
      subroutine imaz(iriinput, imazoutput)
c
c imaz, Ionospheric Model for the Auroral Zone, version 2.0, May 2006
c
c this model was developed using a combination of ionospheric data 
c from the European Incoherent Scatter Radar (EISCAT) in Tromso.
c measurements obtained from Rocket Bourne Experiments
c the technique of neural networks was employed in the development
c of this empirical model
c the model is designed to predict electron densities in the auroral
c zone between the altitudes of about 50 and 150 km.
c
c developed by L.A.McKinnell and M.Friedrich, 
c 1st version in 2003/2004
c 2nd version in 2005/2006
c released for IRI, version 2, 2006
c
c this subroutine takes the inputs as provided by the front end of IRI
c and processes them into the format required by the imaz model
c and then makes the imaz prediction and returns it to IRI or the front
c end program as appropriate
c
c inputs in iriinput as follows
c        iriinput(1) == glat
c        iriinput(2) == glon
c        iriinput(3) == year
c        iriinput(4) == day number
c        iriinput(5) == hour in UT
c        iriinput(6) == riometer absorption, Lv in dB
c        iriinput(7) == magnetic Ap index (planetary A index)
c        iriinput(8) == 10.7cm Solar Radio Flux
c        iriinput(9) == altitude
c
c output in imazoutput as follows
c        imazoutput(1,1) == number of results, 59 if alt is -1, else 1
c        imazoutput(2,i), i=2,60, altitude values in km
c        imazoutput(3,i), i=2,60, log of the electron density values in m-3
c        imazoutput(4,i), i=2,60, uncertainty on the log(ne) predicted
c        if altitude is not equal to -1, then imazoutput(1,1)=1 and
c        the rest of the imazoutput array is set to 0.0
c
c Lee-Anne McKinnell, Hermanus Magnetic Observatory (HMO), South Africa
c May 2006
c
c
c
      REAL iriinput(9), imazoutput(4,60)
      REAL inf(0:7), of(0:2), pressure(59), ninf(0:6)
c      REAL Lv, SF, pi, hls, hlc, za, ic, LMT, chp, lra, li
c      REAL altitude, newpres, ne(2), kt
      REAL Lv, pi, za, ic, LMT, chp, lra, li
      REAL altitude, newpres, ne(2)
c      INTEGER i, j, nolv
      INTEGER i, nolv
      DATA pressure/
     & 8.20E-04, 1.00E-03, 1.20E-03, 1.50E-03, 1.80E-03, 2.20E-03, 
     & 2.70E-03, 3.30E-03, 3.90E-03, 4.70E-03, 5.60E-03, 6.80E-03,
     & 8.20E-03, 1.00E-02, 1.20E-02, 1.50E-02, 1.80E-02, 2.20E-02, 
     & 2.70E-02, 3.30E-02, 3.90E-02, 4.70E-02, 5.60E-02, 6.80E-02,
     & 8.20E-02, 1.00E-01, 1.20E-01, 1.50E-01, 1.80E-01, 2.20E-01, 
     & 2.70E-01, 3.30E-01, 3.90E-01, 4.70E-01, 5.60E-01, 6.80E-01,
     & 8.20E-01, 1.00E+00, 1.20E+00, 1.50E+00, 1.80E+00, 2.20E+00,
     & 2.70E+00, 3.30E+00, 3.90E+00, 4.70E+00, 5.60E+00, 6.80E+00,
     & 8.20E+00, 1.00E+01, 1.20E+01, 1.50E+01, 1.80E+01, 2.20E+01,
     & 2.70E+01, 3.30E+01, 3.90E+01, 4.70E+01, 5.60E+01/

c initialise imazoutput and input arrays
      do i=1,60
         imazoutput(1,i)=0.0
         imazoutput(2,i)=0.0
         imazoutput(3,i)=0.0
         imazoutput(4,i)=0.0
      enddo
      do i=0,7
         inf(i)=0.0
      enddo
      do i=0,6
         ninf(i)=0.0
      enddo

c determine the solar zenith angle from the given inputs
c use za to calculate the inverse chapman function
      call csza(iriinput(4),iriinput(5),iriinput(1),iriinput(2),za)
      call calcchap(za,chp)
      ic=1/chp
c determine the local magnetic time
      call detlmt(iriinput(5),iriinput(1),LMT)
c determine the rest absorption and add it to the given
c riometer absorption to get integral absorption
c if the given Lv is equal to -1, then no Lv was available and
c the network trained with no lv input is used.
      if(iriinput(6).eq.-1.0) then
        nolv=1
      else
        nolv=0
        call calcresabs(iriinput(4),chp,iriinput(8),lra)
        Lv=iriinput(6)
        Li=Lv+lra
      endif

c define pi which is needed to determine cyclic components of LMT
      pi=atan(1.0)*4.0

cc model inputs are read into the inf array
      inf(0)=sin((2*pi*LMT)/24.0)
      inf(1)=cos((2*pi*LMT)/24.0)
      if(nolv.gt.0) then
        ninf(0)=inf(0)
        ninf(1)=inf(1)
        ninf(2)=iriinput(7)
        ninf(3)=za
        ninf(4)=iriinput(8)
      else 
        inf(2)=Li
        inf(3)=iriinput(7)
        inf(4)=za
        inf(5)=iriinput(8)
      endif

c if altitude is not equal to -1, then calculate 1 electron density
c for that altitude
      altitude=iriinput(9)
      if(altitude.ne.-1.0) then
        call atop(iriinput(3),iriinput(4),iriinput(1),altitude,newpres)
        if(nolv.gt.0) then
          ninf(5)=log10(newpres)
          call nolvcase(ninf,of)
        else  
          inf(6)=log10(newpres)
          call iri_imaz(inf,of)
          imazoutput(4,2)=sqrt(abs(of(1)))
        endif
c this is the predicted electron density
c        ne(1)=10**of(0) 
        ne(1)=of(0)
        imazoutput(1,1)=1.0
        imazoutput(2,2)=altitude
        imazoutput(3,2)=ne(1)
      else
c if altitude is equal to -1, then determine entire profile at each pressure
c level from approx. 50 km to 150 km.
        do i=1,59
          if(nolv.gt.0) then
            ninf(5)=log10(pressure(i))
            call nolvcase(ninf,of)
          else  
            inf(6)=log10(pressure(i))
            call iri_imaz(inf,of)
            imazoutput(4,i+1)=sqrt(abs(of(1)))
          endif
c          ne(1)=10**of(0)
          ne(1)=of(0)
          call ptoa(iriinput(3),iriinput(4),iriinput(1),
     &        pressure(i),altitude)
          imazoutput(2,i+1)=altitude
          imazoutput(3,i+1)=ne(1)
        enddo
        imazoutput(1,1)=59.0
      endif

      return
      end
c end of imaz subroutine

      subroutine iri_imaz(inputs, output)
c
c this subroutine normalises the given inputs and presents them to the 
c network file. The output from the network is then converted back to the
c range of the given target values.
c The min and max data structures provide the parameters between which the
c given input space falls
c these data structures need to be changed if the database that the 
c neural network was trained with changes.
c
c Last modified on 22 May 2006, by LAM
c
c Lee-Anne McKinnell, Hermanus Magnetic Observatory, South Africa, May 2006
c
      real inputs(0:6), output(0:1)
      real*8 MIN(0:8), MAX(0:8)
      data MIN/-1.0000,-1.0000,0.0048,0.0000,36.0957,66.2000,-3.2518,
     &  5.5051,0.0000/
      data MAX/1.0000,1.0000,6.9840,300.0000,139.4015,282.6000,1.7482,
     &  12.1830,5.7708/

      real*8 outnet(0:2), innet(0:6)
      integer col

      do col=0,6
       innet(col)= 2.0*(inputs(col)-MIN(col))/(MAX(col)-MIN(col))-1.0
      enddo

c make the prediction
      call imaz_network(innet,outnet)
      outnet(0) = ((outnet(0) - .005)*(MAX(7) - MIN(7))/.99) + MIN(7)  
      output(0)=outnet(0)

c predict the error on the prediction
      call imaz_errnetwork(innet,outnet)
      outnet(0) = ((outnet(0) - .005)*(MAX(8) - MIN(8))/.99) + MIN(8)  
      output(1)=outnet(0)

      RETURN
      end
c end of imaz_main

      subroutine nolvcase(nolinps, nolout)
c
c this subroutine normalises the given inputs and presents them to the 
c network file. The output from the network is then converted back to the
c range of the given target values.
c The min and max data structures provide the parameters between which the
c given input space falls
c these data structures need to be changed if the database that the 
c neural network was trained with changes.
c this is the subroutine that deals with the case where no Lv parameter
c was available
c
c Last modified on 23 May 2006, by LAM
c
c Lee-Anne McKinnell, Hermanus Magnetic Observatory, South Africa, May 2006
c
      real nolinps(0:5), nolout(0:1)
      real*8 MIN(0:7), MAX(0:7)
      data MIN/-1.0000,-1.0000,0.0000,36.0957,66.2000,-3.2518,6.6748,
     &  0.0000/
      data MAX/1.0000,1.0000,300.0000,139.4015,282.6000,1.7482,12.1830,
     &  0.0000/

      real*8 outnet(0:1), innet(0:5)
      integer col

      do col=0,5
       innet(col)= 2.0*(nolinps(col)-MIN(col))/(MAX(col)-MIN(col))-1.0
      enddo

c make the prediction
      call imaz_nolv_network(innet,outnet)
      outnet(0) = ((outnet(0) - .005)*(MAX(6) - MIN(6))/.99) + MIN(6)  
      nolout(0)=outnet(0)

      RETURN
      end
c end of nolvcase

      subroutine imaz_network(in, out)
c
c this is the subroutine that contains the neural network parameters
c the log of the electron density is predicted within this subroutine.
c
c version: 25/5/2006 -- 7-40-1 architecture
c version: 08/8/2006 -- 7-40-1 architecture, IC replaced with ZA,
c                       kt replaced with Ap
c
c Lee-Anne McKinnell, Hermanus Magnetic Observatory, South Africa May 2006
c
      integer N, M, inpnodes, hidnodes, outnodes
      integer member, source, unit
      PARAMETER ( 
     &  inpnodes = 7, 
     &  hidnodes = 40, 
     &  outnodes = 1 )
      
      PARAMETER (
     &  N = (inpnodes*hidnodes+(outnodes*hidnodes)),
     &  M = (inpnodes+hidnodes+outnodes) )

      real*8 in(0:inpnodes-1), out(0:outnodes)
      real*8 weights(0:N-1), biases(0:M-1), act(0:M-1)
      real sum

      data weights/
     & -0.511570, -1.251610, 3.301330, -0.222710, -0.460160, -1.574470,
     & 5.091690, -2.342130, -2.593550, -4.813430, -0.762100, 2.666360,
     & 0.630720, -0.310980, -2.681070, -0.737950, 5.033210, 0.937530,
     & 0.767400, 0.859360, -3.015080, -0.524620, 0.333280, -0.743640,
     & -1.899470, 2.391250, 0.059340, -2.842780, 0.848100, -1.242590,
     & -4.808560, -0.764860, -3.556610, 0.012730, 0.557020, 0.695950,
     & 0.201190, 6.583740, 0.433680, -3.384950, -0.397710, 8.962710,
     & 2.158410, -2.680740, 3.774620, -3.623000, 5.308870, -1.721990,
     & 4.005430, -0.591480, -4.444370, -7.058950, -1.557820, 2.464170,
     & 4.966210, -0.341480, -0.250610, 0.041340, -1.464550, -0.227440,
     & -23.201790, -0.108190, -1.163280, -1.034100, 0.537130, 3.527760,
     & -4.759410, -3.133740, 3.095230, 0.393590, -0.227350, 0.146800,
     & -38.901039, 1.128410, 6.125960, 0.070250, 0.618450, -1.180230,
     & -0.192650, 4.940770, -8.061800, -4.242340, 1.575060, -1.266670,
     & -0.426560, 1.396180, -5.914350, -4.586080, -2.984010, 1.828430,
     & 0.096170, 2.155150, 0.309260, -5.323400, -0.621880, 2.279130,
     & -1.075580, -0.589050, 0.044360, -0.611130, 0.664060, -0.064770,
     & 2.298570, -0.368290, -9.149950, -3.007830, 0.888830, 7.172730,
     & -4.082130, -2.401740, 5.679240, 0.782900, 0.012970, 0.070650,
     & 4.836210, -0.326470, -1.017220, -0.263850, -2.572160, 0.140260,
     & -0.036810, 2.737490, -0.260510, 17.605829, -0.311940, 1.565280,
     & -0.587830, -0.098550, 19.483780, 0.932060, 6.657780, -0.193010,
     & 2.239600, -0.138820, -0.218960, 16.940889, -17.878370, 0.638280,
     & -0.460000, -0.427660, 0.224430, -0.138390, 55.692890, -0.087020,
     & -0.337600, 0.057740, 0.406320, -0.481530, -0.855320, 2.953560,
     & 2.399620, 4.966640, -1.321620, 0.540130, -0.755000, -1.417990,
     & 4.464280, 1.126410, -0.230860, 0.026320, -1.879580, -2.356170,
     & 2.770840, 3.765690, 5.707590, 3.112980, 0.165670, 2.886470,
     & 6.908740, 1.719590, 9.146300, -1.128420, -0.111510, 0.776100,
     & -0.561430, -1.311570, 0.601120, 4.944090, 2.829670, 3.422790,
     & -0.278840, 1.353290, 0.045360, -0.352140, -12.060940, 1.228370,
     & 2.882760, -0.583560, 6.192290, 1.761750, 0.700150, 5.726920,
     & 5.674780, 0.855320, 0.074920, -0.259380, -2.099010, -0.251940,
     & 5.827450, 1.360290, 1.275950, 0.936820, -5.552420, 0.722990,
     & 0.214770, -15.307230, 19.372511, -0.003070, -0.129680, 0.259180,
     & -1.358110, -0.955560, -5.033250, 0.982570, -2.095000, 1.680010,
     & -1.126520, 1.470120, -1.488670, -5.321600, -1.115620, -0.077010,
     & 1.074060, 1.105090, -0.176950, -1.810810, 1.114580, 3.540570,
     & 2.981830, -2.139700, 0.210940, 0.260130, -0.194540, 43.032688,
     & -0.941590, -2.513880, -0.113960, 0.962260, 0.099250, -0.101660,
     & 2.922000, -0.080960, 6.040510, 0.367810, 9.650310, -2.142320,
     & 0.968310, 1.412210, -4.303680, -1.957810, 0.523280, 5.022570,
     & -0.341070, 0.174050, -10.249500, 0.134370, 1.587480, 0.451220,
     & 7.028380, 0.037420, 0.016650, -12.501070, -1.133070, -5.728570,
     & 0.360390, -0.943100, -0.065900, 0.096160, 6.637800, -0.102490,
     & 21.208851, -0.494320, 2.653950, -3.400010, 4.327080, 9.011000,
     & 2.155370, 5.769570, 3.353410, -0.487580, -0.454530, -0.531700,
     & 1.565920, -0.827030, 0.757980, 0.529170, -0.155670, -0.221730,
     & -0.779870, 1.418380, -1.353460, -0.835610, -1.204970, -0.686860,
     & 0.782460, -0.607430, 1.702250, -1.847990, -0.787880, -1.111860,
     & 18.901461, 1.648110, -1.269090, -0.434710, -0.495480, -1.312810,
     & -2.133970, -0.553700, -1.438020, -0.949110, 0.773040, -1.201540,
     & -1.418210, -1.695890, 0.374800, -0.449530, -0.909200, -1.750250,
     & 0.904870, 0.589450/

      data biases/
     & 1.000000, 0.998030, 0.916740, 0.646710, 0.270800, 0.817340,
     & 0.815400, 1.316990, -8.335410, -0.424360, -5.549710, -5.571970,
     & 3.087960, -0.183890, -10.389000, -1.371900, -2.343540, 
     & -38.701450,
     & -4.895420, -11.445310, -8.381840, 2.087660, 0.075450, 0.368070,
     & 1.336170, 17.137079, -1.737150, 58.512260, 0.874180, 3.531080,
     & 3.491330, -1.300910, 0.320340, -15.543630, 8.062280, -1.904190,
     & 3.360630, -4.841030, -8.483840, -2.210050, 42.191399, 4.970690,
     & -3.511160, -6.771350, -11.545940, 3.613440, -0.731090, 
     & -12.939760/

      do member=0,(M-1)
         act(member)=0.0
      enddo

C input layer
      do member=0,(inpnodes-1)
         act(member)=in(member)
      enddo

C hidden layer
      do member=0,(hidnodes-1)
         unit=member+inpnodes
         sum=0.0
         do source=0,(inpnodes-1)
            sum=sum+(act(source)*weights((member*inpnodes)+source))
         enddo
         if((sum+biases(unit)).LT.10000.0) THEN
            act(unit)=1/(1+exp(-sum-biases(unit)))
         else
            act(unit)=0.0
         endif
      enddo

C output layer
      do member=0,(outnodes-1)
         unit=member+inpnodes+hidnodes
         sum=0.0
         do source=0,(hidnodes-1)
            sum=sum+(act(source+inpnodes)*
     &           weights((inpnodes*hidnodes)+source))
         enddo
         if((sum+biases(unit)).LT.10000.0) THEN
            act(unit)=1/(1+exp(-sum-biases(unit)))
         else
            act(unit)=0.0
         endif
      enddo

C the result       
      do member=0,(outnodes-1)
         unit=member+inpnodes+hidnodes
         out(member)=act(unit)
      enddo

      RETURN
      end
c end of IMAZ_prediction

      subroutine imaz_errnetwork(in, out)
c
c this is the subroutine that contains the neural network parameters
c the uncertainty in the predicted log of the electron density is
c predicted within this subroutine.
c
c version: 25/5/2006 -- 7-40-1 architecture
c version: 08/8/2006 -- 7-40-1 architecture, IC replaced with ZA,
c                       kt replaced with Ap
c
c Lee-Anne McKinnell, Hermanus Magnetic Observatory, South Africa May 2006
c
      integer N, M, inpnodes, hidnodes, outnodes
      integer member, source, unit
      PARAMETER ( 
     &  inpnodes = 7, 
     &  hidnodes = 40, 
     &  outnodes = 1 )
      
      PARAMETER (
     &  N = (inpnodes*hidnodes+(outnodes*hidnodes)),
     &  M = (inpnodes+hidnodes+outnodes) )

      real*8 in(0:inpnodes-1), out(0:outnodes)
      real*8 weights(0:N-1), biases(0:M-1), act(0:M-1)
      real sum

      data weights/
     & -0.105310, -1.127650, 0.392770, -0.713520, 0.925380, -1.606360,
     & -0.185840, -2.687430, -0.331280, -2.668760, 0.685700, 0.080520,
     & 0.290320, -1.758100, -4.222380, -2.819080, -0.272980, 0.656130,
     & -2.918490, 3.083010, -1.115220, -0.594210, 0.539000, 0.532970,
     & 0.654520, 1.317070, -0.335230, -5.123020, 2.410500, 0.364150,
     & -1.091370, 0.529740, -0.411480, 0.792080, -1.278130, -0.763840,
     & 0.066010, -1.027990, 1.073050, -0.646970, -0.527620, -0.167790,
     & -0.238830, -0.255150, -6.860000, -0.473680, -0.344860, 0.497470,
     & -1.021240, -0.232050, -0.470680, -5.653240, 0.423440, -0.260550,
     & 1.793200, -0.321980, 0.476480, -0.707270, 1.358910, -0.367560,
     & -6.428190, -0.625990, -1.006140, 0.856220, -0.789120, -0.760610,
     & -0.538160, -0.641960, 1.497400, 1.129090, 1.375940, 0.407700,
     & 0.290040, 3.450290, 5.117750, 5.761180, -2.546970, 1.753940,
     & -0.238730, 0.000570, -3.482250, -0.075710, 0.043220, -0.834340,
     & -0.388200, -0.745690, -1.579420, -0.297170, -0.995950, 1.023240,
     & -2.761320, 1.034540, -0.617020, -2.551870, 0.948890, 2.178500,
     & -0.591860, 3.797240, -0.016980, -0.023830, -15.980340, 0.559820,
     & -0.173800, -0.208790, -0.022000, -2.803690, 0.615610, 3.522490,
     & -1.167470, 1.879290, 1.584760, 0.487610, -0.585090, -0.527860,
     & 3.349250, -3.575380, -0.875330, 0.428970, -2.742800, -0.254160,
     & 2.556020, 0.871240, -1.830670, -0.482570, -1.491150, -0.549490,
     & 2.387760, 1.909200, 3.442310, -0.277560, 1.473900, -0.660990,
     & -0.160740, 0.157010, 2.951870, 0.647280, -1.773040, 0.506610,
     & -0.288830, 0.155560, 0.822180, -3.255350, 1.486360, -1.751770,
     & 0.968230, -0.777840, 1.807800, -0.484200, -0.013440, 1.211970,
     & 1.309910, -1.919040, -7.844890, 0.191000, -0.641110, -0.046510,
     & -2.685450, -1.719270, 0.196330, -1.877280, -1.459280, -0.745170,
     & 4.590540, 4.850230, -2.601460, -2.863660, -0.853890, -0.460120,
     & 1.751990, -1.608570, -2.410580, 0.671090, -2.433770, -0.365370,
     & -2.651160, -0.692420, 1.150480, 1.560750, -0.934590, 8.118160,
     & 0.020940, 1.000870, 0.421630, 1.260740, 1.250440, -0.734110,
     & -0.962600, -2.869310, 0.086850, 0.211140, -1.379770, -0.436460,
     & 0.882940, 0.734310, -0.554430, 0.480130, 0.499690, -0.265170,
     & 5.302570, 2.983970, 0.404040, -2.176810, 0.793890, 1.145530,
     & -0.876100, -4.672580, -1.096890, -1.810390, -0.196010, 2.009740,
     & -0.714520, -0.792040, -0.513540, 0.041310, -1.590940, 5.047720,
     & 0.962710, -0.273990, 0.384170, -1.574580, -2.426650, 1.102960,
     & 4.090620, 0.793280, 2.051070, 0.213290, -0.083290, 2.164570,
     & 0.112460, 0.223030, -3.053580, 0.451040, -0.051420, 4.669130,
     & 0.521860, 1.037520, -0.311460, -2.624650, -0.882530, 0.225230,
     & 5.346270, 1.041920, -0.846060, 1.124440, 0.755260, -1.331540,
     & 1.646660, -3.189990, 0.209910, 1.110460, 0.298320, 2.572130,
     & 2.564860, 2.866820, -5.234080, 1.022460, -2.706360, 1.793960,
     & 0.012040, -0.003550, 0.221290, -11.481430, 1.840110, -0.700490,
     & -0.734690, 0.684000, -0.395550, 1.321550, 0.427310, 1.138070,
     & 1.311320, 0.731340, 3.684480, 0.331500, 0.740000, 0.774620,
     & -0.652340, 0.548040, 0.178820, 0.374390, -2.094760, -2.705800,
     & 1.884110, -2.058660, 2.280880, -1.139060, 6.632080, -6.404510,
     & -3.182430, 0.577210, -1.565290, -2.050070, -2.439870, -4.304150,
     & 8.134030, 2.879330, -3.438850, -2.545030, 3.951750, -2.803160,
     & 2.548610, 5.040560, 2.863360, 2.352620, -2.959460, -3.086090,
     & -2.554770, -1.044640, -3.101920, -3.252140, -2.754210, 3.429000,
     & 3.347190, -3.825990, 6.269090, -3.424240, -2.066190, -9.193190,
     & 7.274360, 0.340360/

      data biases/
     & 1.000000, 0.998030, 0.916740, 0.646710, 0.270800, 0.817340,
     & 0.815400, -1.612240, -1.289640, 0.392060, 1.705290, 0.369720,
     & -0.561920, -4.445460, -6.266580, 0.655770, -0.339970, -0.923520,
     & -0.235640, -1.877280, -1.708540, -13.760660, -0.568650, 
     & 1.575980,
     & -1.143810, 2.232160, 1.077230, 0.157130, -2.537520, -0.022530,
     & 0.459510, -2.074870, -0.851790, -1.600360, 0.527730, 0.066810,
     & -3.757770, 1.011850, -0.042270, -1.960260, 2.299020, 7.068130,
     & -1.950690, -3.009230, -9.800740, -3.752540, -0.901030, 
     & -1.425210/

      do member=0,(M-1)
         act(member)=0.0
      enddo

C input layer
      do member=0,(inpnodes-1)
         act(member)=in(member)
      enddo

C hidden layer
      do member=0,(hidnodes-1)
         unit=member+inpnodes
         sum=0.0
         do source=0,(inpnodes-1)
            sum=sum+(act(source)*weights((member*inpnodes)+source))
         enddo
         if((sum+biases(unit)).LT.10000.0) THEN
            act(unit)=1/(1+exp(-sum-biases(unit)))
         else
            act(unit)=0.0
         endif
      enddo

C output layer
      do member=0,(outnodes-1)
         unit=member+inpnodes+hidnodes
         sum=0.0
         do source=0,(hidnodes-1)
            sum=sum+(act(source+inpnodes)*
     &           weights((inpnodes*hidnodes)+source))
         enddo
         if((sum+biases(unit)).LT.10000.0) THEN
            act(unit)=1/(1+exp(-sum-biases(unit)))
         else
            act(unit)=0.0
         endif
      enddo

C the result       
      do member=0,(outnodes-1)
         unit=member+inpnodes+hidnodes
         out(member)=act(unit)
      enddo

      RETURN
      end
c end of IMAZ_prediction

      subroutine imaz_nolv_network(in, out)
c
c this is the subroutine that contains the neural network parameters
c the log of the electron density is predicted within this subroutine for
c cases where no riometer absorption is available
c please note that no Lv input parameter was used in the training of this NN
c
c version: 23/5/2006 -- 6-70-70-1 architecture
c version: 08/8/2006 -- 6-70-70-1 architecture, IC replaced with ZA,
c                       kt replaced with Ap
c
c Lee-Anne McKinnell, Hermanus Magnetic Observatory, South Africa May 2006
c
      integer N, M, inpnodes, hidnodes1, hidnodes2, outnodes
      integer member, source, unit
      PARAMETER ( 
     &  inpnodes = 6, 
     &  hidnodes1 = 70, 
     &  hidnodes2 = 70, 
     &  outnodes = 1 )
      
      PARAMETER (
     &  N = (inpnodes*hidnodes1)+(hidnodes1*hidnodes2)+
     &        (outnodes*hidnodes2),
     &  M = (inpnodes+hidnodes1+hidnodes2+outnodes) )

      real*8 in(0:inpnodes-1), out(0:outnodes)
      real*8 weights(0:N-1), biases(0:M-1), act(0:M-1)
      real sum

      data weights/
     & -6.887250, -2.170630, 6.462110, -1.419750, -1.518980, 0.571190,
     & -3.923100, 1.292410, 7.167330, 2.509020, 2.100690, -2.027930,
     & -4.537840, -6.222890, -14.064960, -0.852840, 0.036920, -0.906070,
     & 2.512320, -5.297930, 7.630720, 1.508500, -0.249410, 0.773630,
     & 1.684810, -2.468070, -0.654160, 3.071650, 0.068090, 5.374520,
     & -6.580210, 6.999620, -3.563930, -7.570320, 6.693060, 0.956510,
     & 3.807160, 1.763700, 1.001010, 3.615380, -3.283230, 3.470490,
     & 1.119970, -4.526520, -5.923520, 6.088510, -0.985820, -2.882960,
     & -5.813790, 6.589130, 10.073870, 2.743130, 3.414590, -1.231170,
     & -2.853330, -2.299950, -6.551900, 0.084290, -1.816840, 3.498270,
     & -5.877290, -0.624440, 1.974130, 5.144700, 0.790460, -3.413280,
     & 3.066170, -0.813720, 1.951700, 9.704290, 0.202720, -4.564680,
     & -0.719560, 0.743820, 6.212080, 1.659570, 1.861370, 3.720750,
     & -6.287170, 0.596820, 5.103320, -3.534160, -12.815850, 0.632830,
     & -4.772540, -4.739110, -6.807970, 1.447420, -2.708650, 2.902270,
     & -2.108920, -1.722220, 2.374000, 3.009180, -7.958390, 0.606070,
     & -5.942980, -7.880580, 1.296480, -14.888260, -5.612610, 0.245970,
     & -8.003950, -2.110070, 5.699900, 0.081710, 6.235350, 1.256420,
     & -8.684690, 1.836240, 0.236790, 6.441170, -5.302670, -0.067040,
     & -4.439490, 1.315340, -14.892940, -11.228070, 8.431810, 0.600750,
     & 3.840780, 2.995260, 0.697520, 5.029940, 5.141680, -2.283390,
     & 0.459650, 7.573720, 9.397140, -0.488880, 0.913460, 2.757070,
     & 2.197280, 7.202190, 1.332190, -10.052930, -1.071670, 0.408830,
     & -0.240690, 2.439450, 1.192680, -2.404640, 3.082580, 4.185400,
     & 3.984150, -4.392580, 5.598670, 1.369350, 1.417930, -0.611380,
     & -5.692570, 2.930960, 3.799840, -8.822640, 3.892870, -4.316730,
     & -4.101270, -2.578540, 4.038110, 3.523080, -0.156150, -3.290550,
     & 3.844600, -0.275680, 3.286880, 0.897260, 4.387790, 3.608560,
     & -5.665600, 2.221130, 7.153830, -2.696570, -1.931500, 1.770440,
     & -1.673490, -6.091450, 5.014370, 1.660930, 3.748810, 0.670700,
     & 1.728720, -1.062070, 1.141500, -3.655380, -11.357580, -1.441910,
     & -4.784830, 7.543880, 8.568860, 2.858580, 5.835900, 2.161170,
     & 0.475220, 3.400660, 2.489530, -8.286080, -1.934940, -1.183500,
     & 5.293450, -3.983170, 3.949120, -3.209960, -11.408340, 1.156280,
     & -3.740700, 3.759920, 6.791800, 9.255950, -1.351520, 0.302070,
     & 3.381510, 3.304340, 1.757090, -3.659250, -4.553250, -4.739890,
     & -11.201900, -2.752550, -3.743120, -8.669670, -1.472410, 
     & -0.323220,
     & -0.093270, 0.650630, -1.156040, -0.205310, 11.719210, 1.760150,
     & 5.153510, 0.419730, 9.423800, -1.932790, -3.834230, 1.624340,
     & 1.751870, 2.776190, -2.258970, 0.900420, 2.275390, 4.833630,
     & 5.394580, -3.619180, -8.440510, 8.322110, 3.779350, 0.858790,
     & 3.131090, 0.376520, 4.420700, 6.504570, -1.124750, -4.879530,
     & -3.030620, -2.252970, 4.399220, -2.806620, 1.310280, 2.319810,
     & -0.347850, 9.603840, -1.963830, -1.561880, -5.050600, -1.047110,
     & 0.435630, -4.969470, -4.950990, 5.165870, 6.138200, 2.426600,
     & -2.324780, 5.792700, 16.295641, 10.886940, -5.165370, 1.271290,
     & -1.357080, 0.118950, -3.481320, -4.130150, 0.444680, 4.998460,
     & 4.058900, -2.025720, 3.201700, -8.665160, -1.042650, 2.174610,
     & 0.973270, 8.948960, 7.117020, -8.340330, 2.505560, 0.525100,
     & 1.184610, -1.585930, 7.774110, 1.481460, -0.012240, -3.798410,
     & 2.652610, 0.652430, 1.247280, -7.015350, -1.883300, -4.111330,
     & -2.992170, 0.081230, -2.847570, -1.925140, 5.896530, -3.851090,
     & 0.089350, 0.765600, -5.127910, 11.914830, -14.685720, 0.425080,
     & -1.104640, -2.470850, -1.028930, 6.961850, -8.226580, -3.251210,
     & -3.433160, -2.465950, 0.553750, -6.539820, -1.779020, -1.397800,
     & -1.608130, -5.091170, 0.665570, 9.841970, -3.297480, 0.479090,
     & 6.340340, 0.668650, -2.834110, 7.277120, -4.849410, 2.108190,
     & 1.382880, -3.626410, 2.387360, -2.195360, 3.140510, -0.402540,
     & 0.278190, -1.799990, 5.110270, -3.487000, -2.571810, -4.804120,
     & 2.775290, -0.663630, 4.098140, 7.082530, -10.765880, 0.408340,
     & -1.459950, -7.306000, -6.931480, 5.455240, 10.249200, 0.132300,
     & -3.025270, -1.345960, 2.454180, -5.333800, -2.489750, 3.477880,
     & 3.726100, -1.017610, 3.330390, -2.008520, 4.398590, 1.009170,
     & -5.234420, -1.286130, -5.473130, -6.360340, -6.823380, 0.227370,
     & -8.388170, -1.639810, -2.778090, -4.814880, 10.296390, 0.796890,
     & 8.137870, -2.410870, 5.611190, 6.116440, 5.155170, 1.549250,
     & -7.602460, 2.369240, 4.568010, 6.456550, 4.674560, 0.751740,
     & -1.406640, -7.283280, 1.456710, 8.637940, 8.688630, 2.129710,
     & -5.187620, -6.431960, 6.414660, 9.708550, 1.296430, 1.581900,
     & -1.873660, -4.253620, -11.287330, 3.207800, 2.444290, 2.137450,
     & 0.526080, 0.962130, -0.488130, -0.859150, 0.862410, -1.678200,
     & 0.716330, 1.436060, -0.591270, -0.674520, -0.435260, -1.080440,
     & -1.149320, -0.430230, -0.738140, -0.830140, -0.129930, -1.919360,
     & 0.626800, 0.113190, 0.211610, -0.323280, 1.531620, 0.168360,
     & -0.205490, 0.202500, -1.152730, 0.334390, -0.715410, 0.162110,
     & 0.561640, 0.570060, -0.670620, -3.997730, -1.151340, -0.075320,
     & 0.380360, 1.224200, 1.118930, 0.539580, 1.753810, -1.142170,
     & 0.224780, -0.026820, 1.177850, -1.122190, 0.607030, -0.899150,
     & -1.006350, 0.393410, -0.415240, 0.364500, 0.002180, 1.107710,
     & 0.836090, -0.675650, -0.343410, 1.198860, -0.707680, -1.446020,
     & 2.089400, -0.047100, -0.208480, -4.527220, -1.003110, 1.410930,
     & -2.661930, -2.089750, -1.134130, -1.340110, -1.170210, -0.617140,
     & -1.238280, -0.330500, 0.922210, -1.004020, 0.679390, -0.121200,
     & 0.290390, -1.629690, -0.659140, -0.347090, 1.366080, -0.185060,
     & 0.197090, -0.072430, 0.290590, 0.695190, -0.224720, 0.475350,
     & 0.185300, 0.349740, 0.326940, 0.866450, -0.995820, 0.683950,
     & -0.587870, -0.627630, 0.758690, 1.128700, -0.508950, 0.321990,
     & 0.059720, -1.702370, -0.240780, -0.988250, -1.051090, 0.081120,
     & -0.825900, -0.238930, 0.165140, -0.730270, -0.445470, -0.952170,
     & -1.283330, -0.519830, -0.290970, 0.132290, 0.323590, 0.792320,
     & -0.104230, -0.512960, -0.190410, -0.824790, -0.151100, -0.466860,
     & 0.560220, -0.703950, -0.170540, 0.819770, -0.656400, 1.066460,
     & 0.637290, 0.291700, 0.841260, 0.680450, -0.329280, 0.950760,
     & -0.207680, -1.946850, -2.660420, -0.693470, -2.491580, -1.045860,
     & 0.185110, 0.666490, -0.636200, 0.445080, 0.831930, -1.100420,
     & -1.263950, -0.588450, 0.656020, 2.091650, -1.762770, -0.845580,
     & -0.301620, -0.513930, -0.887000, -2.020900, 1.467900, 1.041670,
     & -0.673840, 0.097110, 0.313020, 1.647580, -0.957050, 0.133890,
     & -0.792470, 0.758980, -0.373110, -0.784300, -0.262920, -3.248120,
     & -0.151100, -1.575010, 0.473710, -0.953830, 0.538620, -0.867730,
     & 0.475560, -0.371170, 0.102340, -0.097250, -1.089300, -0.383960,
     & -0.006890, 0.187640, 0.973770, -0.704700, -0.952460, 0.900420,
     & 0.788660, 0.681000, -0.058870, 0.049280, 1.986830, -0.653830,
     & -0.447590, 0.318200, 0.281960, -0.045170, 0.144930, -0.408210,
     & -1.132470, -0.997730, -1.162600, -0.222460, -0.485540, -0.754070,
     & -1.404660, 1.754760, 0.624450, 0.656880, -1.969780, 0.767750,
     & -1.136460, 1.954280, -0.947370, 0.531540, -0.021210, 1.673350,
     & -0.132240, -0.889290, -0.129210, -0.075160, 2.664060, -0.022310,
     & -0.152600, 1.874500, 2.154960, -1.251810, 0.581470, 0.249430,
     & -0.042390, 0.403970, -0.021600, -0.228750, -0.770380, -0.284670,
     & 0.821190, 0.518650, 0.182070, -0.102180, -1.353340, -0.554760,
     & -0.518150, -0.325270, -0.112710, -2.398810, -2.047550, 1.315900,
     & 1.591940, 0.874580, -1.557390, 0.269130, -1.471830, 1.074140,
     & 0.653730, 0.473200, 0.712390, -0.758260, -0.350040, -1.187680,
     & 0.815170, -0.826010, 0.105720, 0.446330, 2.604180, -0.165250,
     & 0.643800, -0.241170, -1.058080, 0.226180, -0.686180, 0.712830,
     & 0.707250, -0.709200, -1.289830, -0.785980, 1.312260, -0.662620,
     & -0.145720, -2.008190, -0.165290, -0.967290, -0.644560, 0.468250,
     & -1.777140, -1.376290, -2.414940, -0.397450, -0.613000, -2.676210,
     & -0.988060, 1.567910, -2.200430, -2.371460, 2.220300, 4.575220,
     & 2.053130, -3.206560, 0.389930, -1.513220, -4.067210, -2.163290,
     & 0.910750, -2.029430, 2.695040, -0.286720, 2.264580, -0.182350,
     & 1.785890, 4.111340, -2.479000, -0.749500, -1.142800, -5.302970,
     & 0.951550, -0.294530, 0.617410, 0.988250, -0.707240, -1.530460,
     & 0.227170, -5.396510, 2.019950, -0.310440, -0.142480, -1.278780,
     & 0.356930, -0.132020, 4.097280, -0.565030, 4.400450, 1.908970,
     & -0.101260, -2.245450, -0.174540, 0.366140, -2.361530, -1.230980,
     & 1.011390, -0.883120, 1.185710, -2.056000, -3.855930, -1.054730,
     & -1.223970, -0.130740, -0.749100, -1.224570, 0.344490, -0.892870,
     & -0.180030, -2.861410, -0.294800, 1.529830, -1.360850, -3.027330,
     & 1.068420, 2.147950, -1.165400, -2.039150, -2.097350, 1.540740,
     & 1.328740, 2.531980, -3.600420, -3.582850, -1.261250, -1.235450,
     & -0.467990, 1.705920, 0.622200, -0.897720, -0.968960, -0.502170,
     & -2.278130, 1.434810, -3.059730, 1.419430, -1.547090, -0.097240,
     & 1.181730, -0.860580, 0.857600, 1.437850, -1.512010, 0.926700,
     & 2.835480, 0.324770, -0.671440, -1.757650, -2.226020, -0.931990,
     & 0.272280, -1.012250, -1.022450, 1.253090, -0.393330, -0.747320,
     & 1.458370, -0.776130, -0.858270, -2.596580, -1.628060, -1.376690,
     & -0.423440, -1.078600, 1.857470, 1.473510, 0.546030, -2.287970,
     & 2.317430, 2.402160, -0.592830, 2.833530, 2.758870, 0.949880,
     & -1.053390, -0.388980, -0.049140, 0.188530, 0.496930, -2.773920,
     & -0.768830, -0.137230, -1.085340, -3.490010, -0.496170, -0.519520,
     & 1.276520, -1.354910, 0.012860, -1.179520, 4.148250, -0.768040,
     & 2.617290, -1.974180, 1.111910, 1.965270, -1.514670, 0.495490,
     & -1.464920, -0.071510, 0.747300, -0.787100, -0.229110, 0.307390,
     & 0.111190, -1.413270, -0.652340, -0.780140, -1.327410, -0.508760,
     & -3.249670, -0.666200, 2.940260, -0.404250, -0.631520, 1.611270,
     & -0.521930, 1.143280, 1.891710, 2.841560, -1.433460, 0.184010,
     & -0.889070, -0.843950, 0.470500, -1.568010, -1.285940, 0.240120,
     & 0.131370, 0.037960, -0.817520, 0.200800, -0.716010, -1.373750,
     & -2.008530, -0.980190, 0.260450, 1.593280, 0.329870, -0.213000,
     & 0.013980, -0.887530, -1.481750, -1.472160, 1.134750, -1.712130,
     & 1.464570, 2.105070, -2.577730, -0.506420, -1.579100, -0.301330,
     & -0.203290, 0.725460, -1.813100, 1.790150, -1.321370, -2.960560,
     & -0.131150, 1.470600, -0.377540, -0.921710, -1.494270, -0.328920,
     & -0.342270, -0.406770, -0.538930, 0.748860, 1.702370, 0.073970,
     & 0.389710, -0.313870, -1.115600, 0.411100, 0.352750, -1.140540,
     & -0.449850, -2.569610, 0.463710, 0.634780, 0.316100, -2.112890,
     & 0.458570, -0.239520, -0.400530, 0.239680, 1.173640, 0.090960,
     & -0.533120, 0.657610, -1.241450, 1.881120, 0.058790, -0.957340,
     & 2.750040, 0.497510, 0.449630, -0.010200, 1.562010, 1.969160,
     & -0.671370, 0.579610, -0.329680, -1.777260, -2.701670, -1.584420,
     & 0.050500, -0.951390, 0.258840, 0.979930, -0.706220, -1.648140,
     & 1.492200, -0.215610, -2.480800, -1.950450, -0.932910, -0.353210,
     & -0.917790, 1.683600, -2.376620, -0.299540, -1.795200, -0.579710,
     & -2.686540, -0.006910, 0.064460, 5.650260, -1.405670, -0.709710,
     & -0.742380, -0.737320, 2.464040, -2.965800, 1.457540, -0.926940,
     & -0.940100, 0.249890, -1.816830, -1.408070, -0.678500, -1.446150,
     & 1.246170, 0.550500, -0.012050, -3.488130, -1.266260, -1.390700,
     & 0.428440, -0.208630, -0.484820, 1.014900, 0.501670, 0.095320,
     & -1.363770, -1.034500, -0.085380, -0.420710, 2.022350, -2.603020,
     & 0.649210, -3.525770, -0.127760, -0.614160, -1.110230, 2.777400,
     & -0.132870, -0.920530, 0.750490, -0.762940, 1.837160, -0.326250,
     & -0.473150, -0.139210, -2.271320, 0.725440, 0.963170, -0.912410,
     & 2.483080, -1.655590, -1.062760, -1.516780, -0.218970, 0.166350,
     & 0.608580, 0.355990, -2.772660, 0.893340, 1.360930, -0.710500,
     & -0.984650, -0.038290, 0.124630, 0.398090, -0.941430, -1.253280,
     & -0.085100, 0.738750, 1.367930, 0.613690, -2.296580, 0.810210,
     & -1.431880, -2.056200, 0.813220, 1.381650, -1.211660, 1.493590,
     & -0.620920, -0.085960, -1.305890, -1.190850, -0.249600, -0.118060,
     & -1.816000, -1.072690, 0.376400, 0.381650, 1.007260, -0.938550,
     & -0.592680, -0.713520, 0.450330, -0.005940, 0.315350, -0.054470,
     & -0.139520, -1.344030, -0.203040, 0.026990, 1.156150, 1.391080,
     & 0.316130, -0.624280, -2.549650, -2.323530, -0.643500, 0.775220,
     & 0.670420, -1.260880, 1.063640, -0.267590, -0.463920, 1.188000,
     & 0.955270, -1.788330, 0.926220, -0.461900, 0.406470, -0.641880,
     & 0.648230, 1.033490, 1.192460, 1.447980, 0.674420, -1.579290,
     & -1.143780, -1.424010, -0.519170, 1.238100, 0.303150, 0.298220,
     & 0.083540, -1.240370, 2.420060, -2.579290, -0.191280, -5.287650,
     & 0.849110, 0.009300, -0.988060, -0.901380, 0.928950, 5.410360,
     & -1.165620, -0.495600, -1.293960, -0.827780, -2.713150, 1.699030,
     & -0.394690, -0.690640, -0.543130, -1.013990, 0.377620, -1.945100,
     & -1.410230, -2.445160, -2.905320, 0.288260, 1.380380, 0.850100,
     & 0.411250, -0.594230, -1.432930, 1.353620, 0.943730, -3.919910,
     & -1.534730, -2.427190, 0.332070, -0.138160, -0.585730, -1.204790,
     & -0.100020, -1.373590, -0.899370, 0.131960, 0.374800, -1.656240,
     & -0.370060, -1.893820, -1.422450, -0.261450, -0.900660, 0.157190,
     & -1.317140, 1.119720, 1.311380, -0.447220, 0.858980, 1.298560,
     & 0.048720, 1.680290, -0.644300, 0.105410, 2.639680, 2.012450,
     & 0.050850, -1.569960, -2.669950, -0.022200, 0.095660, -1.434780,
     & -0.976170, 0.710310, -1.199480, -0.940130, 1.441030, -0.221570,
     & -2.325440, -1.046500, 0.772550, -1.229350, 0.840530, -1.425480,
     & 1.080140, 0.029920, -0.552230, 1.514050, 1.496310, 0.180850,
     & -2.976880, -0.457310, -4.840020, -0.823300, 0.496340, -0.970110,
     & 0.714530, -0.903120, -0.538420, 2.039470, 1.040260, -1.005550,
     & -1.879290, -0.695120, -0.651200, -0.267510, 1.823960, -1.428130,
     & 2.369630, 1.054190, -0.121020, 0.413590, 0.001760, 0.326030,
     & 0.660230, -2.883410, -1.144360, 0.471190, -1.267260, -0.060460,
     & -1.577790, -0.869510, 1.700970, -0.781760, 0.478910, -1.354980,
     & 0.616460, 0.079190, 0.232450, -1.118260, 1.191910, 0.040310,
     & 0.998260, -0.940120, 0.253340, 0.110240, -0.243190, -2.960410,
     & -1.002780, -0.906620, 0.452020, -1.155880, -0.122230, 1.652620,
     & -1.617730, 0.420580, -0.347450, 0.409480, 0.773860, 0.775600,
     & 0.918590, 2.436110, 0.228530, -2.140460, 0.898910, -1.239640,
     & 0.001230, 0.084750, -0.856690, 0.168100, -0.704090, -0.572430,
     & 0.379770, 0.186510, 0.909400, -1.408860, 0.253440, -0.334390,
     & -1.592490, -0.539530, 1.045050, -0.731710, -0.866970, -1.325780,
     & 0.832080, 1.962780, -0.447940, -1.290320, 0.662170, 0.362990,
     & -2.091560, 0.130470, 0.148460, 0.161350, -0.569740, 0.168440,
     & 1.216490, 1.285850, 0.010400, -0.833730, -1.458200, -1.211390,
     & -0.083950, 0.909690, -0.889130, 0.109590, -1.579040, -0.437360,
     & -1.112480, 0.883250, 0.505590, -0.901000, 1.757730, -2.336250,
     & -2.741210, 0.694130, 0.161270, 5.277350, -0.476080, -1.788570,
     & -0.135620, -4.103810, 4.532450, -2.290280, 0.618220, 2.758380,
     & -2.012300, -1.052970, 4.343780, -0.413920, -4.719230, 3.235850,
     & 3.843690, 0.139080, -4.808140, 0.581160, -0.941780, 0.242060,
     & 2.171610, 1.911820, 1.128050, 0.717660, 2.935230, -1.402320,
     & 2.047680, -0.691680, 4.178830, -1.925720, 2.510530, -1.317900,
     & 3.608680, -0.829650, 0.900060, -0.732170, -0.467830, 0.381510,
     & -0.658880, -5.258700, 0.257300, -0.288130, 1.920200, 1.080450,
     & -1.224870, -0.046720, -5.732000, 0.371300, 0.283290, 0.363260,
     & 2.180380, -1.357110, 0.822510, 3.807040, -3.699030, 2.291630,
     & -0.347950, 6.207290, 0.010150, -1.554580, 6.326660, 2.529920,
     & 1.480560, -0.103110, -1.415080, -0.722360, -1.129810, -0.060520,
     & 0.132670, -0.900550, -0.404290, -1.791260, -1.148670, -0.617750,
     & -1.036620, 1.510580, -1.434400, -2.515390, -0.829530, -0.377530,
     & 0.810670, -0.173880, -0.840460, 1.098220, -0.821820, -2.143650,
     & -2.057940, -0.254570, 0.563410, -0.932080, 0.423780, 1.490860,
     & -0.429510, -0.531870, -1.954900, 0.192110, 0.914190, 2.145790,
     & -0.798980, -1.541690, -2.652610, 0.559670, 1.820660, 0.488290,
     & -1.236880, 0.339100, -0.487690, -1.633270, -0.653880, 2.569040,
     & 0.785830, 1.296760, -1.909490, -1.210150, -0.400800, 1.582780,
     & 4.758240, 0.110980, -0.770060, 0.101880, -0.021030, -0.414020,
     & -0.877230, -1.056680, -0.690700, -1.918690, 0.471240, 0.587650,
     & 0.100710, -0.676590, -0.706220, 1.268910, 0.387780, -0.952590,
     & -2.487290, -0.069510, -0.069810, 0.569240, -0.688540, 1.054390,
     & -0.330770, -0.590190, -0.628100, -3.091530, -2.078020, -0.848500,
     & -1.708640, -4.045020, 0.686940, 0.066370, 0.468680, -3.819440,
     & 1.400590, 1.057230, 0.963170, -1.996490, 0.460840, 0.272960,
     & -0.144580, -1.448030, -0.557820, 0.077880, -2.768040, 2.694490,
     & -0.720940, 1.502000, -0.581380, 2.812790, -1.358820, -2.441390,
     & 1.098180, 0.462620, -0.364070, 0.421440, 2.232300, -1.270350,
     & 0.067660, -1.708300, -2.136780, 3.097840, -1.730220, -0.217300,
     & -3.252570, 0.467920, -1.196810, 1.754690, -0.054820, 2.094770,
     & -3.100300, 1.884060, -1.675380, 0.011430, -0.074970, 1.830640,
     & 2.186610, 0.648950, -1.235520, 0.181440, 2.936280, 0.092000,
     & -2.026960, 2.856310, 2.520660, -0.581860, 0.885020, -1.787830,
     & -1.940160, -2.179940, 1.207660, 0.368320, 0.206690, 0.974990,
     & 1.655710, 0.709780, -2.410490, 0.131850, -0.734010, 2.667290,
     & -0.713230, 0.071400, -1.500540, -2.097940, 0.521930, 0.176240,
     & 0.601340, -0.755080, 2.196660, 0.953690, 0.407180, -2.529230,
     & -1.006800, 0.866220, 1.280300, 0.835650, 2.364030, -2.731520,
     & -2.230190, 2.485910, -3.401880, 1.816090, -0.878070, 0.173750,
     & 0.425980, -1.439300, 2.421390, 1.665150, -1.901560, 1.610530,
     & 1.112090, -1.458430, 0.537440, -3.924230, 0.144880, 0.190780,
     & 0.904950, -0.837860, -1.086850, -0.817710, 3.632170, -2.913690,
     & 1.264520, -0.173430, 1.089910, 0.900100, -1.233930, 0.557430,
     & -0.507930, -2.103770, -1.470920, -0.810280, -0.984760, -1.938340,
     & -2.097720, -1.344040, 2.234100, 0.393780, -2.464610, -0.686040,
     & 1.873960, -1.194500, -0.042420, 0.710710, 1.273230, -0.515550,
     & -0.692280, -0.228640, -0.363450, 4.367210, -1.501000, -1.909680,
     & 3.161640, -2.812280, -0.553090, 1.262540, -1.723660, 1.078760,
     & -0.810110, 1.774700, 1.721180, -0.157520, -1.108500, -0.429090,
     & -1.144970, 0.485490, -0.154640, 1.682750, -1.337670, -2.694980,
     & 0.122020, 1.674150, 1.334730, -0.453270, -0.340130, 2.155360,
     & 1.133240, -1.308770, -1.944290, -0.865790, 0.390480, 3.751480,
     & 2.007130, -2.283200, -1.072050, -1.731220, -0.634900, -1.432790,
     & -0.934560, -1.523950, -1.666280, -5.994060, 1.713380, 0.765470,
     & -0.192580, -1.006470, -1.741140, -0.731650, -1.545570, 1.206860,
     & -0.808170, -1.547510, 0.517750, -4.038670, -0.153200, -1.844240,
     & -0.372590, 0.606330, -1.498860, -3.082640, -0.029610, -0.261380,
     & -0.710930, -1.079780, 2.211140, -1.416800, 0.298760, -0.448390,
     & 1.828720, 2.823690, -3.585500, -1.434520, 0.768440, -0.670840,
     & -0.925690, 0.328470, 0.929960, 0.415560, 0.285820, -0.000720,
     & 1.324580, 1.627700, -0.258520, 0.495560, -2.019810, -0.966670,
     & -1.160310, 4.563620, -2.883060, -1.785670, 0.757130, 2.676360,
     & 1.310660, 2.846280, -1.601440, 0.811570, -2.602190, 1.875200,
     & -0.109380, 0.970810, 2.280320, -2.118950, 0.119620, 0.036380,
     & -1.568190, -0.531160, -2.571600, 2.451440, -2.246240, 2.345320,
     & 0.120360, 1.348300, -1.969520, 2.446280, -0.961550, -1.256980,
     & 0.022160, -0.257880, 1.023290, 0.696600, -1.607580, -2.012580,
     & 4.434410, 2.160690, -1.880130, -1.401560, 0.703000, 0.798380,
     & -0.277990, -1.783480, 1.674960, -3.422680, 1.427160, -0.077380,
     & -0.594800, -0.790920, -0.568380, -1.346370, -0.006830, 0.171130,
     & 0.889180, 0.288130, -0.973160, -0.355170, 0.775850, -2.628850,
     & -0.545090, -0.103420, 0.107140, -0.833350, -1.819690, 0.433430,
     & 0.683410, -0.273430, -1.915360, -0.556250, -0.637010, -0.367510,
     & -0.919860, 0.069960, -2.075960, -0.928950, -0.823930, -0.658610,
     & 0.096140, -1.149050, -0.985910, -1.897180, 0.822620, -1.157680,
     & -0.357680, 0.229180, 1.208550, 0.831040, -1.069520, 0.383440,
     & -1.964640, -2.255210, 0.875000, 0.570520, 0.258240, 0.788580,
     & 0.028080, -0.886570, -1.296030, 1.182050, -0.657760, 0.698780,
     & -0.772750, -0.776920, -2.315840, -0.666540, -0.722550, 0.387850,
     & 1.927610, -0.186710, 0.599350, 1.998070, -0.775550, 0.776990,
     & -1.650820, 1.159270, 1.061730, 1.654400, 2.485440, -1.963230,
     & -2.803060, -0.430940, 2.951830, -0.587910, -3.118160, -1.365580,
     & -0.368830, -1.898000, -1.135950, -6.710250, 0.315330, 0.862820,
     & -0.486470, 0.255130, 3.462140, -0.089810, -0.858520, -0.221950,
     & -0.181160, 0.474420, -2.307540, -0.362090, -1.232660, 1.320770,
     & 3.594580, -0.611700, 0.537280, 3.323940, 0.752560, 0.008480,
     & -5.158660, -0.107000, -0.466480, -3.155060, -1.108000, 2.057540,
     & 1.678920, 1.644370, 1.285220, 2.210920, 0.049790, -2.261990,
     & -0.179950, 1.389970, -1.287620, 0.481520, -0.561070, 4.062060,
     & 2.006780, 1.525300, 0.945000, 1.065260, 0.162190, -4.083690,
     & 2.508700, 2.248010, -1.466300, 0.294990, -0.275280, -0.484330,
     & 2.203970, 0.886700, -1.526420, 0.923580, -0.359130, -0.750500,
     & -2.672330, -1.154220, -1.270290, -3.187350, -1.973280, 2.384450,
     & 2.548360, 0.406290, -3.311580, -2.258190, 1.850180, -1.241000,
     & -2.021420, -1.808410, -1.070240, -2.561220, -1.901460, 0.488970,
     & 1.067370, 0.391560, 2.052270, 3.515010, 2.095340, 2.616520,
     & 0.354590, 1.503410, -0.390560, -0.798200, -0.025010, 3.025070,
     & -0.539860, -0.808640, 1.930220, -0.503250, -4.398650, 0.210510,
     & 0.071490, -3.018790, 0.057690, -4.641500, -0.345120, -0.512720,
     & -0.465290, -0.186650, 1.689190, -2.403930, -2.774560, -2.098840,
     & 1.572900, -2.335600, 0.785440, 2.813620, 1.023520, -1.891200,
     & 1.718550, 4.790800, 3.023140, 0.110190, -2.074960, 1.739660,
     & 0.460450, -1.548930, 1.168120, -1.656880, -0.613580, -1.412290,
     & -3.186330, 0.763800, 0.701820, -0.855540, -1.134960, -0.482730,
     & -0.093800, 0.568630, 2.056160, -1.437790, -0.237410, 0.220210,
     & -1.237000, 2.290470, 1.447580, 0.760470, -0.874500, -1.333070,
     & -1.214610, -3.205770, -1.972940, -0.071240, -0.185200, 1.188760,
     & 1.539970, 1.854480, -2.376740, -1.893700, -0.115880, 0.067630,
     & -1.458010, 0.300770, 0.151530, -0.569920, 0.046700, -1.101680,
     & 1.592130, -0.477690, 0.513840, 0.517960, 0.852050, 2.012880,
     & 0.682470, -0.825570, 0.303810, -0.577270, -2.251260, 1.441020,
     & -1.276330, -2.069900, 1.093360, -0.222960, -0.663610, -0.489190,
     & -1.599770, -1.106750, 0.820150, 0.965310, -2.717020, -0.331990,
     & -0.442280, -0.520090, -2.049980, 1.132050, -1.972890, -0.330630,
     & 1.810990, 1.446310, -0.587440, 1.931550, -2.267490, 0.764350,
     & -0.381100, 0.781300, -1.604780, -0.406390, -0.103130, -1.507000,
     & 2.201880, -1.299420, 0.670110, 0.909670, 0.008600, -3.527190,
     & 1.825780, -0.815850, 1.099320, 3.230040, 0.372320, -1.640690,
     & -0.002140, -0.450590, -1.389140, 2.214960, 1.213770, -0.551330,
     & 0.097090, -0.381170, -2.313280, 0.813150, 0.631370, 2.969440,
     & -3.487810, 0.587120, 0.311700, -1.161900, -0.503180, 0.269470,
     & -1.364360, -2.009150, 0.020540, -2.014230, 0.075370, -5.354080,
     & -0.636510, 0.052240, -1.392480, -0.289350, 0.537640, 1.356180,
     & 2.139550, -1.640130, -0.671730, -2.683810, -3.708590, -1.657390,
     & -2.246260, -0.517660, -1.369410, 0.409640, 0.638310, -1.669280,
     & -1.313250, -2.328050, 1.657100, 1.485050, 1.733480, -1.109050,
     & 8.170540, 4.255550, -5.908490, 4.650040, 0.969660, -1.742290,
     & 2.138710, -2.991310, 4.161840, -4.175660, -0.355770, 0.698770,
     & 6.061010, 1.875950, -4.984440, -0.607430, 2.741900, 2.999220,
     & -0.303060, -4.617620, -1.611310, 4.494130, 3.831000, -1.632440,
     & 5.251450, 0.068480, -2.636500, 1.954580, 3.810880, 2.602200,
     & 1.364380, 5.378700, -2.102600, -1.313330, 1.529600, 0.578330,
     & 2.762050, -1.579380, -1.124980, -1.448780, -1.615180, 1.277600,
     & 4.195570, -5.002610, 3.632730, 7.226920, -3.049100, -0.574050,
     & 6.683010, 3.491530, -0.523610, -1.222600, 3.476500, -0.211080,
     & 2.510060, -2.131730, -0.946210, -1.727170, 2.195540, -2.877710,
     & -4.599530, 3.754290, 4.057070, -1.770500, 1.496740, 7.270720,
     & 2.317510, -0.358540, -2.448270, -5.672760, -0.512500, 1.717900,
     & 1.218850, 0.338460, -0.871820, -1.510190, -0.174660, 1.040820,
     & -0.148420, -1.260600, 1.042840, 0.304250, -1.058460, -0.450690,
     & -0.050590, -0.921250, 0.812890, -1.008410, -0.359350, -3.208000,
     & 1.321040, -1.583800, 2.481720, -0.421910, -0.742430, 1.603250,
     & 1.910710, -1.095200, 0.271610, 1.100140, -0.900530, 1.827290,
     & -1.574190, -1.193640, -0.973980, 0.056280, 0.945170, 0.742670,
     & 0.934160, 0.394010, -0.938250, 0.165550, -0.196260, -4.209960,
     & 1.478710, 0.947620, 0.629050, -1.706590, -1.937080, -1.145670,
     & 1.510620, 0.063400, 2.397490, 0.260510, 1.079560, 0.471400,
     & -2.043690, -0.759190, 0.857080, 1.061500, -2.933500, -1.427200,
     & -1.940140, -1.331950, -1.230160, 0.613370, 0.550160, -0.450680,
     & -1.273660, 2.041460, 0.121740, -0.564260, -1.234630, 0.381860,
     & -0.895090, -0.665620, 0.735200, 0.646820, -0.489600, 0.871510,
     & 0.882250, -0.178040, -0.511320, 0.479940, -0.565850, -0.288640,
     & -1.424050, 0.002150, -0.569790, -0.043860, 0.501590, 0.268820,
     & 0.404890, -0.063540, 0.643370, -0.282490, -0.182240, -1.159780,
     & -0.685980, -0.729030, -0.441790, -0.148940, 0.448300, -0.699590,
     & 1.177770, 0.436530, 0.687870, -1.054640, -0.370830, -1.110570,
     & -0.920020, -0.414260, -0.675810, 0.514180, -0.980480, 0.559860,
     & 0.773470, 0.101810, 0.095600, 0.231210, -0.054730, 0.022120,
     & -0.924340, 0.000680, -0.986410, -0.149200, 0.386420, -0.741190,
     & -0.689790, -0.775400, 0.093780, -0.909190, -1.198080, -0.271650,
     & -0.939370, -0.961490, -0.327740, -0.816530, -0.436580, -0.681850,
     & 0.442820, 1.617050, 1.189400, -2.759040, -1.720330, 2.056210,
     & 0.750360, 0.699270, -0.755540, 1.953060, 1.219130, -0.443760,
     & -1.429920, -3.018240, 1.293790, -5.398050, -0.852690, 0.796610,
     & -1.764810, -2.044620, 0.664350, 1.317650, 3.407460, 0.056010,
     & 0.490500, -0.573090, -0.732090, 0.111130, -1.011420, 1.183700,
     & -1.359940, 0.042880, 0.245940, -7.359460, -2.471070, 2.869400,
     & 1.393630, 1.475630, 0.052920, 2.363050, -0.630310, 0.784510,
     & 0.080450, -1.556670, 3.036710, -0.246830, 1.342110, -2.557280,
     & -1.439050, 1.716660, -1.952580, 3.683570, -2.836200, -1.408180,
     & -2.738590, 1.414740, 2.254810, 1.439300, -1.500800, -2.618490,
     & 3.099220, 0.144180, 3.612330, 2.870400, 4.289680, 1.076450,
     & 1.949740, 1.422640, 2.085290, -1.838760, -1.217380, 1.849980,
     & 0.637750, -0.134590, 0.780880, 0.733770, -1.500660, -0.764310,
     & 0.965790, 0.213520, -1.226590, 2.624520, -2.722360, -4.375620,
     & 1.196350, -2.137990, -0.784010, -0.337610, 0.221030, 2.114560,
     & -2.111370, -3.685260, -1.316400, -1.638150, -0.994860, 1.199890,
     & 0.915980, 0.167210, -1.975810, -1.344380, -0.480390, 0.089340,
     & -1.573570, -0.940600, -1.274090, -3.162980, -0.279680, -0.620290,
     & -0.946260, -0.765540, -0.228510, -0.534660, -0.440330, -0.445870,
     & 1.089960, -2.600230, 0.508420, 0.094240, -1.063460, -0.918930,
     & 1.442370, 0.387010, -2.355270, -0.072310, -1.062500, -3.472840,
     & -1.708900, 0.654130, -1.922840, -2.390360, 2.689220, -0.716700,
     & -0.702630, 2.118010, -0.146670, 3.462780, -1.231190, 1.211620,
     & -0.544490, 2.579400, -1.384530, 1.060170, -1.596770, -0.006750,
     & 1.465050, -1.747820, 0.062980, -1.258590, 0.816260, -1.142790,
     & 2.624990, -1.090090, 1.425870, -0.409240, -2.337480, 0.011550,
     & -0.600720, -1.031450, -0.737500, 0.362150, -0.147170, 1.196000,
     & -2.056940, 0.060770, -0.280000, 1.907290, -0.595090, 0.702770,
     & 0.700420, -0.653210, -0.607790, 1.719040, -1.751860, 0.443390,
     & 0.352710, -1.154760, 0.141210, -1.050120, -0.793500, -0.024520,
     & -0.786010, -1.490330, 0.771440, -0.119810, -1.184390, 1.624830,
     & 1.362790, -0.221340, 0.892770, 0.043190, -1.461150, -2.168320,
     & 0.373830, -0.339280, -1.043220, -0.355750, -1.099100, 0.592100,
     & -1.451660, -0.420940, 0.873720, 0.420750, -0.601490, -2.349370,
     & 1.307520, 1.003470, 0.553840, -0.536990, -0.438680, -1.464230,
     & -0.603400, 0.149510, 1.462400, 0.289510, 0.949430, 0.311450,
     & 1.580490, 1.625860, -2.566460, 1.435620, -0.142450, 2.009980,
     & 0.920580, -0.256410, -0.231990, -1.430710, 3.118170, -1.303350,
     & -2.624300, -6.100800, -1.469400, -1.638780, -1.396870, -0.425090,
     & -2.193200, -0.702790, -0.244510, -1.051830, -0.733580, -2.803760,
     & -1.787830, -2.016390, -0.461130, -3.393740, 0.654430, 0.052770,
     & -1.732990, -0.327990, -0.599650, 1.466280, 1.585210, 1.192150,
     & -2.130760, 0.245470, 1.310000, 1.184920, 1.522840, -0.650280,
     & -3.269020, -2.109520, -0.985810, 0.586760, 0.871500, 2.739990,
     & 0.132300, 0.133480, 0.334930, 0.105310, -0.597890, -0.655630,
     & 1.499010, -1.630160, -1.204260, -1.353620, -1.840910, -0.144130,
     & -1.579860, -3.070440, 1.212890, 0.689530, -2.295200, -0.550140,
     & 0.290600, -0.776930, 0.941040, -1.046070, 0.283290, 1.532490,
     & -0.114130, -1.222080, 0.776640, 1.792660, -0.640630, -1.723980,
     & 0.074320, -0.175180, -0.689360, -0.396310, -0.183280, -0.085000,
     & -1.050500, -0.331300, 2.810940, -0.097740, -0.806060, -2.171080,
     & 0.433700, -1.331940, -0.399320, 0.091980, -1.451070, 0.861120,
     & 0.398570, -1.205830, 0.043340, -0.945620, 0.432650, -0.380010,
     & 1.175250, -0.113490, -2.276660, 1.244180, -0.191420, -3.869600,
     & 0.719360, 3.981130, -1.078790, -0.739440, -1.668800, -0.633640,
     & 0.561060, -1.506240, -0.239070, 0.043140, -2.453400, 0.261130,
     & -0.574560, 0.638180, 1.063390, 4.110330, 1.838410, 0.157590,
     & -0.299010, -0.560600, -1.032070, 0.618400, 0.339830, -0.760880,
     & -0.183260, 0.444250, -0.441560, -1.371670, -0.163080, -0.026370,
     & -1.304430, -1.217400, -0.752150, 0.383090, -0.466480, 1.605900,
     & 0.207910, 1.311410, 0.436870, 3.213540, -1.407710, -1.007650,
     & -1.635680, 0.011490, -1.256740, 3.347410, -0.235910, -1.143990,
     & -0.166450, -0.371730, -0.547200, 0.186440, 1.062530, -1.441170,
     & 3.756010, 1.029150, -1.150930, -1.295950, -1.414070, 1.070070,
     & -2.922360, -0.837040, -1.091540, 0.325870, 0.888080, -1.252600,
     & -0.325530, 0.725200, -0.293100, -4.484860, 0.154320, -1.723930,
     & -0.133280, -0.338470, -1.273860, -0.343520, -0.545430, -1.189790,
     & -2.556310, -0.214690, 1.628520, 0.824390, -2.601300, -1.010070,
     & -0.459770, 0.290290, 0.968620, -0.559780, -0.523990, 0.137110,
     & -0.655900, 0.069160, 0.664980, -0.119690, 1.834100, -1.146610,
     & -0.274080, 0.829060, -0.942580, -1.225250, -1.931240, -0.441840,
     & -2.120140, 0.342500, -0.035530, -1.491790, 1.136440, -2.563610,
     & -0.888280, 0.764660, 0.198450, -2.185910, 0.726160, 0.403980,
     & 1.164020, -0.388240, 0.522370, 0.413320, 1.832820, -0.266910,
     & -0.330380, 0.607710, -0.030930, -2.199820, -0.478790, -0.783570,
     & 0.040140, 0.539480, 1.015960, 0.327810, 0.085470, 0.305110,
     & 0.582330, -0.824380, -0.532400, -0.230370, 0.250590, 3.305170,
     & -1.102230, -1.288930, -0.631540, 2.483960, -0.437140, -0.623380,
     & 0.006040, -0.230610, 0.588450, 1.916570, -0.787350, -0.082280,
     & 2.357910, -0.627600, -2.495580, 1.310110, 0.883010, 2.139360,
     & -0.077110, -2.896790, 0.830560, -0.533110, 0.293160, -1.997210,
     & -1.333450, -0.797920, -0.300980, 1.138150, -0.468540, -0.664100,
     & 0.284220, -1.150600, -1.616010, 1.030710, 1.318200, 3.568160,
     & -1.377830, 4.133930, -1.666780, 0.353730, -1.402170, -0.914020,
     & 1.340230, 1.260030, 2.491930, 2.496970, 1.442070, -0.004500,
     & 2.211540, -1.033570, 2.815780, -1.692040, -0.505470, -1.344490,
     & -0.192900, 1.104680, 0.055670, 0.758700, 4.961260, -3.086900,
     & 0.087670, 1.405230, -2.987720, 1.222560, -2.690560, -3.182700,
     & -0.564470, -0.155410, -0.702430, -2.304210, 0.156630, 0.934340,
     & 1.299180, 0.546910, 1.163450, 0.343380, -1.370410, 0.720760,
     & -0.019760, 1.840150, -3.241100, 1.431830, -0.350910, 2.002870,
     & 2.891120, -0.052690, -0.255190, 1.278600, -2.494700, -0.373370,
     & -0.653260, 2.900940, -1.872850, 1.394630, -0.728700, 2.380080,
     & 3.011940, 0.243230, 0.724030, -0.352810, -0.188880, 0.467910,
     & 3.402250, -1.706030, 0.198020, 4.110870, -1.777900, -1.257410,
     & -0.367480, 3.129170, -0.125700, 4.224100, -2.468510, 2.502310,
     & 0.646330, -0.382140, -0.451630, -1.878000, -1.952230, 0.418140,
     & -3.715530, -0.953160, 0.355610, -1.340950, 0.679550, 1.251200,
     & -3.043230, -0.922670, -1.732170, 2.448640, -2.313760, -0.953570,
     & -6.264260, -0.262180, -0.736420, 0.800110, -1.976000, -0.414180,
     & 3.546440, 0.911080, -1.274990, 1.738530, 2.397620, 0.693620,
     & -1.069590, -3.446200, -4.337180, -0.790370, -1.962350, -0.519690,
     & 4.837470, -0.240190, 1.375450, 1.435790, -5.445720, 1.088510,
     & -0.916940, -2.464700, 1.964050, -0.477140, -0.275400, -2.622530,
     & 0.407410, 1.139600, -1.403690, 1.457360, 0.708970, 1.274670,
     & -0.349480, 0.986400, -0.882800, -2.769840, -0.399740, -0.146310,
     & -1.280550, 0.484200, 0.990730, 0.655670, -0.740940, -0.434780,
     & 1.243860, -1.699140, 1.494050, 0.926310, -1.258230, 1.260290,
     & 2.356550, 2.431330, 2.633700, -0.644090, -0.674120, -0.218600,
     & -1.275910, 0.761920, -0.591430, -0.570710, -1.458520, 0.148500,
     & 0.647510, 1.206030, 0.160950, 0.405210, -2.334910, 2.004390,
     & 0.949870, 0.398510, 0.338480, 0.532030, -0.307850, -2.018030,
     & 0.922060, 0.280330, 0.631760, -3.890420, -0.020760, 0.324870,
     & 0.501990, 1.339310, 0.203370, -0.032570, -2.839700, -0.068960,
     & 0.520000, -1.125080, -2.802010, -0.377690, -2.467310, 1.052470,
     & -2.111270, 1.198760, -0.337030, 0.715230, -1.674740, -0.887470,
     & 0.841440, -1.350500, -2.176730, -3.023210, -0.683130, 1.310930,
     & -5.919050, 0.598240, -1.275250, 0.048010, -0.123770, -2.259890,
     & -0.653800, 1.069640, 1.491210, 1.700820, 1.185830, -3.201360,
     & 2.994470, 2.770810, 5.076880, -0.726940, -0.730140, -5.235070,
     & -2.713020, 2.569520, -3.287430, 2.543210, -1.602010, -3.374070,
     & 2.453960, 1.253000, 2.117330, 0.758460, -4.273650, -1.286040,
     & -0.568260, -0.108270, 4.673490, -0.441120, -1.522790, 0.748870,
     & 1.887960, -0.832420, 3.879580, 2.829320, 1.079790, 1.323780,
     & -1.834340, 0.407640, -0.327960, -1.639140, -0.546150, -0.110050,
     & -0.689400, -2.331590, 3.448630, 4.369000, -4.172920, 4.109570,
     & 0.921010, -1.232760, 0.514830, -2.184820, 2.697250, -1.622490,
     & -2.211230, -0.249870, -2.626450, 5.153210, 2.221170, -0.539840,
     & 0.867910, -2.696620, 0.630500, -0.171870, 0.219310, -0.665910,
     & -1.798760, -2.283690, -0.084130, -1.139820, -0.754050, -0.627240,
     & -3.110400, 2.015080, -0.549040, -0.783040, 0.598910, 1.056490,
     & -0.756910, 0.906560, 0.945270, -2.131280, 0.904980, 1.038620,
     & -2.011540, 0.541640, -1.090860, -0.256870, 2.363620, -1.386460,
     & 0.831020, -0.531410, -0.479160, 0.771930, -0.786870, 4.437780,
     & -4.345090, -0.969800, -4.104470, -0.735570, 0.913190, 0.314730,
     & 0.136380, 1.460910, 0.380970, -0.484640, 0.712500, 0.658870,
     & 0.561540, -0.833350, -1.342760, 0.674680, 1.574330, 0.032680,
     & -2.278500, 1.423550, 4.905420, 0.793110, -2.395260, 1.529950,
     & -0.562510, 5.183520, -1.416830, 0.302500, 1.036740, 1.869460,
     & 4.461950, -1.958610, -5.123910, -1.439850, 6.661410, 5.486860,
     & 1.570880, -0.536250, -0.542850, -0.176140, 1.085780, 0.373390,
     & -0.666280, 0.006530, 0.205560, 0.357190, -1.382360, 0.644080,
     & -0.150180, 1.055110, -1.458840, 0.545500, 1.440480, -0.044580,
     & -2.094560, 0.306000, -1.184200, 0.535140, 0.510290, 1.022600,
     & -2.243790, -1.286760, -0.427550, -0.740650, 0.023400, -0.912460,
     & -0.938940, -1.262420, -1.199040, -2.441790, -0.980520, 1.248910,
     & -1.158410, -0.226830, -0.803600, 1.018620, 0.806480, -0.455950,
     & 0.136350, 0.513330, -0.438470, -0.333230, 0.528660, 0.110760,
     & 0.467390, -2.025320, -0.047640, -0.615040, 2.872310, -0.737710,
     & -1.336580, 0.447290, 0.050090, -0.660100, -2.466610, -1.402820,
     & 1.466130, 0.339380, -0.468760, -1.204090, 0.219670, -1.335450,
     & -0.059270, -0.892470, -1.414740, -0.246810, -0.172680, 0.113680,
     & -1.049230, -1.187290, 1.261250, 0.343410, 0.472560, 0.607150,
     & 1.167960, -1.460990, -0.965390, 0.144890, 1.337440, -1.269580,
     & 0.446960, 0.147430, -0.496780, 0.029490, -0.710650, 0.670550,
     & -0.218500, 0.449940, -0.962000, 1.085480, -1.420390, -0.767640,
     & 0.240760, -0.838620, -2.406380, 0.220950, -0.172570, 0.967320,
     & -0.707470, -0.759100, -1.110870, 0.028360, 0.849310, 2.305590,
     & -1.511880, -1.281770, 1.247410, -0.724890, 0.633710, -0.395690,
     & -0.028890, -2.414100, 0.558060, -0.677060, 1.330880, -2.496150,
     & -0.826500, -0.122790, -3.749750, -1.931390, -0.446250, -1.000090,
     & 1.401040, -1.185030, -1.754830, 0.522000, -1.138840, 0.887700,
     & 0.762160, -0.285510, 0.509860, 0.187590, 0.279260, 1.212130,
     & 0.760400, -0.657960, 0.313910, -0.487580, 0.550850, 0.579390,
     & 1.407330, -1.415830, -0.413060, 1.086270, 0.952890, 0.587130,
     & -0.627270, -0.340870, -0.826970, 2.346670, 0.051620, 0.402810,
     & -1.069180, 1.416870, -0.901760, -0.630610, -1.489630, -2.061600,
     & 0.045180, -1.031640, -1.174050, 1.802250, 1.207910, -0.195020,
     & 2.009230, 0.161180, 1.350160, -2.540620, -0.748690, -2.559170,
     & -0.015560, 0.894430, -1.599010, 0.704890, -0.689930, 0.613510,
     & 0.719980, -2.634740, 1.563150, -1.504210, -2.273690, 5.731760,
     & 1.870650, -0.079710, -1.219310, 0.282570, -0.642430, 0.440050,
     & -1.280920, -0.832850, 1.400380, 0.184110, -2.544160, -1.065630,
     & -0.200400, -2.932360, 1.620230, -0.463530, -1.548630, -1.961610,
     & -1.142120, -1.653400, -1.697780, -0.465300, -1.300080, 1.358400,
     & 0.625770, -1.012220, -1.872760, -1.899690, -0.976260, -0.205550,
     & 0.161110, -0.051070, 0.786540, -1.127780, 0.615930, -1.122000,
     & -0.990910, -0.500760, -0.028780, 1.620460, -2.540140, 0.702920,
     & 2.025120, -3.723880, 0.117530, -1.521280, -0.427650, 0.909710,
     & -1.620170, -0.474310, 0.255350, 1.254870, 0.958400, 0.593780,
     & -1.176140, 0.980070, -0.186100, 2.817940, 2.097290, -0.468220,
     & -0.500170, 0.439470, -0.928640, -0.422650, 0.457410, 1.016990,
     & -0.328830, 0.185450, -2.136010, -3.639870, -0.701310, -0.654140,
     & -0.607590, 0.726100, 1.928400, -1.086690, -0.031900, -1.417100,
     & -1.058000, -0.073680, 0.995360, -0.737340, 0.639030, -0.449510,
     & 2.672190, -0.445390, -0.764370, -0.371860, -0.349130, 0.399710,
     & -2.089450, 0.056720, -0.070230, -0.194770, 0.963230, -1.623950,
     & -1.704910, -0.120030, -0.554060, 0.982660, -1.077440, 0.486320,
     & 1.212890, -0.504000, -0.212660, -2.208840, -0.315220, 2.583460,
     & -0.135290, -1.242500, 1.490130, -0.664620, 0.418190, -1.085920,
     & -2.585450, 0.471770, 0.633940, -0.675670, -0.873590, -0.263030,
     & -0.557380, -1.144550, 0.414480, -0.862870, -0.700100, 2.768120,
     & 0.655730, -2.341490, 1.789800, 0.330800, -0.265050, -0.507570,
     & 0.237570, -0.003020, 0.087130, -1.893940, -0.030280, -0.348550,
     & -0.245170, 1.328370, -1.256740, -0.333500, 2.508660, -1.237130,
     & -0.883000, -0.732730, -0.019240, -1.003270, -1.146350, -0.778290,
     & 0.388490, -0.282800, 0.085100, -0.001410, 0.717870, -1.150330,
     & -1.464560, -2.175270, 0.391990, 0.082050, 0.621750, -1.489170,
     & 0.596970, -0.104760, 1.857630, 1.512740, -5.549110, -0.815780,
     & -0.504890, 2.658410, -3.546870, -2.106560, -2.603250, -3.770510,
     & -2.217310, 0.297910, -0.167540, -1.706430, -2.409920, 0.190960,
     & 1.427050, 0.256360, -4.015920, 1.927690, 1.615530, 0.952860,
     & -2.789410, 0.478870, -0.936810, 2.412560, 0.104630, 1.007050,
     & -1.740260, -1.659040, 0.316260, 0.014080, 0.162770, -0.139820,
     & 0.579960, -0.266210, 3.086410, 2.952480, -2.206190, -1.281650,
     & 1.817530, -0.985080, 0.267570, 1.283220, 1.291200, -0.000110,
     & 0.431910, -0.838790, 2.628570, 0.771000, -0.936010, 0.151900,
     & 4.715290, -3.890950, -1.179940, -2.715920, -4.119790, -1.392640,
     & -0.838290, 1.631100, 3.263450, -1.353670, 1.004680, -2.161530,
     & -1.535230, 2.797570, 1.475710, 2.575060, -1.832230, 0.428100,
     & 1.857560, -1.470140, -0.746590, 0.010990, -0.833730, 0.313520,
     & -0.236940, -0.633090, -0.229190, -1.628210, 0.408930, -1.361800,
     & -0.141620, 2.222080, 0.150030, -0.351370, -1.895080, -1.561660,
     & 1.019660, 0.499390, -0.904910, -2.737260, 2.346720, -0.529360,
     & -0.277560, 0.483820, -2.042670, 0.065880, 1.274050, 0.665410,
     & -0.641690, -0.797910, 0.550570, 0.185200, -1.052530, 1.650700,
     & 0.549510, -0.750920, 0.242580, 0.244720, 0.063710, -1.526800,
     & 1.751670, -3.112500, -0.426070, 1.049290, -0.662690, -0.043040,
     & -1.071420, -2.306460, 0.517960, -1.668900, -1.529410, 1.168110,
     & 1.641770, -1.684480, -1.982960, 0.572270, 0.437400, -2.571270,
     & -0.410150, 0.551230, -0.235390, 1.338650, 2.041340, 1.140590,
     & 0.016290, -3.097860, -1.345290, 0.177480, -2.238580, -0.485950,
     & -1.852510, 0.711500, 1.292420, -2.795490, -1.951690, -1.012800,
     & -0.717880, -0.347650, -0.921300, -0.577300, 0.538850, 1.647650,
     & -2.475490, -0.173160, -0.388160, -0.541860, 0.068990, -2.185910,
     & 1.033070, 0.615890, 2.386180, 1.591230, -1.318300, -0.979250,
     & 0.219310, 1.231680, 0.994230, -1.376640, -0.412790, -0.868770,
     & -1.506880, -0.536560, -0.408480, 0.674240, -2.977900, -0.830800,
     & -0.351050, -0.038630, 0.855240, -1.563000, -0.431070, 1.424760,
     & 1.346780, 1.637230, -0.100730, 0.551340, 2.961370, 1.827270,
     & -1.552470, -0.436850, 0.406370, -1.988690, -0.563340, -1.210300,
     & 1.190740, 1.415260, -0.303450, -2.660310, -0.125170, 0.137890,
     & -0.719000, -2.662600, -1.962310, 0.734930, -1.259840, -1.077520,
     & -0.187430, 0.453680, -0.944120, -0.592790, -2.349150, -0.044200,
     & -0.254630, 0.118790, 0.305200, -1.318040, -3.307780, 1.854230,
     & 0.125400, -1.653360, 0.625370, 2.613980, 0.928570, -1.591440,
     & 0.231180, -0.855300, -0.552920, 0.789470, 1.820090, -0.287410,
     & 1.428840, -1.466160, 1.109990, -0.149120, 0.651160, -1.066250,
     & -0.817010, 1.724400, 0.425570, -0.129420, 0.607250, -0.239140,
     & -1.547230, -1.676410, 1.268300, 0.015650, -0.643050, -2.237690,
     & -1.215800, 1.882750, -1.131220, -1.208240, -0.741500, -1.397100,
     & -1.762870, 0.899920, -0.986580, -0.680200, 1.342560, 0.030140,
     & -3.250120, 1.262460, -1.229020, -0.785200, 2.573760, -0.199660,
     & 1.482560, 0.311840, -0.680820, 0.783600, -2.197480, -0.558650,
     & 0.170790, 0.307350, -1.299970, 0.397220, -1.350970, -0.249790,
     & 0.354570, -0.301780, 0.550660, 0.694300, -0.241530, -0.989520,
     & -0.940400, 0.249850, 2.904220, -0.730520, 0.074120, -0.451160,
     & 0.432550, -1.659390, -1.201600, -0.303950, -0.215290, -2.071590,
     & 0.988350, -0.766650, 0.859860, -1.072580, -0.777510, 0.330710,
     & 0.406790, -0.802190, 0.255170, -0.602040, 0.315380, -0.219320,
     & 0.410850, -2.345740, -0.382700, 2.790960, 2.240490, -0.622530,
     & 1.465360, -1.427030, -0.444690, -0.126800, -2.035510, -0.943470,
     & 0.429790, -0.035130, 0.022590, 0.968390, -0.911320, -1.040640,
     & -0.023710, -0.544710, -2.309620, 0.315660, 0.838710, -0.972010,
     & 0.664130, 0.145100, -1.612400, 0.517430, -0.886530, -1.978580,
     & -1.479160, 0.684590, 1.395010, -2.247530, 0.784540, -0.745580,
     & -1.001340, -0.519480, -1.562490, 0.851260, 0.177460, -0.447100,
     & -1.995880, 0.549030, -0.037020, -0.505060, -0.987690, -0.716050,
     & 1.071410, -0.259850, -0.491740, -1.062510, -0.495250, -0.639820,
     & 1.312850, -1.120120, 2.217780, -0.359970, -0.707460, 2.649410,
     & -0.754740, -0.303650, -0.492170, 0.628590, 0.147420, -0.924900,
     & 1.480540, 0.492370, -0.874050, -1.126700, 1.830740, -0.408420,
     & -1.821220, -2.113860, -2.430320, 1.211510, -2.092740, -1.084700,
     & 0.376270, -1.028860, -0.246870, 0.575840, 1.468480, -0.543180,
     & 0.204290, 0.631550, -0.069850, 1.243530, -0.525540, 0.542360,
     & -0.263990, -1.320740, 1.069370, 0.697270, -1.497080, 1.095180,
     & -0.720310, -0.173930, 1.360500, -1.848000, -0.170660, -1.434740,
     & -1.621900, -2.135680, -1.120250, 0.613810, 1.022340, -2.677710,
     & 4.356010, 0.520390, 0.386870, -0.210000, -1.027440, -0.400110,
     & 0.883130, -3.355760, 0.610960, -0.842810, -0.756950, 0.276100,
     & -1.469130, -2.804050, 1.314830, 0.619710, 0.381720, 0.242700,
     & -2.803990, -0.476180, 1.177600, -1.269240, 0.337770, -0.717730,
     & 1.346190, -0.804450, -3.487840, -0.412150, 0.552360, -0.767200,
     & -1.297620, 0.554010, -0.142490, -0.653750, 0.249180, 1.981010,
     & 1.021090, -0.635840, -2.584990, -1.510540, -3.008570, -0.784870,
     & -1.649780, 1.179780, 0.469180, 0.907980, -0.991800, -2.532270,
     & 0.534770, -0.640450, -2.363220, -1.374800, 2.212240, -0.305450,
     & -1.201400, 0.826420, 0.865750, -0.434260, -1.416990, -1.199920,
     & 0.253160, 3.436600, -0.840160, -0.449580, 0.172270, 1.102340,
     & -0.321740, 0.388470, -1.565740, -1.083730, -1.118780, -0.168600,
     & -1.487380, -0.429690, -1.176910, -0.515050, 0.803390, -0.035390,
     & -0.532320, 0.669700, -0.413600, 1.262150, -0.823840, -0.853910,
     & -0.439010, -2.994660, -0.365180, 0.475810, -1.956750, 2.017610,
     & 1.497580, -0.925750, -0.052890, -0.404410, -0.145420, 1.074960,
     & -2.529250, -0.519080, 0.550770, 2.741420, 0.464310, -0.331310,
     & 1.832820, -1.784080, -0.872450, -1.632610, -0.963470, -2.171880,
     & -1.323610, -1.315520, 0.581730, 0.397830, 1.503290, -3.033780,
     & -1.321880, -0.407560, 2.202520, 0.537030, -0.922610, 0.538210,
     & 1.043610, 0.274000, 1.572530, 1.201480, 0.568280, -0.853590,
     & 0.314680, -0.570600, 0.307060, -0.194280, -2.191250, -0.339520,
     & -0.505960, -3.142400, -0.805290, 0.945200, -0.287020, -0.237370,
     & -0.596020, 1.286050, 2.246940, -0.397630, -0.302460, -0.551890,
     & 1.278640, -0.828300, 1.921370, -2.154460, 0.441910, 0.259580,
     & -2.017720, -0.097900, -0.031080, -1.205530, 0.683100, 1.350520,
     & -0.763700, -0.604580, -3.138850, -0.629490, -0.202810, -0.150890,
     & -1.218330, 0.230290, -0.554800, -0.186560, -1.523410, -1.174050,
     & -0.693660, 0.006950, 0.744680, 0.473160, -2.439230, -1.316880,
     & 0.395600, -1.197770, -1.216720, -1.740580, 0.594170, -0.103730,
     & -1.157460, 0.960070, 0.410100, -0.370700, 0.607280, 1.144850,
     & 0.245320, 0.584940, -0.283840, -0.757900, 0.495270, -0.538960,
     & 0.575770, 0.418060, 1.383170, -1.881420, 0.003870, -0.323480,
     & -0.075110, -0.069480, -0.262060, -0.093890, 1.295870, -0.530860,
     & 0.292700, 1.463240, -1.706590, -0.968270, 1.404420, 0.344520,
     & -0.165340, 0.310780, -1.755400, -0.462500, -1.201630, -0.673810,
     & -0.437790, 3.413310, -2.408500, -1.506890, 1.458070, 0.551570,
     & -0.704990, -0.227860, 0.040850, 2.614870, -1.085820, 2.125580,
     & -5.210240, 0.988630, -0.793270, 1.307580, -2.060330, 0.219540,
     & -1.184770, -2.668700, 2.369080, 3.993380, 0.046410, 1.602170,
     & -0.420490, -1.231980, 0.374770, -0.241080, 3.712870, 1.714850,
     & -3.348610, 0.645510, 2.428140, -1.843110, -2.952530, -1.516250,
     & -4.108660, -1.467560, -0.796520, -0.926490, -4.830800, -4.353510,
     & 4.380930, 3.910000, 0.166400, -2.759350, 2.747150, -3.169450,
     & 0.039600, 0.165670, 0.268410, 1.416520, -1.958800, 0.141630,
     & 0.569280, 2.786150, -2.340150, -0.053990, -3.776990, 1.179450,
     & 3.407890, -0.990510, -2.136030, -1.212890, -2.020880, 0.141840,
     & -4.200640, -0.407770, -2.688780, 0.631810, 0.347710, -0.716890,
     & 0.612180, 0.153170, 0.461230, 0.197380, 1.036580, -0.722700,
     & -0.145790, 3.060420, -0.716360, 0.490620, -2.210270, -0.750700,
     & 0.878770, -4.828460, -1.781860, 0.848060, -0.072230, 0.371940,
     & -1.826180, 1.154530, 0.154180, -0.930000, 2.286440, -0.807910,
     & 0.449860, 3.494480, -1.025630, -1.080980, 1.829160, -0.968580,
     & 0.210600, 1.893460, -1.414320, -2.357480, -2.999090, 0.895490,
     & 0.438200, -2.242900, 0.311210, -1.116400, -0.229900, -2.332550,
     & -0.473740, 0.206180, -2.362750, 1.958950, 1.016570, -0.721640,
     & 0.375290, -0.620430, -0.116350, 0.697890, 1.642550, 1.039610,
     & 1.736460, 1.025410, 0.709980, -1.608200, -2.454430, -1.819650,
     & 0.210340, 1.945480, 1.080960, -0.091040, 0.392000, -0.414210,
     & 2.400290, -2.101250, -1.637400, 0.320820, 1.086980, -1.629780,
     & 0.048760, 0.332180, 0.476630, 2.540930, -1.328690, 1.054330,
     & 0.831400, -1.138150, 0.267890, 0.981430, -0.692850, -1.913930,
     & -0.026300, 0.733030, -0.255500, 0.228330, -0.176650, -1.226480,
     & 0.692420, 0.469680, -2.437080, -1.722220, 0.478180, 0.057900,
     & 1.488110, 0.500830, -0.215650, -0.076720, 0.456420, 1.472210,
     & 0.761280, -1.990080, -0.152860, -0.990370, -0.559400, 0.908730,
     & -1.074230, 0.238890, 2.417320, 1.762310, -0.365060, -1.006710,
     & -0.572420, -0.561030, -2.134640, -0.860900, -0.693110, -0.263400,
     & 0.535490, -1.269260, 0.682660, 1.446080, 2.364890, 0.347850,
     & -2.388750, -3.036940, 0.564270, -0.721820, 0.318270, -0.584970,
     & -0.162630, 1.396290, 0.171510, -0.756710, 0.471410, -0.581080,
     & 0.669470, -0.761710, -1.154110, -1.772300, 0.752910, 0.625700,
     & -1.230650, 0.886880, -0.763080, -1.059210, -1.680400, 0.035660,
     & 1.352620, 0.682620, -0.481360, 2.089070, 1.681520, 0.724470,
     & -1.485330, 0.903950, -1.121840, -1.326070, -0.032740, -1.087450,
     & -2.713830, -1.146240, -1.718970, -0.676120, -1.006060, 1.064750,
     & 0.283780, -1.719110, 0.458410, 0.059350, -0.472270, -1.727630,
     & 1.709940, -0.599240, -0.899800, -0.115740, 0.515100, 2.664310,
     & -0.168210, 1.383320, 0.935030, -1.138510, -0.756220, 0.170930,
     & -1.314600, -0.585820, 0.697960, 1.077860, 0.325420, -0.168990,
     & -0.357420, -0.700910, -0.473820, -0.767690, -0.855960, -1.113700,
     & 1.825830, -0.033610, 0.834400, -0.933620, 0.024030, 0.485670,
     & -0.927820, -2.650400, -1.310840, -1.014710, 0.662210, -0.209790,
     & -0.471710, -1.225430, 0.406580, -1.379050, -0.273770, 0.239050,
     & 0.238180, -0.327130, -2.446480, 0.514600, -1.214090, 0.621090,
     & -0.519020, -0.924060, 0.738930, 1.732650, -1.636300, -1.441030,
     & 1.450670, -0.420670, -0.806670, 0.673540, -1.497100, 1.416130,
     & -1.529890, -0.909940, -1.149210, -2.100430, -1.834040, -1.230570,
     & -0.438530, 1.276080, -0.802910, 0.122710, 0.726030, -1.380380,
     & 0.239720, -1.339460, -0.767520, -1.190070, -1.250630, -1.950630,
     & -1.053170, -0.632250, -0.170810, 1.145090, 2.206420, 0.790510,
     & 0.156530, 0.339660, 0.934960, 0.059150, -1.189340, -1.464180,
     & 0.968810, -1.609110, 0.098810, -0.565730, 1.011560, 2.080540,
     & -0.695200, 2.173450, 1.080780, -0.880580, 0.003360, -0.579910,
     & -2.265070, -3.076140, -0.058170, 2.201830, -0.571850, -0.727550,
     & -4.768360, 1.920330, -0.916570, 2.729830, -2.250900, 5.481940,
     & 0.279990, -0.635340, 0.409010, -1.193900, -0.966870, 0.015810,
     & -0.255000, -1.462170, 3.009460, -1.479490, -0.216100, 1.789210,
     & 1.688430, -0.378230, 1.719870, -0.323820, 0.341560, 0.678140,
     & -0.986420, -6.718750, 1.094720, -0.046190, -4.265380, -2.120970,
     & -0.204200, -1.051110, -0.993070, 0.392450, 2.618000, 0.626760,
     & -4.797780, -3.384800, -0.582290, 1.513660, -3.318680, 0.159590,
     & 1.240400, 1.725540, -2.987870, -0.422880, 0.339010, -1.123830,
     & 1.832740, 0.016220, -3.088160, 1.012640, -0.645080, -1.071960,
     & -3.449980, 2.430700, 1.065540, -0.821400, 1.256180, 0.131610,
     & 4.879070, 5.196560, -1.926930, 0.265340, -1.812660, -0.622780,
     & -0.829740, -0.691040, -0.976440, 0.223730, -0.326550, 1.771850,
     & 0.944380, 0.386600, 0.698230, -1.260790, -0.450020, -1.284200,
     & -3.228250, 1.897470, 0.445620, -0.464110, -1.028520, -2.601360,
     & -2.259440, -0.849720, 0.033390, -0.378750, 0.120610, -1.079590,
     & -0.076790, 0.173740, -4.883490, -0.511760, -1.121630, -2.012150,
     & -0.188540, 1.256640, 2.423620, 2.646010, -0.008740, 1.621470,
     & -1.131820, 0.829520, -1.110250, 0.626520, 0.936240, 0.165720,
     & -0.953180, -1.779150, -2.727210, -2.210330, -0.986040, -1.621300,
     & -1.446310, 0.992890, -1.370680, -3.701250, -0.193110, -1.329750,
     & 3.239690, -4.731040, 4.616250, -2.058160, -0.070960, 0.148830,
     & 2.265800, -1.222020, -0.066200, 0.661640, 0.013040, -0.524770,
     & 2.191370, -0.398780, -3.144220, 3.669730, -0.561980, 3.144430,
     & -1.878420, 0.192040, 1.202360, -2.097350, 3.073040, 0.572110,
     & 4.614930, -0.337580, -1.103710, 1.836600, 3.748010, 4.147230,
     & -6.793980, -5.868390, -0.320760, 1.477320, 1.209490, -0.425930,
     & 3.402420, -0.549440, -1.593570, 2.038210, -5.319000, 1.933380,
     & -2.470380, 1.632500, -0.664760, -0.652380, 3.282480, -1.013920,
     & 1.945430, 0.573470, 0.425470, -2.531710, -0.018410, -0.518210,
     & 1.270760, -1.884760, -0.477640, 3.994920, 0.888090, -2.340580,
     & 2.182150, 3.374390, -0.237150, -4.011940, 0.151720, 1.325270,
     & -2.525460, 3.662410, -4.020890, 1.134460, 1.008090, -1.082810,
     & -0.900010, -0.954770, 1.138310, -1.138990, 1.462260, 2.092510,
     & 1.557440, 3.665580, -0.016310, 0.255770, 4.141220, 1.337930,
     & -1.602730, -1.721360, -0.700410, -1.778160, 0.832540, -5.942520,
     & 1.409370, -0.534750, 0.102220, 1.287480, 3.445190, 1.340750,
     & -1.413250, 1.930560, -4.386740, 3.080150, 1.090060, -3.981310,
     & -3.122290, 4.504520, 1.592090, -0.010650, -0.275570, 1.421020,
     & -0.101230, 1.452760, 5.817790, -0.350600, 0.902130, -0.063470,
     & -1.783950, -4.471570, 2.465180, 2.176120, -2.911290, -1.302190,
     & -3.744590, -3.759310, -1.478800, 1.767580, 1.035560, -0.656920,
     & 0.207580, 1.535150, -0.707790, -4.806630, 0.530700, 3.051330,
     & -1.566110, -1.059520, 3.125270, 1.204280, 1.303790, -0.219590,
     & -0.355450, -1.767070, -0.033190, -5.880200, 4.575990, -1.761940,
     & -1.236940, -1.329290, -0.247230, 1.029840, -1.455770, -2.401170,
     & -0.259410, -4.197820, -2.717650, -0.912560, -0.869080, -0.542790,
     & 1.096030, -1.081800, -0.930710, -1.231610, -0.066560, 1.543360,
     & 0.013830, -1.799340, -2.066190, 0.446260, -1.147010, 0.004450,
     & 2.598230, -0.042460, 0.887060, -1.170640, 0.773390, 0.234560,
     & -0.455650, 0.661370, -0.739380, -0.060080, 0.739680, -0.167550,
     & -0.488320, 0.725750, 0.248170, 0.714010, -1.460110, 3.065630,
     & -0.413980, 0.641420, -0.356330, 3.607080, -0.761090, -1.126850,
     & 1.050510, -0.341820, -0.296930, -2.027460, -2.061850, -0.748640,
     & -0.687920, 0.966840, -0.794010, 0.149780, 0.663870, -1.073450,
     & 1.245220, 0.978530, -0.926490, -1.947350, 0.106930, 1.479040,
     & -2.303320, -0.526670, 0.542050, -0.862550, 0.613280, -1.457730,
     & 1.489180, -0.882850, -1.000910, 0.778440, -1.748400, -0.030090,
     & 3.187700, 0.712510, -1.167590, -2.382390, -1.532150, -2.182690,
     & -0.612150, -1.069210, 0.489130, -0.772870, 0.739050, 0.544850,
     & -1.131780, -1.888990, 0.440880, -2.230780, 1.990700, 0.725630,
     & 4.082190, -1.702330, -1.743570, 0.473910, -1.178860, 0.649080,
     & -0.672900, 1.025370, -0.037170, -1.697130, 1.660730, -1.379100,
     & 2.211410, -1.275100, 1.894760, -2.561430, -2.210510, -1.174530,
     & 0.165740, -3.084720, 2.644870, 0.021710, 0.176200, 0.731350,
     & -1.156480, 3.146980, -3.012460, -1.992730, -0.402890, 0.207290,
     & 1.573730, -3.756420, 3.912110, 1.072440, 2.459250, -2.367950,
     & -0.730820, -2.111280, -1.376020, -0.885470, -0.672410, 3.350530,
     & -0.234600, 0.186950, -1.001080, 1.463750, 2.353480, -1.611130,
     & -0.676080, -2.436190, -1.682300, 1.149310, 0.961160, -0.430730,
     & -0.300470, 1.111960, 1.017580, -0.417040, -1.861360, -1.644370,
     & -0.292110, 0.996490, -1.324250, -2.214640, -0.192170, -0.585400,
     & -1.288180, -0.686770, -0.257340, -0.618940, 1.332730, -1.194560,
     & -1.828590, 1.048980, 0.409740, 1.885600, 0.083150, -1.848150,
     & 0.733660, 0.876300, -0.351700, -0.941760, -0.861220, 0.334040,
     & -2.582910, 0.043100, 0.620050, -1.823810, 0.292640, -1.149900,
     & -0.300290, 0.858760, -0.853210, -3.187680, 1.881470, 2.381930,
     & 0.821550, 1.883570, 2.962040, 0.370620, 0.757860, 0.226430,
     & 0.903550, -2.870970, -1.863190, -0.493580, 1.606100, -0.438990,
     & -0.219280, 0.845930, -1.134680, 0.566140, 0.446740, 0.654030,
     & 2.015610, 0.520780, 1.668100, -1.604500, 0.716710, -1.814470,
     & 0.221920, -0.647540, -1.664530, 0.854570, 0.656040, -1.180320,
     & -2.715040, -0.888300, -0.878800, -1.165140, 0.373240, -0.397330,
     & 0.140920, 0.704110, 1.694200, 1.450870, -0.166570, 0.376570,
     & -3.094400, -1.234090, 0.720870, -1.293990, 1.342980, 0.592390,
     & 0.202590, 0.886650, 0.676810, 1.978440, -0.068170, 1.539540,
     & -0.212230, 1.512890, 1.724240, -2.571240, -0.170930, -0.329560,
     & 0.247430, 0.440670, -0.611520, -1.409050, 0.872830, 1.647280,
     & 2.408440, 0.071080, -0.541010, -0.698600, 1.153910, -1.379170,
     & -2.635540, 2.042670, 0.304280, 2.507800, -0.915810, 0.151810,
     & -0.929890, 2.109340, -0.239170, 1.663360, -0.936890, 0.332890,
     & 0.060200, 0.442530, 4.085490, -0.168850, 0.970090, -0.939140,
     & 3.040420, 0.811610, -4.241270, -0.081770, 0.934240, 0.381130,
     & 0.561530, 1.363010, 0.429510, 0.958270, 0.611940, -0.539910,
     & -1.258650, 0.223980, 1.103800, -0.501150, 0.755690, 1.074890,
     & 0.854500, -1.262400, -0.137850, -0.264990, -0.812120, -0.287500,
     & -0.344800, -0.373800, -2.469680, -0.931330, 1.859760, 0.120970,
     & -1.032430, 1.423920, -0.383900, -0.959860, -0.327300, 0.653920,
     & 0.353820, 0.912890, 0.034370, -2.171210, -1.228720, 0.077670,
     & 0.274120, 1.441020, 2.638810, -0.594920, -0.544480, -0.347600,
     & -0.682990, 2.091750, -0.705640, -1.881510, -0.671980, -0.591720,
     & 1.047620, 0.915500, 0.153570, -2.965860, -0.323890, 1.186980,
     & -0.563290, 0.649880, 0.013990, 0.995110, 0.148490, -1.212170,
     & -1.367730, 0.760350, -2.050570, 1.078700, 1.386540, 0.237390,
     & -0.399790, 0.249760, 0.471470, -1.560310, 0.504600, -0.626860,
     & -1.340410, 0.092480, 1.569690, 0.153760, -0.983070, 0.746100,
     & 0.584620, -0.895690, -0.872990, -1.144290, 1.540040, 1.544170,
     & -2.003010, 0.242350, -1.244670, 0.985790, 0.372240, 0.695210,
     & 1.212050, -0.730160, 0.224060, 0.693510, 1.293940, 1.351320,
     & 0.738160, -1.010560, -0.074800, 0.884730, 0.519260, -0.227980,
     & -0.256550, 0.472910, -2.496440, -2.354580, 0.483140, 1.497440,
     & -1.160000, 2.263030, 0.817690, -1.880980, 1.177220, -0.386220,
     & -0.742460, 0.172400, -1.284400, -0.817920, 0.941920, -1.198670,
     & -2.040260, -2.257590, 1.714160, -1.507270, -1.503400, 3.053650,
     & -1.494240, -1.150630, -0.419260, 0.047160, -0.745030, -0.824590,
     & 0.761910, -1.546220, 1.531190, 0.829800, -0.025090, 0.367130,
     & -0.126970, -0.774940, 1.653660, -1.788240, 1.563280, -0.225220,
     & -0.614350, 0.551290, 0.621360, 1.742120, 0.357360, 0.500270,
     & 0.646260, 0.470110, -2.517280, 0.423120, -2.270460, -1.971470,
     & 1.930980, -2.139600, 1.326020, 0.014210, -1.060670, 1.651630,
     & -1.130330, 0.180210, 2.767790, -2.326330, 0.561350, 2.092920,
     & 1.147430, 0.604670, -0.687830, -1.790010, 2.415210, -2.427530,
     & -5.285300, 1.102260, -6.069310, 0.096940, 1.751730, 2.893610,
     & -1.466970, 1.972510, -1.599640, 0.636670, 2.093110, -0.364060,
     & 1.772230, 0.811890, 0.607210, -1.620270, 0.288190, 3.283160,
     & -0.047430, 1.706580, 1.054890, -4.846840, 1.192970, -0.472790,
     & -1.115380, -0.823690, 0.323480, -2.143690, 3.271140, 0.697500,
     & -3.690160, -0.375740, -1.522440, -2.552040, -3.434440, 1.954630,
     & 2.942330, -0.610560, 0.313440, -1.316030, -1.190930, -0.105380,
     & -1.071220, -1.167160, 0.838240, -1.201800, 0.375070, 0.456120,
     & 1.450380, 0.306260, -0.154680, -0.828190, 2.303090, 2.578850,
     & -1.290660, -1.858740, -0.154130, -1.816070, -1.896610, -1.295610,
     & -0.597760, -0.498040, -1.210860, 1.388850, -1.262680, -1.298670,
     & -1.323410, -1.325920, -0.040700, -0.258930, -1.280530, 0.798710,
     & -0.646930, -0.925630, 0.148800, 0.324810, 0.481430, -1.309540,
     & -0.283790, 1.962180, -2.715850, 5.993970, -2.578670, -0.405530,
     & -1.444010, 0.242450, 0.541000, -0.357370, 2.151530, -0.160830,
     & -1.607750, -0.281540, -2.576120, -0.751830, -1.387350, -2.520260,
     & 0.441570, 0.488460, -1.089310, -2.475400, -1.724270, 2.663840,
     & -0.163510, 0.068400, -3.119580, 2.652740, -1.003890, -1.693820,
     & -2.876290, 0.541690, -0.934670, -1.595490, -0.874030, 0.621460,
     & 1.419490, -2.892270, -1.188920, 1.041940, 1.366610, 0.919480,
     & -2.147400, -0.896630, 0.956880, -0.799280, 1.184260, 2.770020,
     & 1.068970, -0.755970, 1.589890, 1.380150, 0.936410, -0.832700,
     & -0.749140, 1.350300, 1.117090, 1.573120, -1.369880, 1.179360,
     & -1.670820, 0.448620, -1.273750, 0.871740, -1.435840, 0.798250,
     & 0.718820, -1.457270, 1.190300, -2.516610, -1.455900, -1.034480,
     & -1.056840, -0.617230, 0.893870, -0.766770, -1.688680, -1.099420,
     & 1.377370, 1.422290, -2.864460, 1.169540, -1.591140, -1.549290,
     & -1.902060, 1.444500, -1.133390, -1.055170, -0.872010, -0.651620,
     & 1.308620, 0.915490, -0.676160, -0.674120, -1.597570, -1.055520,
     & -0.760980, 3.820660/

      data biases/
     & 1.000000, 0.998030, 0.916740, 0.646710, 0.270800, 0.817340,
     & 3.778020, 2.114540, -5.818530, 4.511360, -0.912760, -0.551570,
     & -1.610530, -4.823940, 1.037330, -6.021990, -2.496690, -3.098590,
     & 5.579090, -1.832410, -1.420390, -5.913510, 6.645310, 0.690280,
     & -0.340710, -8.364870, 0.625120, 3.579280, -0.331730, -1.322240,
     & 6.718180, 2.137420, -2.211500, 2.646730, 4.683310, 1.374000,
     & 1.024660, -0.668720, -3.015330, 2.004930, 0.116250, 3.091710,
     & -0.574330, -0.619210, 1.429560, -0.106660, -8.050640, 0.839570,
     & 2.073310, -3.313980, 1.486980, 5.733300, -2.651010, 3.561340,
     & 0.547880, 6.468410, -1.326170, 1.021230, -6.678260, -4.585070,
     & -4.641050, -5.426040, -2.811340, 0.088110, 1.889790, -3.501730,
     & -1.446060, 0.556670, 6.928210, -3.792560, 2.987830, 3.290070,
     & -0.354500, 1.799280, 1.641380, -5.727690, -0.751910, -1.128380,
     & -0.754740, -0.033740, -0.249380, -1.058920, -0.407410, -0.726580,
     & -1.582140, -0.588440, -0.213500, -0.469360, -0.332630, -0.003290,
     & -0.643180, -1.311420, 0.749550, 1.044340, -0.145100, -0.035560,
     & -1.295280, -1.042970, -0.388070, -0.150830, 0.102000, -0.961200,
     & -0.766950, 0.704720, -1.116780, -0.204220, -2.215500, 0.075060,
     & -0.897590, 0.585610, 0.382270, -1.472720, -0.318540, -0.973490,
     & -0.790370, -0.284450, -0.762370, -1.405730, 0.704770, -0.486880,
     & -0.458630, -1.765840, 0.919230, 0.245570, -0.502200, -1.013960,
     & -1.064770, -0.167270, -1.397560, 0.056930, -1.353340, -0.988740,
     & -0.630300, -0.982020, -1.601810, -0.826070, 1.493580, -1.008780,
     & -1.712420, -2.271940, -0.983920, 0.485510, -1.908360, -0.136270,
     & 1.088200, -1.315200, -0.349530/



      do member=0,(M-1)
         act(member)=0.0
      enddo

C input layer
      do member=0,(inpnodes-1)
         act(member)=in(member)
      enddo

C hidden layer 1
      do member=0,(hidnodes1-1)
         unit=member+inpnodes
         sum=0.0
         do source=0,(inpnodes-1)
            sum=sum+(act(source)*weights((member*inpnodes)+source))
         enddo
         if((sum+biases(unit)).LT.10000.0) THEN
            act(unit)=1/(1+exp(-sum-biases(unit)))
         else
            act(unit)=0.0
         endif
      enddo

C hidden layer 2
      do member=0,(hidnodes2-1)
         unit=member+inpnodes+hidnodes1
         sum=0.0
         do source=0,(hidnodes1-1)
            sum=sum+(act(source+inpnodes)*
     &         weights((member*hidnodes1)+(hidnodes1*inpnodes)+source))
         enddo
         if((sum+biases(unit)).LT.10000.0) THEN
            act(unit)=1/(1+exp(-sum-biases(unit)))
         else
            act(unit)=0.0
         endif
      enddo

C output layer
      do member=0,(outnodes-1)
         unit=member+inpnodes+hidnodes1+hidnodes2
         sum=0.0
         do source=0,(hidnodes2-1)
            sum=sum+(act(source+inpnodes+hidnodes1)*
     &         weights((inpnodes*hidnodes1)+(hidnodes1*hidnodes2)
     &                +(hidnodes2*member)+source))
         enddo
         if((sum+biases(unit)).LT.10000.0) THEN
            act(unit)=1/(1+exp(-sum-biases(unit)))
         else
            act(unit)=0.0
         endif
      enddo

C the result       
      do member=0,(outnodes-1)
         unit=member+inpnodes+hidnodes1+hidnodes2
         out(member)=act(unit)
      enddo

      RETURN
      end
c end of IMAZ_prediction for no Lv case

      subroutine detlmt(ah, alat, almt)
c
c this subroutine is designed to determine the local magnetic time
c from the hour and geographic latitude given
c
c inputs: ah --> time in UT
c       alat --> geographic latitude
c
c output: almt --> local magnetic time in hours
c
c Lee-Anne McKinnell, Hermanus Magnetic Observatory, May 2006
c
      real ah, alat, almt, Y

      if(alat.gt.60.0) THEN
         Y=21.433
      else
         Y=6.717
      endif
      almt=Y+(ah-24.0)
      if(almt.lt.0.0) THEN 
         almt=almt+24.0
      endif

      return
      end
c end of detlmt
 
      subroutine csza(ld, t, flat, flon, z)
c
c this subroutine is designed to determine the solar zenith angle
c from the day, hour and geographic coordinates given
c
c inputs: ld --> daynumber
c          t --> time in UT
c       flat --> geographic latitude
c       flon --> geographic longitude
c
c output: z --> zenith angle in degrees
c
c Lee-Anne McKinnell, Hermanus Magnetic Observatory, South Africa, May 2006
c
      real*8 ARGMAX, pi, UMR, humr, dumr
      real*8 wlon, td, te, dcl, dec(0:6), dc
      real*8 tf, eqt, et, fa, phi, a, b, cosx
      real*8 declin, p(0:6)
      real ld, t, flat, flon, z
      data p/ 0.0, 0.017203534, 0.034407068, 0.051610602, 0.068814136,
     &  0.0, 0.103221204/

      ARGMAX=88.0
      pi=atan(1.0)*4.0
      UMR=pi/180.0
      humr=pi/12.0
      dumr = pi / 182.5
	
c s/r is formulated in terms of WEST longitude.......................
      wlon = 360.0 - Elon

c time of equinox for 1980...........................................
      td = ld + (t + wlon/15.0) / 24.0
      te = td + 0.9369

c declination of the sun..............................................
      dec(0) = 23.256 * sin(p(1)*(te-82.242))
      dec(1) = 0.381 * sin(p(2)*(te-44.855))
      dec(2) = 0.167 * sin(p(3)*(te-23.355))
      dec(3) = 0.013 * sin(p(4)*(te+11.97))
      dec(4) = 0.011 * sin(p(6)*(te-10.41))
      dcl =  dec(0) + dec(1) + dec(2) - dec(3) + dec(4) + 0.339137
      declin = dcl
      dc = dcl * UMR

c the equation of time................................................
      tf = te - 0.5
      eqt = -7.38*sin(p(1)*(tf-4.0)) - 9.87*sin(p(2)*(tf+9.0)) +
     & 0.27*sin(p(3)*(tf-53.0)) - 0.2*cos(p(4)*(tf-17.0))
      et = eqt * UMR / 4.0
	
      fa = flat * UMR
      phi = humr * ( t - 12.0) + et

      a = sin(fa) * sin(dc)
      b = cos(fa) * cos(dc)
      cosx = a + b * cos(phi)
      if(abs(cosx).GT.1.0) THEN
 	if(cosx.GE.0.0) THEN cosx=1.0
	if(cosx.LT.0.0) cosx=-1.0
      endif
      z = acos(cosx) / UMR
      
      return
      end
c end of csza

      subroutine calcchap(z, ch)
c
c this subroutine is designed to determine the inverse chapman function
c from the solar zenith angle
c
c inputs:  z --> solar zenith angle in degrees
c
c output: ic --> inverse chapman function
c
c NB:: the file chapman.prn is required for this subroutine to work
c
c Lee-Anne McKinnell, Hermanus Magnetic Observatory, South Africa, May 2006
c
      real values(3,361), ch, m, c, z

c// getting chapman values
      do j=1,361
        do i=1,3
          values(i,j)=0.0
        enddo
      enddo

      open(12,file='chapman.prn',status='old',err=20)


      do j=1,361
         read(12,*) values(1,j), values(2,j), values(3,j)
      enddo

      close(12)

c using chi calculate chapman function
      do j=2,361
	if((z.gt.values(2,j-1)).AND.(z.lt.values(2,j))) then
           m=(values(3,j)-values(3,j-1))/(values(2,j)-values(2,j-1))
	   c=values(3,j)-(m*values(2,j))
	   ch = (m*z)+c
           goto 50
	endif
      enddo

20    print *,'cannot open chapman.prn file'

50    return
      end
c end of calcchap

      subroutine calcresabs(bd, bch, bsf, blra)
c
c this subroutine is designed to determine the rest absorption
c from the day number, solar flux value and the chapman function
c which has already been determined by calcchap
c
c inputs:  bd --> day number
c         bch --> chapman function value
c         bsf --> solar flux value
c
c output: blra --> rest absorption
c
c NB:: the file nighttruequiet.txt is required for this subroutine to work
c
c Lee-Anne McKinnell, Hermanus Magnetic Observatory, South Africa, May 2006
c
      integer j
      real bd, bch, bsf
      real Ln, Lo, m, c, n
      real Lo1, Lo2, n1, n2, F1, F2
      real nttqvalues(2,365), blra


c getting nighttime true quiet values
	do j=1,365
	  nttqvalues(1,j)=0.0
          nttqvalues(2,j)=0.0
	enddo
	blra=0.0

      open(12,file='nighttruequiet.txt',status='old',err=20)

      do j=1,365
         read(12,*) nttqvalues(1,j), nttqvalues(2,j)
      enddo

      close(12)

c calculating rest absorption
      do j=1,365
	if(int(bd).eq.j) then
	  Ln=nttqvalues(2,j)
	  goto 50
        endif
      enddo

20    print *,'cannot open nighttruequiet,txt file'

50	Lo1=0.1395
	Lo2=0.1909
	n1=0.5708
	n2=0.546
	F1=67.0
	F2=200.0
	
	m=(Lo2-Lo1)/(F2-F1)
        c=Lo2-(m*F2)
	Lo=(m*bSF)+c

	m=(n2-n1)/(F2-F1)
        c=n2-(m*F2)
	n=(m*bsf)+c

	blra=Ln+(Lo*(bch**-n))
             
      return
      end
c end of calcresab

      subroutine dntodm(dyr,dd,md)
c
c this subroutine is designed to determine the month and day
c from the daynumber
c
c inputs: dyr --> year
c          dd --> day number
c
c output: md --> md(1) = month, md(2) = day
c
c Lee-Anne McKinnell, Hermanus Magnetic Observatory, South Africa, May 2006
c
      real dyr, dd, md(2)

      if(mod(dyr,4).ne.0) then
        if(dd.le.31.0) then
          md(1)=1.0
          md(2)=dd
        elseif(dd.le.59.0) then
          md(1)=2.0
          md(2)=dd-31.0
        elseif(dd.le.90.0) then
          md(1)=3.0
          md(2)=dd-59.0
        elseif(dd.le.120.0) then
          md(1)=4.0
          md(2)=dd-90.0
        elseif(dd.le.151.0) then
          md(1)=5.0
          md(2)=dd-120.0
        elseif(dd.le.181.0) then
          md(1)=6.0
          md(2)=dd-151.0
        elseif(dd.le.212.0) then
          md(1)=7.0
          md(2)=dd-181.0
        elseif(dd.le.243.0) then
          md(1)=8.0
          md(2)=dd-212.0
        elseif(dd.le.273.0) then
          md(1)=9.0
          md(2)=dd-243.0
        elseif(dd.le.304.0) then
          md(1)=10.0
          md(2)=dd-273.0
        elseif(dd.le.334.0) then
          md(1)=11.0
          md(2)=dd-304.0
        elseif(dd.le.365.0) then
          md(1)=12.0
          md(2)=dd-334.0
        else
          goto 100
        endif
      else 
        if(dd.le.31.0) then
          md(1)=1.0
          md(2)=dd
        elseif(dd.le.60.0) then
          md(1)=2.0
          md(2)=dd-31.0
        elseif(dd.le.91.0) then
          md(1)=3.0
          md(2)=dd-60.0
        elseif(dd.le.121.0) then
          md(1)=4.0
          md(2)=dd-91.0
        elseif(dd.le.152.0) then
          md(1)=5.0
          md(2)=dd-121.0
        elseif(dd.le.182.0) then
          md(1)=6.0
          md(2)=dd-152.0
        elseif(dd.le.213.0) then
          md(1)=7.0
          md(2)=dd-182.0
        elseif(dd.le.244.0) then
          md(1)=8.0
          md(2)=dd-213.0
        elseif(dd.le.274.0) then
          md(1)=9.0
          md(2)=dd-244.0
        elseif(dd.le.305.0) then
          md(1)=10.0
          md(2)=dd-274.0
        elseif(dd.le.335.0) then
          md(1)=11.0
          md(2)=dd-305.0
        elseif(dd.le.366.0) then
          md(1)=12.0
          md(2)=dd-335.0
        else
          goto 100
        endif
      endif
    
100   return
      end
c end of dntodm

      subroutine ptoa(cyr,cd,cgl,cpr,calt)
c
c this subroutine is designed to determine the altitude
c at a certain pressure level
c
c inputs: cyr --> year
c          cd --> day number
c         cgl --> geographic latitude
c         cpr --> pressure level
c
c output: calt --> altitude
c
c NB:: this subroutines calls the subroutine dntodm
c
c Lee-Anne McKinnell, Hermanus Magnetic Observatory, South Africa, May 2006
c
        real cyr, cd, cgl, cpr, calt
        real pres70(111,49), pres60(111,13)
        real mndd(2), m, c, WN, t, d
        integer mm, i, W
        mndd(1)=0.0
        mndd(2)=0.0
	    call dntodm(cyr, cd, mndd)

        if(cgl.lt.67.0) then
          open(12,file='press_60deg.txt',status='old',err=20)
          do 15 N=1,111
             read(12,*) (pres60(N,M),M=1,13)
15        continue
          close(12)
          goto 75

75         mm=int(mndd(1))+1
           if((cpr.gt.pres60(1,mm)).OR.(cpr.lt.pres60(111,mm))) then
              calt=160.0
              goto 100
           endif

	   do i=2,111
             if((cpr.le.pres60(i-1,mm)).AND.(cpr.gt.pres60(i,mm))) then
	       goto 50
             endif
	   enddo
50         m=(pres60(i,1)-pres60(i-1,1))/(log10(pres60(i,mm))-
     &       log10(pres60(i-1,mm)))
	   c=pres60(i-1,1)-(m*log10(pres60(i-1,mm)))
	   calt=(m*log10(cpr))+c
	else 
          open(12,file='press_70deg.txt',status='old',err=20)
          do 10 N=1,111
             read(12,*) (pres70(N,M),M=1,49)
10        continue
          close(12)
          goto 70

20      print *,'can not open pressure file'

70         d=7.75
           if(mndd(1).eq.2.0) then d=7.0
	   if((mndd(1).eq.4.0).or.(mndd(1).eq.6.0).or.(mndd(1).eq.9.0)
     &        .or.(mndd(1).eq.11)) then d=7.5
           t=mndd(2)/d
           WN = ((mndd(1)-1.0)*4.0)+(int(t)+1)
           if(WN.gt.48.0) then WN=48.0
           W=int(WN)+1

           if((cpr.gt.pres70(1,W)).or.(cpr.lt.pres70(111,W))) then
              calt=160.0
              goto 100
	   endif

	   do i=2,111 
             if((cpr.le.pres70(i-1,W)).and.(cpr.gt.pres70(i,W))) then
	       goto 60
             endif
	   enddo
60	   m=(pres70(i,1)-pres70(i-1,1))/(log10(pres70(i,W))-
     &        log10(pres70(i-1,W)))
	   c=pres70(i-1,1)-(m*log10(pres70(i-1,W)))
	   calt=(m*log10(cpr))+c
       endif	

100     return
        end

c end of ptoa

      subroutine atop(eyr,ed,egl,ealt,epr)
c
c this subroutine is designed to determine the pressure
c at a certain altitude
c
c inputs: cyr --> year
c          cd --> day number
c         cgl --> geographic latitude
c         calt --> altitude
c
c output: cpr --> pressure
c
c NB:: this subroutines calls the subroutine dntodm
c
c Lee-Anne McKinnell, Hermanus Magnetic Observatory, South Africa, May 2006
c
        real eyr, ed, egl, epr, ealt
        real pres70(111,49), pres60(111,13)
        real mndd(2), m, WN, t, d
        integer mm, W

        mndd(1)=0.0
        mndd(2)=0.0
	    call dntodm(eyr, ed, mndd)

        a=int(ealt)-50        
        if(egl.lt.67.0) then
          open(12,file='press_60deg.txt',status='old',err=20)
          do 15 N=1,111
             read(12,*) (pres60(N,M),M=1,13)
15        continue
          close(12)
          goto 75

75         mm=int(mndd(1))+1
           epr=pres60(a,mm)

	else 
          open(12,file='press_70deg.txt',status='old',err=20)
          do 10 N=1,111
             read(12,*) (pres70(N,M),M=1,49)
10        continue
          close(12)
          goto 70

20        print *,'can not open pressure file'

70         d=7.75
           if(mndd(1).eq.2.0) then d=7.0
	   if((mndd(1).eq.4.0).or.(mndd(1).eq.6.0).or.(mndd(1).eq.9.0)
     &        .or.(mndd(1).eq.11)) then d=7.5
           t=mndd(2)/d
           WN = ((mndd(1)-1.0)*4.0)+(int(t)+1)
           if(WN.gt.48.0) then WN=48.0
           W=int(WN)+1
           epr=pres70(a,W)
        endif

100     return
        end
c end of atop

