c iridreg.for, version number can be found at the end of this comment.
c-----------------------------------------------------------------------
C
C This file contains the D-region models of Danilov, Rodevich, and
C Smirnova, Adv. Space Res. 15, #2, 165, 1995 (subroutine DRegion)
C and the model by Friedrich and Torkar, Adv. Space Res., to be 
C published, 2001 (subroutine F00 and block data statement).
c
c-----------------------------------------------------------------------
C Corrections/Version Numbers:
C-Version-mm/dd/yy-description (person reporting correction)
C 2001.01 05/07/01 initial version
C 2001.02 07/11/01 new version of F00 (as provided by K. Torkar)
C 2002.01 28/10/02 replace TAB/6 blanks, PARAMETER () (D. Simpson)
C 2007.00 05/18/07 Release of IRI-2007
c-----------------------------------------------------------------------
C
C
      Subroutine DRegion(z,it,f,vKp,f5SW,f6WA,elg)
c-----------------------------------------------------------------------
c D-region model of Danilov, Rodevich, and Smirnova, Adv. Space Res.  
C 15, #2, 165, 1995.
C
C Input:     z    - solar zenith angle in degrees
C            it   - season (month)
C            f    - F10.7 solar radio flux (daily)
C            vKp  - Kp magnetic index (3-hour)
C            f5SW - indicator for Stratospheric Warming (SW) conditions
C                   =0 no SW, =0.5 minor SW, =1 major SW
C            f6WA - indicator for Winter Anomaly (WA) conditions
C                   =0 no WA, =0.5 weak WA, =1 strong WA
C Criteria for SW and WA indicators:
C      SW minor:  Temperature increase at the 30 hPa level by 10 deg.
C      SA major:  The same but by 20 degrees.
C         Temperature data for each year are published  
C         in Beilage zur Berliner Wetterkarte (K. Labitzke et al.).
C      WA weak:   An increase of the absorption in the 2-2.8 MHz  
C                 range at short A3 paths by 15 dB
C      WA strong: The same by 30 dB.
C 
C       Only for month 12 to 2 (winter).
C
C Output:      elg(7)  alog10 of electron density [cm-3] at h=60,65,
C                  70,75,80,85, and 90km
c-----------------------------------------------------------------------
c            
cor   dimension h(7),A0(7),A1(7),A2(7),A3(7),A4(7),A5(7),A6(7),elg(7)
      dimension A0(7),A1(7),A2(7),A3(7),A4(7),A5(7),A6(7),elg(7)
      data A0/1.0,1.2,1.4,1.5,1.6,1.7,3.0/
      data A1/0.6,0.8,1.1,1.2,1.3,1.4,1.0/
      data A2/0.,0.,0.08,0.12,0.05,0.2,0./
      data A3/0.,0.,0.,0.,0.,0.,1./
      data A4/0.,0.,-0.30,0.10,0.20,0.30,0.15/
      data A5/0.,-0.10,-0.20,-0.25,-0.30,-.30,0./
      data A6/0.,0.1,0.3,0.6,1.,1.,0.7/
        pi=3.14159265
         if(z.le.45) then
           f1z=1.
         else
           if(z.lt.90) then
             f1z=1.1892*(cos(z*pi/180))**0.5
           else
             f1z=0.
           endif
         endif
         f4S=1.
       if((it.ge.5).and.(it.le.9))then
         f4S=0.
         f5SW=0
         f6WA=0
       endif
       if((it.eq.3).or.(it.eq.4).or.(it.eq.10).or.(it.eq.11))then
         f4S=0.5
         f5SW=0
         f6WA=0
       endif
         f2Kp=vKp
         if(vKp.gt.2) f2Kp=2.
         f3F=(f-60.)/300.*f1z
         do 1 i=1,7
         elg(i)=A0(i)+A1(i)*f1z+A2(i)*f2Kp+A3(i)*f3F+A4(i)*f4S
     *         +A5(i)*f5SW+A6(i)*f6WA
   1     continue
         end
C
C
      SUBROUTINE F00(HGT,GLAT1,IDAY,ZANG,F107T,EDENS,IERROR)
C---------------------------------------------------------------------
C     PURPOSE:
C     THIS SUBROUTINE COMPUTES "FIRI" ELECTRON DENSITIES
C
C     COMMON BLOCK REQUIRED:
C       REAL EDEN,TABHE,TABLA,TABMO,TABZA,TABFL
C       COMMON/FIRCOM/EDEN(81,5,12,12,3),
C      1              TABHE(81),TABLA(5),TABMO(12),TABZA(12),TABFL(3)
C
C       ARRAY EDEN contains LOG10(tabulated electron density,
C       ordered in (height,latitude,month,zenithangle,f107)
C       Quantity      Minimum  Maximum  Number of steps
C       Height        60       140      81
C       Latitude       0        60       5
C       Month          1        12      12
C       Zenith angle   0       180      12
C       F10.7         75       200       3
C
C     PARAMETERS:
C        HGT   height in km (input, REAL)
C        GLAT1 latitude in degrees, north is positive (input, REAL)
C        IDAY  day of the year (input, INTEGER)
C        ZANG  solar zenith angle in degrees (input, REAL)
C        F107T 10.7cm flux in Ja (input, REAL)
C        EDENS model electron density in m**-3 (output, REAL)
C        IERROR  Error code (INTEGER, output)
C
C       Error code
C         0         no error
C         1         model undefined for given combination of input
C                   parameters, output is set to zero
C         2         input parameters outside valid range, output is invalid
C         3         both error conditions detected, output is zero
C
C     USAGE
C        CALL F00(HGT,GLAT1,IDAY,ZANG,F107T,EDENS,IERROR)
C
C     SUBROUTINES AND FUNCTION SUBPROGRAMS REQUIRED
C        none
C
C     WRITTEN BY K. TORKAR, IWF GRAZ
C     Klaus.Torkar@oeaw.ac.at
C
C     LAST MODIFICATION:  06.07.2001
C
C     VERSION: 1.1
C
C     ------------------------------------------------------------------
C
      REAL HGT,GLAT1,ZANG,F107T,EDENS,F107L
      INTEGER IDAY,IERROR
C
      PARAMETER (NHGT=81)
      PARAMETER (NLAT=5)
      PARAMETER (NMON=12)
      PARAMETER (NZEN=12)
      PARAMETER (NF10=3)
C
      REAL EDEN,TABHE,TABLA,TABMO,TABZA,TABFL
      COMMON/FIRCOM/EDEN(81,5,12,12,3),
     1              TABHE(81),TABLA(5),TABMO(12),TABZA(12),TABFL(3)
C
      INTEGER MON,I,J,L,M,ISTEPJ,I1,I2,J1,J2,K1,K2,L1,L2,M1,M2
      INTEGER TABM(12)
      REAL EDENI(2,2,2,2),EDENIJ(2,2,2),EDENIJK(2,2),EDENIJKL(2)
      REAL STEPJ,DAY1,H1,DEG1,XHI1,FLX1,EL
C
      DATA TABM/0,31,59,90,120,151,181,212,243,273,304,334/
      DATA STEPJ,ISTEPJ/15.0,15/
C
C     INDICES:
C     I=HEIGHT, J=LATITUDE, K=MONTH, L=ZANG, M=F10.7
C
C     CHECK INPUT
C
      IERROR=0
      F107L=ALOG10(MIN(1000.0,MAX(1.0,F107T)))
      IF (HGT.LT.TABHE(1).OR.HGT.GT.TABHE(NHGT).OR.
     1    GLAT1.GT.TABLA(NLAT).OR.GLAT1.LT.-TABLA(NLAT).OR.
     2    IDAY.LT.1.OR.IDAY.GT.366.OR.
     3    ZANG.LT.TABZA(1).OR.ZANG.GT.TABZA(NZEN).OR.
     4    F107L.LT.TABFL(1).OR.F107L.GT.TABFL(NF10)) IERROR=2
C
C     assume height table is in 1 km steps from 60 to 140 km
      I=MIN0(NHGT-1,IFIX(HGT)-59)
      IF(I.LT.1)I=1
      H1=HGT-TABHE(I)
      I1=I
      I2=I+1
C
C     assume latitude table is in 15 deg steps from 0 to 60 deg
      J=MAX0(1,MIN0(NLAT-1,IFIX(ABS(GLAT1))/ISTEPJ))
      DEG1=(ABS(GLAT1)-TABLA(J))/STEPJ
      J1=J
      J2=J+1
C
C     assume month table is given for each month
      MON=12
      DO WHILE (TABM(MON).GT.IDAY)
        MON=MON-1
      END DO
      DAY1=FLOAT(IDAY-TABM(MON)-15)/30.0
      IF (DAY1.LT.0.0) MON=MON-1
      IF(MON.GE.1.AND.MON.LE.11)THEN
        K1=MON
        K2=MON+1
      ELSE
        K1=12
        K2=1
      END IF
C
C     assume zenith angle table has 12 entries between 0 and 180 deg
      DO L=2,NZEN-1
        IF(ZANG.LT.TABZA(L))GOTO 1
      END DO
      L=NZEN
1     L=L-1
      L1=L
      L2=L+1
      XHI1=(ZANG-TABZA(L1))/(TABZA(L2)-TABZA(L1))
C
C     assume solar activity table has 3 entries
      F107L=MIN(TABFL(3),MAX(TABFL(1),F107L))
      IF(F107L.LT.TABFL(NF10-1))THEN
        M1=1
        M2=2
      ELSE
        M1=2
        M2=3
      END IF
      FLX1=(F107L-TABFL(M1))/(TABFL(M2)-TABFL(M1))
C
C     ADJUST SOUTHERN LATITUDES TO NORTH AND MONTH+6
C
      IF(GLAT1.LT.0.0)THEN
        K1=K1+6
        IF(K1.GT.12)K1=K1-12
        K2=K2+6
        IF(K2.GT.12)K2=K2-12
      END IF
C
C     EDEN(hgt,lat,mon,zang,f107)
C          I   J   K   L    M
C
      DO M=M1,M2
C
        MH=M+1-M1
C       INTERPOLATE IN HEIGHT I
        DO L=L1,L2
          IF(EDEN(I1,J1,K1,L,M).EQ.0.0.OR.
     1       EDEN(I2,J1,K1,L,M).EQ.0.0.OR.
     2       EDEN(I1,J2,K1,L,M).EQ.0.0.OR.
     3       EDEN(I2,J2,K1,L,M).EQ.0.0.OR.
     4       EDEN(I1,J1,K2,L,M).EQ.0.0.OR.
     5       EDEN(I2,J1,K2,L,M).EQ.0.0.OR.
     6       EDEN(I1,J2,K2,L,M).EQ.0.0.OR.
     7       EDEN(I2,J2,K2,L,M).EQ.0.0)THEN
               EDENS=0.0
               IERROR=IERROR+1
               RETURN
          END IF
          IF(HGT.LT.TABHE(1))THEN
            EDENI(1,1,L+1-L1,MH)=EDEN(I1,J1,K1,L,M)
            EDENI(2,1,L+1-L1,MH)=EDEN(I1,J2,K1,L,M)
            EDENI(1,2,L+1-L1,MH)=EDEN(I1,J1,K2,L,M)
            EDENI(2,2,L+1-L1,MH)=EDEN(I1,J2,K2,L,M)
          ELSE IF(HGT.GT.TABHE(NHGT))THEN
            EDENI(1,1,L+1-L1,MH)=EDEN(I2,J1,K1,L,M)
            EDENI(2,1,L+1-L1,MH)=EDEN(I2,J2,K1,L,M)
            EDENI(1,2,L+1-L1,MH)=EDEN(I2,J1,K2,L,M)
            EDENI(2,2,L+1-L1,MH)=EDEN(I2,J2,K2,L,M)
          ELSE
            EDENI(1,1,L+1-L1,MH)=EDEN(I1,J1,K1,L,M)+
     1        H1*(EDEN(I2,J1,K1,L,M)-EDEN(I1,J1,K1,L,M))
            EDENI(2,1,L+1-L1,MH)=EDEN(I1,J2,K1,L,M)+
     1        H1*(EDEN(I2,J2,K1,L,M)-EDEN(I1,J2,K1,L,M))
            EDENI(1,2,L+1-L1,MH)=EDEN(I1,J1,K2,L,M)+
     1        H1*(EDEN(I2,J1,K2,L,M)-EDEN(I1,J1,K2,L,M))
            EDENI(2,2,L+1-L1,MH)=EDEN(I1,J2,K2,L,M)+
     1        H1*(EDEN(I2,J2,K2,L,M)-EDEN(I1,J2,K2,L,M))
          END IF
        END DO
C
C       INTERPOLATE IN LATITUDE J
        DO L=1,2
          IF(ABS(GLAT1).GT.TABLA(NLAT))THEN
            EDENIJ(1,L,MH)=EDENI(2,1,L,MH)
            EDENIJ(2,L,MH)=EDENI(2,2,L,MH)
          ELSE
            EDENIJ(1,L,MH)=EDENI(1,1,L,MH)+
     1        DEG1*(EDENI(2,1,L,MH)-EDENI(1,1,L,MH))
            EDENIJ(2,L,MH)=EDENI(1,2,L,MH)+
     1        DEG1*(EDENI(2,2,L,MH)-EDENI(1,2,L,MH))
          END IF
        END DO
C
C       INTERPOLATE IN MONTH K
        EDENIJK(1,MH)=EDENIJ(1,1,MH)+
     &     DAY1*(EDENIJ(2,1,MH)-EDENIJ(1,1,MH))
        EDENIJK(2,MH)=EDENIJ(1,2,MH)+
     &     DAY1*(EDENIJ(2,2,MH)-EDENIJ(1,2,MH))
C
C       INTERPOLATE IN ZENITH ANGLE L
        EDENIJKL(MH)=EDENIJK(1,MH)+XHI1*(EDENIJK(2,MH)-EDENIJK(1,MH))
      END DO
C
      EL=EDENIJKL(1)+FLX1*(EDENIJKL(2)-EDENIJKL(1))
C
      EDENS=10.**EL
C
      RETURN
      END
C
C
      BLOCK DATA
C
C     PURPOSE:
C     DEFINES TABLES OF FIRI(2000) IN
C     ARRAY EDEN(height,latitude,month,zenithangle,f107)
C     Quantity      Minimum  Maximum  Number of steps
C     Height        60       140      81
C     Latitude       0        60       5
C     Month          1        12      12
C     Zenith angle   0       180      12
C     F10.7         75       200       3
C
C     WRITTEN BY K. TORKAR, IWF GRAZ
C     Klaus.Torkar@oeaw.ac.at
C
C     LAST MODIFICATION:  01.09.2000
C
C     VERSION: 1.1
C
C     ------------------------------------------------------------------
C
      PARAMETER (NHGT=81)
      PARAMETER (NLAT=5)
      PARAMETER (NMON=12)
      PARAMETER (NZEN=12)
      PARAMETER (NF10=3)
C
      REAL EDEN,TABHE,TABLA,TABMO,TABZA,TABFL
      COMMON/FIRCOM/EDEN(81,5,12,12,3),
     1              TABHE(81),TABLA(5),TABMO(12),TABZA(12),TABFL(3)
C
      INTEGER I,L
C
C     altitudes in km
      DATA (TABHE(I),I=1,81)/
     * 60.,61.,62.,63.,64.,65.,66.,67.,68.,69.,
     * 70.,71.,72.,73.,74.,75.,76.,77.,78.,79.,
     * 80.,81.,82.,83.,84.,85.,86.,87.,88.,89.,
     * 90.,91.,92.,93.,94.,95.,96.,97.,98.,99.,
     * 100.,101.,102.,103.,104.,105.,106.,107.,108.,109.,
     * 110.,111.,112.,113.,114.,115.,116.,117.,118.,119.,
     * 120.,121.,122.,123.,124.,125.,126.,127.,128.,129.,
     * 130.,131.,132.,133.,134.,135.,136.,137.,138.,139.,140./
C
C     latitudes in degree
      DATA (TABLA(I),I=1,5)/
     * 0.,15.,30.,45.,60./
C
C     months
      DATA (TABMO(I),I=1,12)/
     * 1.,2.,3.,4.,5.,6.,7.,8.,9.,10.,11.,12./
C
C     solar zenith angles in degree
      DATA (TABZA(I),I=1,12)/
     * 0.,30.,45.,60.,75.,80.,85.,90.,95.,100.,130.,180./
C
C     log10(F10.7) for 75,130,200 Jy
      DATA (TABFL(I),I=1,3)/
     * 1.87506, 2.11394, 2.30103/
C
C     log10 electron densities, ordered as
C     I,J,K,L,M = Height,Latitude,Month,Zenithangle,F10.7
C     8 heights in each line
C     12 zenith angles in each DATA statement
C     innermost loop: J (5 latitudes)
C     next loop:      K (12 months)
C     next loop:      M (3 F10.7-fluxes)
C     outermost loop: I (11 groups of heights)
      DATA ((EDEN(I,1,1,L,1),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.290,8.352,8.393,8.435,8.473,
     * 0.000,0.000,0.000,8.501,8.501,8.509,8.516,8.528,
     * 0.000,0.000,0.000,8.394,8.354,8.354,8.352,8.375,
     * 0.000,0.000,0.000,0.000,0.000,0.000,8.114,8.117,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,6.415,6.500,6.590,6.728,6.808,6.906,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,1,L,1),I=1,8),L=1,NZEN)/
     * 7.712,7.829,7.900,7.965,8.053,8.137,8.236,8.324,
     * 7.758,7.834,7.866,7.902,7.961,8.037,8.130,8.220,
     * 7.742,7.792,7.803,7.799,7.817,7.888,7.970,8.072,
     * 7.725,7.764,7.770,7.732,7.715,7.722,7.733,7.818,
     * 7.694,7.724,7.734,7.714,7.710,7.693,7.669,7.646,
     * 7.675,7.701,7.713,7.705,7.710,7.703,7.686,7.658,
     * 7.646,7.667,7.687,7.705,7.732,7.742,7.732,7.701,
     * 0.000,0.000,0.000,0.000,6.086,6.286,6.446,6.617,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,1,L,1),I=1,8),L=1,NZEN)/
     * 7.563,7.632,7.684,7.765,7.866,7.965,8.093,8.217,
     * 7.185,7.253,7.305,7.400,7.512,7.651,7.819,7.985,
     * 7.167,7.210,7.243,7.299,7.375,7.511,7.670,7.848,
     * 7.152,7.182,7.207,7.230,7.270,7.346,7.442,7.607,
     * 7.121,7.140,7.167,7.210,7.262,7.310,7.367,7.427,
     * 7.104,7.117,7.149,7.204,7.267,7.322,7.382,7.433,
     * 7.076,7.083,7.121,7.201,7.286,7.358,7.428,7.474,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.199,6.446,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,1,L,1),I=1,8),L=1,NZEN)/
     * 7.305,7.340,7.389,7.496,7.618,7.744,7.915,8.100,
     * 6.690,6.752,6.830,6.987,7.158,7.352,7.588,7.827,
     * 6.668,6.702,6.756,6.885,7.029,7.220,7.455,7.705,
     * 6.649,6.664,6.707,6.796,6.897,7.045,7.233,7.480,
     * 6.620,6.624,6.669,6.770,6.877,6.980,7.114,7.255,
     * 6.603,6.601,6.649,6.763,6.882,6.993,7.127,7.255,
     * 6.575,6.567,6.623,6.763,6.902,7.029,7.170,7.294,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.029,6.348,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,1,L,1),I=1,8),L=1,NZEN)/
     * 6.925,6.933,6.987,7.137,7.301,7.473,7.705,7.978,
     * 6.190,6.248,6.350,6.568,6.796,7.037,7.342,7.667,
     * 6.167,6.190,6.265,6.462,6.667,6.916,7.220,7.555,
     * 6.146,6.146,6.207,6.360,6.520,6.735,7.004,7.340,
     * 6.117,6.104,6.167,6.326,6.487,6.643,6.845,7.079,
     * 6.100,6.083,6.146,6.320,6.493,6.654,6.857,7.072,
     * 6.072,6.049,6.121,6.320,6.515,6.689,6.900,7.107,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.210,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,2,L,1),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.201,8.272,8.320,8.373,8.420,
     * 0.000,0.000,0.000,8.442,8.442,8.459,8.476,8.497,
     * 0.000,0.000,0.000,8.344,8.301,8.307,8.310,8.342,
     * 0.000,0.000,0.000,0.000,0.000,8.152,8.079,8.090,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,6.382,6.470,6.562,6.702,6.784,6.884,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,2,L,1),I=1,8),L=1,NZEN)/
     * 7.561,7.697,7.786,7.866,7.969,8.064,8.170,8.270,
     * 7.757,7.834,7.867,7.899,7.954,8.033,8.127,8.220,
     * 7.743,7.799,7.814,7.805,7.818,7.887,7.965,8.068,
     * 7.726,7.772,7.783,7.746,7.727,7.732,7.740,7.820,
     * 7.695,7.732,7.747,7.728,7.724,7.711,7.689,7.667,
     * 7.676,7.708,7.726,7.721,7.728,7.723,7.706,7.677,
     * 7.647,7.675,7.701,7.719,7.747,7.760,7.751,7.719,
     * 0.000,0.000,0.000,0.000,6.104,6.303,6.459,6.626,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,2,L,1),I=1,8),L=1,NZEN)/
     * 7.391,7.490,7.572,7.673,7.792,7.910,8.053,8.188,
     * 7.253,7.324,7.380,7.468,7.573,7.705,7.863,8.021,
     * 7.236,7.286,7.324,7.375,7.442,7.567,7.713,7.881,
     * 7.220,7.258,7.290,7.314,7.350,7.415,7.497,7.647,
     * 7.190,7.220,7.255,7.294,7.342,7.386,7.435,7.483,
     * 7.170,7.196,7.236,7.288,7.346,7.398,7.452,7.493,
     * 7.143,7.161,7.210,7.288,7.367,7.435,7.496,7.534,
     * 0.000,0.000,0.000,0.000,0.000,6.029,6.255,6.493,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,2,L,1),I=1,8),L=1,NZEN)/
     * 7.290,7.346,7.413,7.525,7.650,7.775,7.939,8.111,
     * 6.833,6.905,6.990,7.137,7.294,7.468,7.682,7.901,
     * 6.813,6.856,6.915,7.037,7.164,7.340,7.551,7.780,
     * 6.794,6.821,6.871,6.954,7.045,7.176,7.338,7.560,
     * 6.765,6.781,6.834,6.930,7.029,7.121,7.233,7.356,
     * 6.747,6.758,6.814,6.923,7.033,7.130,7.248,7.358,
     * 6.716,6.723,6.787,6.923,7.053,7.167,7.292,7.398,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.137,6.441,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,2,L,1),I=1,8),L=1,NZEN)/
     * 7.000,7.025,7.086,7.230,7.380,7.538,7.749,7.992,
     * 6.401,6.470,6.580,6.782,6.990,7.212,7.484,7.777,
     * 6.377,6.415,6.498,6.681,6.867,7.090,7.364,7.665,
     * 6.358,6.375,6.444,6.589,6.734,6.923,7.158,7.456,
     * 6.328,6.334,6.407,6.559,6.708,6.849,7.025,7.230,
     * 0.000,0.000,6.386,6.553,6.713,6.860,7.041,7.225,
     * 0.000,0.000,6.360,6.553,6.734,6.895,7.083,7.265,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.336,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,3,L,1),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.104,8.176,8.236,8.299,8.354,
     * 0.000,0.000,0.000,8.413,8.415,8.435,8.455,8.480,
     * 0.000,0.000,0.000,0.000,0.000,8.283,8.288,8.324,
     * 0.000,0.000,0.000,0.000,0.000,0.000,8.064,8.072,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,6.364,6.453,6.547,6.688,6.772,6.872,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,3,L,1),I=1,8),L=1,NZEN)/
     * 7.431,7.583,7.686,7.777,7.889,7.991,8.104,8.207,
     * 7.839,7.924,7.962,7.992,8.041,8.114,8.193,8.274,
     * 7.825,7.889,7.910,7.899,7.906,7.967,8.033,8.124,
     * 7.808,7.863,7.880,7.841,7.819,7.816,7.812,7.879,
     * 7.777,7.823,7.844,7.824,7.817,7.796,7.763,7.731,
     * 7.758,7.799,7.824,7.817,7.820,7.809,7.780,7.743,
     * 7.728,7.766,7.798,7.816,7.839,7.846,7.827,7.787,
     * 0.000,0.000,0.000,6.053,6.196,6.386,6.531,6.688,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,3,L,1),I=1,8),L=1,NZEN)/
     * 7.267,7.405,7.517,7.632,7.761,7.886,8.029,8.167,
     * 7.435,7.521,7.587,7.662,7.750,7.863,7.993,8.124,
     * 7.420,7.486,7.535,7.575,7.627,7.730,7.846,7.986,
     * 7.403,7.459,7.507,7.522,7.548,7.592,7.645,7.763,
     * 7.371,7.418,7.468,7.505,7.547,7.574,7.600,7.623,
     * 7.354,7.396,7.447,7.498,7.550,7.587,7.616,7.634,
     * 7.324,7.362,7.422,7.496,7.567,7.621,7.660,7.676,
     * 0.000,0.000,0.000,0.000,0.000,6.193,6.394,6.609,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,3,L,1),I=1,8),L=1,NZEN)/
     * 7.258,7.362,7.470,7.597,7.728,7.858,8.017,8.173,
     * 7.079,7.164,7.255,7.373,7.497,7.644,7.819,8.004,
     * 7.061,7.127,7.204,7.294,7.391,7.528,7.689,7.878,
     * 7.045,7.100,7.170,7.243,7.314,7.403,7.512,7.678,
     * 7.013,7.061,7.137,7.225,7.310,7.382,7.464,7.547,
     * 6.997,7.037,7.117,7.220,7.314,7.394,7.480,7.556,
     * 6.965,7.004,7.090,7.220,7.334,7.431,7.525,7.597,
     * 0.000,0.000,0.000,0.000,0.000,6.045,6.305,6.577,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,3,L,1),I=1,8),L=1,NZEN)/
     * 7.152,7.228,7.326,7.470,7.609,7.744,7.917,8.107,
     * 6.698,6.788,6.905,7.079,7.246,7.430,7.650,7.890,
     * 6.682,6.751,6.850,6.999,7.143,7.320,7.533,7.778,
     * 6.664,6.720,6.814,6.940,7.057,7.193,7.362,7.589,
     * 6.633,6.680,6.777,6.921,7.049,7.161,7.299,7.444,
     * 6.616,6.657,6.757,6.915,7.053,7.176,7.316,7.452,
     * 6.585,6.623,6.732,6.915,7.076,7.212,7.358,7.490,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.199,6.529,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,4,L,1),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.083,8.152,8.215,8.279,8.338,
     * 0.000,0.000,0.000,8.403,8.405,8.428,8.452,8.479,
     * 0.000,0.000,0.000,8.310,8.267,8.279,8.286,8.324,
     * 0.000,0.000,0.000,0.000,0.000,0.000,8.057,8.072,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,6.364,6.455,6.550,6.693,6.777,6.879,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,4,L,1),I=1,8),L=1,NZEN)/
     * 7.400,7.563,7.676,7.769,7.881,7.984,8.093,8.193,
     * 7.973,8.068,8.111,8.137,8.176,8.236,8.299,8.362,
     * 7.958,8.033,8.061,8.045,8.045,8.093,8.140,8.212,
     * 7.941,8.004,8.029,7.990,7.962,7.944,7.920,7.969,
     * 7.910,7.966,7.995,7.971,7.957,7.925,7.876,7.825,
     * 7.890,7.943,7.974,7.963,7.960,7.938,7.892,7.838,
     * 7.861,7.908,7.948,7.962,7.979,7.974,7.937,7.879,
     * 0.000,0.000,6.061,6.204,6.340,6.520,6.646,6.787,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,4,L,1),I=1,8),L=1,NZEN)/
     * 7.199,7.369,7.512,7.635,7.767,7.894,8.025,8.155,
     * 7.686,7.791,7.868,7.925,7.988,8.072,8.161,8.260,
     * 7.673,7.763,7.829,7.853,7.883,7.950,8.021,8.124,
     * 7.656,7.739,7.803,7.812,7.823,7.836,7.846,7.914,
     * 7.623,7.697,7.765,7.795,7.823,7.830,7.823,7.810,
     * 7.605,7.674,7.744,7.788,7.827,7.843,7.839,7.822,
     * 7.575,7.640,7.719,7.787,7.846,7.880,7.885,7.866,
     * 0.000,0.000,0.000,6.033,6.207,6.420,6.588,6.768,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,4,L,1),I=1,8),L=1,NZEN)/
     * 7.146,7.314,7.471,7.612,7.748,7.884,8.029,8.176,
     * 7.427,7.542,7.650,7.734,7.813,7.915,8.025,8.158,
     * 7.413,7.519,7.621,7.685,7.741,7.822,7.908,8.037,
     * 7.396,7.496,7.598,7.658,7.708,7.750,7.792,7.877,
     * 7.365,7.456,7.561,7.643,7.711,7.754,7.790,7.818,
     * 7.346,7.433,7.540,7.636,7.715,7.767,7.806,7.832,
     * 7.314,7.398,7.515,7.637,7.735,7.803,7.851,7.874,
     * 0.000,0.000,0.000,0.000,6.097,6.342,6.549,6.771,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,4,L,1),I=1,8),L=1,NZEN)/
     * 7.170,7.326,7.489,7.639,7.771,7.903,8.049,8.204,
     * 7.146,7.279,7.420,7.545,7.652,7.776,7.913,8.076,
     * 7.137,7.258,7.393,7.507,7.598,7.702,7.816,7.973,
     * 7.121,7.236,7.371,7.484,7.573,7.647,7.728,7.845,
     * 7.086,7.196,7.336,7.471,7.575,7.655,7.732,7.806,
     * 7.068,7.173,7.316,7.465,7.580,7.667,7.747,7.818,
     * 7.037,7.140,7.290,7.465,7.602,7.705,7.794,7.860,
     * 0.000,0.000,0.000,0.000,0.000,6.260,6.509,6.775,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,5,L,1),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.167,8.230,8.286,8.344,8.396,
     * 0.000,0.000,0.000,8.418,8.422,8.444,8.465,8.491,
     * 0.000,0.000,0.000,8.318,8.276,8.290,8.301,8.340,
     * 0.000,0.000,0.000,8.258,8.185,8.130,8.064,8.083,
     * 0.000,0.000,0.000,8.241,8.179,8.104,8.009,7.920,
     * 0.000,0.000,0.000,8.230,8.182,8.117,8.025,7.931,
     * 0.000,0.000,0.000,8.228,8.199,8.152,8.068,7.974,
     * 0.000,0.000,6.382,6.473,6.569,6.713,6.797,6.898,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,5,L,1),I=1,8),L=1,NZEN)/
     * 0.000,0.000,7.782,7.874,7.982,8.076,8.170,8.255,
     * 0.000,0.000,8.262,8.276,8.305,8.352,8.394,8.442,
     * 0.000,0.000,8.212,8.188,8.176,8.210,8.236,8.294,
     * 0.000,0.000,0.000,0.000,0.000,0.000,8.021,8.053,
     * 8.045,8.111,8.146,8.114,8.093,8.049,7.981,7.915,
     * 8.025,8.090,8.127,8.107,8.093,8.061,7.997,7.926,
     * 7.997,8.053,8.100,8.107,8.114,8.097,8.041,7.969,
     * 0.000,6.097,6.207,6.348,6.476,6.645,6.755,6.881,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,5,L,1),I=1,8),L=1,NZEN)/
     * 7.233,7.425,7.580,7.702,7.827,7.946,8.061,8.173,
     * 7.922,8.045,8.134,8.167,8.204,8.262,8.314,8.382,
     * 7.910,8.021,8.100,8.107,8.114,8.149,8.176,8.246,
     * 7.893,7.995,8.076,8.072,8.068,8.057,8.025,8.053,
     * 7.860,7.956,8.037,8.061,8.072,8.057,8.017,7.973,
     * 7.841,7.933,8.017,8.053,8.076,8.068,8.033,7.986,
     * 7.810,7.899,7.993,8.049,8.090,8.104,8.076,8.029,
     * 0.000,0.000,6.100,6.288,6.446,6.638,6.774,6.922,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,5,L,1),I=1,8),L=1,NZEN)/
     * 7.093,7.294,7.479,7.614,7.737,7.866,7.988,8.117,
     * 7.752,7.898,8.025,8.083,8.127,8.185,8.233,8.316,
     * 7.741,7.881,8.009,8.053,8.083,8.114,8.134,8.204,
     * 7.724,7.859,7.986,8.033,8.064,8.068,8.057,8.079,
     * 7.692,7.820,7.951,8.021,8.068,8.083,8.072,8.057,
     * 7.672,7.797,7.930,8.017,8.072,8.097,8.086,8.068,
     * 7.641,7.763,7.903,8.013,8.093,8.134,8.134,8.111,
     * 0.000,0.000,6.004,6.238,6.425,6.642,6.803,6.976,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,5,L,1),I=1,8),L=1,NZEN)/
     * 7.009,7.230,7.446,7.598,7.721,7.838,7.946,8.076,
     * 7.559,7.737,7.911,8.004,8.068,8.124,8.167,8.248,
     * 7.548,7.723,7.896,7.989,8.049,8.086,8.114,8.176,
     * 7.531,7.703,7.877,7.979,8.045,8.076,8.090,8.117,
     * 7.498,7.664,7.841,7.967,8.049,8.093,8.114,8.124,
     * 7.476,7.638,7.820,7.960,8.053,8.107,8.130,8.137,
     * 7.446,7.605,7.794,7.960,8.076,8.146,8.176,8.182,
     * 0.000,0.000,0.000,6.164,6.378,6.618,6.801,7.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,6,L,1),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.272,8.332,8.380,8.428,8.471,
     * 0.000,0.000,0.000,8.450,8.456,8.477,8.498,8.520,
     * 0.000,0.000,0.000,8.344,8.307,8.322,8.334,8.371,
     * 0.000,0.000,0.000,8.274,8.199,8.149,8.090,8.111,
     * 0.000,0.000,0.000,8.253,8.190,8.114,8.017,7.927,
     * 0.000,0.000,0.000,8.246,8.193,8.127,8.033,7.938,
     * 0.000,0.000,0.000,8.243,8.210,8.161,8.079,7.979,
     * 0.000,0.000,6.412,6.502,6.598,6.740,6.823,6.923,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,6,L,1),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.013,8.121,8.201,8.279,8.350,
     * 0.000,0.000,0.000,8.394,8.420,8.458,8.490,8.525,
     * 0.000,0.000,0.000,8.301,8.286,8.316,8.332,8.378,
     * 0.000,0.000,0.000,8.243,8.196,8.161,8.111,8.137,
     * 0.000,0.000,0.000,8.225,8.193,8.140,8.061,7.982,
     * 0.000,0.000,8.241,8.217,8.196,8.152,8.076,7.993,
     * 0.000,0.000,8.215,8.215,8.212,8.188,8.121,8.037,
     * 6.111,6.217,6.336,6.471,6.594,6.757,6.857,6.970,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,6,L,1),I=1,8),L=1,NZEN)/
     * 7.332,7.537,7.701,7.818,7.933,8.041,8.137,8.233,
     * 8.104,8.236,8.332,8.358,8.382,8.425,8.455,8.501,
     * 8.090,8.212,8.299,8.297,8.292,8.314,8.318,8.367,
     * 8.072,8.188,8.276,8.265,8.248,8.220,8.167,8.176,
     * 8.041,8.149,8.241,8.250,8.250,8.223,8.158,8.097,
     * 8.021,8.127,8.220,8.243,8.253,8.236,8.176,8.107,
     * 7.991,8.093,8.196,8.243,8.270,8.270,8.220,8.149,
     * 6.000,6.143,6.310,6.490,6.639,6.820,6.933,7.061,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,6,L,1),I=1,8),L=1,NZEN)/
     * 7.146,7.364,7.555,7.677,7.780,7.892,7.985,8.100,
     * 8.000,8.167,8.310,8.352,8.375,8.400,8.405,8.452,
     * 7.989,8.152,8.297,8.328,8.340,8.342,8.318,8.348,
     * 7.973,8.130,8.274,8.314,8.330,8.312,8.262,8.246,
     * 7.939,8.093,8.241,8.303,8.336,8.328,8.281,8.233,
     * 7.919,8.068,8.220,8.297,8.338,8.340,8.299,8.248,
     * 7.888,8.033,8.193,8.294,8.356,8.377,8.344,8.292,
     * 0.000,6.072,6.279,6.502,6.674,6.870,6.993,7.134,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.004,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,6,L,1),I=1,8),L=1,NZEN)/
     * 6.994,7.241,7.471,7.611,7.713,7.809,7.880,7.985,
     * 7.850,8.064,8.262,8.346,8.391,8.405,8.391,8.413,
     * 7.840,8.053,8.250,8.338,8.384,8.389,8.364,8.365,
     * 7.822,8.033,8.233,8.330,8.384,8.389,8.360,8.338,
     * 7.787,7.991,8.196,8.318,8.387,8.410,8.386,8.356,
     * 7.768,7.968,8.176,8.312,8.393,8.422,8.403,8.369,
     * 7.736,7.935,8.149,8.312,8.413,8.459,8.449,8.415,
     * 0.000,0.000,6.210,6.476,6.666,6.874,7.009,7.161,
     * 0.000,0.000,0.000,0.000,0.000,6.017,6.053,6.093,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,7,L,1),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.297,8.362,8.405,8.452,8.491,
     * 0.000,0.000,0.000,8.479,8.486,8.502,8.519,8.535,
     * 0.000,0.000,0.000,8.367,8.332,8.346,8.356,8.386,
     * 0.000,0.000,0.000,8.292,8.217,8.167,8.107,8.127,
     * 0.000,0.000,0.000,8.272,8.207,8.130,8.029,7.937,
     * 0.000,0.000,0.000,8.262,8.210,8.143,8.045,7.946,
     * 0.000,0.000,0.000,8.260,8.228,8.179,8.090,7.988,
     * 0.000,0.000,6.428,6.517,6.612,6.753,6.836,6.934,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,7,L,1),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.086,8.193,8.270,8.338,8.403,
     * 0.000,0.000,0.000,8.461,8.489,8.522,8.547,8.574,
     * 0.000,0.000,0.000,8.360,8.346,8.373,8.389,8.428,
     * 0.000,0.000,0.000,8.292,8.243,8.210,8.158,8.182,
     * 0.000,0.000,0.000,8.272,8.238,8.179,8.090,8.009,
     * 0.000,0.000,0.000,8.265,8.241,8.190,8.107,8.021,
     * 0.000,0.000,8.265,8.262,8.255,8.225,8.152,8.064,
     * 0.000,6.274,6.393,6.526,6.647,6.807,6.901,7.013,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,7,L,1),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,7.925,8.037,8.137,8.225,8.310,
     * 0.000,0.000,0.000,8.449,8.477,8.520,8.545,8.584,
     * 0.000,0.000,0.000,8.380,8.375,8.400,8.405,8.449,
     * 0.000,0.000,0.000,8.342,8.322,8.292,8.236,8.246,
     * 8.117,8.228,8.320,8.326,8.320,8.286,8.212,8.143,
     * 8.100,8.204,8.299,8.320,8.324,8.297,8.228,8.155,
     * 8.068,8.170,8.270,8.316,8.340,8.332,8.274,8.199,
     * 6.086,6.236,6.405,6.584,6.730,6.907,7.013,7.134,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,7,L,1),I=1,8),L=1,NZEN)/
     * 7.299,7.522,7.719,7.834,7.931,8.033,8.111,8.212,
     * 8.093,8.265,8.412,8.450,8.468,8.494,8.496,8.538,
     * 8.083,8.248,8.391,8.422,8.428,8.430,8.401,8.430,
     * 8.064,8.228,8.373,8.407,8.415,8.393,8.334,8.314,
     * 8.029,8.188,8.338,8.396,8.422,8.407,8.350,8.294,
     * 8.013,8.164,8.318,8.389,8.425,8.420,8.365,8.307,
     * 7.980,8.130,8.290,8.387,8.441,8.456,8.413,8.352,
     * 0.000,6.179,6.393,6.615,6.781,6.971,7.083,7.217,
     * 0.000,0.000,0.000,0.000,0.000,6.037,6.053,6.086,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,7,L,1),I=1,8),L=1,NZEN)/
     * 7.220,7.477,7.714,7.847,7.941,8.021,8.068,8.155,
     * 7.961,8.182,8.384,8.462,8.502,8.507,8.479,8.491,
     * 7.951,8.170,8.373,8.453,8.493,8.487,8.447,8.439,
     * 7.933,8.149,8.354,8.447,8.493,8.489,8.444,8.410,
     * 7.899,8.107,8.318,8.433,8.497,8.508,8.470,8.427,
     * 7.879,8.086,8.297,8.427,8.501,8.521,8.486,8.441,
     * 7.846,8.053,8.272,8.428,8.521,8.559,8.531,8.486,
     * 0.000,6.083,6.334,6.597,6.781,6.982,7.100,7.243,
     * 0.000,0.000,0.000,0.000,6.017,6.127,6.146,6.179,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,8,L,1),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.225,8.299,8.350,8.403,8.449,
     * 0.000,0.000,0.000,8.462,8.470,8.486,8.502,8.521,
     * 0.000,0.000,0.000,8.356,8.318,8.330,8.338,8.371,
     * 0.000,0.000,0.000,8.288,8.215,8.161,8.097,8.114,
     * 0.000,0.000,0.000,8.267,8.207,8.130,8.029,7.938,
     * 0.000,0.000,0.000,8.258,8.210,8.143,8.045,7.948,
     * 0.000,0.000,0.000,8.255,8.225,8.176,8.090,7.990,
     * 0.000,0.000,6.408,6.498,6.592,6.733,6.815,6.914,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,8,L,1),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.072,8.176,8.255,8.330,8.400,
     * 0.000,0.000,0.000,8.418,8.449,8.484,8.513,8.545,
     * 0.000,0.000,0.000,8.318,8.307,8.336,8.356,8.400,
     * 0.000,0.000,0.000,8.253,8.207,8.176,8.127,8.152,
     * 0.000,0.000,8.270,8.233,8.201,8.146,8.064,7.989,
     * 0.000,0.000,8.250,8.225,8.204,8.158,8.083,8.000,
     * 0.000,8.176,8.223,8.223,8.223,8.196,8.127,8.041,
     * 6.121,6.225,6.338,6.471,6.592,6.754,6.854,6.970,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,8,L,1),I=1,8),L=1,NZEN)/
     * 7.529,7.732,7.898,8.009,8.124,8.220,8.305,8.387,
     * 8.121,8.255,8.352,8.391,8.428,8.479,8.515,8.560,
     * 8.107,8.223,8.307,8.314,8.318,8.352,8.371,8.423,
     * 8.090,8.199,8.281,8.270,8.253,8.233,8.190,8.212,
     * 8.057,8.158,8.243,8.253,8.250,8.220,8.158,8.097,
     * 8.037,8.134,8.223,8.246,8.253,8.233,8.176,8.107,
     * 8.009,8.100,8.196,8.243,8.272,8.270,8.220,8.152,
     * 6.029,6.167,6.330,6.509,6.656,6.838,6.953,7.083,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,8,L,1),I=1,8),L=1,NZEN)/
     * 7.542,7.754,7.942,8.057,8.155,8.250,8.328,8.420,
     * 8.000,8.158,8.297,8.352,8.387,8.436,8.465,8.525,
     * 7.988,8.140,8.274,8.312,8.328,8.352,8.354,8.408,
     * 7.971,8.121,8.253,8.290,8.303,8.294,8.258,8.262,
     * 7.938,8.079,8.220,8.279,8.307,8.303,8.262,8.217,
     * 7.919,8.057,8.199,8.272,8.312,8.316,8.279,8.230,
     * 7.886,8.021,8.170,8.270,8.328,8.352,8.324,8.274,
     * 0.000,6.090,6.297,6.524,6.699,6.900,7.029,7.176,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.033,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,8,L,1),I=1,8),L=1,NZEN)/
     * 7.658,7.888,8.100,8.220,8.303,8.377,8.427,8.507,
     * 7.863,8.064,8.250,8.334,8.382,8.420,8.433,8.489,
     * 7.852,8.049,8.236,8.316,8.360,8.378,8.369,8.405,
     * 7.835,8.029,8.217,8.307,8.354,8.362,8.336,8.330,
     * 7.802,7.990,8.182,8.297,8.360,8.380,8.358,8.330,
     * 7.779,7.966,8.161,8.288,8.362,8.391,8.373,8.344,
     * 7.748,7.933,8.137,8.290,8.382,8.428,8.420,8.389,
     * 0.000,0.000,6.228,6.498,6.695,6.911,7.053,7.217,
     * 0.000,0.000,0.000,0.000,0.000,6.025,6.068,6.124,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,9,L,1),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.130,8.207,8.265,8.326,8.382,
     * 0.000,0.000,0.000,8.430,8.435,8.456,8.479,8.502,
     * 0.000,0.000,0.000,8.328,8.288,8.303,8.314,8.350,
     * 0.000,0.000,0.000,8.267,8.196,8.140,8.076,8.093,
     * 0.000,0.000,0.000,8.246,8.188,8.114,8.017,7.928,
     * 0.000,0.000,0.000,8.238,8.190,8.127,8.033,7.939,
     * 0.000,0.000,0.000,8.236,8.210,8.161,8.079,7.981,
     * 0.000,0.000,6.386,6.476,6.572,6.713,6.795,6.896,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,9,L,1),I=1,8),L=1,NZEN)/
     * 0.000,0.000,7.927,8.017,8.121,8.204,8.290,8.367,
     * 0.000,0.000,8.258,8.279,8.316,8.365,8.413,8.461,
     * 0.000,8.167,8.199,8.182,8.179,8.217,8.253,8.312,
     * 8.068,8.140,8.167,8.121,8.083,8.061,8.025,8.064,
     * 8.037,8.100,8.130,8.100,8.079,8.037,7.971,7.907,
     * 8.017,8.076,8.111,8.093,8.083,8.049,7.988,7.918,
     * 7.987,8.041,8.083,8.090,8.097,8.083,8.033,7.961,
     * 0.000,6.090,6.199,6.336,6.465,6.636,6.750,6.879,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,9,L,1),I=1,8),L=1,NZEN)/
     * 7.629,7.810,7.957,8.061,8.170,8.265,8.356,8.444,
     * 7.908,8.029,8.117,8.170,8.225,8.297,8.364,8.436,
     * 7.893,7.996,8.068,8.090,8.111,8.167,8.217,8.297,
     * 7.876,7.970,8.041,8.041,8.037,8.037,8.025,8.079,
     * 7.844,7.930,8.004,8.025,8.037,8.025,7.990,7.953,
     * 7.825,7.907,7.985,8.017,8.041,8.037,8.004,7.964,
     * 7.795,7.874,7.960,8.017,8.061,8.076,8.049,8.004,
     * 0.000,0.000,6.093,6.281,6.441,6.638,6.780,6.936,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,9,L,1),I=1,8),L=1,NZEN)/
     * 7.775,7.943,8.093,8.204,8.301,8.389,8.473,8.558,
     * 7.733,7.869,7.990,8.072,8.140,8.228,8.310,8.407,
     * 7.719,7.844,7.956,8.013,8.053,8.121,8.182,8.283,
     * 7.702,7.821,7.932,7.977,8.004,8.029,8.037,8.100,
     * 7.671,7.782,7.898,7.963,8.004,8.021,8.017,8.009,
     * 7.652,7.760,7.877,7.956,8.009,8.033,8.033,8.021,
     * 7.622,7.725,7.849,7.955,8.029,8.072,8.079,8.064,
     * 0.000,0.000,6.013,6.250,6.442,6.667,6.838,7.021,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,9,L,1),I=1,8),L=1,NZEN)/
     * 7.948,8.100,8.248,8.354,8.435,8.504,8.571,8.648,
     * 7.565,7.723,7.880,7.997,8.086,8.188,8.283,8.403,
     * 7.551,7.700,7.849,7.948,8.017,8.097,8.173,8.292,
     * 7.534,7.678,7.827,7.919,7.977,8.021,8.053,8.134,
     * 7.502,7.638,7.791,7.905,7.980,8.021,8.045,8.064,
     * 7.483,7.616,7.770,7.899,7.984,8.037,8.061,8.076,
     * 7.450,7.581,7.745,7.900,8.004,8.072,8.107,8.117,
     * 0.000,0.000,0.000,6.204,6.427,6.676,6.875,7.086,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,10,L,1),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.114,8.188,8.248,8.310,8.365,
     * 0.000,0.000,0.000,8.413,8.418,8.444,8.468,8.494,
     * 0.000,0.000,0.000,8.314,8.276,8.292,8.303,8.342,
     * 0.000,0.000,0.000,8.253,8.185,8.130,8.064,8.086,
     * 0.000,0.000,0.000,8.233,8.176,8.104,8.009,7.922,
     * 0.000,0.000,0.000,8.225,8.179,8.114,8.025,7.932,
     * 0.000,0.000,0.000,8.225,8.199,8.155,8.072,7.976,
     * 0.000,0.000,6.375,6.467,6.562,6.705,6.789,6.890,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,10,L,1),I=1,8),L=1,NZEN)/
     * 7.664,7.817,7.919,7.997,8.093,8.182,8.272,8.356,
     * 7.949,8.041,8.086,8.114,8.158,8.225,8.292,8.362,
     * 7.934,8.004,8.029,8.017,8.017,8.076,8.130,8.212,
     * 7.918,7.978,8.000,7.957,7.929,7.919,7.902,7.962,
     * 7.885,7.938,7.964,7.940,7.927,7.896,7.849,7.803,
     * 7.867,7.915,7.943,7.932,7.929,7.908,7.865,7.813,
     * 7.836,7.881,7.918,7.929,7.945,7.943,7.910,7.855,
     * 0.000,0.000,6.033,6.179,6.318,6.500,6.631,6.775,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,10,L,1),I=1,8),L=1,NZEN)/
     * 7.707,7.846,7.957,8.049,8.152,8.243,8.342,8.433,
     * 7.652,7.755,7.831,7.899,7.975,8.072,8.173,8.276,
     * 7.637,7.719,7.777,7.812,7.852,7.937,8.025,8.140,
     * 7.621,7.693,7.749,7.757,7.769,7.796,7.822,7.911,
     * 7.589,7.653,7.713,7.740,7.765,7.773,7.772,7.766,
     * 7.571,7.630,7.694,7.732,7.769,7.786,7.787,7.776,
     * 7.540,7.597,7.667,7.732,7.787,7.823,7.834,7.819,
     * 0.000,0.000,0.000,6.004,6.182,6.398,6.574,6.760,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,10,L,1),I=1,8),L=1,NZEN)/
     * 7.808,7.924,8.033,8.134,8.230,8.312,8.410,8.508,
     * 7.410,7.521,7.626,7.737,7.844,7.968,8.104,8.248,
     * 7.394,7.483,7.571,7.652,7.729,7.846,7.973,8.124,
     * 7.378,7.455,7.537,7.591,7.639,7.707,7.779,7.913,
     * 7.342,7.413,7.501,7.574,7.633,7.677,7.715,7.754,
     * 7.324,7.391,7.481,7.567,7.637,7.689,7.732,7.763,
     * 7.297,7.358,7.455,7.567,7.657,7.724,7.775,7.803,
     * 0.000,0.000,0.000,0.000,6.114,6.367,6.589,6.816,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,10,L,1),I=1,8),L=1,NZEN)/
     * 7.784,7.877,7.980,8.093,8.193,8.283,8.394,8.525,
     * 7.173,7.303,7.446,7.612,7.762,7.918,8.093,8.276,
     * 7.152,7.255,7.375,7.519,7.643,7.799,7.971,8.164,
     * 7.130,7.217,7.326,7.438,7.526,7.642,7.771,7.956,
     * 7.100,7.179,7.290,7.413,7.509,7.585,7.665,7.753,
     * 7.083,7.155,7.270,7.407,7.513,7.597,7.679,7.754,
     * 7.053,7.121,7.243,7.407,7.534,7.631,7.721,7.792,
     * 0.000,0.000,0.000,0.000,6.041,6.328,6.591,6.863,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,11,L,1),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.190,8.255,8.307,8.362,8.410,
     * 0.000,0.000,0.000,8.438,8.441,8.461,8.481,8.505,
     * 0.000,0.000,0.000,8.334,8.294,8.307,8.316,8.352,
     * 0.000,0.000,0.000,8.270,8.196,8.140,8.079,8.093,
     * 0.000,0.000,0.000,8.253,8.190,8.114,8.017,7.927,
     * 0.000,0.000,0.000,8.243,8.193,8.127,8.033,7.938,
     * 0.000,0.000,0.000,8.241,8.210,8.161,8.079,7.980,
     * 0.000,0.000,6.389,6.479,6.575,6.716,6.800,6.900,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,11,L,1),I=1,8),L=1,NZEN)/
     * 7.744,7.870,7.947,8.013,8.100,8.179,8.272,8.358,
     * 7.836,7.919,7.956,7.987,8.037,8.114,8.199,8.281,
     * 7.820,7.879,7.896,7.886,7.897,7.964,8.037,8.134,
     * 7.803,7.852,7.864,7.822,7.797,7.799,7.800,7.879,
     * 7.772,7.812,7.828,7.805,7.795,7.773,7.741,7.710,
     * 7.754,7.788,7.808,7.797,7.799,7.786,7.757,7.720,
     * 7.724,7.754,7.781,7.794,7.814,7.821,7.802,7.763,
     * 0.000,0.000,0.000,6.045,6.193,6.384,6.533,6.692,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,11,L,1),I=1,8),L=1,NZEN)/
     * 7.714,7.808,7.880,7.960,8.057,8.143,8.250,8.352,
     * 7.425,7.508,7.573,7.657,7.757,7.873,8.009,8.143,
     * 7.408,7.467,7.511,7.559,7.620,7.733,7.862,8.009,
     * 7.391,7.438,7.476,7.490,7.516,7.571,7.635,7.767,
     * 7.360,7.396,7.438,7.470,7.508,7.534,7.562,7.589,
     * 7.340,7.373,7.417,7.462,7.511,7.545,7.576,7.595,
     * 7.312,7.340,7.391,7.461,7.530,7.582,7.622,7.638,
     * 0.000,0.000,0.000,0.000,0.000,6.182,6.393,6.609,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,11,L,1),I=1,8),L=1,NZEN)/
     * 7.642,7.708,7.777,7.872,7.973,8.072,8.201,8.340,
     * 7.097,7.188,7.286,7.427,7.572,7.726,7.909,8.097,
     * 7.072,7.134,7.201,7.316,7.438,7.595,7.780,7.977,
     * 7.053,7.093,7.149,7.220,7.297,7.415,7.558,7.753,
     * 7.021,7.053,7.111,7.193,7.272,7.342,7.428,7.521,
     * 7.004,7.029,7.090,7.188,7.276,7.356,7.442,7.521,
     * 6.976,6.995,7.064,7.188,7.297,7.391,7.486,7.560,
     * 0.000,0.000,0.000,0.000,0.000,6.072,6.338,6.615,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,11,L,1),I=1,8),L=1,NZEN)/
     * 7.387,7.435,7.509,7.641,7.772,7.905,8.083,8.288,
     * 6.755,6.858,6.991,7.190,7.386,7.581,7.818,8.064,
     * 6.719,6.787,6.890,7.076,7.255,7.458,7.699,7.957,
     * 6.695,6.732,6.811,6.951,7.086,7.267,7.483,7.746,
     * 6.664,6.689,6.769,6.907,7.033,7.149,7.292,7.462,
     * 6.646,6.667,6.748,6.901,7.037,7.158,7.301,7.444,
     * 6.618,6.633,6.723,6.901,7.061,7.193,7.344,7.480,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.243,6.587,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,12,L,1),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.281,8.342,8.382,8.427,8.467,
     * 0.000,0.000,0.000,8.479,8.479,8.489,8.498,8.513,
     * 0.000,0.000,0.000,8.375,8.332,8.334,8.332,8.360,
     * 0.000,0.000,0.000,8.310,8.233,8.167,8.093,8.104,
     * 0.000,0.000,0.000,8.290,8.225,8.143,8.037,7.941,
     * 0.000,0.000,0.000,8.281,8.228,8.155,8.057,7.953,
     * 0.000,0.000,0.000,8.279,8.246,8.193,8.100,7.996,
     * 0.000,0.000,6.408,6.496,6.587,6.724,6.805,6.904,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,12,L,1),I=1,8),L=1,NZEN)/
     * 7.777,7.888,7.953,8.013,8.093,8.173,8.267,8.352,
     * 7.772,7.849,7.880,7.912,7.966,8.041,8.134,8.223,
     * 7.757,7.809,7.820,7.811,7.823,7.892,7.970,8.072,
     * 7.740,7.781,7.788,7.747,7.725,7.729,7.737,7.820,
     * 7.708,7.741,7.752,7.728,7.720,7.702,7.678,7.654,
     * 7.690,7.718,7.731,7.721,7.723,7.715,7.694,7.665,
     * 7.661,7.683,7.705,7.719,7.743,7.752,7.741,7.708,
     * 0.000,0.000,0.000,0.000,6.104,6.303,6.461,6.629,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,12,L,1),I=1,8),L=1,NZEN)/
     * 7.641,7.713,7.766,7.843,7.940,8.033,8.155,8.267,
     * 7.241,7.312,7.367,7.456,7.565,7.698,7.858,8.017,
     * 7.223,7.270,7.303,7.358,7.428,7.558,7.708,7.879,
     * 7.207,7.241,7.270,7.290,7.324,7.394,7.483,7.638,
     * 7.176,7.199,7.230,7.270,7.318,7.360,7.412,7.459,
     * 7.158,7.176,7.210,7.262,7.322,7.371,7.425,7.467,
     * 7.130,7.140,7.185,7.262,7.342,7.408,7.471,7.509,
     * 0.000,0.000,0.000,0.000,0.000,6.009,6.241,6.479,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,12,L,1),I=1,8),L=1,NZEN)/
     * 7.436,7.473,7.520,7.618,7.731,7.848,8.009,8.185,
     * 6.835,6.910,7.000,7.155,7.324,7.504,7.722,7.944,
     * 6.806,6.847,6.906,7.041,7.188,7.371,7.593,7.827,
     * 6.787,6.803,6.844,6.930,7.025,7.179,7.362,7.600,
     * 6.755,6.760,6.805,6.897,6.994,7.086,7.204,7.338,
     * 6.738,6.738,6.785,6.891,6.999,7.097,7.217,7.332,
     * 6.710,6.704,6.757,6.890,7.017,7.130,7.260,7.369,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.140,6.449,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,12,L,1),I=1,8),L=1,NZEN)/
     * 7.057,7.072,7.130,7.276,7.433,7.597,7.818,8.079,
     * 6.447,6.525,6.647,6.862,7.083,7.305,7.581,7.877,
     * 6.403,6.446,6.535,6.740,6.950,7.182,7.467,7.772,
     * 6.375,6.380,6.441,6.600,6.763,6.980,7.246,7.561,
     * 6.344,6.336,6.396,6.545,6.693,6.836,7.021,7.246,
     * 6.328,6.312,6.375,6.538,6.697,6.843,7.025,7.217,
     * 6.299,6.279,6.348,6.538,6.718,6.877,7.068,7.250,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.373,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,1,L,2),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.307,8.380,8.427,8.474,8.520,
     * 0.000,0.000,0.000,0.000,8.538,8.560,8.580,8.595,
     * 0.000,0.000,0.000,0.000,8.400,8.427,8.444,8.476,
     * 0.000,0.000,0.000,0.000,8.320,8.290,8.246,8.267,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,1,L,2),I=1,8),L=1,NZEN)/
     * 7.676,7.810,7.897,7.975,8.079,8.167,8.274,8.367,
     * 7.716,7.807,7.854,7.914,7.996,8.086,8.193,8.288,
     * 7.694,7.759,7.785,7.811,7.856,7.955,8.061,8.173,
     * 7.671,7.732,7.762,7.758,7.768,7.813,7.857,7.963,
     * 7.631,7.703,7.757,7.801,7.848,7.879,7.893,7.886,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,6.267,6.558,6.869,7.124,7.330,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,1,L,2),I=1,8),L=1,NZEN)/
     * 7.519,7.606,7.678,7.775,7.892,7.997,8.134,8.262,
     * 7.130,7.215,7.286,7.407,7.545,7.700,7.881,8.053,
     * 7.111,7.167,7.215,7.307,7.413,7.575,7.760,7.949,
     * 7.086,7.140,7.188,7.248,7.320,7.435,7.566,7.754,
     * 7.049,7.111,7.185,7.290,7.394,7.486,7.580,7.655,
     * 7.025,7.093,7.179,7.322,7.453,7.563,7.669,7.737,
     * 6.988,7.072,7.188,7.386,7.560,7.704,7.834,7.905,
     * 0.000,0.000,0.000,0.000,6.179,6.539,6.870,7.152,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.045,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,1,L,2),I=1,8),L=1,NZEN)/
     * 7.272,7.326,7.393,7.513,7.648,7.781,7.960,8.152,
     * 6.645,6.727,6.824,7.004,7.201,7.407,7.656,7.904,
     * 6.616,6.667,6.738,6.903,7.076,7.297,7.554,7.816,
     * 6.592,6.629,6.694,6.823,6.957,7.146,7.367,7.638,
     * 6.551,6.602,6.691,6.859,7.017,7.164,7.328,7.491,
     * 6.529,6.587,6.688,6.892,7.079,7.241,7.417,7.565,
     * 6.494,6.567,6.696,6.957,7.188,7.380,7.582,7.729,
     * 0.000,0.000,0.000,0.000,0.000,6.303,6.702,7.061,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,1,L,2),I=1,8),L=1,NZEN)/
     * 6.893,6.922,6.995,7.158,7.336,7.513,7.753,8.029,
     * 6.137,6.217,6.342,6.585,6.838,7.097,7.412,7.744,
     * 6.104,6.146,6.243,6.479,6.720,6.993,7.320,7.666,
     * 6.076,6.100,6.182,6.380,6.579,6.836,7.143,7.501,
     * 6.033,6.072,6.173,6.403,6.619,6.816,7.049,7.312,
     * 6.013,6.057,6.173,6.438,6.681,6.892,7.137,7.369,
     * 0.000,6.037,6.182,6.505,6.794,7.033,7.299,7.531,
     * 0.000,0.000,0.000,0.000,0.000,6.004,6.470,6.912,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,2,L,2),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.217,8.299,8.354,8.412,8.464,
     * 0.000,0.000,0.000,0.000,8.479,8.508,8.535,8.563,
     * 0.000,0.000,0.000,0.000,8.344,8.377,8.401,8.442,
     * 0.000,0.000,0.000,0.000,8.279,8.253,8.210,8.238,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,2,L,2),I=1,8),L=1,NZEN)/
     * 7.525,7.678,7.783,7.877,7.994,8.093,8.210,8.312,
     * 7.713,7.806,7.853,7.908,7.985,8.079,8.188,8.286,
     * 7.695,7.764,7.793,7.814,7.852,7.951,8.053,8.167,
     * 7.674,7.740,7.775,7.772,7.783,7.823,7.860,7.963,
     * 7.632,7.711,7.770,7.816,7.862,7.896,7.911,7.904,
     * 7.610,7.695,7.768,7.846,7.919,7.974,8.004,7.991,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,7.340,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,2,L,2),I=1,8),L=1,NZEN)/
     * 7.344,7.462,7.563,7.680,7.815,7.940,8.093,8.233,
     * 7.199,7.286,7.358,7.473,7.602,7.750,7.921,8.086,
     * 7.179,7.241,7.294,7.378,7.474,7.628,7.801,7.981,
     * 7.155,7.217,7.274,7.330,7.396,7.500,7.615,7.788,
     * 7.117,7.188,7.270,7.375,7.476,7.562,7.648,7.712,
     * 7.093,7.173,7.267,7.408,7.534,7.639,7.737,7.797,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,6.253,6.607,6.927,7.199,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.093,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,2,L,2),I=1,8),L=1,NZEN)/
     * 7.253,7.326,7.410,7.537,7.677,7.809,7.981,8.158,
     * 6.787,6.877,6.980,7.149,7.330,7.521,7.748,7.973,
     * 6.761,6.822,6.898,7.053,7.212,7.412,7.646,7.885,
     * 6.738,6.787,6.859,6.981,7.104,7.272,7.468,7.713,
     * 6.698,6.759,6.854,7.017,7.167,7.301,7.452,7.591,
     * 6.676,6.744,6.854,7.049,7.228,7.378,7.539,7.667,
     * 6.640,6.724,6.859,7.114,7.336,7.520,7.703,7.833,
     * 0.000,0.000,0.000,0.000,6.033,6.430,6.813,7.152,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.068,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,2,L,2),I=1,8),L=1,NZEN)/
     * 6.959,7.004,7.083,7.241,7.407,7.572,7.790,8.037,
     * 6.348,6.439,6.568,6.796,7.025,7.262,7.547,7.846,
     * 6.312,6.373,6.476,6.695,6.913,7.161,7.459,7.769,
     * 6.286,6.328,6.420,6.607,6.788,7.017,7.288,7.607,
     * 6.248,6.301,6.413,6.635,6.838,7.021,7.233,7.456,
     * 6.225,6.286,6.413,6.670,6.900,7.097,7.320,7.524,
     * 6.190,6.267,6.422,6.737,7.013,7.238,7.483,7.688,
     * 0.000,0.000,0.000,0.000,0.000,6.170,6.617,7.037,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,3,L,2),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.117,8.204,8.267,8.336,8.398,
     * 0.000,0.000,0.000,0.000,8.449,8.483,8.515,8.547,
     * 0.000,0.000,0.000,0.000,8.318,8.354,8.380,8.425,
     * 0.000,0.000,0.000,0.000,8.260,8.233,8.193,8.220,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,3,L,2),I=1,8),L=1,NZEN)/
     * 7.394,7.562,7.680,7.785,7.912,8.021,8.140,8.250,
     * 7.796,7.894,7.946,7.999,8.072,8.158,8.253,8.340,
     * 7.777,7.854,7.889,7.906,7.941,8.029,8.121,8.223,
     * 7.754,7.831,7.871,7.867,7.875,7.907,7.932,8.021,
     * 7.714,7.801,7.866,7.912,7.957,7.984,7.988,7.969,
     * 7.692,7.785,7.864,7.941,8.009,8.057,8.076,8.057,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,3,L,2),I=1,8),L=1,NZEN)/
     * 7.220,7.373,7.502,7.635,7.782,7.916,8.068,8.210,
     * 7.380,7.481,7.561,7.663,7.775,7.905,8.049,8.190,
     * 7.362,7.441,7.505,7.577,7.657,7.789,7.931,8.083,
     * 7.340,7.417,7.486,7.538,7.593,7.674,7.757,7.899,
     * 7.301,7.391,7.484,7.585,7.675,7.747,7.809,7.849,
     * 7.276,7.375,7.480,7.616,7.733,7.825,7.900,7.936,
     * 7.241,7.354,7.487,7.679,7.840,7.967,8.064,8.104,
     * 0.000,0.000,0.000,6.093,6.428,6.769,7.064,7.314,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.041,6.207,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,3,L,2),I=1,8),L=1,NZEN)/
     * 7.215,7.334,7.456,7.600,7.749,7.888,8.057,8.217,
     * 7.029,7.130,7.233,7.377,7.524,7.688,7.876,8.068,
     * 7.009,7.090,7.179,7.301,7.422,7.587,7.774,7.975,
     * 6.989,7.068,7.158,7.267,7.365,7.489,7.626,7.816,
     * 6.949,7.041,7.155,7.314,7.449,7.562,7.679,7.777,
     * 6.927,7.025,7.155,7.346,7.509,7.641,7.771,7.863,
     * 6.892,7.009,7.161,7.412,7.618,7.782,7.935,8.029,
     * 0.000,0.000,0.000,0.000,6.253,6.630,6.980,7.288,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.207,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,3,L,2),I=1,8),L=1,NZEN)/
     * 7.104,7.196,7.312,7.470,7.625,7.772,7.956,8.152,
     * 6.639,6.746,6.880,7.079,7.270,7.471,7.708,7.956,
     * 6.617,6.702,6.818,7.000,7.170,7.378,7.618,7.876,
     * 6.594,6.673,6.787,6.952,7.100,7.274,7.474,7.727,
     * 6.555,6.647,6.785,6.997,7.179,7.332,7.502,7.665,
     * 6.533,6.633,6.785,7.033,7.238,7.410,7.591,7.746,
     * 6.498,6.615,6.792,7.097,7.350,7.551,7.756,7.911,
     * 0.000,0.000,0.000,0.000,6.041,6.456,6.860,7.228,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.164,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,4,L,2),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.093,8.176,8.246,8.316,8.382,
     * 0.000,0.000,0.000,0.000,8.438,8.476,8.513,8.545,
     * 0.000,0.000,0.000,0.000,8.310,8.346,8.377,8.427,
     * 0.000,0.000,0.000,0.000,8.246,8.225,8.188,8.223,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,4,L,2),I=1,8),L=1,NZEN)/
     * 7.362,7.542,7.667,7.777,7.905,8.013,8.127,8.236,
     * 7.927,8.037,8.097,8.143,8.207,8.281,8.358,8.428,
     * 7.909,7.997,8.041,8.053,8.079,8.155,8.228,8.312,
     * 7.888,7.975,8.021,8.013,8.013,8.033,8.041,8.111,
     * 7.847,7.945,8.017,8.061,8.097,8.111,8.097,8.061,
     * 7.825,7.929,8.017,8.086,8.149,8.188,8.188,8.149,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,4,L,2),I=1,8),L=1,NZEN)/
     * 7.149,7.334,7.491,7.632,7.782,7.918,8.061,8.193,
     * 7.630,7.749,7.840,7.920,8.004,8.107,8.215,8.320,
     * 7.614,7.717,7.797,7.848,7.900,8.000,8.097,8.212,
     * 7.592,7.696,7.784,7.827,7.866,7.913,7.950,8.041,
     * 7.554,7.669,7.778,7.873,7.950,8.004,8.029,8.033,
     * 7.530,7.654,7.778,7.906,8.009,8.079,8.121,8.124,
     * 7.494,7.633,7.782,7.968,8.114,8.220,8.288,8.292,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,7.470,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,4,L,2),I=1,8),L=1,NZEN)/
     * 7.100,7.279,7.449,7.605,7.758,7.905,8.061,8.215,
     * 7.375,7.500,7.616,7.722,7.820,7.943,8.072,8.212,
     * 7.360,7.477,7.590,7.679,7.757,7.865,7.976,8.121,
     * 7.340,7.461,7.584,7.679,7.755,7.825,7.891,7.998,
     * 7.301,7.435,7.581,7.730,7.846,7.933,8.004,8.045,
     * 7.279,7.422,7.580,7.763,7.905,8.013,8.093,8.137,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,6.220,6.574,6.922,7.223,7.479,
     * 0.000,0.000,0.000,0.000,0.000,6.033,6.248,6.410,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,4,L,2),I=1,8),L=1,NZEN)/
     * 7.111,7.276,7.449,7.617,7.768,7.912,8.072,8.236,
     * 7.086,7.225,7.373,7.521,7.648,7.794,7.951,8.127,
     * 7.072,7.204,7.352,7.489,7.599,7.731,7.870,8.045,
     * 7.049,7.190,7.344,7.493,7.607,7.709,7.813,7.952,
     * 7.013,7.164,7.342,7.547,7.702,7.822,7.932,8.021,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,6.057,6.452,6.829,7.170,7.470,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.220,6.422,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,5,L,2),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.176,8.255,8.318,8.382,8.439,
     * 0.000,0.000,0.000,0.000,8.455,8.491,8.526,8.559,
     * 0.000,0.000,0.000,0.000,8.322,8.362,8.393,8.441,
     * 0.000,0.000,0.000,0.000,8.250,8.233,8.196,8.236,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,5,L,2),I=1,8),L=1,NZEN)/
     * 0.000,0.000,7.773,7.881,8.004,8.104,8.207,8.297,
     * 0.000,0.000,8.243,8.281,8.332,8.396,8.452,8.507,
     * 0.000,0.000,8.190,8.193,8.207,8.270,8.322,8.391,
     * 0.000,0.000,8.173,8.158,8.149,8.155,8.140,8.193,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,5,L,2),I=1,8),L=1,NZEN)/
     * 7.182,7.387,7.558,7.697,7.839,7.968,8.093,8.212,
     * 7.866,7.999,8.100,8.158,8.217,8.294,8.364,8.441,
     * 7.851,7.972,8.064,8.097,8.127,8.193,8.248,8.332,
     * 7.829,7.953,8.053,8.086,8.107,8.127,8.124,8.176,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,5,L,2),I=1,8),L=1,NZEN)/
     * 7.045,7.253,7.442,7.594,7.734,7.875,8.009,8.149,
     * 7.702,7.854,7.989,8.064,8.124,8.201,8.270,8.364,
     * 7.688,7.839,7.975,8.041,8.086,8.146,8.188,8.276,
     * 7.669,7.825,7.971,8.053,8.111,8.143,8.152,8.190,
     * 7.630,7.801,7.971,8.107,8.204,8.262,8.283,8.281,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,5,L,2),I=1,8),L=1,NZEN)/
     * 6.946,7.170,7.389,7.551,7.686,7.820,7.946,8.090,
     * 7.494,7.679,7.857,7.966,8.041,8.114,8.179,8.276,
     * 7.481,7.668,7.851,7.964,8.037,8.097,8.140,8.223,
     * 7.462,7.655,7.848,7.985,8.076,8.134,8.167,8.207,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,6,L,2),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.288,8.362,8.413,8.467,8.516,
     * 0.000,0.000,0.000,0.000,8.496,8.529,8.560,8.588,
     * 0.000,0.000,0.000,0.000,8.352,8.394,8.428,8.473,
     * 0.000,0.000,0.000,0.000,8.267,8.253,8.223,8.262,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,6,L,2),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.025,8.143,8.233,8.316,8.394,
     * 0.000,0.000,0.000,8.405,8.453,8.507,8.550,8.592,
     * 0.000,0.000,0.000,8.310,8.320,8.378,8.422,8.477,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,8.279,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,6,L,2),I=1,8),L=1,NZEN)/
     * 7.279,7.496,7.675,7.810,7.943,8.061,8.167,8.272,
     * 8.049,8.190,8.297,8.346,8.393,8.456,8.504,8.561,
     * 0.000,0.000,0.000,0.000,8.307,8.358,8.389,8.453,
     * 0.000,0.000,0.000,0.000,0.000,0.000,8.267,8.299,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,6,L,2),I=1,8),L=1,NZEN)/
     * 7.093,7.316,7.513,7.648,7.766,7.892,8.000,8.127,
     * 7.949,8.124,8.272,8.328,8.365,8.412,8.435,8.496,
     * 0.000,0.000,0.000,0.000,0.000,0.000,8.365,8.415,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,8.352,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,6,L,2),I=1,8),L=1,NZEN)/
     * 6.927,7.176,7.408,7.555,7.665,7.775,7.860,7.981,
     * 7.787,8.004,8.207,8.301,8.354,8.384,8.384,8.422,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,7,L,2),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.316,8.393,8.441,8.491,8.537,
     * 0.000,0.000,0.000,0.000,8.525,8.554,8.582,8.604,
     * 0.000,0.000,0.000,0.000,8.380,8.420,8.452,8.490,
     * 0.000,0.000,0.000,0.000,0.000,0.000,8.243,8.281,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,7,L,2),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.100,8.217,8.301,8.377,8.446,
     * 0.000,0.000,0.000,8.471,8.521,8.569,8.607,8.640,
     * 0.000,0.000,0.000,8.369,8.384,8.439,8.479,8.529,
     * 0.000,0.000,0.000,0.000,8.297,8.301,8.283,8.326,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,7,L,2),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,7.919,8.049,8.158,8.255,8.348,
     * 0.000,0.000,0.000,8.442,8.491,8.555,8.599,8.645,
     * 0.000,0.000,0.000,8.373,8.393,8.449,8.481,8.540,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,7,L,2),I=1,8),L=1,NZEN)/
     * 7.248,7.477,7.677,7.806,7.916,8.033,8.127,8.241,
     * 8.041,8.220,8.373,8.428,8.461,8.508,8.529,8.584,
     * 0.000,0.000,0.000,0.000,0.000,0.000,8.453,8.502,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,8.422,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,7,L,2),I=1,8),L=1,NZEN)/
     * 7.155,7.413,7.650,7.790,7.892,7.985,8.049,8.155,
     * 7.897,8.121,8.328,8.418,8.464,8.484,8.471,8.501,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,8.471,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,8,L,2),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.243,8.328,8.384,8.442,8.493,
     * 0.000,0.000,0.000,0.000,8.505,8.535,8.565,8.590,
     * 0.000,0.000,0.000,0.000,0.000,8.401,8.431,8.473,
     * 0.000,0.000,0.000,0.000,0.000,0.000,8.228,8.265,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,8,L,2),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.083,8.201,8.288,8.369,8.442,
     * 0.000,0.000,0.000,8.430,8.481,8.531,8.574,8.613,
     * 0.000,0.000,0.000,0.000,8.342,8.401,8.444,8.498,
     * 0.000,0.000,0.000,0.000,0.000,8.265,8.248,8.297,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,8,L,2),I=1,8),L=1,NZEN)/
     * 7.479,7.696,7.877,8.009,8.137,8.243,8.338,8.430,
     * 8.064,8.212,8.322,8.387,8.450,8.517,8.568,8.623,
     * 8.049,8.179,8.274,8.312,8.340,8.407,8.452,8.517,
     * 0.000,0.000,0.000,0.000,0.000,0.000,8.297,8.342,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,8,L,2),I=1,8),L=1,NZEN)/
     * 7.493,7.712,7.907,8.037,8.152,8.262,8.354,8.453,
     * 7.949,8.117,8.262,8.336,8.389,8.459,8.509,8.580,
     * 0.000,0.000,0.000,0.000,8.340,8.391,8.417,8.487,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,8,L,2),I=1,8),L=1,NZEN)/
     * 7.592,7.828,8.049,8.179,8.272,8.364,8.431,8.524,
     * 7.799,8.004,8.196,8.297,8.358,8.415,8.449,8.521,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,8.455,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,9,L,2),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.146,8.236,8.299,8.367,8.427,
     * 0.000,0.000,0.000,0.000,8.473,8.507,8.540,8.569,
     * 0.000,0.000,0.000,0.000,8.334,8.375,8.407,8.452,
     * 0.000,0.000,0.000,0.000,0.000,0.000,8.207,8.243,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,9,L,2),I=1,8),L=1,NZEN)/
     * 0.000,0.000,7.923,8.025,8.146,8.238,8.328,8.412,
     * 0.000,0.000,8.243,8.288,8.348,8.413,8.473,8.529,
     * 0.000,0.000,8.179,8.190,8.212,8.283,8.342,8.413,
     * 8.017,8.107,8.158,8.146,8.140,8.152,8.149,8.210,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,9,L,2),I=1,8),L=1,NZEN)/
     * 7.581,7.778,7.942,8.064,8.190,8.292,8.393,8.486,
     * 7.853,7.987,8.090,8.167,8.248,8.336,8.420,8.498,
     * 0.000,0.000,0.000,8.090,8.137,8.223,8.299,8.391,
     * 0.000,0.000,0.000,0.000,8.083,8.117,8.134,8.212,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,9,L,2),I=1,8),L=1,NZEN)/
     * 7.728,7.911,8.076,8.201,8.314,8.412,8.508,8.597,
     * 7.683,7.830,7.961,8.064,8.152,8.260,8.362,8.468,
     * 0.000,0.000,0.000,0.000,8.072,8.170,8.258,8.373,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,8.230,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,9,L,2),I=1,8),L=1,NZEN)/
     * 7.891,8.053,8.215,8.340,8.439,8.524,8.604,8.688,
     * 7.501,7.671,7.838,7.979,8.090,8.212,8.330,8.462,
     * 0.000,7.648,7.810,7.934,8.021,8.137,8.241,8.377,
     * 0.000,0.000,7.801,7.928,8.013,8.086,8.146,8.253,
     * 0.000,0.000,0.000,7.980,8.104,8.190,8.246,8.279,
     * 0.000,0.000,0.000,0.000,0.000,0.000,8.336,8.369,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,10,L,2),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.124,8.210,8.276,8.346,8.408,
     * 0.000,0.000,0.000,0.000,8.453,8.491,8.529,8.562,
     * 0.000,0.000,0.000,0.000,8.318,8.360,8.394,8.444,
     * 0.000,0.000,0.000,0.000,0.000,0.000,8.196,8.236,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,10,L,2),I=1,8),L=1,NZEN)/
     * 7.626,7.795,7.912,8.004,8.117,8.210,8.310,8.398,
     * 7.905,8.013,8.072,8.124,8.190,8.272,8.352,8.428,
     * 0.000,7.970,8.013,8.025,8.057,8.140,8.220,8.312,
     * 0.000,0.000,0.000,0.000,0.000,8.013,8.025,8.107,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,10,L,2),I=1,8),L=1,NZEN)/
     * 7.661,7.818,7.947,8.057,8.173,8.272,8.380,8.477,
     * 7.599,7.715,7.806,7.901,8.004,8.114,8.233,8.344,
     * 0.000,7.675,7.748,7.812,7.880,7.995,8.111,8.238,
     * 0.000,0.000,7.728,7.772,7.814,7.878,7.934,8.049,
     * 0.000,0.000,0.000,7.819,7.895,7.949,7.982,7.992,
     * 0.000,0.000,0.000,0.000,7.950,8.025,8.072,8.079,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,10,L,2),I=1,8),L=1,NZEN)/
     * 7.769,7.902,8.029,8.140,8.253,8.344,8.452,8.555,
     * 7.364,7.489,7.609,7.744,7.875,8.013,8.167,8.316,
     * 7.340,7.446,7.548,7.660,7.763,7.909,8.061,8.225,
     * 7.322,7.422,7.525,7.616,7.692,7.796,7.899,8.057,
     * 7.281,7.394,7.521,7.661,7.770,7.856,7.930,7.986,
     * 7.260,7.378,7.519,7.693,7.829,7.934,8.021,8.068,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,6.037,6.270,6.450,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,10,L,2),I=1,8),L=1,NZEN)/
     * 7.740,7.851,7.973,8.100,8.215,8.314,8.435,8.571,
     * 7.117,7.267,7.428,7.620,7.794,7.965,8.152,8.344,
     * 7.086,7.210,7.346,7.526,7.683,7.865,8.061,8.262,
     * 7.064,7.173,7.303,7.453,7.574,7.731,7.895,8.104,
     * 7.025,7.146,7.297,7.489,7.636,7.754,7.867,7.976,
     * 7.004,7.134,7.297,7.522,7.695,7.831,7.957,8.049,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.272,6.496,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,11,L,2),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.204,8.281,8.338,8.398,8.453,
     * 0.000,0.000,0.000,0.000,8.477,8.511,8.543,8.573,
     * 0.000,0.000,0.000,0.000,8.338,8.377,8.408,8.453,
     * 0.000,0.000,0.000,0.000,0.000,0.000,8.210,8.246,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,11,L,2),I=1,8),L=1,NZEN)/
     * 7.706,7.848,7.940,8.021,8.121,8.210,8.310,8.400,
     * 7.792,7.889,7.941,7.997,8.072,8.161,8.260,8.350,
     * 0.000,0.000,0.000,0.000,7.934,8.029,8.127,8.233,
     * 0.000,0.000,0.000,0.000,0.000,0.000,7.925,8.025,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,7.949,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,11,L,2),I=1,8),L=1,NZEN)/
     * 7.669,7.782,7.873,7.969,8.079,8.176,8.292,8.398,
     * 7.371,7.474,7.556,7.667,7.790,7.922,8.072,8.212,
     * 7.348,7.422,7.481,7.563,7.656,7.797,7.952,8.111,
     * 7.326,7.394,7.455,7.507,7.561,7.656,7.757,7.913,
     * 7.288,7.367,7.450,7.548,7.635,7.709,7.775,7.818,
     * 7.265,7.352,7.449,7.580,7.694,7.787,7.864,7.901,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.033,6.207,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,11,L,2),I=1,8),L=1,NZEN)/
     * 7.610,7.694,7.782,7.890,8.004,8.111,8.248,8.393,
     * 7.057,7.164,7.283,7.446,7.615,7.782,7.978,8.173,
     * 7.021,7.100,7.188,7.338,7.490,7.673,7.881,8.086,
     * 6.996,7.057,7.137,7.250,7.358,7.517,7.695,7.912,
     * 6.956,7.029,7.130,7.281,7.412,7.525,7.645,7.759,
     * 6.934,7.017,7.130,7.314,7.471,7.602,7.733,7.830,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.004,6.238,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,11,L,2),I=1,8),L=1,NZEN)/
     * 7.356,7.423,7.519,7.663,7.806,7.947,8.130,8.344,
     * 6.708,6.836,6.993,7.215,7.433,7.640,7.888,8.143,
     * 6.658,6.751,6.877,7.100,7.314,7.540,7.803,8.072,
     * 6.625,6.688,6.790,6.977,7.149,7.373,7.627,7.911,
     * 6.585,6.657,6.777,6.985,7.164,7.322,7.500,7.697,
     * 6.565,6.643,6.777,7.017,7.223,7.394,7.581,7.744,
     * 6.526,6.623,6.784,7.086,7.336,7.537,7.744,7.905,
     * 0.000,0.000,0.000,0.000,6.076,6.493,6.907,7.288,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.201,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,12,L,2),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.297,8.367,8.415,8.465,8.511,
     * 0.000,0.000,0.000,0.000,8.515,8.538,8.560,8.582,
     * 0.000,0.000,0.000,0.000,8.375,8.405,8.427,8.462,
     * 0.000,0.000,0.000,0.000,0.000,0.000,8.225,8.255,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,12,L,2),I=1,8),L=1,NZEN)/
     * 7.740,7.869,7.950,8.025,8.121,8.207,8.305,8.394,
     * 7.731,7.821,7.867,7.923,8.000,8.090,8.193,8.288,
     * 7.709,7.775,7.801,7.822,7.862,7.959,8.061,8.173,
     * 0.000,0.000,0.000,7.773,7.780,7.822,7.861,7.965,
     * 0.000,0.000,0.000,0.000,0.000,7.890,7.903,7.893,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,7.979,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,12,L,2),I=1,8),L=1,NZEN)/
     * 7.598,7.688,7.760,7.853,7.965,8.064,8.190,8.312,
     * 7.188,7.274,7.346,7.464,7.598,7.746,7.920,8.086,
     * 7.164,7.225,7.274,7.362,7.464,7.621,7.799,7.979,
     * 7.143,7.199,7.250,7.307,7.373,7.481,7.604,7.784,
     * 7.104,7.170,7.246,7.350,7.450,7.537,7.623,7.689,
     * 7.079,7.152,7.241,7.380,7.505,7.613,7.712,7.772,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.079,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,12,L,2),I=1,8),L=1,NZEN)/
     * 7.405,7.459,7.526,7.639,7.767,7.890,8.057,8.238,
     * 6.793,6.890,7.000,7.179,7.371,7.563,7.795,8.025,
     * 6.757,6.816,6.894,7.064,7.243,7.452,7.696,7.941,
     * 6.728,6.768,6.832,6.961,7.093,7.286,7.508,7.765,
     * 6.691,6.740,6.825,6.985,7.134,7.270,7.422,7.579,
     * 6.666,6.723,6.825,7.017,7.193,7.346,7.509,7.642,
     * 6.630,6.705,6.832,7.083,7.303,7.486,7.671,7.805,
     * 0.000,0.000,0.000,0.000,6.029,6.428,6.818,7.164,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.076,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,12,L,2),I=1,8),L=1,NZEN)/
     * 7.025,7.064,7.143,7.303,7.473,7.644,7.873,8.140,
     * 6.405,6.509,6.658,6.893,7.134,7.369,7.660,7.962,
     * 6.344,6.413,6.530,6.773,7.017,7.272,7.576,7.892,
     * 6.305,6.336,6.420,6.630,6.837,7.097,7.401,7.735,
     * 6.265,6.301,6.403,6.621,6.824,7.009,7.230,7.487,
     * 6.243,6.288,6.403,6.656,6.885,7.083,7.307,7.520,
     * 6.207,6.270,6.410,6.723,6.997,7.223,7.470,7.678,
     * 0.000,0.000,0.000,0.000,0.000,6.196,6.649,7.079,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,1,L,3),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.360,8.436,8.484,8.535,8.582,
     * 0.000,0.000,0.000,0.000,8.605,8.632,8.655,8.675,
     * 0.000,0.000,0.000,0.000,8.483,8.522,8.550,8.588,
     * 0.000,0.000,0.000,0.000,0.000,8.422,8.396,8.427,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,1,L,3),I=1,8),L=1,NZEN)/
     * 7.698,7.841,7.938,8.021,8.130,8.225,8.334,8.431,
     * 7.728,7.833,7.894,7.966,8.057,8.158,8.270,8.369,
     * 7.696,7.776,7.818,7.866,7.931,8.045,8.164,8.281,
     * 7.654,7.738,7.791,7.820,7.858,7.931,7.997,8.117,
     * 7.579,7.688,7.780,7.882,7.974,8.049,8.097,8.107,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,1,L,3),I=1,8),L=1,NZEN)/
     * 7.512,7.617,7.706,7.814,7.942,8.053,8.196,8.328,
     * 7.111,7.215,7.303,7.444,7.602,7.767,7.958,8.137,
     * 7.079,7.152,7.217,7.340,7.474,7.658,7.862,8.057,
     * 7.041,7.114,7.185,7.281,7.380,7.530,7.693,7.900,
     * 6.972,7.072,7.182,7.348,7.498,7.629,7.757,7.851,
     * 6.937,7.053,7.185,7.400,7.585,7.744,7.893,7.981,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,1,L,3),I=1,8),L=1,NZEN)/
     * 7.262,7.336,7.423,7.556,7.705,7.844,8.029,8.228,
     * 6.615,6.720,6.841,7.045,7.262,7.480,7.740,7.994,
     * 6.574,6.644,6.736,6.937,7.146,7.386,7.662,7.933,
     * 6.539,6.599,6.685,6.852,7.017,7.246,7.504,7.793,
     * 6.484,6.572,6.694,6.919,7.124,7.303,7.500,7.686,
     * 6.452,6.555,6.702,6.978,7.223,7.427,7.641,7.807,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,1,L,3),I=1,8),L=1,NZEN)/
     * 6.880,6.934,7.033,7.207,7.393,7.577,7.823,8.107,
     * 6.090,6.199,6.356,6.623,6.900,7.170,7.496,7.834,
     * 6.037,6.107,6.230,6.508,6.787,7.083,7.431,7.785,
     * 0.000,6.045,6.146,6.391,6.628,6.931,7.281,7.658,
     * 0.000,6.017,6.152,6.439,6.701,6.933,7.201,7.493,
     * 0.000,6.004,6.164,6.501,6.802,7.053,7.336,7.590,
     * 0.000,0.000,6.193,6.615,6.984,7.274,7.587,7.843,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,2,L,3),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.270,8.354,8.412,8.473,8.526,
     * 0.000,0.000,0.000,0.000,8.544,8.580,8.613,8.643,
     * 0.000,0.000,0.000,0.000,8.428,8.473,8.507,8.553,
     * 0.000,0.000,0.000,0.000,0.000,8.380,8.356,8.394,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,2,L,3),I=1,8),L=1,NZEN)/
     * 7.547,7.709,7.824,7.924,8.045,8.149,8.267,8.375,
     * 7.726,7.829,7.889,7.958,8.049,8.149,8.262,8.365,
     * 7.695,7.779,7.824,7.867,7.925,8.041,8.155,8.276,
     * 7.653,7.744,7.801,7.831,7.867,7.938,7.999,8.114,
     * 7.576,7.694,7.792,7.897,7.991,8.064,8.117,8.124,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,2,L,3),I=1,8),L=1,NZEN)/
     * 7.334,7.471,7.588,7.717,7.863,7.994,8.152,8.294,
     * 7.176,7.281,7.371,7.507,7.656,7.815,7.997,8.167,
     * 7.146,7.225,7.297,7.410,7.533,7.708,7.900,8.086,
     * 7.107,7.190,7.270,7.362,7.455,7.593,7.739,7.931,
     * 7.041,7.152,7.270,7.433,7.577,7.704,7.822,7.905,
     * 7.004,7.130,7.270,7.484,7.667,7.821,7.960,8.037,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,2,L,3),I=1,8),L=1,NZEN)/
     * 7.236,7.328,7.433,7.574,7.727,7.866,8.045,8.228,
     * 6.757,6.869,6.994,7.188,7.389,7.590,7.827,8.057,
     * 6.721,6.801,6.896,7.083,7.274,7.496,7.750,7.997,
     * 6.687,6.757,6.848,7.009,7.161,7.367,7.599,7.861,
     * 6.628,6.728,6.857,7.079,7.272,7.442,7.622,7.782,
     * 6.600,6.715,6.868,7.137,7.371,7.563,7.760,7.907,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,2,L,3),I=1,8),L=1,NZEN)/
     * 6.932,7.000,7.107,7.279,7.455,7.625,7.849,8.100,
     * 6.301,6.420,6.577,6.827,7.079,7.326,7.621,7.926,
     * 6.248,6.332,6.459,6.717,6.971,7.241,7.555,7.875,
     * 6.210,6.272,6.384,6.614,6.831,7.100,7.412,7.751,
     * 6.155,6.246,6.393,6.672,6.921,7.137,7.378,7.628,
     * 6.127,6.233,6.403,6.734,7.021,7.258,7.517,7.740,
     * 6.079,6.223,6.430,6.846,7.201,7.477,7.768,7.995,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,3,L,3),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.170,8.258,8.324,8.394,8.458,
     * 0.000,0.000,0.000,0.000,8.513,8.554,8.591,8.626,
     * 0.000,0.000,0.000,0.000,8.400,8.447,8.484,8.535,
     * 0.000,0.000,0.000,0.000,0.000,8.364,8.340,8.380,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,3,L,3),I=1,8),L=1,NZEN)/
     * 7.415,7.592,7.720,7.831,7.962,8.076,8.199,8.312,
     * 7.806,7.916,7.980,8.045,8.130,8.228,8.328,8.418,
     * 7.777,7.869,7.920,7.960,8.013,8.121,8.223,8.330,
     * 7.736,7.834,7.899,7.928,7.963,8.021,8.068,8.170,
     * 7.659,7.784,7.889,7.991,8.083,8.152,8.193,8.190,
     * 7.622,7.761,7.886,8.037,8.164,8.262,8.320,8.320,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,3,L,3),I=1,8),L=1,NZEN)/
     * 7.207,7.378,7.525,7.669,7.827,7.967,8.124,8.270,
     * 7.358,7.474,7.571,7.694,7.827,7.968,8.121,8.267,
     * 7.328,7.423,7.504,7.604,7.709,7.863,8.025,8.188,
     * 7.292,7.391,7.481,7.568,7.649,7.764,7.877,8.037,
     * 7.225,7.352,7.480,7.640,7.777,7.890,7.985,8.041,
     * 7.190,7.332,7.483,7.693,7.866,8.004,8.121,8.176,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,3,L,3),I=1,8),L=1,NZEN)/
     * 7.193,7.332,7.471,7.629,7.792,7.937,8.111,8.276,
     * 6.995,7.111,7.233,7.400,7.569,7.747,7.948,8.143,
     * 6.965,7.061,7.167,7.320,7.468,7.658,7.866,8.076,
     * 6.936,7.033,7.146,7.286,7.413,7.569,7.737,7.948,
     * 6.880,7.009,7.158,7.371,7.551,7.700,7.846,7.960,
     * 6.851,6.996,7.167,7.430,7.646,7.822,7.989,8.097,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,3,L,3),I=1,8),L=1,NZEN)/
     * 7.061,7.176,7.318,7.496,7.669,7.823,8.017,8.220,
     * 6.582,6.713,6.870,7.097,7.314,7.533,7.785,8.041,
     * 6.547,6.650,6.784,7.004,7.212,7.447,7.712,7.981,
     * 6.515,6.615,6.749,6.951,7.127,7.342,7.580,7.859,
     * 6.464,6.592,6.763,7.033,7.255,7.444,7.642,7.825,
     * 6.433,6.580,6.772,7.093,7.356,7.568,7.786,7.957,
     * 6.391,6.572,6.802,7.207,7.537,7.787,8.037,8.215,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,4,L,3),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.146,8.230,8.303,8.377,8.444,
     * 0.000,0.000,0.000,0.000,8.505,8.549,8.589,8.625,
     * 0.000,0.000,0.000,0.000,8.389,8.441,8.484,8.537,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,4,L,3),I=1,8),L=1,NZEN)/
     * 7.382,7.569,7.706,7.822,7.954,8.068,8.188,8.294,
     * 7.940,8.061,8.130,8.190,8.267,8.350,8.433,8.505,
     * 7.910,8.013,8.072,8.104,8.149,8.243,8.328,8.418,
     * 7.868,7.977,8.049,8.072,8.097,8.146,8.176,8.260,
     * 7.794,7.928,8.037,8.137,8.220,8.279,8.301,8.281,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,4,L,3),I=1,8),L=1,NZEN)/
     * 7.134,7.336,7.509,7.664,7.826,7.967,8.114,8.253,
     * 7.605,7.737,7.842,7.944,8.049,8.167,8.283,8.396,
     * 7.579,7.695,7.790,7.869,7.947,8.068,8.188,8.312,
     * 7.543,7.667,7.775,7.852,7.919,7.997,8.061,8.173,
     * 7.477,7.629,7.775,7.928,8.049,8.143,8.204,8.223,
     * 7.444,7.612,7.778,7.980,8.137,8.258,8.340,8.362,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,4,L,3),I=1,8),L=1,NZEN)/
     * 7.072,7.265,7.449,7.622,7.792,7.946,8.111,8.270,
     * 7.336,7.473,7.602,7.731,7.851,7.989,8.134,8.281,
     * 7.316,7.446,7.572,7.684,7.782,7.917,8.053,8.210,
     * 7.286,7.427,7.569,7.694,7.792,7.891,7.981,8.111,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,4,L,3),I=1,8),L=1,NZEN)/
     * 7.053,7.236,7.425,7.615,7.787,7.945,8.117,8.288,
     * 7.017,7.170,7.332,7.507,7.659,7.825,8.004,8.190,
     * 6.998,7.143,7.301,7.465,7.600,7.763,7.932,8.124,
     * 6.968,7.127,7.301,7.480,7.618,7.748,7.877,8.041,
     * 6.920,7.107,7.316,7.576,7.775,7.928,8.068,8.167,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,5,L,3),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.230,8.312,8.375,8.442,8.504,
     * 0.000,0.000,0.000,0.000,8.521,8.565,8.605,8.640,
     * 0.000,0.000,0.000,0.000,8.403,8.456,8.501,8.553,
     * 0.000,0.000,0.000,0.000,0.000,8.362,8.346,8.394,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,5,L,3),I=1,8),L=1,NZEN)/
     * 0.000,0.000,7.812,7.927,8.057,8.161,8.262,8.358,
     * 0.000,0.000,8.279,8.330,8.394,8.464,8.526,8.583,
     * 0.000,0.000,8.220,8.246,8.279,8.358,8.423,8.497,
     * 0.000,0.000,0.000,8.217,8.233,8.267,8.276,8.342,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,5,L,3),I=1,8),L=1,NZEN)/
     * 7.164,7.386,7.571,7.723,7.878,8.013,8.143,8.267,
     * 7.838,7.984,8.097,8.176,8.255,8.348,8.431,8.512,
     * 7.814,7.949,8.057,8.114,8.170,8.260,8.334,8.428,
     * 7.778,7.923,8.045,8.111,8.158,8.207,8.230,8.301,
     * 7.714,7.886,8.045,8.188,8.294,8.365,8.394,8.382,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,5,L,3),I=1,8),L=1,NZEN)/
     * 7.009,7.228,7.430,7.600,7.758,7.909,8.053,8.201,
     * 7.660,7.823,7.968,8.064,8.140,8.238,8.324,8.427,
     * 7.642,7.804,7.951,8.037,8.104,8.188,8.253,8.356,
     * 7.614,7.789,7.954,8.064,8.143,8.201,8.230,8.292,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,5,L,3),I=1,8),L=1,NZEN)/
     * 6.878,7.111,7.336,7.519,7.670,7.823,7.968,8.124,
     * 7.423,7.615,7.800,7.927,8.021,8.117,8.201,8.316,
     * 7.407,7.603,7.795,7.927,8.017,8.100,8.167,8.270,
     * 7.380,7.592,7.803,7.968,8.083,8.158,8.210,8.270,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,6,L,3),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.338,8.417,8.470,8.526,8.577,
     * 0.000,0.000,0.000,0.000,8.560,8.601,8.638,8.670,
     * 0.000,0.000,0.000,0.000,8.436,8.491,8.535,8.584,
     * 0.000,0.000,0.000,0.000,0.000,8.384,8.375,8.425,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,6,L,3),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.068,8.193,8.286,8.373,8.453,
     * 0.000,0.000,0.000,8.452,8.512,8.573,8.623,8.667,
     * 0.000,0.000,0.000,8.362,8.391,8.465,8.521,8.583,
     * 0.000,0.000,0.000,0.000,8.334,8.367,8.369,8.428,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,6,L,3),I=1,8),L=1,NZEN)/
     * 7.260,7.494,7.688,7.835,7.982,8.107,8.220,8.326,
     * 8.017,8.173,8.292,8.365,8.433,8.511,8.571,8.633,
     * 7.995,8.143,8.255,8.305,8.346,8.422,8.474,8.550,
     * 0.000,0.000,0.000,8.303,8.338,8.373,8.371,8.423,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,8.504,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,6,L,3),I=1,8),L=1,NZEN)/
     * 7.057,7.286,7.491,7.642,7.777,7.917,8.037,8.173,
     * 7.907,8.086,8.246,8.320,8.371,8.439,8.481,8.553,
     * 7.889,8.076,8.238,8.310,8.354,8.401,8.420,8.487,
     * 7.861,8.061,8.243,8.346,8.408,8.439,8.430,8.446,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,6,L,3),I=1,8),L=1,NZEN)/
     * 6.854,7.107,7.342,7.502,7.625,7.755,7.860,7.998,
     * 7.713,7.937,8.146,8.253,8.316,8.365,8.384,8.442,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,7,L,3),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.367,8.447,8.498,8.551,8.599,
     * 0.000,0.000,0.000,0.000,8.591,8.626,8.659,8.686,
     * 0.000,0.000,0.000,0.000,8.464,8.516,8.558,8.601,
     * 0.000,0.000,0.000,0.000,0.000,8.401,8.391,8.439,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,7,L,3),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.146,8.270,8.354,8.435,8.507,
     * 0.000,0.000,0.000,8.521,8.582,8.637,8.682,8.719,
     * 0.000,0.000,0.000,0.000,0.000,8.529,8.581,8.636,
     * 0.000,0.000,0.000,0.000,0.000,8.417,8.420,8.477,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,7,L,3),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,7.945,8.090,8.204,8.307,8.405,
     * 0.000,0.000,0.000,8.464,8.538,8.612,8.667,8.720,
     * 0.000,0.000,0.000,0.000,8.436,8.517,8.571,8.638,
     * 0.000,0.000,0.000,0.000,8.410,8.449,8.449,8.504,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,7,L,3),I=1,8),L=1,NZEN)/
     * 7.207,7.444,7.654,7.801,7.928,8.057,8.167,8.286,
     * 8.000,8.188,8.348,8.422,8.473,8.540,8.580,8.644,
     * 7.983,8.170,8.336,8.407,8.446,8.497,8.513,8.577,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,7,L,3),I=1,8),L=1,NZEN)/
     * 7.083,7.344,7.585,7.739,7.852,7.968,8.053,8.173,
     * 7.823,8.057,8.267,8.369,8.427,8.468,8.476,8.525,
     * 7.808,8.045,8.270,8.386,8.450,8.481,8.473,8.501,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,8,L,3),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.297,8.382,8.441,8.501,8.556,
     * 0.000,0.000,0.000,0.000,8.573,8.609,8.641,8.670,
     * 0.000,0.000,0.000,0.000,8.449,8.498,8.538,8.583,
     * 0.000,0.000,0.000,0.000,0.000,0.000,8.378,8.423,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,8,L,3),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.130,8.255,8.344,8.428,8.504,
     * 0.000,0.000,0.000,8.479,8.542,8.601,8.650,8.692,
     * 0.000,0.000,0.000,8.382,8.418,8.491,8.548,8.607,
     * 0.000,0.000,0.000,0.000,8.348,8.382,8.389,8.449,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,8,L,3),I=1,8),L=1,NZEN)/
     * 7.462,7.697,7.893,8.037,8.179,8.292,8.391,8.486,
     * 8.041,8.201,8.326,8.415,8.497,8.576,8.639,8.698,
     * 8.013,8.158,8.272,8.336,8.389,8.477,8.544,8.618,
     * 7.979,8.130,8.253,8.310,8.346,8.394,8.410,8.476,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,8.509,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,8,L,3),I=1,8),L=1,NZEN)/
     * 7.456,7.688,7.893,8.045,8.176,8.297,8.398,8.504,
     * 7.909,8.086,8.243,8.338,8.412,8.501,8.567,8.645,
     * 7.890,8.064,8.220,8.303,8.358,8.438,8.490,8.574,
     * 7.862,8.049,8.223,8.324,8.387,8.431,8.438,8.487,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,8.618,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,8,L,3),I=1,8),L=1,NZEN)/
     * 7.525,7.767,7.993,8.146,8.262,8.371,8.459,8.563,
     * 7.728,7.941,8.140,8.260,8.338,8.422,8.479,8.566,
     * 0.000,0.000,0.000,0.000,8.330,8.396,8.430,8.511,
     * 0.000,0.000,0.000,0.000,0.000,0.000,8.458,8.487,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,9,L,3),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.199,8.290,8.358,8.427,8.490,
     * 0.000,0.000,0.000,0.000,8.537,8.579,8.617,8.648,
     * 0.000,0.000,0.000,0.000,8.417,8.470,8.511,8.561,
     * 0.000,0.000,0.000,0.000,0.000,8.371,8.356,8.401,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,9,L,3),I=1,8),L=1,NZEN)/
     * 0.000,0.000,7.965,8.072,8.196,8.292,8.387,8.471,
     * 0.000,0.000,0.000,8.338,8.410,8.481,8.548,8.607,
     * 0.000,0.000,0.000,8.246,8.288,8.371,8.444,8.520,
     * 0.000,8.111,8.185,8.207,8.225,8.267,8.286,8.360,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,9,L,3),I=1,8),L=1,NZEN)/
     * 7.569,7.782,7.962,8.097,8.236,8.342,8.449,8.545,
     * 7.829,7.979,8.097,8.199,8.299,8.398,8.491,8.576,
     * 7.801,7.933,8.037,8.114,8.188,8.297,8.394,8.494,
     * 7.766,7.901,8.017,8.083,8.137,8.207,8.250,8.346,
     * 7.700,7.864,8.017,8.158,8.265,8.338,8.373,8.367,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,9,L,3),I=1,8),L=1,NZEN)/
     * 7.702,7.899,8.079,8.223,8.352,8.458,8.561,8.655,
     * 7.646,7.806,7.950,8.079,8.190,8.312,8.428,8.542,
     * 7.623,7.773,7.910,8.017,8.107,8.230,8.342,8.470,
     * 7.594,7.754,7.904,8.017,8.093,8.179,8.238,8.350,
     * 7.542,7.728,7.914,8.104,8.238,8.336,8.396,8.417,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,9,L,3),I=1,8),L=1,NZEN)/
     * 7.834,8.017,8.199,8.348,8.470,8.565,8.656,8.748,
     * 7.436,7.620,7.802,7.973,8.111,8.255,8.394,8.535,
     * 7.412,7.587,7.762,7.918,8.033,8.182,8.316,8.468,
     * 7.387,7.571,7.757,7.917,8.025,8.134,8.223,8.358,
     * 7.336,7.549,7.771,8.009,8.176,8.297,8.382,8.430,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,10,L,3),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.179,8.267,8.336,8.407,8.471,
     * 0.000,0.000,0.000,0.000,8.521,8.566,8.606,8.643,
     * 0.000,0.000,0.000,0.000,8.403,8.458,8.501,8.554,
     * 0.000,0.000,0.000,0.000,0.000,8.360,8.346,8.394,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,10,L,3),I=1,8),L=1,NZEN)/
     * 7.647,7.825,7.952,8.053,8.170,8.267,8.367,8.459,
     * 7.917,8.037,8.107,8.170,8.250,8.340,8.428,8.508,
     * 7.886,7.986,8.041,8.079,8.130,8.230,8.324,8.420,
     * 7.846,7.950,8.017,8.041,8.068,8.124,8.164,8.258,
     * 7.769,7.900,8.009,8.107,8.190,8.250,8.274,8.260,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,10,L,3),I=1,8),L=1,NZEN)/
     * 7.651,7.825,7.971,8.090,8.220,8.324,8.438,8.538,
     * 7.576,7.710,7.819,7.934,8.053,8.179,8.305,8.423,
     * 7.545,7.657,7.746,7.840,7.934,8.072,8.207,8.342,
     * 7.509,7.624,7.725,7.802,7.871,7.968,8.053,8.188,
     * 7.442,7.585,7.723,7.873,7.994,8.090,8.155,8.185,
     * 7.410,7.567,7.726,7.925,8.083,8.204,8.292,8.318,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,10,L,3),I=1,8),L=1,NZEN)/
     * 7.752,7.904,8.049,8.179,8.305,8.401,8.513,8.621,
     * 7.332,7.474,7.613,7.772,7.926,8.079,8.243,8.400,
     * 7.303,7.420,7.537,7.682,7.816,7.985,8.161,8.332,
     * 7.270,7.389,7.512,7.637,7.741,7.881,8.017,8.196,
     * 7.215,7.364,7.524,7.720,7.873,7.995,8.100,8.173,
     * 7.185,7.350,7.530,7.777,7.967,8.114,8.238,8.305,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,10,L,3),I=1,8),L=1,NZEN)/
     * 7.708,7.844,7.991,8.134,8.260,8.367,8.494,8.632,
     * 7.064,7.241,7.427,7.645,7.844,8.025,8.228,8.423,
     * 7.021,7.164,7.322,7.542,7.734,7.941,8.158,8.369,
     * 6.987,7.117,7.265,7.456,7.611,7.810,8.013,8.241,
     * 6.934,7.093,7.276,7.525,7.713,7.866,8.013,8.143,
     * 6.905,7.083,7.286,7.585,7.812,7.989,8.152,8.262,
     * 6.865,7.072,7.314,7.697,7.989,8.207,8.403,8.519,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,11,L,3),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.255,8.336,8.396,8.458,8.515,
     * 0.000,0.000,0.000,0.000,8.544,8.584,8.621,8.652,
     * 0.000,0.000,0.000,0.000,8.422,8.474,8.516,8.566,
     * 0.000,0.000,0.000,0.000,0.000,0.000,8.360,8.405,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,11,L,3),I=1,8),L=1,NZEN)/
     * 7.726,7.877,7.980,8.068,8.173,8.267,8.369,8.462,
     * 7.804,7.914,7.978,8.045,8.134,8.233,8.336,8.428,
     * 7.775,7.862,7.908,7.951,8.009,8.121,8.230,8.342,
     * 7.732,7.824,7.883,7.910,7.942,8.009,8.064,8.179,
     * 7.656,7.775,7.873,7.973,8.061,8.127,8.167,8.167,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,11,L,3),I=1,8),L=1,NZEN)/
     * 7.660,7.791,7.900,8.009,8.130,8.230,8.354,8.464,
     * 7.352,7.471,7.572,7.702,7.844,7.988,8.146,8.294,
     * 7.318,7.408,7.484,7.598,7.718,7.880,8.053,8.217,
     * 7.281,7.369,7.450,7.538,7.622,7.753,7.884,8.057,
     * 7.212,7.330,7.452,7.609,7.744,7.855,7.952,8.013,
     * 7.176,7.312,7.453,7.659,7.830,7.969,8.086,8.146,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,11,L,3),I=1,8),L=1,NZEN)/
     * 7.599,7.704,7.812,7.933,8.061,8.173,8.316,8.467,
     * 7.029,7.164,7.305,7.489,7.678,7.856,8.064,8.265,
     * 6.983,7.083,7.190,7.377,7.559,7.763,7.990,8.207,
     * 6.947,7.029,7.127,7.281,7.422,7.620,7.833,8.068,
     * 6.892,7.000,7.137,7.342,7.517,7.667,7.819,7.955,
     * 6.860,6.986,7.143,7.400,7.614,7.787,7.956,8.072,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,11,L,3),I=1,8),L=1,NZEN)/
     * 7.342,7.433,7.551,7.708,7.865,8.013,8.201,8.420,
     * 6.672,6.830,7.017,7.260,7.497,7.715,7.974,8.236,
     * 6.598,6.721,6.878,7.140,7.387,7.634,7.916,8.193,
     * 6.549,6.634,6.759,6.997,7.212,7.479,7.772,8.072,
     * 6.496,6.603,6.757,7.021,7.246,7.441,7.653,7.884,
     * 6.465,6.592,6.769,7.083,7.346,7.559,7.782,7.967,
     * 6.422,6.582,6.796,7.196,7.524,7.775,8.033,8.217,
     * 0.000,0.000,0.000,0.000,6.422,6.910,7.394,7.810,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,12,L,3),I=1,8),L=1,NZEN)/
     * 0.000,0.000,0.000,8.348,8.423,8.471,8.524,8.573,
     * 0.000,0.000,0.000,0.000,8.581,8.611,8.637,8.663,
     * 0.000,0.000,0.000,0.000,0.000,8.501,8.533,8.574,
     * 0.000,0.000,0.000,0.000,0.000,0.000,8.377,8.413,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,12,L,3),I=1,8),L=1,NZEN)/
     * 7.761,7.899,7.989,8.068,8.173,8.262,8.365,8.456,
     * 7.744,7.846,7.905,7.973,8.061,8.161,8.270,8.369,
     * 7.710,7.792,7.834,7.877,7.937,8.049,8.164,8.281,
     * 7.668,7.754,7.808,7.834,7.867,7.938,8.000,8.117,
     * 7.592,7.703,7.797,7.897,7.985,8.057,8.104,8.114,
     * 7.554,7.679,7.794,7.943,8.068,8.170,8.236,8.246,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,12,L,3),I=1,8),L=1,NZEN)/
     * 7.591,7.699,7.789,7.893,8.017,8.121,8.255,8.378,
     * 7.167,7.272,7.362,7.500,7.654,7.814,7.997,8.167,
     * 7.134,7.210,7.276,7.396,7.526,7.705,7.901,8.090,
     * 7.093,7.173,7.246,7.338,7.433,7.579,7.732,7.930,
     * 7.029,7.134,7.243,7.407,7.551,7.678,7.797,7.884,
     * 6.993,7.114,7.246,7.458,7.641,7.795,7.936,8.017,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,12,L,3),I=1,8),L=1,NZEN)/
     * 7.396,7.471,7.559,7.684,7.824,7.955,8.127,8.316,
     * 6.771,6.892,7.029,7.228,7.439,7.641,7.884,8.121,
     * 6.718,6.801,6.902,7.107,7.318,7.548,7.810,8.064,
     * 6.681,6.741,6.825,6.995,7.161,7.394,7.653,7.928,
     * 6.621,6.709,6.831,7.049,7.241,7.413,7.599,7.780,
     * 6.593,6.696,6.839,7.107,7.336,7.531,7.734,7.888,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,6.396,6.866,7.324,7.703,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,12,L,3),I=1,8),L=1,NZEN)/
     * 7.017,7.079,7.182,7.354,7.534,7.713,7.950,8.225,
     * 6.377,6.512,6.690,6.946,7.207,7.453,7.752,8.064,
     * 6.290,6.393,6.542,6.822,7.100,7.375,7.695,8.021,
     * 6.228,6.286,6.394,6.659,6.913,7.215,7.558,7.908,
     * 6.173,6.248,6.384,6.661,6.909,7.134,7.391,7.688,
     * 6.143,6.238,6.394,6.722,7.009,7.248,7.512,7.749,
     * 6.097,6.225,6.423,6.834,7.188,7.464,7.760,7.996,
     * 0.000,0.000,0.000,0.000,6.100,6.615,7.137,7.604,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.182,6.574,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.104,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,1,L,1),I=9,16),L=1,NZEN)/
     * 8.511,8.515,8.540,8.571,8.618,8.626,8.649,8.676,
     * 8.538,8.517,8.516,8.531,8.563,8.566,8.583,8.610,
     * 8.394,8.387,8.396,8.412,8.447,8.449,8.471,8.501,
     * 8.117,8.124,8.146,8.167,8.212,8.220,8.250,8.292,
     * 0.000,0.000,0.000,0.000,7.656,7.674,7.723,7.792,
     * 0.000,0.000,0.000,0.000,0.000,7.371,7.367,7.439,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,7.064,
     * 6.989,7.009,6.996,6.894,6.844,6.742,6.702,6.688,
     * 0.000,0.000,0.000,0.000,6.076,6.149,6.292,6.346,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,1,L,1),I=9,16),L=1,NZEN)/
     * 8.413,8.459,8.511,8.554,8.611,8.649,8.692,8.733,
     * 8.310,8.360,8.413,8.467,8.533,8.575,8.623,8.667,
     * 8.170,8.233,8.297,8.348,8.417,8.459,8.511,8.560,
     * 7.899,7.976,8.049,8.111,8.185,8.233,8.294,8.354,
     * 7.619,7.599,7.567,7.589,7.634,7.691,7.771,7.857,
     * 7.623,7.566,7.494,7.422,7.378,7.382,7.413,7.505,
     * 7.662,7.591,7.498,7.371,7.276,7.179,7.121,7.114,
     * 6.775,6.860,6.903,6.837,6.816,6.750,6.737,6.737,
     * 0.000,0.000,0.000,0.000,6.057,6.164,6.338,6.407,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,1,L,1),I=9,16),L=1,NZEN)/
     * 8.336,8.417,8.489,8.556,8.633,8.711,8.782,8.844,
     * 8.146,8.258,8.360,8.447,8.545,8.629,8.709,8.775,
     * 8.017,8.140,8.248,8.336,8.435,8.517,8.600,8.670,
     * 7.765,7.900,8.021,8.111,8.215,8.301,8.389,8.470,
     * 7.474,7.522,7.548,7.607,7.686,7.777,7.882,7.987,
     * 7.473,7.477,7.455,7.423,7.415,7.462,7.529,7.638,
     * 7.509,7.500,7.455,7.362,7.294,7.233,7.199,7.215,
     * 6.676,6.818,6.908,6.872,6.875,6.834,6.838,6.844,
     * 0.000,0.000,0.000,0.000,6.130,6.270,6.462,6.537,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,1,L,1),I=9,16),L=1,NZEN)/
     * 8.274,8.412,8.528,8.639,8.755,8.890,9.009,9.086,
     * 8.053,8.223,8.367,8.496,8.628,8.758,8.873,8.960,
     * 7.941,8.117,8.267,8.393,8.525,8.654,8.772,8.864,
     * 7.712,7.900,8.061,8.185,8.320,8.450,8.574,8.678,
     * 7.380,7.505,7.595,7.702,7.822,7.955,8.090,8.220,
     * 7.364,7.431,7.458,7.481,7.521,7.627,7.742,7.878,
     * 7.398,7.444,7.444,7.386,7.348,7.332,7.334,7.393,
     * 6.646,6.841,6.974,6.964,6.987,6.972,6.993,7.009,
     * 0.000,0.000,0.000,0.000,6.272,6.442,6.658,6.735,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,1,L,1),I=9,16),L=1,NZEN)/
     * 8.223,8.431,8.607,8.782,8.960,9.164,9.342,9.418,
     * 7.964,8.196,8.393,8.562,8.735,8.911,9.061,9.161,
     * 7.860,8.097,8.299,8.465,8.638,8.813,8.964,9.072,
     * 7.646,7.895,8.104,8.270,8.442,8.618,8.776,8.898,
     * 7.283,7.486,7.642,7.795,7.959,8.137,8.310,8.462,
     * 7.250,7.382,7.464,7.544,7.637,7.799,7.959,8.127,
     * 7.279,7.384,7.433,7.412,7.410,7.444,7.486,7.590,
     * 6.577,6.827,7.009,7.029,7.079,7.093,7.137,7.170,
     * 0.000,0.000,0.000,6.025,6.375,6.583,6.826,6.914,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,2,L,1),I=9,16),L=1,NZEN)/
     * 8.465,8.479,8.513,8.550,8.603,8.616,8.642,8.669,
     * 8.516,8.504,8.511,8.525,8.559,8.561,8.580,8.604,
     * 8.373,8.373,8.389,8.405,8.439,8.444,8.465,8.497,
     * 8.093,8.107,8.137,8.158,8.207,8.215,8.246,8.286,
     * 0.000,0.000,0.000,0.000,7.647,7.666,7.716,7.785,
     * 0.000,0.000,0.000,0.000,0.000,7.365,7.362,7.433,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,7.064,
     * 6.969,6.991,6.983,6.884,6.836,6.736,6.697,6.683,
     * 0.000,0.000,0.000,0.000,6.064,6.140,6.283,6.338,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,2,L,1),I=9,16),L=1,NZEN)/
     * 8.369,8.423,8.486,8.537,8.602,8.641,8.686,8.723,
     * 8.310,8.362,8.418,8.468,8.531,8.572,8.618,8.662,
     * 8.170,8.230,8.297,8.348,8.417,8.456,8.507,8.554,
     * 7.898,7.975,8.049,8.107,8.185,8.228,8.288,8.344,
     * 7.638,7.613,7.576,7.592,7.632,7.686,7.763,7.847,
     * 7.642,7.584,7.512,7.435,7.386,7.384,7.408,7.497,
     * 7.681,7.610,7.516,7.386,7.290,7.190,7.127,7.117,
     * 6.782,6.865,6.907,6.841,6.820,6.754,6.743,6.741,
     * 0.000,0.000,0.000,0.000,6.053,6.164,6.334,6.403,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,2,L,1),I=9,16),L=1,NZEN)/
     * 8.318,8.401,8.480,8.548,8.624,8.694,8.757,8.812,
     * 8.173,8.279,8.375,8.456,8.549,8.624,8.696,8.754,
     * 8.041,8.158,8.262,8.342,8.435,8.511,8.588,8.650,
     * 7.792,7.919,8.033,8.117,8.215,8.292,8.375,8.447,
     * 7.522,7.559,7.573,7.621,7.688,7.772,7.869,7.965,
     * 7.524,7.521,7.493,7.450,7.431,7.467,7.521,7.621,
     * 7.561,7.545,7.494,7.394,7.322,7.253,7.212,7.217,
     * 6.714,6.851,6.936,6.894,6.893,6.848,6.849,6.851,
     * 0.000,0.000,0.000,0.000,6.143,6.279,6.468,6.539,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,2,L,1),I=9,16),L=1,NZEN)/
     * 8.272,8.394,8.501,8.597,8.699,8.810,8.905,8.979,
     * 8.107,8.260,8.393,8.505,8.625,8.740,8.841,8.917,
     * 7.993,8.152,8.290,8.401,8.521,8.636,8.740,8.821,
     * 7.769,7.941,8.086,8.196,8.316,8.430,8.539,8.631,
     * 7.462,7.566,7.636,7.721,7.822,7.938,8.057,8.173,
     * 7.452,7.504,7.519,7.521,7.543,7.625,7.719,7.838,
     * 7.486,7.520,7.509,7.438,7.391,7.360,7.346,7.384,
     * 6.724,6.907,7.029,7.009,7.025,6.998,7.009,7.013,
     * 0.000,0.000,0.000,0.000,6.303,6.465,6.671,6.740,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,2,L,1),I=9,16),L=1,NZEN)/
     * 8.210,8.393,8.544,8.686,8.833,9.009,9.158,9.246,
     * 8.045,8.253,8.430,8.577,8.732,8.885,9.013,9.100,
     * 7.943,8.155,8.332,8.479,8.631,8.785,8.916,9.009,
     * 7.729,7.953,8.140,8.283,8.436,8.589,8.727,8.832,
     * 7.407,7.573,7.697,7.822,7.959,8.114,8.262,8.393,
     * 7.384,7.493,7.553,7.600,7.663,7.793,7.923,8.064,
     * 7.415,7.501,7.533,7.494,7.476,7.486,7.504,7.571,
     * 6.687,6.922,7.090,7.097,7.137,7.137,7.167,7.182,
     * 0.000,0.000,0.000,6.072,6.417,6.614,6.847,6.922,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,3,L,1),I=9,16),L=1,NZEN)/
     * 8.410,8.431,8.473,8.517,8.579,8.594,8.624,8.650,
     * 8.504,8.494,8.502,8.520,8.555,8.559,8.577,8.605,
     * 8.360,8.362,8.380,8.400,8.438,8.441,8.464,8.496,
     * 8.079,8.097,8.130,8.155,8.201,8.212,8.243,8.286,
     * 0.000,0.000,0.000,7.635,7.643,7.663,7.714,7.784,
     * 0.000,0.000,0.000,0.000,0.000,0.000,7.360,7.433,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,7.064,
     * 6.957,6.980,6.976,6.878,6.833,6.732,6.695,6.683,
     * 0.000,0.000,0.000,0.000,6.057,6.134,6.279,6.336,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,3,L,1),I=9,16),L=1,NZEN)/
     * 8.314,8.377,8.447,8.509,8.585,8.625,8.671,8.706,
     * 8.356,8.396,8.446,8.489,8.548,8.581,8.621,8.659,
     * 8.217,8.270,8.326,8.371,8.431,8.465,8.509,8.551,
     * 7.945,8.013,8.079,8.130,8.201,8.238,8.290,8.342,
     * 7.694,7.657,7.613,7.620,7.651,7.696,7.767,7.846,
     * 7.700,7.632,7.551,7.467,7.412,7.400,7.417,7.500,
     * 7.739,7.658,7.556,7.420,7.318,7.212,7.143,7.127,
     * 6.834,6.908,6.944,6.872,6.846,6.774,6.757,6.751,
     * 0.000,0.000,0.000,0.000,6.079,6.182,6.348,6.413,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,3,L,1),I=9,16),L=1,NZEN)/
     * 8.303,8.391,8.476,8.543,8.621,8.680,8.737,8.782,
     * 8.255,8.342,8.423,8.491,8.572,8.632,8.692,8.738,
     * 8.124,8.220,8.307,8.375,8.456,8.517,8.581,8.631,
     * 7.876,7.981,8.076,8.146,8.233,8.297,8.367,8.427,
     * 7.641,7.651,7.642,7.666,7.711,7.776,7.858,7.941,
     * 7.646,7.625,7.580,7.519,7.481,7.490,7.521,7.605,
     * 7.683,7.648,7.583,7.468,7.386,7.303,7.250,7.241,
     * 6.811,6.932,7.004,6.952,6.943,6.888,6.881,6.875,
     * 0.000,0.000,0.000,0.000,6.179,6.307,6.491,6.554,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,3,L,1),I=9,16),L=1,NZEN)/
     * 8.322,8.427,8.517,8.595,8.681,8.764,8.835,8.890,
     * 8.179,8.305,8.417,8.509,8.611,8.700,8.780,8.836,
     * 8.061,8.193,8.310,8.400,8.501,8.591,8.673,8.733,
     * 7.836,7.976,8.093,8.185,8.288,8.378,8.465,8.534,
     * 7.618,7.672,7.698,7.740,7.799,7.885,7.979,8.068,
     * 7.621,7.642,7.631,7.594,7.579,7.612,7.657,7.744,
     * 7.656,7.665,7.632,7.540,7.476,7.417,7.380,7.375,
     * 6.833,6.994,7.100,7.068,7.076,7.033,7.033,7.025,
     * 0.000,0.000,0.000,6.021,6.324,6.471,6.666,6.726,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,3,L,1),I=9,16),L=1,NZEN)/
     * 8.283,8.422,8.535,8.633,8.739,8.859,8.957,9.029,
     * 8.114,8.283,8.428,8.547,8.673,8.797,8.900,8.970,
     * 8.009,8.182,8.328,8.444,8.569,8.691,8.797,8.872,
     * 7.801,7.982,8.130,8.243,8.367,8.489,8.600,8.683,
     * 7.572,7.678,7.744,7.816,7.903,8.017,8.134,8.236,
     * 7.569,7.637,7.663,7.657,7.671,7.742,7.817,7.919,
     * 7.602,7.655,7.659,7.592,7.550,7.522,7.508,7.522,
     * 6.838,7.041,7.182,7.167,7.188,7.167,7.179,7.173,
     * 0.000,0.000,0.000,6.143,6.462,6.632,6.843,6.901,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,4,L,1),I=9,16),L=1,NZEN)/
     * 8.398,8.420,8.464,8.508,8.569,8.584,8.613,8.638,
     * 8.504,8.496,8.505,8.524,8.560,8.565,8.584,8.614,
     * 8.362,8.365,8.386,8.403,8.442,8.447,8.473,8.507,
     * 8.083,8.104,8.137,8.164,8.212,8.223,8.255,8.299,
     * 0.000,0.000,0.000,7.644,7.659,7.681,7.735,7.807,
     * 0.000,0.000,0.000,0.000,0.000,0.000,7.380,7.456,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,7.068,
     * 6.966,6.987,6.981,6.883,6.836,6.738,6.702,6.689,
     * 0.000,0.000,0.000,0.000,6.072,6.149,6.294,6.352,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,4,L,1),I=9,16),L=1,NZEN)/
     * 8.294,8.356,8.430,8.497,8.581,8.620,8.666,8.697,
     * 8.427,8.453,8.490,8.524,8.575,8.598,8.630,8.663,
     * 8.286,8.324,8.369,8.405,8.459,8.483,8.520,8.555,
     * 8.017,8.068,8.124,8.167,8.230,8.258,8.301,8.348,
     * 7.772,7.721,7.665,7.662,7.686,7.723,7.785,7.857,
     * 7.778,7.696,7.604,7.511,7.449,7.428,7.438,7.515,
     * 7.816,7.721,7.609,7.465,7.356,7.241,7.167,7.143,
     * 6.917,6.977,7.000,6.919,6.886,6.806,6.782,6.771,
     * 0.000,0.000,0.000,0.000,6.124,6.217,6.377,6.436,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,4,L,1),I=9,16),L=1,NZEN)/
     * 8.283,8.371,8.459,8.534,8.623,8.677,8.730,8.763,
     * 8.360,8.418,8.479,8.529,8.593,8.636,8.682,8.715,
     * 8.225,8.294,8.362,8.412,8.479,8.521,8.567,8.604,
     * 7.981,8.057,8.124,8.179,8.250,8.297,8.352,8.398,
     * 7.795,7.770,7.731,7.722,7.738,7.780,7.841,7.911,
     * 7.802,7.754,7.687,7.602,7.545,7.521,7.522,7.587,
     * 7.841,7.781,7.693,7.561,7.464,7.362,7.292,7.262,
     * 6.939,7.033,7.086,7.017,6.997,6.928,6.911,6.896,
     * 0.000,0.000,0.000,0.000,6.228,6.336,6.505,6.561,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,4,L,1),I=9,16),L=1,NZEN)/
     * 8.322,8.423,8.516,8.590,8.677,8.740,8.798,8.829,
     * 8.290,8.380,8.461,8.528,8.609,8.670,8.727,8.761,
     * 8.164,8.260,8.346,8.412,8.491,8.554,8.614,8.651,
     * 7.960,8.049,8.127,8.190,8.272,8.332,8.396,8.442,
     * 7.844,7.844,7.821,7.804,7.810,7.851,7.905,7.968,
     * 7.853,7.837,7.794,7.717,7.666,7.639,7.631,7.676,
     * 7.892,7.864,7.801,7.682,7.594,7.507,7.444,7.407,
     * 6.983,7.114,7.193,7.140,7.130,7.076,7.064,7.045,
     * 0.000,0.000,0.000,6.076,6.358,6.483,6.661,6.713,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,4,L,1),I=9,16),L=1,NZEN)/
     * 8.356,8.459,8.548,8.619,8.702,8.775,8.834,8.865,
     * 8.238,8.356,8.458,8.538,8.630,8.709,8.777,8.812,
     * 8.127,8.250,8.352,8.427,8.516,8.595,8.665,8.702,
     * 7.958,8.068,8.152,8.220,8.307,8.382,8.452,8.497,
     * 7.873,7.904,7.905,7.892,7.900,7.943,7.991,8.045,
     * 7.882,7.900,7.883,7.820,7.782,7.764,7.756,7.785,
     * 7.919,7.925,7.889,7.786,7.713,7.642,7.590,7.549,
     * 7.029,7.193,7.301,7.262,7.267,7.225,7.223,7.201,
     * 0.000,0.000,0.000,6.207,6.498,6.640,6.829,6.879,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,5,L,1),I=9,16),L=1,NZEN)/
     * 8.449,8.464,8.500,8.535,8.588,8.597,8.620,8.644,
     * 8.516,8.505,8.513,8.529,8.562,8.565,8.584,8.614,
     * 8.377,8.377,8.396,8.412,8.449,8.453,8.476,8.509,
     * 8.100,8.121,8.152,8.176,8.223,8.233,8.265,8.307,
     * 7.825,7.744,7.666,7.658,7.679,7.700,7.754,7.823,
     * 7.829,7.709,7.590,7.486,7.413,7.386,7.398,7.476,
     * 7.867,7.736,7.593,7.430,7.305,7.173,7.086,7.068,
     * 6.984,7.004,6.994,6.896,6.849,6.747,6.708,6.692,
     * 0.000,0.000,0.000,0.000,6.097,6.170,6.314,6.369,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,5,L,1),I=9,16),L=1,NZEN)/
     * 8.344,8.396,8.461,8.524,8.602,8.636,8.679,8.708,
     * 8.490,8.502,8.528,8.554,8.599,8.615,8.640,8.667,
     * 8.350,8.375,8.410,8.438,8.484,8.500,8.529,8.559,
     * 8.083,8.124,8.167,8.201,8.255,8.276,8.316,8.354,
     * 7.846,7.781,7.713,7.701,7.716,7.745,7.801,7.868,
     * 7.850,7.756,7.653,7.551,7.483,7.455,7.458,7.528,
     * 7.889,7.782,7.658,7.505,7.389,7.267,7.185,7.158,
     * 6.996,7.041,7.053,6.963,6.924,6.834,6.803,6.787,
     * 0.000,0.000,0.000,0.000,6.164,6.253,6.403,6.456,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,5,L,1),I=9,16),L=1,NZEN)/
     * 8.290,8.367,8.452,8.530,8.626,8.678,8.732,8.757,
     * 8.452,8.489,8.531,8.568,8.622,8.648,8.680,8.702,
     * 8.318,8.364,8.412,8.447,8.504,8.531,8.567,8.592,
     * 8.083,8.130,8.173,8.212,8.274,8.305,8.350,8.384,
     * 7.930,7.876,7.812,7.779,7.773,7.797,7.843,7.902,
     * 7.939,7.866,7.777,7.674,7.601,7.556,7.538,7.587,
     * 7.977,7.893,7.786,7.637,7.526,7.408,7.328,7.286,
     * 7.061,7.134,7.161,7.079,7.045,6.966,6.939,6.918,
     * 0.000,0.000,0.000,6.021,6.283,6.377,6.531,6.581,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,5,L,1),I=9,16),L=1,NZEN)/
     * 8.255,8.354,8.453,8.542,8.646,8.708,8.767,8.790,
     * 8.400,8.458,8.515,8.563,8.630,8.672,8.712,8.731,
     * 8.279,8.340,8.398,8.446,8.513,8.553,8.595,8.617,
     * 8.104,8.146,8.182,8.223,8.286,8.326,8.377,8.407,
     * 8.037,8.000,7.944,7.891,7.865,7.871,7.894,7.939,
     * 8.049,7.999,7.928,7.829,7.760,7.699,7.660,7.674,
     * 8.086,8.029,7.938,7.799,7.697,7.587,7.508,7.452,
     * 7.143,7.238,7.286,7.217,7.196,7.127,7.104,7.083,
     * 0.000,0.000,0.000,6.161,6.422,6.526,6.686,6.732,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,5,L,1),I=9,16),L=1,NZEN)/
     * 8.210,8.312,8.407,8.493,8.597,8.666,8.726,8.747,
     * 8.332,8.398,8.455,8.507,8.574,8.625,8.669,8.688,
     * 8.243,8.303,8.350,8.394,8.456,8.505,8.550,8.569,
     * 8.149,8.179,8.190,8.212,8.253,8.292,8.330,8.354,
     * 8.137,8.121,8.079,8.017,7.978,7.955,7.942,7.949,
     * 8.149,8.124,8.072,7.980,7.919,7.854,7.806,7.775,
     * 8.188,8.152,8.083,7.954,7.862,7.763,7.690,7.626,
     * 7.196,7.320,7.391,7.334,7.326,7.272,7.262,7.241,
     * 0.000,6.057,6.072,6.276,6.537,6.653,6.822,6.870,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,6,L,1),I=9,16),L=1,NZEN)/
     * 8.512,8.517,8.545,8.571,8.613,8.619,8.639,8.662,
     * 8.539,8.525,8.530,8.543,8.573,8.574,8.592,8.619,
     * 8.403,8.401,8.417,8.430,8.462,8.462,8.483,8.515,
     * 8.130,8.146,8.176,8.199,8.241,8.246,8.276,8.316,
     * 7.830,7.753,7.679,7.676,7.701,7.721,7.773,7.841,
     * 7.834,7.713,7.592,7.490,7.422,7.400,7.417,7.496,
     * 7.870,7.736,7.593,7.430,7.305,7.173,7.086,7.072,
     * 7.009,7.025,7.017,6.914,6.865,6.760,6.718,6.698,
     * 0.000,0.000,0.000,0.000,6.124,6.193,6.334,6.384,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,6,L,1),I=9,16),L=1,NZEN)/
     * 8.423,8.462,8.517,8.571,8.641,8.669,8.707,8.731,
     * 8.561,8.562,8.579,8.598,8.634,8.644,8.666,8.687,
     * 8.425,8.438,8.461,8.480,8.521,8.530,8.555,8.581,
     * 8.158,8.188,8.225,8.250,8.297,8.312,8.346,8.380,
     * 7.901,7.832,7.761,7.749,7.766,7.790,7.841,7.903,
     * 7.905,7.801,7.691,7.588,7.517,7.491,7.498,7.566,
     * 7.944,7.827,7.694,7.535,7.415,7.288,7.204,7.176,
     * 7.072,7.107,7.107,7.013,6.966,6.869,6.831,6.810,
     * 0.000,0.000,0.000,0.000,6.220,6.299,6.444,6.491,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,6,L,1),I=9,16),L=1,NZEN)/
     * 8.332,8.400,8.476,8.553,8.647,8.693,8.744,8.767,
     * 8.553,8.573,8.601,8.628,8.675,8.692,8.716,8.731,
     * 8.420,8.449,8.483,8.509,8.558,8.575,8.602,8.621,
     * 8.185,8.217,8.250,8.279,8.334,8.354,8.389,8.417,
     * 8.033,7.961,7.884,7.844,7.833,7.849,7.889,7.941,
     * 8.037,7.949,7.847,7.734,7.654,7.604,7.583,7.627,
     * 8.079,7.976,7.854,7.696,7.576,7.450,7.362,7.314,
     * 7.179,7.230,7.243,7.149,7.107,7.017,6.980,6.951,
     * 0.000,0.000,0.000,6.104,6.354,6.436,6.582,6.623,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,6,L,1),I=9,16),L=1,NZEN)/
     * 8.223,8.314,8.412,8.509,8.626,8.692,8.758,8.780,
     * 8.504,8.539,8.579,8.615,8.671,8.700,8.730,8.739,
     * 8.386,8.423,8.461,8.497,8.554,8.581,8.614,8.625,
     * 8.230,8.241,8.248,8.274,8.324,8.354,8.393,8.413,
     * 8.188,8.124,8.049,7.973,7.929,7.914,7.918,7.951,
     * 8.199,8.124,8.033,7.921,7.839,7.760,7.704,7.702,
     * 8.238,8.155,8.045,7.893,7.780,7.655,7.565,7.497,
     * 7.267,7.338,7.365,7.283,7.253,7.173,7.146,7.117,
     * 6.037,6.068,6.049,6.236,6.479,6.567,6.714,6.756,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,6,L,1),I=9,16),L=1,NZEN)/
     * 8.097,8.196,8.292,8.394,8.519,8.603,8.684,8.714,
     * 8.442,8.470,8.497,8.530,8.585,8.622,8.656,8.668,
     * 8.375,8.389,8.400,8.425,8.471,8.502,8.533,8.544,
     * 8.322,8.305,8.276,8.267,8.283,8.299,8.320,8.328,
     * 8.328,8.279,8.215,8.130,8.079,8.025,7.985,7.964,
     * 8.338,8.283,8.207,8.104,8.033,7.948,7.884,7.829,
     * 8.382,8.316,8.223,8.079,7.977,7.863,7.778,7.702,
     * 7.312,7.405,7.450,7.382,7.367,7.307,7.294,7.274,
     * 6.134,6.170,6.149,6.328,6.563,6.664,6.818,6.866,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,7,L,1),I=9,16),L=1,NZEN)/
     * 8.530,8.533,8.558,8.583,8.626,8.632,8.653,8.676,
     * 8.550,8.533,8.535,8.547,8.576,8.575,8.591,8.617,
     * 8.413,8.408,8.420,8.431,8.464,8.464,8.483,8.513,
     * 8.143,8.155,8.182,8.199,8.241,8.246,8.274,8.314,
     * 7.838,7.759,7.685,7.679,7.699,7.718,7.768,7.833,
     * 7.838,7.718,7.595,7.494,7.423,7.398,7.412,7.487,
     * 7.877,7.742,7.598,7.433,7.307,7.176,7.086,7.072,
     * 7.017,7.033,7.021,6.920,6.868,6.762,6.718,6.698,
     * 0.000,0.000,0.000,0.000,6.124,6.193,6.332,6.380,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,7,L,1),I=9,16),L=1,NZEN)/
     * 8.468,8.501,8.549,8.600,8.667,8.692,8.726,8.751,
     * 8.602,8.599,8.610,8.624,8.658,8.666,8.686,8.708,
     * 8.468,8.476,8.494,8.509,8.545,8.554,8.576,8.602,
     * 8.204,8.228,8.260,8.281,8.326,8.336,8.365,8.401,
     * 7.927,7.857,7.787,7.776,7.794,7.816,7.866,7.927,
     * 7.931,7.822,7.707,7.605,7.537,7.512,7.521,7.589,
     * 7.968,7.846,7.709,7.549,7.428,7.301,7.215,7.190,
     * 7.111,7.140,7.137,7.037,6.990,6.889,6.847,6.822,
     * 0.000,0.000,0.000,0.000,6.248,6.324,6.465,6.509,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,7,L,1),I=9,16),L=1,NZEN)/
     * 8.401,8.459,8.528,8.598,8.686,8.725,8.771,8.790,
     * 8.627,8.638,8.659,8.680,8.722,8.736,8.757,8.772,
     * 8.497,8.519,8.545,8.567,8.610,8.623,8.647,8.666,
     * 8.258,8.286,8.314,8.338,8.387,8.405,8.436,8.464,
     * 8.072,7.999,7.920,7.887,7.884,7.901,7.943,7.994,
     * 8.079,7.981,7.873,7.760,7.681,7.638,7.626,7.675,
     * 8.117,8.009,7.879,7.717,7.594,7.467,7.377,7.332,
     * 7.246,7.288,7.292,7.193,7.149,7.049,7.009,6.975,
     * 0.000,0.000,0.000,6.155,6.405,6.481,6.621,6.659,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,7,L,1),I=9,16),L=1,NZEN)/
     * 8.322,8.403,8.490,8.576,8.682,8.740,8.799,8.816,
     * 8.587,8.615,8.648,8.679,8.729,8.752,8.779,8.785,
     * 8.465,8.496,8.529,8.560,8.613,8.635,8.664,8.674,
     * 8.297,8.305,8.314,8.336,8.386,8.408,8.442,8.462,
     * 8.241,8.167,8.086,8.013,7.968,7.957,7.966,8.000,
     * 8.250,8.167,8.072,7.951,7.866,7.786,7.732,7.736,
     * 8.290,8.196,8.079,7.922,7.803,7.674,7.579,7.512,
     * 7.342,7.403,7.423,7.334,7.297,7.212,7.176,7.140,
     * 6.111,6.134,6.107,6.290,6.530,6.615,6.758,6.793,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,7,L,1),I=9,16),L=1,NZEN)/
     * 8.253,8.334,8.418,8.508,8.619,8.688,8.754,8.773,
     * 8.512,8.535,8.558,8.590,8.644,8.677,8.709,8.717,
     * 8.439,8.450,8.458,8.481,8.528,8.555,8.584,8.593,
     * 8.384,8.358,8.326,8.316,8.334,8.348,8.369,8.380,
     * 8.387,8.330,8.258,8.170,8.114,8.057,8.017,7.999,
     * 8.398,8.334,8.250,8.140,8.064,7.975,7.906,7.851,
     * 8.439,8.365,8.265,8.117,8.009,7.890,7.800,7.719,
     * 7.380,7.462,7.498,7.423,7.403,7.338,7.322,7.297,
     * 6.204,6.230,6.204,6.375,6.605,6.699,6.848,6.893,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,8,L,1),I=9,16),L=1,NZEN)/
     * 8.493,8.504,8.534,8.568,8.618,8.627,8.650,8.675,
     * 8.539,8.522,8.528,8.539,8.569,8.568,8.584,8.610,
     * 8.400,8.396,8.410,8.423,8.455,8.456,8.476,8.504,
     * 8.124,8.140,8.164,8.185,8.228,8.230,8.260,8.299,
     * 7.839,7.756,7.676,7.663,7.677,7.695,7.745,7.812,
     * 7.843,7.723,7.600,7.493,7.417,7.384,7.389,7.464,
     * 7.881,7.747,7.603,7.438,7.312,7.179,7.090,7.068,
     * 6.999,7.017,7.009,6.907,6.857,6.752,6.710,6.692,
     * 0.000,0.000,0.000,0.000,6.100,6.173,6.312,6.364,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,8,L,1),I=9,16),L=1,NZEN)/
     * 8.468,8.504,8.554,8.604,8.672,8.697,8.732,8.757,
     * 8.577,8.579,8.593,8.613,8.649,8.661,8.683,8.707,
     * 8.444,8.456,8.479,8.497,8.537,8.547,8.572,8.599,
     * 8.176,8.204,8.238,8.262,8.310,8.324,8.358,8.396,
     * 7.909,7.839,7.768,7.755,7.771,7.794,7.846,7.910,
     * 7.913,7.808,7.696,7.593,7.522,7.494,7.498,7.567,
     * 7.952,7.834,7.701,7.543,7.423,7.297,7.210,7.185,
     * 7.076,7.111,7.114,7.017,6.972,6.874,6.835,6.814,
     * 0.000,0.000,0.000,0.000,6.215,6.297,6.441,6.489,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,8,L,1),I=9,16),L=1,NZEN)/
     * 8.476,8.528,8.590,8.649,8.727,8.762,8.802,8.822,
     * 8.609,8.626,8.652,8.679,8.725,8.745,8.772,8.794,
     * 8.479,8.505,8.538,8.563,8.610,8.631,8.662,8.686,
     * 8.236,8.270,8.305,8.336,8.389,8.412,8.450,8.483,
     * 8.033,7.970,7.899,7.874,7.879,7.902,7.949,8.004,
     * 8.041,7.949,7.847,7.741,7.668,7.632,7.624,7.680,
     * 8.079,7.975,7.852,7.696,7.579,7.456,7.369,7.332,
     * 7.207,7.258,7.270,7.176,7.134,7.037,7.000,6.970,
     * 0.000,0.000,0.000,6.127,6.382,6.465,6.611,6.650,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,8,L,1),I=9,16),L=1,NZEN)/
     * 8.517,8.581,8.649,8.708,8.787,8.827,8.866,8.879,
     * 8.590,8.628,8.669,8.705,8.761,8.790,8.822,8.836,
     * 8.467,8.511,8.554,8.589,8.645,8.675,8.710,8.728,
     * 8.272,8.305,8.332,8.365,8.422,8.453,8.494,8.522,
     * 8.176,8.117,8.045,7.993,7.969,7.979,8.009,8.053,
     * 8.185,8.114,8.025,7.912,7.834,7.774,7.738,7.765,
     * 8.223,8.140,8.033,7.879,7.765,7.643,7.555,7.498,
     * 7.316,7.387,7.415,7.326,7.292,7.207,7.173,7.137,
     * 6.072,6.107,6.093,6.286,6.535,6.623,6.769,6.803,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,8,L,1),I=9,16),L=1,NZEN)/
     * 8.593,8.655,8.716,8.767,8.838,8.880,8.916,8.922,
     * 8.551,8.602,8.650,8.693,8.757,8.798,8.836,8.847,
     * 8.449,8.496,8.535,8.576,8.638,8.679,8.719,8.732,
     * 8.330,8.344,8.348,8.373,8.420,8.458,8.500,8.522,
     * 8.307,8.262,8.199,8.124,8.079,8.061,8.057,8.079,
     * 8.318,8.265,8.188,8.083,8.009,7.934,7.879,7.860,
     * 8.358,8.294,8.199,8.057,7.950,7.836,7.751,7.681,
     * 7.378,7.470,7.515,7.441,7.418,7.350,7.328,7.297,
     * 6.173,6.220,6.207,6.396,6.640,6.740,6.894,6.932,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,9,L,1),I=9,16),L=1,NZEN)/
     * 8.436,8.455,8.494,8.535,8.594,8.609,8.637,8.661,
     * 8.524,8.512,8.519,8.533,8.566,8.568,8.585,8.611,
     * 8.384,8.382,8.400,8.415,8.450,8.453,8.474,8.504,
     * 8.107,8.124,8.152,8.176,8.220,8.228,8.258,8.299,
     * 7.833,7.748,7.667,7.653,7.667,7.687,7.738,7.806,
     * 7.836,7.717,7.595,7.487,7.412,7.377,7.380,7.455,
     * 7.874,7.743,7.602,7.438,7.312,7.179,7.090,7.064,
     * 6.982,7.004,6.996,6.897,6.849,6.746,6.705,6.690,
     * 0.000,0.000,0.000,0.000,6.086,6.161,6.303,6.356,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,9,L,1),I=9,16),L=1,NZEN)/
     * 8.447,8.489,8.543,8.593,8.659,8.688,8.724,8.752,
     * 8.511,8.525,8.551,8.579,8.623,8.641,8.669,8.699,
     * 8.373,8.400,8.435,8.462,8.509,8.526,8.558,8.591,
     * 8.104,8.143,8.190,8.223,8.279,8.301,8.342,8.386,
     * 7.840,7.781,7.719,7.713,7.735,7.767,7.825,7.895,
     * 7.844,7.752,7.651,7.554,7.489,7.465,7.476,7.549,
     * 7.883,7.778,7.656,7.505,7.391,7.272,7.193,7.167,
     * 6.995,7.045,7.057,6.969,6.931,6.842,6.812,6.796,
     * 0.000,0.000,0.000,0.000,6.170,6.258,6.412,6.465,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,9,L,1),I=9,16),L=1,NZEN)/
     * 8.534,8.584,8.639,8.687,8.748,8.784,8.822,8.848,
     * 8.511,8.550,8.594,8.633,8.690,8.724,8.763,8.795,
     * 8.378,8.428,8.479,8.519,8.576,8.611,8.653,8.688,
     * 8.130,8.190,8.246,8.288,8.352,8.389,8.439,8.484,
     * 7.914,7.873,7.822,7.814,7.833,7.873,7.933,8.000,
     * 7.919,7.850,7.765,7.673,7.611,7.592,7.600,7.668,
     * 7.957,7.874,7.769,7.626,7.519,7.408,7.334,7.305,
     * 7.083,7.155,7.188,7.104,7.072,6.991,6.964,6.943,
     * 0.000,0.000,0.000,6.053,6.318,6.415,6.572,6.620,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,9,L,1),I=9,16),L=1,NZEN)/
     * 8.644,8.695,8.744,8.785,8.841,8.886,8.927,8.954,
     * 8.508,8.574,8.635,8.689,8.759,8.811,8.861,8.896,
     * 8.386,8.459,8.525,8.577,8.647,8.700,8.753,8.791,
     * 8.167,8.241,8.305,8.358,8.431,8.484,8.544,8.591,
     * 8.000,7.980,7.942,7.931,7.946,7.992,8.053,8.121,
     * 8.009,7.963,7.897,7.811,7.754,7.738,7.744,7.806,
     * 8.045,7.988,7.901,7.766,7.667,7.566,7.496,7.464,
     * 7.193,7.290,7.336,7.262,7.238,7.164,7.137,7.111,
     * 0.000,6.009,6.021,6.233,6.498,6.601,6.759,6.801,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,9,L,1),I=9,16),L=1,NZEN)/
     * 8.730,8.784,8.829,8.873,8.933,8.993,9.041,9.072,
     * 8.526,8.615,8.694,8.758,8.840,8.909,8.970,9.009,
     * 8.412,8.505,8.585,8.649,8.731,8.798,8.861,8.903,
     * 8.217,8.305,8.377,8.438,8.519,8.587,8.654,8.706,
     * 8.083,8.083,8.057,8.045,8.061,8.114,8.176,8.246,
     * 8.090,8.068,8.021,7.941,7.891,7.880,7.884,7.942,
     * 8.127,8.093,8.025,7.899,7.810,7.721,7.656,7.621,
     * 7.290,7.408,7.473,7.408,7.391,7.326,7.305,7.274,
     * 6.057,6.146,6.173,6.389,6.657,6.771,6.936,6.972,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,10,L,1),I=9,16),L=1,NZEN)/
     * 8.422,8.441,8.481,8.525,8.584,8.598,8.625,8.651,
     * 8.520,8.509,8.519,8.535,8.569,8.574,8.595,8.624,
     * 8.378,8.380,8.400,8.417,8.453,8.458,8.481,8.516,
     * 8.100,8.121,8.152,8.179,8.228,8.236,8.267,8.310,
     * 7.828,7.744,7.666,7.655,7.673,7.695,7.747,7.818,
     * 7.831,7.713,7.593,7.487,7.413,7.384,7.391,7.468,
     * 7.870,7.739,7.598,7.436,7.312,7.179,7.093,7.072,
     * 6.978,6.999,6.991,6.892,6.844,6.745,6.708,6.694,
     * 0.000,0.000,0.000,0.000,6.086,6.164,6.307,6.362,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,10,L,1),I=9,16),L=1,NZEN)/
     * 8.441,8.484,8.538,8.584,8.645,8.675,8.713,8.743,
     * 8.430,8.462,8.502,8.542,8.595,8.624,8.662,8.700,
     * 8.292,8.334,8.384,8.423,8.480,8.509,8.550,8.593,
     * 8.021,8.079,8.140,8.188,8.255,8.286,8.334,8.387,
     * 7.753,7.710,7.662,7.669,7.704,7.747,7.816,7.894,
     * 7.757,7.679,7.590,7.505,7.450,7.441,7.462,7.544,
     * 7.794,7.704,7.595,7.453,7.348,7.238,7.167,7.152,
     * 6.907,6.970,6.996,6.918,6.888,6.809,6.787,6.778,
     * 0.000,0.000,0.000,0.000,6.130,6.225,6.387,6.449,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,10,L,1),I=9,16),L=1,NZEN)/
     * 8.525,8.576,8.628,8.675,8.734,8.781,8.827,8.867,
     * 8.382,8.449,8.515,8.573,8.645,8.699,8.754,8.797,
     * 8.253,8.328,8.400,8.458,8.533,8.585,8.642,8.692,
     * 8.000,8.086,8.167,8.228,8.310,8.364,8.430,8.489,
     * 7.757,7.747,7.722,7.740,7.782,7.843,7.921,8.004,
     * 7.760,7.717,7.653,7.583,7.540,7.547,7.576,7.662,
     * 7.798,7.741,7.657,7.531,7.438,7.346,7.286,7.274,
     * 6.938,7.033,7.086,7.021,7.000,6.935,6.919,6.908,
     * 0.000,0.000,0.000,0.000,6.250,6.364,6.533,6.591,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,10,L,1),I=9,16),L=1,NZEN)/
     * 8.605,8.671,8.728,8.785,8.852,8.926,8.991,9.045,
     * 8.389,8.489,8.577,8.655,8.747,8.827,8.902,8.960,
     * 8.270,8.378,8.473,8.549,8.638,8.719,8.796,8.859,
     * 8.041,8.158,8.255,8.334,8.428,8.509,8.592,8.665,
     * 7.789,7.822,7.829,7.869,7.931,8.013,8.104,8.199,
     * 7.788,7.780,7.744,7.697,7.676,7.713,7.766,7.864,
     * 7.823,7.801,7.742,7.632,7.554,7.483,7.438,7.442,
     * 7.029,7.158,7.233,7.179,7.167,7.107,7.093,7.079,
     * 0.000,0.000,0.000,6.158,6.442,6.566,6.741,6.792,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,10,L,1),I=9,16),L=1,NZEN)/
     * 8.651,8.754,8.838,8.923,9.017,9.134,9.236,9.303,
     * 8.455,8.590,8.705,8.807,8.921,9.029,9.127,9.193,
     * 8.350,8.490,8.609,8.708,8.819,8.929,9.025,9.100,
     * 8.134,8.283,8.408,8.505,8.619,8.728,8.831,8.915,
     * 7.833,7.919,7.972,8.045,8.137,8.246,8.356,8.465,
     * 7.820,7.850,7.844,7.834,7.848,7.927,8.017,8.130,
     * 7.851,7.862,7.830,7.740,7.679,7.638,7.615,7.649,
     * 7.121,7.281,7.378,7.338,7.336,7.290,7.281,7.267,
     * 0.000,6.009,6.083,6.334,6.631,6.774,6.961,7.009,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,11,L,1),I=9,16),L=1,NZEN)/
     * 8.459,8.473,8.508,8.543,8.594,8.605,8.630,8.656,
     * 8.526,8.515,8.521,8.538,8.573,8.576,8.597,8.627,
     * 8.386,8.386,8.403,8.420,8.458,8.462,8.486,8.521,
     * 8.107,8.127,8.158,8.182,8.230,8.238,8.272,8.316,
     * 7.830,7.747,7.667,7.658,7.677,7.699,7.753,7.824,
     * 7.834,7.714,7.593,7.489,7.415,7.386,7.396,7.473,
     * 7.872,7.740,7.598,7.436,7.312,7.179,7.093,7.072,
     * 6.985,7.004,6.996,6.897,6.849,6.748,6.710,6.696,
     * 0.000,0.000,0.000,0.000,6.093,6.167,6.312,6.367,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,11,L,1),I=9,16),L=1,NZEN)/
     * 8.442,8.483,8.531,8.574,8.629,8.663,8.701,8.736,
     * 8.365,8.408,8.456,8.505,8.567,8.605,8.650,8.695,
     * 8.228,8.281,8.340,8.389,8.453,8.491,8.539,8.589,
     * 7.955,8.025,8.097,8.152,8.225,8.267,8.326,8.384,
     * 7.675,7.646,7.611,7.631,7.678,7.731,7.808,7.893,
     * 7.679,7.613,7.534,7.459,7.412,7.417,7.450,7.542,
     * 7.716,7.636,7.537,7.405,7.305,7.204,7.143,7.137,
     * 6.839,6.913,6.946,6.876,6.853,6.781,6.765,6.761,
     * 0.000,0.000,0.000,0.000,6.097,6.201,6.371,6.436,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,11,L,1),I=9,16),L=1,NZEN)/
     * 8.453,8.516,8.574,8.629,8.695,8.760,8.820,8.873,
     * 8.276,8.365,8.447,8.522,8.609,8.679,8.747,8.806,
     * 8.149,8.246,8.336,8.410,8.497,8.567,8.638,8.702,
     * 7.895,8.009,8.107,8.185,8.279,8.350,8.428,8.500,
     * 7.609,7.632,7.636,7.683,7.750,7.829,7.923,8.017,
     * 7.609,7.590,7.547,7.501,7.480,7.515,7.569,7.671,
     * 7.645,7.612,7.548,7.439,7.360,7.286,7.241,7.248,
     * 6.812,6.931,7.000,6.950,6.943,6.888,6.883,6.881,
     * 0.000,0.000,0.000,0.000,6.201,6.326,6.509,6.575,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,11,L,1),I=9,16),L=1,NZEN)/
     * 8.471,8.577,8.667,8.753,8.848,8.960,9.057,9.130,
     * 8.276,8.408,8.524,8.624,8.735,8.842,8.938,9.013,
     * 8.164,8.303,8.423,8.521,8.630,8.737,8.836,8.916,
     * 7.939,8.090,8.217,8.316,8.427,8.534,8.637,8.729,
     * 7.603,7.695,7.755,7.834,7.931,8.041,8.155,8.270,
     * 7.587,7.618,7.616,7.614,7.632,7.714,7.808,7.930,
     * 7.619,7.631,7.602,7.517,7.459,7.420,7.401,7.446,
     * 6.873,7.033,7.140,7.104,7.107,7.064,7.064,7.064,
     * 0.000,0.000,0.000,6.083,6.391,6.538,6.733,6.794,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,11,L,1),I=9,16),L=1,NZEN)/
     * 8.480,8.645,8.785,8.926,9.072,9.250,9.407,9.477,
     * 8.297,8.479,8.631,8.766,8.906,9.049,9.173,9.255,
     * 8.199,8.384,8.539,8.671,8.812,8.956,9.083,9.167,
     * 7.992,8.188,8.350,8.480,8.620,8.765,8.898,8.997,
     * 7.613,7.773,7.894,8.013,8.146,8.292,8.438,8.569,
     * 7.567,7.656,7.701,7.754,7.825,7.957,8.093,8.236,
     * 7.593,7.652,7.660,7.609,7.579,7.587,7.606,7.694,
     * 6.905,7.114,7.253,7.241,7.260,7.243,7.258,7.265,
     * 0.000,0.000,0.000,6.228,6.560,6.737,6.955,7.017,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,12,L,1),I=9,16),L=1,NZEN)/
     * 8.505,8.511,8.538,8.566,8.612,8.620,8.643,8.669,
     * 8.528,8.511,8.513,8.530,8.563,8.567,8.588,8.615,
     * 8.386,8.382,8.396,8.412,8.449,8.452,8.474,8.507,
     * 8.107,8.121,8.146,8.170,8.217,8.225,8.258,8.299,
     * 7.838,7.748,7.664,7.648,7.661,7.681,7.734,7.803,
     * 7.843,7.720,7.595,7.487,7.410,7.375,7.377,7.452,
     * 7.881,7.747,7.602,7.438,7.312,7.179,7.090,7.068,
     * 6.988,7.004,6.994,6.893,6.844,6.744,6.705,6.690,
     * 0.000,0.000,0.000,0.000,6.083,6.155,6.299,6.354,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,12,L,1),I=9,16),L=1,NZEN)/
     * 8.435,8.476,8.522,8.565,8.619,8.655,8.696,8.736,
     * 8.312,8.362,8.417,8.468,8.534,8.576,8.625,8.671,
     * 8.170,8.233,8.297,8.350,8.418,8.461,8.515,8.563,
     * 7.899,7.977,8.053,8.114,8.190,8.238,8.299,8.358,
     * 7.626,7.603,7.569,7.593,7.639,7.697,7.778,7.865,
     * 7.630,7.571,7.497,7.425,7.378,7.386,7.420,7.513,
     * 7.667,7.595,7.501,7.373,7.276,7.182,7.121,7.114,
     * 6.786,6.866,6.905,6.838,6.817,6.751,6.740,6.740,
     * 0.000,0.000,0.000,0.000,6.061,6.170,6.342,6.412,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,12,L,1),I=9,16),L=1,NZEN)/
     * 8.378,8.453,8.522,8.588,8.661,8.737,8.807,8.868,
     * 8.173,8.279,8.377,8.461,8.556,8.636,8.714,8.779,
     * 8.045,8.158,8.265,8.348,8.446,8.525,8.605,8.676,
     * 7.787,7.919,8.033,8.124,8.225,8.307,8.394,8.474,
     * 7.500,7.542,7.560,7.618,7.695,7.785,7.888,7.991,
     * 7.500,7.498,7.470,7.436,7.423,7.468,7.533,7.642,
     * 7.537,7.521,7.471,7.373,7.303,7.238,7.201,7.217,
     * 6.702,6.838,6.922,6.883,6.883,6.838,6.840,6.846,
     * 0.000,0.000,0.000,0.000,6.140,6.276,6.467,6.539,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,12,L,1),I=9,16),L=1,NZEN)/
     * 8.346,8.479,8.592,8.700,8.814,8.951,9.072,9.149,
     * 8.152,8.310,8.447,8.568,8.695,8.820,8.931,9.013,
     * 8.045,8.210,8.350,8.467,8.592,8.718,8.832,8.921,
     * 7.822,7.999,8.149,8.265,8.391,8.516,8.635,8.737,
     * 7.455,7.583,7.677,7.779,7.896,8.025,8.158,8.286,
     * 7.428,7.490,7.512,7.539,7.583,7.691,7.808,7.944,
     * 7.459,7.497,7.490,7.425,7.382,7.365,7.365,7.431,
     * 6.736,6.921,7.045,7.025,7.037,7.013,7.025,7.033,
     * 0.000,0.000,0.000,6.013,6.334,6.497,6.704,6.772,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,12,L,1),I=9,16),L=1,NZEN)/
     * 8.314,8.521,8.695,8.872,9.049,9.253,9.428,9.500,
     * 8.146,8.360,8.538,8.693,8.853,9.017,9.152,9.246,
     * 8.049,8.267,8.449,8.601,8.760,8.922,9.061,9.161,
     * 7.848,8.076,8.262,8.413,8.571,8.736,8.883,8.995,
     * 7.442,7.645,7.803,7.944,8.097,8.267,8.431,8.575,
     * 7.378,7.505,7.582,7.667,7.768,7.926,8.083,8.246,
     * 7.401,7.491,7.526,7.497,7.486,7.521,7.565,7.679,
     * 6.731,6.968,7.137,7.140,7.173,7.173,7.201,7.225,
     * 0.000,0.000,0.000,6.130,6.481,6.677,6.908,6.983,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,1,L,2),I=9,16),L=1,NZEN)/
     * 8.562,8.572,8.602,8.637,8.689,8.706,8.736,8.773,
     * 8.609,8.594,8.600,8.618,8.655,8.662,8.682,8.714,
     * 8.504,8.498,8.509,8.529,8.569,8.574,8.592,8.623,
     * 8.283,8.294,8.316,8.340,8.391,8.396,8.412,8.442,
     * 0.000,0.000,7.923,7.911,7.931,7.939,7.958,7.997,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,7.673,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,7.723,7.684,7.572,7.524,7.380,7.238,7.111,
     * 0.000,0.000,6.422,6.602,6.856,6.882,6.908,6.827,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,1,L,2),I=9,16),L=1,NZEN)/
     * 8.462,8.512,8.568,8.619,8.682,8.728,8.779,8.830,
     * 8.382,8.436,8.496,8.551,8.622,8.669,8.721,8.772,
     * 8.279,8.344,8.408,8.465,8.539,8.584,8.632,8.682,
     * 8.061,8.143,8.220,8.281,8.364,8.407,8.453,8.505,
     * 7.865,7.847,7.816,7.849,7.909,7.956,8.004,8.064,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,7.739,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 7.496,7.563,7.581,7.505,7.489,7.378,7.267,7.155,
     * 6.290,6.336,6.326,6.543,6.827,6.889,6.945,6.883,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,1,L,2),I=9,16),L=1,NZEN)/
     * 8.387,8.473,8.550,8.624,8.707,8.793,8.873,8.944,
     * 8.223,8.336,8.442,8.537,8.640,8.729,8.812,8.882,
     * 8.127,8.253,8.364,8.458,8.562,8.647,8.726,8.796,
     * 7.928,8.068,8.190,8.286,8.398,8.480,8.554,8.626,
     * 7.709,7.765,7.794,7.867,7.964,8.045,8.121,8.199,
     * 7.780,7.780,7.751,7.727,7.731,7.771,7.803,7.877,
     * 7.943,7.924,7.866,7.769,7.708,7.621,7.525,7.486,
     * 7.384,7.511,7.576,7.530,7.538,7.453,7.358,7.258,
     * 6.193,6.299,6.336,6.585,6.894,6.984,7.061,7.009,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,1,L,2),I=9,16),L=1,NZEN)/
     * 8.330,8.474,8.597,8.713,8.836,8.980,9.107,9.193,
     * 8.134,8.307,8.459,8.592,8.732,8.867,8.985,9.076,
     * 8.057,8.238,8.393,8.522,8.661,8.794,8.909,9.000,
     * 7.885,8.076,8.241,8.369,8.512,8.640,8.751,8.846,
     * 7.622,7.760,7.860,7.977,8.114,8.238,8.342,8.447,
     * 7.675,7.740,7.762,7.796,7.854,7.951,8.029,8.134,
     * 7.832,7.870,7.859,7.797,7.766,7.727,7.667,7.677,
     * 7.360,7.537,7.645,7.627,7.656,7.594,7.515,7.428,
     * 6.193,6.352,6.435,6.709,7.037,7.158,7.255,7.212,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,1,L,2),I=9,16),L=1,NZEN)/
     * 8.281,8.496,8.677,8.857,9.041,9.248,9.428,9.511,
     * 8.045,8.281,8.481,8.657,8.836,9.013,9.158,9.262,
     * 7.979,8.220,8.422,8.593,8.772,8.946,9.093,9.196,
     * 7.820,8.072,8.283,8.453,8.633,8.805,8.947,9.057,
     * 7.524,7.742,7.912,8.072,8.250,8.420,8.559,8.685,
     * 7.550,7.683,7.764,7.860,7.976,8.127,8.250,8.382,
     * 7.703,7.800,7.838,7.816,7.822,7.835,7.818,7.877,
     * 7.276,7.512,7.669,7.682,7.738,7.709,7.650,7.584,
     * 6.121,6.334,6.465,6.775,7.134,7.290,7.415,7.386,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,2,L,2),I=9,16),L=1,NZEN)/
     * 8.515,8.531,8.569,8.613,8.673,8.694,8.728,8.764,
     * 8.588,8.579,8.588,8.610,8.649,8.656,8.677,8.708,
     * 8.479,8.481,8.500,8.521,8.563,8.569,8.588,8.617,
     * 8.255,8.274,8.303,8.330,8.382,8.387,8.407,8.436,
     * 0.000,0.000,0.000,0.000,7.922,7.930,7.950,7.989,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,7.666,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,6.410,6.591,6.844,6.872,6.899,6.820,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,2,L,2),I=9,16),L=1,NZEN)/
     * 8.415,8.476,8.544,8.601,8.672,8.719,8.769,8.818,
     * 8.382,8.436,8.494,8.551,8.622,8.667,8.716,8.764,
     * 8.276,8.340,8.408,8.464,8.538,8.581,8.627,8.674,
     * 8.057,8.137,8.215,8.279,8.360,8.401,8.447,8.496,
     * 7.881,7.857,7.822,7.849,7.904,7.949,7.996,8.053,
     * 7.960,7.897,7.814,7.741,7.701,7.688,7.677,7.727,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 7.502,7.568,7.584,7.508,7.491,7.382,7.270,7.158,
     * 6.301,6.344,6.328,6.543,6.826,6.886,6.942,6.879,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,2,L,2),I=9,16),L=1,NZEN)/
     * 8.367,8.456,8.540,8.613,8.695,8.772,8.845,8.910,
     * 8.246,8.354,8.453,8.542,8.639,8.720,8.796,8.859,
     * 8.152,8.270,8.375,8.461,8.560,8.637,8.711,8.774,
     * 7.950,8.083,8.199,8.288,8.393,8.468,8.538,8.601,
     * 7.755,7.797,7.816,7.876,7.960,8.033,8.104,8.170,
     * 7.831,7.823,7.786,7.750,7.743,7.769,7.790,7.854,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 7.422,7.542,7.602,7.551,7.555,7.465,7.365,7.260,
     * 6.233,6.328,6.358,6.601,6.906,6.990,7.064,7.004,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,2,L,2),I=9,16),L=1,NZEN)/
     * 8.324,8.453,8.563,8.667,8.775,8.895,8.998,9.083,
     * 8.185,8.340,8.477,8.597,8.724,8.844,8.947,9.029,
     * 8.107,8.270,8.410,8.526,8.653,8.769,8.869,8.949,
     * 7.935,8.111,8.258,8.373,8.502,8.615,8.711,8.792,
     * 7.702,7.815,7.892,7.989,8.107,8.212,8.303,8.393,
     * 7.761,7.809,7.816,7.828,7.865,7.940,7.999,8.086,
     * 7.921,7.944,7.920,7.845,7.802,7.745,7.671,7.658,
     * 7.436,7.602,7.702,7.670,7.688,7.616,7.524,7.425,
     * 6.265,6.410,6.479,6.745,7.068,7.176,7.265,7.207,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,2,L,2),I=9,16),L=1,NZEN)/
     * 8.262,8.447,8.603,8.750,8.900,9.079,9.233,9.326,
     * 8.121,8.332,8.511,8.661,8.818,8.972,9.100,9.190,
     * 8.053,8.267,8.449,8.597,8.753,8.904,9.029,9.117,
     * 7.895,8.121,8.310,8.456,8.614,8.760,8.881,8.974,
     * 7.638,7.818,7.953,8.086,8.236,8.378,8.494,8.597,
     * 7.682,7.788,7.844,7.903,7.983,8.100,8.193,8.297,
     * 7.837,7.913,7.932,7.891,7.878,7.862,7.818,7.838,
     * 7.386,7.604,7.748,7.747,7.790,7.742,7.667,7.581,
     * 6.212,6.405,6.517,6.817,7.167,7.312,7.427,7.380,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,3,L,2),I=9,16),L=1,NZEN)/
     * 8.458,8.483,8.529,8.581,8.649,8.672,8.709,8.745,
     * 8.575,8.569,8.583,8.605,8.646,8.654,8.676,8.708,
     * 8.467,8.471,8.493,8.516,8.560,8.566,8.585,8.617,
     * 8.243,8.262,8.294,8.324,8.378,8.386,8.405,8.436,
     * 0.000,0.000,0.000,0.000,7.916,7.927,7.948,7.988,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,6.401,6.583,6.838,6.866,6.895,6.817,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,3,L,2),I=9,16),L=1,NZEN)/
     * 8.362,8.430,8.508,8.574,8.655,8.702,8.754,8.800,
     * 8.428,8.473,8.525,8.575,8.639,8.676,8.719,8.761,
     * 8.322,8.378,8.438,8.487,8.554,8.590,8.630,8.671,
     * 8.104,8.173,8.246,8.299,8.375,8.410,8.450,8.493,
     * 7.936,7.903,7.859,7.876,7.922,7.959,7.998,8.049,
     * 8.017,7.944,7.855,7.773,7.726,7.704,7.684,7.728,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 7.554,7.612,7.621,7.539,7.519,7.401,7.283,7.167,
     * 0.000,6.386,6.362,6.572,6.851,6.905,6.955,6.888,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,3,L,2),I=9,16),L=1,NZEN)/
     * 8.350,8.444,8.534,8.609,8.694,8.760,8.823,8.877,
     * 8.326,8.418,8.504,8.577,8.663,8.728,8.790,8.841,
     * 8.230,8.330,8.420,8.493,8.580,8.642,8.702,8.752,
     * 8.029,8.143,8.241,8.316,8.408,8.468,8.525,8.576,
     * 7.873,7.886,7.879,7.915,7.977,8.033,8.086,8.143,
     * 7.950,7.924,7.870,7.813,7.787,7.787,7.782,7.831,
     * 8.114,8.068,7.989,7.872,7.793,7.683,7.567,7.498,
     * 7.519,7.622,7.670,7.609,7.603,7.502,7.393,7.279,
     * 6.322,6.400,6.410,6.642,6.939,7.017,7.083,7.017,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,3,L,2),I=9,16),L=1,NZEN)/
     * 8.373,8.483,8.579,8.662,8.754,8.845,8.924,8.990,
     * 8.250,8.382,8.498,8.597,8.704,8.798,8.880,8.942,
     * 8.164,8.303,8.425,8.520,8.627,8.718,8.797,8.856,
     * 7.991,8.137,8.262,8.356,8.465,8.553,8.627,8.688,
     * 7.851,7.909,7.937,7.989,8.064,8.143,8.207,8.272,
     * 7.929,7.945,7.925,7.890,7.883,7.907,7.918,7.972,
     * 8.090,8.086,8.041,7.943,7.881,7.793,7.692,7.631,
     * 7.543,7.688,7.770,7.724,7.731,7.642,7.538,7.422,
     * 6.365,6.483,6.526,6.775,7.083,7.176,7.250,7.182,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,3,L,2),I=9,16),L=1,NZEN)/
     * 8.336,8.480,8.599,8.705,8.819,8.947,9.053,9.137,
     * 8.188,8.364,8.513,8.638,8.772,8.901,9.009,9.083,
     * 8.114,8.294,8.446,8.568,8.701,8.826,8.929,9.004,
     * 7.957,8.143,8.299,8.418,8.553,8.672,8.770,8.846,
     * 7.795,7.908,7.981,8.068,8.173,8.283,8.369,8.450,
     * 7.865,7.927,7.945,7.946,7.973,8.037,8.083,8.155,
     * 8.021,8.064,8.053,7.983,7.944,7.889,7.810,7.776,
     * 7.534,7.720,7.835,7.810,7.833,7.763,7.670,7.561,
     * 6.375,6.533,6.612,6.882,7.207,7.322,7.413,7.348,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,4,L,2),I=9,16),L=1,NZEN)/
     * 8.446,8.474,8.524,8.574,8.640,8.662,8.697,8.732,
     * 8.575,8.571,8.585,8.609,8.650,8.660,8.682,8.717,
     * 8.470,8.477,8.500,8.524,8.568,8.574,8.594,8.627,
     * 8.248,8.272,8.305,8.336,8.391,8.398,8.417,8.452,
     * 0.000,0.000,0.000,7.907,7.936,7.948,7.970,8.013,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,7.690,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,6.420,6.600,6.854,6.883,6.912,6.834,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,4,L,2),I=9,16),L=1,NZEN)/
     * 8.342,8.410,8.489,8.562,8.650,8.696,8.749,8.791,
     * 8.498,8.529,8.569,8.610,8.667,8.695,8.729,8.766,
     * 8.394,8.436,8.484,8.524,8.582,8.607,8.639,8.675,
     * 8.176,8.233,8.292,8.336,8.407,8.430,8.461,8.497,
     * 8.013,7.966,7.910,7.918,7.957,7.984,8.013,8.061,
     * 8.097,8.009,7.907,7.817,7.763,7.732,7.704,7.743,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,7.413,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,6.458,6.423,6.623,6.895,6.940,6.982,6.908,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,4,L,2),I=9,16),L=1,NZEN)/
     * 8.330,8.423,8.517,8.599,8.694,8.754,8.814,8.857,
     * 8.427,8.491,8.555,8.613,8.685,8.732,8.779,8.816,
     * 8.326,8.400,8.468,8.525,8.600,8.644,8.688,8.725,
     * 8.127,8.210,8.283,8.342,8.423,8.467,8.508,8.545,
     * 8.025,8.000,7.960,7.962,7.995,8.029,8.064,8.107,
     * 8.107,8.053,7.975,7.891,7.841,7.809,7.774,7.803,
     * 8.274,8.201,8.100,7.962,7.866,7.736,7.603,7.513,
     * 7.646,7.725,7.750,7.671,7.651,7.538,7.417,7.294,
     * 0.000,6.504,6.490,6.703,6.985,7.045,7.093,7.017,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,4,L,2),I=9,16),L=1,NZEN)/
     * 8.369,8.476,8.575,8.656,8.750,8.820,8.883,8.925,
     * 8.352,8.450,8.539,8.613,8.700,8.766,8.826,8.865,
     * 8.260,8.362,8.453,8.525,8.614,8.678,8.733,8.772,
     * 8.097,8.196,8.279,8.348,8.438,8.498,8.551,8.590,
     * 8.072,8.072,8.041,8.033,8.053,8.093,8.121,8.158,
     * 8.158,8.137,8.083,8.004,7.957,7.918,7.869,7.881,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 7.692,7.805,7.857,7.792,7.784,7.680,7.565,7.436,
     * 6.524,6.599,6.604,6.826,7.114,7.185,7.243,7.164,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,4,L,2),I=9,16),L=1,NZEN)/
     * 8.398,8.511,8.607,8.686,8.776,8.858,8.925,8.967,
     * 8.301,8.427,8.535,8.623,8.724,8.809,8.879,8.920,
     * 8.215,8.346,8.456,8.540,8.639,8.721,8.787,8.826,
     * 8.079,8.201,8.297,8.375,8.473,8.548,8.607,8.645,
     * 8.086,8.117,8.114,8.107,8.130,8.170,8.196,8.228,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 7.724,7.870,7.949,7.899,7.905,7.813,7.703,7.573,
     * 6.571,6.676,6.706,6.941,7.238,7.324,7.391,7.312,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,5,L,2),I=9,16),L=1,NZEN)/
     * 8.497,8.517,8.558,8.600,8.659,8.675,8.706,8.739,
     * 8.588,8.581,8.592,8.615,8.655,8.663,8.685,8.719,
     * 8.484,8.489,8.509,8.531,8.574,8.580,8.600,8.631,
     * 8.265,8.288,8.320,8.350,8.403,8.408,8.427,8.459,
     * 0.000,0.000,7.922,7.922,7.955,7.968,7.991,8.029,
     * 0.000,0.000,0.000,0.000,0.000,0.000,7.675,7.713,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,7.117,
     * 0.000,0.000,6.446,6.626,6.879,6.905,6.932,6.850,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,5,L,2),I=9,16),L=1,NZEN)/
     * 8.391,8.447,8.519,8.588,8.673,8.713,8.763,8.801,
     * 8.559,8.577,8.606,8.639,8.690,8.710,8.739,8.769,
     * 8.456,8.484,8.522,8.554,8.606,8.624,8.651,8.679,
     * 8.243,8.286,8.330,8.369,8.431,8.447,8.473,8.502,
     * 0.000,0.000,0.000,7.955,7.986,8.004,8.033,8.068,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,6.481,6.673,6.937,6.974,7.009,6.929,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,5,L,2),I=9,16),L=1,NZEN)/
     * 8.334,8.418,8.508,8.594,8.698,8.754,8.814,8.851,
     * 8.520,8.561,8.609,8.652,8.713,8.746,8.781,8.806,
     * 8.417,8.467,8.520,8.562,8.625,8.655,8.688,8.713,
     * 8.223,8.279,8.332,8.377,8.446,8.474,8.505,8.531,
     * 0.000,0.000,0.000,0.000,8.025,8.045,8.061,8.093,
     * 0.000,0.000,0.000,0.000,0.000,0.000,7.785,7.799,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,6.576,6.772,7.037,7.083,7.121,7.033,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,5,L,2),I=9,16),L=1,NZEN)/
     * 8.294,8.401,8.507,8.604,8.719,8.788,8.854,8.886,
     * 8.461,8.528,8.592,8.647,8.720,8.768,8.813,8.836,
     * 8.365,8.436,8.501,8.556,8.631,8.675,8.715,8.738,
     * 8.228,8.281,8.326,8.377,8.452,8.491,8.529,8.551,
     * 8.267,8.223,8.161,8.114,8.097,8.100,8.100,8.121,
     * 0.000,0.000,0.000,0.000,0.000,0.000,7.886,7.868,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,6.717,6.910,7.173,7.223,7.262,7.176,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,5,L,2),I=9,16),L=1,NZEN)/
     * 8.238,8.354,8.464,8.563,8.679,8.760,8.834,8.866,
     * 8.377,8.456,8.526,8.590,8.672,8.736,8.790,8.815,
     * 8.305,8.378,8.439,8.498,8.577,8.635,8.683,8.708,
     * 8.246,8.283,8.307,8.342,8.403,8.449,8.486,8.508,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,6.811,7.009,7.272,7.334,7.382,7.297,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,6,L,2),I=9,16),L=1,NZEN)/
     * 8.562,8.572,8.602,8.635,8.685,8.697,8.725,8.758,
     * 8.613,8.602,8.612,8.629,8.666,8.672,8.691,8.724,
     * 8.512,8.512,8.528,8.547,8.587,8.591,8.609,8.639,
     * 8.297,8.316,8.346,8.369,8.418,8.422,8.438,8.468,
     * 0.000,0.000,7.940,7.943,7.980,7.990,8.009,8.049,
     * 0.000,0.000,0.000,0.000,0.000,0.000,7.696,7.733,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,6.474,6.653,6.905,6.928,6.953,6.867,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,6,L,2),I=9,16),L=1,NZEN)/
     * 8.471,8.516,8.575,8.636,8.714,8.747,8.790,8.825,
     * 8.633,8.639,8.659,8.683,8.727,8.740,8.763,8.790,
     * 8.531,8.549,8.575,8.599,8.644,8.656,8.678,8.702,
     * 8.320,8.352,8.391,8.420,8.477,8.487,8.505,8.530,
     * 0.000,0.000,8.009,8.004,8.037,8.053,8.072,8.104,
     * 0.000,0.000,0.000,0.000,0.000,0.000,7.766,7.795,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,6.550,6.735,6.993,7.021,7.053,6.964,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,6,L,2),I=9,16),L=1,NZEN)/
     * 8.378,8.450,8.531,8.616,8.719,8.772,8.830,8.862,
     * 8.620,8.645,8.679,8.713,8.766,8.787,8.814,8.833,
     * 8.519,8.553,8.592,8.625,8.680,8.700,8.724,8.742,
     * 8.326,8.367,8.407,8.442,8.505,8.522,8.544,8.562,
     * 0.000,0.000,0.000,8.079,8.086,8.097,8.111,8.134,
     * 0.000,0.000,0.000,0.000,0.000,0.000,7.832,7.841,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,6.671,6.854,7.111,7.143,7.170,7.079,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,6,L,2),I=9,16),L=1,NZEN)/
     * 8.260,8.360,8.464,8.572,8.697,8.772,8.846,8.878,
     * 8.560,8.604,8.651,8.696,8.763,8.797,8.832,8.844,
     * 8.467,8.515,8.560,8.603,8.669,8.702,8.734,8.746,
     * 8.346,8.369,8.387,8.423,8.487,8.516,8.544,8.558,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,6.807,6.981,7.228,7.265,7.294,7.201,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,6,L,2),I=9,16),L=1,NZEN)/
     * 8.111,8.225,8.336,8.458,8.600,8.699,8.793,8.835,
     * 8.467,8.512,8.553,8.605,8.680,8.731,8.777,8.796,
     * 0.000,8.447,8.471,8.513,8.580,8.624,8.665,8.683,
     * 0.000,0.000,8.373,8.378,8.413,8.439,8.461,8.473,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,7.057,7.299,7.344,7.380,7.292,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,7,L,2),I=9,16),L=1,NZEN)/
     * 8.579,8.588,8.618,8.650,8.700,8.712,8.738,8.772,
     * 8.624,8.612,8.618,8.634,8.669,8.674,8.692,8.723,
     * 8.525,8.522,8.537,8.553,8.589,8.591,8.607,8.636,
     * 8.310,8.326,8.352,8.373,8.418,8.422,8.436,8.465,
     * 0.000,0.000,7.945,7.945,7.979,7.987,8.004,8.041,
     * 0.000,0.000,0.000,0.000,0.000,0.000,7.690,7.725,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,6.474,6.654,6.907,6.928,6.951,6.865,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,7,L,2),I=9,16),L=1,NZEN)/
     * 8.517,8.555,8.610,8.667,8.741,8.772,8.812,8.846,
     * 8.675,8.675,8.690,8.711,8.751,8.763,8.785,8.812,
     * 8.576,8.585,8.606,8.628,8.671,8.680,8.699,8.724,
     * 8.365,8.393,8.427,8.452,8.505,8.512,8.528,8.553,
     * 0.000,0.000,8.037,8.033,8.068,8.079,8.100,8.130,
     * 0.000,0.000,0.000,0.000,0.000,0.000,7.792,7.820,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,6.579,6.762,7.021,7.049,7.076,6.983,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,7,L,2),I=9,16),L=1,NZEN)/
     * 8.447,8.509,8.584,8.661,8.757,8.803,8.856,8.885,
     * 8.695,8.713,8.739,8.766,8.814,8.833,8.857,8.876,
     * 8.599,8.624,8.655,8.682,8.732,8.748,8.769,8.787,
     * 8.405,8.439,8.476,8.505,8.562,8.576,8.594,8.613,
     * 0.000,0.000,0.000,8.130,8.143,8.155,8.167,8.190,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,6.728,6.908,7.161,7.190,7.215,7.117,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,7,L,2),I=9,16),L=1,NZEN)/
     * 8.360,8.449,8.543,8.638,8.754,8.820,8.886,8.911,
     * 8.643,8.680,8.721,8.760,8.822,8.849,8.879,8.889,
     * 8.550,8.590,8.631,8.668,8.731,8.757,8.783,8.792,
     * 8.417,8.439,8.456,8.490,8.550,8.573,8.597,8.609,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,7.037,7.281,7.314,7.338,7.238,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,7,L,2),I=9,16),L=1,NZEN)/
     * 8.267,8.365,8.464,8.569,8.696,8.780,8.860,8.890,
     * 8.539,8.577,8.616,8.665,8.736,8.782,8.826,8.841,
     * 8.484,8.511,8.533,8.573,8.637,8.678,8.714,8.729,
     * 0.000,8.452,8.427,8.431,8.467,8.491,8.513,8.524,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,7.104,7.338,7.380,7.410,7.318,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,8,L,2),I=9,16),L=1,NZEN)/
     * 8.542,8.556,8.593,8.632,8.689,8.705,8.736,8.770,
     * 8.613,8.600,8.607,8.625,8.662,8.667,8.686,8.715,
     * 8.508,8.507,8.521,8.540,8.580,8.582,8.598,8.626,
     * 8.290,8.307,8.334,8.358,8.407,8.408,8.422,8.452,
     * 0.000,0.000,7.932,7.928,7.956,7.963,7.981,8.017,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,7.698,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,6.447,6.629,6.883,6.906,6.930,6.847,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,8,L,2),I=9,16),L=1,NZEN)/
     * 8.516,8.556,8.612,8.669,8.744,8.776,8.818,8.852,
     * 8.651,8.656,8.674,8.698,8.742,8.757,8.782,8.810,
     * 8.550,8.563,8.588,8.613,8.660,8.672,8.694,8.721,
     * 8.336,8.367,8.403,8.433,8.487,8.498,8.519,8.545,
     * 0.000,0.000,8.017,8.013,8.045,8.057,8.079,8.114,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,6.538,6.727,6.988,7.021,7.049,6.963,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,8,L,2),I=9,16),L=1,NZEN)/
     * 8.522,8.581,8.649,8.714,8.798,8.839,8.886,8.915,
     * 8.679,8.702,8.732,8.764,8.816,8.841,8.871,8.896,
     * 8.582,8.614,8.650,8.681,8.733,8.756,8.782,8.806,
     * 8.386,8.428,8.468,8.504,8.565,8.584,8.609,8.631,
     * 0.000,0.000,8.130,8.117,8.140,8.155,8.176,8.204,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,7.901,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,6.695,6.881,7.140,7.173,7.204,7.111,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,8,L,2),I=9,16),L=1,NZEN)/
     * 8.559,8.629,8.704,8.772,8.857,8.905,8.952,8.974,
     * 8.655,8.700,8.747,8.788,8.851,8.885,8.920,8.940,
     * 8.560,8.611,8.659,8.701,8.765,8.797,8.829,8.847,
     * 8.403,8.447,8.484,8.524,8.589,8.620,8.649,8.668,
     * 0.000,0.000,0.000,0.000,8.212,8.217,8.223,8.243,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,6.856,7.033,7.288,7.324,7.350,7.253,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,8,L,2),I=9,16),L=1,NZEN)/
     * 8.625,8.697,8.768,8.829,8.908,8.959,9.004,9.017,
     * 8.599,8.660,8.718,8.771,8.845,8.893,8.935,8.951,
     * 8.516,8.575,8.628,8.679,8.753,8.798,8.838,8.853,
     * 0.000,8.458,8.473,8.511,8.574,8.614,8.647,8.664,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,7.127,7.377,7.422,7.456,7.364,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,9,L,2),I=9,16),L=1,NZEN)/
     * 8.486,8.508,8.553,8.600,8.665,8.686,8.721,8.755,
     * 8.597,8.589,8.600,8.619,8.657,8.664,8.684,8.714,
     * 8.491,8.494,8.512,8.533,8.574,8.577,8.595,8.626,
     * 8.272,8.290,8.322,8.346,8.398,8.401,8.418,8.449,
     * 0.000,0.000,7.923,7.917,7.944,7.953,7.973,8.013,
     * 0.000,0.000,0.000,0.000,0.000,0.000,7.656,7.690,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,6.433,6.614,6.869,6.894,6.921,6.839,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,9,L,2),I=9,16),L=1,NZEN)/
     * 8.494,8.542,8.601,8.657,8.730,8.765,8.809,8.846,
     * 8.583,8.602,8.631,8.665,8.716,8.739,8.769,8.803,
     * 8.480,8.509,8.547,8.579,8.631,8.652,8.679,8.713,
     * 8.262,8.307,8.356,8.394,8.456,8.477,8.504,8.537,
     * 0.000,0.000,7.966,7.970,8.009,8.029,8.057,8.097,
     * 0.000,0.000,0.000,0.000,0.000,0.000,7.744,7.780,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,6.480,6.676,6.942,6.981,7.017,6.940,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,9,L,2),I=9,16),L=1,NZEN)/
     * 8.582,8.635,8.695,8.750,8.820,8.862,8.906,8.943,
     * 8.579,8.623,8.672,8.718,8.782,8.821,8.863,8.899,
     * 8.483,8.535,8.589,8.634,8.700,8.736,8.774,8.809,
     * 8.283,8.348,8.408,8.456,8.529,8.562,8.598,8.634,
     * 0.000,0.000,8.057,8.061,8.097,8.130,8.161,8.201,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,7.893,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,6.605,6.806,7.076,7.124,7.164,7.083,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,9,L,2),I=9,16),L=1,NZEN)/
     * 8.690,8.747,8.801,8.849,8.913,8.965,9.013,9.049,
     * 8.577,8.647,8.714,8.775,8.851,8.908,8.962,9.000,
     * 8.487,8.565,8.636,8.695,8.772,8.827,8.877,8.914,
     * 8.314,8.394,8.465,8.525,8.607,8.658,8.704,8.743,
     * 8.230,8.212,8.173,8.173,8.207,8.246,8.281,8.322,
     * 0.000,0.000,0.000,0.000,0.000,0.000,8.000,8.029,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,6.785,6.985,7.255,7.305,7.348,7.258,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,9,L,2),I=9,16),L=1,NZEN)/
     * 8.777,8.837,8.888,8.941,9.009,9.076,9.134,9.176,
     * 8.595,8.690,8.774,8.846,8.937,9.009,9.076,9.117,
     * 8.513,8.612,8.695,8.768,8.859,8.929,8.989,9.033,
     * 8.354,8.455,8.538,8.605,8.697,8.764,8.820,8.863,
     * 8.301,8.301,8.274,8.279,8.314,8.365,8.405,8.450,
     * 8.384,8.356,8.297,8.217,8.176,8.161,8.134,8.161,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,6.919,7.124,7.400,7.461,7.508,7.415,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,10,L,2),I=9,16),L=1,NZEN)/
     * 8.468,8.494,8.540,8.588,8.652,8.675,8.711,8.746,
     * 8.592,8.585,8.598,8.619,8.660,8.669,8.692,8.727,
     * 8.487,8.491,8.512,8.534,8.577,8.584,8.604,8.637,
     * 8.265,8.288,8.322,8.350,8.403,8.410,8.428,8.462,
     * 0.000,0.000,7.922,7.918,7.948,7.960,7.984,8.025,
     * 0.000,0.000,0.000,0.000,0.000,0.000,7.666,7.702,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,6.433,6.614,6.869,6.896,6.925,6.845,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,10,L,2),I=9,16),L=1,NZEN)/
     * 8.489,8.537,8.594,8.647,8.715,8.753,8.797,8.838,
     * 8.504,8.537,8.580,8.625,8.688,8.720,8.760,8.803,
     * 8.398,8.444,8.496,8.540,8.604,8.635,8.673,8.713,
     * 8.182,8.243,8.305,8.356,8.430,8.459,8.496,8.538,
     * 0.000,0.000,7.909,7.927,7.977,8.013,8.049,8.097,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,7.777,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,6.456,6.425,6.629,6.903,6.951,6.997,6.926,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,10,L,2),I=9,16),L=1,NZEN)/
     * 8.576,8.632,8.689,8.741,8.806,8.860,8.914,8.964,
     * 8.455,8.526,8.595,8.658,8.736,8.794,8.852,8.903,
     * 8.360,8.438,8.512,8.576,8.656,8.712,8.766,8.815,
     * 8.158,8.250,8.334,8.400,8.486,8.539,8.591,8.640,
     * 7.987,7.983,7.961,7.992,8.053,8.104,8.152,8.210,
     * 8.068,8.017,7.946,7.880,7.848,7.847,7.843,7.893,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,6.515,6.509,6.727,7.013,7.076,7.130,7.057,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,10,L,2),I=9,16),L=1,NZEN)/
     * 8.657,8.728,8.791,8.855,8.930,9.013,9.083,9.146,
     * 8.464,8.567,8.660,8.745,8.844,8.930,9.009,9.072,
     * 8.382,8.491,8.589,8.671,8.769,8.852,8.925,8.988,
     * 8.204,8.326,8.428,8.512,8.614,8.692,8.761,8.825,
     * 8.025,8.064,8.076,8.127,8.210,8.281,8.344,8.413,
     * 8.097,8.083,8.037,7.998,7.989,8.021,8.041,8.104,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 6.580,6.668,6.688,6.914,7.204,7.276,7.334,7.258,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,10,L,2),I=9,16),L=1,NZEN)/
     * 8.702,8.808,8.897,8.986,9.086,9.210,9.314,9.387,
     * 8.528,8.667,8.785,8.891,9.009,9.121,9.217,9.283,
     * 8.456,8.600,8.721,8.824,8.942,9.049,9.143,9.210,
     * 8.297,8.450,8.579,8.679,8.797,8.901,8.989,9.061,
     * 8.064,8.161,8.223,8.303,8.410,8.509,8.590,8.669,
     * 8.117,8.143,8.134,8.134,8.164,8.236,8.288,8.364,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 6.665,6.787,6.835,7.079,7.382,7.473,7.543,7.464,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,11,L,2),I=9,16),L=1,NZEN)/
     * 8.507,8.525,8.565,8.606,8.665,8.683,8.715,8.751,
     * 8.601,8.592,8.602,8.624,8.665,8.673,8.695,8.732,
     * 8.494,8.497,8.516,8.538,8.581,8.589,8.609,8.642,
     * 8.274,8.294,8.326,8.354,8.407,8.413,8.431,8.467,
     * 0.000,0.000,7.923,7.922,7.954,7.966,7.989,8.029,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,6.442,6.622,6.875,6.902,6.930,6.851,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,11,L,2),I=9,16),L=1,NZEN)/
     * 8.489,8.535,8.590,8.638,8.701,8.741,8.787,8.833,
     * 8.439,8.486,8.539,8.591,8.659,8.702,8.750,8.800,
     * 8.334,8.393,8.455,8.507,8.577,8.618,8.664,8.712,
     * 8.117,8.193,8.267,8.324,8.405,8.444,8.486,8.535,
     * 7.919,7.895,7.862,7.893,7.953,7.997,8.045,8.100,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,6.401,6.382,6.593,6.872,6.928,6.980,6.914,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,11,L,2),I=9,16),L=1,NZEN)/
     * 8.504,8.571,8.634,8.697,8.770,8.841,8.910,8.973,
     * 8.352,8.444,8.530,8.611,8.702,8.778,8.849,8.913,
     * 8.260,8.360,8.452,8.530,8.624,8.696,8.765,8.828,
     * 8.057,8.176,8.281,8.362,8.461,8.529,8.594,8.659,
     * 7.843,7.875,7.885,7.943,8.025,8.097,8.161,8.230,
     * 7.915,7.893,7.844,7.804,7.795,7.822,7.843,7.910,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 6.330,6.415,6.433,6.667,6.965,7.041,7.107,7.045,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,11,L,2),I=9,16),L=1,NZEN)/
     * 8.531,8.641,8.736,8.829,8.931,9.053,9.158,9.238,
     * 8.360,8.496,8.614,8.723,8.841,8.953,9.049,9.130,
     * 8.283,8.425,8.547,8.653,8.771,8.880,8.976,9.053,
     * 8.114,8.267,8.398,8.502,8.621,8.727,8.818,8.900,
     * 7.848,7.949,8.021,8.111,8.225,8.326,8.412,8.501,
     * 7.897,7.927,7.919,7.928,7.965,8.037,8.097,8.188,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 6.410,6.538,6.591,6.845,7.158,7.255,7.332,7.270,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,11,L,2),I=9,16),L=1,NZEN)/
     * 8.539,8.710,8.854,9.004,9.158,9.338,9.496,9.569,
     * 8.380,8.566,8.722,8.862,9.009,9.155,9.276,9.360,
     * 8.318,8.505,8.664,8.801,8.947,9.090,9.210,9.294,
     * 8.170,8.367,8.533,8.667,8.815,8.954,9.068,9.155,
     * 7.860,8.033,8.167,8.294,8.441,8.577,8.690,8.793,
     * 7.869,7.960,8.004,8.076,8.167,8.288,8.386,8.494,
     * 8.021,8.072,8.068,8.017,7.994,7.980,7.943,7.983,
     * 7.606,7.799,7.918,7.897,7.923,7.860,7.774,7.683,
     * 6.428,6.602,6.696,6.981,7.322,7.447,7.549,7.493,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,12,L,2),I=9,16),L=1,NZEN)/
     * 8.555,8.565,8.595,8.631,8.683,8.699,8.729,8.765,
     * 8.601,8.588,8.595,8.616,8.655,8.663,8.685,8.719,
     * 8.494,8.494,8.509,8.530,8.572,8.577,8.597,8.629,
     * 8.274,8.290,8.316,8.342,8.394,8.400,8.418,8.450,
     * 0.000,0.000,7.920,7.912,7.938,7.948,7.969,8.009,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,6.431,6.611,6.862,6.889,6.918,6.838,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,12,L,2),I=9,16),L=1,NZEN)/
     * 8.483,8.530,8.583,8.630,8.690,8.734,8.782,8.833,
     * 8.382,8.438,8.497,8.554,8.625,8.672,8.724,8.775,
     * 8.279,8.344,8.412,8.468,8.542,8.587,8.635,8.687,
     * 8.061,8.143,8.220,8.286,8.369,8.412,8.459,8.511,
     * 7.869,7.849,7.819,7.852,7.914,7.963,8.013,8.072,
     * 7.948,7.884,7.801,7.733,7.697,7.694,7.692,7.747,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 6.305,6.352,6.338,6.554,6.836,6.896,6.951,6.889,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,12,L,2),I=9,16),L=1,NZEN)/
     * 8.428,8.509,8.583,8.654,8.735,8.820,8.898,8.969,
     * 8.246,8.356,8.458,8.549,8.650,8.736,8.816,8.886,
     * 8.149,8.270,8.378,8.468,8.572,8.655,8.732,8.802,
     * 7.950,8.086,8.204,8.297,8.407,8.487,8.561,8.632,
     * 7.734,7.784,7.809,7.879,7.972,8.053,8.127,8.204,
     * 7.806,7.801,7.766,7.738,7.739,7.777,7.808,7.883,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 6.223,6.322,6.356,6.601,6.906,6.992,7.064,7.009,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,12,L,2),I=9,16),L=1,NZEN)/
     * 8.407,8.544,8.663,8.780,8.902,9.045,9.173,9.260,
     * 8.241,8.401,8.542,8.668,8.803,8.932,9.045,9.134,
     * 8.167,8.334,8.479,8.601,8.734,8.862,8.972,9.061,
     * 8.000,8.179,8.332,8.453,8.589,8.713,8.820,8.911,
     * 7.704,7.845,7.950,8.061,8.193,8.314,8.417,8.520,
     * 7.740,7.801,7.821,7.860,7.925,8.025,8.104,8.207,
     * 7.896,7.925,7.906,7.839,7.803,7.764,7.707,7.726,
     * 7.449,7.618,7.719,7.689,7.708,7.638,7.550,7.459,
     * 6.276,6.430,6.508,6.778,7.104,7.215,7.305,7.253,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,12,L,2),I=9,16),L=1,NZEN)/
     * 8.382,8.594,8.776,8.957,9.143,9.348,9.525,9.600,
     * 8.236,8.453,8.636,8.797,8.963,9.127,9.265,9.360,
     * 8.176,8.398,8.582,8.740,8.905,9.068,9.201,9.294,
     * 8.033,8.262,8.455,8.610,8.775,8.933,9.064,9.164,
     * 7.700,7.919,8.093,8.241,8.408,8.566,8.694,8.811,
     * 7.684,7.817,7.899,8.004,8.127,8.272,8.387,8.515,
     * 7.829,7.912,7.938,7.909,7.908,7.925,7.914,7.983,
     * 7.435,7.657,7.805,7.802,7.843,7.799,7.728,7.653,
     * 6.250,6.458,6.582,6.890,7.250,7.394,7.509,7.467,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.041,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,1,L,3),I=9,16),L=1,NZEN)/
     * 8.627,8.639,8.672,8.713,8.771,8.795,8.834,8.881,
     * 8.691,8.679,8.686,8.710,8.753,8.766,8.791,8.829,
     * 8.617,8.612,8.623,8.647,8.694,8.703,8.722,8.755,
     * 8.444,8.455,8.477,8.504,8.559,8.565,8.574,8.601,
     * 0.000,0.000,0.000,8.155,8.182,8.185,8.182,8.199,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,1,L,3),I=9,16),L=1,NZEN)/
     * 8.528,8.582,8.641,8.698,8.766,8.820,8.879,8.940,
     * 8.467,8.522,8.582,8.644,8.721,8.775,8.831,8.889,
     * 8.393,8.458,8.524,8.587,8.667,8.715,8.763,8.816,
     * 8.223,8.303,8.380,8.447,8.537,8.580,8.618,8.665,
     * 8.093,8.079,8.053,8.090,8.161,8.201,8.228,8.267,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,7.140,7.435,7.465,7.449,7.276,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,1,L,3),I=9,16),L=1,NZEN)/
     * 8.455,8.544,8.625,8.707,8.797,8.892,8.980,9.061,
     * 8.305,8.425,8.534,8.633,8.744,8.839,8.927,9.004,
     * 8.243,8.369,8.483,8.581,8.693,8.783,8.862,8.937,
     * 8.086,8.228,8.352,8.453,8.573,8.656,8.724,8.794,
     * 7.914,7.980,8.021,8.104,8.215,8.294,8.348,8.408,
     * 8.033,8.033,8.004,7.990,8.009,8.049,8.057,8.107,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,7.170,7.490,7.551,7.556,7.398,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,1,L,3),I=9,16),L=1,NZEN)/
     * 8.410,8.559,8.686,8.812,8.944,9.097,9.236,9.328,
     * 8.230,8.408,8.562,8.704,8.853,8.996,9.121,9.215,
     * 8.182,8.365,8.522,8.662,8.811,8.949,9.064,9.158,
     * 8.053,8.250,8.417,8.554,8.707,8.837,8.942,9.029,
     * 7.829,7.983,8.100,8.230,8.384,8.505,8.591,8.680,
     * 7.921,7.990,8.017,8.068,8.149,8.248,8.307,8.387,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,7.025,7.301,7.643,7.732,7.758,7.612,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,1,L,3),I=9,16),L=1,NZEN)/
     * 8.362,8.582,8.769,8.958,9.149,9.362,9.545,9.626,
     * 8.140,8.382,8.588,8.769,8.955,9.137,9.290,9.393,
     * 8.104,8.348,8.555,8.734,8.920,9.097,9.241,9.342,
     * 7.991,8.248,8.462,8.638,8.826,8.998,9.134,9.230,
     * 7.725,7.966,8.158,8.330,8.522,8.686,8.806,8.911,
     * 7.776,7.923,8.017,8.134,8.279,8.425,8.524,8.630,
     * 8.017,8.117,8.155,8.140,8.161,8.167,8.121,8.143,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,2,L,3),I=9,16),L=1,NZEN)/
     * 8.579,8.599,8.640,8.689,8.754,8.782,8.825,8.872,
     * 8.669,8.663,8.675,8.702,8.746,8.760,8.785,8.822,
     * 8.592,8.594,8.612,8.638,8.688,8.696,8.714,8.747,
     * 8.420,8.436,8.464,8.494,8.551,8.558,8.567,8.594,
     * 0.000,0.000,8.155,8.143,8.170,8.173,8.173,8.190,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,2,L,3),I=9,16),L=1,NZEN)/
     * 8.481,8.544,8.614,8.679,8.755,8.808,8.867,8.925,
     * 8.465,8.521,8.583,8.643,8.720,8.771,8.824,8.879,
     * 8.389,8.455,8.522,8.583,8.663,8.709,8.756,8.806,
     * 8.217,8.299,8.377,8.442,8.530,8.572,8.607,8.653,
     * 8.107,8.086,8.057,8.086,8.152,8.190,8.215,8.253,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,7.267,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,2,L,3),I=9,16),L=1,NZEN)/
     * 8.433,8.525,8.612,8.691,8.779,8.865,8.946,9.021,
     * 8.328,8.439,8.543,8.634,8.738,8.826,8.907,8.978,
     * 8.262,8.382,8.490,8.581,8.686,8.769,8.841,8.908,
     * 8.107,8.241,8.358,8.450,8.563,8.639,8.702,8.763,
     * 7.958,8.009,8.037,8.107,8.207,8.276,8.324,8.377,
     * 8.079,8.076,8.037,8.009,8.017,8.041,8.037,8.079,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,7.182,7.497,7.551,7.550,7.387,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,2,L,3),I=9,16),L=1,NZEN)/
     * 8.398,8.530,8.644,8.754,8.870,8.999,9.111,9.207,
     * 8.274,8.436,8.575,8.700,8.833,8.960,9.072,9.155,
     * 8.223,8.391,8.534,8.657,8.790,8.912,9.013,9.093,
     * 8.097,8.274,8.427,8.547,8.683,8.797,8.886,8.968,
     * 7.903,8.029,8.121,8.228,8.362,8.465,8.538,8.613,
     * 8.004,8.057,8.064,8.090,8.146,8.220,8.260,8.324,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,7.328,7.663,7.739,7.754,7.593,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,2,L,3),I=9,16),L=1,NZEN)/
     * 8.330,8.520,8.679,8.831,8.988,9.176,9.336,9.430,
     * 8.204,8.417,8.599,8.755,8.918,9.076,9.207,9.297,
     * 8.164,8.382,8.566,8.719,8.880,9.033,9.155,9.243,
     * 8.053,8.281,8.471,8.622,8.786,8.932,9.045,9.127,
     * 7.825,8.025,8.176,8.320,8.486,8.621,8.714,8.799,
     * 7.903,8.017,8.076,8.155,8.258,8.373,8.442,8.521,
     * 8.146,8.223,8.238,8.201,8.199,8.170,8.090,8.076,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,7.083,7.382,7.554,7.681,7.716,7.749,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,3,L,3),I=9,16),L=1,NZEN)/
     * 8.521,8.550,8.600,8.657,8.730,8.760,8.804,8.851,
     * 8.659,8.654,8.669,8.697,8.744,8.757,8.783,8.822,
     * 8.580,8.585,8.606,8.634,8.684,8.694,8.713,8.747,
     * 8.405,8.425,8.455,8.489,8.548,8.555,8.566,8.594,
     * 0.000,0.000,0.000,8.140,8.167,8.170,8.170,8.190,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,3,L,3),I=9,16),L=1,NZEN)/
     * 8.427,8.497,8.576,8.648,8.734,8.789,8.849,8.904,
     * 8.508,8.556,8.611,8.665,8.735,8.777,8.824,8.873,
     * 8.433,8.490,8.549,8.603,8.677,8.715,8.755,8.800,
     * 8.262,8.334,8.403,8.461,8.543,8.577,8.609,8.647,
     * 8.164,8.134,8.090,8.111,8.167,8.199,8.217,8.248,
     * 8.292,8.217,8.127,8.045,8.009,7.979,7.933,7.947,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,3,L,3),I=9,16),L=1,NZEN)/
     * 8.413,8.509,8.603,8.683,8.773,8.847,8.918,8.980,
     * 8.407,8.500,8.588,8.667,8.757,8.828,8.894,8.952,
     * 8.340,8.439,8.531,8.609,8.702,8.769,8.827,8.880,
     * 8.182,8.297,8.398,8.476,8.574,8.634,8.683,8.732,
     * 8.072,8.093,8.093,8.140,8.217,8.270,8.301,8.340,
     * 8.199,8.173,8.117,8.068,8.053,8.049,8.021,8.045,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,3,L,3),I=9,16),L=1,NZEN)/
     * 8.438,8.551,8.651,8.740,8.838,8.937,9.021,9.097,
     * 8.330,8.467,8.587,8.688,8.801,8.903,8.989,9.057,
     * 8.274,8.415,8.538,8.638,8.752,8.848,8.926,8.989,
     * 8.140,8.290,8.420,8.519,8.635,8.723,8.790,8.848,
     * 8.041,8.107,8.143,8.210,8.305,8.378,8.425,8.473,
     * 8.167,8.182,8.161,8.134,8.140,8.164,8.158,8.188,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,3,L,3),I=9,16),L=1,NZEN)/
     * 8.408,8.556,8.679,8.792,8.912,9.049,9.167,9.258,
     * 8.276,8.456,8.609,8.740,8.881,9.017,9.127,9.207,
     * 8.228,8.413,8.568,8.696,8.836,8.966,9.072,9.146,
     * 8.107,8.301,8.462,8.589,8.732,8.854,8.945,9.017,
     * 7.964,8.097,8.185,8.290,8.422,8.529,8.600,8.666,
     * 8.079,8.143,8.161,8.182,8.230,8.301,8.332,8.386,
     * 8.328,8.367,8.350,8.281,8.253,8.185,8.068,8.004,
     * 0.000,8.223,8.322,8.294,8.326,8.228,8.072,7.879,
     * 0.000,0.000,0.000,7.436,7.774,7.857,7.874,7.705,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,4,L,3),I=9,16),L=1,NZEN)/
     * 8.511,8.542,8.593,8.649,8.721,8.750,8.792,8.837,
     * 8.659,8.656,8.673,8.701,8.747,8.763,8.790,8.831,
     * 8.583,8.590,8.612,8.640,8.691,8.702,8.722,8.759,
     * 0.000,0.000,8.467,8.500,8.559,8.566,8.577,8.609,
     * 0.000,0.000,8.152,8.152,8.188,8.193,8.193,8.212,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,7.230,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,4,L,3),I=9,16),L=1,NZEN)/
     * 8.403,8.474,8.556,8.634,8.728,8.781,8.841,8.892,
     * 8.576,8.611,8.654,8.699,8.761,8.794,8.832,8.875,
     * 8.504,8.545,8.593,8.638,8.703,8.732,8.763,8.802,
     * 8.334,8.391,8.449,8.498,8.572,8.597,8.618,8.650,
     * 8.241,8.196,8.140,8.152,8.201,8.223,8.233,8.255,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,4,L,3),I=9,16),L=1,NZEN)/
     * 8.393,8.489,8.584,8.670,8.770,8.836,8.902,8.954,
     * 8.505,8.572,8.637,8.700,8.777,8.828,8.877,8.921,
     * 8.433,8.508,8.581,8.640,8.719,8.766,8.808,8.846,
     * 8.274,8.360,8.438,8.500,8.585,8.627,8.660,8.692,
     * 8.220,8.201,8.164,8.179,8.228,8.258,8.272,8.294,
     * 8.354,8.297,8.217,8.137,8.097,8.064,8.004,8.009,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,4,L,3),I=9,16),L=1,NZEN)/
     * 8.428,8.539,8.642,8.728,8.825,8.903,8.972,9.021,
     * 8.427,8.528,8.619,8.698,8.791,8.862,8.924,8.967,
     * 8.360,8.467,8.562,8.637,8.731,8.797,8.851,8.890,
     * 8.228,8.334,8.425,8.501,8.598,8.658,8.702,8.735,
     * 0.000,8.255,8.228,8.233,8.272,8.307,8.320,8.340,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,4,L,3),I=9,16),L=1,NZEN)/
     * 8.455,8.573,8.675,8.757,8.852,8.940,9.013,9.061,
     * 8.371,8.502,8.616,8.708,8.814,8.904,8.976,9.021,
     * 8.307,8.444,8.560,8.649,8.755,8.839,8.903,8.943,
     * 8.193,8.326,8.435,8.520,8.625,8.702,8.755,8.788,
     * 8.241,8.272,8.270,8.283,8.326,8.371,8.386,8.405,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,8.369,8.377,8.253,8.076,7.854,
     * 0.000,0.000,0.000,0.000,7.785,7.838,7.829,7.637,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,5,L,3),I=9,16),L=1,NZEN)/
     * 8.562,8.585,8.628,8.675,8.737,8.762,8.801,8.844,
     * 8.673,8.667,8.681,8.707,8.751,8.765,8.791,8.833,
     * 8.598,8.603,8.623,8.649,8.697,8.707,8.725,8.763,
     * 8.430,8.452,8.483,8.513,8.569,8.576,8.587,8.617,
     * 0.000,0.000,8.164,8.167,8.207,8.212,8.215,8.233,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,7.248,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,5,L,3),I=9,16),L=1,NZEN)/
     * 8.453,8.513,8.587,8.661,8.751,8.799,8.854,8.900,
     * 8.639,8.660,8.692,8.729,8.784,8.810,8.841,8.877,
     * 8.566,8.595,8.633,8.669,8.727,8.747,8.772,8.804,
     * 8.400,8.442,8.489,8.529,8.597,8.614,8.629,8.654,
     * 0.000,0.000,8.188,8.190,8.230,8.243,8.248,8.262,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,5,L,3),I=9,16),L=1,NZEN)/
     * 8.393,8.480,8.574,8.665,8.771,8.834,8.900,8.943,
     * 8.592,8.638,8.689,8.736,8.802,8.838,8.876,8.906,
     * 8.520,8.573,8.627,8.674,8.743,8.774,8.803,8.829,
     * 8.364,8.425,8.481,8.530,8.606,8.633,8.654,8.674,
     * 8.350,8.299,8.236,8.225,8.253,8.267,8.267,8.279,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,5,L,3),I=9,16),L=1,NZEN)/
     * 8.350,8.462,8.573,8.673,8.790,8.865,8.937,8.975,
     * 8.530,8.601,8.669,8.730,8.809,8.859,8.905,8.930,
     * 8.459,8.535,8.604,8.664,8.744,8.790,8.827,8.849,
     * 8.344,8.410,8.464,8.520,8.601,8.642,8.672,8.688,
     * 0.000,8.401,8.338,8.301,8.299,8.303,8.288,8.292,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,5,L,3),I=9,16),L=1,NZEN)/
     * 8.286,8.410,8.528,8.634,8.757,8.844,8.923,8.960,
     * 8.431,8.521,8.600,8.671,8.760,8.829,8.888,8.916,
     * 8.373,8.458,8.531,8.599,8.688,8.751,8.801,8.827,
     * 8.320,8.375,8.413,8.465,8.543,8.594,8.628,8.648,
     * 0.000,0.000,0.000,0.000,8.340,8.320,8.272,8.260,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.076,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,6,L,3),I=9,16),L=1,NZEN)/
     * 8.627,8.640,8.674,8.712,8.766,8.787,8.821,8.863,
     * 8.698,8.689,8.700,8.722,8.763,8.775,8.799,8.839,
     * 8.626,8.628,8.645,8.667,8.712,8.719,8.736,8.771,
     * 8.462,8.480,8.509,8.537,8.590,8.593,8.601,8.629,
     * 0.000,0.000,8.185,8.190,8.233,8.238,8.236,8.253,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,6,L,3),I=9,16),L=1,NZEN)/
     * 8.533,8.580,8.641,8.708,8.791,8.831,8.881,8.924,
     * 8.711,8.720,8.743,8.772,8.821,8.839,8.866,8.898,
     * 8.641,8.658,8.686,8.715,8.767,8.781,8.801,8.827,
     * 8.477,8.509,8.547,8.580,8.640,8.650,8.661,8.682,
     * 0.000,0.000,8.238,8.243,8.283,8.292,8.290,8.301,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,6,L,3),I=9,16),L=1,NZEN)/
     * 8.438,8.512,8.597,8.685,8.791,8.849,8.913,8.952,
     * 8.697,8.726,8.762,8.798,8.856,8.881,8.909,8.932,
     * 8.623,8.660,8.700,8.736,8.797,8.817,8.838,8.856,
     * 8.468,8.513,8.558,8.595,8.663,8.680,8.694,8.706,
     * 8.450,8.386,8.310,8.292,8.316,8.322,8.316,8.318,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,6,L,3),I=9,16),L=1,NZEN)/
     * 8.312,8.417,8.526,8.637,8.768,8.848,8.927,8.964,
     * 8.628,8.678,8.730,8.779,8.849,8.886,8.922,8.937,
     * 8.555,8.607,8.660,8.708,8.782,8.814,8.843,8.854,
     * 8.456,8.491,8.520,8.563,8.636,8.665,8.684,8.690,
     * 8.590,8.521,8.436,8.375,8.352,8.334,8.303,8.294,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,6,L,3),I=9,16),L=1,NZEN)/
     * 8.146,8.270,8.393,8.522,8.672,8.778,8.881,8.926,
     * 8.507,8.563,8.617,8.679,8.763,8.820,8.872,8.893,
     * 0.000,0.000,8.548,8.603,8.683,8.735,8.777,8.795,
     * 0.000,0.000,0.000,0.000,8.534,8.571,8.593,8.604,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,7,L,3),I=9,16),L=1,NZEN)/
     * 8.644,8.656,8.689,8.727,8.780,8.801,8.834,8.879,
     * 8.708,8.697,8.706,8.727,8.766,8.777,8.801,8.839,
     * 8.639,8.636,8.650,8.671,8.713,8.721,8.737,8.769,
     * 8.474,8.489,8.513,8.539,8.590,8.593,8.600,8.625,
     * 0.000,0.000,8.188,8.193,8.233,8.236,8.230,8.246,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,7,L,3),I=9,16),L=1,NZEN)/
     * 8.581,8.620,8.677,8.739,8.818,8.856,8.904,8.946,
     * 8.757,8.759,8.776,8.801,8.846,8.863,8.889,8.921,
     * 8.687,8.698,8.719,8.745,8.792,8.806,8.823,8.850,
     * 8.525,8.553,8.585,8.614,8.671,8.679,8.686,8.707,
     * 0.000,0.000,8.270,8.274,8.316,8.322,8.318,8.328,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,7,L,3),I=9,16),L=1,NZEN)/
     * 8.505,8.572,8.648,8.729,8.829,8.882,8.941,8.976,
     * 8.775,8.794,8.821,8.852,8.905,8.927,8.954,8.977,
     * 8.704,8.732,8.765,8.795,8.849,8.867,8.885,8.904,
     * 8.550,8.589,8.627,8.660,8.721,8.736,8.747,8.757,
     * 0.000,0.000,8.356,8.348,8.378,8.382,8.375,8.378,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,7,L,3),I=9,16),L=1,NZEN)/
     * 8.412,8.505,8.604,8.705,8.827,8.895,8.967,8.997,
     * 8.713,8.754,8.799,8.841,8.906,8.939,8.970,8.983,
     * 8.641,8.687,8.732,8.775,8.843,8.870,8.895,8.904,
     * 0.000,8.565,8.593,8.632,8.701,8.724,8.740,8.744,
     * 0.000,0.000,8.477,8.420,8.401,8.389,8.362,8.354,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,7,L,3),I=9,16),L=1,NZEN)/
     * 8.301,8.410,8.521,8.635,8.772,8.860,8.946,8.981,
     * 8.582,8.632,8.682,8.740,8.822,8.874,8.921,8.939,
     * 8.531,8.574,8.613,8.665,8.742,8.789,8.829,8.844,
     * 0.000,0.000,8.513,8.538,8.593,8.627,8.649,8.660,
     * 0.000,0.000,0.000,0.000,0.000,0.000,8.316,8.279,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,8,L,3),I=9,16),L=1,NZEN)/
     * 8.610,8.626,8.665,8.709,8.771,8.795,8.833,8.877,
     * 8.695,8.685,8.694,8.717,8.759,8.770,8.793,8.830,
     * 8.622,8.622,8.637,8.660,8.704,8.711,8.726,8.759,
     * 8.455,8.470,8.496,8.521,8.575,8.579,8.585,8.611,
     * 0.000,0.000,8.173,8.173,8.210,8.210,8.204,8.220,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,8,L,3),I=9,16),L=1,NZEN)/
     * 8.582,8.624,8.683,8.744,8.821,8.861,8.910,8.954,
     * 8.733,8.740,8.761,8.790,8.837,8.858,8.886,8.921,
     * 8.662,8.678,8.703,8.732,8.782,8.798,8.819,8.849,
     * 8.497,8.528,8.562,8.594,8.654,8.666,8.676,8.700,
     * 0.000,0.000,8.248,8.250,8.290,8.299,8.299,8.312,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,8,L,3),I=9,16),L=1,NZEN)/
     * 8.584,8.646,8.718,8.787,8.874,8.922,8.974,9.013,
     * 8.759,8.784,8.816,8.852,8.908,8.938,8.970,9.000,
     * 8.690,8.723,8.759,8.794,8.853,8.879,8.904,8.928,
     * 8.534,8.579,8.622,8.660,8.725,8.745,8.760,8.780,
     * 8.459,8.405,8.340,8.338,8.375,8.387,8.386,8.394,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,8,L,3),I=9,16),L=1,NZEN)/
     * 8.615,8.691,8.770,8.842,8.933,8.985,9.037,9.064,
     * 8.727,8.776,8.827,8.874,8.942,8.980,9.017,9.041,
     * 8.659,8.713,8.766,8.812,8.882,8.917,8.947,8.965,
     * 8.530,8.583,8.629,8.675,8.749,8.778,8.799,8.811,
     * 8.583,8.525,8.449,8.418,8.425,8.433,8.423,8.425,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,8,L,3),I=9,16),L=1,NZEN)/
     * 8.675,8.754,8.830,8.895,8.981,9.037,9.090,9.107,
     * 8.658,8.727,8.791,8.849,8.929,8.982,9.029,9.045,
     * 8.592,8.663,8.726,8.782,8.864,8.912,8.950,8.964,
     * 8.512,8.558,8.591,8.641,8.720,8.761,8.789,8.799,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,9,L,3),I=9,16),L=1,NZEN)/
     * 8.550,8.576,8.624,8.677,8.745,8.775,8.818,8.862,
     * 8.677,8.672,8.686,8.710,8.754,8.766,8.791,8.829,
     * 8.604,8.607,8.626,8.651,8.697,8.706,8.724,8.757,
     * 8.435,8.453,8.481,8.511,8.567,8.572,8.581,8.607,
     * 0.000,0.000,8.164,8.161,8.196,8.199,8.196,8.212,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,9,L,3),I=9,16),L=1,NZEN)/
     * 8.559,8.607,8.669,8.732,8.810,8.852,8.903,8.950,
     * 8.666,8.686,8.716,8.755,8.812,8.840,8.874,8.915,
     * 8.591,8.620,8.659,8.696,8.755,8.780,8.807,8.842,
     * 8.423,8.468,8.515,8.556,8.623,8.643,8.663,8.691,
     * 0.000,8.260,8.199,8.210,8.255,8.272,8.276,8.299,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,9,L,3),I=9,16),L=1,NZEN)/
     * 8.645,8.703,8.766,8.824,8.898,8.947,9.000,9.045,
     * 8.662,8.708,8.758,8.808,8.876,8.920,8.965,9.009,
     * 8.591,8.646,8.702,8.751,8.821,8.862,8.899,8.937,
     * 8.433,8.502,8.566,8.616,8.693,8.728,8.756,8.787,
     * 8.340,8.312,8.267,8.286,8.336,8.365,8.377,8.398,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,9,L,3),I=9,16),L=1,NZEN)/
     * 8.754,8.814,8.872,8.925,8.995,9.053,9.111,9.155,
     * 8.657,8.731,8.800,8.865,8.948,9.009,9.068,9.114,
     * 8.594,8.675,8.749,8.812,8.895,8.953,9.004,9.045,
     * 8.453,8.543,8.621,8.685,8.774,8.825,8.863,8.899,
     * 8.417,8.403,8.367,8.386,8.439,8.480,8.498,8.522,
     * 0.000,0.000,0.000,0.000,0.000,8.281,8.233,8.241,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,9,L,3),I=9,16),L=1,NZEN)/
     * 8.844,8.909,8.965,9.021,9.097,9.173,9.241,9.294,
     * 8.679,8.777,8.864,8.942,9.037,9.121,9.188,9.241,
     * 8.617,8.722,8.813,8.890,8.987,9.064,9.127,9.173,
     * 8.486,8.597,8.689,8.766,8.867,8.938,8.989,9.029,
     * 8.456,8.468,8.450,8.479,8.540,8.595,8.624,8.656,
     * 0.000,0.000,0.000,0.000,0.000,0.000,8.364,8.377,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,10,L,3),I=9,16),L=1,NZEN)/
     * 8.534,8.562,8.611,8.665,8.734,8.763,8.807,8.852,
     * 8.677,8.672,8.686,8.713,8.758,8.772,8.799,8.842,
     * 8.600,8.606,8.628,8.654,8.702,8.713,8.732,8.769,
     * 8.430,8.452,8.483,8.515,8.573,8.579,8.589,8.620,
     * 0.000,0.000,8.164,8.164,8.201,8.207,8.204,8.225,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,10,L,3),I=9,16),L=1,NZEN)/
     * 8.553,8.604,8.667,8.723,8.795,8.841,8.893,8.943,
     * 8.585,8.622,8.667,8.717,8.783,8.823,8.867,8.917,
     * 8.512,8.559,8.610,8.659,8.728,8.762,8.799,8.845,
     * 8.342,8.405,8.467,8.520,8.597,8.628,8.657,8.695,
     * 8.225,8.190,8.143,8.167,8.225,8.255,8.272,8.301,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,10,L,3),I=9,16),L=1,NZEN)/
     * 8.639,8.700,8.760,8.818,8.888,8.951,9.013,9.076,
     * 8.539,8.612,8.682,8.751,8.834,8.899,8.961,9.017,
     * 8.470,8.551,8.627,8.695,8.780,8.841,8.897,8.950,
     * 8.312,8.407,8.491,8.562,8.657,8.710,8.754,8.802,
     * 8.188,8.193,8.176,8.220,8.297,8.344,8.373,8.412,
     * 8.314,8.267,8.193,8.137,8.117,8.117,8.090,8.114,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,10,L,3),I=9,16),L=1,NZEN)/
     * 8.728,8.804,8.872,8.941,9.021,9.114,9.199,9.272,
     * 8.553,8.660,8.756,8.846,8.951,9.045,9.130,9.199,
     * 8.497,8.611,8.711,8.799,8.903,8.993,9.072,9.137,
     * 8.362,8.487,8.594,8.682,8.790,8.873,8.939,9.000,
     * 8.217,8.270,8.294,8.362,8.459,8.531,8.579,8.630,
     * 8.338,8.326,8.281,8.253,8.262,8.294,8.299,8.338,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,10,L,3),I=9,16),L=1,NZEN)/
     * 8.768,8.878,8.971,9.064,9.170,9.303,9.417,9.493,
     * 8.611,8.753,8.874,8.984,9.107,9.225,9.324,9.393,
     * 8.568,8.714,8.838,8.944,9.064,9.179,9.272,9.336,
     * 8.449,8.607,8.739,8.843,8.967,9.072,9.155,9.215,
     * 8.243,8.360,8.439,8.534,8.657,8.751,8.812,8.872,
     * 8.334,8.367,8.358,8.378,8.433,8.502,8.535,8.585,
     * 8.583,8.583,8.531,8.442,8.398,8.320,8.201,8.146,
     * 0.000,0.000,0.000,8.477,8.487,8.371,8.199,7.995,
     * 0.000,0.000,0.000,0.000,7.958,8.017,8.013,7.831,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,11,L,3),I=9,16),L=1,NZEN)/
     * 8.571,8.593,8.636,8.684,8.747,8.772,8.812,8.857,
     * 8.680,8.676,8.690,8.716,8.762,8.776,8.803,8.846,
     * 8.609,8.612,8.631,8.658,8.705,8.716,8.736,8.775,
     * 8.438,8.458,8.489,8.520,8.577,8.583,8.593,8.624,
     * 0.000,0.000,8.167,8.167,8.207,8.212,8.210,8.230,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,11,L,3),I=9,16),L=1,NZEN)/
     * 8.555,8.604,8.662,8.715,8.781,8.831,8.886,8.942,
     * 8.519,8.569,8.625,8.683,8.757,8.807,8.859,8.916,
     * 8.449,8.508,8.569,8.626,8.701,8.747,8.793,8.846,
     * 8.279,8.354,8.428,8.490,8.574,8.615,8.650,8.697,
     * 8.149,8.130,8.097,8.134,8.204,8.243,8.267,8.303,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,11,L,3),I=9,16),L=1,NZEN)/
     * 8.573,8.643,8.711,8.780,8.859,8.940,9.017,9.090,
     * 8.436,8.531,8.621,8.707,8.805,8.888,8.966,9.037,
     * 8.375,8.477,8.571,8.655,8.755,8.832,8.901,8.968,
     * 8.217,8.336,8.441,8.528,8.634,8.706,8.764,8.826,
     * 8.049,8.090,8.111,8.179,8.279,8.344,8.389,8.441,
     * 8.167,8.146,8.097,8.068,8.076,8.100,8.100,8.140,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,7.433,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,11,L,3),I=9,16),L=1,NZEN)/
     * 8.611,8.725,8.824,8.928,9.041,9.170,9.290,9.377,
     * 8.455,8.595,8.719,8.835,8.962,9.083,9.188,9.272,
     * 8.408,8.554,8.679,8.794,8.921,9.037,9.134,9.212,
     * 8.283,8.441,8.575,8.687,8.817,8.925,9.013,9.090,
     * 8.057,8.176,8.260,8.365,8.496,8.592,8.660,8.735,
     * 8.146,8.179,8.173,8.201,8.262,8.336,8.375,8.442,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,7.669,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,11,L,3),I=9,16),L=1,NZEN)/
     * 8.622,8.800,8.951,9.107,9.272,9.453,9.614,9.689,
     * 8.479,8.668,8.829,8.976,9.130,9.281,9.405,9.491,
     * 8.446,8.638,8.801,8.944,9.097,9.243,9.360,9.441,
     * 8.340,8.545,8.714,8.855,9.009,9.149,9.258,9.336,
     * 8.072,8.265,8.418,8.555,8.716,8.848,8.941,9.025,
     * 8.097,8.204,8.262,8.354,8.474,8.591,8.666,8.748,
     * 8.336,8.391,8.387,8.344,8.338,8.318,8.250,8.255,
     * 8.130,8.318,8.428,8.408,8.447,8.364,8.217,8.045,
     * 0.000,0.000,0.000,7.562,7.919,8.017,8.049,7.893,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,12,L,3),I=9,16),L=1,NZEN)/
     * 8.620,8.632,8.667,8.708,8.765,8.789,8.826,8.872,
     * 8.684,8.674,8.684,8.708,8.753,8.766,8.793,8.834,
     * 8.607,8.607,8.623,8.648,8.695,8.706,8.725,8.761,
     * 8.436,8.452,8.477,8.508,8.565,8.571,8.581,8.610,
     * 0.000,0.000,8.161,8.158,8.190,8.193,8.190,8.212,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,2,12,L,3),I=9,16),L=1,NZEN)/
     * 8.548,8.598,8.654,8.707,8.772,8.825,8.883,8.943,
     * 8.465,8.522,8.583,8.646,8.724,8.777,8.833,8.892,
     * 8.393,8.459,8.526,8.589,8.668,8.717,8.766,8.820,
     * 8.223,8.303,8.382,8.450,8.539,8.584,8.623,8.671,
     * 8.097,8.083,8.053,8.093,8.164,8.207,8.236,8.274,
     * 8.223,8.158,8.076,8.009,7.985,7.975,7.947,7.973,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,3,12,L,3),I=9,16),L=1,NZEN)/
     * 8.498,8.583,8.660,8.737,8.824,8.920,9.009,9.090,
     * 8.330,8.444,8.549,8.646,8.754,8.848,8.934,9.013,
     * 8.265,8.387,8.498,8.593,8.703,8.792,8.871,8.946,
     * 8.111,8.248,8.369,8.467,8.583,8.665,8.732,8.802,
     * 7.939,8.000,8.033,8.117,8.225,8.303,8.356,8.417,
     * 8.061,8.057,8.017,8.000,8.017,8.057,8.064,8.114,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,7.401,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,4,12,L,3),I=9,16),L=1,NZEN)/
     * 8.491,8.634,8.758,8.884,9.017,9.170,9.310,9.400,
     * 8.340,8.507,8.652,8.787,8.930,9.068,9.190,9.279,
     * 8.297,8.468,8.616,8.748,8.890,9.025,9.137,9.225,
     * 8.176,8.360,8.516,8.645,8.790,8.918,9.021,9.104,
     * 7.920,8.079,8.201,8.324,8.473,8.590,8.675,8.762,
     * 7.992,8.061,8.083,8.143,8.233,8.330,8.391,8.471,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 7.993,8.155,8.241,8.215,8.246,8.152,8.004,7.831,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,7.663,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,5,12,L,3),I=9,16),L=1,NZEN)/
     * 8.474,8.697,8.888,9.076,9.270,9.474,9.650,9.725,
     * 8.344,8.568,8.757,8.924,9.097,9.267,9.407,9.501,
     * 8.314,8.540,8.731,8.897,9.072,9.233,9.364,9.455,
     * 8.217,8.453,8.648,8.810,8.985,9.143,9.270,9.356,
     * 7.928,8.167,8.358,8.516,8.696,8.850,8.960,9.053,
     * 7.921,8.076,8.176,8.299,8.449,8.590,8.685,8.784,
     * 8.155,8.243,8.270,8.250,8.270,8.281,8.241,8.274,
     * 7.963,8.185,8.322,8.326,8.384,8.318,8.188,8.033,
     * 6.848,7.053,7.170,7.484,7.860,7.977,8.025,7.884,
     * 6.303,6.438,6.480,6.490,6.565,6.574,6.513,6.464,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,1,L,1),I=17,24),L=1,NZEN)/
     * 8.707,8.732,8.708,8.687,8.653,8.619,8.601,8.664,
     * 8.642,8.666,8.640,8.610,8.567,8.522,8.496,8.521,
     * 8.540,8.574,8.561,8.539,8.505,8.467,8.450,8.483,
     * 8.346,8.398,8.405,8.403,8.389,8.367,8.371,8.418,
     * 7.884,7.973,8.025,8.068,8.097,8.121,8.179,8.274,
     * 7.542,7.654,7.733,7.804,7.860,7.916,8.017,8.149,
     * 7.076,7.152,7.201,7.290,7.360,7.465,7.624,7.840,
     * 6.732,6.783,6.821,6.827,6.816,6.817,6.893,7.083,
     * 6.468,6.563,6.651,6.688,6.704,6.729,6.838,7.049,
     * 0.000,0.000,0.000,0.000,0.000,6.013,6.346,6.790,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.279,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.233/
      DATA ((EDEN(I,2,1,L,1),I=17,24),L=1,NZEN)/
     * 8.766,8.781,8.751,8.731,8.702,8.677,8.675,8.756,
     * 8.706,8.723,8.696,8.665,8.625,8.592,8.585,8.634,
     * 8.607,8.633,8.617,8.595,8.566,8.539,8.543,8.598,
     * 8.413,8.459,8.465,8.464,8.453,8.441,8.465,8.535,
     * 7.954,8.037,8.090,8.134,8.170,8.204,8.281,8.396,
     * 7.612,7.719,7.797,7.872,7.936,8.004,8.124,8.276,
     * 7.130,7.207,7.258,7.352,7.433,7.555,7.740,7.975,
     * 6.782,6.824,6.857,6.865,6.858,6.872,6.970,7.190,
     * 6.530,6.612,6.692,6.728,6.747,6.785,6.914,7.152,
     * 0.000,0.000,0.000,0.000,0.000,6.079,6.433,6.896,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.382,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.336/
      DATA ((EDEN(I,3,1,L,1),I=17,24),L=1,NZEN)/
     * 8.885,8.892,8.857,8.841,8.822,8.799,8.811,8.899,
     * 8.822,8.832,8.801,8.767,8.730,8.708,8.721,8.790,
     * 8.725,8.746,8.728,8.702,8.673,8.658,8.682,8.757,
     * 8.539,8.579,8.582,8.577,8.569,8.566,8.607,8.698,
     * 8.090,8.170,8.223,8.267,8.307,8.346,8.439,8.572,
     * 7.751,7.856,7.934,8.013,8.086,8.161,8.294,8.464,
     * 7.241,7.322,7.384,7.491,7.591,7.731,7.941,8.188,
     * 6.882,6.911,6.934,6.942,6.938,6.965,7.083,7.338,
     * 6.650,6.714,6.779,6.811,6.832,6.877,7.021,7.288,
     * 0.000,0.000,0.000,0.000,0.000,6.201,6.563,7.041,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.528,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.482/
      DATA ((EDEN(I,4,1,L,1),I=17,24),L=1,NZEN)/
     * 9.137,9.121,9.068,9.053,9.037,9.009,9.025,9.097,
     * 9.017,9.017,8.978,8.941,8.904,8.893,8.928,9.009,
     * 8.930,8.938,8.911,8.883,8.855,8.849,8.890,8.979,
     * 8.759,8.788,8.787,8.775,8.761,8.767,8.827,8.929,
     * 8.334,8.413,8.467,8.501,8.534,8.575,8.680,8.821,
     * 8.004,8.114,8.201,8.274,8.346,8.417,8.558,8.732,
     * 7.449,7.554,7.642,7.773,7.899,8.037,8.255,8.501,
     * 7.041,7.061,7.079,7.090,7.093,7.130,7.270,7.580,
     * 6.839,6.885,6.937,6.967,6.989,7.037,7.196,7.494,
     * 0.000,0.000,0.000,0.000,6.149,6.407,6.775,7.267,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.744,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.698/
      DATA ((EDEN(I,5,1,L,1),I=17,24),L=1,NZEN)/
     * 9.453,9.401,9.314,9.288,9.265,9.228,9.241,9.307,
     * 9.220,9.212,9.170,9.130,9.097,9.093,9.143,9.230,
     * 9.143,9.140,9.107,9.079,9.053,9.053,9.107,9.204,
     * 8.985,9.004,8.992,8.979,8.968,8.980,9.053,9.164,
     * 8.590,8.659,8.706,8.733,8.762,8.806,8.922,9.072,
     * 8.272,8.378,8.465,8.528,8.592,8.663,8.812,8.995,
     * 7.681,7.809,7.925,8.061,8.193,8.328,8.549,8.799,
     * 7.207,7.223,7.241,7.260,7.276,7.330,7.493,7.860,
     * 7.017,7.053,7.100,7.137,7.167,7.223,7.400,7.736,
     * 0.000,0.000,6.009,6.182,6.350,6.617,7.000,7.516,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.107,6.980,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.061,6.934/
      DATA ((EDEN(I,1,2,L,1),I=17,24),L=1,NZEN)/
     * 8.701,8.722,8.694,8.668,8.631,8.594,8.573,8.637,
     * 8.636,8.659,8.633,8.602,8.559,8.516,8.491,8.519,
     * 8.535,8.567,8.551,8.530,8.497,8.461,8.446,8.480,
     * 8.340,8.389,8.396,8.394,8.380,8.360,8.365,8.415,
     * 7.877,7.965,8.017,8.061,8.090,8.114,8.176,8.272,
     * 7.534,7.645,7.722,7.794,7.852,7.911,8.013,8.146,
     * 7.076,7.149,7.199,7.283,7.352,7.461,7.622,7.839,
     * 6.728,6.780,6.820,6.825,6.812,6.814,6.890,7.083,
     * 6.462,6.559,6.647,6.684,6.701,6.726,6.835,7.049,
     * 0.000,0.000,0.000,0.000,0.000,6.004,6.342,6.788,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.276,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.231/
      DATA ((EDEN(I,2,2,L,1),I=17,24),L=1,NZEN)/
     * 8.754,8.767,8.736,8.713,8.682,8.655,8.652,8.736,
     * 8.699,8.714,8.687,8.655,8.615,8.583,8.577,8.624,
     * 8.597,8.622,8.605,8.584,8.555,8.528,8.530,8.585,
     * 8.403,8.447,8.452,8.452,8.441,8.430,8.455,8.522,
     * 7.942,8.021,8.072,8.117,8.155,8.188,8.267,8.382,
     * 7.600,7.706,7.782,7.856,7.919,7.987,8.107,8.260,
     * 7.130,7.201,7.250,7.340,7.418,7.540,7.723,7.959,
     * 6.785,6.827,6.861,6.869,6.862,6.874,6.969,7.185,
     * 6.526,6.610,6.691,6.728,6.750,6.786,6.914,7.149,
     * 0.000,0.000,0.000,0.000,0.000,6.068,6.423,6.889,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.378,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.333/
      DATA ((EDEN(I,3,2,L,1),I=17,24),L=1,NZEN)/
     * 8.848,8.856,8.823,8.809,8.791,8.774,8.787,8.879,
     * 8.795,8.803,8.772,8.739,8.701,8.680,8.693,8.762,
     * 8.699,8.716,8.696,8.672,8.643,8.628,8.649,8.725,
     * 8.511,8.547,8.549,8.545,8.535,8.534,8.575,8.665,
     * 8.061,8.137,8.185,8.230,8.270,8.310,8.401,8.531,
     * 7.726,7.823,7.897,7.975,8.045,8.121,8.255,8.422,
     * 7.236,7.307,7.360,7.462,7.555,7.691,7.896,8.143,
     * 6.886,6.912,6.936,6.942,6.937,6.960,7.072,7.318,
     * 6.650,6.713,6.780,6.812,6.831,6.873,7.013,7.272,
     * 0.000,0.000,0.000,0.000,0.000,6.190,6.554,7.025,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.520,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.474/
      DATA ((EDEN(I,4,2,L,1),I=17,24),L=1,NZEN)/
     * 9.025,9.021,8.986,8.980,8.975,8.958,8.979,9.064,
     * 8.964,8.961,8.926,8.890,8.854,8.845,8.877,8.959,
     * 8.876,8.882,8.857,8.830,8.803,8.797,8.835,8.923,
     * 8.702,8.728,8.726,8.716,8.704,8.710,8.765,8.866,
     * 8.274,8.346,8.398,8.436,8.471,8.512,8.612,8.750,
     * 7.950,8.049,8.130,8.204,8.276,8.348,8.484,8.656,
     * 7.422,7.509,7.584,7.707,7.823,7.961,8.176,8.418,
     * 7.041,7.053,7.064,7.072,7.072,7.100,7.230,7.520,
     * 6.836,6.876,6.924,6.951,6.968,7.013,7.164,7.447,
     * 0.000,0.000,0.000,0.000,6.130,6.386,6.749,7.225,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.719,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.673/
      DATA ((EDEN(I,5,2,L,1),I=17,24),L=1,NZEN)/
     * 9.292,9.262,9.201,9.193,9.188,9.161,9.182,9.255,
     * 9.149,9.137,9.093,9.053,9.021,9.013,9.057,9.146,
     * 9.068,9.061,9.029,8.999,8.972,8.969,9.017,9.111,
     * 8.905,8.919,8.910,8.895,8.882,8.890,8.955,9.061,
     * 8.501,8.563,8.610,8.637,8.667,8.708,8.816,8.959,
     * 8.185,8.279,8.360,8.425,8.490,8.559,8.701,8.875,
     * 7.628,7.728,7.821,7.948,8.076,8.207,8.422,8.665,
     * 7.207,7.210,7.220,7.228,7.233,7.270,7.410,7.737,
     * 7.013,7.041,7.083,7.111,7.130,7.176,7.330,7.636,
     * 0.000,0.000,0.000,6.143,6.305,6.562,6.933,7.422,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.061,6.911,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.015,6.865/
      DATA ((EDEN(I,1,3,L,1),I=17,24),L=1,NZEN)/
     * 8.680,8.700,8.669,8.640,8.599,8.560,8.537,8.603,
     * 8.638,8.664,8.639,8.610,8.568,8.522,8.496,8.521,
     * 8.537,8.571,8.558,8.538,8.507,8.467,8.450,8.483,
     * 8.340,8.394,8.403,8.403,8.389,8.367,8.371,8.420,
     * 7.877,7.968,8.021,8.068,8.100,8.121,8.182,8.274,
     * 7.535,7.649,7.730,7.803,7.862,7.919,8.021,8.152,
     * 7.076,7.152,7.201,7.290,7.362,7.470,7.629,7.844,
     * 6.730,6.783,6.823,6.830,6.820,6.819,6.894,7.086,
     * 6.462,6.560,6.649,6.688,6.706,6.730,6.838,7.049,
     * 0.000,0.000,0.000,0.000,0.000,6.000,6.342,6.788,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.276,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.231/
      DATA ((EDEN(I,2,3,L,1),I=17,24),L=1,NZEN)/
     * 8.733,8.746,8.715,8.693,8.662,8.634,8.628,8.713,
     * 8.692,8.708,8.684,8.657,8.620,8.588,8.576,8.620,
     * 8.591,8.617,8.603,8.585,8.560,8.533,8.530,8.582,
     * 8.398,8.441,8.447,8.452,8.444,8.433,8.452,8.517,
     * 7.936,8.017,8.072,8.121,8.158,8.190,8.262,8.373,
     * 7.599,7.702,7.780,7.857,7.922,7.989,8.104,8.250,
     * 7.137,7.207,7.258,7.348,7.428,7.545,7.722,7.950,
     * 6.794,6.837,6.874,6.884,6.879,6.889,6.979,7.188,
     * 6.534,6.619,6.705,6.744,6.768,6.801,6.924,7.149,
     * 0.000,0.000,0.000,0.000,0.000,6.083,6.436,6.893,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.386,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.340/
      DATA ((EDEN(I,3,3,L,1),I=17,24),L=1,NZEN)/
     * 8.808,8.816,8.787,8.774,8.755,8.743,8.755,8.853,
     * 8.769,8.775,8.746,8.718,8.685,8.663,8.669,8.731,
     * 8.670,8.684,8.667,8.648,8.623,8.609,8.623,8.692,
     * 8.480,8.512,8.515,8.516,8.511,8.509,8.544,8.624,
     * 8.025,8.097,8.146,8.193,8.233,8.272,8.358,8.481,
     * 7.695,7.788,7.862,7.937,8.004,8.079,8.207,8.364,
     * 7.243,7.299,7.342,7.436,7.517,7.644,7.834,8.072,
     * 6.904,6.930,6.958,6.965,6.959,6.975,7.076,7.299,
     * 6.661,6.724,6.796,6.831,6.852,6.891,7.021,7.260,
     * 0.000,0.000,0.000,0.000,0.000,6.193,6.554,7.013,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.517,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.471/
      DATA ((EDEN(I,4,3,L,1),I=17,24),L=1,NZEN)/
     * 8.919,8.921,8.894,8.893,8.888,8.886,8.914,9.013,
     * 8.867,8.861,8.828,8.800,8.769,8.760,8.783,8.860,
     * 8.771,8.774,8.753,8.733,8.710,8.706,8.736,8.819,
     * 8.585,8.607,8.609,8.607,8.602,8.610,8.657,8.752,
     * 8.149,8.207,8.253,8.299,8.340,8.387,8.483,8.616,
     * 7.827,7.907,7.977,8.053,8.124,8.204,8.340,8.507,
     * 7.375,7.422,7.464,7.561,7.652,7.789,7.993,8.236,
     * 7.041,7.045,7.061,7.061,7.057,7.079,7.190,7.435,
     * 6.818,6.857,6.911,6.939,6.955,6.998,7.137,7.389,
     * 0.000,0.000,0.000,0.000,6.079,6.340,6.702,7.158,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.681,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.635/
      DATA ((EDEN(I,5,3,L,1),I=17,24),L=1,NZEN)/
     * 9.064,9.061,9.033,9.045,9.057,9.057,9.093,9.185,
     * 9.004,8.991,8.959,8.933,8.907,8.903,8.936,9.021,
     * 8.914,8.910,8.889,8.872,8.854,8.853,8.892,8.979,
     * 8.738,8.754,8.756,8.755,8.753,8.761,8.815,8.913,
     * 8.318,8.375,8.425,8.471,8.516,8.559,8.656,8.787,
     * 8.009,8.086,8.161,8.241,8.320,8.393,8.526,8.688,
     * 7.531,7.587,7.643,7.760,7.871,8.009,8.215,8.447,
     * 7.182,7.173,7.185,7.188,7.185,7.215,7.334,7.603,
     * 6.981,7.004,7.045,7.076,7.090,7.134,7.276,7.539,
     * 0.000,0.000,0.000,6.104,6.260,6.520,6.879,7.328,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.045,6.861,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.000,6.815/
      DATA ((EDEN(I,1,4,L,1),I=17,24),L=1,NZEN)/
     * 8.667,8.684,8.650,8.621,8.581,8.540,8.515,8.584,
     * 8.649,8.679,8.660,8.634,8.595,8.548,8.517,8.540,
     * 8.548,8.588,8.581,8.565,8.537,8.494,8.474,8.501,
     * 8.358,8.413,8.427,8.431,8.423,8.396,8.394,8.438,
     * 7.902,7.996,8.053,8.104,8.143,8.158,8.212,8.297,
     * 7.562,7.679,7.762,7.843,7.910,7.960,8.053,8.179,
     * 7.083,7.173,7.233,7.330,7.413,7.517,7.675,7.881,
     * 6.736,6.790,6.831,6.843,6.836,6.832,6.904,7.097,
     * 6.477,6.574,6.664,6.704,6.725,6.744,6.848,7.057,
     * 0.000,0.000,0.000,0.000,0.000,6.033,6.365,6.805,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.292,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.246/
      DATA ((EDEN(I,2,4,L,1),I=17,24),L=1,NZEN)/
     * 8.723,8.735,8.708,8.688,8.658,8.632,8.623,8.706,
     * 8.693,8.711,8.691,8.670,8.638,8.606,8.591,8.628,
     * 8.591,8.619,8.611,8.599,8.577,8.549,8.544,8.588,
     * 8.400,8.444,8.455,8.464,8.462,8.449,8.462,8.520,
     * 7.944,8.025,8.083,8.137,8.176,8.207,8.274,8.375,
     * 7.610,7.715,7.797,7.878,7.945,8.009,8.117,8.255,
     * 7.152,7.223,7.276,7.373,7.456,7.569,7.738,7.956,
     * 6.808,6.852,6.894,6.908,6.905,6.914,6.998,7.196,
     * 6.554,6.639,6.729,6.772,6.798,6.829,6.944,7.161,
     * 0.000,0.000,0.000,0.000,0.000,6.124,6.471,6.913,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.410,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.364/
      DATA ((EDEN(I,3,4,L,1),I=17,24),L=1,NZEN)/
     * 8.782,8.786,8.760,8.749,8.730,8.719,8.723,8.822,
     * 8.736,8.740,8.716,8.697,8.670,8.650,8.649,8.702,
     * 8.632,8.645,8.633,8.623,8.605,8.591,8.599,8.656,
     * 8.439,8.470,8.477,8.486,8.486,8.486,8.513,8.584,
     * 7.985,8.053,8.100,8.152,8.196,8.236,8.316,8.430,
     * 7.660,7.747,7.821,7.897,7.961,8.033,8.152,8.303,
     * 7.250,7.294,7.330,7.415,7.487,7.604,7.774,8.004,
     * 6.920,6.947,6.984,6.996,6.993,7.004,7.093,7.294,
     * 6.665,6.735,6.820,6.861,6.886,6.924,7.045,7.265,
     * 0.000,0.000,0.000,0.000,0.000,6.215,6.571,7.017,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.531,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.486/
      DATA ((EDEN(I,4,4,L,1),I=17,24),L=1,NZEN)/
     * 8.838,8.831,8.805,8.799,8.788,8.789,8.806,8.918,
     * 8.775,8.765,8.738,8.722,8.700,8.691,8.702,8.765,
     * 8.670,8.669,8.654,8.645,8.631,8.627,8.647,8.715,
     * 8.476,8.491,8.496,8.504,8.507,8.517,8.556,8.637,
     * 8.025,8.076,8.121,8.170,8.212,8.262,8.350,8.474,
     * 7.720,7.785,7.848,7.919,7.981,8.061,8.185,8.344,
     * 7.377,7.387,7.403,7.473,7.533,7.646,7.814,8.045,
     * 7.057,7.064,7.093,7.104,7.100,7.114,7.204,7.398,
     * 6.802,6.854,6.932,6.972,6.997,7.037,7.161,7.375,
     * 0.000,0.000,0.000,0.000,6.061,6.334,6.695,7.134,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.678,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.632/
      DATA ((EDEN(I,5,4,L,1),I=17,24),L=1,NZEN)/
     * 8.865,8.844,8.816,8.820,8.818,8.831,8.862,8.993,
     * 8.816,8.797,8.771,8.760,8.746,8.744,8.762,8.834,
     * 8.713,8.701,8.685,8.681,8.673,8.675,8.701,8.777,
     * 8.519,8.522,8.524,8.537,8.544,8.559,8.601,8.689,
     * 8.086,8.117,8.158,8.207,8.250,8.299,8.387,8.513,
     * 7.804,7.846,7.897,7.966,8.025,8.100,8.223,8.382,
     * 7.507,7.496,7.500,7.558,7.607,7.710,7.866,8.086,
     * 7.199,7.193,7.217,7.225,7.223,7.236,7.324,7.509,
     * 6.956,6.990,7.061,7.100,7.124,7.161,7.286,7.491,
     * 0.000,0.000,0.000,6.053,6.215,6.489,6.848,7.267,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.049,6.841,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.003,6.795/
      DATA ((EDEN(I,1,5,L,1),I=17,24),L=1,NZEN)/
     * 8.671,8.689,8.657,8.629,8.589,8.551,8.529,8.599,
     * 8.649,8.679,8.662,8.636,8.599,8.551,8.521,8.547,
     * 8.551,8.591,8.584,8.568,8.540,8.498,8.477,8.508,
     * 8.364,8.420,8.435,8.438,8.430,8.401,8.401,8.446,
     * 7.915,8.009,8.068,8.121,8.158,8.170,8.223,8.307,
     * 7.583,7.702,7.786,7.866,7.932,7.980,8.072,8.196,
     * 7.090,7.185,7.250,7.354,7.441,7.547,7.706,7.908,
     * 6.736,6.788,6.827,6.838,6.831,6.831,6.906,7.104,
     * 6.491,6.583,6.667,6.705,6.723,6.744,6.849,7.061,
     * 0.000,0.000,0.000,0.000,0.000,6.068,6.393,6.821,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.312,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.266/
      DATA ((EDEN(I,2,5,L,1),I=17,24),L=1,NZEN)/
     * 8.732,8.744,8.720,8.704,8.678,8.653,8.640,8.720,
     * 8.690,8.708,8.689,8.671,8.642,8.612,8.595,8.631,
     * 8.590,8.617,8.610,8.600,8.580,8.555,8.548,8.589,
     * 8.400,8.442,8.455,8.464,8.462,8.453,8.465,8.520,
     * 7.950,8.029,8.083,8.140,8.179,8.212,8.276,8.373,
     * 7.618,7.722,7.805,7.884,7.950,8.013,8.117,8.253,
     * 7.164,7.233,7.292,7.387,7.468,7.580,7.741,7.955,
     * 6.822,6.865,6.910,6.924,6.922,6.931,7.013,7.207,
     * 6.569,6.655,6.748,6.792,6.818,6.849,6.960,7.170,
     * 0.000,0.000,0.000,0.000,0.000,6.161,6.498,6.928,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.430,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.384/
      DATA ((EDEN(I,3,5,L,1),I=17,24),L=1,NZEN)/
     * 8.772,8.773,8.751,8.740,8.719,8.707,8.702,8.794,
     * 8.714,8.718,8.700,8.688,8.667,8.651,8.646,8.692,
     * 8.611,8.623,8.617,8.613,8.599,8.589,8.592,8.643,
     * 8.417,8.446,8.458,8.471,8.474,8.477,8.501,8.566,
     * 7.969,8.033,8.086,8.137,8.179,8.223,8.297,8.403,
     * 7.649,7.732,7.810,7.885,7.947,8.021,8.130,8.274,
     * 7.267,7.303,7.340,7.423,7.491,7.599,7.752,7.969,
     * 6.940,6.968,7.013,7.029,7.029,7.041,7.124,7.303,
     * 6.681,6.754,6.849,6.897,6.927,6.963,7.076,7.279,
     * 0.000,0.000,0.000,0.000,0.000,6.267,6.611,7.037,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.562,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.517/
      DATA ((EDEN(I,4,5,L,1),I=17,24),L=1,NZEN)/
     * 8.793,8.778,8.754,8.744,8.727,8.719,8.718,8.818,
     * 8.732,8.720,8.700,8.691,8.674,8.669,8.672,8.723,
     * 8.624,8.621,8.613,8.611,8.600,8.600,8.611,8.669,
     * 8.427,8.439,8.449,8.462,8.467,8.479,8.508,8.580,
     * 7.983,8.025,8.076,8.124,8.161,8.210,8.283,8.400,
     * 7.694,7.749,7.813,7.881,7.936,8.009,8.111,8.260,
     * 7.408,7.407,7.420,7.479,7.524,7.615,7.749,7.955,
     * 7.090,7.104,7.146,7.164,7.164,7.176,7.248,7.412,
     * 6.820,6.881,6.980,7.029,7.061,7.100,7.210,7.396,
     * 0.000,0.000,0.000,0.000,6.124,6.398,6.746,7.155,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.715,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.669/
      DATA ((EDEN(I,5,5,L,1),I=17,24),L=1,NZEN)/
     * 8.740,8.719,8.702,8.704,8.698,8.705,8.714,8.827,
     * 8.682,8.665,8.654,8.660,8.658,8.666,8.678,8.740,
     * 8.566,8.558,8.558,8.569,8.573,8.584,8.604,8.676,
     * 8.360,8.362,8.377,8.405,8.423,8.446,8.480,8.567,
     * 7.949,7.967,8.004,8.057,8.097,8.149,8.225,8.352,
     * 7.741,7.748,7.777,7.836,7.881,7.948,8.045,8.199,
     * 7.568,7.533,7.526,7.548,7.555,7.618,7.719,7.905,
     * 7.246,7.248,7.294,7.314,7.316,7.322,7.389,7.529,
     * 6.954,7.009,7.114,7.173,7.212,7.248,7.352,7.522,
     * 0.000,0.000,0.000,6.068,6.241,6.525,6.879,7.281,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.107,6.876,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.061,6.830/
      DATA ((EDEN(I,1,6,L,1),I=17,24),L=1,NZEN)/
     * 8.688,8.708,8.678,8.652,8.615,8.579,8.556,8.625,
     * 8.652,8.680,8.659,8.631,8.591,8.548,8.521,8.550,
     * 8.555,8.593,8.583,8.565,8.533,8.494,8.477,8.512,
     * 8.371,8.427,8.436,8.436,8.423,8.398,8.400,8.449,
     * 7.931,8.021,8.076,8.124,8.158,8.173,8.225,8.316,
     * 7.602,7.716,7.796,7.873,7.936,7.985,8.079,8.204,
     * 7.097,7.196,7.265,7.367,7.452,7.560,7.721,7.925,
     * 6.738,6.787,6.824,6.832,6.822,6.826,6.906,7.111,
     * 6.502,6.591,6.672,6.705,6.719,6.741,6.848,7.064,
     * 0.000,0.000,0.000,0.000,0.000,6.090,6.410,6.832,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.324,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.279/
      DATA ((EDEN(I,2,6,L,1),I=17,24),L=1,NZEN)/
     * 8.752,8.767,8.747,8.735,8.711,8.689,8.676,8.750,
     * 8.708,8.724,8.708,8.692,8.665,8.634,8.617,8.651,
     * 8.609,8.634,8.629,8.621,8.602,8.577,8.568,8.610,
     * 8.422,8.465,8.480,8.490,8.487,8.477,8.487,8.540,
     * 7.981,8.061,8.117,8.170,8.210,8.241,8.301,8.396,
     * 7.655,7.756,7.839,7.920,7.986,8.049,8.149,8.279,
     * 7.182,7.260,7.328,7.428,7.513,7.625,7.785,7.992,
     * 6.839,6.881,6.925,6.942,6.940,6.952,7.033,7.225,
     * 6.599,6.680,6.772,6.816,6.839,6.871,6.980,7.188,
     * 0.000,0.000,0.000,0.000,0.000,6.215,6.543,6.956,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.464,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.418/
      DATA ((EDEN(I,3,6,L,1),I=17,24),L=1,NZEN)/
     * 8.780,8.780,8.760,8.752,8.732,8.718,8.706,8.790,
     * 8.737,8.741,8.728,8.720,8.703,8.688,8.678,8.716,
     * 8.635,8.647,8.643,8.644,8.634,8.624,8.622,8.667,
     * 8.444,8.471,8.486,8.504,8.511,8.513,8.530,8.587,
     * 8.004,8.064,8.121,8.176,8.217,8.260,8.328,8.425,
     * 7.687,7.769,7.852,7.930,7.995,8.064,8.164,8.299,
     * 7.292,7.334,7.384,7.473,7.545,7.650,7.797,8.004,
     * 6.967,6.996,7.045,7.068,7.072,7.083,7.158,7.330,
     * 6.718,6.790,6.888,6.940,6.971,7.004,7.111,7.303,
     * 0.000,0.000,0.000,0.000,6.076,6.338,6.672,7.076,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.605,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.560/
      DATA ((EDEN(I,4,6,L,1),I=17,24),L=1,NZEN)/
     * 8.787,8.771,8.754,8.745,8.726,8.716,8.702,8.785,
     * 8.733,8.721,8.708,8.704,8.688,8.687,8.683,8.727,
     * 8.626,8.621,8.618,8.621,8.612,8.614,8.618,8.668,
     * 8.428,8.439,8.455,8.471,8.476,8.490,8.511,8.575,
     * 7.986,8.029,8.086,8.134,8.170,8.215,8.279,8.386,
     * 7.706,7.758,7.826,7.897,7.952,8.017,8.104,8.246,
     * 7.447,7.441,7.456,7.513,7.554,7.637,7.755,7.943,
     * 7.127,7.140,7.193,7.217,7.220,7.230,7.294,7.438,
     * 6.844,6.911,7.021,7.083,7.121,7.155,7.253,7.423,
     * 0.000,0.000,0.000,6.009,6.188,6.458,6.794,7.188,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.754,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.709/
      DATA ((EDEN(I,5,6,L,1),I=17,24),L=1,NZEN)/
     * 8.721,8.707,8.706,8.715,8.713,8.712,8.702,8.785,
     * 8.658,8.646,8.649,8.665,8.667,8.673,8.670,8.718,
     * 8.535,8.531,8.545,8.568,8.579,8.587,8.591,8.649,
     * 8.324,8.328,8.354,8.394,8.420,8.439,8.461,8.533,
     * 7.939,7.947,7.983,8.037,8.079,8.130,8.193,8.303,
     * 7.775,7.763,7.784,7.836,7.873,7.931,8.009,8.143,
     * 7.635,7.594,7.593,7.605,7.600,7.637,7.704,7.859,
     * 7.283,7.297,7.364,7.394,7.403,7.405,7.455,7.571,
     * 6.954,7.025,7.161,7.236,7.288,7.324,7.420,7.565,
     * 0.000,0.000,0.000,6.097,6.276,6.567,6.915,7.310,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.161,6.917,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.116,6.871/
      DATA ((EDEN(I,1,7,L,1),I=17,24),L=1,NZEN)/
     * 8.702,8.723,8.694,8.667,8.629,8.592,8.569,8.636,
     * 8.649,8.674,8.649,8.617,8.573,8.531,8.507,8.540,
     * 8.551,8.585,8.572,8.550,8.515,8.479,8.462,8.502,
     * 8.367,8.417,8.425,8.422,8.405,8.382,8.387,8.439,
     * 7.922,8.013,8.061,8.107,8.134,8.152,8.210,8.303,
     * 7.592,7.704,7.782,7.854,7.910,7.963,8.061,8.190,
     * 7.093,7.188,7.253,7.346,7.427,7.537,7.700,7.910,
     * 6.738,6.786,6.821,6.826,6.814,6.818,6.899,7.104,
     * 6.497,6.584,6.666,6.698,6.711,6.735,6.843,7.061,
     * 0.000,0.000,0.000,0.000,0.000,6.076,6.400,6.824,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.320,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.274/
      DATA ((EDEN(I,2,7,L,1),I=17,24),L=1,NZEN)/
     * 8.772,8.787,8.769,8.757,8.735,8.712,8.696,8.768,
     * 8.728,8.746,8.731,8.714,8.686,8.657,8.639,8.671,
     * 8.630,8.657,8.652,8.643,8.623,8.600,8.590,8.627,
     * 8.444,8.487,8.502,8.513,8.512,8.501,8.509,8.561,
     * 8.004,8.083,8.143,8.199,8.238,8.267,8.326,8.418,
     * 7.679,7.782,7.868,7.949,8.017,8.079,8.176,8.303,
     * 7.199,7.279,7.350,7.453,7.542,7.657,7.819,8.021,
     * 6.850,6.891,6.937,6.954,6.953,6.966,7.045,7.241,
     * 6.614,6.694,6.784,6.827,6.852,6.884,6.993,7.199,
     * 0.000,0.000,0.000,0.000,0.000,6.233,6.559,6.971,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.476,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.430/
      DATA ((EDEN(I,3,7,L,1),I=17,24),L=1,NZEN)/
     * 8.801,8.801,8.784,8.778,8.761,8.747,8.732,8.813,
     * 8.780,8.786,8.776,8.771,8.755,8.738,8.724,8.759,
     * 8.680,8.695,8.695,8.698,8.689,8.676,8.670,8.709,
     * 8.491,8.521,8.542,8.561,8.568,8.569,8.582,8.633,
     * 8.057,8.121,8.179,8.241,8.286,8.326,8.387,8.477,
     * 7.740,7.825,7.912,7.996,8.064,8.134,8.230,8.356,
     * 7.316,7.369,7.430,7.529,7.612,7.723,7.873,8.072,
     * 6.987,7.013,7.064,7.090,7.093,7.107,7.182,7.360,
     * 6.751,6.818,6.914,6.966,6.997,7.033,7.137,7.326,
     * 0.000,0.000,0.000,0.000,6.127,6.382,6.708,7.104,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.631,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.586/
      DATA ((EDEN(I,4,7,L,1),I=17,24),L=1,NZEN)/
     * 8.816,8.801,8.785,8.780,8.763,8.754,8.739,8.820,
     * 8.778,8.768,8.758,8.758,8.747,8.745,8.739,8.775,
     * 8.673,8.671,8.672,8.679,8.673,8.675,8.677,8.720,
     * 8.477,8.491,8.512,8.533,8.540,8.554,8.573,8.630,
     * 8.037,8.086,8.146,8.201,8.243,8.290,8.350,8.450,
     * 7.750,7.810,7.886,7.961,8.021,8.090,8.182,8.314,
     * 7.462,7.465,7.493,7.562,7.615,7.706,7.829,8.017,
     * 7.146,7.158,7.215,7.241,7.248,7.258,7.320,7.465,
     * 6.875,6.940,7.049,7.111,7.149,7.185,7.281,7.449,
     * 0.000,0.000,0.000,6.057,6.241,6.508,6.838,7.220,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.025,6.783,
     * 0.000,0.000,0.000,0.000,0.000,0.000,5.980,6.737/
      DATA ((EDEN(I,5,7,L,1),I=17,24),L=1,NZEN)/
     * 8.770,8.754,8.755,8.766,8.766,8.765,8.751,8.830,
     * 8.705,8.692,8.697,8.713,8.718,8.724,8.721,8.763,
     * 8.585,8.582,8.598,8.621,8.631,8.642,8.645,8.696,
     * 8.377,8.384,8.417,8.455,8.479,8.500,8.520,8.588,
     * 7.979,7.996,8.045,8.107,8.152,8.204,8.262,8.369,
     * 7.798,7.797,7.831,7.895,7.943,8.009,8.086,8.215,
     * 7.650,7.612,7.615,7.637,7.642,7.695,7.772,7.928,
     * 7.303,7.316,7.386,7.418,7.430,7.433,7.481,7.594,
     * 6.977,7.049,7.188,7.265,7.318,7.354,7.446,7.589,
     * 0.000,0.000,0.000,6.146,6.330,6.616,6.955,7.338,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.201,6.943,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.156,6.897/
      DATA ((EDEN(I,1,8,L,1),I=17,24),L=1,NZEN)/
     * 8.702,8.722,8.691,8.662,8.621,8.581,8.556,8.623,
     * 8.641,8.664,8.637,8.605,8.561,8.520,8.496,8.528,
     * 8.542,8.574,8.560,8.537,8.500,8.464,8.450,8.489,
     * 8.352,8.401,8.407,8.403,8.387,8.367,8.373,8.425,
     * 7.902,7.989,8.037,8.079,8.111,8.130,8.190,8.286,
     * 7.566,7.678,7.755,7.825,7.879,7.936,8.037,8.170,
     * 7.083,7.167,7.225,7.316,7.391,7.500,7.664,7.878,
     * 6.734,6.783,6.820,6.824,6.811,6.814,6.893,7.093,
     * 6.483,6.574,6.658,6.690,6.704,6.729,6.838,7.053,
     * 0.000,0.000,0.000,0.000,0.000,6.045,6.375,6.807,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.301,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.255/
      DATA ((EDEN(I,2,8,L,1),I=17,24),L=1,NZEN)/
     * 8.778,8.794,8.775,8.761,8.736,8.711,8.694,8.765,
     * 8.728,8.746,8.730,8.711,8.680,8.651,8.633,8.665,
     * 8.628,8.654,8.649,8.639,8.619,8.594,8.584,8.622,
     * 8.439,8.483,8.498,8.508,8.505,8.494,8.504,8.555,
     * 7.989,8.072,8.130,8.185,8.225,8.255,8.318,8.412,
     * 7.658,7.762,7.848,7.929,7.997,8.061,8.164,8.294,
     * 7.190,7.265,7.328,7.430,7.515,7.631,7.795,8.004,
     * 6.844,6.887,6.932,6.950,6.949,6.960,7.041,7.233,
     * 6.597,6.679,6.772,6.818,6.844,6.876,6.987,7.196,
     * 0.000,0.000,0.000,0.000,0.000,6.193,6.528,6.953,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.453,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.408/
      DATA ((EDEN(I,3,8,L,1),I=17,24),L=1,NZEN)/
     * 8.831,8.834,8.818,8.815,8.801,8.788,8.779,8.857,
     * 8.807,8.815,8.805,8.798,8.780,8.761,8.748,8.780,
     * 8.704,8.722,8.724,8.726,8.717,8.702,8.698,8.736,
     * 8.516,8.550,8.572,8.592,8.600,8.599,8.614,8.662,
     * 8.072,8.143,8.207,8.270,8.318,8.358,8.423,8.513,
     * 7.750,7.841,7.931,8.021,8.093,8.164,8.267,8.393,
     * 7.318,7.375,7.435,7.539,7.627,7.745,7.904,8.107,
     * 6.985,7.013,7.064,7.086,7.093,7.107,7.185,7.369,
     * 6.744,6.812,6.907,6.960,6.992,7.029,7.137,7.334,
     * 0.000,0.000,0.000,0.000,6.097,6.358,6.692,7.100,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.620,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.574/
      DATA ((EDEN(I,4,8,L,1),I=17,24),L=1,NZEN)/
     * 8.874,8.862,8.846,8.848,8.839,8.834,8.827,8.911,
     * 8.836,8.831,8.824,8.827,8.819,8.813,8.808,8.841,
     * 8.734,8.737,8.741,8.751,8.751,8.749,8.751,8.792,
     * 8.542,8.561,8.584,8.611,8.625,8.637,8.659,8.712,
     * 8.104,8.155,8.220,8.283,8.334,8.384,8.455,8.549,
     * 7.799,7.870,7.955,8.041,8.111,8.188,8.292,8.423,
     * 7.458,7.476,7.515,7.604,7.677,7.785,7.929,8.127,
     * 7.140,7.155,7.207,7.233,7.241,7.255,7.326,7.487,
     * 6.884,6.944,7.049,7.104,7.140,7.179,7.283,7.462,
     * 0.000,0.000,0.000,6.064,6.243,6.509,6.842,7.233,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.013,6.785,
     * 0.000,0.000,0.000,0.000,0.000,0.000,5.967,6.739/
      DATA ((EDEN(I,5,8,L,1),I=17,24),L=1,NZEN)/
     * 8.903,8.879,8.867,8.874,8.871,8.874,8.870,8.960,
     * 8.835,8.818,8.815,8.824,8.822,8.827,8.827,8.865,
     * 8.728,8.719,8.728,8.743,8.747,8.755,8.762,8.809,
     * 8.531,8.538,8.565,8.593,8.611,8.631,8.654,8.715,
     * 8.100,8.137,8.199,8.262,8.312,8.362,8.427,8.528,
     * 7.843,7.879,7.947,8.029,8.093,8.164,8.255,8.387,
     * 7.620,7.598,7.613,7.670,7.711,7.797,7.913,8.093,
     * 7.297,7.303,7.364,7.394,7.405,7.415,7.476,7.607,
     * 7.009,7.068,7.190,7.258,7.301,7.340,7.436,7.594,
     * 0.000,0.000,0.000,6.185,6.365,6.642,6.979,7.360,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.201,6.948,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.156,6.902/
      DATA ((EDEN(I,1,9,L,1),I=17,24),L=1,NZEN)/
     * 8.689,8.708,8.676,8.645,8.603,8.562,8.537,8.602,
     * 8.642,8.667,8.644,8.614,8.572,8.526,8.500,8.528,
     * 8.544,8.577,8.565,8.544,8.511,8.471,8.453,8.489,
     * 8.352,8.405,8.412,8.412,8.398,8.375,8.378,8.427,
     * 7.898,7.987,8.037,8.083,8.114,8.134,8.190,8.286,
     * 7.559,7.672,7.752,7.825,7.883,7.937,8.033,8.164,
     * 7.079,7.164,7.220,7.312,7.387,7.494,7.655,7.867,
     * 6.734,6.785,6.823,6.829,6.816,6.818,6.894,7.090,
     * 6.477,6.571,6.658,6.692,6.708,6.731,6.839,7.053,
     * 0.000,0.000,0.000,0.000,0.000,6.033,6.365,6.803,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.294,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.249/
      DATA ((EDEN(I,2,9,L,1),I=17,24),L=1,NZEN)/
     * 8.775,8.789,8.766,8.748,8.720,8.692,8.678,8.750,
     * 8.726,8.744,8.726,8.705,8.673,8.641,8.625,8.657,
     * 8.625,8.652,8.645,8.634,8.612,8.587,8.579,8.618,
     * 8.433,8.479,8.493,8.501,8.497,8.487,8.501,8.553,
     * 7.979,8.061,8.121,8.173,8.215,8.248,8.314,8.410,
     * 7.643,7.748,7.833,7.915,7.983,8.049,8.155,8.290,
     * 7.176,7.250,7.310,7.408,7.491,7.609,7.777,7.993,
     * 6.831,6.874,6.919,6.934,6.932,6.943,7.025,7.223,
     * 6.579,6.664,6.754,6.798,6.823,6.858,6.973,7.188,
     * 0.000,0.000,0.000,0.000,0.000,6.161,6.501,6.938,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.433,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.387/
      DATA ((EDEN(I,3,9,L,1),I=17,24),L=1,NZEN)/
     * 8.863,8.871,8.856,8.852,8.840,8.827,8.825,8.902,
     * 8.817,8.827,8.816,8.804,8.783,8.763,8.756,8.792,
     * 8.717,8.737,8.736,8.734,8.722,8.707,8.708,8.750,
     * 8.526,8.565,8.584,8.602,8.609,8.607,8.627,8.683,
     * 8.076,8.152,8.217,8.279,8.330,8.371,8.442,8.540,
     * 7.749,7.844,7.933,8.025,8.104,8.176,8.288,8.423,
     * 7.301,7.362,7.422,7.528,7.619,7.745,7.919,8.134,
     * 6.962,6.993,7.037,7.061,7.064,7.083,7.167,7.365,
     * 6.719,6.789,6.881,6.929,6.959,6.998,7.117,7.328,
     * 0.000,0.000,0.000,0.000,6.053,6.314,6.657,7.086,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.595,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.550/
      DATA ((EDEN(I,4,9,L,1),I=17,24),L=1,NZEN)/
     * 8.961,8.961,8.950,8.960,8.962,8.960,8.967,9.049,
     * 8.912,8.917,8.912,8.910,8.901,8.888,8.885,8.925,
     * 8.814,8.828,8.834,8.841,8.840,8.833,8.839,8.883,
     * 8.628,8.659,8.687,8.712,8.728,8.735,8.760,8.816,
     * 8.188,8.255,8.326,8.396,8.458,8.505,8.581,8.676,
     * 7.872,7.959,8.053,8.152,8.238,8.318,8.433,8.563,
     * 7.444,7.493,7.554,7.668,7.769,7.900,8.076,8.286,
     * 7.117,7.130,7.176,7.201,7.210,7.230,7.320,7.520,
     * 6.884,6.937,7.025,7.076,7.111,7.152,7.270,7.480,
     * 0.000,0.000,0.000,6.064,6.238,6.501,6.841,7.253,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.781,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.735/
      DATA ((EDEN(I,5,9,L,1),I=17,24),L=1,NZEN)/
     * 9.076,9.072,9.068,9.093,9.111,9.117,9.130,9.207,
     * 9.017,9.017,9.021,9.029,9.029,9.021,9.021,9.061,
     * 8.922,8.930,8.944,8.960,8.968,8.968,8.979,9.017,
     * 8.738,8.764,8.799,8.833,8.859,8.872,8.900,8.950,
     * 8.305,8.367,8.446,8.525,8.594,8.647,8.727,8.816,
     * 7.996,8.076,8.176,8.283,8.378,8.465,8.583,8.706,
     * 7.590,7.626,7.689,7.811,7.919,8.057,8.238,8.438,
     * 7.267,7.272,7.318,7.348,7.362,7.391,7.483,7.678,
     * 7.041,7.083,7.173,7.228,7.265,7.314,7.433,7.635,
     * 0.000,0.000,6.049,6.243,6.418,6.683,7.021,7.417,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.207,6.966,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.161,6.920/
      DATA ((EDEN(I,1,10,L,1),I=17,24),L=1,NZEN)/
     * 8.681,8.699,8.667,8.638,8.597,8.558,8.533,8.600,
     * 8.659,8.689,8.671,8.644,8.605,8.555,8.521,8.542,
     * 8.560,8.600,8.592,8.575,8.547,8.501,8.477,8.505,
     * 8.367,8.427,8.441,8.444,8.435,8.405,8.401,8.442,
     * 7.912,8.009,8.064,8.117,8.155,8.167,8.217,8.301,
     * 7.574,7.692,7.776,7.856,7.922,7.968,8.057,8.182,
     * 7.090,7.179,7.241,7.338,7.422,7.524,7.679,7.881,
     * 6.740,6.794,6.834,6.845,6.839,6.835,6.907,7.100,
     * 6.486,6.581,6.669,6.709,6.730,6.748,6.851,7.061,
     * 0.000,0.000,0.000,0.000,0.000,6.045,6.375,6.809,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.297,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.251/
      DATA ((EDEN(I,2,10,L,1),I=17,24),L=1,NZEN)/
     * 8.768,8.782,8.754,8.736,8.708,8.679,8.668,8.743,
     * 8.733,8.755,8.738,8.716,8.684,8.647,8.629,8.662,
     * 8.634,8.665,8.658,8.646,8.625,8.594,8.585,8.625,
     * 8.442,8.491,8.507,8.515,8.513,8.497,8.511,8.562,
     * 7.985,8.072,8.130,8.188,8.233,8.260,8.324,8.422,
     * 7.646,7.757,7.844,7.927,7.998,8.061,8.167,8.303,
     * 7.167,7.246,7.307,7.410,7.500,7.616,7.787,8.004,
     * 6.818,6.865,6.907,6.923,6.922,6.932,7.017,7.220,
     * 6.567,6.653,6.743,6.786,6.812,6.844,6.962,7.182,
     * 0.000,0.000,0.000,0.000,0.000,6.143,6.487,6.932,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.420,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.374/
      DATA ((EDEN(I,3,10,L,1),I=17,24),L=1,NZEN)/
     * 8.892,8.903,8.884,8.880,8.869,8.852,8.854,8.929,
     * 8.829,8.843,8.827,8.808,8.782,8.760,8.758,8.803,
     * 8.731,8.754,8.751,8.741,8.724,8.707,8.713,8.765,
     * 8.542,8.582,8.599,8.611,8.614,8.612,8.638,8.703,
     * 8.090,8.170,8.233,8.292,8.342,8.380,8.459,8.567,
     * 7.756,7.859,7.947,8.037,8.114,8.188,8.310,8.455,
     * 7.281,7.350,7.413,7.522,7.620,7.752,7.940,8.164,
     * 6.936,6.967,7.004,7.021,7.025,7.045,7.146,7.364,
     * 6.695,6.765,6.847,6.891,6.920,6.962,7.090,7.322,
     * 0.000,0.000,0.000,0.000,6.017,6.279,6.627,7.076,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.574,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.528/
      DATA ((EDEN(I,4,10,L,1),I=17,24),L=1,NZEN)/
     * 9.076,9.083,9.068,9.079,9.086,9.079,9.086,9.146,
     * 8.996,9.004,8.993,8.978,8.959,8.946,8.955,9.004,
     * 8.903,8.921,8.921,8.915,8.904,8.895,8.911,8.967,
     * 8.723,8.760,8.783,8.797,8.804,8.808,8.843,8.910,
     * 8.286,8.367,8.439,8.504,8.561,8.601,8.681,8.788,
     * 7.961,8.064,8.164,8.262,8.354,8.427,8.547,8.688,
     * 7.455,7.537,7.620,7.755,7.880,8.021,8.217,8.433,
     * 7.093,7.114,7.149,7.173,7.185,7.215,7.324,7.567,
     * 6.880,6.930,7.004,7.049,7.083,7.127,7.262,7.507,
     * 0.000,0.000,0.000,6.064,6.236,6.491,6.838,7.279,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.782,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.737/
      DATA ((EDEN(I,5,10,L,1),I=17,24),L=1,NZEN)/
     * 9.338,9.328,9.305,9.326,9.344,9.328,9.332,9.371,
     * 9.230,9.223,9.204,9.188,9.170,9.158,9.173,9.217,
     * 9.143,9.146,9.140,9.134,9.121,9.114,9.137,9.185,
     * 8.975,8.999,9.017,9.025,9.029,9.033,9.072,9.134,
     * 8.559,8.631,8.708,8.762,8.810,8.846,8.924,9.025,
     * 8.236,8.338,8.447,8.540,8.626,8.691,8.806,8.937,
     * 7.682,7.783,7.897,8.049,8.190,8.324,8.513,8.714,
     * 7.274,7.281,7.318,7.348,7.369,7.410,7.531,7.803,
     * 7.079,7.114,7.182,7.230,7.267,7.318,7.456,7.713,
     * 0.000,0.000,6.107,6.292,6.464,6.717,7.061,7.498,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.204,6.999,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.158,6.953/
      DATA ((EDEN(I,1,11,L,1),I=17,24),L=1,NZEN)/
     * 8.685,8.706,8.678,8.652,8.616,8.579,8.556,8.624,
     * 8.664,8.695,8.679,8.652,8.614,8.562,8.528,8.547,
     * 8.566,8.605,8.599,8.583,8.554,8.509,8.486,8.509,
     * 8.375,8.433,8.447,8.452,8.442,8.412,8.407,8.444,
     * 7.919,8.013,8.072,8.124,8.164,8.173,8.223,8.303,
     * 7.579,7.697,7.782,7.865,7.932,7.976,8.064,8.185,
     * 7.093,7.182,7.246,7.346,7.430,7.530,7.683,7.884,
     * 6.741,6.796,6.837,6.849,6.844,6.839,6.909,7.100,
     * 6.491,6.587,6.674,6.713,6.734,6.751,6.853,7.061,
     * 0.000,0.000,0.000,0.000,0.000,6.053,6.382,6.814,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.299,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.253/
      DATA ((EDEN(I,2,11,L,1),I=17,24),L=1,NZEN)/
     * 8.766,8.780,8.750,8.730,8.702,8.676,8.670,8.749,
     * 8.736,8.759,8.740,8.715,8.681,8.642,8.627,8.664,
     * 8.635,8.668,8.662,8.647,8.623,8.591,8.587,8.628,
     * 8.446,8.497,8.511,8.517,8.513,8.496,8.509,8.566,
     * 7.988,8.079,8.137,8.193,8.238,8.262,8.328,8.428,
     * 7.649,7.761,7.846,7.931,8.004,8.064,8.173,8.312,
     * 7.155,7.241,7.305,7.410,7.502,7.618,7.792,8.013,
     * 6.804,6.851,6.890,6.904,6.903,6.913,7.004,7.217,
     * 6.556,6.641,6.727,6.769,6.794,6.826,6.946,7.176,
     * 0.000,0.000,0.000,0.000,0.000,6.134,6.476,6.926,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.412,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.366/
      DATA ((EDEN(I,3,11,L,1),I=17,24),L=1,NZEN)/
     * 8.908,8.918,8.892,8.883,8.869,8.850,8.857,8.932,
     * 8.849,8.863,8.841,8.813,8.780,8.757,8.763,8.818,
     * 8.751,8.775,8.766,8.749,8.726,8.708,8.723,8.785,
     * 8.565,8.607,8.621,8.624,8.621,8.617,8.651,8.727,
     * 8.114,8.199,8.260,8.314,8.358,8.394,8.480,8.597,
     * 7.778,7.885,7.971,8.057,8.137,8.210,8.334,8.489,
     * 7.270,7.354,7.423,7.538,7.642,7.777,7.976,8.210,
     * 6.913,6.944,6.976,6.991,6.993,7.017,7.127,7.367,
     * 6.683,6.750,6.823,6.861,6.886,6.929,7.068,7.318,
     * 0.000,0.000,0.000,0.000,6.000,6.260,6.613,7.076,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.562,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.517/
      DATA ((EDEN(I,4,11,L,1),I=17,24),L=1,NZEN)/
     * 9.176,9.167,9.130,9.130,9.130,9.104,9.107,9.164,
     * 9.061,9.064,9.037,9.009,8.982,8.968,8.989,9.049,
     * 8.973,8.986,8.973,8.953,8.929,8.921,8.950,9.017,
     * 8.801,8.835,8.846,8.843,8.838,8.841,8.888,8.969,
     * 8.375,8.458,8.524,8.569,8.610,8.647,8.739,8.860,
     * 8.049,8.158,8.258,8.340,8.420,8.486,8.613,8.768,
     * 7.493,7.600,7.701,7.838,7.969,8.104,8.307,8.534,
     * 7.086,7.107,7.137,7.155,7.167,7.204,7.328,7.616,
     * 6.886,6.934,6.996,7.033,7.064,7.111,7.258,7.535,
     * 0.000,0.000,0.000,6.064,6.233,6.487,6.840,7.310,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.790,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.745/
      DATA ((EDEN(I,5,11,L,1),I=17,24),L=1,NZEN)/
     * 9.509,9.470,9.405,9.396,9.389,9.352,9.354,9.393,
     * 9.303,9.292,9.260,9.230,9.201,9.193,9.228,9.288,
     * 9.223,9.220,9.199,9.176,9.155,9.152,9.193,9.262,
     * 9.068,9.086,9.086,9.079,9.072,9.079,9.137,9.220,
     * 8.679,8.746,8.801,8.834,8.865,8.906,9.004,9.130,
     * 8.365,8.468,8.565,8.631,8.696,8.763,8.896,9.053,
     * 7.773,7.900,8.029,8.167,8.301,8.430,8.632,8.853,
     * 7.286,7.299,7.326,7.352,7.371,7.427,7.576,7.913,
     * 7.100,7.137,7.190,7.230,7.262,7.318,7.479,7.787,
     * 0.000,0.000,6.124,6.301,6.471,6.729,7.090,7.575,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.204,7.041,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.158,6.996/
      DATA ((EDEN(I,1,12,L,1),I=17,24),L=1,NZEN)/
     * 8.699,8.723,8.697,8.677,8.644,8.611,8.592,8.656,
     * 8.649,8.677,8.655,8.627,8.588,8.538,8.507,8.530,
     * 8.548,8.585,8.575,8.556,8.526,8.486,8.465,8.494,
     * 8.356,8.410,8.422,8.423,8.412,8.386,8.386,8.430,
     * 7.895,7.987,8.041,8.090,8.127,8.143,8.196,8.286,
     * 7.556,7.671,7.752,7.828,7.890,7.941,8.033,8.161,
     * 7.083,7.164,7.220,7.312,7.387,7.491,7.648,7.856,
     * 6.734,6.787,6.826,6.834,6.825,6.824,6.899,7.090,
     * 6.477,6.572,6.660,6.697,6.716,6.737,6.843,7.053,
     * 0.000,0.000,0.000,0.000,0.000,6.033,6.362,6.801,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.288,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.242/
      DATA ((EDEN(I,2,12,L,1),I=17,24),L=1,NZEN)/
     * 8.769,8.783,8.753,8.735,8.709,8.681,8.678,8.758,
     * 8.710,8.731,8.708,8.679,8.643,8.610,8.601,8.644,
     * 8.610,8.641,8.629,8.612,8.584,8.555,8.556,8.607,
     * 8.420,8.468,8.477,8.479,8.473,8.459,8.481,8.547,
     * 7.961,8.049,8.100,8.152,8.193,8.223,8.299,8.410,
     * 7.621,7.731,7.812,7.891,7.959,8.025,8.140,8.290,
     * 7.134,7.215,7.272,7.369,7.455,7.575,7.757,7.989,
     * 6.784,6.828,6.863,6.873,6.869,6.883,6.980,7.199,
     * 6.534,6.619,6.701,6.737,6.759,6.794,6.922,7.158,
     * 0.000,0.000,0.000,0.000,0.000,6.100,6.449,6.908,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.393,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.347/
      DATA ((EDEN(I,3,12,L,1),I=17,24),L=1,NZEN)/
     * 8.910,8.917,8.884,8.869,8.851,8.829,8.839,8.919,
     * 8.826,8.839,8.814,8.782,8.747,8.725,8.738,8.803,
     * 8.732,8.754,8.739,8.718,8.692,8.675,8.696,8.769,
     * 8.544,8.587,8.595,8.594,8.588,8.585,8.626,8.713,
     * 8.093,8.176,8.233,8.283,8.326,8.364,8.456,8.585,
     * 7.757,7.863,7.944,8.029,8.107,8.179,8.312,8.476,
     * 7.243,7.328,7.393,7.505,7.607,7.747,7.955,8.199,
     * 6.884,6.914,6.941,6.951,6.950,6.978,7.097,7.350,
     * 6.652,6.718,6.786,6.821,6.843,6.890,7.037,7.299,
     * 0.000,0.000,0.000,0.000,0.000,6.220,6.581,7.057,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.540,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.495/
      DATA ((EDEN(I,4,12,L,1),I=17,24),L=1,NZEN)/
     * 9.196,9.176,9.124,9.111,9.097,9.064,9.072,9.134,
     * 9.068,9.068,9.033,8.996,8.959,8.947,8.978,9.049,
     * 8.984,8.992,8.967,8.939,8.910,8.903,8.941,9.021,
     * 8.817,8.845,8.844,8.832,8.818,8.822,8.877,8.972,
     * 8.398,8.479,8.534,8.568,8.599,8.636,8.735,8.869,
     * 8.072,8.185,8.274,8.346,8.415,8.480,8.615,8.781,
     * 7.497,7.614,7.717,7.851,7.979,8.111,8.322,8.558,
     * 7.064,7.086,7.104,7.121,7.130,7.170,7.312,7.626,
     * 6.869,6.914,6.967,6.999,7.025,7.076,7.233,7.533,
     * 0.000,0.000,0.000,6.049,6.212,6.465,6.827,7.312,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.784,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.738/
      DATA ((EDEN(I,5,12,L,1),I=17,24),L=1,NZEN)/
     * 9.529,9.474,9.387,9.362,9.340,9.299,9.305,9.362,
     * 9.301,9.288,9.246,9.204,9.170,9.164,9.207,9.286,
     * 9.225,9.220,9.185,9.152,9.124,9.124,9.176,9.262,
     * 9.072,9.086,9.072,9.057,9.041,9.053,9.121,9.223,
     * 8.695,8.756,8.798,8.820,8.843,8.885,8.997,9.140,
     * 8.387,8.486,8.569,8.624,8.679,8.748,8.893,9.064,
     * 7.781,7.918,8.045,8.170,8.299,8.427,8.641,8.880,
     * 7.255,7.270,7.288,7.310,7.326,7.393,7.565,7.941,
     * 7.072,7.107,7.152,7.185,7.215,7.276,7.455,7.797,
     * 0.000,0.000,6.100,6.272,6.436,6.698,7.076,7.585,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.185,7.041,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.139,6.996/
      DATA ((EDEN(I,1,1,L,2),I=17,24),L=1,NZEN)/
     * 8.813,8.851,8.839,8.831,8.810,8.790,8.784,8.842,
     * 8.754,8.791,8.779,8.760,8.728,8.697,8.680,8.708,
     * 8.668,8.716,8.716,8.708,8.686,8.659,8.647,8.675,
     * 8.500,8.569,8.594,8.609,8.609,8.595,8.595,8.624,
     * 8.083,8.199,8.276,8.344,8.398,8.428,8.470,8.521,
     * 7.768,7.909,8.017,8.117,8.204,8.270,8.346,8.423,
     * 7.336,7.449,7.531,7.654,7.759,7.881,8.013,8.152,
     * 7.121,7.212,7.288,7.332,7.352,7.354,7.360,7.433,
     * 6.908,7.049,7.182,7.258,7.307,7.328,7.354,7.427,
     * 0.000,0.000,0.000,6.173,6.367,6.609,6.852,7.158,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.623,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.577/
      DATA ((EDEN(I,2,1,L,2),I=17,24),L=1,NZEN)/
     * 8.872,8.901,8.883,8.876,8.861,8.848,8.859,8.935,
     * 8.818,8.848,8.833,8.814,8.786,8.766,8.771,8.819,
     * 8.733,8.775,8.773,8.763,8.745,8.730,8.739,8.787,
     * 8.568,8.631,8.653,8.667,8.670,8.668,8.688,8.740,
     * 8.155,8.265,8.342,8.412,8.470,8.508,8.567,8.640,
     * 7.839,7.976,8.083,8.188,8.279,8.356,8.450,8.548,
     * 7.389,7.501,7.589,7.719,7.834,7.972,8.127,8.286,
     * 7.167,7.253,7.326,7.371,7.394,7.412,7.441,7.542,
     * 6.966,7.097,7.220,7.297,7.352,7.387,7.433,7.531,
     * 0.000,0.000,6.009,6.228,6.427,6.678,6.941,7.270,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.017,6.728,
     * 0.000,0.000,0.000,0.000,0.000,0.000,5.971,6.682/
      DATA ((EDEN(I,3,1,L,2),I=17,24),L=1,NZEN)/
     * 8.995,9.013,8.990,8.987,8.979,8.968,8.992,9.072,
     * 8.936,8.957,8.937,8.914,8.887,8.880,8.906,8.973,
     * 8.856,8.889,8.882,8.869,8.852,8.848,8.876,8.943,
     * 8.699,8.755,8.775,8.783,8.787,8.792,8.829,8.899,
     * 8.297,8.405,8.483,8.548,8.606,8.650,8.723,8.812,
     * 7.984,8.121,8.230,8.336,8.435,8.512,8.618,8.728,
     * 7.502,7.622,7.723,7.867,8.000,8.149,8.326,8.494,
     * 7.265,7.336,7.403,7.450,7.480,7.508,7.558,7.696,
     * 7.083,7.196,7.307,7.382,7.438,7.481,7.547,7.674,
     * 0.000,0.000,6.137,6.350,6.548,6.801,7.076,7.420,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.155,6.873,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.110,6.828/
      DATA ((EDEN(I,4,1,L,2),I=17,24),L=1,NZEN)/
     * 9.250,9.243,9.196,9.193,9.190,9.167,9.190,9.260,
     * 9.137,9.146,9.117,9.093,9.068,9.068,9.114,9.190,
     * 9.068,9.086,9.072,9.053,9.037,9.041,9.086,9.164,
     * 8.928,8.972,8.983,8.983,8.982,8.994,9.045,9.127,
     * 8.560,8.661,8.736,8.788,8.836,8.879,8.959,9.057,
     * 8.258,8.396,8.512,8.606,8.695,8.766,8.874,8.990,
     * 7.728,7.873,8.004,8.167,8.324,8.462,8.635,8.801,
     * 7.431,7.496,7.556,7.609,7.648,7.688,7.757,7.948,
     * 7.279,7.375,7.474,7.549,7.609,7.656,7.733,7.894,
     * 0.000,6.114,6.354,6.568,6.766,7.017,7.299,7.658,
     * 0.000,0.000,0.000,0.000,6.057,6.204,6.382,7.100,
     * 0.000,0.000,0.000,0.000,6.011,6.158,6.336,7.055/
      DATA ((EDEN(I,5,1,L,2),I=17,24),L=1,NZEN)/
     * 9.550,9.502,9.420,9.401,9.387,9.358,9.380,9.447,
     * 9.328,9.324,9.288,9.262,9.241,9.246,9.301,9.386,
     * 9.265,9.272,9.248,9.228,9.212,9.220,9.281,9.364,
     * 9.143,9.167,9.167,9.164,9.161,9.182,9.248,9.336,
     * 8.806,8.890,8.952,8.993,9.037,9.083,9.173,9.281,
     * 8.521,8.647,8.756,8.835,8.913,8.985,9.104,9.228,
     * 7.963,8.127,8.279,8.436,8.591,8.723,8.900,9.068,
     * 7.594,7.655,7.716,7.775,7.823,7.876,7.969,8.212,
     * 7.453,7.542,7.635,7.713,7.780,7.834,7.926,8.124,
     * 6.083,6.310,6.543,6.760,6.964,7.223,7.517,7.896,
     * 0.000,0.000,0.000,6.104,6.250,6.410,6.605,7.332,
     * 0.000,0.000,0.000,6.058,6.205,6.364,6.560,7.287/
      DATA ((EDEN(I,1,2,L,2),I=17,24),L=1,NZEN)/
     * 8.804,8.840,8.825,8.812,8.787,8.764,8.755,8.817,
     * 8.747,8.784,8.772,8.752,8.720,8.691,8.676,8.705,
     * 8.662,8.708,8.708,8.699,8.678,8.652,8.641,8.671,
     * 8.494,8.562,8.587,8.600,8.600,8.588,8.589,8.620,
     * 8.076,8.190,8.267,8.334,8.387,8.420,8.462,8.516,
     * 7.760,7.902,8.009,8.111,8.193,8.262,8.340,8.420,
     * 7.332,7.444,7.526,7.647,7.751,7.874,8.009,8.149,
     * 7.117,7.210,7.286,7.328,7.348,7.350,7.356,7.430,
     * 6.903,7.045,7.176,7.253,7.303,7.326,7.350,7.425,
     * 0.000,0.000,0.000,6.164,6.358,6.602,6.849,7.158,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.620,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.574/
      DATA ((EDEN(I,2,2,L,2),I=17,24),L=1,NZEN)/
     * 8.859,8.886,8.869,8.859,8.843,8.829,8.838,8.917,
     * 8.809,8.837,8.822,8.803,8.776,8.757,8.762,8.811,
     * 8.723,8.762,8.759,8.752,8.736,8.723,8.731,8.779,
     * 8.558,8.618,8.639,8.653,8.658,8.658,8.679,8.730,
     * 8.143,8.250,8.324,8.394,8.453,8.494,8.555,8.626,
     * 7.825,7.960,8.064,8.170,8.260,8.338,8.436,8.531,
     * 7.387,7.494,7.577,7.706,7.818,7.955,8.107,8.270,
     * 7.167,7.253,7.326,7.371,7.396,7.410,7.436,7.533,
     * 6.960,7.090,7.217,7.297,7.352,7.386,7.431,7.525,
     * 0.000,0.000,0.000,6.212,6.410,6.666,6.930,7.260,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.004,6.719,
     * 0.000,0.000,0.000,0.000,0.000,0.000,5.959,6.674/
      DATA ((EDEN(I,3,2,L,2),I=17,24),L=1,NZEN)/
     * 8.955,8.975,8.954,8.953,8.947,8.943,8.970,9.057,
     * 8.908,8.927,8.907,8.886,8.861,8.856,8.881,8.950,
     * 8.828,8.857,8.850,8.839,8.823,8.821,8.848,8.918,
     * 8.666,8.719,8.738,8.747,8.751,8.760,8.798,8.869,
     * 8.262,8.364,8.439,8.507,8.567,8.613,8.685,8.775,
     * 7.954,8.083,8.185,8.292,8.389,8.471,8.577,8.689,
     * 7.490,7.600,7.692,7.832,7.960,8.107,8.281,8.452,
     * 7.262,7.332,7.396,7.442,7.470,7.496,7.540,7.670,
     * 7.079,7.190,7.301,7.375,7.430,7.471,7.531,7.651,
     * 0.000,0.000,6.117,6.332,6.528,6.785,7.061,7.398,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.143,6.860,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.097,6.815/
      DATA ((EDEN(I,4,2,L,2),I=17,24),L=1,NZEN)/
     * 9.137,9.146,9.117,9.124,9.127,9.121,9.152,9.233,
     * 9.083,9.090,9.064,9.041,9.017,9.021,9.068,9.146,
     * 9.009,9.025,9.013,8.998,8.982,8.991,9.037,9.117,
     * 8.865,8.905,8.917,8.920,8.921,8.938,8.990,9.072,
     * 8.491,8.588,8.662,8.719,8.770,8.815,8.894,8.990,
     * 8.193,8.322,8.431,8.530,8.623,8.698,8.807,8.922,
     * 7.688,7.816,7.931,8.090,8.243,8.386,8.560,8.723,
     * 7.418,7.474,7.530,7.577,7.611,7.646,7.709,7.885,
     * 7.265,7.354,7.450,7.521,7.575,7.620,7.690,7.838,
     * 0.000,6.086,6.328,6.539,6.733,6.984,7.260,7.604,
     * 0.000,0.000,0.000,0.000,6.041,6.188,6.358,7.064,
     * 0.000,0.000,0.000,0.000,5.996,6.142,6.312,7.019/
      DATA ((EDEN(I,5,2,L,2),I=17,24),L=1,NZEN)/
     * 9.380,9.356,9.303,9.305,9.310,9.294,9.326,9.398,
     * 9.243,9.236,9.201,9.176,9.155,9.164,9.223,9.310,
     * 9.179,9.182,9.155,9.137,9.121,9.134,9.193,9.279,
     * 9.045,9.072,9.072,9.068,9.068,9.086,9.149,9.238,
     * 8.698,8.779,8.842,8.887,8.930,8.976,9.064,9.164,
     * 8.412,8.530,8.633,8.719,8.800,8.872,8.984,9.104,
     * 7.887,8.021,8.149,8.307,8.458,8.593,8.767,8.931,
     * 7.572,7.619,7.672,7.720,7.757,7.795,7.868,8.079,
     * 7.431,7.508,7.595,7.665,7.720,7.763,7.839,8.009,
     * 6.033,6.253,6.484,6.697,6.894,7.146,7.430,7.785,
     * 0.000,0.000,0.000,6.061,6.201,6.350,6.533,7.246,
     * 0.000,0.000,0.000,6.015,6.156,6.304,6.487,7.200/
      DATA ((EDEN(I,1,3,L,2),I=17,24),L=1,NZEN)/
     * 8.785,8.818,8.800,8.786,8.760,8.732,8.720,8.785,
     * 8.748,8.787,8.775,8.758,8.728,8.698,8.683,8.711,
     * 8.663,8.711,8.713,8.705,8.686,8.661,8.650,8.679,
     * 8.494,8.565,8.591,8.606,8.609,8.597,8.599,8.627,
     * 8.076,8.193,8.272,8.344,8.400,8.430,8.471,8.522,
     * 7.760,7.904,8.013,8.117,8.204,8.272,8.350,8.427,
     * 7.334,7.446,7.529,7.654,7.760,7.885,8.017,8.158,
     * 7.117,7.212,7.290,7.334,7.354,7.356,7.362,7.435,
     * 6.900,7.041,7.176,7.255,7.307,7.330,7.356,7.428,
     * 0.000,0.000,0.000,6.155,6.352,6.599,6.849,7.158,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.618,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.572/
      DATA ((EDEN(I,2,3,L,2),I=17,24),L=1,NZEN)/
     * 8.838,8.865,8.848,8.840,8.823,8.810,8.816,8.897,
     * 8.802,8.832,8.820,8.805,8.780,8.763,8.766,8.812,
     * 8.717,8.757,8.757,8.753,8.740,8.726,8.732,8.778,
     * 8.550,8.611,8.635,8.653,8.661,8.660,8.677,8.725,
     * 8.134,8.243,8.320,8.394,8.455,8.496,8.553,8.621,
     * 7.822,7.955,8.061,8.170,8.265,8.340,8.435,8.528,
     * 7.391,7.498,7.582,7.712,7.825,7.960,8.107,8.265,
     * 7.176,7.260,7.336,7.384,7.410,7.422,7.442,7.534,
     * 6.968,7.100,7.228,7.310,7.367,7.398,7.438,7.526,
     * 0.000,0.000,0.000,6.220,6.422,6.678,6.941,7.262,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.017,6.726,
     * 0.000,0.000,0.000,0.000,0.000,0.000,5.971,6.680/
      DATA ((EDEN(I,3,3,L,2),I=17,24),L=1,NZEN)/
     * 8.914,8.934,8.919,8.921,8.916,8.919,8.945,9.037,
     * 8.879,8.897,8.881,8.866,8.845,8.841,8.863,8.927,
     * 8.796,8.823,8.820,8.815,8.804,8.803,8.825,8.889,
     * 8.631,8.681,8.702,8.718,8.725,8.737,8.772,8.836,
     * 8.223,8.320,8.394,8.465,8.529,8.579,8.648,8.732,
     * 7.916,8.037,8.140,8.248,8.344,8.430,8.535,8.640,
     * 7.487,7.583,7.664,7.794,7.911,8.057,8.223,8.389,
     * 7.274,7.342,7.410,7.455,7.481,7.501,7.535,7.644,
     * 7.083,7.193,7.310,7.387,7.442,7.481,7.531,7.633,
     * 0.000,0.000,6.097,6.318,6.516,6.777,7.049,7.378,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.140,6.851,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.094,6.805/
      DATA ((EDEN(I,4,3,L,2),I=17,24),L=1,NZEN)/
     * 9.029,9.041,9.029,9.041,9.053,9.061,9.104,9.193,
     * 8.980,8.986,8.966,8.952,8.934,8.943,8.981,9.061,
     * 8.899,8.915,8.907,8.902,8.894,8.904,8.943,9.021,
     * 8.743,8.780,8.797,8.810,8.819,8.839,8.886,8.963,
     * 8.348,8.436,8.507,8.575,8.636,8.691,8.771,8.864,
     * 8.053,8.164,8.262,8.369,8.468,8.555,8.667,8.781,
     * 7.618,7.706,7.787,7.926,8.057,8.207,8.382,8.551,
     * 7.403,7.450,7.507,7.549,7.574,7.601,7.646,7.779,
     * 7.233,7.320,7.420,7.490,7.540,7.583,7.643,7.760,
     * 0.000,6.017,6.258,6.470,6.663,6.919,7.193,7.519,
     * 0.000,0.000,0.000,0.000,6.004,6.149,6.312,7.009,
     * 0.000,0.000,0.000,0.000,5.959,6.103,6.266,6.963/
      DATA ((EDEN(I,5,3,L,2),I=17,24),L=1,NZEN)/
     * 9.182,9.188,9.170,9.196,9.223,9.233,9.276,9.362,
     * 9.124,9.124,9.100,9.090,9.079,9.093,9.143,9.228,
     * 9.049,9.057,9.049,9.045,9.041,9.057,9.107,9.188,
     * 8.903,8.933,8.949,8.961,8.972,8.995,9.053,9.130,
     * 8.531,8.615,8.689,8.755,8.816,8.867,8.948,9.041,
     * 8.243,8.354,8.462,8.567,8.667,8.747,8.857,8.967,
     * 7.777,7.878,7.980,8.137,8.286,8.433,8.607,8.766,
     * 7.537,7.573,7.625,7.672,7.705,7.739,7.795,7.955,
     * 7.387,7.458,7.550,7.621,7.674,7.719,7.783,7.914,
     * 0.000,6.201,6.439,6.650,6.843,7.097,7.369,7.692,
     * 0.000,0.000,0.000,6.057,6.193,6.338,6.504,7.185,
     * 0.000,0.000,0.000,6.011,6.147,6.293,6.458,7.139/
      DATA ((EDEN(I,1,4,L,2),I=17,24),L=1,NZEN)/
     * 8.770,8.802,8.784,8.769,8.741,8.715,8.704,8.770,
     * 8.760,8.802,8.796,8.781,8.754,8.724,8.708,8.733,
     * 8.676,8.729,8.735,8.732,8.716,8.688,8.675,8.700,
     * 8.512,8.587,8.617,8.635,8.640,8.624,8.622,8.648,
     * 8.100,8.223,8.305,8.382,8.441,8.465,8.501,8.548,
     * 7.789,7.936,8.049,8.158,8.253,8.314,8.386,8.456,
     * 7.344,7.468,7.562,7.697,7.814,7.936,8.068,8.199,
     * 7.124,7.220,7.299,7.348,7.373,7.371,7.375,7.450,
     * 6.917,7.061,7.193,7.274,7.328,7.346,7.369,7.441,
     * 0.000,0.000,0.000,6.193,6.391,6.632,6.877,7.179,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.638,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.593/
      DATA ((EDEN(I,2,4,L,2),I=17,24),L=1,NZEN)/
     * 8.827,8.852,8.839,8.833,8.818,8.807,8.812,8.891,
     * 8.801,8.833,8.826,8.818,8.799,8.784,8.783,8.823,
     * 8.716,8.757,8.763,8.765,8.757,8.745,8.748,8.787,
     * 8.550,8.614,8.642,8.666,8.678,8.677,8.689,8.732,
     * 8.140,8.250,8.330,8.408,8.474,8.513,8.565,8.624,
     * 7.831,7.966,8.076,8.188,8.286,8.362,8.447,8.531,
     * 7.405,7.512,7.602,7.735,7.851,7.983,8.124,8.272,
     * 7.188,7.272,7.354,7.407,7.435,7.444,7.459,7.543,
     * 6.985,7.117,7.250,7.334,7.393,7.423,7.456,7.535,
     * 0.000,0.000,6.033,6.260,6.464,6.714,6.968,7.276,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.049,6.746,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.003,6.700/
      DATA ((EDEN(I,3,4,L,2),I=17,24),L=1,NZEN)/
     * 8.886,8.904,8.893,8.897,8.893,8.897,8.918,9.009,
     * 8.844,8.862,8.852,8.848,8.835,8.832,8.845,8.899,
     * 8.758,8.785,8.785,8.790,8.785,8.787,8.803,8.857,
     * 8.589,8.635,8.660,8.685,8.701,8.713,8.739,8.795,
     * 8.179,8.270,8.344,8.422,8.487,8.540,8.606,8.679,
     * 7.873,7.989,8.090,8.199,8.294,8.384,8.484,8.581,
     * 7.487,7.567,7.640,7.764,7.873,8.009,8.158,8.318,
     * 7.283,7.352,7.428,7.477,7.505,7.521,7.542,7.625,
     * 7.079,7.196,7.324,7.408,7.467,7.502,7.542,7.623,
     * 0.000,0.000,6.097,6.324,6.528,6.790,7.057,7.367,
     * 0.000,0.000,0.000,0.000,0.000,6.004,6.155,6.854,
     * 0.000,0.000,0.000,0.000,0.000,5.959,6.110,6.809/
      DATA ((EDEN(I,4,4,L,2),I=17,24),L=1,NZEN)/
     * 8.944,8.952,8.942,8.954,8.960,8.975,9.009,9.114,
     * 8.885,8.889,8.876,8.876,8.870,8.879,8.903,8.967,
     * 8.797,8.810,8.807,8.815,8.816,8.829,8.857,8.919,
     * 8.625,8.658,8.679,8.706,8.725,8.748,8.787,8.851,
     * 8.215,8.292,8.364,8.438,8.504,8.566,8.641,8.723,
     * 7.924,8.021,8.117,8.220,8.312,8.408,8.516,8.621,
     * 7.599,7.646,7.699,7.811,7.908,8.045,8.190,8.354,
     * 7.410,7.458,7.525,7.573,7.599,7.615,7.636,7.716,
     * 7.210,7.305,7.425,7.507,7.562,7.601,7.643,7.720,
     * 0.000,0.000,6.212,6.435,6.631,6.898,7.170,7.471,
     * 0.000,0.000,0.000,0.000,6.004,6.155,6.307,6.987,
     * 0.000,0.000,0.000,0.000,5.959,6.110,6.262,6.941/
      DATA ((EDEN(I,5,4,L,2),I=17,24),L=1,NZEN)/
     * 8.978,8.974,8.962,8.984,9.004,9.033,9.079,9.201,
     * 8.932,8.927,8.915,8.922,8.925,8.940,8.972,9.041,
     * 8.843,8.844,8.843,8.856,8.865,8.883,8.918,8.986,
     * 8.671,8.692,8.712,8.742,8.766,8.792,8.835,8.903,
     * 8.270,8.332,8.398,8.474,8.540,8.602,8.679,8.760,
     * 7.997,8.072,8.158,8.262,8.354,8.447,8.551,8.655,
     * 0.000,7.737,7.777,7.882,7.973,8.100,8.236,8.393,
     * 7.535,7.566,7.627,7.674,7.700,7.714,7.736,7.806,
     * 7.346,7.423,7.538,7.616,7.671,7.708,7.747,7.816,
     * 0.000,0.000,6.352,6.569,6.764,7.029,7.299,7.582,
     * 0.000,0.000,0.000,6.037,6.176,6.324,6.474,7.130,
     * 0.000,0.000,0.000,5.992,6.130,6.279,6.428,7.085/
      DATA ((EDEN(I,1,5,L,2),I=17,24),L=1,NZEN)/
     * 8.776,8.808,8.790,8.777,8.753,8.728,8.719,8.784,
     * 8.761,8.805,8.799,8.785,8.758,8.728,8.713,8.739,
     * 8.679,8.732,8.740,8.736,8.720,8.694,8.680,8.706,
     * 8.519,8.593,8.623,8.641,8.647,8.631,8.628,8.656,
     * 8.117,8.238,8.322,8.396,8.456,8.479,8.512,8.556,
     * 7.813,7.959,8.072,8.182,8.276,8.334,8.400,8.468,
     * 7.354,7.484,7.585,7.723,7.844,7.966,8.097,8.225,
     * 7.127,7.220,7.299,7.344,7.369,7.369,7.377,7.459,
     * 6.930,7.068,7.201,7.276,7.328,7.348,7.371,7.446,
     * 0.000,0.000,6.013,6.238,6.435,6.667,6.903,7.196,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.658,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.612/
      DATA ((EDEN(I,2,5,L,2),I=17,24),L=1,NZEN)/
     * 8.835,8.862,8.852,8.851,8.839,8.831,8.834,8.908,
     * 8.801,8.832,8.827,8.823,8.808,8.792,8.787,8.824,
     * 8.715,8.756,8.763,8.769,8.762,8.751,8.750,8.787,
     * 8.550,8.612,8.642,8.668,8.681,8.682,8.692,8.730,
     * 8.146,8.250,8.332,8.412,8.477,8.517,8.563,8.620,
     * 7.839,7.973,8.083,8.196,8.290,8.365,8.449,8.529,
     * 0.000,7.521,7.614,7.748,7.863,7.994,8.130,8.270,
     * 0.000,7.283,7.367,7.420,7.449,7.459,7.471,7.548,
     * 7.000,7.130,7.267,7.352,7.410,7.439,7.468,7.540,
     * 0.000,0.000,6.064,6.297,6.501,6.749,6.997,7.288,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.083,6.765,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.037,6.719/
      DATA ((EDEN(I,3,5,L,2),I=17,24),L=1,NZEN)/
     * 8.876,8.890,8.883,8.889,8.887,8.889,8.900,8.989,
     * 8.824,8.841,8.837,8.842,8.837,8.836,8.842,8.887,
     * 8.736,8.762,8.769,8.781,8.783,8.787,8.799,8.843,
     * 8.567,8.612,8.639,8.671,8.691,8.707,8.729,8.775,
     * 8.158,8.246,8.324,8.405,8.471,8.525,8.584,8.649,
     * 7.858,7.971,8.079,8.188,8.279,8.367,8.458,8.548,
     * 0.000,7.571,7.645,7.766,7.870,7.999,8.130,8.281,
     * 0.000,7.369,7.452,7.507,7.538,7.551,7.563,7.627,
     * 7.093,7.212,7.350,7.439,7.502,7.537,7.567,7.630,
     * 0.000,0.000,6.134,6.369,6.577,6.837,7.090,7.380,
     * 0.000,0.000,0.000,0.000,0.000,6.064,6.207,6.881,
     * 0.000,0.000,0.000,0.000,0.000,6.019,6.161,6.836/
      DATA ((EDEN(I,4,5,L,2),I=17,24),L=1,NZEN)/
     * 8.900,8.900,8.892,8.902,8.904,8.913,8.928,9.029,
     * 8.843,8.845,8.839,8.849,8.849,8.860,8.876,8.924,
     * 8.752,8.762,8.766,8.781,8.787,8.803,8.823,8.873,
     * 8.575,8.603,8.629,8.661,8.681,8.708,8.738,8.790,
     * 8.164,8.233,8.310,8.384,8.447,8.508,8.571,8.643,
     * 7.885,7.974,8.068,8.170,8.258,8.344,8.433,8.530,
     * 7.624,7.657,7.704,7.802,7.883,7.996,8.107,8.253,
     * 0.000,7.490,7.572,7.624,7.653,7.664,7.668,7.713,
     * 7.220,7.326,7.465,7.558,7.621,7.654,7.680,7.726,
     * 0.000,0.000,6.246,6.480,6.686,6.949,7.207,7.480,
     * 0.000,0.000,0.000,0.000,6.086,6.230,6.365,7.017,
     * 0.000,0.000,0.000,0.000,6.041,6.185,6.320,6.971/
      DATA ((EDEN(I,5,5,L,2),I=17,24),L=1,NZEN)/
     * 8.871,8.864,8.862,8.884,8.898,8.922,8.948,9.057,
     * 8.817,8.814,8.818,8.841,8.856,8.877,8.901,8.950,
     * 8.715,8.720,8.732,8.760,8.780,8.807,8.835,8.889,
     * 8.524,8.548,8.583,8.625,8.657,8.693,8.728,8.788,
     * 8.114,8.170,8.246,8.328,8.400,8.465,8.529,8.610,
     * 0.000,7.950,8.021,8.124,8.210,8.297,8.380,8.481,
     * 0.000,0.000,7.779,7.844,7.891,7.985,8.076,8.204,
     * 7.573,7.613,7.698,7.753,7.782,7.788,7.785,7.807,
     * 7.336,7.433,7.581,7.679,7.749,7.781,7.802,7.830,
     * 0.000,0.000,6.338,6.573,6.777,7.053,7.322,7.583,
     * 0.000,0.000,0.000,6.090,6.241,6.387,6.525,7.155,
     * 0.000,0.000,0.000,6.044,6.195,6.342,6.479,7.110/
      DATA ((EDEN(I,1,6,L,2),I=17,24),L=1,NZEN)/
     * 8.795,8.829,8.814,8.803,8.780,8.757,8.749,8.812,
     * 8.766,8.807,8.799,8.783,8.754,8.725,8.710,8.740,
     * 8.685,8.736,8.740,8.732,8.712,8.688,8.678,8.705,
     * 8.526,8.599,8.626,8.641,8.643,8.627,8.625,8.655,
     * 8.137,8.253,8.332,8.403,8.456,8.479,8.512,8.560,
     * 7.833,7.977,8.086,8.193,8.281,8.338,8.403,8.474,
     * 7.364,7.497,7.602,7.739,7.858,7.979,8.107,8.238,
     * 0.000,7.220,7.294,7.340,7.362,7.367,7.377,7.464,
     * 6.944,7.079,7.204,7.276,7.324,7.344,7.369,7.446,
     * 0.000,0.000,6.049,6.270,6.464,6.691,6.922,7.204,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.009,6.672,
     * 0.000,0.000,0.000,0.000,0.000,0.000,5.963,6.626/
      DATA ((EDEN(I,2,6,L,2),I=17,24),L=1,NZEN)/
     * 8.856,8.886,8.881,8.883,8.874,8.866,8.867,8.936,
     * 8.820,8.850,8.848,8.845,8.831,8.817,8.811,8.845,
     * 8.736,8.775,8.784,8.790,8.785,8.775,8.773,8.806,
     * 8.575,8.634,8.666,8.692,8.705,8.706,8.713,8.747,
     * 8.179,8.283,8.365,8.444,8.508,8.547,8.590,8.640,
     * 7.878,8.009,8.121,8.230,8.326,8.400,8.477,8.550,
     * 7.438,7.551,7.653,7.790,7.908,8.037,8.173,8.305,
     * 0.000,7.301,7.386,7.438,7.468,7.480,7.490,7.568,
     * 7.029,7.158,7.294,7.377,7.435,7.462,7.489,7.556,
     * 0.000,0.000,6.127,6.358,6.562,6.803,7.041,7.318,
     * 0.000,0.000,0.000,0.000,0.000,6.004,6.140,6.798,
     * 0.000,0.000,0.000,0.000,0.000,5.959,6.094,6.752/
      DATA ((EDEN(I,3,6,L,2),I=17,24),L=1,NZEN)/
     * 8.884,8.899,8.895,8.904,8.901,8.902,8.906,8.988,
     * 8.846,8.865,8.866,8.876,8.875,8.874,8.877,8.914,
     * 8.760,8.786,8.797,8.814,8.820,8.825,8.833,8.869,
     * 8.592,8.637,8.670,8.705,8.728,8.743,8.759,8.797,
     * 8.190,8.279,8.362,8.446,8.512,8.565,8.616,8.671,
     * 7.897,8.009,8.121,8.230,8.324,8.410,8.493,8.572,
     * 7.525,7.605,7.690,7.817,7.924,8.049,8.173,8.312,
     * 7.328,7.396,7.484,7.544,7.577,7.591,7.597,7.653,
     * 7.134,7.250,7.391,7.483,7.545,7.579,7.601,7.653,
     * 0.000,0.000,6.207,6.446,6.656,6.909,7.152,7.417,
     * 0.000,0.000,0.000,0.000,0.000,6.140,6.272,6.922,
     * 0.000,0.000,0.000,0.000,0.000,6.094,6.226,6.876/
      DATA ((EDEN(I,4,6,L,2),I=17,24),L=1,NZEN)/
     * 8.894,8.893,8.889,8.902,8.904,8.912,8.916,9.004,
     * 8.844,8.846,8.847,8.862,8.866,8.879,8.889,8.925,
     * 8.751,8.760,8.771,8.791,8.800,8.818,8.832,8.871,
     * 8.575,8.602,8.634,8.669,8.691,8.719,8.741,8.783,
     * 8.161,8.233,8.314,8.391,8.453,8.512,8.563,8.628,
     * 0.000,7.979,8.079,8.182,8.267,8.350,8.423,8.512,
     * 0.000,7.686,7.733,7.831,7.908,8.013,8.107,8.236,
     * 7.471,7.525,7.617,7.676,7.707,7.713,7.708,7.732,
     * 7.246,7.356,7.508,7.607,7.676,7.707,7.722,7.750,
     * 0.000,0.000,6.292,6.535,6.748,7.009,7.258,7.508,
     * 0.000,0.000,0.000,6.000,6.161,6.301,6.425,7.053,
     * 0.000,0.000,0.000,5.954,6.116,6.255,6.379,7.007/
      DATA ((EDEN(I,5,6,L,2),I=17,24),L=1,NZEN)/
     * 8.854,8.851,8.863,8.889,8.905,8.921,8.927,9.013,
     * 8.796,8.794,8.808,8.838,8.856,8.876,8.887,8.922,
     * 8.688,8.696,8.722,8.756,8.778,8.801,8.815,8.856,
     * 8.484,8.513,8.563,8.613,8.649,8.679,8.698,8.747,
     * 8.083,8.134,8.212,8.303,8.377,8.436,8.483,8.550,
     * 0.000,7.945,8.009,8.104,8.185,8.265,8.332,8.415,
     * 0.000,7.810,7.836,7.885,7.912,7.983,8.037,8.140,
     * 7.609,7.659,7.763,7.827,7.862,7.862,7.843,7.839,
     * 7.334,7.447,7.623,7.737,7.819,7.851,7.861,7.866,
     * 0.000,0.000,6.352,6.597,6.809,7.093,7.354,7.606,
     * 0.000,0.000,0.000,6.143,6.305,6.450,6.575,7.190,
     * 0.000,0.000,0.000,6.097,6.260,6.404,6.529,7.145/
      DATA ((EDEN(I,1,7,L,2),I=17,24),L=1,NZEN)/
     * 8.810,8.845,8.830,8.818,8.794,8.770,8.762,8.823,
     * 8.763,8.801,8.789,8.770,8.740,8.710,8.695,8.728,
     * 8.680,8.728,8.728,8.719,8.695,8.673,8.665,8.695,
     * 8.522,8.590,8.615,8.626,8.624,8.612,8.612,8.643,
     * 8.127,8.241,8.316,8.384,8.435,8.459,8.496,8.545,
     * 7.822,7.964,8.072,8.173,8.258,8.318,8.387,8.461,
     * 7.358,7.487,7.588,7.719,7.833,7.956,8.086,8.220,
     * 0.000,7.217,7.292,7.334,7.352,7.358,7.369,7.455,
     * 6.942,7.076,7.199,7.270,7.316,7.338,7.362,7.438,
     * 0.000,0.000,6.037,6.255,6.449,6.678,6.909,7.196,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.000,6.664,
     * 0.000,0.000,0.000,0.000,0.000,0.000,5.954,6.618/
      DATA ((EDEN(I,2,7,L,2),I=17,24),L=1,NZEN)/
     * 8.876,8.906,8.903,8.905,8.897,8.889,8.889,8.954,
     * 8.841,8.872,8.872,8.867,8.851,8.838,8.833,8.865,
     * 8.757,8.798,8.808,8.814,8.808,8.797,8.795,8.826,
     * 8.597,8.658,8.692,8.718,8.730,8.729,8.734,8.768,
     * 8.204,8.310,8.394,8.473,8.535,8.573,8.615,8.662,
     * 7.903,8.037,8.152,8.262,8.358,8.431,8.507,8.575,
     * 7.453,7.572,7.678,7.819,7.941,8.072,8.204,8.334,
     * 7.230,7.312,7.398,7.452,7.483,7.496,7.507,7.585,
     * 7.045,7.173,7.307,7.391,7.447,7.477,7.504,7.572,
     * 0.000,0.000,6.146,6.378,6.583,6.822,7.057,7.332,
     * 0.000,0.000,0.000,0.000,0.000,6.021,6.155,6.812,
     * 0.000,0.000,0.000,0.000,0.000,5.975,6.110,6.766/
      DATA ((EDEN(I,3,7,L,2),I=17,24),L=1,NZEN)/
     * 8.905,8.921,8.920,8.931,8.931,8.932,8.933,9.013,
     * 8.892,8.912,8.916,8.927,8.927,8.926,8.926,8.960,
     * 8.806,8.834,8.849,8.867,8.874,8.877,8.880,8.912,
     * 8.643,8.690,8.725,8.763,8.787,8.800,8.812,8.845,
     * 8.248,8.338,8.425,8.509,8.579,8.630,8.679,8.726,
     * 7.953,8.068,8.185,8.301,8.400,8.483,8.561,8.631,
     * 0.000,7.644,7.742,7.879,7.997,8.127,8.255,8.382,
     * 0.000,7.418,7.508,7.569,7.605,7.621,7.626,7.688,
     * 7.167,7.279,7.420,7.512,7.575,7.609,7.630,7.683,
     * 0.000,0.000,6.258,6.498,6.708,6.957,7.193,7.450,
     * 0.000,0.000,0.000,0.000,6.041,6.185,6.316,6.954,
     * 0.000,0.000,0.000,0.000,5.996,6.139,6.270,6.908/
      DATA ((EDEN(I,4,7,L,2),I=17,24),L=1,NZEN)/
     * 8.920,8.920,8.920,8.937,8.943,8.953,8.957,9.037,
     * 8.888,8.893,8.898,8.919,8.926,8.939,8.945,8.976,
     * 8.797,8.809,8.824,8.849,8.863,8.880,8.891,8.924,
     * 8.624,8.655,8.691,8.731,8.757,8.785,8.805,8.840,
     * 8.217,8.292,8.378,8.461,8.528,8.589,8.638,8.694,
     * 0.000,8.033,8.143,8.253,8.344,8.430,8.504,8.583,
     * 0.000,7.715,7.776,7.887,7.976,8.090,8.190,8.316,
     * 0.000,7.547,7.641,7.703,7.737,7.748,7.743,7.767,
     * 7.279,7.387,7.540,7.639,7.708,7.739,7.753,7.779,
     * 0.000,6.061,6.346,6.590,6.803,7.061,7.303,7.545,
     * 0.000,0.000,0.000,6.045,6.207,6.348,6.470,7.086,
     * 0.000,0.000,0.000,6.000,6.161,6.303,6.424,7.041/
      DATA ((EDEN(I,5,7,L,2),I=17,24),L=1,NZEN)/
     * 8.898,8.894,8.907,8.938,8.956,8.974,8.978,9.061,
     * 8.838,8.837,8.854,8.886,8.905,8.928,8.939,8.969,
     * 8.733,8.742,8.771,8.808,8.832,8.858,8.870,8.906,
     * 8.534,8.563,8.617,8.670,8.709,8.742,8.760,8.803,
     * 0.000,8.188,8.276,8.373,8.452,8.513,8.559,8.620,
     * 0.000,7.987,8.064,8.173,8.260,8.344,8.412,8.490,
     * 0.000,7.830,7.863,7.926,7.966,8.049,8.114,8.215,
     * 7.628,7.679,7.787,7.855,7.893,7.896,7.876,7.869,
     * 7.358,7.473,7.653,7.770,7.852,7.885,7.892,7.893,
     * 0.000,0.000,6.394,6.646,6.864,7.143,7.396,7.639,
     * 0.000,0.000,0.000,6.193,6.356,6.500,6.618,6.954,
     * 0.000,0.000,0.000,6.147,6.310,6.454,6.572,6.908/
      DATA ((EDEN(I,1,8,L,2),I=17,24),L=1,NZEN)/
     * 8.809,8.843,8.826,8.811,8.784,8.759,8.749,8.811,
     * 8.754,8.790,8.777,8.757,8.724,8.696,8.683,8.714,
     * 8.669,8.716,8.715,8.706,8.683,8.660,8.650,8.680,
     * 8.508,8.575,8.598,8.610,8.609,8.595,8.595,8.629,
     * 8.104,8.217,8.290,8.358,8.410,8.438,8.477,8.529,
     * 7.794,7.934,8.037,8.140,8.225,8.290,8.364,8.439,
     * 0.000,7.465,7.558,7.685,7.794,7.918,8.049,8.188,
     * 7.121,7.212,7.288,7.328,7.346,7.350,7.360,7.441,
     * 6.925,7.061,7.188,7.260,7.307,7.328,7.354,7.430,
     * 0.000,0.000,0.000,6.215,6.407,6.642,6.880,7.176,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.643,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.598/
      DATA ((EDEN(I,2,8,L,2),I=17,24),L=1,NZEN)/
     * 8.884,8.913,8.908,8.908,8.897,8.886,8.886,8.951,
     * 8.840,8.871,8.867,8.862,8.844,8.829,8.823,8.856,
     * 8.756,8.796,8.804,8.808,8.800,8.790,8.788,8.820,
     * 8.592,8.652,8.684,8.709,8.722,8.723,8.732,8.764,
     * 8.188,8.294,8.380,8.459,8.524,8.563,8.609,8.658,
     * 7.882,8.017,8.130,8.243,8.338,8.413,8.493,8.567,
     * 0.000,7.555,7.654,7.793,7.913,8.045,8.182,8.318,
     * 0.000,7.310,7.394,7.447,7.477,7.490,7.501,7.579,
     * 7.029,7.158,7.297,7.382,7.439,7.470,7.498,7.568,
     * 0.000,0.000,6.100,6.332,6.537,6.783,7.029,7.316,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.117,6.791,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.072,6.745/
      DATA ((EDEN(I,3,8,L,2),I=17,24),L=1,NZEN)/
     * 8.934,8.953,8.953,8.967,8.970,8.971,8.977,9.049,
     * 8.916,8.939,8.943,8.951,8.948,8.945,8.946,8.979,
     * 8.831,8.862,8.878,8.895,8.901,8.901,8.905,8.938,
     * 8.667,8.719,8.756,8.793,8.818,8.829,8.843,8.875,
     * 8.267,8.362,8.452,8.539,8.613,8.663,8.713,8.761,
     * 7.967,8.090,8.210,8.328,8.431,8.516,8.598,8.668,
     * 0.000,7.651,7.751,7.894,8.017,8.155,8.288,8.420,
     * 0.000,7.422,7.511,7.573,7.609,7.625,7.635,7.704,
     * 7.161,7.276,7.417,7.511,7.575,7.610,7.636,7.695,
     * 0.000,0.000,6.233,6.471,6.681,6.936,7.182,7.455,
     * 0.000,0.000,0.000,0.000,6.004,6.152,6.290,6.946,
     * 0.000,0.000,0.000,0.000,5.959,6.107,6.244,6.900/
      DATA ((EDEN(I,4,8,L,2),I=17,24),L=1,NZEN)/
     * 8.979,8.983,8.984,9.004,9.017,9.029,9.041,9.121,
     * 8.946,8.956,8.964,8.984,8.993,9.004,9.009,9.041,
     * 8.859,8.877,8.895,8.922,8.938,8.951,8.962,8.993,
     * 8.691,8.728,8.769,8.812,8.843,8.868,8.890,8.922,
     * 8.290,8.369,8.461,8.550,8.625,8.688,8.745,8.795,
     * 8.004,8.104,8.220,8.338,8.441,8.533,8.619,8.694,
     * 0.000,7.736,7.813,7.942,8.053,8.182,8.303,8.435,
     * 0.000,7.547,7.638,7.702,7.739,7.755,7.759,7.804,
     * 7.292,7.396,7.542,7.640,7.708,7.744,7.766,7.807,
     * 0.000,0.000,6.360,6.602,6.813,7.072,7.318,7.571,
     * 0.000,0.000,0.000,6.037,6.196,6.340,6.468,7.093,
     * 0.000,0.000,0.000,5.992,6.150,6.295,6.423,7.048/
      DATA ((EDEN(I,5,8,L,2),I=17,24),L=1,NZEN)/
     * 9.009,9.000,9.004,9.033,9.053,9.076,9.093,9.185,
     * 8.947,8.944,8.955,8.982,8.997,9.021,9.033,9.064,
     * 8.855,8.860,8.880,8.913,8.933,8.960,8.977,9.013,
     * 8.677,8.702,8.745,8.792,8.826,8.861,8.886,8.926,
     * 0.000,8.336,8.428,8.517,8.590,8.657,8.713,8.772,
     * 0.000,8.090,8.196,8.310,8.408,8.497,8.574,8.655,
     * 0.000,7.829,7.877,7.977,8.057,8.167,8.265,8.384,
     * 0.000,7.673,7.771,7.838,7.876,7.887,7.880,7.894,
     * 7.396,7.500,7.662,7.769,7.843,7.879,7.894,7.912,
     * 0.000,0.000,6.447,6.695,6.910,7.179,7.428,7.671,
     * 0.000,0.000,6.009,6.201,6.360,6.502,6.624,7.230,
     * 0.000,0.000,5.963,6.156,6.314,6.457,6.579,7.185/
      DATA ((EDEN(I,1,9,L,2),I=17,24),L=1,NZEN)/
     * 8.794,8.827,8.809,8.792,8.763,8.736,8.723,8.787,
     * 8.754,8.792,8.782,8.763,8.732,8.702,8.686,8.714,
     * 8.671,8.719,8.720,8.712,8.691,8.665,8.653,8.681,
     * 8.505,8.575,8.601,8.615,8.616,8.602,8.602,8.632,
     * 8.097,8.215,8.290,8.360,8.413,8.441,8.480,8.530,
     * 7.786,7.928,8.037,8.140,8.225,8.290,8.364,8.438,
     * 7.342,7.461,7.550,7.679,7.790,7.912,8.041,8.179,
     * 7.121,7.212,7.290,7.334,7.354,7.356,7.362,7.441,
     * 6.920,7.057,7.188,7.262,7.312,7.332,7.356,7.431,
     * 0.000,0.000,0.000,6.201,6.396,6.633,6.873,7.173,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.637,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.592/
      DATA ((EDEN(I,2,9,L,2),I=17,24),L=1,NZEN)/
     * 8.879,8.906,8.897,8.894,8.880,8.867,8.867,8.934,
     * 8.837,8.867,8.862,8.853,8.833,8.818,8.814,8.846,
     * 8.751,8.792,8.800,8.801,8.792,8.780,8.780,8.812,
     * 8.587,8.649,8.680,8.703,8.715,8.715,8.726,8.760,
     * 8.176,8.286,8.369,8.449,8.513,8.553,8.601,8.657,
     * 7.868,8.004,8.117,8.228,8.324,8.401,8.486,8.563,
     * 0.000,7.540,7.635,7.772,7.890,8.025,8.164,8.305,
     * 0.000,7.297,7.380,7.433,7.464,7.476,7.491,7.572,
     * 7.013,7.143,7.279,7.364,7.423,7.455,7.487,7.562,
     * 0.000,0.000,6.072,6.301,6.504,6.754,7.004,7.305,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.086,6.773,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.041,6.727/
      DATA ((EDEN(I,3,9,L,2),I=17,24),L=1,NZEN)/
     * 8.968,8.991,8.989,9.000,9.004,9.004,9.017,9.086,
     * 8.927,8.951,8.952,8.954,8.946,8.942,8.946,8.985,
     * 8.842,8.877,8.890,8.902,8.903,8.903,8.911,8.948,
     * 8.679,8.735,8.772,8.804,8.825,8.835,8.853,8.893,
     * 8.274,8.375,8.465,8.553,8.627,8.677,8.732,8.789,
     * 7.969,8.097,8.215,8.336,8.444,8.529,8.619,8.698,
     * 0.000,7.644,7.744,7.888,8.017,8.158,8.307,8.447,
     * 0.000,7.405,7.491,7.551,7.588,7.606,7.626,7.709,
     * 7.140,7.258,7.394,7.484,7.549,7.588,7.623,7.699,
     * 0.000,0.000,6.201,6.435,6.642,6.899,7.152,7.447,
     * 0.000,0.000,0.000,0.000,0.000,6.104,6.250,6.926,
     * 0.000,0.000,0.000,0.000,0.000,6.058,6.205,6.881/
      DATA ((EDEN(I,4,9,L,2),I=17,24),L=1,NZEN)/
     * 9.068,9.083,9.086,9.114,9.130,9.143,9.164,9.238,
     * 9.025,9.045,9.053,9.064,9.068,9.072,9.083,9.124,
     * 8.943,8.971,8.991,9.013,9.021,9.029,9.045,9.083,
     * 8.782,8.831,8.874,8.916,8.947,8.964,8.987,9.025,
     * 8.387,8.481,8.579,8.673,8.755,8.810,8.869,8.923,
     * 8.090,8.210,8.334,8.464,8.580,8.669,8.761,8.837,
     * 0.000,7.772,7.873,8.029,8.167,8.314,8.464,8.599,
     * 0.000,7.535,7.619,7.683,7.723,7.751,7.775,7.859,
     * 7.299,7.398,7.531,7.624,7.691,7.734,7.772,7.844,
     * 0.000,0.000,6.380,6.615,6.822,7.079,7.328,7.609,
     * 0.000,0.000,0.000,6.021,6.176,6.318,6.458,7.104,
     * 0.000,0.000,0.000,5.975,6.130,6.272,6.412,7.058/
      DATA ((EDEN(I,5,9,L,2),I=17,24),L=1,NZEN)/
     * 9.193,9.201,9.212,9.253,9.286,9.305,9.330,9.398,
     * 9.140,9.152,9.167,9.185,9.196,9.207,9.223,9.258,
     * 9.057,9.079,9.107,9.137,9.155,9.170,9.188,9.220,
     * 8.899,8.942,8.994,9.041,9.083,9.107,9.134,9.164,
     * 8.511,8.600,8.705,8.808,8.899,8.957,9.017,9.064,
     * 8.217,8.330,8.464,8.603,8.727,8.820,8.913,8.980,
     * 0.000,7.899,8.009,8.173,8.320,8.474,8.626,8.751,
     * 0.000,7.663,7.751,7.822,7.869,7.903,7.931,8.013,
     * 7.444,7.531,7.667,7.767,7.839,7.888,7.929,7.996,
     * 0.000,6.274,6.543,6.782,6.992,7.250,7.502,7.768,
     * 0.000,0.000,6.041,6.223,6.377,6.521,6.660,7.281,
     * 0.000,0.000,5.996,6.177,6.331,6.475,6.614,7.235/
      DATA ((EDEN(I,1,10,L,2),I=17,24),L=1,NZEN)/
     * 8.785,8.817,8.799,8.783,8.756,8.730,8.719,8.783,
     * 8.770,8.813,8.808,8.792,8.765,8.732,8.713,8.736,
     * 8.686,8.740,8.746,8.741,8.724,8.695,8.680,8.702,
     * 8.522,8.598,8.628,8.645,8.650,8.632,8.628,8.652,
     * 8.114,8.236,8.318,8.394,8.453,8.474,8.507,8.551,
     * 7.801,7.949,8.061,8.173,8.265,8.322,8.389,8.458,
     * 0.000,7.476,7.572,7.706,7.822,7.942,8.068,8.199,
     * 7.127,7.223,7.303,7.352,7.377,7.375,7.377,7.453,
     * 6.928,7.068,7.201,7.281,7.334,7.350,7.371,7.442,
     * 0.000,0.000,0.000,6.212,6.410,6.645,6.884,7.182,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.643,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.598/
      DATA ((EDEN(I,2,10,L,2),I=17,24),L=1,NZEN)/
     * 8.872,8.900,8.887,8.882,8.867,8.852,8.855,8.926,
     * 8.844,8.878,8.873,8.862,8.841,8.822,8.818,8.852,
     * 8.760,8.806,8.813,8.814,8.803,8.787,8.786,8.820,
     * 8.595,8.663,8.695,8.718,8.728,8.723,8.733,8.770,
     * 8.185,8.299,8.386,8.464,8.530,8.565,8.613,8.670,
     * 7.873,8.013,8.127,8.243,8.342,8.413,8.497,8.577,
     * 0.000,7.540,7.637,7.776,7.897,8.033,8.176,8.318,
     * 0.000,7.290,7.373,7.425,7.456,7.468,7.486,7.573,
     * 7.004,7.137,7.270,7.354,7.413,7.444,7.480,7.563,
     * 0.000,0.000,6.061,6.288,6.493,6.740,6.992,7.303,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.068,6.764,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.022,6.718/
      DATA ((EDEN(I,3,10,L,2),I=17,24),L=1,NZEN)/
     * 8.999,9.021,9.017,9.025,9.025,9.025,9.041,9.107,
     * 8.943,8.968,8.963,8.956,8.942,8.934,8.945,8.991,
     * 8.859,8.895,8.904,8.907,8.903,8.899,8.912,8.958,
     * 8.696,8.756,8.790,8.816,8.831,8.839,8.863,8.910,
     * 8.290,8.398,8.487,8.571,8.641,8.688,8.747,8.814,
     * 7.982,8.114,8.233,8.352,8.458,8.540,8.636,8.726,
     * 0.000,7.641,7.743,7.889,8.021,8.167,8.326,8.477,
     * 7.310,7.384,7.465,7.521,7.558,7.581,7.613,7.715,
     * 7.121,7.238,7.369,7.455,7.517,7.559,7.606,7.701,
     * 0.000,0.000,6.185,6.412,6.616,6.870,7.130,7.446,
     * 0.000,0.000,0.000,0.000,0.000,6.068,6.220,6.913,
     * 0.000,0.000,0.000,0.000,0.000,6.022,6.174,6.867/
      DATA ((EDEN(I,4,10,L,2),I=17,24),L=1,NZEN)/
     * 9.188,9.204,9.201,9.225,9.246,9.246,9.265,9.318,
     * 9.114,9.134,9.130,9.127,9.117,9.124,9.146,9.193,
     * 9.037,9.068,9.079,9.086,9.086,9.090,9.114,9.161,
     * 8.885,8.939,8.978,9.004,9.021,9.037,9.068,9.117,
     * 8.497,8.605,8.704,8.787,8.859,8.904,8.965,9.029,
     * 8.196,8.332,8.464,8.588,8.702,8.779,8.871,8.956,
     * 0.000,7.836,7.962,8.137,8.297,8.444,8.604,8.743,
     * 7.468,7.531,7.611,7.675,7.719,7.754,7.798,7.927,
     * 7.305,7.405,7.526,7.616,7.683,7.731,7.785,7.893,
     * 0.000,0.000,6.403,6.630,6.836,7.086,7.344,7.655,
     * 0.000,0.000,0.000,6.004,6.155,6.301,6.452,7.124,
     * 0.000,0.000,0.000,5.959,6.110,6.255,6.406,7.078/
      DATA ((EDEN(I,5,10,L,2),I=17,24),L=1,NZEN)/
     * 9.428,9.423,9.408,9.441,9.470,9.462,9.477,9.515,
     * 9.320,9.322,9.314,9.310,9.301,9.310,9.338,9.384,
     * 9.253,9.267,9.272,9.272,9.270,9.279,9.307,9.352,
     * 9.117,9.152,9.179,9.201,9.215,9.230,9.265,9.310,
     * 8.754,8.848,8.942,9.013,9.072,9.114,9.173,9.236,
     * 8.459,8.588,8.723,8.834,8.936,9.004,9.093,9.170,
     * 0.000,8.072,8.225,8.407,8.575,8.711,8.860,8.985,
     * 0.000,7.691,7.769,7.839,7.890,7.934,7.989,8.143,
     * 7.497,7.579,7.693,7.785,7.854,7.905,7.964,8.083,
     * 6.140,6.362,6.619,6.846,7.049,7.299,7.555,7.859,
     * 0.000,0.000,0.000,6.083,6.212,6.332,6.456,7.093,
     * 0.000,0.000,0.000,6.037,6.166,6.287,6.411,7.048/
      DATA ((EDEN(I,1,11,L,2),I=17,24),L=1,NZEN)/
     * 8.790,8.825,8.809,8.797,8.774,8.751,8.743,8.806,
     * 8.776,8.820,8.814,8.799,8.772,8.738,8.718,8.738,
     * 8.692,8.747,8.754,8.749,8.732,8.702,8.686,8.706,
     * 8.528,8.604,8.636,8.654,8.659,8.640,8.634,8.657,
     * 8.117,8.241,8.326,8.401,8.462,8.481,8.512,8.555,
     * 7.807,7.955,8.068,8.179,8.276,8.330,8.396,8.461,
     * 0.000,7.480,7.577,7.714,7.833,7.949,8.076,8.201,
     * 7.130,7.228,7.307,7.356,7.380,7.378,7.382,7.456,
     * 6.933,7.072,7.204,7.286,7.340,7.356,7.375,7.446,
     * 0.000,0.000,0.000,6.223,6.420,6.654,6.890,7.188,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.647,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.602/
      DATA ((EDEN(I,2,11,L,2),I=17,24),L=1,NZEN)/
     * 8.870,8.898,8.882,8.877,8.863,8.848,8.854,8.928,
     * 8.847,8.882,8.875,8.862,8.839,8.816,8.814,8.852,
     * 8.763,8.810,8.816,8.813,8.801,8.783,8.785,8.822,
     * 8.599,8.668,8.700,8.719,8.728,8.722,8.735,8.775,
     * 8.190,8.307,8.393,8.470,8.535,8.567,8.615,8.676,
     * 7.878,8.021,8.134,8.248,8.348,8.417,8.500,8.585,
     * 0.000,7.538,7.636,7.778,7.904,8.037,8.185,8.328,
     * 0.000,7.279,7.358,7.410,7.442,7.455,7.477,7.573,
     * 6.996,7.127,7.258,7.340,7.400,7.430,7.470,7.561,
     * 0.000,0.000,6.057,6.286,6.487,6.732,6.985,7.301,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.057,6.758,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.011,6.712/
      DATA ((EDEN(I,3,11,L,2),I=17,24),L=1,NZEN)/
     * 9.017,9.037,9.021,9.025,9.025,9.017,9.033,9.104,
     * 8.963,8.987,8.975,8.960,8.939,8.929,8.946,9.000,
     * 8.883,8.919,8.921,8.915,8.903,8.898,8.918,8.971,
     * 8.726,8.785,8.812,8.828,8.836,8.842,8.872,8.928,
     * 8.324,8.433,8.519,8.593,8.658,8.699,8.763,8.838,
     * 8.013,8.149,8.267,8.380,8.484,8.560,8.657,8.754,
     * 0.000,7.653,7.760,7.912,8.053,8.196,8.360,8.516,
     * 0.000,7.369,7.444,7.497,7.533,7.560,7.601,7.725,
     * 7.114,7.230,7.350,7.431,7.491,7.534,7.590,7.704,
     * 0.000,0.000,6.182,6.407,6.609,6.857,7.121,7.452,
     * 0.000,0.000,0.000,0.000,0.000,6.045,6.204,6.908,
     * 0.000,0.000,0.000,0.000,0.000,6.000,6.158,6.863/
      DATA ((EDEN(I,4,11,L,2),I=17,24),L=1,NZEN)/
     * 9.290,9.290,9.262,9.272,9.279,9.260,9.274,9.324,
     * 9.185,9.196,9.176,9.161,9.143,9.143,9.176,9.230,
     * 9.114,9.137,9.130,9.124,9.111,9.114,9.146,9.204,
     * 8.975,9.021,9.041,9.049,9.057,9.068,9.107,9.167,
     * 8.603,8.708,8.794,8.854,8.907,8.948,9.017,9.093,
     * 8.303,8.442,8.568,8.672,8.767,8.837,8.933,9.029,
     * 7.771,7.919,8.061,8.230,8.393,8.529,8.690,8.834,
     * 7.477,7.542,7.613,7.675,7.720,7.759,7.818,7.987,
     * 7.324,7.422,7.531,7.616,7.681,7.728,7.794,7.936,
     * 0.000,6.176,6.425,6.647,6.850,7.097,7.364,7.699,
     * 0.000,0.000,0.000,0.000,6.146,6.290,6.453,7.146,
     * 0.000,0.000,0.000,0.000,6.100,6.244,6.408,7.100/
      DATA ((EDEN(I,5,11,L,2),I=17,24),L=1,NZEN)/
     * 9.605,9.571,9.513,9.512,9.512,9.483,9.491,9.531,
     * 9.410,9.407,9.382,9.364,9.346,9.348,9.389,9.447,
     * 9.350,9.356,9.342,9.330,9.320,9.324,9.367,9.425,
     * 9.225,9.253,9.265,9.267,9.272,9.286,9.334,9.398,
     * 8.898,8.981,9.053,9.100,9.143,9.185,9.260,9.340,
     * 8.616,8.740,8.858,8.943,9.021,9.090,9.188,9.286,
     * 8.057,8.220,8.384,8.548,8.703,8.829,8.985,9.127,
     * 7.675,7.733,7.804,7.871,7.924,7.977,8.057,8.270,
     * 7.540,7.623,7.727,7.811,7.879,7.932,8.009,8.179,
     * 0.000,6.412,6.659,6.882,7.086,7.336,7.611,7.957,
     * 0.000,0.000,6.061,6.230,6.382,6.531,6.705,7.398,
     * 0.000,0.000,6.015,6.185,6.336,6.486,6.659,7.352/
      DATA ((EDEN(I,1,12,L,2),I=17,24),L=1,NZEN)/
     * 8.805,8.842,8.829,8.821,8.801,8.780,8.774,8.834,
     * 8.760,8.801,8.792,8.776,8.747,8.714,8.695,8.720,
     * 8.676,8.727,8.730,8.725,8.708,8.679,8.665,8.688,
     * 8.509,8.582,8.611,8.626,8.629,8.614,8.612,8.638,
     * 8.097,8.217,8.297,8.369,8.427,8.452,8.486,8.534,
     * 7.783,7.927,8.037,8.143,8.236,8.294,8.365,8.439,
     * 0.000,7.461,7.550,7.679,7.790,7.910,8.037,8.173,
     * 7.121,7.217,7.294,7.340,7.362,7.362,7.367,7.441,
     * 6.920,7.061,7.190,7.270,7.320,7.340,7.362,7.435,
     * 0.000,0.000,0.000,6.199,6.394,6.630,6.870,7.170,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.634,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.589/
      DATA ((EDEN(I,2,12,L,2),I=17,24),L=1,NZEN)/
     * 8.875,8.903,8.886,8.881,8.867,8.853,8.862,8.938,
     * 8.823,8.856,8.844,8.827,8.803,8.782,8.784,8.828,
     * 8.740,8.783,8.784,8.778,8.763,8.747,8.753,8.798,
     * 8.575,8.640,8.667,8.683,8.690,8.686,8.702,8.751,
     * 8.164,8.276,8.356,8.430,8.491,8.528,8.582,8.652,
     * 7.850,7.989,8.097,8.207,8.303,8.375,8.467,8.560,
     * 0.000,7.512,7.603,7.737,7.856,7.992,8.143,8.299,
     * 0.000,7.258,7.332,7.380,7.408,7.423,7.452,7.551,
     * 6.973,7.104,7.230,7.310,7.364,7.398,7.444,7.542,
     * 0.000,0.000,6.029,6.255,6.455,6.700,6.956,7.281,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.033,6.739,
     * 0.000,0.000,0.000,0.000,0.000,0.000,5.988,6.693/
      DATA ((EDEN(I,3,12,L,2),I=17,24),L=1,NZEN)/
     * 9.021,9.037,9.017,9.013,9.009,8.996,9.017,9.090,
     * 8.942,8.967,8.952,8.931,8.906,8.897,8.919,8.982,
     * 8.863,8.898,8.897,8.886,8.871,8.866,8.892,8.954,
     * 8.705,8.763,8.787,8.799,8.803,8.810,8.845,8.911,
     * 8.303,8.413,8.496,8.566,8.627,8.669,8.738,8.824,
     * 7.993,8.130,8.243,8.352,8.453,8.530,8.632,8.740,
     * 0.000,7.628,7.733,7.883,8.021,8.167,8.338,8.505,
     * 7.267,7.340,7.410,7.461,7.493,7.522,7.574,7.708,
     * 7.086,7.201,7.316,7.394,7.452,7.498,7.562,7.688,
     * 0.000,0.000,6.155,6.373,6.573,6.823,7.093,7.433,
     * 0.000,0.000,0.000,0.000,0.000,6.013,6.176,6.888,
     * 0.000,0.000,0.000,0.000,0.000,5.967,6.130,6.842/
      DATA ((EDEN(I,4,12,L,2),I=17,24),L=1,NZEN)/
     * 9.314,9.303,9.255,9.250,9.246,9.220,9.233,9.294,
     * 9.196,9.201,9.176,9.149,9.121,9.121,9.158,9.228,
     * 9.127,9.143,9.127,9.111,9.093,9.093,9.134,9.204,
     * 8.993,9.033,9.041,9.041,9.037,9.049,9.097,9.170,
     * 8.632,8.732,8.808,8.854,8.897,8.937,9.013,9.104,
     * 8.332,8.473,8.591,8.680,8.764,8.831,8.932,9.041,
     * 7.786,7.943,8.086,8.250,8.407,8.537,8.701,8.856,
     * 7.461,7.528,7.592,7.649,7.692,7.735,7.806,8.000,
     * 7.316,7.410,7.511,7.589,7.649,7.699,7.775,7.936,
     * 0.000,6.179,6.423,6.637,6.834,7.079,7.354,7.704,
     * 0.000,0.000,0.000,0.000,6.127,6.270,6.441,7.146,
     * 0.000,0.000,0.000,0.000,6.081,6.224,6.395,7.100/
      DATA ((EDEN(I,5,12,L,2),I=17,24),L=1,NZEN)/
     * 9.632,9.583,9.501,9.481,9.464,9.430,9.444,9.498,
     * 9.418,9.412,9.375,9.344,9.320,9.320,9.369,9.441,
     * 9.358,9.360,9.334,9.314,9.294,9.299,9.350,9.423,
     * 9.241,9.262,9.260,9.253,9.248,9.262,9.320,9.400,
     * 8.924,9.000,9.057,9.093,9.127,9.170,9.253,9.350,
     * 8.649,8.768,8.871,8.942,9.009,9.079,9.188,9.301,
     * 8.079,8.250,8.412,8.561,8.707,8.830,8.994,9.149,
     * 7.656,7.715,7.778,7.839,7.890,7.953,8.053,8.301,
     * 7.522,7.606,7.699,7.778,7.844,7.902,7.994,8.193,
     * 6.188,6.415,6.651,6.866,7.068,7.318,7.604,7.975,
     * 0.000,0.000,6.041,6.204,6.354,6.509,6.697,7.408,
     * 0.000,0.000,5.996,6.158,6.308,6.463,6.651,7.362/
      DATA ((EDEN(I,1,1,L,3),I=17,24),L=1,NZEN)/
     * 8.931,8.979,8.977,8.970,8.952,8.929,8.921,8.968,
     * 8.877,8.924,8.922,8.908,8.881,8.851,8.833,8.851,
     * 8.808,8.870,8.884,8.883,8.869,8.843,8.827,8.839,
     * 8.664,8.753,8.797,8.821,8.832,8.822,8.814,8.821,
     * 8.286,8.428,8.530,8.620,8.695,8.727,8.750,8.764,
     * 7.988,8.158,8.294,8.423,8.537,8.606,8.661,8.690,
     * 0.000,7.723,7.838,7.996,8.134,8.270,8.378,8.455,
     * 7.450,7.581,7.695,7.772,7.821,7.831,7.789,7.771,
     * 0.000,0.000,7.627,7.741,7.823,7.852,7.820,7.785,
     * 0.000,6.097,6.408,6.664,6.886,7.124,7.303,7.507,
     * 0.000,0.000,0.000,0.000,6.167,6.301,6.371,6.942,
     * 0.000,0.000,0.000,0.000,6.122,6.255,6.325,6.896/
      DATA ((EDEN(I,2,1,L,3),I=17,24),L=1,NZEN)/
     * 8.992,9.029,9.017,9.017,9.004,8.987,8.995,9.061,
     * 8.945,8.985,8.980,8.964,8.940,8.920,8.922,8.961,
     * 8.876,8.929,8.939,8.938,8.928,8.912,8.915,8.949,
     * 8.734,8.816,8.857,8.881,8.895,8.895,8.906,8.934,
     * 8.358,8.496,8.598,8.688,8.766,8.806,8.846,8.881,
     * 0.000,8.228,8.360,8.494,8.614,8.692,8.762,8.811,
     * 0.000,7.777,7.897,8.064,8.212,8.360,8.489,8.584,
     * 0.000,7.621,7.732,7.814,7.869,7.891,7.872,7.881,
     * 7.332,7.504,7.668,7.785,7.872,7.912,7.902,7.894,
     * 0.000,0.000,6.464,6.721,6.949,7.196,7.389,7.618,
     * 0.000,0.000,0.000,6.041,6.228,6.369,6.458,7.049,
     * 0.000,0.000,0.000,5.996,6.182,6.323,6.412,7.003/
      DATA ((EDEN(I,3,1,L,3),I=17,24),L=1,NZEN)/
     * 9.124,9.146,9.130,9.127,9.121,9.107,9.127,9.201,
     * 9.068,9.097,9.086,9.064,9.041,9.033,9.061,9.117,
     * 9.004,9.049,9.053,9.045,9.033,9.029,9.053,9.107,
     * 8.874,8.945,8.980,8.997,9.009,9.017,9.045,9.093,
     * 8.509,8.644,8.748,8.829,8.901,8.944,8.998,9.049,
     * 8.215,8.382,8.522,8.650,8.769,8.847,8.928,8.992,
     * 0.000,7.906,8.041,8.223,8.391,8.543,8.687,8.790,
     * 0.000,7.709,7.815,7.899,7.961,7.997,8.000,8.045,
     * 7.449,7.605,7.760,7.876,7.966,8.017,8.025,8.049,
     * 0.000,6.294,6.592,6.846,7.072,7.324,7.533,7.780,
     * 0.000,0.000,0.000,6.161,6.346,6.496,6.602,7.204,
     * 0.000,0.000,0.000,6.116,6.301,6.450,6.556,7.158/
      DATA ((EDEN(I,4,1,L,3),I=17,24),L=1,NZEN)/
     * 9.391,9.386,9.340,9.334,9.328,9.305,9.324,9.387,
     * 9.283,9.297,9.272,9.250,9.228,9.228,9.267,9.332,
     * 9.233,9.258,9.246,9.236,9.223,9.225,9.265,9.324,
     * 9.117,9.170,9.190,9.201,9.207,9.220,9.262,9.318,
     * 8.797,8.915,9.009,9.072,9.130,9.173,9.233,9.292,
     * 8.515,8.678,8.818,8.927,9.029,9.097,9.179,9.248,
     * 0.000,8.179,8.348,8.538,8.719,8.854,8.990,9.090,
     * 0.000,7.881,7.984,8.076,8.146,8.193,8.215,8.307,
     * 7.656,7.797,7.941,8.057,8.152,8.207,8.225,8.279,
     * 6.258,6.540,6.825,7.076,7.305,7.555,7.772,8.029,
     * 0.000,0.000,6.193,6.398,6.582,6.731,6.846,7.447,
     * 0.000,0.000,6.147,6.352,6.536,6.685,6.801,7.401/
      DATA ((EDEN(I,5,1,L,3),I=17,24),L=1,NZEN)/
     * 9.666,9.622,9.544,9.522,9.507,9.477,9.500,9.565,
     * 9.458,9.459,9.427,9.405,9.387,9.387,9.439,9.513,
     * 9.413,9.427,9.407,9.393,9.384,9.389,9.439,9.509,
     * 9.316,9.352,9.360,9.365,9.373,9.387,9.442,9.508,
     * 9.029,9.127,9.204,9.258,9.312,9.356,9.427,9.496,
     * 8.769,8.914,9.041,9.134,9.223,9.294,9.386,9.462,
     * 8.233,8.428,8.612,8.790,8.961,9.090,9.225,9.328,
     * 7.936,8.037,8.140,8.236,8.314,8.369,8.410,8.551,
     * 7.829,7.962,8.104,8.220,8.318,8.375,8.408,8.493,
     * 6.461,6.733,7.013,7.265,7.500,7.754,7.983,8.258,
     * 0.000,0.000,6.377,6.584,6.771,6.931,7.064,7.674,
     * 0.000,0.000,6.331,6.539,6.725,6.885,7.019,7.628/
      DATA ((EDEN(I,1,2,L,3),I=17,24),L=1,NZEN)/
     * 8.922,8.967,8.962,8.952,8.931,8.906,8.896,8.945,
     * 8.870,8.916,8.914,8.899,8.873,8.844,8.828,8.847,
     * 8.801,8.862,8.874,8.873,8.859,8.836,8.821,8.834,
     * 8.658,8.744,8.786,8.812,8.824,8.814,8.807,8.816,
     * 8.279,8.418,8.520,8.611,8.686,8.719,8.743,8.759,
     * 0.000,8.149,8.283,8.413,8.525,8.599,8.655,8.686,
     * 0.000,7.718,7.830,7.985,8.121,8.260,8.371,8.450,
     * 0.000,7.577,7.691,7.769,7.817,7.826,7.785,7.767,
     * 7.270,7.450,7.622,7.736,7.818,7.848,7.816,7.782,
     * 0.000,6.086,6.396,6.652,6.875,7.114,7.294,7.501,
     * 0.000,0.000,0.000,0.000,6.158,6.292,6.364,6.937,
     * 0.000,0.000,0.000,0.000,6.113,6.246,6.318,6.891/
      DATA ((EDEN(I,2,2,L,3),I=17,24),L=1,NZEN)/
     * 8.975,9.009,9.000,8.995,8.982,8.967,8.975,9.041,
     * 8.932,8.971,8.966,8.951,8.927,8.910,8.913,8.953,
     * 8.863,8.915,8.925,8.924,8.913,8.901,8.907,8.940,
     * 8.721,8.800,8.840,8.865,8.879,8.881,8.894,8.923,
     * 8.342,8.477,8.579,8.669,8.748,8.792,8.834,8.868,
     * 8.045,8.210,8.342,8.474,8.593,8.675,8.749,8.799,
     * 0.000,7.766,7.881,8.045,8.190,8.340,8.473,8.569,
     * 0.000,7.618,7.729,7.810,7.865,7.886,7.866,7.873,
     * 7.322,7.496,7.663,7.780,7.867,7.907,7.895,7.885,
     * 0.000,0.000,6.441,6.699,6.928,7.179,7.377,7.606,
     * 0.000,0.000,0.000,6.021,6.207,6.352,6.444,7.037,
     * 0.000,0.000,0.000,5.975,6.161,6.306,6.398,6.992/
      DATA ((EDEN(I,3,2,L,3),I=17,24),L=1,NZEN)/
     * 9.076,9.104,9.090,9.090,9.086,9.079,9.104,9.182,
     * 9.033,9.061,9.049,9.033,9.013,9.009,9.029,9.090,
     * 8.970,9.013,9.017,9.013,9.000,9.000,9.025,9.076,
     * 8.835,8.904,8.939,8.959,8.973,8.985,9.017,9.061,
     * 8.470,8.599,8.699,8.784,8.860,8.909,8.965,9.017,
     * 8.179,8.338,8.471,8.603,8.723,8.807,8.890,8.954,
     * 0.000,7.874,8.000,8.179,8.340,8.497,8.642,8.747,
     * 0.000,7.695,7.799,7.881,7.941,7.975,7.975,8.013,
     * 7.435,7.590,7.745,7.860,7.948,7.997,8.000,8.017,
     * 0.000,0.000,6.561,6.817,7.045,7.299,7.509,7.750,
     * 0.000,0.000,0.000,6.143,6.324,6.476,6.581,7.179,
     * 0.000,0.000,0.000,6.097,6.279,6.430,6.535,7.133/
      DATA ((EDEN(I,4,2,L,3),I=17,24),L=1,NZEN)/
     * 9.270,9.279,9.255,9.260,9.265,9.255,9.283,9.356,
     * 9.217,9.230,9.210,9.190,9.170,9.176,9.217,9.286,
     * 9.161,9.188,9.182,9.173,9.164,9.170,9.210,9.272,
     * 9.045,9.100,9.121,9.134,9.146,9.161,9.204,9.258,
     * 8.716,8.834,8.927,8.997,9.061,9.107,9.170,9.228,
     * 8.436,8.593,8.731,8.847,8.955,9.029,9.114,9.182,
     * 0.000,8.107,8.262,8.453,8.633,8.777,8.916,9.017,
     * 0.000,7.843,7.940,8.029,8.093,8.137,8.155,8.236,
     * 7.626,7.760,7.900,8.013,8.104,8.155,8.173,8.215,
     * 6.217,6.496,6.780,7.029,7.258,7.508,7.721,7.968,
     * 0.000,0.000,6.158,6.362,6.540,6.690,6.801,7.394,
     * 0.000,0.000,6.113,6.316,6.495,6.644,6.755,7.349/
      DATA ((EDEN(I,5,2,L,3),I=17,24),L=1,NZEN)/
     * 9.484,9.462,9.410,9.410,9.413,9.400,9.430,9.502,
     * 9.352,9.350,9.320,9.299,9.279,9.290,9.344,9.425,
     * 9.301,9.312,9.294,9.281,9.272,9.286,9.338,9.412,
     * 9.199,9.233,9.243,9.250,9.258,9.276,9.332,9.400,
     * 8.899,8.995,9.072,9.130,9.188,9.236,9.305,9.373,
     * 8.633,8.775,8.900,8.999,9.093,9.167,9.260,9.334,
     * 8.124,8.297,8.459,8.639,8.812,8.948,9.090,9.193,
     * 0.000,7.971,8.064,8.149,8.215,8.262,8.288,8.405,
     * 7.777,7.898,8.029,8.140,8.225,8.276,8.297,8.360,
     * 6.375,6.639,6.918,7.167,7.396,7.649,7.872,8.127,
     * 0.000,0.000,6.297,6.496,6.673,6.827,6.952,7.555,
     * 0.000,0.000,6.251,6.450,6.627,6.782,6.906,7.509/
      DATA ((EDEN(I,1,3,L,3),I=17,24),L=1,NZEN)/
     * 8.900,8.943,8.936,8.924,8.901,8.873,8.861,8.914,
     * 8.872,8.919,8.918,8.906,8.881,8.852,8.835,8.854,
     * 8.802,8.865,8.880,8.879,8.866,8.843,8.829,8.842,
     * 8.658,8.747,8.791,8.818,8.832,8.823,8.816,8.823,
     * 8.279,8.422,8.525,8.618,8.694,8.728,8.753,8.768,
     * 7.979,8.152,8.288,8.422,8.537,8.609,8.665,8.695,
     * 0.000,7.720,7.834,7.993,8.130,8.270,8.384,8.461,
     * 0.000,7.579,7.694,7.773,7.824,7.833,7.790,7.772,
     * 7.267,7.450,7.623,7.740,7.825,7.853,7.821,7.787,
     * 0.000,6.076,6.387,6.644,6.869,7.111,7.294,7.504,
     * 0.000,0.000,0.000,0.000,6.152,6.286,6.358,6.937,
     * 0.000,0.000,0.000,0.000,6.107,6.240,6.312,6.891/
      DATA ((EDEN(I,2,3,L,3),I=17,24),L=1,NZEN)/
     * 8.952,8.987,8.979,8.974,8.960,8.947,8.954,9.025,
     * 8.922,8.960,8.957,8.947,8.929,8.915,8.918,8.954,
     * 8.853,8.906,8.919,8.922,8.916,8.906,8.910,8.940,
     * 8.710,8.790,8.833,8.863,8.881,8.885,8.897,8.921,
     * 8.334,8.468,8.571,8.667,8.748,8.793,8.834,8.866,
     * 8.037,8.201,8.336,8.473,8.593,8.678,8.751,8.796,
     * 0.000,7.767,7.883,8.049,8.196,8.346,8.474,8.566,
     * 0.000,7.622,7.736,7.819,7.874,7.895,7.870,7.870,
     * 7.324,7.498,7.668,7.788,7.878,7.916,7.900,7.884,
     * 0.000,0.000,6.442,6.704,6.934,7.188,7.384,7.606,
     * 0.000,0.000,0.000,6.029,6.215,6.360,6.450,7.041,
     * 0.000,0.000,0.000,5.984,6.169,6.314,6.404,6.996/
      DATA ((EDEN(I,3,3,L,3),I=17,24),L=1,NZEN)/
     * 9.025,9.057,9.049,9.053,9.049,9.053,9.079,9.158,
     * 8.998,9.025,9.017,9.009,8.991,8.989,9.009,9.064,
     * 8.931,8.971,8.979,8.980,8.975,8.978,8.999,9.045,
     * 8.792,8.859,8.896,8.924,8.945,8.959,8.986,9.025,
     * 8.423,8.545,8.644,8.737,8.818,8.873,8.929,8.975,
     * 8.130,8.283,8.417,8.550,8.672,8.763,8.848,8.908,
     * 0.000,7.846,7.962,8.130,8.283,8.444,8.588,8.692,
     * 0.000,7.693,7.799,7.881,7.939,7.968,7.957,7.980,
     * 0.000,7.582,7.741,7.857,7.944,7.992,7.989,7.989,
     * 0.000,0.000,6.531,6.790,7.017,7.279,7.486,7.719,
     * 0.000,0.000,0.000,6.121,6.305,6.458,6.558,7.155,
     * 0.000,0.000,0.000,6.075,6.260,6.412,6.512,7.110/
      DATA ((EDEN(I,4,3,L,3),I=17,24),L=1,NZEN)/
     * 9.146,9.164,9.155,9.170,9.182,9.190,9.230,9.310,
     * 9.100,9.114,9.104,9.093,9.079,9.086,9.124,9.190,
     * 9.037,9.064,9.068,9.068,9.064,9.076,9.111,9.173,
     * 8.907,8.961,8.993,9.017,9.037,9.061,9.097,9.152,
     * 8.554,8.666,8.763,8.849,8.927,8.986,9.049,9.104,
     * 8.270,8.413,8.544,8.676,8.797,8.888,8.977,9.045,
     * 0.000,7.968,8.090,8.267,8.431,8.593,8.744,8.849,
     * 0.000,7.794,7.889,7.970,8.029,8.064,8.068,8.111,
     * 0.000,7.702,7.844,7.956,8.041,8.090,8.097,8.114,
     * 6.121,6.394,6.682,6.934,7.158,7.417,7.628,7.859,
     * 0.000,0.000,6.093,6.297,6.471,6.620,6.722,7.310,
     * 0.000,0.000,6.048,6.251,6.426,6.574,6.676,7.264/
      DATA ((EDEN(I,5,3,L,3),I=17,24),L=1,NZEN)/
     * 9.312,9.318,9.305,9.328,9.352,9.358,9.400,9.476,
     * 9.255,9.260,9.246,9.236,9.228,9.241,9.288,9.360,
     * 9.201,9.217,9.217,9.217,9.215,9.233,9.276,9.340,
     * 9.083,9.124,9.152,9.176,9.193,9.217,9.262,9.320,
     * 8.752,8.859,8.956,9.037,9.111,9.164,9.228,9.281,
     * 8.477,8.620,8.757,8.883,9.000,9.083,9.170,9.236,
     * 8.009,8.152,8.299,8.493,8.674,8.825,8.970,9.068,
     * 7.836,7.915,8.009,8.097,8.164,8.210,8.225,8.297,
     * 7.720,7.836,7.975,8.090,8.179,8.233,8.248,8.276,
     * 6.312,6.574,6.857,7.111,7.338,7.595,7.810,8.037,
     * 0.000,6.053,6.270,6.473,6.647,6.799,6.909,7.486,
     * 0.000,6.007,6.224,6.427,6.602,6.754,6.863,7.440/
      DATA ((EDEN(I,1,4,L,3),I=17,24),L=1,NZEN)/
     * 8.885,8.927,8.920,8.907,8.883,8.856,8.844,8.898,
     * 8.883,8.936,8.939,8.928,8.906,8.877,8.860,8.876,
     * 8.816,8.882,8.901,8.904,8.893,8.869,8.852,8.863,
     * 8.676,8.769,8.818,8.846,8.861,8.849,8.839,8.844,
     * 8.303,8.452,8.561,8.656,8.736,8.764,8.783,8.794,
     * 8.009,8.188,8.328,8.465,8.587,8.651,8.701,8.725,
     * 0.000,7.744,7.869,8.037,8.188,8.324,8.435,8.504,
     * 7.456,7.589,7.706,7.789,7.843,7.851,7.807,7.794,
     * 7.286,7.468,7.640,7.760,7.846,7.872,7.836,7.804,
     * 0.000,0.000,6.425,6.684,6.910,7.149,7.324,7.529,
     * 0.000,0.000,0.000,6.004,6.185,6.316,6.387,6.960,
     * 0.000,0.000,0.000,5.959,6.139,6.270,6.342,6.914/
      DATA ((EDEN(I,2,4,L,3),I=17,24),L=1,NZEN)/
     * 8.937,8.973,8.969,8.967,8.955,8.944,8.949,9.017,
     * 8.919,8.960,8.964,8.960,8.946,8.931,8.930,8.961,
     * 8.851,8.906,8.924,8.933,8.931,8.922,8.922,8.946,
     * 8.708,8.790,8.838,8.873,8.896,8.900,8.907,8.925,
     * 8.336,8.471,8.579,8.680,8.767,8.811,8.847,8.870,
     * 8.045,8.210,8.348,8.490,8.615,8.695,8.762,8.800,
     * 0.000,7.780,7.901,8.072,8.220,8.369,8.491,8.575,
     * 0.000,7.632,7.751,7.837,7.895,7.914,7.883,7.877,
     * 0.000,7.513,7.688,7.811,7.901,7.938,7.915,7.890,
     * 0.000,0.000,6.471,6.738,6.971,7.223,7.412,7.619,
     * 0.000,0.000,0.000,6.064,6.253,6.396,6.479,7.057,
     * 0.000,0.000,0.000,6.019,6.207,6.350,6.433,7.011/
      DATA ((EDEN(I,3,4,L,3),I=17,24),L=1,NZEN)/
     * 8.991,9.017,9.013,9.021,9.025,9.029,9.049,9.130,
     * 8.956,8.982,8.981,8.981,8.973,8.973,8.987,9.033,
     * 8.885,8.925,8.939,8.951,8.954,8.959,8.973,9.013,
     * 8.740,8.806,8.848,8.887,8.915,8.932,8.953,8.985,
     * 8.367,8.484,8.585,8.687,8.775,8.835,8.888,8.925,
     * 8.079,8.223,8.356,8.496,8.618,8.716,8.800,8.851,
     * 0.000,7.818,7.926,8.086,8.230,8.389,8.521,8.621,
     * 0.000,7.691,7.804,7.890,7.949,7.975,7.951,7.948,
     * 7.417,7.573,7.742,7.866,7.957,8.000,7.986,7.965,
     * 0.000,0.000,6.516,6.782,7.017,7.281,7.483,7.695,
     * 0.000,0.000,0.000,6.130,6.318,6.470,6.561,7.146,
     * 0.000,0.000,0.000,6.085,6.272,6.424,6.515,7.100/
      DATA ((EDEN(I,4,4,L,3),I=17,24),L=1,NZEN)/
     * 9.045,9.064,9.061,9.076,9.083,9.100,9.130,9.225,
     * 8.994,9.009,9.004,9.009,9.004,9.013,9.037,9.093,
     * 8.922,8.947,8.957,8.973,8.982,8.997,9.021,9.068,
     * 8.775,8.825,8.863,8.905,8.938,8.966,8.999,9.037,
     * 8.398,8.501,8.599,8.700,8.789,8.860,8.924,8.968,
     * 8.121,8.248,8.373,8.509,8.630,8.738,8.833,8.893,
     * 0.000,7.880,7.969,8.121,8.258,8.417,8.550,8.659,
     * 0.000,7.780,7.883,7.967,8.021,8.049,8.029,8.021,
     * 7.530,7.666,7.825,7.945,8.033,8.079,8.068,8.045,
     * 0.000,6.316,6.613,6.873,7.100,7.367,7.575,7.781,
     * 0.000,0.000,6.061,6.272,6.447,6.597,6.687,7.260,
     * 0.000,0.000,6.015,6.226,6.401,6.551,6.641,7.214/
      DATA ((EDEN(I,5,4,L,3),I=17,24),L=1,NZEN)/
     * 9.076,9.079,9.076,9.100,9.121,9.149,9.196,9.305,
     * 9.037,9.041,9.033,9.045,9.049,9.068,9.100,9.164,
     * 8.964,8.978,8.986,9.009,9.021,9.041,9.076,9.130,
     * 8.818,8.856,8.892,8.935,8.971,9.004,9.045,9.086,
     * 8.450,8.538,8.628,8.731,8.822,8.895,8.963,9.009,
     * 0.000,8.292,8.412,8.545,8.667,8.775,8.871,8.932,
     * 0.000,7.948,8.029,8.179,8.310,8.465,8.595,8.701,
     * 7.792,7.862,7.961,8.041,8.097,8.127,8.107,8.093,
     * 7.639,7.757,7.912,8.029,8.117,8.161,8.152,8.121,
     * 6.173,6.428,6.719,6.979,7.207,7.476,7.687,7.874,
     * 0.000,0.000,6.196,6.403,6.577,6.730,6.823,7.378,
     * 0.000,0.000,6.150,6.357,6.532,6.684,6.778,7.333/
      DATA ((EDEN(I,1,5,L,3),I=17,24),L=1,NZEN)/
     * 8.890,8.932,8.923,8.914,8.893,8.867,8.856,8.911,
     * 8.886,8.939,8.943,8.932,8.908,8.879,8.862,8.880,
     * 8.820,8.886,8.906,8.908,8.899,8.872,8.854,8.866,
     * 8.682,8.776,8.825,8.854,8.868,8.854,8.844,8.849,
     * 8.322,8.470,8.580,8.674,8.752,8.775,8.790,8.800,
     * 8.033,8.212,8.352,8.490,8.610,8.670,8.714,8.736,
     * 0.000,7.762,7.895,8.072,8.225,8.358,8.462,8.525,
     * 7.459,7.591,7.706,7.788,7.843,7.852,7.812,7.803,
     * 7.301,7.480,7.649,7.765,7.849,7.874,7.841,7.809,
     * 0.000,0.000,6.474,6.730,6.954,7.185,7.354,7.547,
     * 0.000,0.000,0.000,6.049,6.230,6.360,6.425,6.981,
     * 0.000,0.000,0.000,6.003,6.185,6.314,6.379,6.935/
      DATA ((EDEN(I,2,5,L,3),I=17,24),L=1,NZEN)/
     * 8.943,8.979,8.979,8.981,8.974,8.965,8.967,9.029,
     * 8.916,8.957,8.961,8.962,8.951,8.938,8.932,8.960,
     * 8.847,8.901,8.921,8.935,8.937,8.926,8.922,8.943,
     * 8.708,8.787,8.836,8.874,8.900,8.903,8.905,8.920,
     * 8.338,8.471,8.580,8.682,8.769,8.814,8.846,8.864,
     * 8.053,8.215,8.354,8.496,8.619,8.701,8.764,8.795,
     * 0.000,7.788,7.913,8.083,8.230,8.378,8.496,8.574,
     * 0.000,7.641,7.762,7.849,7.908,7.926,7.890,7.879,
     * 0.000,7.526,7.702,7.826,7.916,7.952,7.924,7.892,
     * 0.000,6.188,6.504,6.772,7.009,7.253,7.435,7.628,
     * 0.000,0.000,0.000,6.104,6.292,6.433,6.509,7.072,
     * 0.000,0.000,0.000,6.058,6.246,6.387,6.463,7.026/
      DATA ((EDEN(I,3,5,L,3),I=17,24),L=1,NZEN)/
     * 8.975,8.999,8.999,9.009,9.013,9.017,9.029,9.107,
     * 8.931,8.957,8.960,8.971,8.972,8.972,8.978,9.013,
     * 8.858,8.897,8.916,8.938,8.948,8.953,8.961,8.990,
     * 8.713,8.776,8.822,8.869,8.903,8.922,8.938,8.959,
     * 8.340,8.455,8.559,8.665,8.754,8.818,8.866,8.892,
     * 0.000,8.201,8.336,8.474,8.594,8.695,8.773,8.816,
     * 0.000,7.889,8.079,8.303,8.494,8.678,8.796,8.862,
     * 0.000,7.702,7.822,7.912,7.973,7.995,7.961,7.939,
     * 0.000,7.582,7.761,7.890,7.984,8.025,8.000,7.960,
     * 0.000,0.000,6.547,6.820,7.061,7.318,7.511,7.698,
     * 0.000,0.000,0.000,6.182,6.373,6.521,6.601,7.164,
     * 0.000,0.000,0.000,6.136,6.327,6.475,6.555,7.119/
      DATA ((EDEN(I,4,5,L,3),I=17,24),L=1,NZEN)/
     * 8.993,8.999,8.996,9.009,9.017,9.029,9.045,9.140,
     * 8.943,8.952,8.953,8.968,8.972,8.985,9.000,9.041,
     * 8.866,8.888,8.902,8.927,8.942,8.959,8.975,9.009,
     * 8.715,8.760,8.803,8.849,8.884,8.917,8.943,8.969,
     * 8.336,8.433,8.531,8.632,8.718,8.793,8.849,8.884,
     * 0.000,8.188,8.314,8.444,8.559,8.665,8.746,8.798,
     * 0.000,7.875,7.956,8.093,8.212,8.348,8.452,8.548,
     * 0.000,7.800,7.918,8.004,8.061,8.083,8.041,7.998,
     * 0.000,7.676,7.854,7.984,8.076,8.117,8.086,8.029,
     * 0.000,6.324,6.632,6.905,7.143,7.407,7.602,7.774,
     * 0.000,0.000,6.111,6.330,6.513,6.659,6.733,7.274,
     * 0.000,0.000,6.065,6.285,6.467,6.613,6.687,7.228/
      DATA ((EDEN(I,5,5,L,3),I=17,24),L=1,NZEN)/
     * 8.970,8.966,8.967,8.990,9.004,9.029,9.057,9.161,
     * 8.924,8.925,8.932,8.957,8.973,8.997,9.021,9.064,
     * 8.839,8.852,8.874,8.908,8.932,8.962,8.988,9.029,
     * 8.670,8.709,8.760,8.816,8.861,8.904,8.937,8.972,
     * 8.281,8.367,8.471,8.581,8.675,8.757,8.816,8.862,
     * 0.000,8.143,8.258,8.394,8.513,8.620,8.700,8.762,
     * 0.000,7.936,7.992,8.107,8.201,8.328,8.413,8.504,
     * 0.000,7.889,8.009,8.100,8.161,8.176,8.130,8.068,
     * 7.615,7.751,7.937,8.072,8.170,8.212,8.182,8.107,
     * 6.124,6.380,6.688,6.963,7.199,7.480,7.688,7.851,
     * 0.000,0.000,6.217,6.439,6.622,6.775,6.853,7.386,
     * 0.000,0.000,6.172,6.394,6.576,6.729,6.807,7.340/
      DATA ((EDEN(I,1,6,L,3),I=17,24),L=1,NZEN)/
     * 8.910,8.954,8.947,8.939,8.919,8.894,8.884,8.937,
     * 8.890,8.941,8.943,8.929,8.903,8.874,8.857,8.876,
     * 8.825,8.889,8.906,8.906,8.893,8.866,8.849,8.862,
     * 8.693,8.783,8.829,8.854,8.865,8.851,8.839,8.845,
     * 8.338,8.484,8.590,8.680,8.753,8.776,8.790,8.799,
     * 8.057,8.230,8.369,8.502,8.617,8.676,8.719,8.739,
     * 0.000,7.778,7.915,8.090,8.241,8.371,8.474,8.535,
     * 0.000,7.593,7.706,7.785,7.836,7.849,7.810,7.806,
     * 0.000,7.490,7.654,7.766,7.846,7.873,7.839,7.809,
     * 0.000,6.199,6.508,6.762,6.984,7.210,7.373,7.555,
     * 0.000,0.000,0.000,6.083,6.265,6.389,6.452,6.993,
     * 0.000,0.000,0.000,6.037,6.219,6.343,6.406,6.948/
      DATA ((EDEN(I,2,6,L,3),I=17,24),L=1,NZEN)/
     * 8.965,9.004,9.004,9.013,9.004,8.999,9.000,9.057,
     * 8.934,8.975,8.982,8.983,8.972,8.959,8.951,8.975,
     * 8.866,8.920,8.942,8.956,8.957,8.947,8.941,8.957,
     * 8.732,8.810,8.860,8.899,8.924,8.925,8.924,8.933,
     * 8.373,8.504,8.613,8.714,8.800,8.841,8.867,8.880,
     * 8.093,8.253,8.394,8.535,8.657,8.736,8.792,8.816,
     * 0.000,7.821,7.955,8.130,8.281,8.427,8.538,8.605,
     * 0.000,7.661,7.782,7.870,7.928,7.948,7.912,7.899,
     * 7.387,7.555,7.730,7.852,7.940,7.975,7.944,7.906,
     * 0.000,0.000,6.569,6.836,7.068,7.307,7.479,7.655,
     * 0.000,0.000,0.000,6.167,6.356,6.494,6.562,7.104,
     * 0.000,0.000,0.000,6.122,6.310,6.448,6.517,7.058/
      DATA ((EDEN(I,3,6,L,3),I=17,24),L=1,NZEN)/
     * 8.980,9.004,9.009,9.021,9.021,9.025,9.033,9.104,
     * 8.951,8.978,8.987,9.004,9.009,9.009,9.009,9.037,
     * 8.880,8.919,8.942,8.968,8.982,8.987,8.989,9.009,
     * 8.739,8.801,8.851,8.901,8.938,8.956,8.965,8.978,
     * 8.375,8.487,8.593,8.702,8.794,8.855,8.894,8.911,
     * 8.097,8.238,8.378,8.520,8.641,8.738,8.806,8.839,
     * 0.000,7.851,7.970,8.134,8.276,8.425,8.535,8.613,
     * 0.000,7.730,7.856,7.950,8.013,8.033,7.994,7.963,
     * 0.000,7.620,7.803,7.933,8.025,8.068,8.033,7.982,
     * 0.000,6.292,6.619,6.896,7.137,7.389,7.569,7.733,
     * 0.000,0.000,6.025,6.258,6.453,6.599,6.671,7.204,
     * 0.000,0.000,5.980,6.212,6.408,6.553,6.625,7.158/
      DATA ((EDEN(I,4,6,L,3),I=17,24),L=1,NZEN)/
     * 8.984,8.986,8.986,9.004,9.009,9.021,9.025,9.111,
     * 8.941,8.948,8.954,8.974,8.982,8.997,9.004,9.037,
     * 8.864,8.883,8.903,8.932,8.948,8.969,8.981,9.004,
     * 8.710,8.754,8.801,8.851,8.887,8.922,8.941,8.958,
     * 8.330,8.425,8.530,8.632,8.717,8.792,8.839,8.865,
     * 0.000,8.185,8.316,8.449,8.562,8.663,8.730,8.775,
     * 0.000,7.898,7.980,8.117,8.230,8.358,8.444,8.525,
     * 0.000,7.832,7.959,8.053,8.111,8.127,8.076,8.009,
     * 0.000,7.702,7.895,8.029,8.127,8.164,8.124,8.045,
     * 6.076,6.354,6.678,6.958,7.199,7.461,7.644,7.794,
     * 0.000,0.000,6.164,6.394,6.585,6.729,6.793,7.310,
     * 0.000,0.000,6.119,6.349,6.540,6.683,6.747,7.264/
      DATA ((EDEN(I,5,6,L,3),I=17,24),L=1,NZEN)/
     * 8.947,8.945,8.958,8.981,8.994,9.013,9.021,9.111,
     * 8.897,8.899,8.916,8.943,8.960,8.983,8.994,9.025,
     * 8.805,8.820,8.853,8.890,8.915,8.942,8.953,8.984,
     * 8.622,8.666,8.729,8.790,8.837,8.874,8.890,8.918,
     * 0.000,8.312,8.423,8.539,8.635,8.711,8.753,8.790,
     * 0.000,8.111,8.217,8.356,8.473,8.572,8.631,8.680,
     * 0.000,7.977,8.029,8.124,8.193,8.299,8.354,8.418,
     * 0.000,7.925,8.064,8.164,8.225,8.238,8.176,8.083,
     * 0.000,7.758,7.971,8.124,8.233,8.272,8.228,8.127,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,7,L,3),I=17,24),L=1,NZEN)/
     * 8.926,8.969,8.962,8.952,8.931,8.907,8.897,8.948,
     * 8.887,8.933,8.930,8.915,8.887,8.859,8.843,8.863,
     * 8.822,8.882,8.895,8.892,8.875,8.851,8.835,8.849,
     * 8.688,8.775,8.817,8.839,8.846,8.834,8.823,8.831,
     * 8.330,8.473,8.575,8.661,8.730,8.757,8.773,8.784,
     * 8.045,8.217,8.352,8.481,8.592,8.653,8.699,8.723,
     * 0.000,7.767,7.899,8.068,8.212,8.346,8.452,8.517,
     * 0.000,7.590,7.701,7.777,7.826,7.839,7.800,7.795,
     * 0.000,7.484,7.649,7.759,7.836,7.863,7.831,7.799,
     * 0.000,0.000,6.496,6.748,6.968,7.196,7.360,7.544,
     * 0.000,0.000,0.000,6.072,6.250,6.377,6.441,6.983,
     * 0.000,0.000,0.000,6.026,6.205,6.331,6.395,6.937/
      DATA ((EDEN(I,2,7,L,3),I=17,24),L=1,NZEN)/
     * 8.985,9.025,9.029,9.037,9.029,9.021,9.021,9.076,
     * 8.958,8.997,9.004,9.004,8.995,8.982,8.974,8.996,
     * 8.890,8.944,8.967,8.979,8.978,8.969,8.962,8.978,
     * 8.757,8.835,8.885,8.923,8.947,8.950,8.948,8.956,
     * 8.401,8.533,8.644,8.745,8.829,8.869,8.892,8.903,
     * 8.121,8.283,8.427,8.567,8.689,8.766,8.820,8.841,
     * 0.000,7.843,7.983,8.161,8.316,8.461,8.572,8.636,
     * 0.000,7.674,7.794,7.885,7.944,7.966,7.930,7.920,
     * 7.405,7.572,7.747,7.869,7.956,7.993,7.962,7.924,
     * 0.000,0.000,6.591,6.857,7.090,7.330,7.501,7.676,
     * 0.000,0.000,0.000,6.185,6.377,6.515,6.583,7.121,
     * 0.000,0.000,0.000,6.139,6.331,6.469,6.537,7.075/
      DATA ((EDEN(I,3,7,L,3),I=17,24),L=1,NZEN)/
     * 9.004,9.025,9.033,9.049,9.057,9.057,9.061,9.127,
     * 8.998,9.025,9.041,9.053,9.057,9.057,9.057,9.083,
     * 8.928,8.969,8.997,9.025,9.037,9.041,9.041,9.057,
     * 8.790,8.855,8.909,8.960,8.997,9.013,9.017,9.025,
     * 8.435,8.550,8.662,8.772,8.866,8.922,8.955,8.965,
     * 8.158,8.303,8.446,8.593,8.720,8.812,8.875,8.897,
     * 0.000,7.897,8.029,8.204,8.358,8.508,8.617,8.685,
     * 0.000,7.756,7.884,7.980,8.045,8.072,8.029,8.004,
     * 7.498,7.654,7.836,7.968,8.061,8.100,8.068,8.017,
     * 6.045,6.346,6.675,6.953,7.193,7.442,7.617,7.774,
     * 0.000,0.000,6.076,6.307,6.502,6.648,6.718,7.241,
     * 0.000,0.000,6.030,6.262,6.457,6.603,6.672,7.195/
      DATA ((EDEN(I,4,7,L,3),I=17,24),L=1,NZEN)/
     * 9.013,9.017,9.021,9.041,9.049,9.064,9.068,9.146,
     * 8.987,8.998,9.009,9.033,9.045,9.061,9.064,9.090,
     * 8.913,8.936,8.960,8.994,9.017,9.033,9.041,9.057,
     * 8.763,8.810,8.862,8.917,8.958,8.992,9.009,9.017,
     * 8.393,8.490,8.600,8.707,8.797,8.871,8.915,8.931,
     * 0.000,8.248,8.387,8.526,8.644,8.747,8.814,8.848,
     * 0.000,7.934,8.029,8.182,8.305,8.441,8.530,8.606,
     * 0.000,7.857,7.988,8.086,8.146,8.167,8.114,8.053,
     * 0.000,7.736,7.929,8.068,8.164,8.201,8.161,8.083,
     * 6.127,6.407,6.733,7.017,7.260,7.520,7.699,7.838,
     * 0.000,0.000,6.210,6.444,6.636,6.780,6.841,7.344,
     * 0.000,0.000,6.164,6.398,6.591,6.734,6.795,7.299/
      DATA ((EDEN(I,5,7,L,3),I=17,24),L=1,NZEN)/
     * 8.994,8.991,9.004,9.033,9.053,9.072,9.079,9.164,
     * 8.940,8.943,8.965,8.997,9.017,9.041,9.053,9.079,
     * 8.852,8.869,8.905,8.946,8.975,9.004,9.017,9.037,
     * 8.679,8.722,8.789,8.853,8.902,8.943,8.960,8.980,
     * 8.283,8.375,8.497,8.615,8.713,8.792,8.834,8.864,
     * 0.000,8.164,8.286,8.430,8.553,8.655,8.714,8.759,
     * 0.000,8.000,8.064,8.176,8.260,8.377,8.439,8.500,
     * 0.000,7.949,8.093,8.196,8.262,8.276,8.215,8.121,
     * 0.000,7.786,8.004,8.158,8.272,8.312,8.267,8.164,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,8,L,3),I=17,24),L=1,NZEN)/
     * 8.926,8.967,8.959,8.947,8.924,8.896,8.884,8.936,
     * 8.877,8.924,8.922,8.905,8.876,8.848,8.832,8.852,
     * 8.811,8.870,8.882,8.879,8.862,8.838,8.823,8.838,
     * 8.671,8.757,8.799,8.822,8.832,8.820,8.812,8.820,
     * 8.305,8.446,8.548,8.635,8.707,8.736,8.757,8.770,
     * 8.017,8.185,8.318,8.447,8.559,8.625,8.676,8.702,
     * 0.000,7.743,7.866,8.029,8.170,8.305,8.415,8.487,
     * 7.455,7.582,7.694,7.770,7.818,7.829,7.790,7.780,
     * 7.292,7.470,7.635,7.747,7.825,7.853,7.821,7.789,
     * 0.000,0.000,6.450,6.704,6.925,7.158,7.330,7.524,
     * 0.000,0.000,0.000,6.029,6.210,6.340,6.407,6.962,
     * 0.000,0.000,0.000,5.984,6.164,6.295,6.361,6.916/
      DATA ((EDEN(I,2,8,L,3),I=17,24),L=1,NZEN)/
     * 8.996,9.033,9.037,9.041,9.033,9.021,9.021,9.076,
     * 8.959,8.999,9.004,9.004,8.990,8.977,8.970,8.992,
     * 8.891,8.944,8.965,8.977,8.977,8.967,8.959,8.975,
     * 8.753,8.831,8.881,8.918,8.941,8.945,8.946,8.954,
     * 8.386,8.520,8.629,8.732,8.816,8.859,8.886,8.899,
     * 8.097,8.260,8.403,8.545,8.668,8.750,8.809,8.835,
     * 0.000,7.825,7.957,8.130,8.283,8.433,8.549,8.620,
     * 0.000,7.670,7.792,7.882,7.941,7.962,7.927,7.913,
     * 7.387,7.558,7.736,7.859,7.949,7.987,7.960,7.923,
     * 0.000,0.000,6.544,6.812,7.045,7.292,7.473,7.660,
     * 0.000,0.000,0.000,6.140,6.330,6.473,6.547,7.100,
     * 0.000,0.000,0.000,6.094,6.285,6.427,6.501,7.055/
      DATA ((EDEN(I,3,8,L,3),I=17,24),L=1,NZEN)/
     * 9.037,9.064,9.072,9.090,9.093,9.100,9.107,9.170,
     * 9.029,9.061,9.072,9.086,9.086,9.086,9.086,9.111,
     * 8.959,9.004,9.033,9.057,9.068,9.072,9.072,9.090,
     * 8.820,8.889,8.945,8.994,9.029,9.045,9.053,9.061,
     * 8.456,8.580,8.695,8.807,8.901,8.957,8.992,9.004,
     * 8.176,8.326,8.476,8.624,8.755,8.847,8.912,8.937,
     * 0.000,7.908,8.041,8.223,8.380,8.537,8.655,8.724,
     * 0.000,7.763,7.890,7.989,8.057,8.083,8.049,8.029,
     * 0.000,7.657,7.838,7.971,8.068,8.111,8.083,8.037,
     * 6.029,6.330,6.656,6.933,7.173,7.428,7.610,7.785,
     * 0.000,0.000,6.049,6.279,6.473,6.622,6.698,7.241,
     * 0.000,0.000,6.003,6.233,6.427,6.576,6.652,7.195/
      DATA ((EDEN(I,4,8,L,3),I=17,24),L=1,NZEN)/
     * 9.076,9.086,9.097,9.121,9.134,9.149,9.161,9.233,
     * 9.053,9.072,9.086,9.111,9.124,9.134,9.140,9.161,
     * 8.982,9.013,9.041,9.076,9.100,9.114,9.121,9.137,
     * 8.837,8.892,8.950,9.009,9.053,9.083,9.097,9.104,
     * 8.473,8.579,8.694,8.810,8.908,8.979,9.025,9.033,
     * 0.000,8.330,8.476,8.625,8.756,8.862,8.936,8.962,
     * 0.000,7.969,8.083,8.253,8.398,8.550,8.660,8.734,
     * 0.000,7.867,7.996,8.097,8.161,8.188,8.149,8.107,
     * 0.000,7.756,7.943,8.079,8.179,8.223,8.190,8.127,
     * 6.155,6.438,6.760,7.041,7.283,7.544,7.727,7.880,
     * 0.000,0.000,6.217,6.447,6.638,6.785,6.854,7.369,
     * 0.000,0.000,6.172,6.401,6.593,6.740,6.809,7.323/
      DATA ((EDEN(I,5,8,L,3),I=17,24),L=1,NZEN)/
     * 9.104,9.100,9.107,9.140,9.161,9.188,9.207,9.292,
     * 9.049,9.049,9.068,9.097,9.117,9.143,9.155,9.179,
     * 8.969,8.985,9.017,9.057,9.083,9.114,9.130,9.146,
     * 8.816,8.855,8.914,8.975,9.025,9.068,9.090,9.100,
     * 0.000,8.529,8.646,8.760,8.856,8.938,8.989,9.009,
     * 0.000,8.290,8.431,8.577,8.702,8.810,8.881,8.920,
     * 0.000,8.021,8.104,8.250,8.373,8.508,8.594,8.669,
     * 0.000,7.958,8.093,8.196,8.265,8.288,8.238,8.164,
     * 0.000,0.000,8.025,8.173,8.279,8.322,8.286,8.199,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
      DATA ((EDEN(I,1,9,L,3),I=17,24),L=1,NZEN)/
     * 8.911,8.952,8.943,8.929,8.904,8.876,8.863,8.915,
     * 8.877,8.926,8.926,8.911,8.884,8.854,8.837,8.856,
     * 8.810,8.872,8.886,8.885,8.872,8.846,8.829,8.842,
     * 8.670,8.759,8.802,8.828,8.839,8.827,8.816,8.825,
     * 8.301,8.442,8.545,8.635,8.709,8.739,8.760,8.772,
     * 8.009,8.179,8.314,8.446,8.559,8.625,8.677,8.703,
     * 0.000,7.737,7.859,8.021,8.164,8.299,8.407,8.480,
     * 0.000,7.583,7.696,7.775,7.823,7.833,7.792,7.779,
     * 7.286,7.465,7.634,7.747,7.828,7.856,7.823,7.790,
     * 0.000,0.000,6.435,6.690,6.913,7.146,7.320,7.519,
     * 0.000,0.000,0.000,6.013,6.196,6.324,6.393,6.956,
     * 0.000,0.000,0.000,5.967,6.150,6.279,6.347,6.910/
      DATA ((EDEN(I,2,9,L,3),I=17,24),L=1,NZEN)/
     * 8.993,9.029,9.029,9.029,9.017,9.004,9.004,9.061,
     * 8.957,8.997,9.000,8.998,8.984,8.969,8.964,8.987,
     * 8.889,8.943,8.964,8.973,8.970,8.959,8.955,8.972,
     * 8.747,8.829,8.879,8.914,8.938,8.940,8.942,8.953,
     * 8.378,8.513,8.622,8.723,8.808,8.851,8.884,8.900,
     * 0.000,8.250,8.391,8.531,8.655,8.736,8.800,8.831,
     * 0.000,7.812,7.939,8.111,8.262,8.412,8.531,8.607,
     * 0.000,7.660,7.782,7.870,7.929,7.950,7.918,7.907,
     * 0.000,7.544,7.720,7.844,7.934,7.974,7.950,7.919,
     * 0.000,0.000,6.519,6.785,7.017,7.265,7.449,7.648,
     * 0.000,0.000,0.000,6.111,6.299,6.442,6.521,7.086,
     * 0.000,0.000,0.000,6.065,6.253,6.397,6.475,7.041/
      DATA ((EDEN(I,3,9,L,3),I=17,24),L=1,NZEN)/
     * 9.079,9.111,9.114,9.130,9.137,9.140,9.149,9.210,
     * 9.045,9.079,9.090,9.093,9.093,9.090,9.093,9.121,
     * 8.978,9.025,9.049,9.068,9.076,9.079,9.083,9.104,
     * 8.838,8.912,8.967,9.013,9.041,9.057,9.068,9.083,
     * 8.471,8.599,8.714,8.824,8.918,8.971,9.013,9.029,
     * 0.000,8.340,8.489,8.638,8.772,8.862,8.931,8.964,
     * 0.000,7.906,8.041,8.223,8.387,8.544,8.670,8.748,
     * 0.000,7.754,7.879,7.976,8.045,8.072,8.049,8.041,
     * 7.486,7.645,7.824,7.954,8.049,8.097,8.079,8.053,
     * 0.000,6.312,6.631,6.905,7.143,7.401,7.592,7.790,
     * 0.000,0.000,6.017,6.243,6.438,6.588,6.670,7.233,
     * 0.000,0.000,5.971,6.197,6.392,6.542,6.624,7.187/
      DATA ((EDEN(I,4,9,L,3),I=17,24),L=1,NZEN)/
     * 9.182,9.204,9.215,9.246,9.265,9.276,9.294,9.356,
     * 9.146,9.173,9.190,9.207,9.212,9.217,9.225,9.255,
     * 9.083,9.121,9.152,9.179,9.199,9.207,9.215,9.238,
     * 8.945,9.013,9.072,9.127,9.167,9.188,9.201,9.215,
     * 8.587,8.708,8.831,8.947,9.049,9.107,9.149,9.164,
     * 8.348,8.456,8.612,8.769,8.911,9.004,9.072,9.100,
     * 0.000,8.029,8.170,8.364,8.537,8.700,8.829,8.899,
     * 0.000,7.875,8.000,8.104,8.176,8.212,8.190,8.188,
     * 0.000,7.776,7.952,8.086,8.188,8.238,8.225,8.193,
     * 0.000,0.000,6.801,7.076,7.314,7.572,7.762,7.943,
     * 0.000,0.000,6.225,6.450,6.639,6.787,6.866,7.401,
     * 0.000,0.000,6.180,6.404,6.594,6.741,6.821,7.356/
      DATA ((EDEN(I,5,9,L,3),I=17,24),L=1,NZEN)/
     * 9.318,9.334,9.352,9.391,9.423,9.439,9.459,9.517,
     * 9.270,9.290,9.316,9.336,9.350,9.358,9.367,9.391,
     * 9.204,9.238,9.279,9.310,9.334,9.346,9.358,9.373,
     * 9.072,9.134,9.201,9.260,9.307,9.330,9.346,9.352,
     * 8.720,8.836,8.967,9.090,9.196,9.255,9.297,9.305,
     * 8.441,8.585,8.750,8.915,9.064,9.158,9.228,9.248,
     * 0.000,8.152,8.307,8.513,8.698,8.863,8.992,9.053,
     * 0.000,7.992,8.124,8.236,8.316,8.360,8.346,8.338,
     * 7.763,7.897,8.079,8.223,8.330,8.389,8.380,8.342,
     * 6.360,6.631,6.952,7.233,7.479,7.739,7.930,8.097,
     * 0.000,6.143,6.396,6.627,6.820,6.972,7.053,7.567,
     * 0.000,6.097,6.350,6.582,6.774,6.926,7.007,7.521/
      DATA ((EDEN(I,1,10,L,3),I=17,24),L=1,NZEN)/
     * 8.900,8.943,8.936,8.923,8.899,8.872,8.859,8.912,
     * 8.895,8.947,8.950,8.940,8.916,8.885,8.866,8.880,
     * 8.827,8.893,8.912,8.913,8.901,8.876,8.859,8.867,
     * 8.688,8.780,8.829,8.857,8.872,8.857,8.846,8.849,
     * 8.314,8.464,8.574,8.668,8.747,8.771,8.786,8.797,
     * 8.021,8.199,8.340,8.479,8.598,8.659,8.704,8.727,
     * 0.000,7.752,7.879,8.049,8.199,8.330,8.436,8.502,
     * 7.461,7.594,7.711,7.794,7.848,7.856,7.812,7.797,
     * 7.294,7.477,7.649,7.767,7.852,7.876,7.839,7.807,
     * 0.000,0.000,6.446,6.703,6.928,7.161,7.332,7.534,
     * 0.000,0.000,0.000,6.021,6.201,6.330,6.398,6.965,
     * 0.000,0.000,0.000,5.975,6.156,6.285,6.352,6.919/
      DATA ((EDEN(I,2,10,L,3),I=17,24),L=1,NZEN)/
     * 8.989,9.025,9.021,9.017,9.009,8.992,8.993,9.053,
     * 8.968,9.013,9.017,9.009,8.994,8.975,8.970,8.996,
     * 8.900,8.958,8.979,8.985,8.982,8.968,8.964,8.984,
     * 8.769,8.844,8.895,8.929,8.951,8.949,8.952,8.966,
     * 8.387,8.528,8.639,8.738,8.822,8.862,8.894,8.914,
     * 0.000,8.262,8.405,8.548,8.673,8.749,8.811,8.844,
     * 0.000,7.814,7.943,8.117,8.274,8.420,8.540,8.620,
     * 0.000,7.657,7.777,7.866,7.928,7.947,7.918,7.914,
     * 7.365,7.540,7.714,7.839,7.931,7.969,7.948,7.926,
     * 0.000,0.000,6.513,6.777,7.009,7.255,7.441,7.653,
     * 0.000,0.000,0.000,6.097,6.286,6.425,6.505,7.083,
     * 0.000,0.000,0.000,6.051,6.240,6.379,6.459,7.037/
      DATA ((EDEN(I,3,10,L,3),I=17,24),L=1,NZEN)/
     * 9.121,9.149,9.149,9.161,9.167,9.161,9.173,9.233,
     * 9.068,9.100,9.107,9.104,9.093,9.086,9.097,9.130,
     * 9.004,9.049,9.072,9.079,9.083,9.079,9.090,9.117,
     * 8.865,8.941,8.991,9.025,9.053,9.061,9.079,9.104,
     * 8.497,8.629,8.744,8.846,8.934,8.982,9.025,9.053,
     * 8.207,8.367,8.515,8.660,8.792,8.876,8.947,8.989,
     * 0.000,7.913,8.049,8.233,8.400,8.556,8.689,8.774,
     * 7.630,7.745,7.865,7.959,8.025,8.057,8.045,8.053,
     * 7.477,7.637,7.809,7.937,8.033,8.083,8.076,8.061,
     * 0.000,6.314,6.626,6.894,7.130,7.382,7.579,7.794,
     * 0.000,0.000,0.000,6.223,6.415,6.563,6.652,7.228,
     * 0.000,0.000,0.000,6.177,6.369,6.518,6.606,7.182/
      DATA ((EDEN(I,4,10,L,3),I=17,24),L=1,NZEN)/
     * 9.318,9.340,9.340,9.365,9.384,9.380,9.394,9.441,
     * 9.250,9.274,9.281,9.281,9.274,9.276,9.297,9.332,
     * 9.190,9.230,9.253,9.262,9.267,9.272,9.288,9.320,
     * 9.068,9.134,9.188,9.220,9.246,9.260,9.283,9.307,
     * 8.719,8.851,8.974,9.068,9.155,9.201,9.243,9.272,
     * 8.433,8.600,8.763,8.906,9.037,9.114,9.182,9.217,
     * 0.000,8.121,8.286,8.494,8.686,8.836,8.965,9.037,
     * 0.000,7.895,8.017,8.121,8.199,8.241,8.241,8.276,
     * 7.662,7.806,7.973,8.104,8.207,8.262,8.262,8.265,
     * 6.260,6.544,6.851,7.117,7.354,7.605,7.801,8.013,
     * 0.000,0.000,6.243,6.464,6.651,6.797,6.888,7.447,
     * 0.000,0.000,6.197,6.418,6.606,6.752,6.842,7.401/
      DATA ((EDEN(I,5,10,L,3),I=17,24),L=1,NZEN)/
     * 9.535,9.534,9.520,9.549,9.575,9.569,9.583,9.619,
     * 9.433,9.439,9.436,9.435,9.430,9.436,9.462,9.501,
     * 9.382,9.400,9.410,9.418,9.422,9.430,9.455,9.487,
     * 9.274,9.316,9.356,9.384,9.405,9.422,9.449,9.474,
     * 8.957,9.068,9.179,9.260,9.330,9.377,9.420,9.444,
     * 8.680,8.834,8.992,9.117,9.233,9.305,9.369,9.403,
     * 8.170,8.344,8.531,8.739,8.931,9.068,9.185,9.246,
     * 7.948,8.041,8.158,8.267,8.350,8.400,8.408,8.465,
     * 7.839,7.966,8.127,8.258,8.360,8.417,8.422,8.433,
     * 6.477,6.746,7.049,7.316,7.551,7.799,7.994,8.199,
     * 0.000,6.207,6.444,6.664,6.851,6.997,7.086,7.634,
     * 0.000,6.161,6.398,6.618,6.806,6.951,7.041,7.589/
      DATA ((EDEN(I,1,11,L,3),I=17,24),L=1,NZEN)/
     * 8.906,8.949,8.943,8.936,8.916,8.892,8.882,8.933,
     * 8.900,8.954,8.958,8.947,8.924,8.892,8.872,8.885,
     * 8.833,8.900,8.920,8.922,8.910,8.884,8.866,8.872,
     * 8.692,8.787,8.838,8.866,8.881,8.866,8.852,8.855,
     * 8.322,8.471,8.582,8.678,8.757,8.781,8.796,8.802,
     * 8.029,8.207,8.348,8.487,8.609,8.667,8.711,8.732,
     * 0.000,7.757,7.886,8.057,8.210,8.340,8.442,8.507,
     * 0.000,7.598,7.715,7.799,7.855,7.860,7.816,7.799,
     * 7.301,7.483,7.654,7.773,7.859,7.883,7.846,7.812,
     * 0.000,0.000,6.455,6.714,6.940,7.170,7.338,7.538,
     * 0.000,0.000,0.000,6.029,6.212,6.338,6.403,6.969,
     * 0.000,0.000,0.000,5.984,6.166,6.293,6.357,6.923/
      DATA ((EDEN(I,2,11,L,3),I=17,24),L=1,NZEN)/
     * 8.990,9.025,9.017,9.017,9.004,8.989,8.994,9.057,
     * 8.973,9.017,9.017,9.009,8.991,8.971,8.969,8.998,
     * 8.906,8.965,8.984,8.987,8.981,8.965,8.962,8.987,
     * 8.768,8.854,8.902,8.932,8.951,8.947,8.952,8.972,
     * 8.396,8.539,8.649,8.746,8.828,8.864,8.896,8.921,
     * 0.000,8.274,8.417,8.556,8.682,8.753,8.814,8.852,
     * 0.000,7.816,7.947,8.124,8.283,8.427,8.548,8.628,
     * 0.000,7.649,7.766,7.855,7.917,7.938,7.913,7.916,
     * 0.000,7.537,7.707,7.829,7.921,7.958,7.941,7.927,
     * 0.000,6.204,6.516,6.778,7.009,7.250,7.436,7.654,
     * 0.000,0.000,0.000,6.090,6.281,6.420,6.501,7.083,
     * 0.000,0.000,0.000,6.044,6.235,6.374,6.455,7.037/
      DATA ((EDEN(I,3,11,L,3),I=17,24),L=1,NZEN)/
     * 9.149,9.173,9.161,9.167,9.167,9.155,9.170,9.233,
     * 9.097,9.127,9.124,9.114,9.097,9.086,9.100,9.146,
     * 9.033,9.079,9.093,9.093,9.086,9.083,9.097,9.134,
     * 8.900,8.975,9.017,9.041,9.061,9.068,9.090,9.121,
     * 8.537,8.672,8.782,8.873,8.952,8.995,9.041,9.079,
     * 8.241,8.408,8.555,8.694,8.820,8.894,8.965,9.017,
     * 0.000,7.935,8.076,8.265,8.438,8.588,8.722,8.812,
     * 0.000,7.741,7.855,7.946,8.013,8.049,8.045,8.072,
     * 7.480,7.638,7.802,7.924,8.017,8.068,8.068,8.076,
     * 0.000,6.334,6.639,6.900,7.134,7.382,7.581,7.811,
     * 0.000,0.000,6.000,6.220,6.408,6.555,6.650,7.236,
     * 0.000,0.000,5.954,6.174,6.362,6.509,6.605,7.190/
      DATA ((EDEN(I,4,11,L,3),I=17,24),L=1,NZEN)/
     * 9.435,9.436,9.408,9.413,9.418,9.396,9.407,9.452,
     * 9.332,9.346,9.334,9.320,9.305,9.301,9.328,9.375,
     * 9.279,9.307,9.310,9.305,9.299,9.301,9.328,9.367,
     * 9.167,9.223,9.253,9.270,9.283,9.294,9.324,9.360,
     * 8.842,8.965,9.068,9.140,9.204,9.246,9.294,9.334,
     * 8.559,8.726,8.877,8.994,9.100,9.170,9.241,9.288,
     * 0.000,8.225,8.405,8.604,8.790,8.923,9.045,9.124,
     * 0.000,7.928,8.041,8.143,8.220,8.267,8.279,8.346,
     * 7.702,7.846,8.000,8.127,8.228,8.279,8.288,8.320,
     * 6.318,6.602,6.899,7.158,7.389,7.635,7.838,8.072,
     * 0.000,0.000,6.267,6.481,6.668,6.815,6.916,7.493,
     * 0.000,0.000,6.221,6.436,6.623,6.769,6.870,7.447/
      DATA ((EDEN(I,5,11,L,3),I=17,24),L=1,NZEN)/
     * 9.727,9.694,9.638,9.634,9.631,9.602,9.611,9.647,
     * 9.544,9.544,9.522,9.508,9.494,9.493,9.528,9.576,
     * 9.498,9.511,9.502,9.496,9.490,9.493,9.528,9.572,
     * 9.405,9.439,9.458,9.470,9.480,9.493,9.531,9.571,
     * 9.127,9.220,9.303,9.364,9.418,9.459,9.516,9.558,
     * 8.869,9.009,9.143,9.243,9.334,9.401,9.474,9.524,
     * 8.332,8.528,8.722,8.904,9.076,9.196,9.314,9.391,
     * 8.021,8.121,8.230,8.334,8.417,8.471,8.500,8.611,
     * 7.919,8.049,8.196,8.320,8.422,8.477,8.494,8.551,
     * 6.567,6.839,7.130,7.389,7.625,7.872,8.079,8.322,
     * 6.049,6.260,6.493,6.710,6.901,7.053,7.167,7.740,
     * 6.003,6.214,6.447,6.664,6.855,7.007,7.122,7.694/
      DATA ((EDEN(I,1,12,L,3),I=17,24),L=1,NZEN)/
     * 8.921,8.968,8.966,8.960,8.943,8.921,8.914,8.962,
     * 8.884,8.934,8.934,8.924,8.900,8.869,8.850,8.865,
     * 8.816,8.881,8.898,8.899,8.887,8.860,8.841,8.851,
     * 8.674,8.766,8.812,8.840,8.854,8.841,8.831,8.834,
     * 8.301,8.446,8.551,8.644,8.722,8.749,8.766,8.778,
     * 8.004,8.179,8.316,8.450,8.567,8.632,8.681,8.707,
     * 0.000,7.736,7.857,8.021,8.164,8.297,8.403,8.474,
     * 0.000,7.588,7.702,7.782,7.833,7.842,7.800,7.782,
     * 0.000,7.468,7.639,7.755,7.838,7.865,7.831,7.796,
     * 0.000,0.000,6.431,6.689,6.914,7.146,7.320,7.521,
     * 0.000,0.000,0.000,6.009,6.193,6.320,6.386,6.953,
     * 0.000,0.000,0.000,5.963,6.147,6.274,6.340,6.908/
      DATA ((EDEN(I,2,12,L,3),I=17,24),L=1,NZEN)/
     * 8.996,9.033,9.025,9.021,9.009,8.992,9.000,9.064,
     * 8.949,8.991,8.989,8.975,8.954,8.933,8.934,8.971,
     * 8.882,8.939,8.953,8.952,8.943,8.928,8.930,8.960,
     * 8.741,8.825,8.870,8.897,8.912,8.910,8.918,8.945,
     * 8.369,8.509,8.614,8.706,8.785,8.823,8.861,8.893,
     * 8.072,8.241,8.378,8.515,8.635,8.710,8.777,8.823,
     * 0.000,7.789,7.913,8.079,8.230,8.378,8.505,8.595,
     * 0.000,7.628,7.740,7.824,7.883,7.904,7.884,7.893,
     * 0.000,7.512,7.680,7.798,7.886,7.926,7.914,7.905,
     * 0.000,6.179,6.487,6.747,6.976,7.217,7.408,7.631,
     * 0.000,0.000,0.000,6.064,6.253,6.393,6.477,7.061,
     * 0.000,0.000,0.000,6.019,6.207,6.347,6.431,7.015/
      DATA ((EDEN(I,3,12,L,3),I=17,24),L=1,NZEN)/
     * 9.149,9.176,9.158,9.155,9.149,9.134,9.152,9.217,
     * 9.079,9.111,9.104,9.083,9.061,9.053,9.072,9.127,
     * 9.017,9.061,9.072,9.064,9.053,9.049,9.068,9.117,
     * 8.882,8.956,8.995,9.017,9.029,9.037,9.061,9.104,
     * 8.519,8.655,8.763,8.847,8.922,8.964,9.013,9.061,
     * 8.225,8.393,8.535,8.667,8.788,8.865,8.943,9.000,
     * 0.000,7.914,8.053,8.238,8.408,8.559,8.699,8.799,
     * 0.000,7.717,7.825,7.912,7.975,8.013,8.017,8.057,
     * 7.453,7.613,7.772,7.890,7.982,8.033,8.041,8.057,
     * 0.000,6.316,6.617,6.872,7.100,7.348,7.553,7.794,
     * 0.000,0.000,0.000,6.190,6.377,6.522,6.624,7.217,
     * 0.000,0.000,0.000,6.145,6.331,6.477,6.579,7.172/
      DATA ((EDEN(I,4,12,L,3),I=17,24),L=1,NZEN)/
     * 9.458,9.447,9.401,9.393,9.386,9.358,9.369,9.425,
     * 9.344,9.354,9.330,9.307,9.286,9.281,9.314,9.371,
     * 9.294,9.318,9.307,9.294,9.281,9.281,9.314,9.365,
     * 9.188,9.238,9.255,9.262,9.267,9.276,9.312,9.360,
     * 8.876,8.991,9.083,9.140,9.193,9.233,9.288,9.338,
     * 8.597,8.760,8.901,9.004,9.097,9.161,9.238,9.297,
     * 0.000,8.260,8.439,8.626,8.803,8.930,9.053,9.143,
     * 7.815,7.922,8.029,8.124,8.199,8.246,8.267,8.362,
     * 7.702,7.843,7.989,8.107,8.204,8.258,8.274,8.324,
     * 6.336,6.619,6.905,7.155,7.384,7.627,7.835,8.083,
     * 0.000,0.000,6.267,6.476,6.660,6.806,6.914,7.500,
     * 0.000,0.000,6.221,6.430,6.614,6.760,6.869,7.454/
      DATA ((EDEN(I,5,12,L,3),I=17,24),L=1,NZEN)/
     * 9.759,9.713,9.634,9.611,9.591,9.556,9.569,9.622,
     * 9.560,9.558,9.524,9.497,9.474,9.471,9.513,9.575,
     * 9.519,9.526,9.505,9.487,9.473,9.473,9.516,9.574,
     * 9.428,9.458,9.462,9.464,9.465,9.476,9.522,9.576,
     * 9.164,9.250,9.318,9.365,9.410,9.449,9.512,9.569,
     * 8.916,9.049,9.167,9.250,9.330,9.393,9.474,9.539,
     * 8.373,8.574,8.764,8.928,9.083,9.201,9.326,9.415,
     * 8.021,8.121,8.223,8.320,8.400,8.461,8.504,8.646,
     * 7.919,8.049,8.188,8.305,8.401,8.458,8.489,8.573,
     * 6.593,6.863,7.143,7.391,7.619,7.866,8.086,8.348,
     * 6.061,6.276,6.501,6.711,6.899,7.053,7.179,7.763,
     * 6.015,6.231,6.455,6.665,6.853,7.007,7.133,7.718/
      DATA ((EDEN(I,1,1,L,1),I=25,32),L=1,NZEN)/
     * 8.758,8.845,9.000,9.182,9.362,9.549,9.724,9.892,
     * 8.576,8.708,8.906,9.130,9.354,9.545,9.723,9.887,
     * 8.545,8.679,8.872,9.083,9.292,9.461,9.616,9.777,
     * 8.491,8.634,8.825,9.033,9.246,9.387,9.517,9.653,
     * 8.389,8.553,8.746,8.966,9.193,9.344,9.487,9.612,
     * 8.301,8.489,8.694,8.924,9.161,9.324,9.479,9.611,
     * 8.072,8.316,8.561,8.820,9.093,9.274,9.450,9.599,
     * 7.283,7.618,7.913,8.290,8.692,8.950,9.204,9.412,
     * 7.265,7.562,7.806,8.143,8.507,8.730,8.950,9.146,
     * 7.238,7.531,7.768,8.107,8.473,8.695,8.914,9.107,
     * 7.127,7.394,7.602,7.940,8.305,8.517,8.725,8.901,
     * 7.081,7.349,7.556,7.894,8.260,8.471,8.679,8.856/
      DATA ((EDEN(I,2,1,L,1),I=25,32),L=1,NZEN)/
     * 8.866,8.983,9.158,9.348,9.525,9.696,9.856,10.009,
     * 8.710,8.860,9.064,9.281,9.481,9.648,9.802,9.949,
     * 8.679,8.831,9.029,9.233,9.422,9.565,9.696,9.839,
     * 8.628,8.790,8.988,9.188,9.377,9.494,9.601,9.719,
     * 8.533,8.716,8.917,9.127,9.328,9.456,9.575,9.679,
     * 8.447,8.654,8.867,9.086,9.301,9.439,9.568,9.679,
     * 8.225,8.490,8.742,8.988,9.236,9.393,9.543,9.669,
     * 7.415,7.783,8.100,8.464,8.838,9.068,9.294,9.483,
     * 7.389,7.719,7.980,8.301,8.637,8.833,9.025,9.204,
     * 7.360,7.687,7.943,8.265,8.604,8.799,8.990,9.164,
     * 7.243,7.543,7.772,8.097,8.436,8.620,8.801,8.960,
     * 7.197,7.497,7.727,8.051,8.390,8.574,8.756,8.914/
      DATA ((EDEN(I,3,1,L,1),I=25,32),L=1,NZEN)/
     * 9.013,9.143,9.316,9.497,9.649,9.795,9.928,10.061,
     * 8.885,9.049,9.258,9.461,9.638,9.781,9.911,10.041,
     * 8.856,9.021,9.225,9.410,9.569,9.691,9.801,9.926,
     * 8.810,8.987,9.188,9.367,9.521,9.612,9.691,9.794,
     * 8.720,8.921,9.130,9.314,9.480,9.576,9.662,9.747,
     * 8.649,8.872,9.090,9.281,9.458,9.562,9.657,9.749,
     * 8.447,8.728,8.984,9.196,9.398,9.521,9.635,9.742,
     * 7.598,8.025,8.389,8.708,9.025,9.217,9.405,9.571,
     * 7.553,7.929,8.228,8.505,8.787,8.945,9.100,9.255,
     * 7.521,7.897,8.190,8.471,8.755,8.911,9.064,9.217,
     * 7.391,7.744,8.017,8.301,8.591,8.736,8.877,9.013,
     * 7.345,7.699,7.971,8.255,8.545,8.690,8.832,8.967/
      DATA ((EDEN(I,4,1,L,1),I=25,32),L=1,NZEN)/
     * 9.196,9.324,9.483,9.644,9.765,9.876,9.973,10.083,
     * 9.114,9.281,9.479,9.671,9.824,9.943,10.049,10.158,
     * 9.086,9.250,9.438,9.607,9.740,9.842,9.932,10.041,
     * 9.049,9.223,9.407,9.560,9.677,9.743,9.797,9.889,
     * 8.977,9.173,9.364,9.519,9.643,9.704,9.755,9.819,
     * 8.919,9.137,9.336,9.494,9.625,9.695,9.754,9.823,
     * 8.756,9.021,9.255,9.430,9.580,9.664,9.740,9.822,
     * 7.889,8.356,8.747,9.009,9.253,9.400,9.542,9.679,
     * 7.789,8.204,8.529,8.744,8.950,9.064,9.173,9.307,
     * 7.754,8.170,8.494,8.710,8.919,9.029,9.140,9.270,
     * 7.602,8.009,8.322,8.547,8.762,8.862,8.957,9.072,
     * 7.556,7.963,8.276,8.501,8.716,8.816,8.911,9.026/
      DATA ((EDEN(I,5,1,L,1),I=25,32),L=1,NZEN)/
     * 9.394,9.526,9.679,9.829,9.925,10.004,10.064,10.152,
     * 9.338,9.511,9.700,9.881,10.013,10.100,10.176,10.272,
     * 9.316,9.481,9.659,9.816,9.924,10.000,10.064,10.152,
     * 9.290,9.461,9.633,9.768,9.855,9.894,9.921,9.993,
     * 9.236,9.433,9.612,9.744,9.834,9.857,9.870,9.907,
     * 9.188,9.405,9.593,9.729,9.825,9.854,9.873,9.914,
     * 9.057,9.318,9.533,9.679,9.792,9.834,9.867,9.919,
     * 8.223,8.699,9.090,9.305,9.500,9.600,9.695,9.797,
     * 8.064,8.494,8.821,8.987,9.134,9.201,9.265,9.371,
     * 8.021,8.456,8.787,8.954,9.100,9.170,9.230,9.334,
     * 7.844,8.281,8.613,8.786,8.938,8.996,9.049,9.137,
     * 7.799,8.235,8.567,8.740,8.892,8.950,9.003,9.091/
      DATA ((EDEN(I,1,2,L,1),I=25,32),L=1,NZEN)/
     * 8.733,8.828,8.992,9.179,9.365,9.556,9.735,9.904,
     * 8.576,8.710,8.909,9.137,9.362,9.555,9.737,9.903,
     * 8.543,8.679,8.874,9.086,9.297,9.467,9.625,9.789,
     * 8.491,8.634,8.827,9.037,9.248,9.389,9.520,9.659,
     * 8.389,8.554,8.748,8.968,9.193,9.346,9.487,9.612,
     * 8.301,8.490,8.697,8.926,9.164,9.326,9.480,9.611,
     * 8.072,8.318,8.566,8.823,9.093,9.279,9.453,9.601,
     * 7.283,7.619,7.915,8.294,8.699,8.955,9.207,9.415,
     * 7.265,7.562,7.806,8.143,8.507,8.728,8.948,9.143,
     * 7.238,7.531,7.770,8.107,8.473,8.694,8.912,9.104,
     * 7.124,7.394,7.604,7.942,8.307,8.517,8.725,8.900,
     * 7.078,7.349,7.558,7.896,8.262,8.471,8.679,8.854/
      DATA ((EDEN(I,2,2,L,1),I=25,32),L=1,NZEN)/
     * 8.848,8.967,9.143,9.334,9.511,9.683,9.843,9.997,
     * 8.698,8.851,9.057,9.276,9.481,9.650,9.808,9.959,
     * 8.667,8.822,9.025,9.225,9.417,9.563,9.700,9.846,
     * 8.616,8.779,8.979,9.179,9.367,9.486,9.593,9.716,
     * 8.517,8.701,8.903,9.114,9.316,9.444,9.562,9.668,
     * 8.430,8.638,8.854,9.072,9.288,9.425,9.555,9.668,
     * 8.207,8.474,8.729,8.974,9.220,9.378,9.530,9.659,
     * 7.405,7.770,8.086,8.450,8.828,9.061,9.288,9.477,
     * 7.382,7.708,7.967,8.288,8.627,8.823,9.017,9.193,
     * 7.354,7.677,7.930,8.253,8.593,8.787,8.980,9.155,
     * 7.241,7.535,7.760,8.086,8.428,8.612,8.793,8.951,
     * 7.195,7.490,7.714,8.041,8.382,8.566,8.747,8.906/
      DATA ((EDEN(I,3,2,L,1),I=25,32),L=1,NZEN)/
     * 8.997,9.127,9.301,9.477,9.627,9.771,9.903,10.037,
     * 8.857,9.025,9.238,9.446,9.629,9.777,9.913,10.045,
     * 8.824,8.995,9.201,9.387,9.553,9.680,9.797,9.928,
     * 8.775,8.954,9.158,9.338,9.498,9.591,9.674,9.785,
     * 8.679,8.882,9.093,9.279,9.452,9.548,9.636,9.725,
     * 8.603,8.828,9.049,9.246,9.427,9.533,9.631,9.727,
     * 8.400,8.680,8.939,9.158,9.367,9.491,9.610,9.719,
     * 7.565,7.980,8.332,8.664,8.997,9.190,9.382,9.553,
     * 7.526,7.890,8.176,8.465,8.760,8.921,9.079,9.238,
     * 7.498,7.860,8.143,8.431,8.727,8.886,9.045,9.201,
     * 7.380,7.716,7.971,8.265,8.566,8.713,8.859,9.000,
     * 7.334,7.670,7.926,8.219,8.520,8.667,8.813,8.954/
      DATA ((EDEN(I,4,2,L,1),I=25,32),L=1,NZEN)/
     * 9.173,9.299,9.459,9.615,9.735,9.848,9.950,10.064,
     * 9.064,9.238,9.444,9.640,9.803,9.927,10.041,10.158,
     * 9.033,9.201,9.394,9.568,9.708,9.819,9.920,10.033,
     * 8.985,9.164,9.354,9.509,9.631,9.704,9.767,9.869,
     * 8.903,9.104,9.299,9.456,9.588,9.653,9.711,9.780,
     * 8.841,9.061,9.265,9.427,9.566,9.640,9.708,9.782,
     * 8.669,8.940,9.176,9.356,9.517,9.607,9.692,9.780,
     * 7.810,8.267,8.650,8.925,9.190,9.344,9.497,9.640,
     * 7.726,8.127,8.435,8.668,8.897,9.021,9.140,9.279,
     * 7.695,8.093,8.401,8.635,8.866,8.987,9.107,9.243,
     * 7.562,7.946,8.236,8.477,8.713,8.821,8.926,9.045,
     * 7.517,7.900,8.190,8.431,8.668,8.775,8.881,9.000/
      DATA ((EDEN(I,5,2,L,1),I=25,32),L=1,NZEN)/
     * 9.348,9.479,9.631,9.775,9.873,9.957,10.029,10.124,
     * 9.253,9.436,9.638,9.825,9.968,10.064,10.155,10.258,
     * 9.223,9.396,9.583,9.746,9.865,9.953,10.033,10.130,
     * 9.182,9.362,9.543,9.680,9.775,9.824,9.865,9.951,
     * 9.114,9.314,9.498,9.635,9.735,9.765,9.787,9.836,
     * 9.061,9.281,9.473,9.612,9.717,9.755,9.786,9.838,
     * 8.913,9.179,9.400,9.553,9.677,9.728,9.774,9.838,
     * 8.061,8.537,8.926,9.161,9.377,9.490,9.600,9.715,
     * 7.935,8.354,8.671,8.864,9.041,9.124,9.207,9.326,
     * 7.901,8.322,8.639,8.833,9.009,9.093,9.173,9.288,
     * 7.753,8.167,8.477,8.675,8.859,8.929,8.998,9.097,
     * 7.707,8.122,8.431,8.629,8.813,8.884,8.953,9.051/
      DATA ((EDEN(I,1,3,L,1),I=25,32),L=1,NZEN)/
     * 8.702,8.806,8.979,9.176,9.369,9.567,9.752,9.922,
     * 8.576,8.709,8.908,9.137,9.362,9.556,9.739,9.904,
     * 8.545,8.679,8.873,9.086,9.294,9.467,9.627,9.790,
     * 8.494,8.635,8.825,9.033,9.246,9.387,9.519,9.659,
     * 8.389,8.553,8.746,8.966,9.193,9.344,9.486,9.610,
     * 8.303,8.491,8.697,8.925,9.161,9.324,9.477,9.610,
     * 8.076,8.320,8.563,8.822,9.093,9.274,9.450,9.599,
     * 7.283,7.620,7.918,8.297,8.699,8.955,9.207,9.417,
     * 7.265,7.563,7.806,8.143,8.505,8.727,8.945,9.140,
     * 7.238,7.533,7.769,8.107,8.471,8.692,8.910,9.100,
     * 7.127,7.394,7.603,7.940,8.305,8.516,8.723,8.897,
     * 7.081,7.349,7.557,7.894,8.260,8.470,8.677,8.851/
      DATA ((EDEN(I,2,3,L,1),I=25,32),L=1,NZEN)/
     * 8.827,8.947,9.124,9.310,9.483,9.655,9.817,9.975,
     * 8.692,8.844,9.049,9.265,9.471,9.644,9.807,9.958,
     * 8.659,8.812,9.013,9.212,9.401,9.551,9.691,9.841,
     * 8.605,8.767,8.966,9.161,9.350,9.467,9.577,9.705,
     * 8.501,8.685,8.886,9.090,9.294,9.422,9.542,9.650,
     * 8.417,8.622,8.835,9.049,9.267,9.403,9.533,9.649,
     * 8.193,8.455,8.706,8.949,9.199,9.354,9.508,9.639,
     * 7.400,7.758,8.068,8.430,8.809,9.041,9.270,9.462,
     * 7.377,7.697,7.949,8.270,8.606,8.803,8.998,9.179,
     * 7.350,7.666,7.913,8.233,8.573,8.769,8.963,9.140,
     * 7.243,7.531,7.749,8.068,8.410,8.593,8.777,8.938,
     * 7.197,7.486,7.703,8.022,8.364,8.548,8.731,8.892/
      DATA ((EDEN(I,3,3,L,1),I=25,32),L=1,NZEN)/
     * 8.977,9.104,9.274,9.439,9.585,9.729,9.866,10.004,
     * 8.818,8.988,9.199,9.407,9.594,9.745,9.889,10.029,
     * 8.783,8.953,9.158,9.344,9.515,9.645,9.770,9.908,
     * 8.725,8.904,9.111,9.290,9.455,9.549,9.637,9.757,
     * 8.622,8.823,9.029,9.220,9.401,9.500,9.593,9.688,
     * 8.538,8.762,8.982,9.179,9.371,9.480,9.584,9.688,
     * 8.324,8.604,8.863,9.086,9.305,9.436,9.562,9.679,
     * 7.525,7.912,8.236,8.581,8.932,9.134,9.336,9.515,
     * 7.498,7.837,8.097,8.396,8.708,8.877,9.049,9.212,
     * 7.474,7.808,8.064,8.364,8.677,8.843,9.013,9.173,
     * 7.369,7.678,7.904,8.204,8.516,8.671,8.829,8.975,
     * 7.323,7.632,7.858,8.158,8.470,8.625,8.784,8.929/
      DATA ((EDEN(I,4,3,L,1),I=25,32),L=1,NZEN)/
     * 9.134,9.262,9.423,9.568,9.687,9.804,9.915,10.041,
     * 8.960,9.146,9.362,9.560,9.732,9.862,9.989,10.117,
     * 8.922,9.104,9.310,9.484,9.634,9.751,9.865,9.989,
     * 8.866,9.053,9.255,9.415,9.553,9.629,9.703,9.816,
     * 8.763,8.971,9.173,9.342,9.493,9.566,9.636,9.716,
     * 8.686,8.913,9.124,9.301,9.462,9.545,9.628,9.714,
     * 8.487,8.768,9.013,9.212,9.401,9.504,9.606,9.708,
     * 7.679,8.090,8.425,8.738,9.049,9.225,9.401,9.559,
     * 7.637,7.991,8.255,8.526,8.801,8.943,9.090,9.238,
     * 7.613,7.964,8.223,8.494,8.770,8.910,9.053,9.201,
     * 7.513,7.839,8.072,8.342,8.618,8.745,8.875,9.004,
     * 7.467,7.793,8.026,8.297,8.572,8.699,8.829,8.959/
      DATA ((EDEN(I,5,3,L,1),I=25,32),L=1,NZEN)/
     * 9.299,9.427,9.580,9.710,9.805,9.897,9.986,10.093,
     * 9.124,9.320,9.537,9.726,9.881,9.988,10.090,10.207,
     * 9.083,9.272,9.471,9.636,9.768,9.867,9.963,10.076,
     * 9.025,9.217,9.413,9.554,9.663,9.721,9.777,9.882,
     * 8.931,9.143,9.336,9.481,9.601,9.639,9.678,9.741,
     * 8.861,9.093,9.297,9.446,9.572,9.619,9.667,9.736,
     * 8.686,8.965,9.199,9.367,9.516,9.582,9.648,9.729,
     * 7.869,8.303,8.652,8.927,9.196,9.328,9.465,9.600,
     * 7.796,8.164,8.430,8.668,8.904,9.013,9.127,9.262,
     * 7.772,8.137,8.398,8.637,8.873,8.980,9.093,9.225,
     * 7.668,8.013,8.248,8.489,8.726,8.821,8.922,9.033,
     * 7.623,7.967,8.202,8.443,8.680,8.775,8.876,8.988/
      DATA ((EDEN(I,1,4,L,1),I=25,32),L=1,NZEN)/
     * 8.684,8.794,8.974,9.176,9.378,9.579,9.768,9.940,
     * 8.592,8.721,8.914,9.140,9.364,9.559,9.741,9.906,
     * 8.558,8.688,8.879,9.086,9.294,9.468,9.628,9.790,
     * 8.507,8.645,8.832,9.037,9.246,9.386,9.517,9.658,
     * 8.405,8.565,8.754,8.970,9.193,9.342,9.483,9.606,
     * 8.324,8.505,8.705,8.930,9.164,9.324,9.476,9.606,
     * 8.104,8.340,8.579,8.831,9.097,9.276,9.450,9.598,
     * 7.297,7.640,7.944,8.316,8.711,8.964,9.212,9.420,
     * 7.272,7.574,7.822,8.149,8.505,8.723,8.940,9.134,
     * 7.248,7.544,7.783,8.114,8.473,8.690,8.904,9.093,
     * 7.134,7.407,7.619,7.951,8.312,8.517,8.719,8.893,
     * 7.088,7.361,7.573,7.905,8.266,8.471,8.673,8.847/
      DATA ((EDEN(I,2,4,L,1),I=25,32),L=1,NZEN)/
     * 8.816,8.932,9.107,9.283,9.452,9.622,9.786,9.947,
     * 8.693,8.840,9.041,9.250,9.453,9.626,9.792,9.947,
     * 8.658,8.806,9.000,9.193,9.380,9.531,9.677,9.829,
     * 8.601,8.757,8.952,9.140,9.324,9.441,9.554,9.687,
     * 8.496,8.673,8.869,9.068,9.267,9.393,9.513,9.626,
     * 8.410,8.611,8.818,9.025,9.236,9.373,9.505,9.625,
     * 8.188,8.446,8.691,8.926,9.170,9.326,9.479,9.615,
     * 7.401,7.754,8.057,8.412,8.786,9.017,9.246,9.442,
     * 7.380,7.691,7.934,8.246,8.582,8.777,8.975,9.158,
     * 7.356,7.661,7.898,8.212,8.549,8.743,8.939,9.121,
     * 7.253,7.531,7.740,8.053,8.387,8.571,8.756,8.920,
     * 7.207,7.486,7.694,8.007,8.342,8.525,8.710,8.874/
      DATA ((EDEN(I,3,4,L,1),I=25,32),L=1,NZEN)/
     * 8.945,9.068,9.241,9.394,9.537,9.684,9.829,9.976,
     * 8.778,8.944,9.152,9.354,9.543,9.698,9.851,9.998,
     * 8.738,8.905,9.107,9.290,9.461,9.595,9.728,9.875,
     * 8.678,8.852,9.053,9.228,9.396,9.493,9.590,9.719,
     * 8.561,8.756,8.958,9.146,9.334,9.436,9.539,9.642,
     * 8.470,8.688,8.903,9.100,9.301,9.413,9.528,9.639,
     * 8.243,8.516,8.766,8.996,9.230,9.365,9.502,9.629,
     * 7.496,7.841,8.124,8.479,8.846,9.057,9.274,9.462,
     * 7.481,7.784,8.009,8.316,8.640,8.819,9.004,9.179,
     * 7.459,7.758,7.977,8.281,8.606,8.785,8.970,9.140,
     * 7.371,7.641,7.831,8.130,8.447,8.614,8.787,8.941,
     * 7.325,7.596,7.785,8.085,8.401,8.568,8.742,8.895/
      DATA ((EDEN(I,4,4,L,1),I=25,32),L=1,NZEN)/
     * 9.053,9.193,9.367,9.511,9.636,9.765,9.896,10.037,
     * 8.849,9.033,9.250,9.447,9.631,9.772,9.914,10.057,
     * 8.804,8.987,9.196,9.369,9.531,9.656,9.784,9.926,
     * 8.736,8.924,9.127,9.292,9.449,9.531,9.619,9.747,
     * 8.612,8.814,9.013,9.193,9.369,9.455,9.545,9.639,
     * 8.517,8.740,8.947,9.140,9.326,9.427,9.530,9.632,
     * 8.286,8.559,8.799,9.021,9.248,9.371,9.502,9.619,
     * 7.592,7.911,8.155,8.504,8.862,9.068,9.283,9.461,
     * 7.585,7.866,8.057,8.350,8.659,8.835,9.021,9.188,
     * 7.568,7.843,8.025,8.318,8.626,8.801,8.988,9.149,
     * 7.496,7.745,7.901,8.176,8.467,8.632,8.810,8.953,
     * 7.450,7.699,7.856,8.130,8.421,8.587,8.764,8.908/
      DATA ((EDEN(I,5,4,L,1),I=25,32),L=1,NZEN)/
     * 9.143,9.305,9.491,9.622,9.735,9.847,9.966,10.100,
     * 8.926,9.134,9.360,9.554,9.731,9.856,9.988,10.121,
     * 8.873,9.072,9.288,9.458,9.612,9.727,9.849,9.984,
     * 8.794,8.998,9.207,9.362,9.504,9.572,9.649,9.778,
     * 8.650,8.867,9.068,9.243,9.407,9.465,9.535,9.619,
     * 8.550,8.784,8.993,9.176,9.356,9.428,9.512,9.604,
     * 8.314,8.595,8.832,9.049,9.267,9.365,9.476,9.584,
     * 7.692,7.993,8.210,8.540,8.877,9.068,9.270,9.438,
     * 7.691,7.953,8.114,8.386,8.667,8.840,9.029,9.188,
     * 7.678,7.936,8.090,8.356,8.632,8.806,8.995,9.149,
     * 7.623,7.856,7.985,8.223,8.474,8.639,8.821,8.957,
     * 7.577,7.810,7.940,8.177,8.428,8.594,8.775,8.911/
      DATA ((EDEN(I,1,5,L,1),I=25,32),L=1,NZEN)/
     * 8.700,8.808,8.984,9.190,9.394,9.598,9.787,9.960,
     * 8.601,8.730,8.923,9.149,9.371,9.567,9.750,9.913,
     * 8.568,8.697,8.885,9.093,9.299,9.471,9.634,9.795,
     * 8.519,8.655,8.839,9.041,9.246,9.386,9.517,9.658,
     * 8.418,8.576,8.766,8.976,9.193,9.342,9.480,9.604,
     * 8.338,8.520,8.719,8.938,9.164,9.324,9.474,9.605,
     * 8.127,8.364,8.600,8.844,9.104,9.281,9.450,9.597,
     * 7.312,7.666,7.979,8.340,8.726,8.975,9.220,9.425,
     * 7.281,7.587,7.837,8.158,8.507,8.722,8.934,9.127,
     * 7.253,7.555,7.801,8.124,8.474,8.688,8.899,9.090,
     * 7.140,7.420,7.640,7.964,8.316,8.517,8.716,8.889,
     * 7.094,7.374,7.595,7.918,8.270,8.471,8.670,8.843/
      DATA ((EDEN(I,2,5,L,1),I=25,32),L=1,NZEN)/
     * 8.829,8.936,9.100,9.267,9.430,9.599,9.764,9.928,
     * 8.695,8.838,9.033,9.238,9.438,9.613,9.782,9.941,
     * 8.655,8.801,8.993,9.179,9.362,9.515,9.663,9.820,
     * 8.599,8.751,8.943,9.121,9.301,9.420,9.535,9.672,
     * 8.491,8.665,8.857,9.049,9.246,9.367,9.490,9.605,
     * 8.405,8.600,8.804,9.004,9.212,9.348,9.481,9.604,
     * 8.182,8.435,8.677,8.904,9.143,9.299,9.455,9.593,
     * 7.405,7.751,8.049,8.396,8.766,8.994,9.225,9.423,
     * 7.384,7.685,7.919,8.225,8.559,8.754,8.954,9.140,
     * 7.358,7.656,7.885,8.193,8.528,8.721,8.919,9.104,
     * 7.260,7.531,7.733,8.037,8.371,8.551,8.738,8.903,
     * 7.214,7.486,7.687,7.992,8.325,8.506,8.692,8.857/
      DATA ((EDEN(I,3,5,L,1),I=25,32),L=1,NZEN)/
     * 8.912,9.037,9.210,9.358,9.502,9.652,9.806,9.964,
     * 8.761,8.921,9.124,9.314,9.504,9.660,9.820,9.975,
     * 8.718,8.877,9.072,9.246,9.417,9.555,9.697,9.851,
     * 8.650,8.818,9.013,9.179,9.348,9.447,9.550,9.688,
     * 8.526,8.713,8.907,9.090,9.281,9.384,9.491,9.603,
     * 8.431,8.640,8.846,9.041,9.243,9.358,9.480,9.599,
     * 8.199,8.461,8.702,8.930,9.170,9.307,9.453,9.588,
     * 7.489,7.804,8.061,8.410,8.784,8.999,9.225,9.420,
     * 7.480,7.756,7.955,8.255,8.583,8.769,8.967,9.146,
     * 7.461,7.731,7.923,8.223,8.549,8.734,8.931,9.111,
     * 7.382,7.624,7.786,8.076,8.389,8.563,8.750,8.910,
     * 7.336,7.579,7.740,8.030,8.343,8.518,8.704,8.864/
      DATA ((EDEN(I,4,5,L,1),I=25,32),L=1,NZEN)/
     * 8.940,9.104,9.301,9.456,9.606,9.754,9.911,10.072,
     * 8.797,8.971,9.176,9.367,9.554,9.702,9.862,10.013,
     * 8.749,8.920,9.117,9.286,9.452,9.583,9.726,9.880,
     * 8.670,8.846,9.037,9.199,9.362,9.452,9.553,9.695,
     * 8.528,8.716,8.901,9.079,9.267,9.362,9.471,9.576,
     * 8.420,8.627,8.821,9.013,9.215,9.326,9.452,9.567,
     * 8.170,8.425,8.647,8.880,9.124,9.265,9.418,9.550,
     * 7.573,7.830,8.013,8.352,8.713,8.945,9.196,9.389,
     * 7.579,7.808,7.949,8.225,8.531,8.737,8.962,9.140,
     * 7.563,7.788,7.923,8.196,8.494,8.702,8.927,9.104,
     * 7.505,7.705,7.814,8.057,8.328,8.528,8.748,8.905,
     * 7.459,7.659,7.768,8.011,8.283,8.482,8.702,8.859/
      DATA ((EDEN(I,5,5,L,1),I=25,32),L=1,NZEN)/
     * 8.960,9.158,9.382,9.542,9.697,9.837,9.995,10.161,
     * 8.822,9.013,9.223,9.413,9.602,9.742,9.899,10.049,
     * 8.765,8.950,9.152,9.318,9.484,9.610,9.754,9.909,
     * 8.668,8.859,9.057,9.212,9.373,9.450,9.548,9.696,
     * 8.491,8.692,8.880,9.057,9.243,9.322,9.423,9.525,
     * 8.364,8.583,8.777,8.969,9.170,9.270,9.389,9.501,
     * 8.097,8.352,8.567,8.799,9.045,9.182,9.340,9.471,
     * 7.667,7.872,7.995,8.281,8.589,8.843,9.121,9.314,
     * 7.685,7.878,7.972,8.190,8.435,8.676,8.943,9.124,
     * 7.676,7.865,7.953,8.158,8.394,8.637,8.907,9.086,
     * 7.637,7.804,7.869,8.029,8.220,8.461,8.728,8.888,
     * 7.592,7.758,7.823,7.984,8.174,8.415,8.682,8.842/
      DATA ((EDEN(I,1,6,L,1),I=25,32),L=1,NZEN)/
     * 8.725,8.829,9.000,9.207,9.412,9.615,9.806,9.980,
     * 8.610,8.739,8.933,9.158,9.382,9.577,9.762,9.926,
     * 8.575,8.705,8.894,9.100,9.305,9.481,9.645,9.807,
     * 8.525,8.662,8.848,9.045,9.246,9.389,9.520,9.664,
     * 8.430,8.588,8.775,8.982,9.196,9.344,9.481,9.604,
     * 8.350,8.530,8.729,8.945,9.170,9.326,9.474,9.604,
     * 8.146,8.380,8.616,8.855,9.107,9.283,9.452,9.597,
     * 7.324,7.687,8.009,8.364,8.742,8.988,9.230,9.431,
     * 7.286,7.597,7.852,8.167,8.509,8.721,8.931,9.121,
     * 7.258,7.566,7.817,8.134,8.477,8.688,8.895,9.086,
     * 7.143,7.428,7.656,7.975,8.322,8.519,8.713,8.885,
     * 7.097,7.382,7.610,7.929,8.276,8.473,8.668,8.840/
      DATA ((EDEN(I,2,6,L,1),I=25,32),L=1,NZEN)/
     * 8.851,8.949,9.104,9.265,9.423,9.594,9.763,9.926,
     * 8.713,8.852,9.045,9.243,9.441,9.617,9.790,9.949,
     * 8.677,8.815,9.000,9.179,9.358,9.513,9.668,9.827,
     * 8.619,8.765,8.949,9.117,9.292,9.412,9.529,9.671,
     * 8.509,8.678,8.865,9.045,9.236,9.356,9.479,9.595,
     * 8.427,8.616,8.814,9.004,9.204,9.336,9.470,9.594,
     * 8.212,8.458,8.693,8.908,9.140,9.292,9.447,9.585,
     * 7.427,7.777,8.079,8.412,8.772,8.996,9.225,9.423,
     * 7.396,7.698,7.933,8.225,8.549,8.740,8.939,9.124,
     * 7.371,7.670,7.900,8.193,8.517,8.707,8.904,9.086,
     * 7.274,7.545,7.747,8.041,8.364,8.540,8.724,8.890,
     * 7.228,7.500,7.702,7.996,8.318,8.495,8.679,8.844/
      DATA ((EDEN(I,3,6,L,1),I=25,32),L=1,NZEN)/
     * 8.899,9.021,9.196,9.340,9.489,9.645,9.810,9.974,
     * 8.779,8.930,9.124,9.307,9.493,9.651,9.816,9.976,
     * 8.736,8.884,9.072,9.233,9.400,9.542,9.691,9.849,
     * 8.663,8.821,9.009,9.161,9.324,9.423,9.533,9.678,
     * 8.540,8.716,8.900,9.072,9.253,9.356,9.467,9.582,
     * 8.447,8.644,8.839,9.021,9.217,9.330,9.455,9.577,
     * 8.220,8.468,8.698,8.912,9.146,9.281,9.428,9.567,
     * 7.507,7.815,8.061,8.401,8.769,8.981,9.210,9.407,
     * 7.496,7.759,7.945,8.233,8.554,8.738,8.939,9.124,
     * 7.476,7.735,7.915,8.201,8.521,8.704,8.904,9.086,
     * 7.398,7.629,7.781,8.053,8.362,8.535,8.725,8.888,
     * 7.352,7.584,7.735,8.007,8.316,8.490,8.679,8.842/
      DATA ((EDEN(I,4,6,L,1),I=25,32),L=1,NZEN)/
     * 8.892,9.061,9.270,9.436,9.607,9.766,9.939,10.121,
     * 8.792,8.953,9.149,9.328,9.515,9.667,9.836,9.995,
     * 8.739,8.898,9.083,9.243,9.408,9.545,9.699,9.862,
     * 8.657,8.821,9.000,9.152,9.314,9.407,9.519,9.670,
     * 8.507,8.682,8.855,9.025,9.210,9.310,9.428,9.543,
     * 8.396,8.589,8.769,8.953,9.155,9.272,9.408,9.530,
     * 8.140,8.378,8.588,8.812,9.061,9.207,9.373,9.513,
     * 7.582,7.808,7.963,8.286,8.638,8.883,9.152,9.352,
     * 7.590,7.792,7.908,8.164,8.456,8.677,8.922,9.107,
     * 7.577,7.775,7.883,8.134,8.420,8.640,8.887,9.072,
     * 7.521,7.695,7.780,7.997,8.253,8.467,8.707,8.872,
     * 7.475,7.650,7.734,7.952,8.207,8.421,8.661,8.826/
      DATA ((EDEN(I,5,6,L,1),I=25,32),L=1,NZEN)/
     * 8.886,9.100,9.340,9.520,9.708,9.864,10.045,10.243,
     * 8.782,8.953,9.146,9.330,9.524,9.673,9.847,10.009,
     * 8.723,8.892,9.076,9.236,9.407,9.540,9.700,9.868,
     * 8.620,8.795,8.976,9.127,9.294,9.378,9.490,9.652,
     * 8.427,8.612,8.784,8.957,9.149,9.241,9.360,9.474,
     * 8.288,8.491,8.670,8.855,9.061,9.176,9.320,9.447,
     * 8.021,8.248,8.435,8.662,8.915,9.076,9.265,9.410,
     * 7.682,7.842,7.920,8.155,8.425,8.716,9.041,9.250,
     * 7.704,7.860,7.919,8.090,8.301,8.577,8.888,9.083,
     * 7.696,7.849,7.902,8.061,8.260,8.538,8.849,9.041,
     * 7.666,7.796,7.825,7.933,8.083,8.356,8.663,8.841,
     * 7.620,7.750,7.779,7.888,8.037,8.310,8.617,8.795/
      DATA ((EDEN(I,1,7,L,1),I=25,32),L=1,NZEN)/
     * 8.735,8.838,9.009,9.217,9.422,9.623,9.813,9.989,
     * 8.603,8.736,8.933,9.158,9.384,9.582,9.766,9.932,
     * 8.571,8.702,8.892,9.100,9.307,9.484,9.649,9.812,
     * 8.519,8.659,8.848,9.045,9.248,9.391,9.522,9.667,
     * 8.422,8.582,8.772,8.981,9.196,9.344,9.483,9.605,
     * 8.342,8.526,8.729,8.945,9.170,9.326,9.476,9.605,
     * 8.137,8.373,8.612,8.853,9.107,9.286,9.453,9.599,
     * 7.318,7.679,8.004,8.360,8.741,8.988,9.230,9.433,
     * 7.281,7.592,7.848,8.164,8.509,8.721,8.931,9.121,
     * 7.253,7.561,7.813,8.130,8.477,8.688,8.896,9.086,
     * 7.137,7.425,7.651,7.971,8.320,8.519,8.713,8.885,
     * 7.091,7.379,7.606,7.926,8.274,8.473,8.668,8.840/
      DATA ((EDEN(I,2,7,L,1),I=25,32),L=1,NZEN)/
     * 8.867,8.960,9.111,9.270,9.430,9.600,9.768,9.934,
     * 8.730,8.865,9.053,9.253,9.450,9.625,9.799,9.959,
     * 8.691,8.825,9.009,9.185,9.362,9.520,9.678,9.836,
     * 8.635,8.777,8.958,9.124,9.292,9.413,9.531,9.676,
     * 8.529,8.692,8.875,9.049,9.236,9.358,9.480,9.597,
     * 8.449,8.633,8.827,9.009,9.207,9.338,9.471,9.595,
     * 8.241,8.480,8.710,8.919,9.143,9.294,9.449,9.589,
     * 7.442,7.797,8.100,8.428,8.782,9.004,9.233,9.430,
     * 7.408,7.712,7.949,8.233,8.550,8.740,8.936,9.121,
     * 7.384,7.681,7.911,8.199,8.519,8.706,8.900,9.086,
     * 7.281,7.555,7.760,8.045,8.364,8.539,8.723,8.888,
     * 7.235,7.509,7.714,8.000,8.318,8.493,8.677,8.842/
      DATA ((EDEN(I,3,7,L,1),I=25,32),L=1,NZEN)/
     * 8.918,9.037,9.204,9.348,9.496,9.654,9.820,9.985,
     * 8.818,8.960,9.146,9.324,9.508,9.665,9.830,9.989,
     * 8.772,8.912,9.090,9.246,9.408,9.551,9.703,9.862,
     * 8.706,8.854,9.029,9.173,9.326,9.428,9.540,9.687,
     * 8.584,8.751,8.927,9.086,9.260,9.360,9.470,9.584,
     * 8.497,8.684,8.869,9.037,9.225,9.336,9.458,9.581,
     * 8.279,8.519,8.738,8.938,9.158,9.290,9.435,9.572,
     * 7.539,7.859,8.117,8.441,8.794,9.000,9.225,9.420,
     * 7.516,7.784,7.975,8.250,8.560,8.737,8.932,9.114,
     * 7.498,7.760,7.943,8.217,8.526,8.703,8.898,9.079,
     * 7.415,7.650,7.806,8.072,8.371,8.538,8.720,8.883,
     * 7.369,7.605,7.760,8.026,8.325,8.492,8.674,8.837/
      DATA ((EDEN(I,4,7,L,1),I=25,32),L=1,NZEN)/
     * 8.923,9.086,9.286,9.449,9.619,9.776,9.951,10.134,
     * 8.833,8.988,9.176,9.348,9.529,9.679,9.846,10.009,
     * 8.784,8.933,9.111,9.258,9.417,9.555,9.712,9.873,
     * 8.705,8.859,9.029,9.167,9.316,9.412,9.525,9.679,
     * 8.562,8.727,8.890,9.045,9.220,9.316,9.431,9.547,
     * 8.458,8.640,8.811,8.980,9.170,9.281,9.413,9.535,
     * 8.212,8.444,8.642,8.849,9.083,9.220,9.382,9.520,
     * 7.609,7.844,8.009,8.324,8.675,8.907,9.164,9.364,
     * 7.612,7.815,7.931,8.185,8.474,8.681,8.915,9.100,
     * 7.597,7.796,7.905,8.152,8.436,8.644,8.880,9.064,
     * 7.538,7.713,7.797,8.013,8.272,8.473,8.702,8.866,
     * 7.492,7.667,7.752,7.967,8.226,8.427,8.657,8.821/
      DATA ((EDEN(I,5,7,L,1),I=25,32),L=1,NZEN)/
     * 8.928,9.134,9.369,9.539,9.721,9.877,10.061,10.258,
     * 8.823,8.985,9.170,9.342,9.529,9.674,9.847,10.013,
     * 8.765,8.925,9.100,9.248,9.408,9.543,9.704,9.873,
     * 8.669,8.833,9.004,9.140,9.294,9.380,9.494,9.659,
     * 8.486,8.661,8.823,8.979,9.155,9.246,9.365,9.480,
     * 8.354,8.547,8.716,8.885,9.079,9.190,9.330,9.455,
     * 8.090,8.314,8.494,8.707,8.947,9.097,9.276,9.422,
     * 7.705,7.866,7.945,8.185,8.462,8.741,9.053,9.262,
     * 7.725,7.878,7.933,8.100,8.314,8.580,8.881,9.076,
     * 7.716,7.865,7.914,8.072,8.276,8.542,8.844,9.037,
     * 7.679,7.806,7.833,7.943,8.097,8.360,8.660,8.835,
     * 7.634,7.760,7.787,7.898,8.051,8.314,8.614,8.789/
      DATA ((EDEN(I,1,8,L,1),I=25,32),L=1,NZEN)/
     * 8.722,8.827,9.000,9.210,9.417,9.619,9.810,9.985,
     * 8.590,8.723,8.922,9.149,9.377,9.573,9.758,9.922,
     * 8.555,8.690,8.884,9.093,9.301,9.479,9.642,9.806,
     * 8.504,8.646,8.838,9.041,9.246,9.389,9.520,9.664,
     * 8.405,8.568,8.761,8.975,9.196,9.344,9.483,9.605,
     * 8.324,8.509,8.713,8.936,9.167,9.326,9.476,9.605,
     * 8.107,8.352,8.595,8.842,9.104,9.281,9.452,9.598,
     * 7.301,7.653,7.967,8.332,8.723,8.974,9.220,9.427,
     * 7.272,7.579,7.829,8.155,8.507,8.723,8.935,9.127,
     * 7.243,7.549,7.797,8.121,8.474,8.688,8.900,9.090,
     * 7.130,7.412,7.631,7.959,8.314,8.516,8.716,8.889,
     * 7.085,7.366,7.586,7.913,8.268,8.470,8.670,8.843/
      DATA ((EDEN(I,2,8,L,1),I=25,32),L=1,NZEN)/
     * 8.863,8.960,9.114,9.279,9.441,9.614,9.785,9.948,
     * 8.723,8.859,9.049,9.246,9.444,9.619,9.791,9.952,
     * 8.687,8.822,9.004,9.185,9.364,9.519,9.671,9.831,
     * 8.630,8.774,8.956,9.127,9.301,9.420,9.538,9.679,
     * 8.525,8.690,8.874,9.057,9.246,9.369,9.493,9.609,
     * 8.441,8.629,8.825,9.013,9.215,9.350,9.484,9.607,
     * 8.228,8.470,8.702,8.919,9.152,9.305,9.459,9.599,
     * 7.433,7.780,8.079,8.415,8.777,9.004,9.233,9.433,
     * 7.407,7.707,7.942,8.236,8.558,8.750,8.948,9.134,
     * 7.380,7.678,7.906,8.201,8.526,8.716,8.913,9.097,
     * 7.276,7.547,7.747,8.041,8.369,8.547,8.732,8.898,
     * 7.231,7.501,7.702,7.996,8.323,8.501,8.686,8.852/
      DATA ((EDEN(I,3,8,L,1),I=25,32),L=1,NZEN)/
     * 8.961,9.072,9.228,9.369,9.513,9.666,9.825,9.985,
     * 8.834,8.975,9.161,9.336,9.516,9.673,9.836,9.992,
     * 8.796,8.933,9.111,9.265,9.423,9.563,9.709,9.865,
     * 8.731,8.878,9.053,9.199,9.348,9.449,9.556,9.697,
     * 8.619,8.785,8.960,9.117,9.288,9.386,9.494,9.605,
     * 8.534,8.720,8.904,9.072,9.255,9.365,9.484,9.603,
     * 8.318,8.560,8.779,8.975,9.190,9.320,9.461,9.594,
     * 7.554,7.885,8.155,8.474,8.823,9.029,9.248,9.439,
     * 7.531,7.809,8.009,8.283,8.588,8.763,8.952,9.134,
     * 7.509,7.783,7.978,8.250,8.556,8.729,8.918,9.097,
     * 7.418,7.665,7.831,8.097,8.398,8.560,8.739,8.898,
     * 7.373,7.619,7.785,8.051,8.352,8.514,8.693,8.852/
      DATA ((EDEN(I,4,8,L,1),I=25,32),L=1,NZEN)/
     * 9.017,9.158,9.336,9.480,9.627,9.772,9.932,10.097,
     * 8.896,9.045,9.228,9.396,9.567,9.710,9.867,10.025,
     * 8.851,8.996,9.167,9.310,9.459,9.591,9.737,9.892,
     * 8.783,8.931,9.097,9.225,9.365,9.455,9.560,9.705,
     * 8.657,8.818,8.978,9.124,9.286,9.371,9.476,9.583,
     * 8.565,8.746,8.914,9.072,9.243,9.342,9.462,9.576,
     * 8.334,8.566,8.766,8.955,9.164,9.290,9.435,9.565,
     * 7.648,7.923,8.127,8.439,8.781,8.990,9.223,9.412,
     * 7.639,7.870,8.013,8.267,8.558,8.740,8.945,9.124,
     * 7.622,7.848,7.984,8.236,8.524,8.705,8.911,9.086,
     * 7.550,7.751,7.862,8.093,8.360,8.535,8.733,8.890,
     * 7.504,7.706,7.816,8.048,8.314,8.490,8.687,8.845/
      DATA ((EDEN(I,5,8,L,1),I=25,32),L=1,NZEN)/
     * 9.068,9.238,9.435,9.575,9.721,9.856,10.013,10.188,
     * 8.921,9.083,9.265,9.427,9.594,9.729,9.889,10.045,
     * 8.873,9.029,9.199,9.334,9.477,9.601,9.750,9.907,
     * 8.792,8.950,9.114,9.236,9.369,9.446,9.548,9.701,
     * 8.639,8.810,8.967,9.107,9.262,9.334,9.433,9.539,
     * 8.529,8.719,8.884,9.033,9.207,9.294,9.410,9.521,
     * 8.279,8.513,8.706,8.894,9.107,9.223,9.369,9.500,
     * 7.738,7.949,8.079,8.360,8.674,8.899,9.155,9.346,
     * 7.747,7.933,8.021,8.228,8.471,8.684,8.928,9.107,
     * 7.736,7.917,7.998,8.196,8.433,8.646,8.892,9.072,
     * 7.687,7.844,7.900,8.061,8.258,8.470,8.714,8.873,
     * 7.641,7.798,7.854,8.015,8.212,8.424,8.669,8.828/
      DATA ((EDEN(I,1,9,L,1),I=25,32),L=1,NZEN)/
     * 8.700,8.808,8.985,9.193,9.396,9.598,9.787,9.961,
     * 8.585,8.717,8.914,9.140,9.362,9.556,9.738,9.903,
     * 8.553,8.686,8.877,9.086,9.294,9.467,9.627,9.788,
     * 8.501,8.641,8.832,9.037,9.243,9.386,9.516,9.655,
     * 8.401,8.562,8.754,8.970,9.190,9.342,9.483,9.606,
     * 8.316,8.501,8.705,8.929,9.161,9.322,9.474,9.606,
     * 8.097,8.336,8.579,8.831,9.097,9.276,9.449,9.597,
     * 7.294,7.639,7.944,8.314,8.708,8.961,9.210,9.418,
     * 7.270,7.573,7.822,8.149,8.505,8.723,8.940,9.134,
     * 7.243,7.542,7.783,8.114,8.471,8.689,8.904,9.097,
     * 7.130,7.405,7.620,7.950,8.310,8.516,8.719,8.893,
     * 7.085,7.359,7.574,7.905,8.264,8.470,8.673,8.847/
      DATA ((EDEN(I,2,9,L,1),I=25,32),L=1,NZEN)/
     * 8.850,8.955,9.117,9.290,9.456,9.630,9.800,9.962,
     * 8.716,8.853,9.045,9.243,9.441,9.612,9.777,9.934,
     * 8.682,8.820,9.004,9.190,9.371,9.520,9.665,9.818,
     * 8.628,8.775,8.960,9.137,9.316,9.435,9.548,9.680,
     * 8.528,8.695,8.881,9.068,9.265,9.389,9.512,9.625,
     * 8.442,8.633,8.833,9.029,9.236,9.371,9.505,9.625,
     * 8.223,8.468,8.705,8.930,9.167,9.324,9.480,9.616,
     * 7.425,7.774,8.072,8.417,8.784,9.013,9.246,9.441,
     * 7.403,7.708,7.944,8.246,8.575,8.770,8.968,9.152,
     * 7.377,7.678,7.911,8.212,8.543,8.736,8.933,9.114,
     * 7.270,7.543,7.747,8.049,8.382,8.563,8.749,8.913,
     * 7.224,7.497,7.702,8.003,8.336,8.518,8.703,8.868/
      DATA ((EDEN(I,3,9,L,1),I=25,32),L=1,NZEN)/
     * 9.004,9.107,9.258,9.400,9.537,9.683,9.832,9.984,
     * 8.851,8.993,9.176,9.356,9.530,9.681,9.834,9.985,
     * 8.815,8.957,9.137,9.294,9.450,9.582,9.716,9.864,
     * 8.759,8.908,9.086,9.238,9.389,9.484,9.582,9.712,
     * 8.656,8.826,9.004,9.167,9.334,9.433,9.537,9.642,
     * 8.572,8.765,8.955,9.127,9.305,9.415,9.529,9.641,
     * 8.358,8.605,8.831,9.029,9.243,9.371,9.507,9.633,
     * 7.567,7.916,8.207,8.525,8.866,9.068,9.281,9.467,
     * 7.542,7.843,8.068,8.340,8.639,8.810,8.991,9.164,
     * 7.516,7.813,8.029,8.305,8.606,8.775,8.956,9.127,
     * 7.415,7.685,7.874,8.149,8.447,8.605,8.774,8.929,
     * 7.369,7.639,7.829,8.103,8.401,8.560,8.728,8.883/
      DATA ((EDEN(I,4,9,L,1),I=25,32),L=1,NZEN)/
     * 9.149,9.258,9.400,9.522,9.639,9.766,9.901,10.045,
     * 8.987,9.130,9.307,9.468,9.625,9.757,9.898,10.041,
     * 8.947,9.086,9.255,9.394,9.529,9.646,9.775,9.916,
     * 8.889,9.037,9.199,9.324,9.449,9.528,9.615,9.744,
     * 8.785,8.950,9.111,9.248,9.387,9.462,9.550,9.646,
     * 8.705,8.891,9.064,9.207,9.356,9.442,9.542,9.644,
     * 8.502,8.738,8.943,9.111,9.292,9.400,9.520,9.637,
     * 7.718,8.061,8.328,8.619,8.931,9.111,9.310,9.484,
     * 7.686,7.971,8.167,8.413,8.684,8.836,9.004,9.170,
     * 7.663,7.943,8.134,8.380,8.650,8.801,8.969,9.134,
     * 7.566,7.822,7.986,8.228,8.493,8.632,8.790,8.937,
     * 7.520,7.776,7.941,8.182,8.447,8.587,8.745,8.891/
      DATA ((EDEN(I,5,9,L,1),I=25,32),L=1,NZEN)/
     * 9.303,9.412,9.544,9.643,9.736,9.841,9.962,10.100,
     * 9.114,9.262,9.430,9.575,9.717,9.830,9.959,10.097,
     * 9.076,9.215,9.367,9.490,9.607,9.710,9.828,9.965,
     * 9.017,9.161,9.310,9.412,9.513,9.572,9.647,9.775,
     * 8.915,9.076,9.220,9.332,9.447,9.494,9.559,9.644,
     * 8.838,9.017,9.173,9.290,9.415,9.473,9.548,9.638,
     * 8.641,8.870,9.057,9.199,9.354,9.430,9.525,9.630,
     * 7.870,8.199,8.442,8.709,8.996,9.149,9.328,9.489,
     * 7.832,8.097,8.262,8.480,8.723,8.857,9.017,9.173,
     * 7.809,8.068,8.230,8.447,8.689,8.823,8.981,9.137,
     * 7.718,7.954,8.090,8.297,8.531,8.657,8.807,8.943,
     * 7.672,7.908,8.044,8.251,8.486,8.611,8.761,8.897/
      DATA ((EDEN(I,1,10,L,1),I=25,32),L=1,NZEN)/
     * 8.698,8.804,8.980,9.179,9.375,9.574,9.760,9.932,
     * 8.592,8.719,8.910,9.134,9.352,9.542,9.719,9.881,
     * 8.560,8.688,8.876,9.083,9.290,9.456,9.611,9.770,
     * 8.509,8.644,8.829,9.033,9.243,9.382,9.512,9.648,
     * 8.410,8.566,8.753,8.968,9.190,9.340,9.483,9.606,
     * 8.324,8.504,8.702,8.927,9.161,9.322,9.474,9.606,
     * 8.100,8.336,8.574,8.827,9.093,9.274,9.447,9.594,
     * 7.299,7.637,7.937,8.307,8.702,8.954,9.204,9.412,
     * 7.276,7.574,7.818,8.146,8.505,8.726,8.943,9.137,
     * 7.248,7.544,7.782,8.114,8.473,8.692,8.908,9.100,
     * 7.134,7.407,7.619,7.950,8.310,8.517,8.721,8.897,
     * 7.088,7.361,7.573,7.904,8.264,8.471,8.675,8.851/
      DATA ((EDEN(I,2,10,L,1),I=25,32),L=1,NZEN)/
     * 8.846,8.957,9.127,9.305,9.476,9.648,9.814,9.973,
     * 8.722,8.859,9.049,9.250,9.446,9.611,9.769,9.920,
     * 8.691,8.829,9.013,9.201,9.384,9.526,9.662,9.810,
     * 8.638,8.785,8.970,9.155,9.338,9.453,9.563,9.687,
     * 8.539,8.708,8.895,9.090,9.288,9.413,9.535,9.645,
     * 8.455,8.647,8.846,9.049,9.258,9.396,9.529,9.646,
     * 8.233,8.483,8.721,8.951,9.190,9.348,9.502,9.635,
     * 7.430,7.782,8.083,8.431,8.799,9.029,9.260,9.452,
     * 7.405,7.716,7.959,8.267,8.599,8.794,8.992,9.173,
     * 7.378,7.686,7.925,8.233,8.566,8.760,8.955,9.137,
     * 7.265,7.547,7.759,8.068,8.401,8.584,8.768,8.931,
     * 7.219,7.501,7.713,8.022,8.356,8.539,8.722,8.886/
      DATA ((EDEN(I,3,10,L,1),I=25,32),L=1,NZEN)/
     * 9.029,9.134,9.286,9.435,9.571,9.713,9.852,9.995,
     * 8.872,9.017,9.204,9.386,9.555,9.700,9.841,9.981,
     * 8.841,8.987,9.170,9.334,9.489,9.610,9.729,9.866,
     * 8.790,8.945,9.127,9.286,9.439,9.529,9.618,9.733,
     * 8.693,8.872,9.057,9.225,9.391,9.489,9.585,9.684,
     * 8.614,8.816,9.013,9.190,9.365,9.473,9.580,9.685,
     * 8.401,8.659,8.894,9.097,9.303,9.428,9.556,9.676,
     * 7.585,7.960,8.274,8.590,8.921,9.117,9.322,9.501,
     * 7.553,7.882,8.134,8.407,8.699,8.864,9.033,9.204,
     * 7.525,7.851,8.097,8.371,8.666,8.829,8.999,9.164,
     * 7.410,7.709,7.927,8.204,8.502,8.655,8.814,8.963,
     * 7.364,7.664,7.882,8.158,8.457,8.609,8.768,8.918/
      DATA ((EDEN(I,4,10,L,1),I=25,32),L=1,NZEN)/
     * 9.228,9.320,9.446,9.566,9.667,9.781,9.894,10.021,
     * 9.072,9.215,9.386,9.548,9.691,9.813,9.937,10.064,
     * 9.041,9.179,9.344,9.483,9.606,9.712,9.818,9.944,
     * 8.995,9.140,9.303,9.428,9.542,9.610,9.681,9.790,
     * 8.910,9.079,9.243,9.375,9.498,9.566,9.636,9.719,
     * 8.841,9.029,9.204,9.342,9.476,9.550,9.630,9.719,
     * 8.657,8.898,9.107,9.265,9.420,9.513,9.613,9.715,
     * 7.811,8.212,8.542,8.803,9.072,9.236,9.405,9.563,
     * 7.747,8.090,8.346,8.568,8.803,8.933,9.072,9.225,
     * 7.717,8.061,8.310,8.534,8.772,8.899,9.037,9.188,
     * 7.593,7.915,8.146,8.375,8.617,8.732,8.857,8.990,
     * 7.548,7.869,8.100,8.329,8.571,8.686,8.811,8.945/
      DATA ((EDEN(I,5,10,L,1),I=25,32),L=1,NZEN)/
     * 9.428,9.508,9.613,9.708,9.780,9.862,9.949,10.061,
     * 9.281,9.417,9.573,9.713,9.832,9.928,10.029,10.149,
     * 9.250,9.380,9.522,9.639,9.736,9.820,9.909,10.021,
     * 9.210,9.344,9.483,9.579,9.656,9.702,9.752,9.853,
     * 9.137,9.292,9.435,9.533,9.618,9.650,9.689,9.755,
     * 9.076,9.253,9.405,9.508,9.600,9.639,9.687,9.757,
     * 8.920,9.143,9.324,9.442,9.556,9.610,9.673,9.757,
     * 8.072,8.486,8.817,9.029,9.243,9.360,9.489,9.624,
     * 7.963,8.316,8.567,8.739,8.919,9.009,9.114,9.250,
     * 7.929,8.281,8.531,8.706,8.888,8.977,9.079,9.215,
     * 7.788,8.127,8.364,8.544,8.732,8.811,8.903,9.021,
     * 7.742,8.081,8.318,8.498,8.687,8.765,8.857,8.975/
      DATA ((EDEN(I,1,11,L,1),I=25,32),L=1,NZEN)/
     * 8.723,8.821,8.988,9.176,9.362,9.555,9.735,9.904,
     * 8.594,8.719,8.908,9.127,9.346,9.534,9.708,9.870,
     * 8.563,8.689,8.874,9.083,9.288,9.452,9.603,9.762,
     * 8.509,8.644,8.829,9.033,9.243,9.382,9.511,9.644,
     * 8.410,8.566,8.753,8.968,9.190,9.340,9.483,9.606,
     * 8.324,8.504,8.702,8.927,9.161,9.322,9.476,9.607,
     * 8.100,8.334,8.571,8.823,9.090,9.272,9.447,9.593,
     * 7.299,7.635,7.931,8.303,8.697,8.951,9.199,9.407,
     * 7.276,7.575,7.818,8.149,8.507,8.728,8.946,9.140,
     * 7.250,7.544,7.781,8.114,8.473,8.693,8.910,9.104,
     * 7.137,7.405,7.616,7.948,8.307,8.517,8.723,8.899,
     * 7.091,7.359,7.570,7.902,8.262,8.471,8.678,8.854/
      DATA ((EDEN(I,2,11,L,1),I=25,32),L=1,NZEN)/
     * 8.855,8.969,9.140,9.326,9.500,9.673,9.836,9.991,
     * 8.728,8.866,9.057,9.262,9.458,9.620,9.772,9.919,
     * 8.696,8.837,9.025,9.217,9.401,9.540,9.670,9.812,
     * 8.646,8.796,8.982,9.173,9.358,9.474,9.582,9.699,
     * 8.549,8.722,8.912,9.114,9.312,9.439,9.559,9.665,
     * 8.467,8.662,8.863,9.072,9.283,9.422,9.551,9.666,
     * 8.248,8.500,8.740,8.975,9.215,9.373,9.526,9.655,
     * 7.435,7.794,8.104,8.455,8.820,9.049,9.276,9.467,
     * 7.407,7.728,7.981,8.292,8.621,8.817,9.013,9.190,
     * 7.378,7.695,7.945,8.258,8.589,8.783,8.976,9.152,
     * 7.260,7.551,7.774,8.086,8.422,8.605,8.788,8.948,
     * 7.214,7.506,7.728,8.041,8.376,8.560,8.742,8.903/
      DATA ((EDEN(I,3,11,L,1),I=25,32),L=1,NZEN)/
     * 9.033,9.146,9.305,9.468,9.609,9.751,9.884,10.021,
     * 8.897,9.045,9.238,9.425,9.593,9.732,9.861,9.994,
     * 8.869,9.021,9.207,9.377,9.530,9.647,9.757,9.885,
     * 8.823,8.983,9.167,9.336,9.487,9.575,9.657,9.762,
     * 8.732,8.918,9.111,9.283,9.446,9.542,9.631,9.722,
     * 8.658,8.865,9.068,9.250,9.422,9.526,9.626,9.723,
     * 8.453,8.718,8.960,9.161,9.360,9.484,9.603,9.715,
     * 7.611,8.013,8.352,8.665,8.982,9.176,9.369,9.539,
     * 7.569,7.924,8.201,8.473,8.753,8.912,9.072,9.236,
     * 7.538,7.890,8.164,8.436,8.720,8.879,9.037,9.196,
     * 7.410,7.740,7.989,8.267,8.556,8.703,8.851,8.994,
     * 7.364,7.695,7.943,8.221,8.511,8.658,8.805,8.949/
      DATA ((EDEN(I,4,11,L,1),I=25,32),L=1,NZEN)/
     * 9.241,9.342,9.477,9.612,9.716,9.823,9.924,10.041,
     * 9.134,9.276,9.450,9.621,9.761,9.877,9.986,10.100,
     * 9.107,9.248,9.415,9.562,9.681,9.780,9.872,9.984,
     * 9.068,9.220,9.384,9.517,9.625,9.689,9.747,9.842,
     * 8.996,9.170,9.338,9.476,9.592,9.654,9.712,9.782,
     * 8.934,9.130,9.310,9.450,9.573,9.643,9.711,9.787,
     * 8.768,9.013,9.228,9.384,9.528,9.613,9.695,9.785,
     * 7.902,8.340,8.706,8.950,9.190,9.340,9.491,9.635,
     * 7.808,8.196,8.493,8.695,8.898,9.017,9.134,9.276,
     * 7.773,8.161,8.456,8.661,8.867,8.982,9.100,9.238,
     * 7.624,8.000,8.283,8.496,8.709,8.812,8.917,9.037,
     * 7.579,7.954,8.238,8.450,8.664,8.766,8.871,8.992/
      DATA ((EDEN(I,5,11,L,1),I=25,32),L=1,NZEN)/
     * 9.452,9.549,9.668,9.782,9.859,9.929,9.993,10.086,
     * 9.369,9.512,9.672,9.822,9.935,10.021,10.104,10.204,
     * 9.348,9.484,9.632,9.760,9.852,9.924,9.992,10.086,
     * 9.320,9.464,9.610,9.717,9.790,9.825,9.856,9.934,
     * 9.267,9.435,9.585,9.692,9.769,9.793,9.814,9.860,
     * 9.217,9.405,9.567,9.677,9.760,9.790,9.817,9.867,
     * 9.079,9.314,9.504,9.626,9.727,9.768,9.809,9.871,
     * 8.248,8.694,9.053,9.248,9.431,9.530,9.630,9.744,
     * 8.090,8.487,8.784,8.929,9.068,9.137,9.212,9.328,
     * 8.053,8.453,8.750,8.896,9.033,9.104,9.176,9.292,
     * 7.871,8.274,8.574,8.727,8.873,8.933,8.996,9.093,
     * 7.825,8.228,8.528,8.681,8.827,8.887,8.950,9.048/
      DATA ((EDEN(I,1,12,L,1),I=25,32),L=1,NZEN)/
     * 8.752,8.841,8.999,9.182,9.360,9.547,9.721,9.889,
     * 8.583,8.712,8.906,9.127,9.346,9.534,9.709,9.873,
     * 8.550,8.681,8.872,9.079,9.288,9.453,9.605,9.763,
     * 8.501,8.638,8.825,9.033,9.243,9.384,9.513,9.646,
     * 8.396,8.558,8.748,8.966,9.190,9.342,9.484,9.610,
     * 8.310,8.494,8.697,8.924,9.161,9.322,9.476,9.609,
     * 8.083,8.322,8.563,8.819,9.090,9.272,9.449,9.595,
     * 7.290,7.623,7.918,8.292,8.690,8.947,9.199,9.408,
     * 7.270,7.567,7.810,8.143,8.507,8.730,8.950,9.146,
     * 7.243,7.537,7.772,8.107,8.473,8.695,8.913,9.107,
     * 7.130,7.398,7.607,7.943,8.305,8.517,8.725,8.901,
     * 7.085,7.352,7.562,7.897,8.260,8.471,8.679,8.856/
      DATA ((EDEN(I,2,12,L,1),I=25,32),L=1,NZEN)/
     * 8.867,8.982,9.155,9.344,9.519,9.692,9.852,10.004,
     * 8.715,8.862,9.061,9.272,9.470,9.633,9.785,9.929,
     * 8.687,8.834,9.029,9.228,9.415,9.555,9.683,9.823,
     * 8.637,8.794,8.987,9.185,9.373,9.490,9.595,9.710,
     * 8.543,8.721,8.916,9.124,9.326,9.453,9.573,9.677,
     * 8.456,8.660,8.870,9.083,9.297,9.436,9.566,9.677,
     * 8.236,8.496,8.746,8.986,9.230,9.387,9.539,9.666,
     * 7.423,7.788,8.104,8.461,8.833,9.064,9.290,9.477,
     * 7.398,7.723,7.983,8.301,8.635,8.831,9.025,9.201,
     * 7.367,7.692,7.947,8.265,8.601,8.796,8.989,9.164,
     * 7.248,7.547,7.775,8.093,8.433,8.617,8.800,8.958,
     * 7.202,7.501,7.729,8.048,8.387,8.571,8.754,8.912/
      DATA ((EDEN(I,3,12,L,1),I=25,32),L=1,NZEN)/
     * 9.025,9.149,9.318,9.493,9.640,9.784,9.915,10.049,
     * 8.892,9.053,9.253,9.452,9.623,9.762,9.889,10.017,
     * 8.864,9.025,9.223,9.403,9.560,9.677,9.783,9.908,
     * 8.821,8.992,9.188,9.364,9.516,9.605,9.684,9.785,
     * 8.732,8.928,9.130,9.312,9.476,9.572,9.659,9.745,
     * 8.655,8.877,9.093,9.281,9.453,9.559,9.655,9.747,
     * 8.456,8.731,8.983,9.193,9.394,9.517,9.633,9.740,
     * 7.607,8.029,8.386,8.702,9.017,9.210,9.400,9.565,
     * 7.562,7.935,8.228,8.504,8.785,8.942,9.097,9.255,
     * 7.530,7.902,8.193,8.470,8.751,8.907,9.061,9.217,
     * 7.398,7.747,8.013,8.297,8.587,8.732,8.874,9.013,
     * 7.352,7.702,7.967,8.251,8.541,8.686,8.828,8.967/
      DATA ((EDEN(I,4,12,L,1),I=25,32),L=1,NZEN)/
     * 9.223,9.338,9.490,9.641,9.756,9.862,9.957,10.068,
     * 9.143,9.301,9.486,9.667,9.814,9.927,10.029,10.140,
     * 9.121,9.272,9.449,9.609,9.732,9.831,9.918,10.025,
     * 9.086,9.246,9.420,9.565,9.675,9.738,9.791,9.880,
     * 9.017,9.201,9.382,9.528,9.644,9.704,9.755,9.819,
     * 8.960,9.167,9.356,9.505,9.628,9.695,9.754,9.823,
     * 8.803,9.057,9.276,9.441,9.585,9.667,9.741,9.823,
     * 7.940,8.398,8.783,9.029,9.260,9.405,9.543,9.679,
     * 7.828,8.236,8.554,8.755,8.949,9.061,9.170,9.303,
     * 7.791,8.201,8.520,8.723,8.919,9.029,9.134,9.267,
     * 7.630,8.037,8.346,8.556,8.760,8.858,8.952,9.068,
     * 7.585,7.992,8.301,8.511,8.715,8.812,8.906,9.022/
      DATA ((EDEN(I,5,12,L,1),I=25,32),L=1,NZEN)/
     * 9.436,9.554,9.695,9.831,9.919,9.991,10.053,10.137,
     * 9.384,9.542,9.718,9.885,10.004,10.090,10.164,10.255,
     * 9.364,9.516,9.679,9.825,9.924,9.994,10.053,10.140,
     * 9.340,9.500,9.659,9.782,9.862,9.897,9.921,9.989,
     * 9.294,9.477,9.643,9.766,9.848,9.869,9.881,9.916,
     * 9.248,9.453,9.629,9.753,9.841,9.867,9.886,9.925,
     * 9.124,9.371,9.573,9.708,9.811,9.850,9.881,9.931,
     * 8.314,8.772,9.146,9.348,9.528,9.621,9.711,9.810,
     * 8.130,8.547,8.859,9.009,9.140,9.207,9.267,9.373,
     * 8.086,8.509,8.826,8.977,9.111,9.173,9.233,9.336,
     * 7.894,8.322,8.647,8.804,8.943,8.998,9.049,9.137,
     * 7.848,8.276,8.602,8.758,8.898,8.952,9.003,9.091/
      DATA ((EDEN(I,1,1,L,2),I=25,32),L=1,NZEN)/
     * 8.931,9.009,9.152,9.342,9.528,9.728,9.914,10.083,
     * 8.766,8.879,9.057,9.272,9.484,9.682,9.866,10.029,
     * 8.732,8.844,9.017,9.212,9.408,9.582,9.741,9.901,
     * 8.684,8.794,8.959,9.146,9.342,9.489,9.622,9.752,
     * 8.600,8.717,8.872,9.057,9.260,9.413,9.554,9.665,
     * 8.526,8.657,8.818,9.004,9.215,9.375,9.524,9.637,
     * 8.320,8.493,8.682,8.881,9.117,9.294,9.458,9.582,
     * 7.529,7.771,7.999,8.283,8.629,8.876,9.107,9.279,
     * 7.521,7.710,7.873,8.100,8.394,8.598,8.785,8.928,
     * 7.489,7.672,7.829,8.057,8.356,8.556,8.743,8.885,
     * 7.338,7.493,7.618,7.854,8.161,8.356,8.535,8.664,
     * 7.293,7.447,7.572,7.809,8.116,8.310,8.490,8.618/
      DATA ((EDEN(I,2,1,L,2),I=25,32),L=1,NZEN)/
     * 9.041,9.149,9.312,9.513,9.698,9.880,10.049,10.196,
     * 8.895,9.029,9.215,9.422,9.615,9.786,9.944,10.086,
     * 8.863,8.995,9.176,9.364,9.540,9.688,9.822,9.962,
     * 8.819,8.951,9.124,9.303,9.477,9.599,9.707,9.816,
     * 8.739,8.878,9.045,9.217,9.400,9.528,9.642,9.732,
     * 8.670,8.823,8.994,9.167,9.356,9.491,9.613,9.706,
     * 8.470,8.665,8.864,9.049,9.262,9.413,9.550,9.651,
     * 7.665,7.942,8.193,8.462,8.780,8.998,9.201,9.350,
     * 7.650,7.872,8.057,8.267,8.534,8.707,8.863,8.987,
     * 7.616,7.833,8.013,8.223,8.496,8.666,8.822,8.944,
     * 7.456,7.647,7.797,8.021,8.301,8.465,8.615,8.724,
     * 7.411,7.602,7.752,7.975,8.255,8.420,8.569,8.679/
      DATA ((EDEN(I,3,1,L,2),I=25,32),L=1,NZEN)/
     * 9.179,9.303,9.470,9.660,9.822,9.977,10.117,10.248,
     * 9.064,9.215,9.405,9.602,9.772,9.919,10.053,10.176,
     * 9.037,9.182,9.365,9.539,9.690,9.815,9.927,10.049,
     * 8.994,9.143,9.320,9.480,9.623,9.717,9.797,9.892,
     * 8.922,9.079,9.250,9.405,9.553,9.648,9.731,9.802,
     * 8.861,9.029,9.210,9.360,9.513,9.615,9.704,9.777,
     * 8.684,8.893,9.093,9.255,9.427,9.543,9.645,9.727,
     * 7.853,8.179,8.471,8.702,8.968,9.149,9.314,9.441,
     * 7.818,8.083,8.297,8.470,8.687,8.822,8.942,9.045,
     * 7.782,8.045,8.255,8.428,8.647,8.782,8.900,9.000,
     * 7.610,7.850,8.037,8.225,8.458,8.583,8.695,8.783,
     * 7.564,7.804,7.992,8.180,8.412,8.537,8.649,8.737/
      DATA ((EDEN(I,4,1,L,2),I=25,32),L=1,NZEN)/
     * 9.354,9.477,9.633,9.803,9.934,10.057,10.164,10.272,
     * 9.290,9.441,9.622,9.810,9.960,10.079,10.188,10.290,
     * 9.265,9.407,9.576,9.736,9.861,9.967,10.061,10.161,
     * 9.230,9.373,9.534,9.671,9.779,9.847,9.902,9.986,
     * 9.173,9.324,9.480,9.605,9.713,9.775,9.823,9.873,
     * 9.127,9.288,9.446,9.568,9.679,9.747,9.799,9.851,
     * 8.985,9.179,9.356,9.480,9.605,9.684,9.748,9.806,
     * 8.155,8.509,8.816,8.993,9.196,9.332,9.455,9.550,
     * 8.068,8.358,8.587,8.701,8.849,8.942,9.021,9.100,
     * 8.029,8.318,8.545,8.662,8.812,8.904,8.980,9.057,
     * 7.834,8.121,8.340,8.465,8.625,8.709,8.777,8.843,
     * 7.789,8.075,8.295,8.420,8.580,8.664,8.732,8.797/
      DATA ((EDEN(I,5,1,L,2),I=25,32),L=1,NZEN)/
     * 9.534,9.667,9.821,9.980,10.086,10.179,10.255,10.338,
     * 9.490,9.652,9.833,10.013,10.143,10.238,10.318,10.398,
     * 9.468,9.618,9.782,9.933,10.037,10.121,10.188,10.270,
     * 9.442,9.591,9.744,9.866,9.947,9.989,10.017,10.083,
     * 9.403,9.559,9.705,9.813,9.891,9.915,9.925,9.950,
     * 9.369,9.533,9.680,9.782,9.862,9.890,9.904,9.928,
     * 9.250,9.442,9.605,9.708,9.799,9.837,9.860,9.889,
     * 8.468,8.826,9.124,9.270,9.425,9.516,9.592,9.656,
     * 8.330,8.627,8.851,8.926,9.021,9.076,9.111,9.164,
     * 8.288,8.588,8.814,8.888,8.985,9.037,9.072,9.121,
     * 8.072,8.380,8.614,8.695,8.798,8.841,8.867,8.907,
     * 8.026,8.334,8.568,8.649,8.752,8.795,8.822,8.862/
      DATA ((EDEN(I,1,2,L,2),I=25,32),L=1,NZEN)/
     * 8.908,8.993,9.146,9.340,9.533,9.736,9.928,10.097,
     * 8.764,8.880,9.061,9.276,9.493,9.694,9.882,10.045,
     * 8.731,8.844,9.017,9.215,9.413,9.590,9.754,9.915,
     * 8.681,8.794,8.961,9.149,9.344,9.491,9.625,9.758,
     * 8.598,8.716,8.874,9.057,9.262,9.415,9.554,9.666,
     * 8.526,8.658,8.820,9.004,9.215,9.377,9.524,9.638,
     * 8.320,8.494,8.685,8.884,9.117,9.297,9.461,9.583,
     * 7.528,7.772,8.004,8.290,8.635,8.883,9.114,9.286,
     * 7.520,7.709,7.873,8.100,8.396,8.597,8.782,8.925,
     * 7.487,7.672,7.829,8.057,8.356,8.556,8.741,8.882,
     * 7.336,7.493,7.620,7.856,8.164,8.356,8.535,8.662,
     * 7.291,7.447,7.574,7.810,8.119,8.310,8.490,8.616/
      DATA ((EDEN(I,2,2,L,2),I=25,32),L=1,NZEN)/
     * 9.025,9.134,9.301,9.500,9.684,9.866,10.037,10.188,
     * 8.887,9.021,9.212,9.420,9.614,9.789,9.952,10.097,
     * 8.855,8.988,9.170,9.356,9.534,9.686,9.825,9.970,
     * 8.808,8.941,9.117,9.292,9.467,9.590,9.700,9.814,
     * 8.724,8.865,9.033,9.207,9.387,9.515,9.630,9.723,
     * 8.653,8.808,8.980,9.155,9.342,9.479,9.601,9.696,
     * 8.455,8.650,8.852,9.037,9.248,9.400,9.539,9.642,
     * 7.652,7.928,8.179,8.450,8.769,8.989,9.193,9.344,
     * 7.639,7.859,8.041,8.250,8.520,8.694,8.854,8.979,
     * 7.606,7.820,7.995,8.207,8.481,8.653,8.812,8.936,
     * 7.453,7.638,7.785,8.004,8.288,8.453,8.605,8.715,
     * 7.408,7.593,7.740,7.959,8.242,8.408,8.560,8.669/
      DATA ((EDEN(I,3,2,L,2),I=25,32),L=1,NZEN)/
     * 9.170,9.290,9.456,9.639,9.799,9.953,10.097,10.228,
     * 9.045,9.196,9.389,9.588,9.763,9.915,10.053,10.182,
     * 9.013,9.158,9.344,9.520,9.674,9.806,9.925,10.053,
     * 8.965,9.114,9.292,9.453,9.600,9.695,9.780,9.883,
     * 8.887,9.045,9.217,9.373,9.524,9.620,9.705,9.780,
     * 8.823,8.994,9.173,9.326,9.483,9.585,9.678,9.755,
     * 8.642,8.852,9.057,9.217,9.396,9.515,9.619,9.704,
     * 7.819,8.140,8.425,8.664,8.941,9.124,9.292,9.423,
     * 7.788,8.045,8.253,8.431,8.657,8.795,8.920,9.025,
     * 7.754,8.009,8.210,8.389,8.619,8.756,8.879,8.984,
     * 7.593,7.821,7.997,8.188,8.430,8.559,8.675,8.766,
     * 7.548,7.775,7.951,8.142,8.384,8.513,8.629,8.721/
      DATA ((EDEN(I,4,2,L,2),I=25,32),L=1,NZEN)/
     * 9.334,9.456,9.611,9.776,9.905,10.029,10.140,10.253,
     * 9.250,9.405,9.592,9.783,9.940,10.064,10.182,10.286,
     * 9.217,9.364,9.537,9.698,9.829,9.943,10.045,10.152,
     * 9.173,9.320,9.486,9.622,9.733,9.808,9.872,9.964,
     * 9.107,9.260,9.418,9.545,9.658,9.724,9.779,9.835,
     * 9.057,9.220,9.380,9.502,9.620,9.692,9.753,9.810,
     * 8.906,9.104,9.286,9.412,9.544,9.628,9.701,9.764,
     * 8.076,8.428,8.732,8.920,9.137,9.279,9.410,9.512,
     * 8.000,8.281,8.502,8.629,8.794,8.895,8.984,9.068,
     * 7.963,8.246,8.464,8.591,8.757,8.857,8.944,9.029,
     * 7.787,8.057,8.262,8.400,8.575,8.666,8.744,8.814,
     * 7.742,8.011,8.217,8.354,8.529,8.620,8.698,8.768/
      DATA ((EDEN(I,5,2,L,2),I=25,32),L=1,NZEN)/
     * 9.491,9.622,9.775,9.930,10.041,10.134,10.220,10.314,
     * 9.418,9.589,9.779,9.962,10.100,10.201,10.292,10.382,
     * 9.386,9.542,9.713,9.867,9.980,10.072,10.155,10.246,
     * 9.344,9.498,9.659,9.780,9.867,9.919,9.960,10.041,
     * 9.286,9.444,9.595,9.705,9.790,9.822,9.843,9.880,
     * 9.241,9.410,9.562,9.667,9.754,9.791,9.817,9.853,
     * 9.111,9.310,9.477,9.584,9.683,9.732,9.767,9.810,
     * 8.301,8.668,8.975,9.130,9.303,9.407,9.498,9.577,
     * 8.188,8.487,8.713,8.807,8.927,8.993,9.045,9.114,
     * 8.152,8.450,8.675,8.769,8.890,8.954,9.004,9.068,
     * 7.969,8.265,8.486,8.584,8.712,8.766,8.809,8.859,
     * 7.924,8.219,8.440,8.539,8.666,8.721,8.763,8.813/
      DATA ((EDEN(I,1,3,L,2),I=25,32),L=1,NZEN)/
     * 8.881,8.973,9.134,9.338,9.542,9.749,9.944,10.111,
     * 8.769,8.883,9.061,9.279,9.493,9.694,9.882,10.045,
     * 8.736,8.846,9.017,9.215,9.413,9.590,9.754,9.915,
     * 8.686,8.797,8.961,9.149,9.342,9.490,9.624,9.757,
     * 8.603,8.719,8.874,9.057,9.262,9.413,9.551,9.663,
     * 8.531,8.660,8.820,9.004,9.215,9.375,9.521,9.635,
     * 8.326,8.498,8.688,8.886,9.121,9.297,9.458,9.582,
     * 7.531,7.777,8.009,8.292,8.637,8.883,9.114,9.286,
     * 7.522,7.712,7.876,8.100,8.394,8.594,8.780,8.922,
     * 7.489,7.674,7.833,8.057,8.356,8.555,8.738,8.879,
     * 7.338,7.496,7.622,7.857,8.164,8.356,8.533,8.660,
     * 7.293,7.450,7.576,7.812,8.119,8.310,8.487,8.614/
      DATA ((EDEN(I,2,3,L,2),I=25,32),L=1,NZEN)/
     * 9.004,9.114,9.281,9.474,9.656,9.840,10.013,10.167,
     * 8.886,9.017,9.204,9.408,9.604,9.781,9.948,10.097,
     * 8.852,8.981,9.161,9.342,9.520,9.675,9.820,9.967,
     * 8.799,8.930,9.104,9.274,9.447,9.571,9.683,9.802,
     * 8.714,8.851,9.017,9.185,9.364,9.491,9.609,9.703,
     * 8.644,8.794,8.963,9.134,9.318,9.455,9.579,9.677,
     * 8.444,8.635,8.833,9.017,9.225,9.377,9.517,9.623,
     * 7.645,7.916,8.161,8.430,8.749,8.969,9.176,9.330,
     * 7.634,7.846,8.021,8.228,8.500,8.673,8.834,8.963,
     * 7.601,7.809,7.977,8.188,8.459,8.632,8.793,8.920,
     * 7.455,7.631,7.770,7.986,8.267,8.433,8.588,8.701,
     * 7.409,7.586,7.724,7.941,8.221,8.387,8.542,8.655/
      DATA ((EDEN(I,3,3,L,2),I=25,32),L=1,NZEN)/
     * 9.152,9.270,9.430,9.605,9.760,9.913,10.057,10.196,
     * 9.017,9.164,9.356,9.551,9.729,9.884,10.033,10.164,
     * 8.978,9.124,9.307,9.479,9.634,9.769,9.897,10.033,
     * 8.924,9.072,9.248,9.405,9.555,9.653,9.743,9.856,
     * 8.837,8.991,9.161,9.314,9.471,9.571,9.661,9.743,
     * 8.766,8.934,9.111,9.265,9.427,9.533,9.630,9.715,
     * 8.577,8.784,8.985,9.149,9.336,9.458,9.571,9.663,
     * 7.772,8.072,8.334,8.583,8.876,9.064,9.246,9.382,
     * 7.753,7.988,8.176,8.364,8.604,8.749,8.884,8.997,
     * 7.723,7.952,8.130,8.320,8.566,8.709,8.844,8.956,
     * 7.579,7.780,7.929,8.124,8.377,8.513,8.640,8.738,
     * 7.533,7.734,7.884,8.078,8.331,8.467,8.595,8.692/
      DATA ((EDEN(I,4,3,L,2),I=25,32),L=1,NZEN)/
     * 9.307,9.425,9.576,9.732,9.859,9.987,10.111,10.230,
     * 9.158,9.322,9.517,9.704,9.866,10.000,10.134,10.248,
     * 9.121,9.274,9.458,9.616,9.753,9.874,9.989,10.111,
     * 9.061,9.217,9.394,9.530,9.651,9.732,9.807,9.913,
     * 8.977,9.137,9.303,9.435,9.561,9.635,9.704,9.770,
     * 8.913,9.086,9.253,9.384,9.516,9.598,9.674,9.742,
     * 8.739,8.947,9.137,9.276,9.430,9.526,9.617,9.692,
     * 7.926,8.248,8.521,8.740,8.996,9.158,9.312,9.430,
     * 7.890,8.140,8.326,8.487,8.695,8.814,8.926,9.025,
     * 7.860,8.104,8.288,8.449,8.657,8.775,8.886,8.982,
     * 7.719,7.937,8.090,8.260,8.474,8.584,8.687,8.767,
     * 7.674,7.891,8.044,8.214,8.428,8.539,8.641,8.721/
      DATA ((EDEN(I,5,3,L,2),I=25,32),L=1,NZEN)/
     * 9.464,9.588,9.735,9.874,9.978,10.079,10.179,10.283,
     * 9.330,9.502,9.694,9.872,10.017,10.127,10.233,10.332,
     * 9.288,9.447,9.621,9.769,9.888,9.990,10.090,10.193,
     * 9.230,9.389,9.554,9.669,9.761,9.821,9.877,9.974,
     * 9.149,9.312,9.468,9.574,9.667,9.707,9.742,9.791,
     * 9.093,9.265,9.423,9.525,9.622,9.668,9.709,9.759,
     * 8.942,9.146,9.320,9.428,9.542,9.600,9.653,9.709,
     * 8.127,8.468,8.752,8.932,9.140,9.260,9.375,9.470,
     * 8.057,8.316,8.505,8.630,8.795,8.882,8.963,9.045,
     * 8.025,8.281,8.465,8.592,8.758,8.843,8.923,9.004,
     * 7.877,8.111,8.272,8.407,8.582,8.657,8.728,8.792,
     * 7.831,8.065,8.226,8.361,8.536,8.611,8.683,8.747/
      DATA ((EDEN(I,1,4,L,2),I=25,32),L=1,NZEN)/
     * 8.867,8.965,9.130,9.344,9.554,9.764,9.962,10.130,
     * 8.788,8.896,9.068,9.283,9.497,9.697,9.885,10.045,
     * 8.755,8.859,9.025,9.217,9.413,9.592,9.758,9.917,
     * 8.705,8.810,8.969,9.152,9.342,9.489,9.622,9.757,
     * 8.624,8.734,8.883,9.064,9.262,9.412,9.550,9.661,
     * 8.554,8.678,8.832,9.013,9.215,9.375,9.519,9.633,
     * 8.360,8.522,8.702,8.894,9.124,9.299,9.458,9.581,
     * 7.551,7.802,8.041,8.314,8.650,8.892,9.121,9.288,
     * 7.537,7.728,7.893,8.111,8.394,8.592,8.775,8.918,
     * 7.502,7.690,7.851,8.068,8.356,8.553,8.733,8.874,
     * 7.350,7.511,7.643,7.869,8.164,8.354,8.529,8.655,
     * 7.304,7.465,7.598,7.823,8.119,8.308,8.483,8.609/
      DATA ((EDEN(I,2,4,L,2),I=25,32),L=1,NZEN)/
     * 8.998,9.100,9.262,9.447,9.623,9.806,9.981,10.137,
     * 8.890,9.017,9.196,9.394,9.588,9.766,9.937,10.086,
     * 8.853,8.976,9.149,9.324,9.498,9.655,9.805,9.957,
     * 8.799,8.923,9.090,9.253,9.422,9.544,9.660,9.786,
     * 8.711,8.841,9.000,9.161,9.336,9.462,9.581,9.680,
     * 8.639,8.782,8.945,9.107,9.290,9.425,9.550,9.652,
     * 8.442,8.625,8.814,8.990,9.199,9.348,9.489,9.599,
     * 7.646,7.911,8.149,8.412,8.728,8.945,9.155,9.312,
     * 7.634,7.837,8.004,8.204,8.471,8.645,8.810,8.940,
     * 7.603,7.801,7.962,8.164,8.433,8.605,8.769,8.898,
     * 7.461,7.629,7.759,7.967,8.243,8.408,8.565,8.680,
     * 7.415,7.584,7.713,7.921,8.197,8.362,8.519,8.635/
      DATA ((EDEN(I,3,4,L,2),I=25,32),L=1,NZEN)/
     * 9.127,9.238,9.396,9.559,9.709,9.866,10.021,10.167,
     * 8.978,9.121,9.310,9.498,9.678,9.837,9.994,10.137,
     * 8.934,9.076,9.258,9.422,9.580,9.719,9.856,9.999,
     * 8.874,9.017,9.188,9.342,9.496,9.595,9.695,9.816,
     * 8.775,8.922,9.086,9.238,9.403,9.505,9.604,9.695,
     * 8.700,8.860,9.029,9.182,9.356,9.465,9.573,9.667,
     * 8.497,8.699,8.895,9.064,9.260,9.387,9.511,9.612,
     * 7.728,7.990,8.217,8.477,8.791,8.988,9.182,9.330,
     * 7.722,7.921,8.072,8.272,8.533,8.688,8.838,8.961,
     * 7.694,7.887,8.029,8.230,8.493,8.647,8.797,8.917,
     * 7.568,7.728,7.838,8.037,8.303,8.452,8.595,8.701,
     * 7.522,7.683,7.792,7.992,8.257,8.406,8.550,8.655/
      DATA ((EDEN(I,4,4,L,2),I=25,32),L=1,NZEN)/
     * 9.241,9.365,9.526,9.678,9.813,9.951,10.093,10.225,
     * 9.053,9.215,9.408,9.592,9.763,9.908,10.057,10.188,
     * 9.004,9.161,9.344,9.501,9.647,9.778,9.911,10.045,
     * 8.937,9.090,9.265,9.405,9.543,9.630,9.721,9.840,
     * 8.825,8.981,9.143,9.283,9.433,9.520,9.609,9.690,
     * 8.746,8.912,9.076,9.220,9.377,9.474,9.573,9.657,
     * 8.538,8.741,8.927,9.090,9.274,9.391,9.508,9.601,
     * 7.810,8.045,8.238,8.497,8.805,8.996,9.188,9.328,
     * 7.810,7.985,8.100,8.294,8.547,8.699,8.853,8.967,
     * 7.786,7.955,8.064,8.255,8.508,8.659,8.812,8.924,
     * 7.681,7.816,7.890,8.072,8.316,8.464,8.612,8.709,
     * 7.635,7.770,7.845,8.026,8.270,8.418,8.566,8.664/
      DATA ((EDEN(I,5,4,L,2),I=25,32),L=1,NZEN)/
     * 9.342,9.486,9.654,9.794,9.916,10.037,10.161,10.286,
     * 9.130,9.312,9.516,9.696,9.861,9.989,10.124,10.246,
     * 9.072,9.246,9.436,9.587,9.725,9.844,9.969,10.100,
     * 8.989,9.161,9.340,9.471,9.594,9.667,9.746,9.867,
     * 8.860,9.029,9.193,9.326,9.464,9.525,9.593,9.664,
     * 8.775,8.952,9.114,9.253,9.400,9.470,9.548,9.621,
     * 8.563,8.776,8.961,9.114,9.290,9.380,9.479,9.560,
     * 7.889,8.111,8.274,8.524,8.820,8.994,9.173,9.301,
     * 7.894,8.049,8.137,8.314,8.548,8.696,8.851,8.960,
     * 7.876,8.025,8.100,8.274,8.505,8.655,8.810,8.916,
     * 7.794,7.909,7.951,8.104,8.310,8.459,8.614,8.704,
     * 7.748,7.863,7.906,8.058,8.264,8.414,8.568,8.658/
      DATA ((EDEN(I,1,5,L,2),I=25,32),L=1,NZEN)/
     * 8.881,8.978,9.143,9.358,9.571,9.781,9.980,10.146,
     * 8.796,8.903,9.076,9.292,9.505,9.706,9.894,10.053,
     * 8.762,8.865,9.029,9.223,9.417,9.595,9.763,9.922,
     * 8.713,8.817,8.975,9.155,9.342,9.489,9.622,9.757,
     * 8.629,8.740,8.891,9.068,9.262,9.412,9.548,9.658,
     * 8.565,8.688,8.841,9.017,9.220,9.377,9.519,9.632,
     * 8.378,8.539,8.717,8.904,9.127,9.301,9.459,9.581,
     * 7.567,7.825,8.072,8.338,8.668,8.906,9.130,9.294,
     * 7.542,7.736,7.906,8.117,8.396,8.591,8.769,8.911,
     * 7.509,7.700,7.864,8.076,8.360,8.551,8.729,8.868,
     * 7.356,7.521,7.659,7.880,8.173,8.356,8.526,8.651,
     * 7.310,7.475,7.613,7.834,8.127,8.310,8.481,8.606/
      DATA ((EDEN(I,2,5,L,2),I=25,32),L=1,NZEN)/
     * 9.009,9.104,9.258,9.431,9.601,9.783,9.960,10.121,
     * 8.888,9.013,9.188,9.382,9.574,9.754,9.929,10.083,
     * 8.851,8.969,9.140,9.307,9.480,9.640,9.795,9.949,
     * 8.794,8.914,9.076,9.233,9.400,9.522,9.640,9.772,
     * 8.702,8.829,8.984,9.137,9.312,9.438,9.558,9.660,
     * 8.633,8.772,8.930,9.086,9.267,9.400,9.526,9.632,
     * 8.435,8.614,8.799,8.968,9.173,9.322,9.465,9.579,
     * 7.645,7.903,8.137,8.393,8.708,8.925,9.137,9.294,
     * 7.631,7.827,7.985,8.182,8.447,8.621,8.788,8.923,
     * 7.600,7.790,7.942,8.140,8.408,8.582,8.747,8.879,
     * 7.465,7.623,7.744,7.947,8.223,8.387,8.545,8.664,
     * 7.420,7.577,7.698,7.901,8.177,8.342,8.500,8.618/
      DATA ((EDEN(I,3,5,L,2),I=25,32),L=1,NZEN)/
     * 9.104,9.212,9.369,9.524,9.674,9.836,10.000,10.155,
     * 8.957,9.097,9.279,9.459,9.639,9.803,9.969,10.117,
     * 8.912,9.049,9.223,9.378,9.535,9.679,9.824,9.976,
     * 8.844,8.980,9.149,9.292,9.446,9.549,9.655,9.786,
     * 8.736,8.877,9.037,9.182,9.348,9.452,9.559,9.656,
     * 8.659,8.812,8.975,9.124,9.297,9.410,9.524,9.625,
     * 8.452,8.645,8.835,8.999,9.201,9.328,9.461,9.571,
     * 7.710,7.945,8.146,8.407,8.729,8.931,9.137,9.290,
     * 7.710,7.883,8.009,8.204,8.471,8.633,8.797,8.926,
     * 7.684,7.849,7.967,8.164,8.433,8.594,8.757,8.884,
     * 7.572,7.702,7.782,7.976,8.241,8.398,8.555,8.667,
     * 7.526,7.656,7.737,7.930,8.195,8.352,8.509,8.622/
      DATA ((EDEN(I,4,5,L,2),I=25,32),L=1,NZEN)/
     * 9.149,9.292,9.473,9.633,9.790,9.943,10.107,10.253,
     * 8.994,9.146,9.334,9.511,9.687,9.840,10.004,10.146,
     * 8.943,9.090,9.265,9.415,9.567,9.705,9.854,10.000,
     * 8.863,9.009,9.173,9.307,9.453,9.549,9.654,9.787,
     * 8.735,8.877,9.025,9.167,9.326,9.425,9.533,9.624,
     * 8.645,8.797,8.946,9.090,9.260,9.371,9.491,9.588,
     * 8.417,8.605,8.780,8.946,9.149,9.281,9.423,9.529,
     * 7.772,7.944,8.072,8.332,8.655,8.872,9.097,9.253,
     * 7.786,7.907,7.970,8.155,8.412,8.594,8.786,8.914,
     * 7.767,7.880,7.933,8.117,8.369,8.553,8.745,8.870,
     * 7.680,7.761,7.782,7.939,8.167,8.352,8.544,8.655,
     * 7.635,7.715,7.736,7.893,8.122,8.306,8.498,8.609/
      DATA ((EDEN(I,5,5,L,2),I=25,32),L=1,NZEN)/
     * 9.188,9.364,9.566,9.728,9.889,10.029,10.190,10.338,
     * 9.017,9.190,9.382,9.556,9.728,9.874,10.037,10.176,
     * 8.962,9.124,9.301,9.444,9.591,9.724,9.874,10.025,
     * 8.867,9.021,9.188,9.316,9.455,9.539,9.640,9.782,
     * 8.707,8.856,9.000,9.137,9.290,9.375,9.474,9.563,
     * 8.598,8.756,8.902,9.041,9.204,9.303,9.417,9.512,
     * 8.350,8.539,8.702,8.865,9.064,9.190,9.334,9.439,
     * 7.841,7.959,8.021,8.241,8.522,8.760,9.013,9.167,
     * 7.869,7.949,7.961,8.093,8.297,8.517,8.754,8.886,
     * 7.856,7.929,7.933,8.053,8.248,8.473,8.713,8.843,
     * 7.796,7.840,7.814,7.888,8.037,8.265,8.509,8.625,
     * 7.750,7.794,7.768,7.842,7.992,8.219,8.463,8.580/
      DATA ((EDEN(I,1,6,L,2),I=25,32),L=1,NZEN)/
     * 8.905,8.999,9.161,9.377,9.590,9.800,9.998,10.164,
     * 8.799,8.908,9.083,9.301,9.517,9.719,9.908,10.064,
     * 8.763,8.869,9.037,9.228,9.423,9.605,9.775,9.934,
     * 8.714,8.820,8.980,9.158,9.344,9.491,9.626,9.763,
     * 8.635,8.747,8.897,9.072,9.265,9.413,9.549,9.658,
     * 8.572,8.695,8.848,9.021,9.220,9.377,9.520,9.632,
     * 8.391,8.553,8.731,8.914,9.134,9.305,9.462,9.582,
     * 7.576,7.843,8.097,8.360,8.683,8.919,9.140,9.303,
     * 7.543,7.744,7.918,8.124,8.400,8.591,8.767,8.908,
     * 7.512,7.707,7.876,8.083,8.360,8.551,8.727,8.865,
     * 7.356,7.526,7.669,7.886,8.176,8.358,8.525,8.649,
     * 7.310,7.481,7.624,7.841,8.130,8.312,8.479,8.604/
      DATA ((EDEN(I,2,6,L,2),I=25,32),L=1,NZEN)/
     * 9.033,9.117,9.262,9.430,9.597,9.777,9.955,10.114,
     * 8.907,9.025,9.199,9.387,9.580,9.759,9.936,10.090,
     * 8.867,8.980,9.143,9.307,9.477,9.638,9.797,9.955,
     * 8.809,8.923,9.079,9.230,9.391,9.515,9.635,9.771,
     * 8.716,8.837,8.987,9.134,9.303,9.427,9.545,9.650,
     * 8.647,8.781,8.934,9.083,9.258,9.387,9.515,9.622,
     * 8.461,8.633,8.813,8.970,9.167,9.314,9.456,9.571,
     * 7.666,7.927,8.164,8.408,8.716,8.928,9.137,9.297,
     * 7.643,7.836,7.992,8.179,8.438,8.607,8.772,8.908,
     * 7.613,7.800,7.950,8.137,8.400,8.568,8.732,8.866,
     * 7.476,7.634,7.754,7.946,8.215,8.375,8.531,8.650,
     * 7.430,7.589,7.708,7.901,8.169,8.329,8.486,8.605/
      DATA ((EDEN(I,3,6,L,2),I=25,32),L=1,NZEN)/
     * 9.093,9.201,9.356,9.509,9.665,9.830,10.004,10.158,
     * 8.976,9.107,9.281,9.453,9.631,9.795,9.967,10.117,
     * 8.928,9.053,9.217,9.365,9.520,9.667,9.821,9.975,
     * 8.860,8.985,9.143,9.274,9.420,9.526,9.638,9.778,
     * 8.749,8.879,9.025,9.161,9.320,9.423,9.533,9.634,
     * 8.671,8.812,8.963,9.100,9.270,9.382,9.500,9.604,
     * 8.470,8.648,8.823,8.978,9.176,9.303,9.438,9.550,
     * 7.728,7.954,8.146,8.398,8.715,8.915,9.121,9.279,
     * 7.722,7.880,7.992,8.179,8.442,8.603,8.769,8.902,
     * 7.696,7.849,7.954,8.140,8.401,8.563,8.729,8.860,
     * 7.587,7.705,7.774,7.955,8.215,8.369,8.529,8.645,
     * 7.541,7.659,7.728,7.909,8.169,8.323,8.483,8.600/
      DATA ((EDEN(I,4,6,L,2),I=25,32),L=1,NZEN)/
     * 9.111,9.262,9.450,9.621,9.798,9.958,10.134,10.292,
     * 8.984,9.127,9.305,9.473,9.648,9.806,9.980,10.130,
     * 8.931,9.068,9.233,9.371,9.521,9.665,9.823,9.983,
     * 8.846,8.979,9.134,9.260,9.401,9.502,9.619,9.763,
     * 8.713,8.841,8.975,9.107,9.267,9.371,9.489,9.590,
     * 8.618,8.755,8.892,9.029,9.199,9.314,9.446,9.550,
     * 8.384,8.559,8.719,8.878,9.083,9.223,9.377,9.490,
     * 7.772,7.914,8.013,8.260,8.577,8.808,9.053,9.215,
     * 7.791,7.884,7.919,8.086,8.334,8.531,8.744,8.880,
     * 7.773,7.859,7.886,8.049,8.290,8.490,8.702,8.836,
     * 7.695,7.749,7.742,7.873,8.086,8.286,8.501,8.621,
     * 7.650,7.703,7.696,7.827,8.041,8.240,8.455,8.575/
      DATA ((EDEN(I,5,6,L,2),I=25,32),L=1,NZEN)/
     * 9.121,9.318,9.543,9.720,9.905,10.057,10.233,10.401,
     * 8.975,9.130,9.307,9.473,9.649,9.803,9.982,10.137,
     * 8.914,9.061,9.228,9.364,9.513,9.653,9.818,9.983,
     * 8.812,8.954,9.107,9.230,9.373,9.464,9.581,9.737,
     * 8.633,8.771,8.905,9.029,9.190,9.286,9.407,9.509,
     * 8.512,8.658,8.792,8.923,9.093,9.207,9.344,9.453,
     * 8.255,8.425,8.568,8.727,8.935,9.083,9.255,9.375,
     * 7.846,7.914,7.927,8.100,8.346,8.623,8.924,9.097,
     * 7.880,7.922,7.897,7.982,8.152,8.410,8.692,8.840,
     * 7.870,7.905,7.870,7.943,8.100,8.364,8.649,8.795,
     * 7.817,7.825,7.763,7.782,7.888,8.152,8.439,8.574,
     * 7.771,7.779,7.717,7.736,7.842,8.107,8.394,8.528/
      DATA ((EDEN(I,1,7,L,2),I=25,32),L=1,NZEN)/
     * 8.916,9.009,9.170,9.387,9.601,9.812,10.009,10.173,
     * 8.790,8.904,9.083,9.303,9.521,9.724,9.914,10.072,
     * 8.755,8.864,9.033,9.228,9.425,9.609,9.779,9.939,
     * 8.705,8.814,8.977,9.158,9.346,9.494,9.629,9.766,
     * 8.624,8.740,8.894,9.068,9.265,9.413,9.549,9.659,
     * 8.561,8.688,8.845,9.021,9.220,9.378,9.521,9.633,
     * 8.378,8.543,8.725,8.912,9.134,9.305,9.464,9.583,
     * 7.566,7.833,8.086,8.354,8.682,8.919,9.143,9.305,
     * 7.537,7.736,7.910,8.121,8.400,8.591,8.768,8.908,
     * 7.504,7.699,7.867,8.079,8.360,8.551,8.728,8.865,
     * 7.350,7.520,7.663,7.883,8.176,8.358,8.525,8.649,
     * 7.304,7.474,7.617,7.837,8.130,8.312,8.479,8.604/
      DATA ((EDEN(I,2,7,L,2),I=25,32),L=1,NZEN)/
     * 9.045,9.130,9.270,9.435,9.601,9.783,9.963,10.124,
     * 8.924,9.037,9.207,9.396,9.587,9.769,9.947,10.100,
     * 8.884,8.993,9.152,9.314,9.483,9.645,9.806,9.963,
     * 8.827,8.937,9.090,9.236,9.391,9.516,9.638,9.777,
     * 8.736,8.852,8.998,9.140,9.303,9.427,9.547,9.651,
     * 8.667,8.797,8.947,9.090,9.260,9.389,9.516,9.623,
     * 8.487,8.654,8.827,8.980,9.173,9.316,9.458,9.573,
     * 7.686,7.949,8.190,8.427,8.726,8.938,9.146,9.303,
     * 7.659,7.852,8.009,8.188,8.438,8.606,8.771,8.906,
     * 7.627,7.816,7.966,8.146,8.400,8.567,8.730,8.863,
     * 7.487,7.647,7.769,7.955,8.217,8.375,8.530,8.648,
     * 7.441,7.602,7.723,7.909,8.172,8.329,8.484,8.603/
      DATA ((EDEN(I,3,7,L,2),I=25,32),L=1,NZEN)/
     * 9.114,9.217,9.367,9.519,9.674,9.839,10.013,10.170,
     * 9.017,9.140,9.305,9.473,9.646,9.809,9.980,10.130,
     * 8.969,9.083,9.238,9.380,9.529,9.678,9.834,9.989,
     * 8.902,9.017,9.164,9.288,9.425,9.531,9.645,9.785,
     * 8.796,8.916,9.053,9.176,9.328,9.428,9.535,9.637,
     * 8.723,8.853,8.993,9.121,9.279,9.387,9.502,9.607,
     * 8.531,8.701,8.867,9.004,9.188,9.312,9.444,9.556,
     * 7.767,8.004,8.207,8.441,8.743,8.937,9.137,9.292,
     * 7.751,7.914,8.029,8.201,8.449,8.604,8.765,8.898,
     * 7.723,7.880,7.989,8.161,8.410,8.565,8.724,8.854,
     * 7.607,7.730,7.804,7.974,8.223,8.371,8.526,8.641,
     * 7.562,7.684,7.758,7.928,8.177,8.325,8.481,8.596/
      DATA ((EDEN(I,4,7,L,2),I=25,32),L=1,NZEN)/
     * 9.140,9.286,9.470,9.636,9.812,9.969,10.143,10.303,
     * 9.029,9.164,9.334,9.493,9.663,9.818,9.990,10.140,
     * 8.978,9.104,9.258,9.389,9.534,9.678,9.839,9.996,
     * 8.896,9.017,9.161,9.276,9.408,9.509,9.627,9.775,
     * 8.768,8.886,9.013,9.130,9.281,9.378,9.494,9.594,
     * 8.681,8.807,8.932,9.053,9.215,9.326,9.452,9.556,
     * 8.459,8.622,8.771,8.914,9.107,9.238,9.386,9.500,
     * 7.806,7.958,8.064,8.305,8.619,8.836,9.068,9.230,
     * 7.818,7.913,7.950,8.111,8.352,8.538,8.740,8.875,
     * 7.799,7.886,7.915,8.072,8.310,8.497,8.699,8.831,
     * 7.713,7.769,7.764,7.894,8.107,8.294,8.500,8.617,
     * 7.668,7.723,7.718,7.848,8.061,8.249,8.454,8.571/
      DATA ((EDEN(I,5,7,L,2),I=25,32),L=1,NZEN)/
     * 9.164,9.354,9.571,9.739,9.920,10.068,10.243,10.412,
     * 9.017,9.161,9.330,9.486,9.655,9.808,9.987,10.143,
     * 8.959,9.097,9.250,9.375,9.516,9.656,9.823,9.989,
     * 8.864,8.994,9.137,9.243,9.375,9.467,9.587,9.744,
     * 8.697,8.823,8.945,9.057,9.204,9.294,9.413,9.516,
     * 8.584,8.719,8.842,8.956,9.114,9.220,9.356,9.464,
     * 8.330,8.494,8.631,8.772,8.967,9.104,9.272,9.391,
     * 7.873,7.944,7.960,8.137,8.393,8.655,8.943,9.114,
     * 7.905,7.944,7.917,8.000,8.173,8.418,8.689,8.836,
     * 7.893,7.926,7.890,7.963,8.124,8.373,8.646,8.792,
     * 7.079,7.301,7.602,7.797,7.907,8.161,8.441,8.572,
     * 7.033,7.255,7.556,7.751,7.862,8.116,8.395,8.526/
      DATA ((EDEN(I,1,8,L,2),I=25,32),L=1,NZEN)/
     * 8.903,8.998,9.161,9.382,9.597,9.808,10.004,10.170,
     * 8.777,8.892,9.072,9.292,9.511,9.713,9.903,10.064,
     * 8.741,8.853,9.025,9.223,9.420,9.602,9.771,9.932,
     * 8.692,8.804,8.969,9.152,9.342,9.491,9.626,9.763,
     * 8.610,8.728,8.885,9.064,9.262,9.413,9.550,9.660,
     * 8.543,8.673,8.834,9.013,9.217,9.377,9.520,9.633,
     * 8.352,8.522,8.708,8.900,9.127,9.301,9.461,9.582,
     * 7.545,7.806,8.053,8.328,8.664,8.904,9.130,9.297,
     * 7.526,7.723,7.893,8.111,8.396,8.591,8.771,8.912,
     * 7.494,7.685,7.848,8.068,8.356,8.551,8.730,8.868,
     * 7.342,7.507,7.643,7.870,8.167,8.354,8.526,8.652,
     * 7.297,7.461,7.598,7.824,8.122,8.308,8.481,8.606/
      DATA ((EDEN(I,2,8,L,2),I=25,32),L=1,NZEN)/
     * 9.045,9.130,9.272,9.444,9.615,9.798,9.978,10.137,
     * 8.916,9.029,9.201,9.389,9.580,9.760,9.938,10.093,
     * 8.877,8.989,9.149,9.314,9.481,9.644,9.803,9.959,
     * 8.823,8.934,9.090,9.238,9.398,9.524,9.644,9.778,
     * 8.733,8.852,8.999,9.146,9.314,9.439,9.560,9.663,
     * 8.667,8.798,8.948,9.093,9.267,9.401,9.530,9.635,
     * 8.476,8.645,8.822,8.980,9.176,9.326,9.470,9.583,
     * 7.676,7.934,8.170,8.413,8.720,8.935,9.146,9.305,
     * 7.656,7.848,8.004,8.190,8.447,8.618,8.783,8.918,
     * 7.625,7.814,7.964,8.149,8.408,8.579,8.743,8.874,
     * 7.483,7.641,7.761,7.954,8.223,8.384,8.540,8.659,
     * 7.437,7.596,7.715,7.908,8.177,8.338,8.495,8.613/
      DATA ((EDEN(I,3,8,L,2),I=25,32),L=1,NZEN)/
     * 9.149,9.246,9.389,9.539,9.688,9.851,10.017,10.173,
     * 9.037,9.155,9.316,9.483,9.654,9.814,9.981,10.130,
     * 8.994,9.104,9.258,9.396,9.544,9.689,9.839,9.991,
     * 8.930,9.041,9.188,9.312,9.449,9.551,9.661,9.796,
     * 8.831,8.950,9.086,9.210,9.358,9.456,9.561,9.660,
     * 8.761,8.892,9.033,9.155,9.310,9.417,9.530,9.630,
     * 8.573,8.742,8.907,9.041,9.220,9.342,9.471,9.580,
     * 7.790,8.033,8.246,8.476,8.770,8.962,9.158,9.312,
     * 7.772,7.946,8.072,8.241,8.481,8.632,8.787,8.916,
     * 7.745,7.912,8.029,8.199,8.442,8.592,8.747,8.873,
     * 7.617,7.752,7.839,8.009,8.255,8.398,8.547,8.658,
     * 7.571,7.706,7.793,7.963,8.210,8.352,8.501,8.612/
      DATA ((EDEN(I,4,8,L,2),I=25,32),L=1,NZEN)/
     * 9.225,9.348,9.509,9.660,9.814,9.961,10.124,10.274,
     * 9.093,9.223,9.386,9.542,9.702,9.850,10.013,10.155,
     * 9.045,9.167,9.316,9.441,9.576,9.713,9.864,10.013,
     * 8.975,9.093,9.233,9.338,9.461,9.554,9.662,9.800,
     * 8.865,8.981,9.104,9.212,9.348,9.436,9.539,9.633,
     * 8.787,8.913,9.037,9.149,9.292,9.391,9.502,9.600,
     * 8.584,8.747,8.895,9.021,9.193,9.310,9.441,9.547,
     * 7.864,8.057,8.201,8.430,8.727,8.922,9.130,9.281,
     * 7.862,7.986,8.053,8.210,8.446,8.603,8.775,8.901,
     * 7.838,7.955,8.013,8.170,8.405,8.562,8.734,8.859,
     * 7.735,7.820,7.844,7.987,8.210,8.365,8.535,8.645,
     * 7.689,7.774,7.799,7.941,8.164,8.320,8.490,8.600/
      DATA ((EDEN(I,5,8,L,2),I=25,32),L=1,NZEN)/
     * 9.292,9.442,9.619,9.763,9.913,10.045,10.204,10.356,
     * 9.117,9.258,9.423,9.569,9.724,9.864,10.029,10.173,
     * 9.064,9.196,9.346,9.461,9.588,9.717,9.870,10.021,
     * 8.982,9.107,9.246,9.342,9.456,9.538,9.643,9.789,
     * 8.846,8.969,9.090,9.188,9.318,9.391,9.489,9.581,
     * 8.752,8.884,9.004,9.107,9.248,9.332,9.442,9.537,
     * 8.519,8.688,8.834,8.958,9.130,9.238,9.371,9.474,
     * 7.921,8.049,8.124,8.332,8.615,8.822,9.049,9.204,
     * 7.940,8.017,8.029,8.146,8.348,8.535,8.747,8.877,
     * 7.925,7.994,7.995,8.107,8.301,8.491,8.704,8.833,
     * 7.849,7.887,7.854,7.930,8.090,8.286,8.504,8.618,
     * 7.804,7.841,7.809,7.885,8.044,8.240,8.458,8.572/
      DATA ((EDEN(I,1,9,L,2),I=25,32),L=1,NZEN)/
     * 8.882,8.979,9.143,9.360,9.574,9.785,9.984,10.149,
     * 8.773,8.886,9.064,9.281,9.494,9.695,9.882,10.041,
     * 8.740,8.850,9.021,9.215,9.413,9.590,9.754,9.914,
     * 8.692,8.802,8.965,9.149,9.342,9.489,9.621,9.754,
     * 8.610,8.724,8.879,9.061,9.260,9.412,9.549,9.660,
     * 8.539,8.668,8.828,9.009,9.215,9.375,9.519,9.633,
     * 8.342,8.512,8.696,8.891,9.121,9.297,9.458,9.580,
     * 7.542,7.794,8.033,8.312,8.649,8.890,9.117,9.286,
     * 7.526,7.720,7.887,8.107,8.394,8.592,8.775,8.918,
     * 7.494,7.681,7.842,8.064,8.356,8.553,8.734,8.874,
     * 7.342,7.504,7.635,7.866,8.167,8.356,8.529,8.655,
     * 7.297,7.458,7.590,7.820,8.122,8.310,8.483,8.609/
      DATA ((EDEN(I,2,9,L,2),I=25,32),L=1,NZEN)/
     * 9.029,9.124,9.274,9.453,9.627,9.812,9.991,10.149,
     * 8.907,9.025,9.196,9.386,9.574,9.751,9.923,10.076,
     * 8.873,8.987,9.152,9.318,9.487,9.643,9.794,9.945,
     * 8.822,8.937,9.093,9.250,9.415,9.538,9.654,9.780,
     * 8.737,8.859,9.009,9.161,9.332,9.459,9.580,9.679,
     * 8.667,8.802,8.957,9.111,9.290,9.423,9.550,9.653,
     * 8.471,8.646,8.827,8.994,9.196,9.346,9.489,9.600,
     * 7.671,7.929,8.164,8.413,8.724,8.942,9.152,9.310,
     * 7.658,7.855,8.017,8.207,8.467,8.639,8.805,8.936,
     * 7.626,7.817,7.969,8.164,8.428,8.600,8.764,8.895,
     * 7.479,7.640,7.763,7.965,8.238,8.403,8.560,8.676,
     * 7.433,7.595,7.718,7.919,8.192,8.357,8.514,8.630/
      DATA ((EDEN(I,3,9,L,2),I=25,32),L=1,NZEN)/
     * 9.185,9.276,9.417,9.566,9.710,9.867,10.025,10.173,
     * 9.049,9.167,9.332,9.501,9.666,9.822,9.978,10.121,
     * 9.009,9.127,9.283,9.427,9.571,9.706,9.843,9.987,
     * 8.956,9.072,9.223,9.354,9.490,9.588,9.688,9.808,
     * 8.868,8.993,9.134,9.262,9.407,9.505,9.604,9.695,
     * 8.799,8.937,9.083,9.210,9.362,9.468,9.575,9.668,
     * 8.609,8.785,8.958,9.097,9.272,9.393,9.516,9.617,
     * 7.810,8.072,8.301,8.526,8.812,9.000,9.190,9.338,
     * 7.791,7.988,8.137,8.301,8.535,8.681,8.827,8.951,
     * 7.760,7.951,8.093,8.260,8.496,8.641,8.787,8.907,
     * 7.619,7.780,7.892,8.064,8.305,8.446,8.584,8.691,
     * 7.573,7.735,7.846,8.019,8.260,8.400,8.539,8.645/
      DATA ((EDEN(I,4,9,L,2),I=25,32),L=1,NZEN)/
     * 9.332,9.428,9.561,9.694,9.820,9.953,10.093,10.230,
     * 9.185,9.307,9.462,9.614,9.761,9.897,10.041,10.173,
     * 9.143,9.260,9.405,9.526,9.648,9.771,9.901,10.037,
     * 9.086,9.201,9.338,9.439,9.548,9.629,9.719,9.839,
     * 8.996,9.117,9.243,9.340,9.455,9.533,9.617,9.700,
     * 8.931,9.061,9.190,9.288,9.408,9.493,9.587,9.670,
     * 8.751,8.916,9.068,9.176,9.320,9.422,9.529,9.620,
     * 7.958,8.212,8.420,8.619,8.877,9.045,9.220,9.356,
     * 7.931,8.111,8.236,8.371,8.577,8.705,8.841,8.955,
     * 7.901,8.076,8.190,8.330,8.539,8.666,8.800,8.912,
     * 7.766,7.912,7.998,8.140,8.352,8.473,8.600,8.697,
     * 7.720,7.866,7.953,8.094,8.306,8.427,8.554,8.651/
      DATA ((EDEN(I,5,9,L,2),I=25,32),L=1,NZEN)/
     * 9.486,9.584,9.708,9.817,9.920,10.029,10.155,10.281,
     * 9.312,9.438,9.585,9.720,9.851,9.966,10.097,10.223,
     * 9.272,9.386,9.519,9.622,9.726,9.832,9.952,10.083,
     * 9.212,9.326,9.449,9.526,9.611,9.671,9.747,9.866,
     * 9.127,9.241,9.352,9.425,9.513,9.560,9.622,9.693,
     * 9.064,9.188,9.299,9.371,9.467,9.520,9.588,9.661,
     * 8.890,9.049,9.182,9.262,9.378,9.447,9.530,9.610,
     * 8.104,8.348,8.535,8.709,8.942,9.079,9.233,9.356,
     * 8.072,8.233,8.322,8.436,8.619,8.727,8.848,8.953,
     * 8.045,8.199,8.283,8.396,8.579,8.687,8.808,8.912,
     * 7.911,8.037,8.090,8.204,8.386,8.491,8.611,8.700,
     * 7.865,7.992,8.044,8.158,8.340,8.446,8.565,8.654/
      DATA ((EDEN(I,1,10,L,2),I=25,32),L=1,NZEN)/
     * 8.879,8.973,9.137,9.342,9.545,9.755,9.952,10.121,
     * 8.788,8.893,9.064,9.274,9.483,9.679,9.863,10.021,
     * 8.755,8.858,9.021,9.212,9.407,9.579,9.738,9.896,
     * 8.707,8.810,8.966,9.149,9.340,9.486,9.617,9.746,
     * 8.624,8.733,8.881,9.061,9.258,9.410,9.549,9.660,
     * 8.554,8.677,8.830,9.009,9.215,9.373,9.519,9.633,
     * 8.356,8.519,8.696,8.889,9.117,9.292,9.455,9.579,
     * 7.553,7.799,8.029,8.305,8.639,8.882,9.111,9.279,
     * 7.537,7.728,7.893,8.111,8.394,8.594,8.778,8.921,
     * 7.504,7.689,7.848,8.068,8.356,8.554,8.737,8.878,
     * 7.352,7.512,7.642,7.868,8.164,8.356,8.531,8.659,
     * 7.306,7.466,7.597,7.822,8.119,8.310,8.486,8.613/
      DATA ((EDEN(I,2,10,L,2),I=25,32),L=1,NZEN)/
     * 9.025,9.124,9.281,9.470,9.648,9.832,10.009,10.161,
     * 8.914,9.029,9.201,9.393,9.577,9.748,9.912,10.061,
     * 8.882,8.997,9.161,9.332,9.501,9.648,9.787,9.933,
     * 8.834,8.949,9.107,9.270,9.436,9.556,9.670,9.785,
     * 8.752,8.875,9.025,9.185,9.358,9.484,9.604,9.700,
     * 8.683,8.819,8.975,9.134,9.312,9.447,9.574,9.673,
     * 8.484,8.662,8.844,9.017,9.217,9.369,9.512,9.619,
     * 7.680,7.942,8.179,8.431,8.739,8.956,9.164,9.318,
     * 7.666,7.870,8.037,8.233,8.491,8.665,8.828,8.958,
     * 7.632,7.833,7.994,8.190,8.453,8.625,8.787,8.915,
     * 7.479,7.650,7.783,7.989,8.260,8.427,8.581,8.695,
     * 7.433,7.605,7.737,7.943,8.214,8.381,8.535,8.650/
      DATA ((EDEN(I,3,10,L,2),I=25,32),L=1,NZEN)/
     * 9.201,9.297,9.439,9.598,9.742,9.895,10.045,10.185,
     * 9.061,9.188,9.358,9.529,9.690,9.838,9.983,10.121,
     * 9.029,9.152,9.316,9.465,9.606,9.732,9.854,9.990,
     * 8.981,9.107,9.262,9.401,9.539,9.633,9.723,9.831,
     * 8.903,9.037,9.185,9.320,9.462,9.561,9.654,9.739,
     * 8.837,8.984,9.140,9.272,9.420,9.525,9.625,9.712,
     * 8.648,8.835,9.017,9.158,9.330,9.450,9.566,9.660,
     * 7.835,8.121,8.367,8.590,8.863,9.049,9.230,9.369,
     * 7.812,8.033,8.207,8.371,8.595,8.737,8.873,8.990,
     * 7.779,7.997,8.164,8.330,8.556,8.697,8.833,8.947,
     * 7.622,7.814,7.954,8.130,8.365,8.500,8.627,8.728,
     * 7.576,7.768,7.908,8.085,8.320,8.454,8.582,8.683/
      DATA ((EDEN(I,4,10,L,2),I=25,32),L=1,NZEN)/
     * 9.394,9.479,9.600,9.729,9.841,9.964,10.086,10.212,
     * 9.262,9.384,9.539,9.691,9.828,9.952,10.079,10.199,
     * 9.230,9.346,9.490,9.615,9.728,9.835,9.944,10.064,
     * 9.188,9.303,9.439,9.544,9.642,9.713,9.785,9.888,
     * 9.117,9.238,9.367,9.465,9.567,9.635,9.703,9.772,
     * 9.061,9.196,9.326,9.422,9.528,9.602,9.676,9.747,
     * 8.899,9.068,9.223,9.324,9.447,9.535,9.622,9.699,
     * 8.072,8.373,8.630,8.800,9.017,9.164,9.314,9.433,
     * 8.017,8.248,8.420,8.533,8.702,8.808,8.914,9.013,
     * 7.980,8.210,8.380,8.493,8.664,8.769,8.874,8.973,
     * 7.811,8.025,8.173,8.297,8.480,8.576,8.672,8.757,
     * 7.765,7.980,8.127,8.251,8.434,8.531,8.626,8.712/
      DATA ((EDEN(I,5,10,L,2),I=25,32),L=1,NZEN)/
     * 9.572,9.652,9.757,9.863,9.947,10.041,10.140,10.250,
     * 9.449,9.573,9.717,9.851,9.966,10.064,10.170,10.276,
     * 9.417,9.529,9.656,9.762,9.851,9.940,10.033,10.140,
     * 9.375,9.486,9.604,9.682,9.749,9.795,9.847,9.942,
     * 9.312,9.428,9.539,9.606,9.674,9.707,9.744,9.797,
     * 9.265,9.391,9.502,9.566,9.636,9.675,9.717,9.771,
     * 9.124,9.281,9.412,9.479,9.563,9.613,9.666,9.726,
     * 8.307,8.620,8.875,9.000,9.170,9.274,9.384,9.481,
     * 8.212,8.450,8.620,8.688,8.807,8.877,8.950,9.037,
     * 8.173,8.412,8.579,8.647,8.769,8.838,8.911,8.994,
     * 7.747,7.954,8.111,8.228,8.407,8.515,8.618,8.711,
     * 7.701,7.908,8.065,8.182,8.361,8.469,8.572,8.665/
      DATA ((EDEN(I,1,11,L,2),I=25,32),L=1,NZEN)/
     * 8.900,8.985,9.140,9.336,9.530,9.734,9.925,10.093,
     * 8.790,8.892,9.061,9.270,9.477,9.669,9.850,10.009,
     * 8.757,8.857,9.017,9.210,9.403,9.573,9.729,9.886,
     * 8.708,8.810,8.964,9.146,9.338,9.484,9.616,9.743,
     * 8.626,8.733,8.879,9.057,9.258,9.410,9.550,9.661,
     * 8.554,8.675,8.826,9.004,9.212,9.373,9.519,9.633,
     * 8.356,8.515,8.691,8.885,9.117,9.292,9.455,9.577,
     * 7.554,7.795,8.025,8.299,8.633,8.876,9.104,9.274,
     * 7.540,7.726,7.887,8.107,8.394,8.595,8.781,8.924,
     * 7.507,7.689,7.845,8.064,8.356,8.555,8.739,8.881,
     * 7.352,7.509,7.636,7.864,8.164,8.356,8.533,8.661,
     * 7.306,7.463,7.591,7.818,8.119,8.310,8.487,8.615/
      DATA ((EDEN(I,2,11,L,2),I=25,32),L=1,NZEN)/
     * 9.029,9.134,9.297,9.490,9.672,9.855,10.025,10.179,
     * 8.918,9.037,9.212,9.405,9.589,9.756,9.913,10.057,
     * 8.887,9.004,9.170,9.348,9.519,9.663,9.796,9.935,
     * 8.841,8.960,9.121,9.290,9.458,9.579,9.688,9.797,
     * 8.763,8.889,9.041,9.207,9.380,9.509,9.626,9.719,
     * 8.696,8.835,8.992,9.155,9.336,9.473,9.597,9.692,
     * 8.497,8.679,8.865,9.041,9.243,9.394,9.534,9.636,
     * 7.689,7.957,8.201,8.455,8.760,8.977,9.182,9.332,
     * 7.672,7.884,8.057,8.258,8.516,8.688,8.849,8.976,
     * 7.636,7.845,8.013,8.215,8.476,8.647,8.807,8.931,
     * 7.477,7.660,7.803,8.009,8.281,8.447,8.600,8.712,
     * 7.431,7.614,7.757,7.963,8.235,8.401,8.554,8.666/
      DATA ((EDEN(I,3,11,L,2),I=25,32),L=1,NZEN)/
     * 9.201,9.307,9.459,9.629,9.778,9.932,10.076,10.212,
     * 9.079,9.210,9.386,9.566,9.726,9.870,10.004,10.134,
     * 9.049,9.179,9.348,9.508,9.650,9.771,9.883,10.009,
     * 9.009,9.140,9.303,9.450,9.588,9.679,9.763,9.859,
     * 8.936,9.076,9.233,9.375,9.517,9.613,9.701,9.777,
     * 8.872,9.025,9.190,9.328,9.477,9.580,9.673,9.752,
     * 8.691,8.885,9.072,9.220,9.387,9.505,9.613,9.700,
     * 7.866,8.173,8.442,8.662,8.925,9.107,9.276,9.408,
     * 7.834,8.079,8.276,8.438,8.651,8.787,8.914,9.021,
     * 7.799,8.041,8.230,8.394,8.613,8.748,8.873,8.980,
     * 7.628,7.847,8.013,8.190,8.422,8.549,8.667,8.761,
     * 7.583,7.801,7.967,8.145,8.376,8.503,8.621,8.715/
      DATA ((EDEN(I,4,11,L,2),I=25,32),L=1,NZEN)/
     * 9.398,9.496,9.625,9.769,9.881,10.000,10.111,10.228,
     * 9.310,9.438,9.597,9.760,9.896,10.013,10.127,10.233,
     * 9.283,9.405,9.553,9.689,9.802,9.904,9.997,10.107,
     * 9.250,9.371,9.513,9.628,9.727,9.792,9.851,9.938,
     * 9.193,9.322,9.458,9.563,9.663,9.724,9.779,9.836,
     * 9.143,9.286,9.423,9.525,9.627,9.695,9.754,9.813,
     * 8.997,9.170,9.326,9.435,9.551,9.630,9.702,9.766,
     * 8.170,8.498,8.778,8.939,9.134,9.270,9.398,9.504,
     * 8.090,8.352,8.555,8.654,8.797,8.893,8.979,9.068,
     * 8.049,8.312,8.513,8.614,8.760,8.854,8.939,9.025,
     * 7.856,8.111,8.305,8.418,8.575,8.660,8.736,8.809,
     * 7.810,8.065,8.260,8.373,8.529,8.614,8.690,8.763/
      DATA ((EDEN(I,5,11,L,2),I=25,32),L=1,NZEN)/
     * 9.590,9.688,9.808,9.933,10.017,10.100,10.179,10.276,
     * 9.525,9.655,9.805,9.953,10.064,10.155,10.241,10.330,
     * 9.502,9.622,9.758,9.878,9.965,10.041,10.114,10.201,
     * 9.479,9.597,9.723,9.816,9.881,9.919,9.951,10.021,
     * 9.438,9.563,9.683,9.763,9.827,9.850,9.868,9.902,
     * 9.401,9.537,9.657,9.732,9.799,9.825,9.847,9.881,
     * 9.281,9.446,9.581,9.656,9.733,9.770,9.801,9.839,
     * 8.496,8.825,9.093,9.212,9.354,9.444,9.528,9.602,
     * 8.358,8.625,8.821,8.872,8.959,9.009,9.053,9.121,
     * 8.314,8.584,8.782,8.833,8.921,8.971,9.013,9.076,
     * 8.100,8.377,8.577,8.636,8.732,8.777,8.814,8.863,
     * 8.055,8.331,8.532,8.591,8.687,8.731,8.768,8.817/
      DATA ((EDEN(I,1,12,L,2),I=25,32),L=1,NZEN)/
     * 8.924,9.004,9.152,9.340,9.526,9.725,9.911,10.079,
     * 8.775,8.883,9.057,9.267,9.477,9.671,9.853,10.013,
     * 8.741,8.848,9.017,9.210,9.405,9.574,9.732,9.888,
     * 8.695,8.801,8.960,9.146,9.340,9.486,9.618,9.745,
     * 8.610,8.723,8.875,9.057,9.258,9.412,9.551,9.663,
     * 8.539,8.664,8.820,9.004,9.212,9.373,9.521,9.635,
     * 8.332,8.501,8.685,8.881,9.114,9.292,9.455,9.579,
     * 7.538,7.780,8.009,8.288,8.627,8.872,9.100,9.272,
     * 7.528,7.716,7.879,8.100,8.394,8.597,8.784,8.927,
     * 7.496,7.679,7.836,8.061,8.354,8.555,8.742,8.884,
     * 7.344,7.498,7.624,7.857,8.161,8.356,8.535,8.664,
     * 7.299,7.453,7.579,7.812,8.116,8.310,8.490,8.618/
      DATA ((EDEN(I,2,12,L,2),I=25,32),L=1,NZEN)/
     * 9.041,9.146,9.310,9.508,9.691,9.873,10.045,10.193,
     * 8.901,9.029,9.212,9.413,9.602,9.770,9.926,10.068,
     * 8.871,8.998,9.173,9.358,9.533,9.677,9.809,9.946,
     * 8.825,8.953,9.124,9.299,9.473,9.593,9.702,9.808,
     * 8.747,8.883,9.045,9.217,9.396,9.524,9.639,9.730,
     * 8.677,8.825,8.994,9.167,9.354,9.489,9.611,9.704,
     * 8.479,8.668,8.863,9.049,9.258,9.410,9.548,9.648,
     * 7.673,7.946,8.196,8.459,8.773,8.991,9.193,9.342,
     * 7.658,7.877,8.057,8.265,8.531,8.704,8.862,8.987,
     * 7.624,7.838,8.013,8.223,8.491,8.663,8.821,8.943,
     * 7.462,7.651,7.799,8.017,8.297,8.462,8.613,8.723,
     * 7.417,7.606,7.753,7.971,8.251,8.417,8.567,8.677/
      DATA ((EDEN(I,3,12,L,2),I=25,32),L=1,NZEN)/
     * 9.193,9.310,9.471,9.655,9.812,9.966,10.107,10.238,
     * 9.068,9.215,9.400,9.590,9.755,9.899,10.029,10.155,
     * 9.041,9.185,9.364,9.531,9.680,9.801,9.910,10.029,
     * 9.000,9.146,9.320,9.476,9.617,9.710,9.790,9.883,
     * 8.932,9.086,9.253,9.403,9.548,9.644,9.728,9.800,
     * 8.869,9.037,9.210,9.358,9.509,9.611,9.701,9.775,
     * 8.691,8.895,9.093,9.250,9.422,9.538,9.641,9.723,
     * 7.862,8.182,8.465,8.694,8.958,9.137,9.305,9.431,
     * 7.829,8.086,8.297,8.465,8.682,8.818,8.939,9.041,
     * 7.792,8.045,8.250,8.423,8.643,8.777,8.898,9.000,
     * 7.617,7.851,8.033,8.220,8.452,8.579,8.692,8.781,
     * 7.571,7.806,7.988,8.174,8.406,8.533,8.646,8.735/
      DATA ((EDEN(I,4,12,L,2),I=25,32),L=1,NZEN)/
     * 9.380,9.491,9.638,9.799,9.921,10.041,10.146,10.258,
     * 9.318,9.458,9.627,9.806,9.947,10.064,10.170,10.272,
     * 9.294,9.427,9.584,9.734,9.852,9.954,10.045,10.146,
     * 9.262,9.394,9.547,9.674,9.775,9.841,9.894,9.975,
     * 9.212,9.352,9.497,9.613,9.714,9.775,9.822,9.872,
     * 9.167,9.318,9.464,9.577,9.681,9.747,9.799,9.851,
     * 9.029,9.212,9.375,9.490,9.609,9.685,9.747,9.806,
     * 8.210,8.553,8.848,9.013,9.201,9.334,9.453,9.550,
     * 8.111,8.391,8.611,8.712,8.849,8.940,9.017,9.097,
     * 8.068,8.350,8.569,8.672,8.812,8.901,8.976,9.053,
     * 7.867,8.146,8.362,8.476,8.626,8.706,8.773,8.839,
     * 7.822,8.100,8.316,8.430,8.581,8.660,8.727,8.793/
      DATA ((EDEN(I,5,12,L,2),I=25,32),L=1,NZEN)/
     * 9.574,9.694,9.835,9.982,10.079,10.164,10.236,10.322,
     * 9.534,9.682,9.848,10.017,10.137,10.225,10.299,10.380,
     * 9.516,9.651,9.802,9.940,10.033,10.111,10.176,10.255,
     * 9.497,9.630,9.770,9.881,9.954,9.991,10.013,10.079,
     * 9.464,9.605,9.740,9.836,9.905,9.926,9.934,9.957,
     * 9.431,9.582,9.717,9.808,9.879,9.904,9.915,9.938,
     * 9.320,9.498,9.646,9.736,9.818,9.851,9.872,9.899,
     * 8.563,8.901,9.179,9.310,9.452,9.537,9.607,9.668,
     * 8.403,8.683,8.891,8.951,9.033,9.079,9.114,9.164,
     * 8.358,8.642,8.854,8.913,8.997,9.041,9.072,9.124,
     * 8.130,8.427,8.647,8.713,8.806,8.844,8.868,8.907,
     * 8.085,8.381,8.602,8.668,8.760,8.798,8.822,8.862/
      DATA ((EDEN(I,1,1,L,3),I=25,32),L=1,NZEN)/
     * 9.045,9.117,9.258,9.447,9.632,9.830,10.013,10.176,
     * 8.900,8.996,9.158,9.362,9.563,9.750,9.922,10.076,
     * 8.883,8.971,9.124,9.310,9.500,9.664,9.816,9.960,
     * 8.860,8.941,9.079,9.253,9.438,9.585,9.719,9.836,
     * 8.811,8.889,9.013,9.170,9.356,9.508,9.643,9.745,
     * 8.752,8.836,8.961,9.114,9.303,9.461,9.601,9.702,
     * 8.565,8.679,8.822,8.976,9.185,9.358,9.511,9.615,
     * 7.787,7.950,8.121,8.326,8.622,8.856,9.068,9.210,
     * 7.786,7.887,7.986,8.121,8.360,8.537,8.691,8.792,
     * 7.747,7.844,7.938,8.076,8.316,8.493,8.646,8.745,
     * 7.549,7.620,7.688,7.840,8.100,8.274,8.425,8.513,
     * 7.503,7.574,7.642,7.794,8.055,8.228,8.379,8.467/
      DATA ((EDEN(I,2,1,L,3),I=25,32),L=1,NZEN)/
     * 9.155,9.260,9.418,9.618,9.801,9.980,10.146,10.290,
     * 9.029,9.146,9.316,9.511,9.692,9.854,10.004,10.137,
     * 9.013,9.121,9.283,9.461,9.632,9.772,9.898,10.021,
     * 8.992,9.093,9.243,9.408,9.574,9.699,9.810,9.903,
     * 8.946,9.045,9.182,9.328,9.496,9.623,9.735,9.815,
     * 8.890,8.997,9.130,9.274,9.446,9.579,9.695,9.772,
     * 8.709,8.844,8.996,9.140,9.328,9.476,9.604,9.686,
     * 7.923,8.117,8.305,8.498,8.769,8.978,9.164,9.283,
     * 7.920,8.049,8.164,8.286,8.497,8.648,8.777,8.856,
     * 7.879,8.004,8.114,8.238,8.455,8.604,8.732,8.810,
     * 7.671,7.774,7.861,8.004,8.238,8.386,8.511,8.579,
     * 7.625,7.728,7.815,7.959,8.192,8.340,8.465,8.533/
      DATA ((EDEN(I,3,1,L,3),I=25,32),L=1,NZEN)/
     * 9.299,9.417,9.581,9.769,9.928,10.079,10.215,10.340,
     * 9.201,9.332,9.508,9.691,9.850,9.985,10.107,10.220,
     * 9.185,9.312,9.477,9.638,9.782,9.896,9.997,10.104,
     * 9.167,9.288,9.441,9.588,9.723,9.818,9.898,9.977,
     * 9.130,9.248,9.387,9.516,9.649,9.745,9.824,9.885,
     * 9.083,9.207,9.346,9.467,9.603,9.702,9.785,9.844,
     * 8.921,9.068,9.223,9.344,9.494,9.606,9.700,9.761,
     * 8.124,8.358,8.577,8.737,8.962,9.134,9.281,9.377,
     * 8.100,8.267,8.407,8.490,8.654,8.768,8.860,8.918,
     * 8.057,8.223,8.360,8.446,8.612,8.724,8.814,8.870,
     * 7.835,7.986,8.111,8.212,8.398,8.508,8.594,8.641,
     * 7.789,7.941,8.065,8.166,8.352,8.462,8.549,8.596/
      DATA ((EDEN(I,4,1,L,3),I=25,32),L=1,NZEN)/
     * 9.476,9.594,9.747,9.914,10.041,10.158,10.265,10.369,
     * 9.422,9.555,9.720,9.891,10.025,10.137,10.233,10.328,
     * 9.408,9.533,9.685,9.829,9.943,10.037,10.117,10.207,
     * 9.398,9.515,9.652,9.776,9.876,9.941,9.991,10.057,
     * 9.375,9.487,9.611,9.713,9.808,9.870,9.915,9.955,
     * 9.340,9.456,9.577,9.670,9.766,9.831,9.880,9.916,
     * 9.212,9.348,9.477,9.565,9.670,9.745,9.802,9.840,
     * 8.427,8.680,8.906,9.021,9.185,9.314,9.420,9.489,
     * 8.360,8.540,8.683,8.716,8.816,8.890,8.942,8.979,
     * 8.314,8.497,8.639,8.673,8.775,8.848,8.900,8.933,
     * 8.076,8.260,8.407,8.450,8.566,8.634,8.681,8.705,
     * 8.030,8.214,8.361,8.404,8.520,8.589,8.635,8.659/
      DATA ((EDEN(I,5,1,L,3),I=25,32),L=1,NZEN)/
     * 9.649,9.781,9.934,10.093,10.196,10.283,10.356,10.436,
     * 9.610,9.756,9.922,10.086,10.201,10.286,10.354,10.430,
     * 9.600,9.734,9.884,10.021,10.114,10.182,10.233,10.305,
     * 9.594,9.720,9.856,9.965,10.037,10.076,10.093,10.140,
     * 9.585,9.704,9.823,9.910,9.977,10.000,10.009,10.021,
     * 9.560,9.681,9.797,9.872,9.939,9.966,9.975,9.986,
     * 9.456,9.592,9.711,9.778,9.852,9.887,9.903,9.914,
     * 8.717,8.975,9.193,9.279,9.405,9.489,9.551,9.589,
     * 8.604,8.790,8.928,8.929,8.986,9.021,9.033,9.041,
     * 8.560,8.748,8.887,8.887,8.944,8.977,8.988,8.995,
     * 8.307,8.511,8.663,8.669,8.734,8.763,8.769,8.769,
     * 8.262,8.465,8.617,8.624,8.688,8.717,8.724,8.723/
      DATA ((EDEN(I,1,2,L,3),I=25,32),L=1,NZEN)/
     * 9.025,9.104,9.248,9.446,9.639,9.840,10.029,10.190,
     * 8.897,8.995,9.161,9.364,9.568,9.757,9.933,10.086,
     * 8.879,8.969,9.124,9.310,9.501,9.668,9.822,9.970,
     * 8.855,8.939,9.079,9.253,9.439,9.588,9.721,9.839,
     * 8.807,8.887,9.013,9.170,9.356,9.509,9.644,9.746,
     * 8.749,8.836,8.962,9.114,9.305,9.462,9.603,9.703,
     * 8.563,8.679,8.824,8.979,9.188,9.360,9.513,9.617,
     * 7.785,7.949,8.121,8.330,8.630,8.864,9.076,9.217,
     * 7.784,7.885,7.984,8.121,8.360,8.537,8.691,8.790,
     * 7.745,7.842,7.936,8.072,8.316,8.493,8.646,8.744,
     * 7.547,7.617,7.684,7.839,8.100,8.274,8.425,8.513,
     * 7.501,7.571,7.638,7.793,8.055,8.228,8.379,8.467/
      DATA ((EDEN(I,2,2,L,3),I=25,32),L=1,NZEN)/
     * 9.140,9.246,9.407,9.603,9.787,9.966,10.134,10.279,
     * 9.021,9.140,9.312,9.507,9.690,9.854,10.004,10.143,
     * 9.004,9.114,9.276,9.453,9.625,9.767,9.897,10.025,
     * 8.980,9.086,9.236,9.398,9.562,9.686,9.796,9.896,
     * 8.932,9.033,9.170,9.318,9.483,9.611,9.723,9.803,
     * 8.879,8.986,9.121,9.262,9.433,9.566,9.682,9.762,
     * 8.695,8.833,8.988,9.130,9.318,9.464,9.593,9.677,
     * 7.912,8.107,8.294,8.490,8.763,8.971,9.158,9.279,
     * 7.907,8.037,8.152,8.272,8.486,8.636,8.766,8.846,
     * 7.868,7.993,8.104,8.225,8.442,8.593,8.721,8.799,
     * 7.665,7.766,7.852,7.992,8.228,8.375,8.500,8.569,
     * 7.619,7.721,7.807,7.946,8.182,8.329,8.454,8.524/
      DATA ((EDEN(I,3,2,L,3),I=25,32),L=1,NZEN)/
     * 9.283,9.400,9.563,9.747,9.904,10.053,10.193,10.322,
     * 9.173,9.310,9.487,9.672,9.835,9.975,10.104,10.223,
     * 9.155,9.283,9.450,9.614,9.761,9.880,9.987,10.100,
     * 9.134,9.255,9.412,9.558,9.696,9.792,9.876,9.961,
     * 9.093,9.212,9.352,9.483,9.619,9.716,9.797,9.861,
     * 9.045,9.167,9.310,9.433,9.573,9.673,9.759,9.821,
     * 8.881,9.029,9.188,9.310,9.465,9.577,9.674,9.740,
     * 8.083,8.316,8.534,8.700,8.934,9.107,9.262,9.362,
     * 8.061,8.223,8.362,8.452,8.625,8.741,8.837,8.898,
     * 8.021,8.182,8.314,8.407,8.582,8.697,8.792,8.851,
     * 7.810,7.951,8.064,8.176,8.371,8.483,8.573,8.623,
     * 7.764,7.906,8.019,8.130,8.325,8.437,8.527,8.577/
      DATA ((EDEN(I,4,2,L,3),I=25,32),L=1,NZEN)/
     * 9.450,9.568,9.719,9.884,10.013,10.130,10.241,10.350,
     * 9.375,9.515,9.684,9.857,9.998,10.114,10.220,10.322,
     * 9.358,9.484,9.641,9.787,9.906,10.004,10.097,10.193,
     * 9.338,9.458,9.601,9.723,9.827,9.898,9.957,10.029,
     * 9.310,9.423,9.550,9.652,9.751,9.818,9.870,9.916,
     * 9.272,9.391,9.515,9.607,9.708,9.777,9.833,9.876,
     * 9.140,9.279,9.412,9.500,9.611,9.691,9.756,9.800,
     * 8.346,8.602,8.831,8.951,9.127,9.262,9.380,9.453,
     * 8.286,8.464,8.605,8.647,8.761,8.843,8.905,8.946,
     * 8.243,8.422,8.561,8.604,8.719,8.800,8.860,8.900,
     * 8.017,8.190,8.326,8.384,8.515,8.588,8.642,8.673,
     * 7.971,8.145,8.281,8.338,8.469,8.542,8.597,8.627/
      DATA ((EDEN(I,5,2,L,3),I=25,32),L=1,NZEN)/
     * 9.593,9.724,9.878,10.033,10.143,10.236,10.318,10.407,
     * 9.526,9.680,9.855,10.021,10.146,10.243,10.324,10.408,
     * 9.505,9.645,9.803,9.943,10.045,10.124,10.193,10.274,
     * 9.487,9.617,9.759,9.869,9.949,9.993,10.025,10.086,
     * 9.461,9.585,9.709,9.797,9.871,9.904,9.923,9.950,
     * 9.431,9.558,9.678,9.755,9.829,9.865,9.886,9.910,
     * 9.318,9.461,9.585,9.656,9.738,9.783,9.812,9.836,
     * 8.544,8.816,9.049,9.143,9.283,9.382,9.459,9.511,
     * 8.449,8.644,8.792,8.808,8.884,8.932,8.960,8.984,
     * 8.407,8.604,8.751,8.766,8.843,8.889,8.916,8.938,
     * 8.182,8.382,8.530,8.554,8.640,8.681,8.702,8.713,
     * 8.136,8.336,8.484,8.508,8.595,8.635,8.657,8.667/
      DATA ((EDEN(I,1,3,L,3),I=25,32),L=1,NZEN)/
     * 8.998,9.083,9.238,9.442,9.643,9.848,10.041,10.204,
     * 8.905,9.000,9.161,9.365,9.568,9.758,9.936,10.090,
     * 8.886,8.975,9.127,9.312,9.501,9.668,9.822,9.970,
     * 8.863,8.943,9.083,9.253,9.438,9.585,9.719,9.838,
     * 8.816,8.893,9.017,9.170,9.356,9.507,9.642,9.743,
     * 8.757,8.841,8.965,9.114,9.305,9.461,9.600,9.701,
     * 8.573,8.686,8.828,8.981,9.188,9.360,9.511,9.615,
     * 7.790,7.957,8.130,8.334,8.630,8.864,9.076,9.217,
     * 7.789,7.892,7.994,8.124,8.360,8.535,8.688,8.787,
     * 7.751,7.849,7.945,8.079,8.316,8.491,8.643,8.740,
     * 7.551,7.625,7.695,7.846,8.104,8.274,8.423,8.511,
     * 7.506,7.580,7.650,7.800,8.058,8.228,8.377,8.465/
      DATA ((EDEN(I,2,3,L,3),I=25,32),L=1,NZEN)/
     * 9.121,9.225,9.386,9.577,9.759,9.940,10.111,10.258,
     * 9.017,9.134,9.305,9.496,9.679,9.843,9.998,10.140,
     * 8.999,9.107,9.267,9.439,9.606,9.752,9.886,10.021,
     * 8.975,9.076,9.223,9.380,9.542,9.666,9.777,9.882,
     * 8.927,9.025,9.158,9.297,9.459,9.585,9.699,9.784,
     * 8.871,8.975,9.107,9.243,9.410,9.542,9.658,9.742,
     * 8.689,8.822,8.974,9.111,9.297,9.441,9.569,9.657,
     * 7.903,8.097,8.286,8.474,8.745,8.952,9.140,9.265,
     * 7.899,8.025,8.140,8.253,8.464,8.615,8.745,8.829,
     * 7.860,7.983,8.090,8.207,8.422,8.571,8.700,8.782,
     * 7.664,7.761,7.843,7.976,8.207,8.354,8.480,8.553,
     * 7.618,7.715,7.797,7.930,8.161,8.308,8.434,8.507/
      DATA ((EDEN(I,3,3,L,3),I=25,32),L=1,NZEN)/
     * 9.265,9.377,9.534,9.708,9.862,10.013,10.155,10.290,
     * 9.146,9.279,9.455,9.633,9.797,9.942,10.079,10.204,
     * 9.121,9.248,9.413,9.572,9.719,9.842,9.957,10.079,
     * 9.093,9.215,9.367,9.509,9.648,9.746,9.834,9.927,
     * 9.045,9.161,9.301,9.427,9.566,9.665,9.751,9.823,
     * 8.995,9.117,9.255,9.375,9.517,9.620,9.711,9.781,
     * 8.824,8.973,9.127,9.248,9.407,9.524,9.626,9.698,
     * 8.033,8.253,8.458,8.629,8.874,9.049,9.212,9.320,
     * 8.017,8.170,8.294,8.389,8.573,8.694,8.798,8.867,
     * 7.981,8.127,8.248,8.344,8.529,8.649,8.753,8.820,
     * 7.786,7.909,8.004,8.117,8.318,8.435,8.534,8.591,
     * 7.740,7.863,7.959,8.072,8.272,8.389,8.488,8.545/
      DATA ((EDEN(I,4,3,L,3),I=25,32),L=1,NZEN)/
     * 9.413,9.529,9.678,9.832,9.959,10.086,10.207,10.324,
     * 9.283,9.427,9.604,9.775,9.923,10.049,10.167,10.279,
     * 9.258,9.391,9.555,9.699,9.825,9.932,10.033,10.143,
     * 9.228,9.354,9.502,9.626,9.739,9.816,9.886,9.971,
     * 9.182,9.303,9.435,9.540,9.650,9.725,9.792,9.848,
     * 9.137,9.260,9.391,9.489,9.602,9.681,9.751,9.806,
     * 8.980,9.130,9.274,9.371,9.500,9.590,9.670,9.727,
     * 8.182,8.423,8.635,8.780,8.992,9.143,9.281,9.369,
     * 8.155,8.314,8.438,8.508,8.660,8.757,8.839,8.893,
     * 8.117,8.272,8.391,8.464,8.617,8.713,8.794,8.846,
     * 7.923,8.057,8.155,8.241,8.412,8.502,8.579,8.620,
     * 7.877,8.011,8.110,8.195,8.366,8.457,8.533,8.574/
      DATA ((EDEN(I,5,3,L,3),I=25,32),L=1,NZEN)/
     * 9.572,9.693,9.837,9.975,10.079,10.176,10.272,10.375,
     * 9.453,9.605,9.778,9.937,10.064,10.164,10.262,10.356,
     * 9.425,9.565,9.720,9.848,9.949,10.037,10.121,10.215,
     * 9.396,9.526,9.667,9.766,9.846,9.895,9.939,10.013,
     * 9.358,9.480,9.604,9.680,9.755,9.792,9.823,9.862,
     * 9.320,9.446,9.565,9.632,9.709,9.750,9.783,9.818,
     * 9.188,9.330,9.459,9.525,9.613,9.664,9.706,9.742,
     * 8.393,8.651,8.872,8.975,9.137,9.246,9.342,9.408,
     * 8.330,8.501,8.625,8.658,8.764,8.825,8.874,8.912,
     * 8.290,8.461,8.582,8.615,8.723,8.782,8.831,8.866,
     * 8.086,8.243,8.348,8.396,8.520,8.575,8.618,8.641,
     * 8.041,8.197,8.303,8.350,8.474,8.529,8.572,8.596/
      DATA ((EDEN(I,1,4,L,3),I=25,32),L=1,NZEN)/
     * 8.983,9.076,9.236,9.446,9.653,9.862,10.057,10.223,
     * 8.924,9.013,9.170,9.371,9.572,9.762,9.939,10.093,
     * 8.905,8.988,9.134,9.316,9.501,9.670,9.826,9.971,
     * 8.881,8.958,9.090,9.258,9.438,9.584,9.718,9.835,
     * 8.837,8.908,9.025,9.176,9.356,9.505,9.639,9.741,
     * 8.782,8.860,8.978,9.121,9.305,9.461,9.599,9.699,
     * 8.606,8.712,8.848,8.993,9.193,9.362,9.511,9.614,
     * 7.817,7.988,8.167,8.362,8.645,8.875,9.083,9.223,
     * 7.807,7.913,8.017,8.137,8.362,8.534,8.684,8.782,
     * 7.768,7.870,7.968,8.090,8.318,8.490,8.639,8.736,
     * 7.567,7.646,7.723,7.861,8.107,8.274,8.420,8.507,
     * 7.521,7.601,7.677,7.815,8.061,8.228,8.374,8.461/
      DATA ((EDEN(I,2,4,L,3),I=25,32),L=1,NZEN)/
     * 9.111,9.210,9.365,9.550,9.726,9.905,10.076,10.230,
     * 9.021,9.130,9.294,9.479,9.658,9.826,9.986,10.130,
     * 8.999,9.100,9.255,9.418,9.582,9.729,9.869,10.004,
     * 8.972,9.068,9.207,9.356,9.513,9.637,9.751,9.861,
     * 8.922,9.013,9.140,9.272,9.430,9.555,9.670,9.759,
     * 8.867,8.965,9.090,9.217,9.380,9.509,9.628,9.717,
     * 8.689,8.815,8.959,9.086,9.265,9.410,9.542,9.632,
     * 7.904,8.090,8.274,8.456,8.724,8.929,9.117,9.246,
     * 7.897,8.017,8.121,8.228,8.435,8.585,8.719,8.806,
     * 7.859,7.973,8.072,8.182,8.393,8.542,8.674,8.760,
     * 7.667,7.756,7.829,7.953,8.182,8.326,8.455,8.530,
     * 7.622,7.710,7.783,7.908,8.136,8.281,8.409,8.484/
      DATA ((EDEN(I,3,4,L,3),I=25,32),L=1,NZEN)/
     * 9.236,9.342,9.496,9.658,9.809,9.963,10.114,10.258,
     * 9.104,9.230,9.405,9.579,9.744,9.893,10.037,10.173,
     * 9.076,9.196,9.358,9.511,9.660,9.787,9.911,10.041,
     * 9.041,9.155,9.305,9.442,9.584,9.685,9.781,9.885,
     * 8.989,9.097,9.228,9.352,9.497,9.598,9.693,9.773,
     * 8.929,9.045,9.179,9.297,9.446,9.551,9.651,9.730,
     * 8.748,8.892,9.041,9.164,9.334,9.453,9.565,9.646,
     * 7.973,8.167,8.346,8.528,8.792,8.975,9.149,9.267,
     * 7.973,8.093,8.193,8.299,8.501,8.629,8.747,8.825,
     * 7.937,8.053,8.146,8.253,8.456,8.585,8.702,8.779,
     * 7.762,7.847,7.905,8.025,8.243,8.369,8.484,8.550,
     * 7.716,7.801,7.859,7.980,8.197,8.323,8.439,8.504/
      DATA ((EDEN(I,4,4,L,3),I=25,32),L=1,NZEN)/
     * 9.344,9.465,9.622,9.773,9.909,10.045,10.182,10.314,
     * 9.173,9.316,9.494,9.660,9.815,9.951,10.090,10.217,
     * 9.140,9.274,9.438,9.580,9.715,9.832,9.951,10.076,
     * 9.100,9.225,9.373,9.496,9.621,9.708,9.795,9.895,
     * 9.037,9.152,9.281,9.391,9.519,9.605,9.691,9.762,
     * 8.978,9.097,9.225,9.330,9.462,9.556,9.648,9.717,
     * 8.792,8.941,9.086,9.196,9.348,9.456,9.562,9.633,
     * 8.041,8.217,8.367,8.548,8.807,8.984,9.155,9.262,
     * 8.045,8.146,8.215,8.318,8.513,8.637,8.756,8.826,
     * 8.013,8.107,8.167,8.270,8.468,8.593,8.711,8.780,
     * 7.860,7.920,7.942,8.049,8.255,8.378,8.494,8.551,
     * 7.815,7.874,7.896,8.003,8.210,8.333,8.448,8.506/
      DATA ((EDEN(I,5,4,L,3),I=25,32),L=1,NZEN)/
     * 9.438,9.579,9.744,9.884,10.004,10.124,10.248,10.371,
     * 9.248,9.410,9.593,9.754,9.898,10.017,10.146,10.265,
     * 9.204,9.354,9.524,9.654,9.775,9.879,9.990,10.114,
     * 9.152,9.292,9.444,9.551,9.659,9.724,9.795,9.895,
     * 9.079,9.207,9.336,9.430,9.539,9.598,9.661,9.721,
     * 9.013,9.146,9.274,9.364,9.477,9.543,9.612,9.671,
     * 8.828,8.986,9.130,9.225,9.362,9.442,9.525,9.585,
     * 8.107,8.276,8.408,8.577,8.823,8.978,9.134,9.228,
     * 8.114,8.199,8.241,8.330,8.509,8.626,8.743,8.808,
     * 8.086,8.167,8.199,8.286,8.464,8.582,8.699,8.762,
     * 7.957,7.998,7.991,8.068,8.243,8.364,8.484,8.535,
     * 7.911,7.952,7.945,8.022,8.197,8.318,8.439,8.490/
      DATA ((EDEN(I,1,5,L,3),I=25,32),L=1,NZEN)/
     * 8.996,9.086,9.248,9.462,9.672,9.880,10.076,10.238,
     * 8.928,9.017,9.176,9.375,9.576,9.766,9.943,10.097,
     * 8.908,8.992,9.137,9.318,9.504,9.673,9.829,9.975,
     * 8.886,8.961,9.093,9.258,9.438,9.584,9.717,9.836,
     * 8.843,8.913,9.029,9.179,9.356,9.505,9.638,9.739,
     * 8.792,8.868,8.985,9.127,9.307,9.461,9.598,9.698,
     * 8.621,8.724,8.857,9.000,9.199,9.365,9.512,9.614,
     * 7.829,8.009,8.190,8.382,8.664,8.890,9.093,9.230,
     * 7.813,7.920,8.025,8.143,8.362,8.533,8.681,8.780,
     * 7.775,7.877,7.977,8.097,8.320,8.490,8.636,8.732,
     * 7.572,7.654,7.734,7.868,8.107,8.274,8.418,8.504,
     * 7.526,7.608,7.688,7.822,8.061,8.228,8.373,8.458/
      DATA ((EDEN(I,2,5,L,3),I=25,32),L=1,NZEN)/
     * 9.124,9.212,9.360,9.533,9.703,9.883,10.057,10.212,
     * 9.017,9.121,9.283,9.462,9.642,9.812,9.975,10.124,
     * 8.991,9.090,9.241,9.398,9.561,9.709,9.851,9.994,
     * 8.962,9.053,9.190,9.332,9.489,9.613,9.730,9.844,
     * 8.912,8.999,9.121,9.248,9.403,9.528,9.645,9.738,
     * 8.857,8.949,9.072,9.193,9.354,9.483,9.604,9.696,
     * 8.682,8.802,8.942,9.064,9.241,9.386,9.519,9.613,
     * 7.898,8.079,8.258,8.436,8.705,8.910,9.100,9.230,
     * 7.890,8.000,8.097,8.201,8.410,8.560,8.695,8.787,
     * 7.854,7.959,8.049,8.155,8.365,8.516,8.651,8.740,
     * 7.667,7.744,7.806,7.928,8.158,8.303,8.435,8.512,
     * 7.622,7.698,7.760,7.882,8.113,8.257,8.389,8.466/
      DATA ((EDEN(I,3,5,L,3),I=25,32),L=1,NZEN)/
     * 9.210,9.314,9.465,9.622,9.775,9.933,10.093,10.243,
     * 9.076,9.199,9.365,9.534,9.702,9.854,10.009,10.152,
     * 9.045,9.158,9.316,9.461,9.613,9.744,9.877,10.017,
     * 9.009,9.114,9.255,9.386,9.530,9.633,9.737,9.851,
     * 8.945,9.045,9.170,9.290,9.438,9.540,9.643,9.732,
     * 8.885,8.992,9.117,9.230,9.386,9.494,9.601,9.688,
     * 8.957,9.053,9.127,9.161,9.301,9.408,9.509,9.572,
     * 7.944,8.114,8.265,8.452,8.728,8.916,9.100,9.228,
     * 7.948,8.041,8.114,8.223,8.436,8.573,8.702,8.788,
     * 7.915,8.004,8.064,8.176,8.393,8.529,8.657,8.740,
     * 7.754,7.806,7.831,7.951,8.179,8.312,8.439,8.513,
     * 7.709,7.760,7.785,7.905,8.133,8.266,8.394,8.467/
      DATA ((EDEN(I,4,5,L,3),I=25,32),L=1,NZEN)/
     * 9.253,9.387,9.561,9.723,9.882,10.033,10.193,10.336,
     * 9.104,9.238,9.410,9.571,9.732,9.877,10.033,10.173,
     * 9.064,9.190,9.346,9.481,9.623,9.751,9.888,10.029,
     * 9.017,9.130,9.267,9.386,9.520,9.616,9.721,9.836,
     * 8.942,9.041,9.155,9.265,9.403,9.502,9.607,9.691,
     * 8.874,8.979,9.090,9.196,9.342,9.449,9.561,9.644,
     * 8.669,8.803,8.933,9.049,9.220,9.344,9.473,9.559,
     * 7.980,8.093,8.185,8.375,8.657,8.857,9.061,9.185,
     * 7.997,8.041,8.053,8.161,8.373,8.526,8.682,8.767,
     * 7.971,8.009,8.009,8.114,8.326,8.480,8.636,8.719,
     * 7.843,7.842,7.805,7.898,8.100,8.258,8.420,8.491,
     * 7.797,7.796,7.759,7.852,8.055,8.212,8.374,8.446/
      DATA ((EDEN(I,5,5,L,3),I=25,32),L=1,NZEN)/
     * 9.288,9.455,9.647,9.811,9.972,10.114,10.272,10.417,
     * 9.130,9.274,9.444,9.598,9.751,9.892,10.049,10.190,
     * 9.086,9.217,9.369,9.493,9.624,9.747,9.887,10.029,
     * 9.025,9.140,9.270,9.375,9.496,9.580,9.679,9.801,
     * 8.928,9.025,9.127,9.220,9.346,9.430,9.528,9.609,
     * 8.846,8.949,9.049,9.140,9.272,9.364,9.471,9.551,
     * 8.615,8.747,8.867,8.971,9.134,9.246,9.373,9.458,
     * 8.025,8.083,8.104,8.265,8.519,8.734,8.960,9.086,
     * 8.057,8.057,8.013,8.072,8.241,8.431,8.631,8.722,
     * 8.037,8.029,7.976,8.025,8.188,8.380,8.585,8.675,
     * 7.941,7.897,7.807,7.818,7.939,8.146,8.364,8.444,
     * 7.895,7.851,7.761,7.772,7.893,8.100,8.318,8.398/
      DATA ((EDEN(I,1,6,L,3),I=25,32),L=1,NZEN)/
     * 9.021,9.111,9.267,9.480,9.691,9.898,10.093,10.255,
     * 8.926,9.021,9.179,9.384,9.588,9.779,9.957,10.107,
     * 8.905,8.990,9.137,9.320,9.508,9.680,9.839,9.985,
     * 8.883,8.960,9.093,9.260,9.438,9.585,9.719,9.839,
     * 8.841,8.913,9.033,9.179,9.358,9.507,9.638,9.739,
     * 8.792,8.869,8.987,9.130,9.310,9.464,9.600,9.698,
     * 8.630,8.732,8.865,9.004,9.204,9.369,9.516,9.617,
     * 7.838,8.021,8.210,8.398,8.678,8.903,9.107,9.241,
     * 7.814,7.921,8.025,8.143,8.362,8.533,8.679,8.775,
     * 7.774,7.877,7.977,8.097,8.320,8.490,8.635,8.729,
     * 7.571,7.654,7.734,7.869,8.111,8.276,8.418,8.502,
     * 7.525,7.608,7.688,7.823,8.065,8.231,8.373,8.457/
      DATA ((EDEN(I,2,6,L,3),I=25,32),L=1,NZEN)/
     * 9.143,9.225,9.364,9.530,9.695,9.874,10.049,10.210,
     * 9.029,9.130,9.286,9.464,9.643,9.815,9.983,10.130,
     * 9.000,9.093,9.241,9.394,9.556,9.708,9.855,10.000,
     * 8.971,9.057,9.190,9.326,9.479,9.603,9.723,9.841,
     * 8.922,9.004,9.121,9.241,9.393,9.516,9.634,9.728,
     * 8.870,8.957,9.076,9.188,9.342,9.471,9.593,9.687,
     * 8.704,8.817,8.948,9.061,9.236,9.377,9.511,9.605,
     * 7.918,8.100,8.281,8.450,8.713,8.915,9.107,9.236,
     * 7.901,8.009,8.100,8.196,8.398,8.547,8.682,8.774,
     * 7.863,7.965,8.053,8.149,8.356,8.504,8.637,8.727,
     * 7.678,7.752,7.812,7.924,8.146,8.290,8.422,8.501,
     * 7.632,7.706,7.766,7.879,8.100,8.244,8.376,8.455/
      DATA ((EDEN(I,3,6,L,3),I=25,32),L=1,NZEN)/
     * 9.201,9.303,9.452,9.606,9.762,9.925,10.093,10.248,
     * 9.090,9.201,9.360,9.522,9.691,9.846,10.009,10.152,
     * 9.057,9.161,9.305,9.444,9.592,9.728,9.870,10.017,
     * 9.017,9.111,9.243,9.364,9.504,9.609,9.718,9.838,
     * 8.955,9.041,9.158,9.265,9.410,9.512,9.618,9.710,
     * 8.899,8.992,9.104,9.207,9.358,9.465,9.575,9.667,
     * 8.718,8.840,8.972,9.079,9.248,9.369,9.491,9.585,
     * 7.960,8.121,8.267,8.444,8.718,8.905,9.090,9.217,
     * 7.958,8.037,8.097,8.196,8.407,8.542,8.675,8.765,
     * 7.926,7.999,8.045,8.149,8.362,8.497,8.630,8.717,
     * 7.768,7.805,7.816,7.925,8.149,8.283,8.415,8.491,
     * 7.722,7.759,7.770,7.880,8.103,8.238,8.369,8.446/
      DATA ((EDEN(I,4,6,L,3),I=25,32),L=1,NZEN)/
     * 9.212,9.356,9.535,9.706,9.881,10.041,10.215,10.373,
     * 9.086,9.212,9.371,9.525,9.688,9.841,10.009,10.155,
     * 9.049,9.158,9.301,9.430,9.573,9.708,9.858,10.009,
     * 8.997,9.093,9.217,9.328,9.462,9.565,9.681,9.808,
     * 8.915,8.999,9.097,9.199,9.338,9.444,9.561,9.653,
     * 8.843,8.932,9.029,9.127,9.276,9.389,9.513,9.605,
     * 8.631,8.751,8.868,8.978,9.152,9.283,9.425,9.520,
     * 7.971,8.053,8.107,8.292,8.580,8.792,9.013,9.146,
     * 7.993,8.004,7.985,8.079,8.290,8.459,8.636,8.730,
     * 7.968,7.973,7.943,8.033,8.243,8.413,8.592,8.684,
     * 7.851,7.817,7.748,7.820,8.009,8.188,8.375,8.455,
     * 7.806,7.771,7.702,7.774,7.963,8.142,8.329,8.409/
      DATA ((EDEN(I,5,6,L,3),I=25,32),L=1,NZEN)/
     * 9.220,9.412,9.627,9.801,9.984,10.137,10.314,10.479,
     * 9.076,9.210,9.367,9.511,9.667,9.819,9.996,10.149,
     * 9.033,9.152,9.290,9.405,9.537,9.670,9.827,9.988,
     * 8.965,9.064,9.182,9.279,9.401,9.496,9.613,9.749,
     * 8.848,8.935,9.025,9.107,9.233,9.330,9.452,9.547,
     * 8.750,8.844,8.937,9.017,9.146,9.258,9.389,9.484,
     * 8.501,8.622,8.731,8.828,8.994,9.130,9.288,9.389,
     * 8.013,8.017,7.985,8.104,8.330,8.589,8.865,9.009,
     * 8.053,8.013,7.928,7.943,8.079,8.312,8.561,8.670,
     * 0.000,0.000,0.000,7.896,8.021,8.258,8.513,8.621,
     * 0.000,0.000,0.000,7.694,7.769,8.017,8.286,8.386,
     * 0.000,0.000,0.000,7.648,7.724,7.971,8.240,8.340/
      DATA ((EDEN(I,1,7,L,3),I=25,32),L=1,NZEN)/
     * 9.029,9.117,9.274,9.489,9.700,9.907,10.100,10.265,
     * 8.915,9.013,9.173,9.380,9.588,9.781,9.961,10.114,
     * 8.894,8.982,9.134,9.318,9.509,9.682,9.843,9.990,
     * 8.870,8.950,9.086,9.255,9.438,9.587,9.720,9.841,
     * 8.827,8.903,9.025,9.176,9.358,9.507,9.639,9.740,
     * 8.780,8.860,8.980,9.124,9.310,9.464,9.600,9.699,
     * 8.617,8.723,8.857,9.004,9.201,9.369,9.517,9.618,
     * 7.825,8.004,8.196,8.391,8.678,8.904,9.107,9.243,
     * 7.804,7.910,8.013,8.134,8.362,8.533,8.680,8.777,
     * 7.764,7.866,7.964,8.090,8.318,8.489,8.635,8.729,
     * 7.561,7.642,7.719,7.860,8.107,8.274,8.418,8.504,
     * 7.515,7.597,7.674,7.814,8.061,8.228,8.373,8.458/
      DATA ((EDEN(I,2,7,L,3),I=25,32),L=1,NZEN)/
     * 9.158,9.236,9.371,9.537,9.702,9.881,10.057,10.215,
     * 9.045,9.143,9.297,9.471,9.651,9.823,9.993,10.140,
     * 9.021,9.111,9.250,9.401,9.560,9.713,9.862,10.009,
     * 8.992,9.072,9.199,9.330,9.480,9.605,9.725,9.844,
     * 8.943,9.017,9.130,9.246,9.396,9.519,9.635,9.729,
     * 8.893,8.975,9.086,9.196,9.346,9.474,9.595,9.689,
     * 8.732,8.838,8.964,9.072,9.241,9.382,9.513,9.609,
     * 7.941,8.124,8.307,8.468,8.724,8.925,9.114,9.243,
     * 7.919,8.025,8.114,8.204,8.401,8.548,8.681,8.772,
     * 7.882,7.982,8.068,8.158,8.358,8.504,8.637,8.725,
     * 7.692,7.766,7.827,7.933,8.149,8.292,8.422,8.500,
     * 7.646,7.721,7.782,7.888,8.103,8.246,8.376,8.454/
      DATA ((EDEN(I,3,7,L,3),I=25,32),L=1,NZEN)/
     * 9.220,9.318,9.464,9.616,9.772,9.933,10.104,10.258,
     * 9.130,9.236,9.386,9.542,9.703,9.857,10.017,10.164,
     * 9.097,9.193,9.328,9.458,9.601,9.737,9.881,10.025,
     * 9.057,9.143,9.267,9.377,9.509,9.613,9.723,9.844,
     * 9.000,9.079,9.185,9.281,9.417,9.516,9.621,9.713,
     * 8.945,9.029,9.137,9.228,9.365,9.471,9.580,9.670,
     * 8.779,8.889,9.009,9.104,9.260,9.377,9.498,9.590,
     * 8.009,8.176,8.328,8.487,8.744,8.925,9.107,9.233,
     * 7.994,8.076,8.134,8.217,8.413,8.544,8.673,8.762,
     * 7.960,8.037,8.086,8.173,8.371,8.500,8.628,8.714,
     * 7.794,7.836,7.855,7.949,8.161,8.288,8.413,8.489,
     * 7.748,7.791,7.809,7.904,8.116,8.242,8.368,8.443/
      DATA ((EDEN(I,4,7,L,3),I=25,32),L=1,NZEN)/
     * 9.246,9.382,9.555,9.723,9.897,10.053,10.225,10.384,
     * 9.134,9.248,9.398,9.545,9.703,9.853,10.021,10.167,
     * 9.093,9.196,9.330,9.449,9.584,9.719,9.870,10.021,
     * 9.045,9.134,9.246,9.346,9.473,9.573,9.688,9.816,
     * 8.970,9.045,9.137,9.223,9.354,9.453,9.567,9.659,
     * 8.905,8.984,9.072,9.155,9.292,9.400,9.521,9.612,
     * 8.707,8.813,8.918,9.013,9.173,9.299,9.436,9.529,
     * 8.013,8.104,8.170,8.342,8.619,8.822,9.033,9.164,
     * 8.029,8.041,8.025,8.111,8.312,8.470,8.636,8.728,
     * 8.000,8.009,7.980,8.064,8.265,8.423,8.591,8.680,
     * 7.874,7.843,7.778,7.846,8.033,8.201,8.373,8.453,
     * 7.828,7.797,7.732,7.801,7.988,8.156,8.327,8.408/
      DATA ((EDEN(I,5,7,L,3),I=25,32),L=1,NZEN)/
     * 9.270,9.447,9.654,9.820,9.999,10.146,10.322,10.489,
     * 9.124,9.246,9.391,9.524,9.674,9.823,10.000,10.155,
     * 9.079,9.188,9.316,9.418,9.543,9.676,9.835,9.996,
     * 9.021,9.111,9.215,9.299,9.412,9.502,9.620,9.760,
     * 8.914,8.990,9.072,9.140,9.253,9.346,9.464,9.559,
     * 8.823,8.905,8.986,9.053,9.176,9.279,9.405,9.500,
     * 8.583,8.695,8.797,8.877,9.029,9.158,9.307,9.407,
     * 8.049,8.057,8.029,8.152,8.384,8.626,8.889,9.029,
     * 8.083,8.041,7.958,7.972,8.111,8.326,8.562,8.669,
     * 0.000,0.000,7.922,7.925,8.053,8.274,8.516,8.621,
     * 0.000,0.000,0.000,7.717,7.800,8.037,8.290,8.387,
     * 0.000,0.000,0.000,7.671,7.754,7.992,8.244,8.342/
      DATA ((EDEN(I,1,8,L,3),I=25,32),L=1,NZEN)/
     * 9.017,9.107,9.267,9.481,9.694,9.902,10.097,10.260,
     * 8.904,9.000,9.167,9.373,9.580,9.772,9.951,10.107,
     * 8.884,8.974,9.127,9.314,9.504,9.677,9.836,9.984,
     * 8.860,8.942,9.083,9.253,9.436,9.585,9.719,9.839,
     * 8.816,8.893,9.017,9.170,9.354,9.505,9.639,9.739,
     * 8.762,8.846,8.970,9.117,9.305,9.461,9.599,9.699,
     * 8.592,8.701,8.840,8.990,9.196,9.364,9.513,9.615,
     * 7.804,7.979,8.161,8.364,8.656,8.887,9.097,9.233,
     * 7.792,7.895,7.997,8.127,8.358,8.531,8.682,8.780,
     * 7.753,7.852,7.949,8.079,8.316,8.489,8.638,8.732,
     * 7.553,7.629,7.703,7.850,8.104,8.274,8.420,8.505,
     * 7.507,7.584,7.658,7.804,8.058,8.228,8.374,8.459/
      DATA ((EDEN(I,2,8,L,3),I=25,32),L=1,NZEN)/
     * 9.155,9.236,9.375,9.547,9.718,9.897,10.072,10.230,
     * 9.041,9.140,9.294,9.470,9.649,9.819,9.985,10.134,
     * 9.017,9.111,9.250,9.405,9.563,9.714,9.860,10.004,
     * 8.990,9.072,9.204,9.338,9.490,9.615,9.733,9.850,
     * 8.942,9.021,9.137,9.255,9.407,9.530,9.648,9.741,
     * 8.890,8.975,9.090,9.201,9.358,9.487,9.609,9.700,
     * 8.721,8.833,8.963,9.076,9.248,9.391,9.524,9.618,
     * 7.931,8.111,8.288,8.458,8.719,8.922,9.114,9.243,
     * 7.919,8.025,8.117,8.210,8.412,8.560,8.694,8.785,
     * 7.880,7.983,8.072,8.167,8.369,8.516,8.649,8.739,
     * 7.688,7.766,7.829,7.939,8.158,8.303,8.433,8.511,
     * 7.643,7.720,7.783,7.893,8.113,8.257,8.387,8.465/
      DATA ((EDEN(I,3,8,L,3),I=25,32),L=1,NZEN)/
     * 9.258,9.350,9.489,9.638,9.789,9.947,10.111,10.262,
     * 9.158,9.260,9.407,9.558,9.714,9.866,10.021,10.164,
     * 9.130,9.220,9.356,9.483,9.621,9.753,9.891,10.029,
     * 9.093,9.179,9.299,9.408,9.538,9.638,9.744,9.859,
     * 9.041,9.121,9.225,9.320,9.449,9.548,9.648,9.737,
     * 8.987,9.072,9.176,9.267,9.400,9.504,9.609,9.695,
     * 8.822,8.932,9.053,9.143,9.292,9.410,9.526,9.615,
     * 8.037,8.212,8.375,8.526,8.773,8.952,9.127,9.250,
     * 8.025,8.121,8.190,8.265,8.450,8.575,8.698,8.782,
     * 7.990,8.079,8.143,8.220,8.407,8.531,8.653,8.735,
     * 7.812,7.870,7.904,7.993,8.193,8.318,8.436,8.509,
     * 7.766,7.825,7.858,7.947,8.147,8.272,8.390,8.463/
      DATA ((EDEN(I,4,8,L,3),I=25,32),L=1,NZEN)/
     * 9.328,9.446,9.601,9.751,9.905,10.049,10.210,10.358,
     * 9.207,9.316,9.461,9.601,9.749,9.889,10.045,10.185,
     * 9.176,9.272,9.398,9.512,9.637,9.763,9.901,10.041,
     * 9.134,9.217,9.328,9.422,9.535,9.626,9.730,9.849,
     * 9.068,9.143,9.236,9.314,9.430,9.520,9.620,9.704,
     * 9.013,9.090,9.179,9.253,9.375,9.470,9.576,9.659,
     * 8.832,8.938,9.041,9.121,9.262,9.373,9.493,9.579,
     * 8.090,8.217,8.318,8.474,8.729,8.910,9.097,9.217,
     * 8.090,8.134,8.146,8.223,8.412,8.542,8.679,8.762,
     * 8.061,8.097,8.100,8.176,8.365,8.497,8.634,8.716,
     * 7.910,7.912,7.878,7.953,8.146,8.279,8.418,8.489,
     * 7.864,7.866,7.832,7.908,8.100,8.233,8.373,8.443/
      DATA ((EDEN(I,5,8,L,3),I=25,32),L=1,NZEN)/
     * 9.394,9.537,9.705,9.848,9.997,10.130,10.288,10.436,
     * 9.220,9.342,9.487,9.616,9.755,9.888,10.045,10.188,
     * 9.182,9.290,9.418,9.519,9.633,9.749,9.888,10.033,
     * 9.130,9.223,9.332,9.412,9.516,9.592,9.691,9.816,
     * 9.049,9.130,9.215,9.281,9.387,9.461,9.554,9.635,
     * 8.978,9.064,9.149,9.212,9.324,9.403,9.504,9.584,
     * 8.765,8.882,8.986,9.061,9.199,9.297,9.413,9.497,
     * 8.114,8.182,8.217,8.367,8.618,8.806,9.009,9.134,
     * 8.137,8.137,8.097,8.143,8.312,8.465,8.633,8.723,
     * 0.000,0.000,0.000,8.097,8.260,8.417,8.589,8.675,
     * 0.000,0.000,0.000,7.878,8.017,8.185,8.371,8.447,
     * 0.000,0.000,0.000,7.832,7.971,8.139,8.325,8.401/
      DATA ((EDEN(I,1,9,L,3),I=25,32),L=1,NZEN)/
     * 8.999,9.090,9.248,9.462,9.672,9.881,10.076,10.241,
     * 8.906,9.000,9.161,9.365,9.568,9.757,9.933,10.086,
     * 8.886,8.975,9.124,9.310,9.500,9.667,9.822,9.968,
     * 8.864,8.944,9.083,9.250,9.435,9.583,9.717,9.834,
     * 8.818,8.894,9.017,9.170,9.354,9.505,9.638,9.739,
     * 8.762,8.846,8.969,9.117,9.303,9.459,9.599,9.698,
     * 8.585,8.695,8.834,8.985,9.190,9.360,9.511,9.613,
     * 7.799,7.970,8.149,8.350,8.641,8.872,9.079,9.220,
     * 7.792,7.896,7.997,8.127,8.358,8.533,8.685,8.782,
     * 7.753,7.852,7.949,8.079,8.316,8.489,8.639,8.736,
     * 7.554,7.630,7.703,7.849,8.104,8.274,8.420,8.507,
     * 7.508,7.585,7.658,7.804,8.058,8.228,8.374,8.461/
      DATA ((EDEN(I,2,9,L,3),I=25,32),L=1,NZEN)/
     * 9.143,9.233,9.378,9.559,9.734,9.914,10.090,10.243,
     * 9.037,9.140,9.294,9.471,9.646,9.812,9.973,10.117,
     * 9.017,9.111,9.258,9.413,9.575,9.719,9.858,9.996,
     * 8.992,9.079,9.212,9.354,9.509,9.632,9.749,9.858,
     * 8.945,9.029,9.146,9.272,9.428,9.554,9.670,9.760,
     * 8.892,8.980,9.097,9.217,9.377,9.508,9.628,9.718,
     * 8.714,8.831,8.967,9.090,9.265,9.408,9.542,9.633,
     * 7.928,8.107,8.279,8.455,8.720,8.925,9.117,9.246,
     * 7.921,8.029,8.127,8.228,8.433,8.582,8.716,8.804,
     * 7.881,7.986,8.079,8.182,8.389,8.537,8.670,8.756,
     * 7.687,7.766,7.831,7.950,8.176,8.322,8.452,8.528,
     * 7.641,7.720,7.785,7.904,8.130,8.276,8.406,8.482/
      DATA ((EDEN(I,3,9,L,3),I=25,32),L=1,NZEN)/
     * 9.294,9.384,9.519,9.669,9.814,9.968,10.124,10.265,
     * 9.176,9.279,9.428,9.581,9.732,9.877,10.021,10.161,
     * 9.152,9.250,9.387,9.519,9.654,9.778,9.903,10.033,
     * 9.124,9.215,9.340,9.455,9.583,9.680,9.778,9.882,
     * 9.076,9.164,9.274,9.373,9.501,9.599,9.695,9.776,
     * 9.025,9.117,9.228,9.320,9.452,9.555,9.654,9.735,
     * 8.854,8.973,9.100,9.193,9.342,9.458,9.571,9.652,
     * 8.064,8.253,8.423,8.573,8.810,8.987,9.158,9.276,
     * 8.053,8.164,8.253,8.326,8.504,8.625,8.740,8.819,
     * 8.017,8.124,8.207,8.281,8.461,8.582,8.696,8.772,
     * 7.824,7.905,7.961,8.053,8.248,8.367,8.479,8.545,
     * 7.778,7.859,7.916,8.007,8.202,8.322,8.433,8.500/
      DATA ((EDEN(I,4,9,L,3),I=25,32),L=1,NZEN)/
     * 9.441,9.533,9.662,9.792,9.916,10.049,10.188,10.320,
     * 9.307,9.412,9.551,9.686,9.818,9.944,10.079,10.207,
     * 9.281,9.377,9.501,9.610,9.720,9.829,9.945,10.072,
     * 9.250,9.336,9.447,9.535,9.634,9.713,9.799,9.899,
     * 9.204,9.283,9.377,9.447,9.547,9.622,9.705,9.776,
     * 9.152,9.238,9.332,9.396,9.496,9.577,9.664,9.734,
     * 8.994,9.104,9.210,9.274,9.391,9.486,9.582,9.654,
     * 8.210,8.387,8.542,8.665,8.877,9.033,9.190,9.294,
     * 8.188,8.283,8.344,8.394,8.545,8.649,8.753,8.822,
     * 8.152,8.243,8.299,8.348,8.504,8.606,8.708,8.775,
     * 7.964,8.029,8.061,8.121,8.290,8.391,8.491,8.549,
     * 7.919,7.984,8.015,8.075,8.244,8.345,8.446,8.503/
      DATA ((EDEN(I,5,9,L,3),I=25,32),L=1,NZEN)/
     * 9.593,9.688,9.808,9.913,10.013,10.121,10.243,10.369,
     * 9.435,9.540,9.670,9.785,9.897,10.004,10.127,10.246,
     * 9.408,9.502,9.616,9.702,9.791,9.880,9.984,10.104,
     * 9.380,9.462,9.560,9.621,9.695,9.747,9.813,9.908,
     * 9.336,9.410,9.489,9.531,9.602,9.646,9.703,9.763,
     * 9.288,9.367,9.444,9.480,9.553,9.601,9.661,9.719,
     * 9.134,9.238,9.326,9.362,9.450,9.511,9.581,9.639,
     * 8.356,8.524,8.657,8.754,8.942,9.068,9.199,9.290,
     * 8.328,8.405,8.438,8.462,8.591,8.670,8.757,8.818,
     * 8.290,8.364,8.391,8.417,8.548,8.626,8.712,8.771,
     * 8.104,8.149,8.149,8.188,8.330,8.410,8.496,8.545,
     * 8.058,8.103,8.103,8.142,8.285,8.364,8.450,8.500/
      DATA ((EDEN(I,1,10,L,3),I=25,32),L=1,NZEN)/
     * 8.995,9.083,9.241,9.447,9.649,9.857,10.049,10.215,
     * 8.925,9.013,9.167,9.364,9.560,9.744,9.916,10.068,
     * 8.906,8.988,9.134,9.312,9.497,9.662,9.812,9.955,
     * 8.884,8.959,9.090,9.255,9.436,9.583,9.715,9.831,
     * 8.839,8.908,9.025,9.176,9.354,9.505,9.639,9.741,
     * 8.782,8.859,8.975,9.121,9.303,9.459,9.599,9.699,
     * 8.603,8.707,8.840,8.986,9.188,9.358,9.508,9.612,
     * 7.817,7.984,8.158,8.350,8.633,8.864,9.072,9.212,
     * 7.809,7.913,8.013,8.137,8.360,8.535,8.687,8.787,
     * 7.770,7.870,7.967,8.090,8.316,8.491,8.641,8.740,
     * 7.568,7.645,7.719,7.859,8.104,8.274,8.422,8.509,
     * 7.522,7.600,7.674,7.813,8.058,8.228,8.376,8.463/
      DATA ((EDEN(I,2,10,L,3),I=25,32),L=1,NZEN)/
     * 9.140,9.236,9.387,9.574,9.753,9.933,10.104,10.255,
     * 9.049,9.149,9.305,9.484,9.658,9.817,9.968,10.107,
     * 9.033,9.127,9.272,9.433,9.593,9.734,9.865,9.993,
     * 9.009,9.097,9.230,9.378,9.534,9.657,9.769,9.869,
     * 8.964,9.049,9.167,9.299,9.455,9.581,9.695,9.781,
     * 8.908,8.999,9.121,9.246,9.405,9.537,9.655,9.740,
     * 8.729,8.848,8.985,9.111,9.290,9.435,9.565,9.652,
     * 7.943,8.124,8.299,8.473,8.732,8.938,9.127,9.250,
     * 7.937,8.053,8.155,8.258,8.459,8.609,8.740,8.825,
     * 7.898,8.009,8.107,8.212,8.415,8.565,8.695,8.777,
     * 7.695,7.785,7.860,7.980,8.201,8.348,8.474,8.548,
     * 7.649,7.739,7.814,7.934,8.156,8.303,8.428,8.502/
      DATA ((EDEN(I,3,10,L,3),I=25,32),L=1,NZEN)/
     * 9.316,9.408,9.548,9.704,9.846,9.995,10.140,10.279,
     * 9.190,9.303,9.458,9.616,9.764,9.900,10.033,10.161,
     * 9.173,9.276,9.423,9.562,9.697,9.813,9.925,10.045,
     * 9.152,9.250,9.384,9.508,9.636,9.732,9.821,9.912,
     * 9.107,9.204,9.324,9.431,9.560,9.657,9.747,9.821,
     * 9.057,9.158,9.279,9.380,9.512,9.613,9.707,9.779,
     * 8.885,9.013,9.149,9.253,9.400,9.515,9.620,9.695,
     * 8.093,8.294,8.480,8.629,8.860,9.033,9.196,9.305,
     * 8.079,8.212,8.318,8.393,8.563,8.682,8.788,8.859,
     * 8.041,8.167,8.272,8.348,8.520,8.638,8.744,8.814,
     * 7.835,7.942,8.021,8.114,8.305,8.422,8.522,8.583,
     * 7.789,7.896,7.975,8.068,8.260,8.376,8.477,8.537/
      DATA ((EDEN(I,4,10,L,3),I=25,32),L=1,NZEN)/
     * 9.508,9.590,9.708,9.836,9.946,10.064,10.185,10.305,
     * 9.391,9.496,9.633,9.770,9.892,10.004,10.121,10.233,
     * 9.373,9.470,9.594,9.706,9.808,9.905,10.000,10.107,
     * 9.354,9.442,9.555,9.647,9.738,9.808,9.876,9.958,
     * 9.322,9.405,9.502,9.574,9.662,9.729,9.794,9.852,
     * 9.281,9.369,9.464,9.528,9.616,9.688,9.757,9.813,
     * 9.134,9.243,9.352,9.412,9.515,9.598,9.676,9.733,
     * 8.338,8.549,8.734,8.835,9.013,9.152,9.281,9.371,
     * 8.292,8.428,8.529,8.554,8.668,8.754,8.833,8.888,
     * 8.250,8.386,8.481,8.508,8.626,8.711,8.787,8.841,
     * 8.033,8.155,8.241,8.281,8.417,8.498,8.571,8.614,
     * 7.988,8.110,8.195,8.235,8.371,8.453,8.525,8.568/
      DATA ((EDEN(I,5,10,L,3),I=25,32),L=1,NZEN)/
     * 9.674,9.755,9.860,9.966,10.049,10.140,10.236,10.342,
     * 9.560,9.667,9.797,9.916,10.017,10.107,10.201,10.303,
     * 9.539,9.634,9.748,9.839,9.916,9.992,10.072,10.170,
     * 9.520,9.606,9.705,9.772,9.833,9.874,9.919,9.993,
     * 9.491,9.572,9.654,9.699,9.755,9.789,9.823,9.867,
     * 9.458,9.540,9.619,9.656,9.713,9.749,9.786,9.827,
     * 9.330,9.433,9.520,9.550,9.618,9.665,9.710,9.751,
     * 8.548,8.769,8.953,9.017,9.149,9.248,9.344,9.413,
     * 8.470,8.611,8.704,8.691,8.765,8.816,8.863,8.904,
     * 8.427,8.568,8.661,8.648,8.724,8.773,8.819,8.859,
     * 8.204,8.340,8.428,8.428,8.519,8.563,8.604,8.633,
     * 8.158,8.295,8.382,8.382,8.473,8.518,8.558,8.588/
      DATA ((EDEN(I,1,11,L,3),I=25,32),L=1,NZEN)/
     * 9.017,9.097,9.246,9.442,9.636,9.836,10.021,10.188,
     * 8.928,9.013,9.167,9.362,9.556,9.738,9.908,10.061,
     * 8.910,8.990,9.134,9.312,9.496,9.657,9.805,9.946,
     * 8.889,8.961,9.090,9.258,9.436,9.583,9.714,9.829,
     * 8.841,8.910,9.025,9.176,9.356,9.505,9.639,9.741,
     * 8.785,8.861,8.977,9.121,9.303,9.459,9.599,9.699,
     * 8.604,8.707,8.838,8.985,9.188,9.356,9.507,9.611,
     * 7.817,7.982,8.155,8.348,8.630,8.859,9.064,9.207,
     * 7.812,7.914,8.013,8.137,8.360,8.535,8.688,8.789,
     * 7.773,7.872,7.968,8.090,8.318,8.493,8.643,8.742,
     * 7.571,7.646,7.719,7.858,8.104,8.274,8.422,8.511,
     * 7.525,7.601,7.674,7.812,8.058,8.228,8.376,8.465/
      DATA ((EDEN(I,2,11,L,3),I=25,32),L=1,NZEN)/
     * 9.146,9.246,9.403,9.595,9.775,9.954,10.124,10.272,
     * 9.057,9.158,9.316,9.500,9.674,9.829,9.974,10.107,
     * 9.037,9.137,9.286,9.452,9.614,9.751,9.875,9.999,
     * 9.021,9.111,9.248,9.400,9.558,9.679,9.789,9.885,
     * 8.975,9.064,9.188,9.322,9.480,9.606,9.719,9.801,
     * 8.920,9.013,9.137,9.267,9.430,9.561,9.678,9.758,
     * 8.741,8.862,9.000,9.134,9.312,9.458,9.588,9.671,
     * 7.952,8.140,8.320,8.498,8.756,8.959,9.143,9.265,
     * 7.946,8.068,8.179,8.283,8.484,8.632,8.761,8.843,
     * 7.906,8.025,8.130,8.238,8.441,8.589,8.716,8.796,
     * 7.697,7.797,7.880,8.004,8.225,8.371,8.494,8.565,
     * 7.651,7.751,7.834,7.959,8.180,8.325,8.448,8.519/
      DATA ((EDEN(I,3,11,L,3),I=25,32),L=1,NZEN)/
     * 9.318,9.422,9.569,9.740,9.888,10.037,10.176,10.307,
     * 9.215,9.330,9.490,9.656,9.804,9.936,10.057,10.179,
     * 9.199,9.310,9.461,9.609,9.744,9.855,9.956,10.064,
     * 9.182,9.288,9.427,9.559,9.688,9.782,9.866,9.946,
     * 9.143,9.246,9.371,9.487,9.615,9.710,9.794,9.859,
     * 9.093,9.201,9.328,9.436,9.567,9.667,9.754,9.819,
     * 8.929,9.061,9.204,9.310,9.456,9.568,9.667,9.734,
     * 8.134,8.348,8.549,8.696,8.915,9.086,9.241,9.342,
     * 8.114,8.262,8.386,8.459,8.618,8.733,8.831,8.895,
     * 8.072,8.217,8.338,8.412,8.575,8.689,8.786,8.849,
     * 7.851,7.983,8.086,8.182,8.364,8.473,8.566,8.618,
     * 7.806,7.937,8.041,8.136,8.318,8.427,8.520,8.572/
      DATA ((EDEN(I,4,11,L,3),I=25,32),L=1,NZEN)/
     * 9.520,9.613,9.740,9.882,9.994,10.107,10.215,10.326,
     * 9.442,9.554,9.696,9.843,9.964,10.072,10.170,10.274,
     * 9.431,9.533,9.664,9.787,9.889,9.978,10.057,10.155,
     * 9.420,9.515,9.632,9.736,9.826,9.891,9.947,10.013,
     * 9.398,9.487,9.590,9.672,9.758,9.820,9.871,9.917,
     * 9.362,9.455,9.554,9.628,9.715,9.780,9.834,9.879,
     * 9.228,9.342,9.452,9.520,9.616,9.692,9.755,9.801,
     * 8.442,8.670,8.871,8.966,9.124,9.253,9.365,9.442,
     * 8.380,8.535,8.654,8.671,8.766,8.841,8.901,8.945,
     * 8.336,8.491,8.610,8.627,8.724,8.798,8.857,8.898,
     * 8.097,8.255,8.375,8.405,8.516,8.584,8.637,8.670,
     * 8.051,8.210,8.329,8.359,8.470,8.539,8.592,8.624/
      DATA ((EDEN(I,5,11,L,3),I=25,32),L=1,NZEN)/
     * 9.706,9.801,9.920,10.045,10.130,10.210,10.283,10.375,
     * 9.645,9.760,9.895,10.029,10.127,10.207,10.281,10.364,
     * 9.635,9.740,9.862,9.968,10.045,10.107,10.161,10.241,
     * 9.631,9.728,9.834,9.916,9.976,10.009,10.029,10.083,
     * 9.620,9.711,9.802,9.860,9.913,9.937,9.951,9.975,
     * 9.594,9.688,9.774,9.822,9.876,9.901,9.918,9.938,
     * 9.489,9.597,9.688,9.727,9.787,9.820,9.843,9.865,
     * 8.746,8.974,9.164,9.223,9.336,9.417,9.486,9.533,
     * 8.633,8.791,8.900,8.876,8.922,8.956,8.975,8.995,
     * 8.589,8.747,8.857,8.833,8.880,8.913,8.930,8.948,
     * 8.338,8.509,8.630,8.614,8.671,8.699,8.713,8.722,
     * 8.293,8.463,8.585,8.568,8.625,8.653,8.667,8.676/
      DATA ((EDEN(I,1,12,L,3),I=25,32),L=1,NZEN)/
     * 9.041,9.114,9.255,9.446,9.632,9.829,10.013,10.176,
     * 8.911,9.004,9.161,9.358,9.556,9.740,9.911,10.061,
     * 8.893,8.978,9.127,9.307,9.496,9.659,9.808,9.950,
     * 8.870,8.948,9.083,9.253,9.436,9.584,9.717,9.831,
     * 8.822,8.896,9.017,9.170,9.354,9.507,9.642,9.743,
     * 8.765,8.846,8.966,9.114,9.303,9.461,9.600,9.701,
     * 8.579,8.688,8.826,8.978,9.185,9.356,9.508,9.612,
     * 7.799,7.961,8.130,8.330,8.622,8.854,9.064,9.204,
     * 7.797,7.898,7.996,8.124,8.360,8.537,8.690,8.791,
     * 7.758,7.854,7.947,8.079,8.316,8.491,8.645,8.744,
     * 7.556,7.630,7.700,7.847,8.100,8.274,8.423,8.513,
     * 7.511,7.585,7.654,7.801,8.055,8.228,8.377,8.467/
      DATA ((EDEN(I,2,12,L,3),I=25,32),L=1,NZEN)/
     * 9.158,9.258,9.417,9.614,9.795,9.974,10.140,10.286,
     * 9.033,9.146,9.314,9.505,9.685,9.841,9.986,10.121,
     * 9.017,9.124,9.283,9.458,9.626,9.764,9.888,10.009,
     * 9.000,9.100,9.243,9.405,9.571,9.693,9.802,9.897,
     * 8.955,9.053,9.182,9.328,9.493,9.620,9.733,9.813,
     * 8.900,9.004,9.134,9.274,9.442,9.575,9.692,9.770,
     * 8.718,8.848,8.996,9.140,9.324,9.471,9.601,9.682,
     * 7.934,8.124,8.305,8.496,8.763,8.970,9.155,9.274,
     * 7.929,8.053,8.164,8.283,8.494,8.645,8.775,8.854,
     * 7.888,8.009,8.117,8.238,8.452,8.601,8.729,8.808,
     * 7.679,7.780,7.864,8.000,8.233,8.382,8.507,8.576,
     * 7.633,7.734,7.818,7.954,8.187,8.336,8.461,8.531/
      DATA ((EDEN(I,3,12,L,3),I=25,32),L=1,NZEN)/
     * 9.310,9.423,9.582,9.763,9.919,10.068,10.207,10.332,
     * 9.204,9.332,9.504,9.679,9.833,9.965,10.086,10.201,
     * 9.190,9.314,9.474,9.631,9.771,9.883,9.982,10.090,
     * 9.173,9.290,9.441,9.584,9.718,9.812,9.893,9.969,
     * 9.137,9.250,9.389,9.513,9.645,9.741,9.822,9.883,
     * 9.090,9.210,9.346,9.464,9.599,9.699,9.783,9.843,
     * 8.926,9.072,9.220,9.338,9.487,9.601,9.695,9.758,
     * 8.130,8.358,8.569,8.726,8.949,9.121,9.270,9.367,
     * 8.107,8.267,8.403,8.486,8.648,8.762,8.854,8.914,
     * 8.064,8.225,8.358,8.441,8.606,8.720,8.812,8.869,
     * 7.842,7.988,8.107,8.207,8.393,8.502,8.591,8.639,
     * 7.796,7.942,8.061,8.161,8.347,8.457,8.545,8.594/
      DATA ((EDEN(I,4,12,L,3),I=25,32),L=1,NZEN)/
     * 9.502,9.612,9.754,9.912,10.033,10.149,10.250,10.356,
     * 9.450,9.573,9.727,9.888,10.017,10.121,10.215,10.312,
     * 9.439,9.553,9.695,9.831,9.938,10.029,10.104,10.193,
     * 9.431,9.538,9.667,9.781,9.876,9.938,9.987,10.049,
     * 9.413,9.515,9.626,9.721,9.810,9.870,9.914,9.955,
     * 9.380,9.484,9.594,9.679,9.769,9.832,9.879,9.918,
     * 9.258,9.378,9.496,9.574,9.672,9.745,9.801,9.840,
     * 8.481,8.722,8.934,9.033,9.190,9.316,9.420,9.489,
     * 8.403,8.572,8.703,8.726,8.818,8.890,8.943,8.977,
     * 8.356,8.528,8.661,8.682,8.775,8.846,8.896,8.930,
     * 8.114,8.288,8.427,8.459,8.566,8.632,8.678,8.702,
     * 8.068,8.242,8.381,8.414,8.520,8.587,8.632,8.657/
      DATA ((EDEN(I,5,12,L,3),I=25,32),L=1,NZEN)/
     * 9.695,9.813,9.954,10.097,10.193,10.272,10.338,10.420,
     * 9.659,9.791,9.943,10.093,10.199,10.276,10.342,10.417,
     * 9.652,9.773,9.910,10.033,10.117,10.176,10.225,10.294,
     * 9.652,9.763,9.886,9.985,10.053,10.079,10.093,10.140,
     * 9.648,9.754,9.860,9.934,9.993,10.013,10.017,10.029,
     * 9.626,9.734,9.835,9.899,9.957,9.979,9.984,9.995,
     * 9.526,9.648,9.753,9.808,9.872,9.901,9.913,9.923,
     * 8.814,9.053,9.250,9.320,9.431,9.509,9.566,9.600,
     * 8.682,8.850,8.969,8.956,9.000,9.029,9.033,9.045,
     * 8.635,8.807,8.928,8.913,8.959,8.985,8.992,8.997,
     * 8.373,8.563,8.702,8.694,8.747,8.770,8.772,8.770,
     * 8.327,8.518,8.656,8.648,8.702,8.724,8.727,8.724/
      DATA ((EDEN(I,1,1,L,1),I=33,40),L=1,NZEN)/
     * 10.029,10.152,10.270,10.400,10.542,10.658,10.780,10.866,
     * 10.017,10.140,10.250,10.373,10.508,10.617,10.731,10.814,
     * 9.906,10.017,10.114,10.233,10.362,10.483,10.607,10.704,
     * 9.758,9.852,9.932,10.025,10.127,10.246,10.365,10.483,
     * 9.703,9.767,9.814,9.866,9.921,9.981,10.041,10.114,
     * 9.709,9.777,9.823,9.866,9.911,9.953,9.995,10.033,
     * 9.713,9.790,9.843,9.886,9.925,9.954,9.981,9.999,
     * 9.585,9.706,9.792,9.851,9.895,9.913,9.924,9.924,
     * 9.305,9.433,9.524,9.602,9.662,9.702,9.731,9.707,
     * 9.265,9.393,9.483,9.559,9.615,9.652,9.680,9.655,
     * 9.041,9.158,9.233,9.303,9.352,9.384,9.407,9.382,
     * 8.996,9.113,9.187,9.257,9.306,9.338,9.361,9.336/
      DATA ((EDEN(I,2,1,L,1),I=33,40),L=1,NZEN)/
     * 10.140,10.272,10.394,10.511,10.635,10.732,10.834,10.901,
     * 10.076,10.196,10.307,10.422,10.542,10.643,10.750,10.826,
     * 9.961,10.072,10.170,10.281,10.396,10.509,10.626,10.715,
     * 9.816,9.910,9.990,10.076,10.164,10.274,10.386,10.496,
     * 9.762,9.823,9.865,9.911,9.954,10.009,10.061,10.127,
     * 9.769,9.833,9.874,9.911,9.943,9.977,10.013,10.041,
     * 9.773,9.846,9.894,9.930,9.958,9.980,10.000,10.013,
     * 9.648,9.764,9.845,9.897,9.929,9.940,9.943,9.936,
     * 9.356,9.484,9.575,9.646,9.694,9.727,9.751,9.719,
     * 9.316,9.444,9.533,9.602,9.646,9.678,9.699,9.667,
     * 9.093,9.210,9.286,9.348,9.386,9.412,9.427,9.393,
     * 9.048,9.164,9.240,9.303,9.340,9.366,9.381,9.347/
      DATA ((EDEN(I,3,1,L,1),I=33,40),L=1,NZEN)/
     * 10.185,10.314,10.433,10.539,10.649,10.741,10.838,10.900,
     * 10.161,10.283,10.396,10.501,10.609,10.696,10.790,10.852,
     * 10.041,10.155,10.255,10.360,10.465,10.566,10.670,10.747,
     * 9.890,9.985,10.064,10.143,10.223,10.326,10.435,10.531,
     * 9.822,9.880,9.919,9.955,9.983,10.033,10.083,10.143,
     * 9.831,9.890,9.926,9.950,9.964,9.994,10.025,10.045,
     * 9.839,9.906,9.948,9.970,9.978,9.995,10.009,10.009,
     * 9.726,9.833,9.906,9.943,9.955,9.959,9.956,9.936,
     * 9.400,9.528,9.616,9.680,9.714,9.741,9.757,9.719,
     * 9.360,9.486,9.573,9.634,9.667,9.692,9.707,9.667,
     * 9.140,9.255,9.328,9.382,9.405,9.425,9.435,9.393,
     * 9.094,9.210,9.283,9.336,9.359,9.379,9.389,9.347/
      DATA ((EDEN(I,4,1,L,1),I=33,40),L=1,NZEN)/
     * 10.196,10.318,10.428,10.528,10.623,10.717,10.815,10.880,
     * 10.274,10.400,10.516,10.606,10.693,10.765,10.840,10.888,
     * 10.152,10.267,10.371,10.467,10.556,10.641,10.730,10.789,
     * 9.984,10.083,10.167,10.243,10.310,10.405,10.502,10.584,
     * 9.885,9.948,9.992,10.017,10.029,10.072,10.111,10.170,
     * 9.894,9.955,9.994,10.004,9.997,10.013,10.029,10.045,
     * 9.906,9.975,10.017,10.025,10.009,10.009,10.009,9.995,
     * 9.816,9.920,9.989,10.009,9.994,9.981,9.959,9.926,
     * 9.441,9.566,9.653,9.708,9.727,9.747,9.756,9.710,
     * 9.401,9.526,9.611,9.663,9.680,9.698,9.704,9.657,
     * 9.185,9.297,9.369,9.412,9.420,9.431,9.433,9.384,
     * 9.139,9.251,9.323,9.366,9.374,9.386,9.387,9.338/
      DATA ((EDEN(I,5,1,L,1),I=33,40),L=1,NZEN)/
     * 10.253,10.369,10.476,10.563,10.643,10.730,10.821,10.881,
     * 10.384,10.512,10.629,10.705,10.772,10.827,10.886,10.920,
     * 10.255,10.377,10.484,10.569,10.643,10.713,10.785,10.827,
     * 10.083,10.179,10.262,10.336,10.400,10.483,10.569,10.633,
     * 9.960,10.017,10.053,10.072,10.076,10.111,10.149,10.207,
     * 9.970,10.017,10.045,10.045,10.021,10.033,10.041,10.053,
     * 9.986,10.041,10.068,10.061,10.029,10.021,10.009,9.986,
     * 9.913,10.000,10.057,10.057,10.021,9.997,9.962,9.919,
     * 9.490,9.611,9.693,9.738,9.744,9.757,9.759,9.704,
     * 9.452,9.572,9.652,9.695,9.696,9.708,9.708,9.652,
     * 9.238,9.344,9.412,9.444,9.438,9.442,9.436,9.380,
     * 9.192,9.299,9.366,9.398,9.392,9.397,9.390,9.334/
      DATA ((EDEN(I,1,2,L,1),I=33,40),L=1,NZEN)/
     * 10.041,10.167,10.286,10.413,10.554,10.667,10.787,10.869,
     * 10.037,10.158,10.270,10.393,10.526,10.632,10.744,10.824,
     * 9.921,10.033,10.130,10.253,10.384,10.501,10.623,10.717,
     * 9.766,9.863,9.945,10.041,10.146,10.262,10.386,10.500,
     * 9.703,9.767,9.814,9.867,9.923,9.984,10.045,10.121,
     * 9.709,9.776,9.822,9.865,9.909,9.950,9.990,10.029,
     * 9.714,9.791,9.843,9.885,9.923,9.951,9.976,9.995,
     * 9.590,9.709,9.795,9.853,9.896,9.912,9.921,9.921,
     * 9.301,9.428,9.520,9.597,9.654,9.693,9.722,9.698,
     * 9.260,9.387,9.477,9.553,9.609,9.644,9.671,9.645,
     * 9.037,9.152,9.228,9.297,9.344,9.377,9.400,9.373,
     * 8.992,9.107,9.182,9.251,9.299,9.331,9.354,9.327/
      DATA ((EDEN(I,2,2,L,1),I=33,40),L=1,NZEN)/
     * 10.130,10.260,10.382,10.500,10.625,10.723,10.826,10.894,
     * 10.090,10.212,10.326,10.442,10.566,10.664,10.766,10.838,
     * 9.973,10.086,10.188,10.303,10.422,10.533,10.647,10.732,
     * 9.818,9.916,10.000,10.090,10.182,10.294,10.408,10.515,
     * 9.752,9.816,9.861,9.910,9.957,10.013,10.068,10.134,
     * 9.760,9.825,9.869,9.907,9.941,9.977,10.013,10.041,
     * 9.765,9.839,9.889,9.927,9.955,9.978,9.997,10.009,
     * 9.643,9.760,9.843,9.895,9.928,9.940,9.942,9.934,
     * 9.346,9.476,9.566,9.638,9.687,9.719,9.742,9.709,
     * 9.307,9.433,9.522,9.593,9.640,9.671,9.690,9.657,
     * 9.086,9.201,9.279,9.340,9.378,9.403,9.418,9.384,
     * 9.041,9.156,9.233,9.295,9.333,9.357,9.373,9.338/
      DATA ((EDEN(I,3,2,L,1),I=33,40),L=1,NZEN)/
     * 10.167,10.294,10.412,10.522,10.635,10.730,10.828,10.892,
     * 10.170,10.297,10.415,10.522,10.630,10.717,10.807,10.865,
     * 10.049,10.167,10.272,10.382,10.490,10.590,10.691,10.762,
     * 9.887,9.987,10.072,10.158,10.243,10.350,10.458,10.551,
     * 9.804,9.866,9.909,9.951,9.985,10.037,10.093,10.155,
     * 9.812,9.873,9.914,9.942,9.959,9.991,10.021,10.045,
     * 9.819,9.889,9.934,9.960,9.971,9.990,10.004,10.004,
     * 9.710,9.820,9.897,9.937,9.951,9.956,9.952,9.930,
     * 9.386,9.513,9.604,9.670,9.707,9.734,9.750,9.709,
     * 9.346,9.473,9.562,9.626,9.660,9.686,9.699,9.657,
     * 9.127,9.241,9.316,9.373,9.400,9.418,9.425,9.384,
     * 9.081,9.195,9.270,9.327,9.354,9.373,9.379,9.338/
      DATA ((EDEN(I,4,2,L,1),I=33,40),L=1,NZEN)/
     * 10.182,10.305,10.417,10.521,10.622,10.718,10.816,10.879,
     * 10.276,10.407,10.528,10.621,10.712,10.782,10.854,10.897,
     * 10.149,10.272,10.384,10.483,10.577,10.662,10.747,10.801,
     * 9.971,10.076,10.167,10.255,10.332,10.428,10.525,10.602,
     * 9.851,9.920,9.971,10.004,10.025,10.076,10.121,10.185,
     * 9.859,9.923,9.968,9.985,9.987,10.009,10.029,10.045,
     * 9.869,9.942,9.991,10.004,9.996,10.000,10.000,9.987,
     * 9.782,9.891,9.966,9.990,9.983,9.972,9.951,9.918,
     * 9.417,9.545,9.636,9.695,9.719,9.740,9.748,9.700,
     * 9.378,9.505,9.595,9.652,9.674,9.693,9.698,9.648,
     * 9.164,9.276,9.352,9.400,9.412,9.425,9.427,9.375,
     * 9.119,9.231,9.306,9.354,9.366,9.379,9.381,9.329/
      DATA ((EDEN(I,5,2,L,1),I=33,40),L=1,NZEN)/
     * 10.230,10.352,10.464,10.560,10.648,10.737,10.828,10.886,
     * 10.373,10.508,10.631,10.713,10.785,10.840,10.897,10.926,
     * 10.241,10.371,10.487,10.579,10.660,10.728,10.797,10.836,
     * 10.053,10.161,10.258,10.342,10.413,10.500,10.585,10.646,
     * 9.898,9.966,10.017,10.049,10.064,10.111,10.152,10.215,
     * 9.903,9.962,10.000,10.013,10.004,10.021,10.037,10.053,
     * 9.915,9.981,10.021,10.029,10.009,10.004,9.997,9.975,
     * 9.843,9.942,10.009,10.021,10.000,9.981,9.951,9.908,
     * 9.453,9.581,9.671,9.724,9.737,9.752,9.754,9.697,
     * 9.415,9.540,9.627,9.679,9.690,9.703,9.703,9.645,
     * 9.204,9.316,9.389,9.431,9.430,9.438,9.430,9.371,
     * 9.158,9.270,9.343,9.386,9.384,9.392,9.384,9.325/
      DATA ((EDEN(I,1,3,L,1),I=33,40),L=1,NZEN)/
     * 10.061,10.188,10.305,10.431,10.572,10.680,10.794,10.873,
     * 10.037,10.161,10.274,10.396,10.530,10.635,10.746,10.824,
     * 9.921,10.033,10.137,10.255,10.386,10.504,10.626,10.719,
     * 9.768,9.865,9.949,10.045,10.146,10.267,10.389,10.504,
     * 9.702,9.766,9.812,9.865,9.922,9.983,10.045,10.121,
     * 9.708,9.775,9.819,9.863,9.906,9.946,9.986,10.029,
     * 9.713,9.789,9.840,9.883,9.921,9.948,9.972,9.991,
     * 9.591,9.709,9.794,9.851,9.894,9.910,9.918,9.918,
     * 9.297,9.425,9.515,9.591,9.650,9.688,9.717,9.692,
     * 9.258,9.384,9.471,9.545,9.602,9.638,9.667,9.640,
     * 9.037,9.149,9.225,9.294,9.342,9.373,9.394,9.367,
     * 8.992,9.103,9.180,9.249,9.297,9.327,9.349,9.322/
      DATA ((EDEN(I,2,3,L,1),I=33,40),L=1,NZEN)/
     * 10.111,10.241,10.362,10.484,10.613,10.713,10.817,10.885,
     * 10.086,10.215,10.332,10.452,10.576,10.675,10.775,10.843,
     * 9.970,10.086,10.193,10.312,10.435,10.545,10.656,10.738,
     * 9.811,9.912,10.000,10.097,10.193,10.307,10.422,10.525,
     * 9.736,9.803,9.851,9.905,9.956,10.013,10.068,10.140,
     * 9.744,9.811,9.859,9.901,9.938,9.975,10.009,10.041,
     * 9.749,9.825,9.879,9.920,9.951,9.976,9.995,10.004,
     * 9.630,9.750,9.836,9.892,9.926,9.939,9.941,9.931,
     * 9.332,9.462,9.556,9.631,9.681,9.714,9.736,9.702,
     * 9.294,9.422,9.512,9.587,9.635,9.667,9.686,9.650,
     * 9.072,9.190,9.267,9.332,9.373,9.400,9.413,9.378,
     * 9.026,9.145,9.221,9.287,9.327,9.354,9.368,9.333/
      DATA ((EDEN(I,3,3,L,1),I=33,40),L=1,NZEN)/
     * 10.134,10.262,10.386,10.505,10.626,10.723,10.822,10.885,
     * 10.158,10.290,10.413,10.529,10.643,10.730,10.817,10.871,
     * 10.037,10.158,10.272,10.387,10.504,10.603,10.703,10.771,
     * 9.866,9.971,10.064,10.161,10.255,10.365,10.474,10.563,
     * 9.772,9.839,9.891,9.940,9.981,10.037,10.093,10.158,
     * 9.779,9.845,9.893,9.927,9.950,9.989,10.021,10.045,
     * 9.784,9.860,9.913,9.946,9.963,9.985,10.000,9.997,
     * 9.677,9.792,9.876,9.923,9.944,9.953,9.949,9.926,
     * 9.364,9.494,9.590,9.663,9.705,9.732,9.747,9.704,
     * 9.322,9.453,9.548,9.619,9.659,9.685,9.696,9.652,
     * 9.107,9.223,9.303,9.365,9.398,9.418,9.427,9.380,
     * 9.061,9.177,9.257,9.320,9.352,9.373,9.381,9.334/
      DATA ((EDEN(I,4,3,L,1),I=33,40),L=1,NZEN)/
     * 10.164,10.292,10.415,10.529,10.640,10.736,10.830,10.888,
     * 10.246,10.382,10.511,10.617,10.719,10.792,10.863,10.903,
     * 10.114,10.243,10.365,10.479,10.587,10.673,10.758,10.809,
     * 9.929,10.041,10.146,10.246,10.334,10.438,10.535,10.610,
     * 9.796,9.872,9.933,9.983,10.021,10.076,10.124,10.188,
     * 9.800,9.872,9.926,9.958,9.973,10.004,10.025,10.041,
     * 9.807,9.888,9.946,9.975,9.981,9.993,9.995,9.980,
     * 9.714,9.833,9.922,9.960,9.968,9.964,9.947,9.911,
     * 9.386,9.519,9.618,9.688,9.723,9.745,9.752,9.700,
     * 9.346,9.479,9.576,9.644,9.676,9.696,9.701,9.647,
     * 9.130,9.250,9.334,9.393,9.415,9.430,9.428,9.375,
     * 9.085,9.205,9.289,9.347,9.369,9.384,9.382,9.329/
      DATA ((EDEN(I,5,3,L,1),I=33,40),L=1,NZEN)/
     * 10.212,10.342,10.467,10.580,10.682,10.772,10.859,10.908,
     * 10.336,10.476,10.610,10.706,10.792,10.851,10.907,10.931,
     * 10.199,10.334,10.464,10.572,10.667,10.740,10.808,10.842,
     * 9.997,10.117,10.228,10.330,10.420,10.511,10.597,10.653,
     * 9.816,9.895,9.962,10.017,10.053,10.111,10.158,10.220,
     * 9.814,9.884,9.938,9.971,9.982,10.013,10.033,10.045,
     * 9.820,9.898,9.956,9.982,9.981,9.989,9.987,9.965,
     * 9.744,9.857,9.942,9.977,9.975,9.966,9.942,9.897,
     * 9.403,9.539,9.642,9.711,9.738,9.756,9.757,9.696,
     * 9.365,9.500,9.601,9.667,9.693,9.708,9.707,9.644,
     * 9.155,9.274,9.358,9.417,9.431,9.441,9.433,9.369,
     * 9.110,9.228,9.312,9.371,9.386,9.395,9.387,9.323/
      DATA ((EDEN(I,1,4,L,1),I=33,40),L=1,NZEN)/
     * 10.079,10.212,10.334,10.459,10.595,10.699,10.808,10.881,
     * 10.037,10.161,10.274,10.394,10.529,10.632,10.742,10.820,
     * 9.921,10.033,10.137,10.255,10.386,10.502,10.622,10.715,
     * 9.766,9.863,9.945,10.041,10.146,10.265,10.387,10.501,
     * 9.698,9.762,9.808,9.861,9.918,9.979,10.041,10.117,
     * 9.705,9.771,9.816,9.859,9.901,9.942,9.982,10.021,
     * 9.711,9.786,9.836,9.879,9.916,9.943,9.968,9.987,
     * 9.591,9.709,9.792,9.849,9.890,9.905,9.913,9.913,
     * 9.290,9.417,9.507,9.583,9.642,9.681,9.711,9.687,
     * 9.250,9.377,9.465,9.540,9.597,9.632,9.659,9.634,
     * 9.033,9.143,9.217,9.288,9.336,9.367,9.389,9.362,
     * 8.988,9.097,9.172,9.242,9.291,9.322,9.343,9.316/
      DATA ((EDEN(I,2,4,L,1),I=33,40),L=1,NZEN)/
     * 10.086,10.217,10.342,10.471,10.606,10.709,10.814,10.882,
     * 10.079,10.210,10.330,10.453,10.583,10.680,10.778,10.844,
     * 9.960,10.079,10.190,10.314,10.442,10.553,10.662,10.740,
     * 9.797,9.901,9.995,10.097,10.199,10.314,10.428,10.529,
     * 9.716,9.785,9.839,9.899,9.955,10.013,10.072,10.140,
     * 9.723,9.792,9.845,9.892,9.934,9.974,10.009,10.041,
     * 9.727,9.807,9.865,9.912,9.947,9.974,9.993,10.000,
     * 9.613,9.734,9.824,9.884,9.924,9.938,9.940,9.928,
     * 9.316,9.446,9.543,9.622,9.677,9.711,9.732,9.697,
     * 9.274,9.405,9.501,9.579,9.631,9.664,9.682,9.645,
     * 9.057,9.173,9.255,9.324,9.369,9.396,9.410,9.373,
     * 9.011,9.127,9.210,9.279,9.323,9.350,9.364,9.327/
      DATA ((EDEN(I,3,4,L,1),I=33,40),L=1,NZEN)/
     * 10.111,10.246,10.375,10.505,10.635,10.734,10.830,10.889,
     * 10.134,10.270,10.403,10.526,10.651,10.740,10.825,10.875,
     * 10.009,10.137,10.258,10.386,10.512,10.614,10.713,10.776,
     * 9.834,9.946,10.049,10.155,10.262,10.375,10.484,10.571,
     * 9.732,9.806,9.866,9.926,9.977,10.041,10.100,10.164,
     * 9.737,9.810,9.866,9.911,9.943,9.987,10.021,10.041,
     * 9.741,9.823,9.886,9.929,9.955,9.982,10.000,9.993,
     * 9.633,9.756,9.851,9.907,9.937,9.951,9.949,9.923,
     * 9.336,9.470,9.573,9.654,9.706,9.736,9.749,9.702,
     * 9.297,9.430,9.530,9.611,9.659,9.687,9.697,9.650,
     * 9.079,9.199,9.286,9.358,9.396,9.420,9.427,9.377,
     * 9.033,9.153,9.240,9.312,9.350,9.374,9.381,9.331/
      DATA ((EDEN(I,4,4,L,1),I=33,40),L=1,NZEN)/
     * 10.173,10.314,10.455,10.582,10.705,10.793,10.875,10.918,
     * 10.193,10.338,10.480,10.605,10.724,10.802,10.873,10.909,
     * 10.064,10.201,10.334,10.467,10.591,10.684,10.769,10.816,
     * 9.870,9.993,10.111,10.228,10.336,10.446,10.547,10.617,
     * 9.730,9.814,9.886,9.957,10.013,10.076,10.127,10.190,
     * 9.731,9.810,9.877,9.928,9.962,10.000,10.025,10.041,
     * 9.732,9.823,9.897,9.944,9.969,9.988,9.993,9.976,
     * 9.632,9.763,9.867,9.928,9.955,9.960,9.946,9.907,
     * 9.346,9.486,9.598,9.683,9.731,9.757,9.761,9.703,
     * 9.307,9.446,9.555,9.638,9.685,9.708,9.710,9.651,
     * 9.090,9.215,9.312,9.386,9.423,9.441,9.438,9.378,
     * 9.044,9.169,9.266,9.340,9.377,9.395,9.392,9.333/
      DATA ((EDEN(I,5,4,L,1),I=33,40),L=1,NZEN)/
     * 10.238,10.389,10.542,10.668,10.785,10.862,10.929,10.956,
     * 10.262,10.413,10.565,10.688,10.798,10.866,10.923,10.942,
     * 10.124,10.272,10.417,10.550,10.670,10.752,10.822,10.852,
     * 9.914,10.045,10.176,10.305,10.417,10.520,10.611,10.663,
     * 9.709,9.805,9.892,9.976,10.037,10.107,10.161,10.223,
     * 9.702,9.787,9.863,9.924,9.960,10.004,10.029,10.045,
     * 9.698,9.794,9.876,9.931,9.958,9.980,9.983,9.959,
     * 9.609,9.744,9.856,9.922,9.950,9.955,9.938,9.890,
     * 9.348,9.497,9.618,9.706,9.751,9.775,9.774,9.705,
     * 9.310,9.455,9.574,9.661,9.704,9.725,9.721,9.652,
     * 9.097,9.228,9.330,9.408,9.442,9.458,9.449,9.378,
     * 9.051,9.182,9.285,9.362,9.397,9.412,9.403,9.333/
      DATA ((EDEN(I,1,5,L,1),I=33,40),L=1,NZEN)/
     * 10.100,10.238,10.364,10.486,10.621,10.720,10.824,10.895,
     * 10.045,10.167,10.279,10.400,10.531,10.634,10.742,10.819,
     * 9.925,10.037,10.140,10.258,10.389,10.504,10.623,10.714,
     * 9.768,9.866,9.949,10.045,10.146,10.267,10.389,10.502,
     * 9.695,9.759,9.805,9.859,9.915,9.976,10.037,10.117,
     * 9.703,9.768,9.812,9.855,9.898,9.938,9.978,10.017,
     * 9.708,9.783,9.833,9.875,9.913,9.939,9.963,9.983,
     * 9.595,9.710,9.791,9.847,9.888,9.903,9.909,9.909,
     * 9.283,9.412,9.501,9.577,9.634,9.674,9.703,9.680,
     * 9.243,9.369,9.456,9.533,9.589,9.625,9.652,9.627,
     * 9.025,9.140,9.212,9.279,9.326,9.358,9.380,9.356,
     * 8.980,9.094,9.166,9.233,9.281,9.312,9.334,9.310/
      DATA ((EDEN(I,2,5,L,1),I=33,40),L=1,NZEN)/
     * 10.068,10.204,10.336,10.470,10.611,10.716,10.822,10.888,
     * 10.076,10.207,10.332,10.458,10.591,10.688,10.783,10.846,
     * 9.954,10.076,10.190,10.318,10.450,10.560,10.668,10.745,
     * 9.786,9.893,9.989,10.097,10.207,10.322,10.436,10.535,
     * 9.697,9.770,9.829,9.893,9.954,10.017,10.076,10.143,
     * 9.703,9.777,9.833,9.885,9.931,9.973,10.009,10.037,
     * 9.708,9.792,9.854,9.905,9.944,9.973,9.993,9.998,
     * 9.595,9.720,9.815,9.880,9.923,9.938,9.940,9.927,
     * 9.299,9.431,9.531,9.615,9.673,9.708,9.730,9.693,
     * 9.260,9.391,9.489,9.571,9.627,9.661,9.679,9.641,
     * 9.041,9.158,9.243,9.318,9.365,9.394,9.408,9.369,
     * 8.996,9.113,9.197,9.272,9.320,9.349,9.362,9.323/
      DATA ((EDEN(I,3,5,L,1),I=33,40),L=1,NZEN)/
     * 10.107,10.250,10.393,10.530,10.668,10.765,10.855,10.906,
     * 10.117,10.258,10.396,10.530,10.663,10.752,10.835,10.881,
     * 9.990,10.121,10.250,10.389,10.525,10.628,10.724,10.784,
     * 9.810,9.926,10.037,10.155,10.272,10.389,10.498,10.581,
     * 9.698,9.777,9.846,9.916,9.977,10.045,10.104,10.170,
     * 9.702,9.780,9.844,9.898,9.940,9.987,10.021,10.041,
     * 9.705,9.793,9.864,9.916,9.951,9.982,9.999,9.990,
     * 9.598,9.726,9.828,9.894,9.934,9.950,9.949,9.920,
     * 9.307,9.447,9.556,9.646,9.704,9.737,9.749,9.700,
     * 9.270,9.405,9.512,9.601,9.657,9.688,9.698,9.647,
     * 9.049,9.176,9.267,9.348,9.394,9.422,9.427,9.375,
     * 9.003,9.130,9.221,9.303,9.349,9.376,9.381,9.329/
      DATA ((EDEN(I,4,5,L,1),I=33,40),L=1,NZEN)/
     * 10.225,10.380,10.540,10.670,10.794,10.867,10.928,10.952,
     * 10.161,10.312,10.464,10.603,10.736,10.818,10.888,10.919,
     * 10.029,10.170,10.314,10.462,10.603,10.699,10.783,10.825,
     * 9.829,9.960,10.090,10.225,10.350,10.464,10.565,10.630,
     * 9.677,9.767,9.852,9.940,10.013,10.083,10.137,10.199,
     * 9.674,9.762,9.840,9.908,9.957,10.000,10.029,10.041,
     * 9.674,9.772,9.857,9.921,9.962,9.988,9.994,9.973,
     * 9.573,9.711,9.828,9.906,9.949,9.960,9.946,9.904,
     * 9.310,9.455,9.575,9.673,9.734,9.763,9.766,9.704,
     * 9.270,9.413,9.533,9.628,9.687,9.713,9.714,9.651,
     * 9.053,9.182,9.288,9.375,9.425,9.447,9.442,9.378,
     * 9.007,9.136,9.242,9.329,9.379,9.401,9.397,9.333/
      DATA ((EDEN(I,5,5,L,1),I=33,40),L=1,NZEN)/
     * 10.332,10.498,10.671,10.796,10.908,10.961,10.999,10.999,
     * 10.204,10.365,10.534,10.679,10.810,10.884,10.940,10.953,
     * 10.068,10.223,10.384,10.540,10.682,10.771,10.842,10.866,
     * 9.848,9.994,10.143,10.294,10.427,10.538,10.630,10.679,
     * 9.630,9.737,9.842,9.951,10.037,10.117,10.176,10.238,
     * 9.615,9.713,9.806,9.892,9.953,10.004,10.037,10.049,
     * 9.602,9.713,9.816,9.897,9.946,9.978,9.985,9.956,
     * 9.507,9.658,9.793,9.886,9.939,9.953,9.939,9.886,
     * 9.303,9.459,9.593,9.701,9.765,9.791,9.786,9.711,
     * 9.262,9.418,9.551,9.656,9.717,9.741,9.734,9.658,
     * 9.045,9.188,9.305,9.401,9.453,9.473,9.461,9.386,
     * 9.000,9.142,9.260,9.356,9.408,9.427,9.415,9.340/
      DATA ((EDEN(I,1,6,L,1),I=33,40),L=1,NZEN)/
     * 10.121,10.260,10.387,10.509,10.643,10.739,10.839,10.906,
     * 10.057,10.182,10.294,10.413,10.543,10.642,10.747,10.823,
     * 9.937,10.049,10.152,10.274,10.403,10.516,10.633,10.722,
     * 9.775,9.873,9.957,10.057,10.161,10.279,10.401,10.512,
     * 9.694,9.758,9.805,9.859,9.917,9.979,10.041,10.121,
     * 9.702,9.767,9.812,9.854,9.896,9.937,9.978,10.017,
     * 9.708,9.782,9.833,9.874,9.911,9.937,9.960,9.979,
     * 9.599,9.713,9.794,9.848,9.888,9.900,9.906,9.906,
     * 9.276,9.403,9.493,9.569,9.628,9.667,9.696,9.673,
     * 9.238,9.364,9.450,9.525,9.581,9.618,9.645,9.620,
     * 9.021,9.134,9.207,9.274,9.320,9.352,9.373,9.348,
     * 8.975,9.088,9.161,9.228,9.274,9.306,9.327,9.303/
      DATA ((EDEN(I,2,6,L,1),I=33,40),L=1,NZEN)/
     * 10.068,10.207,10.340,10.480,10.625,10.731,10.834,10.898,
     * 10.086,10.217,10.346,10.474,10.607,10.702,10.794,10.853,
     * 9.962,10.086,10.204,10.334,10.468,10.576,10.682,10.754,
     * 9.789,9.898,9.997,10.111,10.223,10.340,10.453,10.549,
     * 9.688,9.763,9.824,9.893,9.958,10.021,10.079,10.152,
     * 9.695,9.769,9.827,9.883,9.931,9.975,10.009,10.041,
     * 9.700,9.784,9.849,9.902,9.944,9.972,9.991,9.997,
     * 9.594,9.718,9.812,9.879,9.924,9.939,9.939,9.925,
     * 9.283,9.418,9.519,9.605,9.667,9.702,9.723,9.686,
     * 9.246,9.377,9.476,9.561,9.620,9.654,9.671,9.632,
     * 9.029,9.146,9.233,9.310,9.362,9.389,9.400,9.362,
     * 8.984,9.100,9.187,9.264,9.316,9.343,9.354,9.316/
      DATA ((EDEN(I,3,6,L,1),I=33,40),L=1,NZEN)/
     * 10.121,10.272,10.422,10.563,10.704,10.796,10.879,10.922,
     * 10.121,10.262,10.407,10.544,10.681,10.769,10.847,10.889,
     * 9.990,10.127,10.260,10.403,10.545,10.647,10.740,10.794,
     * 9.806,9.925,10.041,10.170,10.294,10.412,10.517,10.595,
     * 9.679,9.762,9.837,9.915,9.984,10.053,10.114,10.179,
     * 9.683,9.763,9.832,9.893,9.941,9.989,10.025,10.041,
     * 9.687,9.777,9.852,9.910,9.950,9.982,9.998,9.988,
     * 9.587,9.715,9.821,9.892,9.934,9.952,9.949,9.919,
     * 9.286,9.427,9.539,9.635,9.700,9.733,9.744,9.693,
     * 9.246,9.384,9.494,9.590,9.652,9.684,9.693,9.640,
     * 9.029,9.155,9.253,9.338,9.393,9.418,9.423,9.369,
     * 8.984,9.110,9.207,9.293,9.347,9.373,9.377,9.323/
      DATA ((EDEN(I,4,6,L,1),I=33,40),L=1,NZEN)/
     * 10.292,10.452,10.619,10.743,10.860,10.919,10.963,10.972,
     * 10.146,10.303,10.465,10.614,10.755,10.836,10.902,10.927,
     * 10.017,10.164,10.318,10.476,10.624,10.720,10.801,10.838,
     * 9.813,9.947,10.083,10.233,10.369,10.486,10.587,10.647,
     * 9.647,9.743,9.836,9.936,10.021,10.093,10.149,10.215,
     * 9.643,9.734,9.820,9.899,9.959,10.004,10.033,10.045,
     * 9.642,9.744,9.836,9.911,9.962,9.990,9.996,9.972,
     * 9.543,9.685,9.809,9.897,9.950,9.962,9.948,9.903,
     * 9.281,9.430,9.556,9.665,9.735,9.765,9.766,9.701,
     * 9.241,9.389,9.513,9.620,9.688,9.716,9.716,9.648,
     * 9.025,9.158,9.272,9.369,9.427,9.450,9.444,9.375,
     * 8.980,9.113,9.226,9.323,9.381,9.404,9.398,9.329/
      DATA ((EDEN(I,5,6,L,1),I=33,40),L=1,NZEN)/
     * 10.439,10.601,10.775,10.886,10.986,11.021,11.033,11.017,
     * 10.173,10.342,10.524,10.683,10.829,10.905,10.959,10.967,
     * 10.037,10.201,10.375,10.545,10.702,10.793,10.862,10.880,
     * 9.814,9.969,10.130,10.299,10.449,10.565,10.656,10.698,
     * 9.589,9.702,9.820,9.945,10.049,10.134,10.196,10.258,
     * 9.571,9.675,9.779,9.880,9.955,10.013,10.045,10.061,
     * 9.555,9.672,9.786,9.881,9.946,9.982,9.989,9.956,
     * 9.455,9.616,9.763,9.871,9.938,9.957,9.942,9.886,
     * 9.274,9.433,9.576,9.697,9.773,9.801,9.793,9.712,
     * 9.233,9.393,9.535,9.652,9.724,9.751,9.740,9.659,
     * 9.013,9.158,9.286,9.396,9.461,9.481,9.465,9.384,
     * 8.967,9.113,9.240,9.350,9.415,9.436,9.420,9.338/
      DATA ((EDEN(I,1,7,L,1),I=33,40),L=1,NZEN)/
     * 10.134,10.272,10.400,10.520,10.653,10.747,10.846,10.911,
     * 10.064,10.190,10.305,10.423,10.553,10.652,10.756,10.830,
     * 9.944,10.061,10.161,10.283,10.415,10.526,10.641,10.728,
     * 9.780,9.880,9.965,10.064,10.170,10.290,10.412,10.521,
     * 9.695,9.760,9.807,9.862,9.920,9.982,10.045,10.127,
     * 9.702,9.768,9.812,9.854,9.897,9.938,9.978,10.021,
     * 9.710,9.785,9.834,9.874,9.911,9.937,9.960,9.979,
     * 9.601,9.715,9.795,9.849,9.889,9.901,9.906,9.906,
     * 9.276,9.403,9.493,9.569,9.626,9.665,9.694,9.669,
     * 9.238,9.364,9.450,9.525,9.581,9.617,9.643,9.618,
     * 9.021,9.134,9.207,9.272,9.320,9.350,9.371,9.344,
     * 8.975,9.088,9.161,9.226,9.274,9.304,9.325,9.299/
      DATA ((EDEN(I,2,7,L,1),I=33,40),L=1,NZEN)/
     * 10.079,10.217,10.352,10.491,10.636,10.740,10.841,10.903,
     * 10.097,10.230,10.360,10.487,10.620,10.712,10.801,10.859,
     * 9.970,10.097,10.215,10.346,10.483,10.589,10.692,10.762,
     * 9.796,9.906,10.009,10.121,10.236,10.352,10.465,10.559,
     * 9.689,9.764,9.827,9.897,9.963,10.025,10.083,10.158,
     * 9.695,9.769,9.827,9.883,9.933,9.976,10.013,10.041,
     * 9.702,9.785,9.849,9.903,9.945,9.973,9.990,9.995,
     * 9.600,9.723,9.817,9.883,9.926,9.940,9.939,9.925,
     * 9.279,9.413,9.515,9.602,9.664,9.700,9.719,9.681,
     * 9.241,9.373,9.471,9.558,9.617,9.650,9.668,9.628,
     * 9.025,9.143,9.228,9.305,9.354,9.384,9.396,9.358,
     * 8.980,9.097,9.182,9.260,9.308,9.338,9.350,9.312/
      DATA ((EDEN(I,3,7,L,1),I=33,40),L=1,NZEN)/
     * 10.134,10.283,10.435,10.576,10.718,10.806,10.885,10.926,
     * 10.134,10.279,10.423,10.560,10.695,10.780,10.854,10.894,
     * 10.004,10.140,10.276,10.420,10.562,10.661,10.750,10.801,
     * 9.817,9.937,10.053,10.185,10.312,10.428,10.531,10.606,
     * 9.682,9.764,9.839,9.920,9.991,10.064,10.121,10.190,
     * 9.686,9.764,9.833,9.895,9.944,9.993,10.029,10.045,
     * 9.691,9.779,9.852,9.912,9.953,9.984,9.999,9.988,
     * 9.594,9.722,9.826,9.897,9.939,9.955,9.950,9.919,
     * 9.279,9.418,9.531,9.628,9.695,9.728,9.739,9.687,
     * 9.238,9.377,9.489,9.584,9.647,9.679,9.688,9.634,
     * 9.025,9.149,9.246,9.332,9.387,9.415,9.418,9.364,
     * 8.980,9.103,9.200,9.287,9.342,9.369,9.373,9.318/
      DATA ((EDEN(I,4,7,L,1),I=33,40),L=1,NZEN)/
     * 10.305,10.465,10.632,10.755,10.872,10.927,10.968,10.975,
     * 10.158,10.316,10.480,10.628,10.769,10.848,10.911,10.932,
     * 10.025,10.176,10.332,10.491,10.640,10.734,10.811,10.844,
     * 9.823,9.960,10.100,10.250,10.389,10.504,10.601,10.657,
     * 9.650,9.747,9.843,9.945,10.029,10.107,10.161,10.225,
     * 9.647,9.738,9.824,9.904,9.965,10.013,10.037,10.053,
     * 9.647,9.748,9.840,9.915,9.967,9.993,9.997,9.972,
     * 9.550,9.692,9.817,9.904,9.956,9.967,9.950,9.904,
     * 9.272,9.422,9.550,9.659,9.730,9.761,9.762,9.695,
     * 9.233,9.380,9.507,9.614,9.683,9.712,9.711,9.643,
     * 9.017,9.152,9.262,9.362,9.422,9.446,9.439,9.371,
     * 8.971,9.107,9.217,9.316,9.376,9.400,9.394,9.325/
      DATA ((EDEN(I,5,7,L,1),I=33,40),L=1,NZEN)/
     * 10.456,10.614,10.784,10.895,10.993,11.025,11.037,11.017,
     * 10.179,10.350,10.534,10.695,10.841,10.916,10.967,10.971,
     * 10.041,10.210,10.386,10.559,10.717,10.806,10.872,10.886,
     * 9.822,9.977,10.140,10.314,10.467,10.581,10.669,10.708,
     * 9.593,9.708,9.828,9.957,10.061,10.149,10.212,10.272,
     * 9.576,9.679,9.785,9.888,9.967,10.025,10.053,10.068,
     * 9.563,9.679,9.793,9.889,9.955,9.989,9.992,9.959,
     * 9.468,9.625,9.770,9.879,9.947,9.964,9.946,9.888,
     * 9.265,9.425,9.568,9.691,9.768,9.797,9.790,9.708,
     * 9.223,9.384,9.528,9.647,9.723,9.749,9.739,9.655,
     * 9.004,9.152,9.281,9.393,9.459,9.480,9.464,9.380,
     * 8.959,9.107,9.235,9.347,9.414,9.434,9.418,9.334/
      DATA ((EDEN(I,1,8,L,1),I=33,40),L=1,NZEN)/
     * 10.130,10.265,10.393,10.513,10.646,10.740,10.840,10.906,
     * 10.057,10.182,10.294,10.417,10.548,10.648,10.754,10.829,
     * 9.937,10.053,10.155,10.276,10.408,10.521,10.639,10.728,
     * 9.775,9.873,9.957,10.057,10.164,10.286,10.408,10.519,
     * 9.695,9.760,9.808,9.862,9.920,9.982,10.045,10.127,
     * 9.703,9.769,9.814,9.856,9.899,9.939,9.978,10.021,
     * 9.708,9.784,9.834,9.876,9.913,9.939,9.962,9.980,
     * 9.598,9.713,9.794,9.849,9.890,9.903,9.908,9.908,
     * 9.283,9.410,9.498,9.574,9.631,9.670,9.700,9.676,
     * 9.243,9.369,9.456,9.530,9.585,9.622,9.648,9.623,
     * 9.025,9.137,9.212,9.279,9.326,9.356,9.377,9.350,
     * 8.980,9.091,9.166,9.233,9.281,9.310,9.331,9.304/
      DATA ((EDEN(I,2,8,L,1),I=33,40),L=1,NZEN)/
     * 10.090,10.228,10.364,10.500,10.641,10.742,10.841,10.903,
     * 10.090,10.223,10.350,10.477,10.611,10.705,10.797,10.856,
     * 9.967,10.090,10.207,10.338,10.473,10.580,10.686,10.757,
     * 9.796,9.903,10.000,10.114,10.225,10.344,10.458,10.553,
     * 9.699,9.772,9.831,9.899,9.962,10.025,10.083,10.155,
     * 9.706,9.778,9.834,9.888,9.936,9.978,10.013,10.041,
     * 9.713,9.794,9.855,9.907,9.949,9.975,9.993,9.998,
     * 9.605,9.727,9.819,9.884,9.929,9.942,9.940,9.927,
     * 9.292,9.425,9.525,9.610,9.669,9.705,9.725,9.688,
     * 9.253,9.384,9.483,9.566,9.623,9.656,9.673,9.634,
     * 9.033,9.152,9.236,9.314,9.364,9.393,9.403,9.364,
     * 8.988,9.107,9.190,9.268,9.318,9.347,9.357,9.318/
      DATA ((EDEN(I,3,8,L,1),I=33,40),L=1,NZEN)/
     * 10.130,10.276,10.423,10.562,10.702,10.792,10.874,10.920,
     * 10.134,10.274,10.417,10.551,10.686,10.771,10.848,10.890,
     * 10.004,10.137,10.270,10.412,10.551,10.651,10.743,10.796,
     * 9.823,9.940,10.053,10.179,10.301,10.417,10.522,10.599,
     * 9.699,9.779,9.850,9.926,9.993,10.061,10.121,10.185,
     * 9.705,9.781,9.846,9.904,9.950,9.996,10.029,10.045,
     * 9.710,9.796,9.867,9.922,9.960,9.989,10.000,9.992,
     * 9.611,9.736,9.836,9.904,9.944,9.959,9.953,9.922,
     * 9.292,9.431,9.540,9.636,9.699,9.732,9.742,9.692,
     * 9.253,9.391,9.500,9.592,9.653,9.684,9.692,9.639,
     * 9.037,9.161,9.255,9.340,9.391,9.417,9.420,9.367,
     * 8.992,9.116,9.210,9.295,9.345,9.371,9.374,9.322/
      DATA ((EDEN(I,4,8,L,1),I=33,40),L=1,NZEN)/
     * 10.255,10.410,10.572,10.701,10.823,10.889,10.941,10.959,
     * 10.170,10.324,10.483,10.625,10.761,10.839,10.902,10.927,
     * 10.037,10.185,10.334,10.487,10.632,10.725,10.802,10.838,
     * 9.842,9.973,10.104,10.248,10.380,10.494,10.590,10.648,
     * 9.683,9.775,9.865,9.957,10.033,10.104,10.158,10.220,
     * 9.682,9.769,9.849,9.921,9.974,10.017,10.041,10.053,
     * 9.686,9.782,9.867,9.934,9.977,10.000,10.000,9.977,
     * 9.591,9.726,9.841,9.921,9.966,9.973,9.954,9.908,
     * 9.290,9.436,9.559,9.663,9.728,9.759,9.761,9.696,
     * 9.250,9.394,9.516,9.618,9.682,9.710,9.710,9.644,
     * 9.037,9.167,9.274,9.367,9.422,9.444,9.438,9.373,
     * 8.992,9.122,9.228,9.322,9.376,9.398,9.392,9.327/
      DATA ((EDEN(I,5,8,L,1),I=33,40),L=1,NZEN)/
     * 10.364,10.525,10.697,10.820,10.931,10.980,11.009,11.000,
     * 10.201,10.367,10.543,10.694,10.833,10.905,10.956,10.963,
     * 10.064,10.225,10.396,10.560,10.708,10.794,10.860,10.877,
     * 9.853,10.000,10.152,10.314,10.458,10.568,10.655,10.696,
     * 9.643,9.750,9.858,9.974,10.064,10.146,10.201,10.260,
     * 9.633,9.728,9.823,9.914,9.980,10.029,10.053,10.064,
     * 9.628,9.735,9.836,9.920,9.973,10.000,9.998,9.965,
     * 9.535,9.681,9.814,9.910,9.965,9.975,9.953,9.895,
     * 9.286,9.441,9.580,9.695,9.764,9.792,9.785,9.706,
     * 9.246,9.400,9.534,9.649,9.719,9.744,9.733,9.653,
     * 9.029,9.170,9.290,9.396,9.456,9.476,9.459,9.380,
     * 8.984,9.125,9.244,9.350,9.411,9.430,9.414,9.334/
      DATA ((EDEN(I,1,9,L,1),I=33,40),L=1,NZEN)/
     * 10.104,10.238,10.364,10.486,10.621,10.719,10.822,10.893,
     * 10.037,10.158,10.270,10.391,10.524,10.629,10.740,10.819,
     * 9.918,10.029,10.130,10.250,10.382,10.498,10.620,10.713,
     * 9.764,9.859,9.941,10.037,10.143,10.262,10.384,10.497,
     * 9.698,9.761,9.807,9.860,9.917,9.977,10.037,10.114,
     * 9.705,9.770,9.815,9.858,9.901,9.942,9.982,10.021,
     * 9.710,9.785,9.835,9.878,9.916,9.944,9.969,9.987,
     * 9.591,9.708,9.791,9.848,9.890,9.905,9.913,9.913,
     * 9.290,9.418,9.509,9.585,9.644,9.683,9.712,9.688,
     * 9.250,9.377,9.465,9.540,9.597,9.633,9.662,9.636,
     * 9.033,9.146,9.220,9.288,9.336,9.367,9.391,9.364,
     * 8.988,9.100,9.174,9.242,9.291,9.322,9.345,9.318/
      DATA ((EDEN(I,2,9,L,1),I=33,40),L=1,NZEN)/
     * 10.100,10.238,10.369,10.500,10.636,10.736,10.836,10.899,
     * 10.068,10.196,10.320,10.446,10.579,10.678,10.776,10.841,
     * 9.948,10.068,10.179,10.305,10.438,10.549,10.659,10.738,
     * 9.790,9.893,9.986,10.090,10.196,10.310,10.423,10.525,
     * 9.715,9.784,9.837,9.899,9.957,10.017,10.072,10.137,
     * 9.723,9.792,9.844,9.894,9.938,9.978,10.013,10.041,
     * 9.727,9.806,9.865,9.914,9.952,9.978,9.996,10.000,
     * 9.611,9.731,9.821,9.885,9.928,9.942,9.942,9.930,
     * 9.310,9.442,9.540,9.623,9.681,9.717,9.737,9.701,
     * 9.270,9.400,9.496,9.579,9.635,9.668,9.687,9.648,
     * 9.049,9.167,9.253,9.326,9.375,9.403,9.415,9.377,
     * 9.003,9.122,9.207,9.281,9.329,9.357,9.369,9.331/
      DATA ((EDEN(I,3,9,L,1),I=33,40),L=1,NZEN)/
     * 10.121,10.260,10.398,10.533,10.667,10.763,10.854,10.906,
     * 10.121,10.258,10.391,10.521,10.651,10.741,10.825,10.874,
     * 9.998,10.124,10.246,10.380,10.512,10.615,10.712,10.775,
     * 9.827,9.937,10.041,10.152,10.262,10.377,10.481,10.567,
     * 9.732,9.804,9.866,9.931,9.987,10.049,10.100,10.161,
     * 9.738,9.810,9.867,9.918,9.956,9.997,10.029,10.045,
     * 9.745,9.825,9.888,9.937,9.968,9.994,10.009,9.998,
     * 9.636,9.758,9.853,9.915,9.949,9.961,9.956,9.927,
     * 9.322,9.458,9.563,9.651,9.708,9.740,9.754,9.705,
     * 9.283,9.417,9.521,9.609,9.664,9.693,9.702,9.653,
     * 9.064,9.188,9.276,9.354,9.401,9.427,9.431,9.380,
     * 9.019,9.142,9.231,9.308,9.356,9.381,9.386,9.334/
      DATA ((EDEN(I,4,9,L,1),I=33,40),L=1,NZEN)/
     * 10.185,10.326,10.473,10.605,10.732,10.817,10.892,10.927,
     * 10.182,10.326,10.471,10.603,10.729,10.808,10.876,10.908,
     * 10.053,10.190,10.326,10.465,10.597,10.689,10.771,10.815,
     * 9.867,9.988,10.107,10.233,10.346,10.455,10.550,10.617,
     * 9.736,9.819,9.894,9.969,10.029,10.086,10.134,10.193,
     * 9.741,9.820,9.888,9.944,9.982,10.017,10.037,10.045,
     * 9.747,9.836,9.910,9.962,9.991,10.009,10.009,9.985,
     * 9.652,9.778,9.881,9.946,9.978,9.980,9.959,9.916,
     * 9.326,9.468,9.584,9.677,9.732,9.761,9.766,9.705,
     * 9.288,9.427,9.539,9.632,9.688,9.713,9.713,9.652,
     * 9.072,9.199,9.297,9.382,9.427,9.446,9.441,9.380,
     * 9.026,9.153,9.251,9.336,9.381,9.400,9.395,9.334/
      DATA ((EDEN(I,5,9,L,1),I=33,40),L=1,NZEN)/
     * 10.241,10.391,10.548,10.682,10.805,10.879,10.939,10.959,
     * 10.238,10.391,10.549,10.682,10.803,10.872,10.925,10.941,
     * 10.104,10.253,10.403,10.547,10.676,10.758,10.825,10.851,
     * 9.904,10.033,10.167,10.303,10.425,10.528,10.614,10.662,
     * 9.732,9.822,9.909,9.997,10.064,10.127,10.173,10.228,
     * 9.732,9.813,9.888,9.954,9.995,10.033,10.049,10.053,
     * 9.737,9.827,9.906,9.966,9.996,10.013,10.004,9.974,
     * 9.651,9.778,9.887,9.958,9.989,9.988,9.960,9.906,
     * 9.330,9.477,9.601,9.702,9.759,9.785,9.781,9.708,
     * 9.292,9.438,9.560,9.658,9.712,9.736,9.729,9.656,
     * 9.079,9.210,9.316,9.407,9.450,9.468,9.456,9.382,
     * 9.033,9.164,9.270,9.361,9.404,9.423,9.411,9.336/
      DATA ((EDEN(I,1,10,L,1),I=33,40),L=1,NZEN)/
     * 10.072,10.204,10.324,10.450,10.589,10.694,10.804,10.880,
     * 10.013,10.130,10.241,10.362,10.496,10.605,10.720,10.804,
     * 9.898,10.009,10.107,10.223,10.352,10.471,10.595,10.694,
     * 9.752,9.845,9.924,10.017,10.117,10.233,10.354,10.471,
     * 9.699,9.762,9.807,9.859,9.914,9.974,10.033,10.104,
     * 9.706,9.772,9.817,9.860,9.905,9.945,9.986,10.025,
     * 9.710,9.786,9.837,9.880,9.920,9.949,9.975,9.994,
     * 9.583,9.702,9.787,9.846,9.890,9.908,9.918,9.919,
     * 9.297,9.427,9.517,9.595,9.654,9.695,9.726,9.702,
     * 9.258,9.384,9.474,9.550,9.609,9.646,9.675,9.650,
     * 9.037,9.152,9.228,9.297,9.344,9.378,9.403,9.378,
     * 8.992,9.107,9.182,9.251,9.299,9.333,9.357,9.333/
      DATA ((EDEN(I,2,10,L,1),I=33,40),L=1,NZEN)/
     * 10.111,10.246,10.375,10.500,10.631,10.731,10.832,10.897,
     * 10.049,10.173,10.288,10.412,10.540,10.644,10.750,10.823,
     * 9.935,10.049,10.152,10.272,10.396,10.511,10.625,10.713,
     * 9.788,9.885,9.971,10.064,10.158,10.272,10.386,10.491,
     * 9.733,9.797,9.846,9.900,9.951,10.009,10.061,10.121,
     * 9.740,9.807,9.856,9.901,9.941,9.979,10.013,10.041,
     * 9.744,9.821,9.875,9.921,9.955,9.981,10.000,10.009,
     * 9.619,9.738,9.825,9.886,9.927,9.941,9.944,9.934,
     * 9.330,9.461,9.556,9.637,9.693,9.728,9.751,9.717,
     * 9.290,9.420,9.513,9.592,9.646,9.679,9.699,9.665,
     * 9.068,9.188,9.267,9.338,9.386,9.413,9.428,9.391,
     * 9.022,9.142,9.221,9.293,9.340,9.368,9.382,9.345/
      DATA ((EDEN(I,3,10,L,1),I=33,40),L=1,NZEN)/
     * 10.127,10.260,10.389,10.516,10.643,10.740,10.836,10.896,
     * 10.107,10.238,10.364,10.486,10.609,10.702,10.794,10.852,
     * 9.991,10.111,10.223,10.344,10.465,10.572,10.675,10.746,
     * 9.836,9.936,10.025,10.127,10.220,10.330,10.436,10.530,
     * 9.769,9.833,9.884,9.937,9.981,10.037,10.086,10.140,
     * 9.776,9.842,9.892,9.932,9.962,9.998,10.025,10.041,
     * 9.781,9.857,9.912,9.952,9.976,9.999,10.013,10.009,
     * 9.666,9.782,9.869,9.924,9.953,9.963,9.959,9.934,
     * 9.356,9.489,9.589,9.670,9.720,9.751,9.766,9.721,
     * 9.316,9.449,9.548,9.626,9.674,9.702,9.714,9.668,
     * 9.097,9.215,9.301,9.373,9.412,9.435,9.442,9.394,
     * 9.051,9.169,9.255,9.327,9.366,9.389,9.397,9.349/
      DATA ((EDEN(I,4,10,L,1),I=33,40),L=1,NZEN)/
     * 10.146,10.279,10.407,10.531,10.652,10.750,10.842,10.895,
     * 10.190,10.326,10.459,10.579,10.691,10.772,10.846,10.888,
     * 10.068,10.193,10.316,10.439,10.554,10.647,10.735,10.789,
     * 9.898,10.009,10.111,10.212,10.305,10.408,10.505,10.581,
     * 9.797,9.870,9.932,9.985,10.025,10.072,10.114,10.167,
     * 9.806,9.877,9.934,9.972,9.993,10.021,10.037,10.041,
     * 9.814,9.895,9.957,9.992,10.004,10.017,10.013,9.993,
     * 9.717,9.835,9.926,9.973,9.988,9.985,9.963,9.924,
     * 9.373,9.509,9.616,9.695,9.739,9.764,9.770,9.716,
     * 9.334,9.468,9.573,9.650,9.692,9.716,9.720,9.665,
     * 9.117,9.241,9.328,9.400,9.431,9.450,9.449,9.391,
     * 9.072,9.195,9.283,9.354,9.386,9.404,9.403,9.345/
      DATA ((EDEN(I,5,10,L,1),I=33,40),L=1,NZEN)/
     * 10.179,10.312,10.444,10.569,10.685,10.779,10.864,10.910,
     * 10.274,10.415,10.555,10.668,10.769,10.836,10.895,10.920,
     * 10.143,10.276,10.408,10.530,10.639,10.719,10.790,10.825,
     * 9.961,10.076,10.185,10.294,10.387,10.484,10.569,10.627,
     * 9.828,9.901,9.965,10.025,10.061,10.111,10.146,10.196,
     * 9.834,9.901,9.956,9.995,10.013,10.033,10.045,10.049,
     * 9.846,9.920,9.979,10.013,10.017,10.025,10.013,9.983,
     * 9.765,9.874,9.959,10.004,10.009,9.998,9.967,9.916,
     * 9.393,9.530,9.638,9.719,9.757,9.779,9.778,9.714,
     * 9.352,9.489,9.597,9.675,9.711,9.731,9.727,9.662,
     * 9.140,9.262,9.354,9.423,9.450,9.464,9.455,9.389,
     * 9.094,9.217,9.308,9.377,9.404,9.418,9.409,9.343/
      DATA ((EDEN(I,1,11,L,1),I=33,40),L=1,NZEN)/
     * 10.041,10.170,10.288,10.417,10.558,10.670,10.787,10.870,
     * 10.000,10.117,10.223,10.346,10.480,10.592,10.708,10.796,
     * 9.889,9.996,10.090,10.207,10.332,10.455,10.581,10.682,
     * 9.746,9.838,9.916,10.004,10.104,10.217,10.336,10.456,
     * 9.699,9.763,9.808,9.860,9.914,9.972,10.029,10.097,
     * 9.706,9.773,9.819,9.863,9.907,9.949,9.990,10.025,
     * 9.708,9.785,9.838,9.883,9.922,9.953,9.980,9.999,
     * 9.580,9.700,9.785,9.846,9.892,9.911,9.922,9.923,
     * 9.301,9.431,9.524,9.603,9.664,9.704,9.735,9.713,
     * 9.262,9.389,9.480,9.558,9.617,9.655,9.684,9.660,
     * 9.041,9.155,9.233,9.303,9.354,9.389,9.412,9.387,
     * 8.996,9.110,9.187,9.257,9.308,9.343,9.366,9.342/
      DATA ((EDEN(I,2,11,L,1),I=33,40),L=1,NZEN)/
     * 10.127,10.258,10.382,10.504,10.632,10.731,10.833,10.899,
     * 10.045,10.164,10.274,10.393,10.517,10.622,10.731,10.810,
     * 9.932,10.041,10.140,10.250,10.369,10.486,10.604,10.696,
     * 9.794,9.886,9.965,10.049,10.140,10.248,10.360,10.470,
     * 9.749,9.811,9.855,9.904,9.949,10.000,10.053,10.111,
     * 9.757,9.822,9.866,9.907,9.943,9.979,10.013,10.041,
     * 9.761,9.835,9.885,9.926,9.958,9.982,10.004,10.013,
     * 9.632,9.749,9.833,9.890,9.927,9.941,9.945,9.937,
     * 9.344,9.476,9.571,9.646,9.699,9.734,9.758,9.726,
     * 9.305,9.436,9.529,9.602,9.652,9.685,9.707,9.673,
     * 9.083,9.201,9.281,9.346,9.389,9.417,9.435,9.400,
     * 9.037,9.156,9.235,9.301,9.343,9.371,9.389,9.354/
      DATA ((EDEN(I,3,11,L,1),I=33,40),L=1,NZEN)/
     * 10.152,10.281,10.403,10.521,10.640,10.736,10.834,10.897,
     * 10.117,10.241,10.354,10.468,10.583,10.678,10.774,10.839,
     * 10.004,10.114,10.215,10.326,10.438,10.544,10.651,10.729,
     * 9.856,9.949,10.029,10.117,10.199,10.305,10.410,10.507,
     * 9.801,9.860,9.903,9.945,9.979,10.029,10.076,10.130,
     * 9.810,9.871,9.913,9.945,9.966,10.000,10.029,10.045,
     * 9.815,9.885,9.932,9.964,9.980,10.000,10.017,10.013,
     * 9.696,9.808,9.888,9.934,9.955,9.963,9.960,9.938,
     * 9.384,9.513,9.609,9.681,9.723,9.754,9.769,9.728,
     * 9.344,9.473,9.566,9.636,9.677,9.704,9.718,9.676,
     * 9.124,9.241,9.320,9.384,9.417,9.438,9.446,9.403,
     * 9.078,9.195,9.274,9.338,9.371,9.392,9.400,9.357/
      DATA ((EDEN(I,4,11,L,1),I=33,40),L=1,NZEN)/
     * 10.158,10.281,10.396,10.509,10.618,10.718,10.817,10.880,
     * 10.217,10.346,10.467,10.571,10.670,10.749,10.828,10.876,
     * 10.097,10.215,10.324,10.431,10.531,10.622,10.713,10.773,
     * 9.937,10.037,10.127,10.212,10.286,10.384,10.479,10.560,
     * 9.853,9.919,9.968,10.004,10.025,10.068,10.104,10.155,
     * 9.863,9.927,9.974,9.995,10.000,10.021,10.033,10.041,
     * 9.873,9.946,9.997,10.017,10.013,10.017,10.017,9.999,
     * 9.776,9.886,9.964,9.996,9.996,9.987,9.965,9.929,
     * 9.413,9.545,9.641,9.707,9.737,9.761,9.771,9.721,
     * 9.375,9.505,9.599,9.663,9.691,9.713,9.719,9.669,
     * 9.158,9.274,9.356,9.412,9.430,9.446,9.447,9.394,
     * 9.113,9.228,9.310,9.366,9.384,9.400,9.401,9.349/
      DATA ((EDEN(I,5,11,L,1),I=33,40),L=1,NZEN)/
     * 10.196,10.316,10.430,10.535,10.632,10.728,10.822,10.881,
     * 10.318,10.450,10.575,10.669,10.752,10.816,10.878,10.910,
     * 10.193,10.316,10.431,10.531,10.620,10.696,10.770,10.813,
     * 10.025,10.124,10.215,10.301,10.373,10.461,10.547,10.611,
     * 9.918,9.978,10.021,10.053,10.068,10.107,10.140,10.188,
     * 9.928,9.982,10.017,10.033,10.025,10.037,10.045,10.049,
     * 9.943,10.004,10.045,10.053,10.033,10.029,10.017,9.991,
     * 9.866,9.960,10.025,10.041,10.025,10.004,9.970,9.923,
     * 9.453,9.580,9.673,9.736,9.756,9.774,9.775,9.717,
     * 9.415,9.542,9.632,9.692,9.709,9.725,9.723,9.665,
     * 9.201,9.314,9.391,9.441,9.450,9.459,9.450,9.391,
     * 9.156,9.268,9.345,9.395,9.404,9.414,9.404,9.345/
      DATA ((EDEN(I,1,12,L,1),I=33,40),L=1,NZEN)/
     * 10.025,10.152,10.267,10.398,10.542,10.659,10.781,10.867,
     * 10.004,10.121,10.228,10.352,10.486,10.598,10.714,10.801,
     * 9.889,9.998,10.097,10.212,10.338,10.461,10.587,10.688,
     * 9.749,9.841,9.920,10.009,10.107,10.225,10.342,10.462,
     * 9.702,9.765,9.811,9.862,9.917,9.975,10.033,10.104,
     * 9.708,9.775,9.822,9.866,9.910,9.952,9.995,10.029,
     * 9.710,9.787,9.841,9.885,9.925,9.955,9.982,10.000,
     * 9.582,9.702,9.787,9.848,9.893,9.912,9.924,9.925,
     * 9.305,9.435,9.526,9.605,9.665,9.706,9.737,9.714,
     * 9.265,9.393,9.483,9.560,9.618,9.657,9.686,9.662,
     * 9.041,9.158,9.236,9.307,9.358,9.391,9.413,9.389,
     * 8.996,9.113,9.190,9.262,9.312,9.345,9.368,9.343/
      DATA ((EDEN(I,2,12,L,1),I=33,40),L=1,NZEN)/
     * 10.140,10.272,10.393,10.511,10.636,10.734,10.836,10.903,
     * 10.053,10.173,10.281,10.398,10.520,10.625,10.734,10.813,
     * 9.943,10.053,10.146,10.258,10.373,10.487,10.607,10.700,
     * 9.804,9.895,9.972,10.057,10.143,10.253,10.364,10.474,
     * 9.759,9.820,9.862,9.907,9.951,10.004,10.057,10.117,
     * 9.767,9.831,9.873,9.911,9.944,9.979,10.013,10.041,
     * 9.771,9.844,9.892,9.930,9.959,9.982,10.004,10.013,
     * 9.640,9.757,9.840,9.894,9.928,9.941,9.944,9.938,
     * 9.354,9.484,9.576,9.650,9.700,9.734,9.757,9.727,
     * 9.316,9.444,9.535,9.605,9.652,9.684,9.707,9.674,
     * 9.093,9.210,9.288,9.350,9.389,9.417,9.435,9.401,
     * 9.048,9.164,9.242,9.304,9.343,9.371,9.389,9.356/
      DATA ((EDEN(I,3,12,L,1),I=33,40),L=1,NZEN)/
     * 10.176,10.305,10.427,10.537,10.649,10.742,10.838,10.901,
     * 10.134,10.258,10.369,10.477,10.585,10.679,10.775,10.841,
     * 10.025,10.134,10.230,10.336,10.439,10.544,10.651,10.731,
     * 9.877,9.968,10.045,10.124,10.201,10.305,10.410,10.509,
     * 9.822,9.879,9.916,9.953,9.981,10.029,10.076,10.130,
     * 9.830,9.888,9.925,9.952,9.967,9.997,10.025,10.045,
     * 9.836,9.904,9.946,9.972,9.981,10.000,10.013,10.013,
     * 9.719,9.827,9.901,9.942,9.957,9.962,9.959,9.938,
     * 9.400,9.528,9.619,9.685,9.722,9.750,9.766,9.728,
     * 9.360,9.487,9.576,9.640,9.675,9.701,9.716,9.676,
     * 9.140,9.255,9.330,9.387,9.412,9.435,9.446,9.403,
     * 9.094,9.210,9.285,9.342,9.366,9.389,9.400,9.357/
      DATA ((EDEN(I,4,12,L,1),I=33,40),L=1,NZEN)/
     * 10.185,10.307,10.418,10.520,10.620,10.715,10.814,10.879,
     * 10.253,10.378,10.493,10.587,10.676,10.751,10.828,10.877,
     * 10.134,10.248,10.350,10.447,10.538,10.625,10.715,10.777,
     * 9.971,10.068,10.149,10.225,10.292,10.387,10.484,10.567,
     * 9.885,9.947,9.990,10.017,10.029,10.068,10.104,10.158,
     * 9.895,9.956,9.995,10.004,10.000,10.017,10.033,10.045,
     * 9.907,9.976,10.021,10.029,10.017,10.017,10.013,10.000,
     * 9.816,9.920,9.989,10.009,9.998,9.985,9.963,9.929,
     * 9.438,9.565,9.653,9.710,9.732,9.754,9.763,9.717,
     * 9.398,9.524,9.611,9.666,9.686,9.705,9.713,9.665,
     * 9.182,9.294,9.369,9.415,9.425,9.438,9.441,9.391,
     * 9.136,9.249,9.323,9.369,9.379,9.392,9.395,9.345/
      DATA ((EDEN(I,5,12,L,1),I=33,40),L=1,NZEN)/
     * 10.238,10.352,10.458,10.551,10.636,10.725,10.818,10.879,
     * 10.364,10.491,10.609,10.688,10.758,10.817,10.878,10.912,
     * 10.238,10.358,10.465,10.553,10.629,10.700,10.772,10.816,
     * 10.072,10.167,10.248,10.322,10.382,10.467,10.553,10.618,
     * 9.966,10.021,10.053,10.072,10.072,10.107,10.140,10.193,
     * 9.978,10.025,10.053,10.049,10.029,10.037,10.045,10.053,
     * 9.996,10.049,10.076,10.068,10.037,10.029,10.013,9.991,
     * 9.923,10.009,10.061,10.064,10.029,10.004,9.968,9.923,
     * 9.489,9.609,9.692,9.740,9.747,9.762,9.765,9.710,
     * 9.450,9.569,9.650,9.697,9.702,9.714,9.714,9.659,
     * 9.238,9.342,9.410,9.447,9.442,9.449,9.442,9.386,
     * 9.192,9.297,9.364,9.401,9.397,9.403,9.397,9.340/
      DATA ((EDEN(I,1,1,L,2),I=33,40),L=1,NZEN)/
     * 10.217,10.326,10.425,10.533,10.653,10.758,10.867,10.947,
     * 10.158,10.260,10.352,10.453,10.567,10.669,10.777,10.863,
     * 10.029,10.130,10.220,10.316,10.425,10.533,10.645,10.744,
     * 9.849,9.941,10.021,10.107,10.204,10.303,10.408,10.521,
     * 9.744,9.799,9.841,9.895,9.959,10.021,10.090,10.167,
     * 9.720,9.775,9.817,9.862,9.914,9.965,10.017,10.072,
     * 9.672,9.732,9.778,9.825,9.877,9.920,9.965,10.004,
     * 9.415,9.509,9.587,9.656,9.727,9.772,9.816,9.853,
     * 9.037,9.134,9.210,9.299,9.386,9.461,9.535,9.550,
     * 8.993,9.086,9.158,9.248,9.334,9.408,9.480,9.496,
     * 8.758,8.842,8.907,8.989,9.068,9.137,9.207,9.220,
     * 8.712,8.796,8.861,8.943,9.022,9.091,9.161,9.174/
      DATA ((EDEN(I,2,1,L,2),I=33,40),L=1,NZEN)/
     * 10.324,10.436,10.538,10.633,10.738,10.825,10.916,10.980,
     * 10.210,10.312,10.405,10.498,10.599,10.695,10.795,10.873,
     * 10.083,10.182,10.272,10.362,10.456,10.558,10.664,10.754,
     * 9.906,9.996,10.076,10.152,10.236,10.328,10.427,10.530,
     * 9.802,9.854,9.892,9.939,9.989,10.049,10.111,10.176,
     * 9.779,9.830,9.867,9.907,9.949,9.991,10.037,10.083,
     * 9.732,9.788,9.829,9.870,9.912,9.948,9.986,10.021,
     * 9.477,9.566,9.638,9.701,9.759,9.798,9.838,9.866,
     * 9.090,9.185,9.260,9.342,9.418,9.487,9.555,9.563,
     * 9.045,9.137,9.210,9.292,9.367,9.435,9.501,9.508,
     * 8.811,8.894,8.959,9.033,9.100,9.164,9.228,9.233,
     * 8.765,8.848,8.913,8.988,9.055,9.119,9.182,9.187/
      DATA ((EDEN(I,3,1,L,2),I=33,40),L=1,NZEN)/
     * 10.369,10.479,10.577,10.665,10.754,10.835,10.921,10.980,
     * 10.292,10.394,10.486,10.571,10.657,10.742,10.831,10.898,
     * 10.164,10.262,10.352,10.433,10.516,10.606,10.702,10.782,
     * 9.978,10.068,10.146,10.215,10.286,10.375,10.467,10.560,
     * 9.864,9.912,9.946,9.983,10.017,10.072,10.130,10.190,
     * 9.842,9.888,9.920,9.945,9.967,10.004,10.049,10.083,
     * 9.799,9.848,9.883,9.910,9.931,9.962,9.994,10.013,
     * 9.558,9.638,9.702,9.749,9.786,9.818,9.850,9.866,
     * 9.137,9.228,9.303,9.377,9.439,9.502,9.563,9.562,
     * 9.093,9.182,9.253,9.326,9.389,9.449,9.509,9.508,
     * 8.860,8.941,9.004,9.068,9.121,9.179,9.236,9.233,
     * 8.815,8.895,8.959,9.022,9.075,9.133,9.190,9.187/
      DATA ((EDEN(I,4,1,L,2),I=33,40),L=1,NZEN)/
     * 10.386,10.490,10.584,10.663,10.739,10.818,10.903,10.963,
     * 10.396,10.498,10.589,10.664,10.735,10.806,10.880,10.932,
     * 10.265,10.365,10.453,10.526,10.595,10.675,10.758,10.822,
     * 10.072,10.164,10.246,10.305,10.360,10.441,10.524,10.605,
     * 9.926,9.979,10.017,10.045,10.064,10.107,10.155,10.210,
     * 9.905,9.954,9.989,9.999,10.000,10.029,10.057,10.083,
     * 9.865,9.918,9.955,9.965,9.964,9.976,9.991,9.999,
     * 9.647,9.728,9.790,9.817,9.828,9.839,9.850,9.852,
     * 9.179,9.272,9.344,9.408,9.453,9.509,9.562,9.554,
     * 9.134,9.225,9.297,9.358,9.403,9.456,9.508,9.500,
     * 8.908,8.988,9.045,9.100,9.140,9.188,9.236,9.225,
     * 8.863,8.942,9.000,9.055,9.094,9.142,9.190,9.180/
      DATA ((EDEN(I,5,1,L,2),I=33,40),L=1,NZEN)/
     * 10.439,10.543,10.633,10.702,10.762,10.835,10.913,10.966,
     * 10.494,10.599,10.691,10.754,10.810,10.866,10.927,10.966,
     * 10.365,10.464,10.551,10.617,10.674,10.740,10.810,10.859,
     * 10.167,10.255,10.334,10.389,10.433,10.507,10.582,10.649,
     * 9.989,10.037,10.072,10.090,10.100,10.140,10.182,10.236,
     * 9.967,10.004,10.029,10.033,10.021,10.037,10.061,10.083,
     * 9.932,9.972,9.996,9.994,9.976,9.980,9.986,9.985,
     * 9.734,9.799,9.846,9.857,9.849,9.851,9.851,9.842,
     * 9.230,9.316,9.382,9.438,9.471,9.519,9.565,9.549,
     * 9.188,9.270,9.334,9.387,9.418,9.465,9.511,9.494,
     * 8.960,9.033,9.086,9.134,9.155,9.199,9.238,9.220,
     * 8.915,8.988,9.041,9.088,9.110,9.153,9.192,9.174/
      DATA ((EDEN(I,1,2,L,2),I=33,40),L=1,NZEN)/
     * 10.230,10.340,10.439,10.545,10.665,10.765,10.871,10.949,
     * 10.176,10.276,10.367,10.468,10.583,10.683,10.789,10.872,
     * 10.045,10.146,10.236,10.332,10.442,10.549,10.660,10.756,
     * 9.859,9.951,10.033,10.117,10.215,10.318,10.425,10.535,
     * 9.744,9.799,9.841,9.895,9.959,10.025,10.097,10.173,
     * 9.720,9.775,9.816,9.862,9.914,9.963,10.017,10.072,
     * 9.673,9.733,9.778,9.825,9.877,9.919,9.962,10.004,
     * 9.422,9.515,9.591,9.659,9.728,9.772,9.816,9.853,
     * 9.033,9.127,9.204,9.292,9.380,9.453,9.528,9.542,
     * 8.989,9.079,9.155,9.241,9.326,9.400,9.473,9.487,
     * 8.755,8.838,8.903,8.982,9.061,9.130,9.199,9.212,
     * 8.709,8.792,8.857,8.937,9.015,9.085,9.153,9.166/
      DATA ((EDEN(I,2,2,L,2),I=33,40),L=1,NZEN)/
     * 10.316,10.427,10.528,10.624,10.728,10.817,10.909,10.973,
     * 10.223,10.326,10.420,10.516,10.619,10.713,10.810,10.884,
     * 10.093,10.196,10.288,10.378,10.477,10.577,10.682,10.769,
     * 9.907,10.000,10.083,10.164,10.253,10.348,10.447,10.549,
     * 9.794,9.847,9.889,9.940,9.994,10.053,10.117,10.188,
     * 9.770,9.823,9.862,9.903,9.946,9.990,10.037,10.083,
     * 9.724,9.782,9.825,9.867,9.908,9.945,9.984,10.017,
     * 9.473,9.565,9.638,9.702,9.760,9.797,9.834,9.863,
     * 9.083,9.176,9.253,9.334,9.412,9.480,9.548,9.553,
     * 9.037,9.130,9.204,9.286,9.360,9.428,9.493,9.498,
     * 8.803,8.886,8.951,9.025,9.097,9.158,9.217,9.225,
     * 8.757,8.840,8.906,8.980,9.051,9.113,9.172,9.180/
      DATA ((EDEN(I,3,2,L,2),I=33,40),L=1,NZEN)/
     * 10.350,10.459,10.560,10.650,10.744,10.827,10.913,10.971,
     * 10.299,10.403,10.500,10.588,10.678,10.761,10.848,10.910,
     * 10.170,10.272,10.364,10.450,10.537,10.628,10.722,10.797,
     * 9.975,10.072,10.155,10.230,10.303,10.394,10.489,10.579,
     * 9.846,9.898,9.938,9.979,10.017,10.076,10.137,10.201,
     * 9.822,9.872,9.908,9.936,9.961,10.004,10.045,10.083,
     * 9.779,9.833,9.872,9.901,9.925,9.957,9.989,10.009,
     * 9.542,9.627,9.695,9.744,9.782,9.816,9.848,9.861,
     * 9.121,9.215,9.292,9.369,9.433,9.496,9.555,9.553,
     * 9.076,9.170,9.246,9.320,9.382,9.442,9.501,9.498,
     * 8.845,8.927,8.991,9.061,9.114,9.173,9.228,9.225,
     * 8.799,8.882,8.945,9.015,9.068,9.127,9.182,9.180/
      DATA ((EDEN(I,4,2,L,2),I=33,40),L=1,NZEN)/
     * 10.369,10.477,10.574,10.658,10.738,10.820,10.904,10.961,
     * 10.394,10.501,10.597,10.677,10.753,10.822,10.894,10.942,
     * 10.265,10.367,10.459,10.538,10.614,10.693,10.774,10.834,
     * 10.057,10.155,10.246,10.314,10.377,10.458,10.543,10.620,
     * 9.892,9.950,9.997,10.033,10.061,10.111,10.158,10.217,
     * 9.869,9.922,9.961,9.981,9.992,10.021,10.049,10.079,
     * 9.829,9.885,9.927,9.944,9.950,9.967,9.983,9.990,
     * 9.615,9.699,9.766,9.800,9.818,9.833,9.846,9.846,
     * 9.155,9.250,9.328,9.396,9.447,9.502,9.555,9.545,
     * 9.111,9.204,9.279,9.344,9.394,9.449,9.501,9.490,
     * 8.883,8.965,9.029,9.090,9.130,9.179,9.225,9.215,
     * 8.837,8.919,8.984,9.044,9.085,9.133,9.180,9.169/
      DATA ((EDEN(I,5,2,L,2),I=33,40),L=1,NZEN)/
     * 10.422,10.526,10.622,10.698,10.766,10.842,10.920,10.972,
     * 10.483,10.591,10.689,10.760,10.822,10.879,10.938,10.972,
     * 10.350,10.455,10.549,10.622,10.687,10.754,10.822,10.867,
     * 10.137,10.236,10.324,10.389,10.444,10.520,10.597,10.661,
     * 9.929,9.988,10.033,10.072,10.093,10.140,10.188,10.243,
     * 9.903,9.951,9.986,10.000,10.000,10.029,10.053,10.079,
     * 9.864,9.914,9.951,9.961,9.954,9.965,9.975,9.975,
     * 9.667,9.743,9.802,9.825,9.828,9.836,9.841,9.833,
     * 9.190,9.283,9.360,9.423,9.462,9.513,9.560,9.543,
     * 9.146,9.236,9.312,9.373,9.412,9.461,9.505,9.487,
     * 8.921,9.000,9.064,9.117,9.149,9.190,9.230,9.212,
     * 8.875,8.954,9.019,9.072,9.103,9.145,9.185,9.166/
      DATA ((EDEN(I,1,3,L,2),I=33,40),L=1,NZEN)/
     * 10.248,10.358,10.456,10.561,10.679,10.776,10.877,10.952,
     * 10.176,10.276,10.371,10.471,10.587,10.686,10.790,10.873,
     * 10.045,10.146,10.238,10.334,10.444,10.551,10.663,10.757,
     * 9.859,9.952,10.033,10.121,10.220,10.320,10.428,10.538,
     * 9.742,9.797,9.839,9.895,9.959,10.025,10.093,10.173,
     * 9.718,9.772,9.814,9.858,9.911,9.960,10.013,10.068,
     * 9.672,9.732,9.776,9.823,9.877,9.917,9.959,10.000,
     * 9.422,9.515,9.590,9.658,9.726,9.769,9.812,9.849,
     * 9.029,9.124,9.201,9.288,9.375,9.449,9.522,9.537,
     * 8.985,9.079,9.152,9.238,9.322,9.394,9.467,9.481,
     * 8.753,8.835,8.899,8.979,9.057,9.127,9.193,9.207,
     * 8.707,8.789,8.853,8.933,9.011,9.081,9.147,9.161/
      DATA ((EDEN(I,2,3,L,2),I=33,40),L=1,NZEN)/
     * 10.297,10.408,10.511,10.612,10.719,10.809,10.901,10.965,
     * 10.223,10.328,10.425,10.525,10.630,10.723,10.819,10.890,
     * 10.093,10.196,10.290,10.386,10.489,10.589,10.691,10.776,
     * 9.900,9.997,10.086,10.170,10.260,10.358,10.458,10.558,
     * 9.777,9.834,9.880,9.934,9.993,10.057,10.121,10.190,
     * 9.753,9.808,9.852,9.897,9.944,9.989,10.037,10.083,
     * 9.708,9.768,9.816,9.861,9.906,9.944,9.983,10.013,
     * 9.461,9.554,9.632,9.699,9.760,9.799,9.836,9.863,
     * 9.068,9.161,9.241,9.326,9.407,9.476,9.542,9.547,
     * 9.021,9.117,9.193,9.279,9.356,9.423,9.487,9.491,
     * 8.789,8.874,8.941,9.017,9.090,9.152,9.215,9.217,
     * 8.743,8.828,8.895,8.971,9.044,9.107,9.169,9.172/
      DATA ((EDEN(I,3,3,L,2),I=33,40),L=1,NZEN)/
     * 10.322,10.433,10.539,10.636,10.736,10.821,10.907,10.967,
     * 10.288,10.396,10.497,10.593,10.691,10.775,10.859,10.917,
     * 10.158,10.262,10.360,10.455,10.549,10.641,10.735,10.806,
     * 9.958,10.057,10.146,10.230,10.314,10.407,10.501,10.589,
     * 9.814,9.870,9.918,9.968,10.017,10.079,10.143,10.204,
     * 9.789,9.843,9.886,9.923,9.956,10.004,10.049,10.083,
     * 9.744,9.802,9.849,9.885,9.915,9.952,9.987,10.004,
     * 9.508,9.598,9.674,9.730,9.775,9.812,9.844,9.857,
     * 9.097,9.196,9.276,9.360,9.430,9.493,9.553,9.549,
     * 9.053,9.149,9.228,9.310,9.378,9.441,9.498,9.493,
     * 8.822,8.907,8.976,9.049,9.114,9.170,9.225,9.220,
     * 8.776,8.861,8.931,9.003,9.068,9.125,9.180,9.174/
      DATA ((EDEN(I,4,3,L,2),I=33,40),L=1,NZEN)/
     * 10.352,10.464,10.569,10.663,10.754,10.836,10.918,10.970,
     * 10.365,10.477,10.582,10.673,10.760,10.833,10.904,10.949,
     * 10.233,10.340,10.442,10.534,10.621,10.704,10.785,10.842,
     * 10.017,10.124,10.223,10.303,10.380,10.468,10.554,10.629,
     * 9.836,9.901,9.958,10.009,10.053,10.111,10.164,10.223,
     * 9.810,9.869,9.919,9.952,9.976,10.013,10.045,10.076,
     * 9.766,9.830,9.883,9.914,9.934,9.959,9.979,9.985,
     * 9.547,9.639,9.719,9.768,9.801,9.823,9.839,9.839,
     * 9.121,9.220,9.305,9.386,9.449,9.507,9.558,9.544,
     * 9.076,9.173,9.258,9.336,9.396,9.453,9.504,9.489,
     * 8.846,8.933,9.009,9.079,9.130,9.182,9.228,9.215,
     * 8.800,8.888,8.963,9.033,9.085,9.136,9.182,9.169/
      DATA ((EDEN(I,5,3,L,2),I=33,40),L=1,NZEN)/
     * 10.400,10.513,10.622,10.713,10.796,10.873,10.946,10.992,
     * 10.444,10.558,10.667,10.753,10.829,10.889,10.947,10.977,
     * 10.310,10.420,10.525,10.614,10.693,10.764,10.832,10.873,
     * 10.083,10.190,10.294,10.377,10.449,10.529,10.606,10.667,
     * 9.851,9.921,9.982,10.037,10.079,10.137,10.188,10.243,
     * 9.819,9.877,9.927,9.961,9.980,10.017,10.049,10.076,
     * 9.775,9.837,9.889,9.918,9.930,9.950,9.963,9.965,
     * 9.574,9.662,9.738,9.782,9.805,9.822,9.831,9.822,
     * 9.134,9.238,9.328,9.408,9.464,9.517,9.562,9.540,
     * 9.090,9.193,9.281,9.358,9.412,9.464,9.508,9.486,
     * 8.866,8.955,9.029,9.100,9.146,9.193,9.233,9.210,
     * 8.821,8.909,8.984,9.055,9.100,9.147,9.187,9.164/
      DATA ((EDEN(I,1,4,L,2),I=33,40),L=1,NZEN)/
     * 10.265,10.375,10.477,10.581,10.698,10.791,10.889,10.960,
     * 10.176,10.276,10.371,10.471,10.583,10.682,10.786,10.868,
     * 10.045,10.146,10.236,10.332,10.442,10.548,10.659,10.754,
     * 9.859,9.951,10.033,10.121,10.217,10.318,10.423,10.534,
     * 9.740,9.794,9.836,9.891,9.955,10.021,10.090,10.170,
     * 9.715,9.769,9.810,9.854,9.906,9.956,10.009,10.064,
     * 9.670,9.728,9.772,9.820,9.873,9.913,9.955,9.996,
     * 9.423,9.515,9.589,9.656,9.723,9.766,9.808,9.844,
     * 9.025,9.121,9.193,9.281,9.367,9.442,9.516,9.530,
     * 8.981,9.072,9.143,9.230,9.316,9.389,9.461,9.476,
     * 8.747,8.829,8.892,8.972,9.049,9.121,9.188,9.201,
     * 8.702,8.784,8.846,8.926,9.003,9.075,9.142,9.156/
      DATA ((EDEN(I,2,4,L,2),I=33,40),L=1,NZEN)/
     * 10.272,10.386,10.491,10.599,10.713,10.805,10.899,10.961,
     * 10.217,10.322,10.422,10.526,10.635,10.729,10.823,10.892,
     * 10.086,10.190,10.286,10.387,10.494,10.594,10.696,10.779,
     * 9.890,9.989,10.079,10.170,10.262,10.362,10.462,10.561,
     * 9.757,9.816,9.867,9.928,9.992,10.057,10.121,10.190,
     * 9.732,9.790,9.839,9.888,9.939,9.988,10.037,10.083,
     * 9.686,9.749,9.802,9.852,9.903,9.942,9.979,10.009,
     * 9.444,9.539,9.621,9.692,9.757,9.798,9.836,9.860,
     * 9.045,9.146,9.228,9.318,9.401,9.471,9.538,9.542,
     * 9.004,9.100,9.182,9.270,9.350,9.418,9.483,9.486,
     * 8.771,8.857,8.928,9.009,9.083,9.149,9.210,9.212,
     * 8.725,8.812,8.882,8.963,9.037,9.103,9.164,9.166/
      DATA ((EDEN(I,3,4,L,2),I=33,40),L=1,NZEN)/
     * 10.299,10.413,10.525,10.633,10.744,10.831,10.916,10.971,
     * 10.265,10.377,10.486,10.592,10.699,10.785,10.867,10.923,
     * 10.130,10.241,10.346,10.452,10.558,10.652,10.744,10.812,
     * 9.925,10.029,10.134,10.225,10.318,10.417,10.511,10.597,
     * 9.772,9.835,9.892,9.952,10.009,10.079,10.143,10.207,
     * 9.746,9.806,9.859,9.905,9.946,9.998,10.045,10.079,
     * 9.700,9.766,9.822,9.867,9.906,9.949,9.986,10.000,
     * 9.464,9.561,9.647,9.713,9.767,9.808,9.841,9.852,
     * 9.068,9.170,9.258,9.350,9.430,9.496,9.554,9.545,
     * 9.021,9.121,9.210,9.299,9.377,9.441,9.498,9.490,
     * 8.791,8.880,8.957,9.041,9.111,9.170,9.225,9.215,
     * 8.745,8.834,8.911,8.996,9.065,9.125,9.180,9.169/
      DATA ((EDEN(I,4,4,L,2),I=33,40),L=1,NZEN)/
     * 10.356,10.477,10.597,10.705,10.808,10.886,10.958,10.997,
     * 10.318,10.436,10.554,10.663,10.766,10.844,10.915,10.956,
     * 10.182,10.299,10.413,10.522,10.626,10.714,10.797,10.849,
     * 9.956,10.072,10.188,10.288,10.382,10.477,10.565,10.637,
     * 9.767,9.841,9.910,9.980,10.041,10.107,10.164,10.223,
     * 9.737,9.805,9.868,9.920,9.962,10.009,10.045,10.072,
     * 9.690,9.763,9.830,9.880,9.919,9.950,9.973,9.977,
     * 9.461,9.566,9.662,9.732,9.784,9.815,9.835,9.831,
     * 9.076,9.182,9.281,9.377,9.452,9.515,9.566,9.547,
     * 9.029,9.137,9.230,9.326,9.401,9.461,9.511,9.491,
     * 8.801,8.895,8.981,9.068,9.137,9.193,9.236,9.217,
     * 8.755,8.850,8.935,9.022,9.091,9.147,9.190,9.172/
      DATA ((EDEN(I,5,4,L,2),I=33,40),L=1,NZEN)/
     * 10.417,10.545,10.676,10.785,10.884,10.952,11.013,11.033,
     * 10.375,10.500,10.627,10.736,10.834,10.904,10.963,10.987,
     * 10.236,10.358,10.483,10.595,10.697,10.777,10.847,10.884,
     * 9.995,10.121,10.246,10.352,10.446,10.538,10.619,10.677,
     * 9.740,9.825,9.908,9.993,10.061,10.130,10.190,10.243,
     * 9.701,9.775,9.847,9.908,9.953,10.004,10.045,10.068,
     * 9.647,9.727,9.803,9.862,9.901,9.934,9.955,9.953,
     * 9.431,9.542,9.645,9.721,9.773,9.804,9.821,9.811,
     * 9.072,9.188,9.297,9.398,9.473,9.533,9.576,9.547,
     * 9.025,9.140,9.248,9.346,9.422,9.479,9.521,9.491,
     * 8.798,8.901,8.996,9.086,9.155,9.210,9.246,9.217,
     * 8.752,8.855,8.950,9.041,9.110,9.164,9.200,9.172/
      DATA ((EDEN(I,1,5,L,2),I=33,40),L=1,NZEN)/
     * 10.281,10.394,10.498,10.602,10.719,10.809,10.904,10.972,
     * 10.182,10.281,10.373,10.473,10.585,10.683,10.786,10.867,
     * 10.049,10.149,10.238,10.334,10.444,10.550,10.660,10.754,
     * 9.859,9.952,10.033,10.121,10.217,10.320,10.427,10.535,
     * 9.736,9.791,9.833,9.888,9.951,10.017,10.090,10.167,
     * 9.713,9.766,9.807,9.853,9.906,9.955,10.004,10.061,
     * 9.669,9.727,9.771,9.817,9.868,9.907,9.948,9.991,
     * 9.427,9.517,9.589,9.655,9.722,9.763,9.804,9.841,
     * 9.017,9.111,9.188,9.274,9.362,9.435,9.509,9.524,
     * 8.973,9.064,9.137,9.223,9.310,9.382,9.453,9.468,
     * 8.743,8.823,8.886,8.966,9.045,9.114,9.182,9.196,
     * 8.697,8.778,8.841,8.920,9.000,9.068,9.136,9.150/
      DATA ((EDEN(I,2,5,L,2),I=33,40),L=1,NZEN)/
     * 10.255,10.373,10.483,10.597,10.717,10.811,10.904,10.967,
     * 10.212,10.320,10.423,10.530,10.643,10.736,10.829,10.895,
     * 10.079,10.185,10.286,10.391,10.502,10.603,10.704,10.783,
     * 9.880,9.982,10.076,10.170,10.270,10.371,10.471,10.567,
     * 9.740,9.802,9.857,9.923,9.991,10.057,10.124,10.193,
     * 9.713,9.775,9.828,9.882,9.937,9.988,10.037,10.083,
     * 9.668,9.735,9.791,9.846,9.900,9.941,9.979,10.009,
     * 9.430,9.528,9.613,9.688,9.757,9.798,9.834,9.860,
     * 9.033,9.130,9.215,9.312,9.398,9.468,9.534,9.537,
     * 8.985,9.083,9.167,9.260,9.346,9.417,9.479,9.481,
     * 8.756,8.843,8.916,9.004,9.083,9.146,9.204,9.207,
     * 8.710,8.797,8.871,8.959,9.037,9.100,9.158,9.161/
      DATA ((EDEN(I,3,5,L,2),I=33,40),L=1,NZEN)/
     * 10.292,10.415,10.535,10.652,10.771,10.857,10.937,10.983,
     * 10.248,10.364,10.479,10.594,10.711,10.797,10.878,10.929,
     * 10.114,10.228,10.340,10.455,10.568,10.666,10.756,10.820,
     * 9.902,10.013,10.121,10.225,10.328,10.430,10.524,10.606,
     * 9.738,9.807,9.872,9.943,10.013,10.083,10.149,10.210,
     * 9.711,9.776,9.837,9.892,9.942,9.999,10.045,10.079,
     * 9.665,9.735,9.799,9.854,9.901,9.947,9.985,9.996,
     * 9.430,9.533,9.626,9.701,9.763,9.808,9.843,9.849,
     * 9.037,9.143,9.241,9.340,9.428,9.496,9.553,9.542,
     * 8.993,9.097,9.190,9.290,9.377,9.442,9.498,9.487,
     * 8.763,8.856,8.939,9.029,9.111,9.173,9.223,9.212,
     * 8.717,8.810,8.893,8.984,9.065,9.127,9.177,9.166/
      DATA ((EDEN(I,4,5,L,2),I=33,40),L=1,NZEN)/
     * 10.394,10.526,10.661,10.776,10.885,10.951,11.004,11.025,
     * 10.283,10.410,10.539,10.662,10.779,10.860,10.930,10.965,
     * 10.146,10.270,10.396,10.521,10.639,10.732,10.812,10.860,
     * 9.914,10.037,10.164,10.283,10.393,10.493,10.581,10.649,
     * 9.711,9.792,9.874,9.963,10.041,10.114,10.173,10.230,
     * 9.678,9.754,9.828,9.897,9.955,10.009,10.045,10.072,
     * 9.627,9.709,9.789,9.856,9.910,9.949,9.974,9.974,
     * 9.398,9.512,9.621,9.708,9.775,9.813,9.834,9.829,
     * 9.033,9.149,9.255,9.365,9.456,9.521,9.569,9.547,
     * 8.987,9.100,9.207,9.316,9.405,9.467,9.515,9.490,
     * 8.757,8.859,8.955,9.057,9.137,9.199,9.241,9.215,
     * 8.712,8.813,8.909,9.011,9.091,9.153,9.195,9.169/
      DATA ((EDEN(I,5,5,L,2),I=33,40),L=1,NZEN)/
     * 10.489,10.630,10.779,10.892,10.992,11.041,11.072,11.072,
     * 10.320,10.456,10.599,10.730,10.849,10.923,10.981,11.000,
     * 10.179,10.312,10.453,10.588,10.710,10.797,10.868,10.898,
     * 9.926,10.068,10.212,10.342,10.459,10.558,10.639,10.692,
     * 9.655,9.751,9.852,9.964,10.061,10.140,10.201,10.255,
     * 9.610,9.695,9.785,9.871,9.940,10.000,10.045,10.068,
     * 9.548,9.640,9.736,9.820,9.884,9.928,9.952,9.946,
     * 9.322,9.449,9.574,9.677,9.754,9.797,9.818,9.803,
     * 9.017,9.143,9.270,9.389,9.484,9.547,9.588,9.551,
     * 8.973,9.097,9.220,9.338,9.430,9.493,9.533,9.497,
     * 8.742,8.855,8.966,9.079,9.161,9.220,9.255,9.220,
     * 8.696,8.809,8.920,9.033,9.116,9.174,9.210,9.174/
      DATA ((EDEN(I,1,6,L,2),I=33,40),L=1,NZEN)/
     * 10.299,10.413,10.519,10.621,10.737,10.825,10.918,10.983,
     * 10.193,10.292,10.384,10.484,10.597,10.693,10.793,10.873,
     * 10.064,10.161,10.250,10.346,10.456,10.560,10.668,10.760,
     * 9.867,9.961,10.045,10.130,10.228,10.328,10.435,10.544,
     * 9.736,9.791,9.834,9.890,9.955,10.021,10.090,10.173,
     * 9.713,9.766,9.806,9.851,9.903,9.953,10.004,10.061,
     * 9.669,9.727,9.770,9.816,9.867,9.906,9.948,9.989,
     * 9.435,9.522,9.593,9.658,9.723,9.763,9.804,9.841,
     * 9.017,9.107,9.179,9.267,9.352,9.427,9.501,9.516,
     * 8.968,9.061,9.130,9.217,9.303,9.375,9.447,9.462,
     * 8.739,8.819,8.881,8.960,9.037,9.107,9.173,9.188,
     * 8.693,8.773,8.835,8.915,8.992,9.061,9.127,9.142/
      DATA ((EDEN(I,2,6,L,2),I=33,40),L=1,NZEN)/
     * 10.253,10.371,10.487,10.604,10.728,10.823,10.915,10.976,
     * 10.220,10.328,10.435,10.544,10.659,10.751,10.839,10.903,
     * 10.086,10.193,10.297,10.405,10.519,10.619,10.718,10.793,
     * 9.882,9.986,10.083,10.185,10.288,10.387,10.486,10.580,
     * 9.730,9.795,9.854,9.924,9.996,10.064,10.130,10.201,
     * 9.705,9.767,9.822,9.880,9.939,9.990,10.037,10.083,
     * 9.661,9.728,9.787,9.844,9.900,9.943,9.981,10.009,
     * 9.430,9.528,9.613,9.689,9.760,9.801,9.836,9.859,
     * 9.017,9.117,9.204,9.303,9.393,9.464,9.528,9.529,
     * 8.973,9.072,9.155,9.250,9.340,9.410,9.473,9.474,
     * 8.743,8.831,8.905,8.994,9.076,9.140,9.199,9.201,
     * 8.697,8.785,8.859,8.948,9.030,9.094,9.153,9.156/
      DATA ((EDEN(I,3,6,L,2),I=33,40),L=1,NZEN)/
     * 10.301,10.428,10.556,10.679,10.803,10.884,10.958,10.998,
     * 10.250,10.369,10.487,10.609,10.729,10.814,10.891,10.938,
     * 10.114,10.230,10.348,10.468,10.589,10.685,10.773,10.832,
     * 9.900,10.013,10.127,10.238,10.348,10.450,10.543,10.621,
     * 9.719,9.792,9.863,9.943,10.017,10.093,10.158,10.220,
     * 9.692,9.760,9.825,9.886,9.943,10.004,10.053,10.083,
     * 9.645,9.719,9.789,9.849,9.901,9.949,9.985,9.995,
     * 9.418,9.522,9.619,9.699,9.766,9.812,9.844,9.850,
     * 9.017,9.124,9.223,9.328,9.422,9.491,9.548,9.535,
     * 8.972,9.076,9.173,9.279,9.371,9.438,9.493,9.480,
     * 8.742,8.836,8.923,9.021,9.104,9.167,9.217,9.204,
     * 8.696,8.791,8.877,8.975,9.058,9.122,9.172,9.158/
      DATA ((EDEN(I,4,6,L,2),I=33,40),L=1,NZEN)/
     * 10.444,10.581,10.724,10.838,10.944,10.998,11.037,11.045,
     * 10.272,10.401,10.538,10.671,10.797,10.879,10.946,10.975,
     * 10.134,10.262,10.396,10.531,10.659,10.753,10.831,10.873,
     * 9.898,10.029,10.164,10.292,10.412,10.515,10.602,10.666,
     * 9.681,9.768,9.857,9.958,10.049,10.124,10.185,10.243,
     * 9.646,9.726,9.808,9.888,9.956,10.013,10.053,10.079,
     * 9.595,9.680,9.767,9.845,9.908,9.949,9.972,9.971,
     * 9.367,9.486,9.602,9.699,9.777,9.816,9.837,9.828,
     * 9.004,9.124,9.238,9.358,9.456,9.522,9.569,9.543,
     * 8.960,9.076,9.188,9.307,9.403,9.468,9.515,9.487,
     * 8.730,8.834,8.938,9.049,9.137,9.199,9.241,9.212,
     * 8.684,8.789,8.892,9.003,9.091,9.153,9.195,9.166/
      DATA ((EDEN(I,5,6,L,2),I=33,40),L=1,NZEN)/
     * 10.569,10.712,10.866,10.972,11.064,11.097,11.107,11.090,
     * 10.292,10.436,10.589,10.734,10.866,10.944,11.000,11.013,
     * 10.149,10.290,10.442,10.593,10.730,10.820,10.889,10.914,
     * 9.893,10.041,10.199,10.346,10.479,10.582,10.664,10.711,
     * 9.611,9.715,9.828,9.957,10.068,10.155,10.217,10.274,
     * 9.560,9.654,9.755,9.857,9.941,10.009,10.057,10.079,
     * 9.494,9.597,9.705,9.803,9.881,9.930,9.956,9.948,
     * 9.265,9.401,9.542,9.660,9.752,9.800,9.821,9.803,
     * 8.985,9.117,9.250,9.384,9.489,9.555,9.594,9.553,
     * 8.939,9.068,9.199,9.332,9.438,9.502,9.539,9.497,
     * 8.707,8.825,8.946,9.072,9.170,9.230,9.262,9.220,
     * 8.661,8.779,8.901,9.026,9.125,9.185,9.217,9.174/
      DATA ((EDEN(I,1,7,L,2),I=33,40),L=1,NZEN)/
     * 10.307,10.422,10.529,10.630,10.746,10.832,10.923,10.987,
     * 10.199,10.301,10.393,10.493,10.605,10.701,10.801,10.879,
     * 10.068,10.167,10.258,10.354,10.465,10.569,10.678,10.768,
     * 9.872,9.967,10.053,10.137,10.236,10.338,10.446,10.553,
     * 9.737,9.792,9.836,9.893,9.959,10.025,10.097,10.179,
     * 9.713,9.766,9.807,9.853,9.906,9.955,10.004,10.064,
     * 9.670,9.728,9.772,9.817,9.867,9.907,9.948,9.989,
     * 9.436,9.524,9.597,9.660,9.725,9.764,9.804,9.841,
     * 9.017,9.107,9.179,9.267,9.352,9.425,9.498,9.513,
     * 8.968,9.061,9.130,9.217,9.301,9.373,9.444,9.459,
     * 8.739,8.819,8.880,8.959,9.037,9.104,9.170,9.185,
     * 8.693,8.773,8.834,8.913,8.992,9.058,9.125,9.139/
      DATA ((EDEN(I,2,7,L,2),I=33,40),L=1,NZEN)/
     * 10.260,10.380,10.496,10.614,10.740,10.833,10.923,10.982,
     * 10.228,10.338,10.444,10.554,10.670,10.760,10.848,10.908,
     * 10.097,10.204,10.305,10.417,10.531,10.630,10.727,10.800,
     * 9.892,9.996,10.093,10.193,10.297,10.400,10.498,10.590,
     * 9.732,9.797,9.856,9.927,10.000,10.072,10.137,10.207,
     * 9.706,9.768,9.823,9.881,9.940,9.991,10.037,10.083,
     * 9.662,9.729,9.787,9.845,9.901,9.943,9.978,10.009,
     * 9.435,9.531,9.617,9.693,9.763,9.803,9.837,9.860,
     * 9.017,9.114,9.201,9.299,9.389,9.459,9.522,9.525,
     * 8.969,9.068,9.152,9.248,9.336,9.407,9.470,9.470,
     * 8.740,8.827,8.902,8.991,9.072,9.137,9.196,9.196,
     * 8.695,8.782,8.856,8.945,9.026,9.091,9.150,9.150/
      DATA ((EDEN(I,3,7,L,2),I=33,40),L=1,NZEN)/
     * 10.312,10.438,10.567,10.690,10.814,10.894,10.965,11.004,
     * 10.262,10.380,10.500,10.621,10.742,10.825,10.899,10.943,
     * 10.127,10.243,10.360,10.481,10.603,10.697,10.782,10.838,
     * 9.908,10.025,10.137,10.250,10.362,10.464,10.555,10.631,
     * 9.722,9.795,9.867,9.950,10.029,10.104,10.167,10.230,
     * 9.695,9.761,9.827,9.890,9.947,10.004,10.053,10.086,
     * 9.650,9.722,9.790,9.851,9.905,9.950,9.983,9.994,
     * 9.430,9.531,9.626,9.706,9.772,9.816,9.847,9.852,
     * 9.013,9.117,9.215,9.324,9.418,9.487,9.543,9.530,
     * 8.965,9.072,9.167,9.274,9.365,9.433,9.487,9.474,
     * 8.737,8.831,8.918,9.017,9.100,9.164,9.215,9.199,
     * 8.691,8.785,8.872,8.971,9.055,9.119,9.169,9.153/
      DATA ((EDEN(I,4,7,L,2),I=33,40),L=1,NZEN)/
     * 10.455,10.591,10.736,10.848,10.953,11.004,11.041,11.049,
     * 10.281,10.412,10.549,10.683,10.810,10.890,10.954,10.982,
     * 10.143,10.272,10.408,10.544,10.674,10.765,10.840,10.879,
     * 9.913,10.041,10.173,10.305,10.427,10.530,10.616,10.676,
     * 9.686,9.773,9.865,9.968,10.061,10.140,10.199,10.255,
     * 9.652,9.731,9.813,9.895,9.964,10.021,10.057,10.083,
     * 9.603,9.687,9.773,9.851,9.915,9.956,9.978,9.974,
     * 9.382,9.496,9.611,9.708,9.785,9.823,9.841,9.831,
     * 8.999,9.114,9.230,9.352,9.452,9.519,9.565,9.538,
     * 8.953,9.068,9.182,9.301,9.400,9.465,9.511,9.481,
     * 8.724,8.828,8.930,9.041,9.134,9.193,9.236,9.207,
     * 8.679,8.782,8.885,8.996,9.088,9.147,9.190,9.161/
      DATA ((EDEN(I,5,7,L,2),I=33,40),L=1,NZEN)/
     * 10.581,10.721,10.873,10.979,11.072,11.100,11.107,11.090,
     * 10.297,10.441,10.597,10.745,10.879,10.955,11.009,11.017,
     * 10.155,10.297,10.452,10.604,10.744,10.833,10.898,10.920,
     * 9.900,10.049,10.207,10.358,10.494,10.597,10.676,10.721,
     * 9.618,9.723,9.838,9.969,10.083,10.170,10.230,10.283,
     * 9.569,9.662,9.763,9.868,9.954,10.021,10.064,10.086,
     * 9.507,9.606,9.713,9.814,9.892,9.940,9.961,9.950,
     * 9.281,9.413,9.551,9.671,9.765,9.809,9.827,9.806,
     * 8.978,9.111,9.246,9.380,9.489,9.554,9.591,9.549,
     * 8.933,9.061,9.196,9.328,9.436,9.500,9.537,9.493,
     * 8.701,8.819,8.941,9.068,9.170,9.228,9.260,9.217,
     * 8.655,8.773,8.895,9.022,9.125,9.182,9.214,9.172/
      DATA ((EDEN(I,1,8,L,2),I=33,40),L=1,NZEN)/
     * 10.303,10.418,10.524,10.625,10.740,10.826,10.918,10.983,
     * 10.193,10.294,10.387,10.489,10.601,10.698,10.800,10.879,
     * 10.064,10.161,10.253,10.350,10.461,10.566,10.675,10.766,
     * 9.867,9.962,10.045,10.134,10.230,10.334,10.441,10.550,
     * 9.737,9.793,9.836,9.893,9.959,10.025,10.097,10.176,
     * 9.714,9.768,9.808,9.854,9.906,9.955,10.004,10.064,
     * 9.669,9.728,9.772,9.818,9.869,9.907,9.948,9.991,
     * 9.430,9.520,9.593,9.659,9.725,9.764,9.804,9.841,
     * 9.017,9.111,9.185,9.272,9.358,9.431,9.505,9.520,
     * 8.973,9.064,9.137,9.223,9.307,9.378,9.450,9.464,
     * 8.744,8.824,8.886,8.965,9.041,9.111,9.176,9.190,
     * 8.698,8.778,8.841,8.919,8.996,9.065,9.130,9.145/
      DATA ((EDEN(I,2,8,L,2),I=33,40),L=1,NZEN)/
     * 10.274,10.393,10.507,10.621,10.743,10.834,10.924,10.982,
     * 10.223,10.332,10.436,10.545,10.662,10.753,10.842,10.905,
     * 10.090,10.199,10.299,10.408,10.522,10.622,10.720,10.796,
     * 9.887,9.990,10.086,10.185,10.288,10.391,10.490,10.583,
     * 9.740,9.803,9.860,9.928,10.000,10.068,10.134,10.204,
     * 9.716,9.776,9.829,9.885,9.943,9.994,10.041,10.086,
     * 9.671,9.737,9.793,9.849,9.905,9.945,9.980,10.009,
     * 9.438,9.534,9.618,9.694,9.763,9.805,9.840,9.863,
     * 9.025,9.124,9.210,9.305,9.394,9.465,9.529,9.530,
     * 8.980,9.076,9.161,9.258,9.344,9.413,9.474,9.476,
     * 8.750,8.837,8.911,8.999,9.079,9.143,9.201,9.201,
     * 8.704,8.791,8.865,8.953,9.033,9.097,9.156,9.156/
      DATA ((EDEN(I,3,8,L,2),I=33,40),L=1,NZEN)/
     * 10.310,10.435,10.560,10.679,10.800,10.882,10.956,10.997,
     * 10.262,10.378,10.496,10.615,10.732,10.816,10.892,10.938,
     * 10.127,10.241,10.356,10.474,10.593,10.688,10.774,10.833,
     * 9.915,10.025,10.137,10.246,10.354,10.453,10.545,10.623,
     * 9.741,9.810,9.877,9.954,10.025,10.100,10.164,10.225,
     * 9.715,9.779,9.839,9.900,9.954,10.009,10.053,10.086,
     * 9.672,9.740,9.803,9.862,9.912,9.956,9.988,10.000,
     * 9.447,9.544,9.636,9.713,9.777,9.820,9.849,9.855,
     * 9.025,9.130,9.228,9.332,9.422,9.491,9.547,9.534,
     * 8.982,9.083,9.179,9.281,9.371,9.438,9.491,9.480,
     * 8.752,8.844,8.928,9.025,9.107,9.170,9.217,9.207,
     * 8.706,8.798,8.882,8.980,9.061,9.125,9.172,9.161/
      DATA ((EDEN(I,4,8,L,2),I=33,40),L=1,NZEN)/
     * 10.415,10.548,10.686,10.801,10.910,10.972,11.021,11.033,
     * 10.292,10.420,10.553,10.680,10.802,10.881,10.945,10.975,
     * 10.155,10.281,10.410,10.542,10.665,10.755,10.831,10.872,
     * 9.931,10.053,10.182,10.305,10.420,10.520,10.604,10.667,
     * 9.719,9.801,9.887,9.982,10.064,10.137,10.193,10.248,
     * 9.688,9.763,9.839,9.913,9.974,10.025,10.061,10.083,
     * 9.642,9.721,9.801,9.872,9.928,9.964,9.983,9.980,
     * 9.422,9.531,9.639,9.728,9.797,9.831,9.846,9.835,
     * 9.017,9.130,9.243,9.358,9.452,9.519,9.565,9.539,
     * 8.973,9.086,9.193,9.307,9.401,9.465,9.511,9.483,
     * 8.745,8.846,8.943,9.049,9.137,9.196,9.236,9.210,
     * 8.699,8.800,8.898,9.003,9.091,9.150,9.190,9.164/
      DATA ((EDEN(I,5,8,L,2),I=33,40),L=1,NZEN)/
     * 10.508,10.647,10.797,10.911,11.013,11.057,11.083,11.076,
     * 10.318,10.456,10.603,10.742,10.868,10.943,10.996,11.009,
     * 10.176,10.314,10.459,10.603,10.734,10.820,10.885,10.910,
     * 9.934,10.072,10.220,10.360,10.486,10.584,10.662,10.708,
     * 9.672,9.769,9.873,9.991,10.090,10.170,10.225,10.276,
     * 9.631,9.715,9.806,9.898,9.971,10.029,10.064,10.086,
     * 9.576,9.667,9.762,9.849,9.915,9.954,9.969,9.959,
     * 9.356,9.477,9.602,9.708,9.787,9.825,9.837,9.816,
     * 9.004,9.130,9.255,9.384,9.486,9.550,9.588,9.548,
     * 8.960,9.083,9.210,9.334,9.433,9.497,9.533,9.493,
     * 8.730,8.841,8.955,9.076,9.164,9.225,9.258,9.217,
     * 8.684,8.796,8.909,9.030,9.119,9.180,9.212,9.172/
      DATA ((EDEN(I,1,9,L,2),I=33,40),L=1,NZEN)/
     * 10.286,10.398,10.501,10.604,10.720,10.810,10.904,10.972,
     * 10.170,10.272,10.365,10.467,10.580,10.679,10.783,10.866,
     * 10.041,10.143,10.233,10.328,10.438,10.545,10.656,10.751,
     * 9.855,9.947,10.029,10.117,10.212,10.314,10.422,10.531,
     * 9.739,9.793,9.835,9.891,9.955,10.021,10.090,10.167,
     * 9.715,9.769,9.810,9.854,9.906,9.956,10.009,10.064,
     * 9.669,9.728,9.772,9.820,9.873,9.913,9.955,9.996,
     * 9.422,9.513,9.588,9.655,9.723,9.766,9.808,9.844,
     * 9.025,9.121,9.193,9.281,9.369,9.442,9.517,9.531,
     * 8.981,9.072,9.143,9.230,9.316,9.389,9.462,9.477,
     * 8.748,8.830,8.893,8.974,9.053,9.121,9.188,9.201,
     * 8.702,8.784,8.847,8.928,9.007,9.075,9.142,9.156/
      DATA ((EDEN(I,2,9,L,2),I=33,40),L=1,NZEN)/
     * 10.286,10.401,10.512,10.621,10.738,10.828,10.918,10.977,
     * 10.204,10.312,10.413,10.519,10.631,10.726,10.821,10.889,
     * 10.072,10.179,10.276,10.380,10.490,10.592,10.695,10.776,
     * 9.883,9.981,10.072,10.164,10.262,10.362,10.459,10.556,
     * 9.756,9.815,9.865,9.928,9.994,10.057,10.121,10.190,
     * 9.732,9.790,9.838,9.891,9.944,9.993,10.037,10.083,
     * 9.687,9.749,9.801,9.854,9.907,9.945,9.981,10.013,
     * 9.441,9.537,9.619,9.693,9.760,9.801,9.838,9.862,
     * 9.041,9.140,9.225,9.320,9.407,9.477,9.542,9.545,
     * 9.000,9.097,9.176,9.270,9.356,9.425,9.487,9.490,
     * 8.767,8.853,8.925,9.013,9.090,9.155,9.212,9.215,
     * 8.721,8.807,8.879,8.967,9.044,9.110,9.166,9.169/
      DATA ((EDEN(I,3,9,L,2),I=33,40),L=1,NZEN)/
     * 10.305,10.425,10.542,10.655,10.769,10.856,10.936,10.985,
     * 10.248,10.362,10.474,10.587,10.698,10.785,10.867,10.921,
     * 10.117,10.228,10.336,10.447,10.556,10.653,10.744,10.809,
     * 9.914,10.021,10.124,10.223,10.320,10.418,10.511,10.593,
     * 9.773,9.834,9.892,9.959,10.021,10.086,10.146,10.207,
     * 9.748,9.807,9.860,9.912,9.958,10.009,10.053,10.083,
     * 9.704,9.767,9.824,9.876,9.921,9.961,9.994,10.004,
     * 9.470,9.563,9.649,9.720,9.780,9.820,9.851,9.859,
     * 9.057,9.158,9.250,9.348,9.433,9.501,9.559,9.548,
     * 9.013,9.111,9.204,9.299,9.382,9.447,9.504,9.494,
     * 8.782,8.871,8.950,9.041,9.117,9.179,9.230,9.220,
     * 8.736,8.825,8.904,8.996,9.072,9.133,9.185,9.174/
      DATA ((EDEN(I,4,9,L,2),I=33,40),L=1,NZEN)/
     * 10.362,10.484,10.609,10.723,10.832,10.907,10.973,11.004,
     * 10.303,10.422,10.544,10.660,10.770,10.849,10.918,10.955,
     * 10.167,10.286,10.403,10.521,10.631,10.720,10.799,10.848,
     * 9.954,10.068,10.182,10.290,10.389,10.484,10.568,10.636,
     * 9.776,9.848,9.920,9.995,10.061,10.124,10.173,10.225,
     * 9.749,9.816,9.881,9.939,9.985,10.029,10.057,10.079,
     * 9.706,9.777,9.845,9.901,9.943,9.972,9.987,9.985,
     * 9.484,9.584,9.680,9.754,9.809,9.838,9.851,9.843,
     * 9.061,9.167,9.270,9.373,9.458,9.521,9.569,9.549,
     * 9.017,9.121,9.220,9.324,9.407,9.468,9.515,9.493,
     * 8.787,8.883,8.971,9.068,9.143,9.199,9.243,9.220,
     * 8.742,8.837,8.925,9.022,9.097,9.153,9.197,9.174/
      DATA ((EDEN(I,5,9,L,2),I=33,40),L=1,NZEN)/
     * 10.412,10.542,10.677,10.794,10.900,10.968,11.021,11.037,
     * 10.350,10.477,10.610,10.731,10.839,10.910,10.966,10.986,
     * 10.215,10.338,10.467,10.591,10.702,10.783,10.849,10.882,
     * 9.989,10.111,10.236,10.354,10.456,10.548,10.622,10.675,
     * 9.768,9.846,9.928,10.017,10.090,10.155,10.204,10.250,
     * 9.737,9.806,9.876,9.943,9.992,10.037,10.064,10.083,
     * 9.692,9.764,9.838,9.900,9.943,9.971,9.981,9.972,
     * 9.480,9.581,9.680,9.760,9.816,9.841,9.848,9.829,
     * 9.061,9.173,9.286,9.398,9.483,9.544,9.584,9.551,
     * 9.017,9.127,9.236,9.346,9.431,9.491,9.529,9.496,
     * 8.790,8.889,8.987,9.090,9.167,9.220,9.255,9.220,
     * 8.744,8.844,8.941,9.044,9.122,9.174,9.210,9.174/
      DATA ((EDEN(I,1,10,L,2),I=33,40),L=1,NZEN)/
     * 10.258,10.369,10.470,10.574,10.691,10.786,10.886,10.959,
     * 10.149,10.250,10.342,10.442,10.555,10.657,10.765,10.852,
     * 10.021,10.121,10.210,10.305,10.412,10.520,10.633,10.732,
     * 9.843,9.932,10.013,10.097,10.190,10.290,10.396,10.508,
     * 9.740,9.793,9.835,9.889,9.951,10.017,10.083,10.158,
     * 9.715,9.770,9.812,9.857,9.911,9.960,10.013,10.064,
     * 9.668,9.728,9.773,9.820,9.873,9.915,9.959,10.000,
     * 9.413,9.507,9.582,9.651,9.721,9.766,9.812,9.848,
     * 9.029,9.124,9.201,9.292,9.380,9.455,9.530,9.547,
     * 8.985,9.079,9.155,9.243,9.328,9.401,9.476,9.491,
     * 8.753,8.836,8.900,8.983,9.064,9.134,9.201,9.217,
     * 8.707,8.790,8.855,8.937,9.019,9.088,9.156,9.172/
      DATA ((EDEN(I,2,10,L,2),I=33,40),L=1,NZEN)/
     * 10.297,10.410,10.517,10.622,10.734,10.823,10.914,10.975,
     * 10.185,10.290,10.387,10.489,10.597,10.695,10.793,10.869,
     * 10.057,10.161,10.255,10.350,10.453,10.558,10.661,10.750,
     * 9.879,9.971,10.053,10.140,10.233,10.328,10.425,10.526,
     * 9.773,9.828,9.873,9.929,9.987,10.049,10.111,10.173,
     * 9.750,9.805,9.849,9.897,9.946,9.993,10.037,10.083,
     * 9.702,9.763,9.810,9.860,9.908,9.947,9.985,10.017,
     * 9.449,9.543,9.621,9.692,9.757,9.797,9.834,9.862,
     * 9.064,9.161,9.243,9.334,9.418,9.489,9.555,9.561,
     * 9.017,9.114,9.196,9.283,9.365,9.436,9.501,9.505,
     * 8.785,8.871,8.941,9.025,9.100,9.164,9.225,9.230,
     * 8.739,8.825,8.895,8.980,9.055,9.119,9.180,9.185/
      DATA ((EDEN(I,3,10,L,2),I=33,40),L=1,NZEN)/
     * 10.314,10.428,10.538,10.643,10.750,10.836,10.921,10.975,
     * 10.243,10.350,10.452,10.555,10.658,10.747,10.836,10.898,
     * 10.114,10.217,10.318,10.417,10.515,10.612,10.707,10.782,
     * 9.925,10.021,10.111,10.199,10.286,10.378,10.470,10.559,
     * 9.810,9.865,9.912,9.965,10.017,10.076,10.134,10.188,
     * 9.786,9.839,9.884,9.927,9.965,10.013,10.053,10.083,
     * 9.741,9.799,9.848,9.891,9.928,9.966,9.998,10.013,
     * 9.496,9.587,9.666,9.729,9.782,9.821,9.853,9.863,
     * 9.093,9.190,9.276,9.367,9.446,9.512,9.571,9.565,
     * 9.049,9.143,9.228,9.318,9.394,9.459,9.516,9.509,
     * 8.814,8.901,8.975,9.057,9.127,9.188,9.241,9.236,
     * 8.768,8.856,8.930,9.011,9.081,9.142,9.195,9.190/
      DATA ((EDEN(I,4,10,L,2),I=33,40),L=1,NZEN)/
     * 10.334,10.449,10.560,10.665,10.764,10.849,10.928,10.976,
     * 10.316,10.427,10.535,10.637,10.734,10.812,10.886,10.931,
     * 10.185,10.294,10.400,10.498,10.593,10.680,10.762,10.820,
     * 9.988,10.090,10.188,10.274,10.358,10.444,10.526,10.601,
     * 9.839,9.901,9.959,10.013,10.057,10.111,10.158,10.207,
     * 9.816,9.874,9.927,9.967,9.996,10.029,10.057,10.079,
     * 9.773,9.836,9.893,9.931,9.958,9.981,9.997,9.997,
     * 9.549,9.640,9.723,9.780,9.820,9.843,9.855,9.851,
     * 9.111,9.212,9.303,9.394,9.464,9.526,9.576,9.561,
     * 9.068,9.167,9.255,9.344,9.413,9.473,9.521,9.505,
     * 8.838,8.926,9.004,9.086,9.149,9.204,9.248,9.230,
     * 8.792,8.881,8.959,9.041,9.103,9.158,9.202,9.185/
      DATA ((EDEN(I,5,10,L,2),I=33,40),L=1,NZEN)/
     * 10.367,10.483,10.600,10.704,10.798,10.880,10.952,10.993,
     * 10.389,10.504,10.618,10.719,10.807,10.874,10.933,10.963,
     * 10.255,10.367,10.477,10.579,10.668,10.746,10.814,10.856,
     * 10.045,10.149,10.253,10.346,10.427,10.509,10.582,10.641,
     * 9.857,9.922,9.983,10.041,10.086,10.140,10.182,10.225,
     * 9.833,9.889,9.941,9.982,10.009,10.041,10.061,10.076,
     * 9.792,9.852,9.906,9.944,9.963,9.982,9.989,9.981,
     * 9.585,9.671,9.749,9.803,9.834,9.851,9.854,9.838,
     * 9.127,9.230,9.324,9.417,9.484,9.542,9.584,9.559,
     * 9.083,9.185,9.276,9.367,9.433,9.489,9.530,9.504,
     * 8.790,8.872,8.940,9.021,9.086,9.149,9.201,9.196,
     * 8.744,8.826,8.894,8.975,9.041,9.103,9.156,9.150/
      DATA ((EDEN(I,1,11,L,2),I=33,40),L=1,NZEN)/
     * 10.228,10.338,10.439,10.547,10.666,10.766,10.872,10.950,
     * 10.140,10.238,10.328,10.428,10.539,10.644,10.754,10.843,
     * 10.009,10.107,10.196,10.290,10.398,10.507,10.619,10.721,
     * 9.837,9.925,10.000,10.086,10.179,10.279,10.382,10.494,
     * 9.740,9.794,9.836,9.888,9.948,10.013,10.079,10.152,
     * 9.715,9.770,9.812,9.860,9.914,9.962,10.013,10.064,
     * 9.667,9.728,9.774,9.823,9.877,9.920,9.965,10.004,
     * 9.410,9.502,9.580,9.649,9.720,9.766,9.812,9.849,
     * 9.033,9.130,9.207,9.297,9.387,9.464,9.539,9.555,
     * 8.989,9.083,9.158,9.248,9.336,9.410,9.484,9.500,
     * 8.755,8.839,8.906,8.988,9.068,9.140,9.210,9.225,
     * 8.709,8.794,8.860,8.942,9.022,9.094,9.164,9.180/
      DATA ((EDEN(I,2,11,L,2),I=33,40),L=1,NZEN)/
     * 10.310,10.422,10.526,10.626,10.734,10.822,10.914,10.978,
     * 10.179,10.283,10.377,10.473,10.575,10.673,10.775,10.856,
     * 10.053,10.152,10.246,10.334,10.431,10.535,10.641,10.733,
     * 9.884,9.971,10.045,10.127,10.215,10.307,10.403,10.507,
     * 9.789,9.841,9.882,9.932,9.985,10.041,10.104,10.167,
     * 9.766,9.818,9.859,9.902,9.947,9.991,10.037,10.079,
     * 9.719,9.776,9.820,9.865,9.910,9.947,9.985,10.017,
     * 9.459,9.550,9.625,9.693,9.755,9.796,9.836,9.864,
     * 9.079,9.176,9.255,9.342,9.423,9.494,9.562,9.571,
     * 9.033,9.127,9.207,9.294,9.373,9.441,9.508,9.513,
     * 8.800,8.885,8.953,9.033,9.104,9.170,9.233,9.241,
     * 8.754,8.840,8.908,8.988,9.058,9.125,9.187,9.195/
      DATA ((EDEN(I,3,11,L,2),I=33,40),L=1,NZEN)/
     * 10.336,10.447,10.551,10.648,10.747,10.831,10.918,10.976,
     * 10.250,10.352,10.447,10.540,10.634,10.723,10.814,10.882,
     * 10.121,10.223,10.314,10.401,10.490,10.585,10.683,10.763,
     * 9.945,10.033,10.114,10.190,10.265,10.354,10.444,10.537,
     * 9.843,9.892,9.930,9.973,10.013,10.068,10.124,10.179,
     * 9.820,9.868,9.905,9.939,9.968,10.009,10.049,10.083,
     * 9.775,9.827,9.867,9.903,9.932,9.966,9.999,10.017,
     * 9.526,9.612,9.681,9.737,9.782,9.818,9.851,9.865,
     * 9.121,9.215,9.294,9.378,9.449,9.515,9.575,9.573,
     * 9.076,9.170,9.246,9.328,9.398,9.461,9.520,9.517,
     * 8.843,8.926,8.994,9.068,9.130,9.190,9.246,9.243,
     * 8.797,8.881,8.948,9.022,9.085,9.145,9.200,9.197/
      DATA ((EDEN(I,4,11,L,2),I=33,40),L=1,NZEN)/
     * 10.344,10.453,10.556,10.646,10.734,10.819,10.905,10.961,
     * 10.340,10.447,10.544,10.631,10.713,10.790,10.866,10.918,
     * 10.215,10.316,10.410,10.493,10.572,10.656,10.740,10.804,
     * 10.025,10.117,10.204,10.274,10.338,10.422,10.504,10.582,
     * 9.893,9.949,9.996,10.029,10.057,10.104,10.149,10.196,
     * 9.872,9.925,9.968,9.991,10.004,10.033,10.057,10.079,
     * 9.831,9.888,9.933,9.955,9.966,9.984,9.998,10.004,
     * 9.607,9.693,9.763,9.803,9.826,9.844,9.856,9.855,
     * 9.155,9.250,9.330,9.407,9.465,9.524,9.576,9.566,
     * 9.111,9.204,9.283,9.356,9.413,9.471,9.522,9.511,
     * 8.880,8.964,9.033,9.100,9.149,9.201,9.248,9.236,
     * 8.834,8.919,8.988,9.055,9.103,9.156,9.202,9.190/
      DATA ((EDEN(I,5,11,L,2),I=33,40),L=1,NZEN)/
     * 10.384,10.490,10.591,10.677,10.754,10.835,10.915,10.967,
     * 10.433,10.539,10.640,10.720,10.790,10.854,10.915,10.953,
     * 10.303,10.407,10.501,10.581,10.651,10.724,10.795,10.844,
     * 10.104,10.199,10.283,10.354,10.412,10.487,10.561,10.626,
     * 9.947,9.997,10.037,10.072,10.090,10.134,10.170,10.217,
     * 9.926,9.968,10.000,10.017,10.021,10.041,10.061,10.076,
     * 9.889,9.934,9.968,9.983,9.980,9.988,9.993,9.986,
     * 9.687,9.757,9.814,9.842,9.849,9.856,9.857,9.843,
     * 9.193,9.286,9.362,9.435,9.481,9.535,9.581,9.561,
     * 9.149,9.238,9.312,9.384,9.431,9.483,9.528,9.507,
     * 8.922,9.000,9.064,9.127,9.164,9.212,9.253,9.230,
     * 8.876,8.954,9.019,9.081,9.119,9.166,9.207,9.185/
      DATA ((EDEN(I,1,12,L,2),I=33,40),L=1,NZEN)/
     * 10.215,10.324,10.425,10.533,10.653,10.757,10.867,10.947,
     * 10.143,10.243,10.332,10.433,10.547,10.650,10.760,10.849,
     * 10.013,10.114,10.201,10.297,10.405,10.513,10.626,10.728,
     * 9.841,9.928,10.004,10.090,10.185,10.286,10.389,10.501,
     * 9.742,9.797,9.839,9.890,9.951,10.017,10.083,10.158,
     * 9.718,9.772,9.815,9.861,9.914,9.963,10.017,10.068,
     * 9.669,9.730,9.776,9.824,9.877,9.920,9.965,10.004,
     * 9.410,9.504,9.581,9.651,9.723,9.769,9.816,9.853,
     * 9.037,9.134,9.212,9.301,9.389,9.465,9.542,9.556,
     * 8.993,9.086,9.161,9.250,9.338,9.413,9.486,9.501,
     * 8.758,8.843,8.908,8.990,9.072,9.143,9.212,9.225,
     * 8.712,8.797,8.863,8.945,9.026,9.097,9.166,9.180/
      DATA ((EDEN(I,2,12,L,2),I=33,40),L=1,NZEN)/
     * 10.322,10.433,10.535,10.633,10.739,10.826,10.918,10.981,
     * 10.190,10.292,10.384,10.477,10.577,10.675,10.777,10.859,
     * 10.064,10.161,10.253,10.340,10.435,10.538,10.643,10.736,
     * 9.895,9.981,10.057,10.137,10.217,10.310,10.407,10.511,
     * 9.800,9.851,9.890,9.937,9.986,10.045,10.104,10.170,
     * 9.776,9.827,9.866,9.906,9.949,9.992,10.037,10.083,
     * 9.728,9.785,9.827,9.870,9.912,9.948,9.987,10.021,
     * 9.470,9.559,9.631,9.696,9.757,9.795,9.834,9.865,
     * 9.090,9.185,9.262,9.346,9.423,9.494,9.563,9.571,
     * 9.041,9.137,9.215,9.297,9.373,9.441,9.508,9.515,
     * 8.810,8.894,8.960,9.037,9.107,9.170,9.233,9.241,
     * 8.764,8.848,8.914,8.992,9.061,9.125,9.187,9.195/
      DATA ((EDEN(I,3,12,L,2),I=33,40),L=1,NZEN)/
     * 10.360,10.471,10.572,10.662,10.754,10.836,10.922,10.980,
     * 10.270,10.371,10.462,10.548,10.636,10.724,10.816,10.885,
     * 10.143,10.241,10.330,10.412,10.494,10.588,10.685,10.766,
     * 9.966,10.053,10.127,10.199,10.270,10.356,10.446,10.540,
     * 9.863,9.909,9.943,9.980,10.013,10.068,10.127,10.182,
     * 9.840,9.886,9.919,9.947,9.972,10.009,10.049,10.083,
     * 9.795,9.846,9.881,9.911,9.934,9.966,9.998,10.017,
     * 9.548,9.630,9.695,9.745,9.785,9.818,9.851,9.866,
     * 9.137,9.230,9.305,9.382,9.447,9.511,9.573,9.572,
     * 9.093,9.185,9.258,9.332,9.394,9.458,9.519,9.517,
     * 8.860,8.942,9.004,9.072,9.130,9.188,9.243,9.243,
     * 8.814,8.896,8.959,9.026,9.085,9.142,9.197,9.197/
      DATA ((EDEN(I,4,12,L,2),I=33,40),L=1,NZEN)/
     * 10.373,10.479,10.575,10.657,10.736,10.817,10.903,10.960,
     * 10.375,10.477,10.569,10.645,10.719,10.792,10.868,10.922,
     * 10.248,10.346,10.435,10.509,10.579,10.659,10.743,10.808,
     * 10.061,10.149,10.228,10.290,10.346,10.427,10.508,10.588,
     * 9.925,9.978,10.017,10.041,10.061,10.104,10.146,10.199,
     * 9.905,9.954,9.990,10.004,10.009,10.029,10.053,10.079,
     * 9.865,9.918,9.956,9.968,9.968,9.982,9.996,10.004,
     * 9.646,9.726,9.789,9.817,9.831,9.844,9.856,9.857,
     * 9.179,9.270,9.344,9.412,9.461,9.516,9.571,9.562,
     * 9.134,9.225,9.297,9.362,9.408,9.462,9.516,9.507,
     * 8.905,8.985,9.045,9.104,9.143,9.193,9.241,9.233,
     * 8.859,8.939,9.000,9.058,9.097,9.147,9.195,9.187/
      DATA ((EDEN(I,5,12,L,2),I=33,40),L=1,NZEN)/
     * 10.423,10.526,10.620,10.692,10.755,10.831,10.910,10.964,
     * 10.476,10.579,10.672,10.739,10.797,10.856,10.918,10.957,
     * 10.350,10.447,10.534,10.601,10.659,10.728,10.799,10.849,
     * 10.155,10.243,10.320,10.375,10.420,10.493,10.568,10.634,
     * 9.994,10.037,10.072,10.090,10.097,10.137,10.173,10.225,
     * 9.974,10.009,10.033,10.037,10.025,10.041,10.061,10.079,
     * 9.939,9.977,10.000,10.000,9.982,9.987,9.992,9.988,
     * 9.743,9.805,9.849,9.862,9.855,9.856,9.855,9.844,
     * 9.228,9.314,9.382,9.441,9.476,9.525,9.573,9.556,
     * 9.188,9.270,9.332,9.389,9.423,9.471,9.517,9.501,
     * 8.959,9.033,9.086,9.134,9.161,9.204,9.243,9.225,
     * 8.913,8.988,9.041,9.088,9.116,9.158,9.197,9.180/
      DATA ((EDEN(I,1,1,L,3),I=33,40),L=1,NZEN)/
     * 10.307,10.415,10.512,10.616,10.732,10.833,10.940,11.017,
     * 10.199,10.294,10.380,10.474,10.581,10.685,10.794,10.888,
     * 10.072,10.164,10.248,10.336,10.438,10.543,10.653,10.758,
     * 9.921,9.995,10.061,10.140,10.230,10.322,10.420,10.531,
     * 9.815,9.861,9.900,9.952,10.017,10.079,10.146,10.223,
     * 9.771,9.816,9.855,9.903,9.964,10.017,10.079,10.140,
     * 9.687,9.734,9.776,9.827,9.890,9.942,9.999,10.057,
     * 9.320,9.394,9.465,9.542,9.632,9.699,9.772,9.839,
     * 8.858,8.921,8.983,9.076,9.185,9.286,9.393,9.439,
     * 8.809,8.870,8.930,9.025,9.130,9.230,9.336,9.382,
     * 8.568,8.622,8.675,8.762,8.862,8.958,9.061,9.107,
     * 8.522,8.576,8.629,8.716,8.816,8.912,9.015,9.061/
      DATA ((EDEN(I,2,1,L,3),I=33,40),L=1,NZEN)/
     * 10.415,10.522,10.620,10.712,10.812,10.896,10.985,11.049,
     * 10.250,10.346,10.431,10.517,10.612,10.708,10.810,10.896,
     * 10.127,10.217,10.299,10.380,10.468,10.566,10.668,10.766,
     * 9.975,10.049,10.114,10.182,10.260,10.346,10.438,10.539,
     * 9.874,9.918,9.954,9.998,10.049,10.104,10.167,10.233,
     * 9.829,9.870,9.905,9.946,9.996,10.041,10.097,10.152,
     * 9.746,9.789,9.825,9.870,9.921,9.968,10.021,10.072,
     * 9.380,9.450,9.516,9.588,9.667,9.727,9.792,9.852,
     * 8.912,8.975,9.033,9.124,9.217,9.312,9.413,9.453,
     * 8.863,8.924,8.982,9.068,9.164,9.258,9.356,9.396,
     * 8.622,8.675,8.725,8.807,8.896,8.986,9.083,9.121,
     * 8.576,8.629,8.679,8.761,8.850,8.940,9.037,9.075/
      DATA ((EDEN(I,3,1,L,3),I=33,40),L=1,NZEN)/
     * 10.459,10.565,10.660,10.744,10.829,10.907,10.990,11.049,
     * 10.326,10.423,10.508,10.585,10.667,10.753,10.845,10.920,
     * 10.204,10.292,10.371,10.444,10.520,10.611,10.706,10.792,
     * 10.045,10.117,10.179,10.238,10.303,10.384,10.470,10.563,
     * 9.938,9.975,10.004,10.037,10.072,10.124,10.182,10.238,
     * 9.893,9.929,9.959,9.986,10.017,10.057,10.107,10.149,
     * 9.813,9.849,9.879,9.909,9.941,9.982,10.029,10.064,
     * 9.462,9.524,9.581,9.635,9.693,9.746,9.804,9.850,
     * 8.963,9.025,9.079,9.158,9.241,9.328,9.422,9.453,
     * 8.914,8.972,9.025,9.104,9.185,9.272,9.365,9.396,
     * 8.677,8.725,8.771,8.843,8.917,9.000,9.090,9.121,
     * 8.631,8.679,8.725,8.797,8.871,8.954,9.044,9.075/
      DATA ((EDEN(I,4,1,L,3),I=33,40),L=1,NZEN)/
     * 10.477,10.581,10.672,10.746,10.817,10.893,10.975,11.033,
     * 10.428,10.522,10.606,10.675,10.741,10.816,10.894,10.956,
     * 10.299,10.389,10.470,10.533,10.594,10.674,10.758,10.831,
     * 10.124,10.201,10.270,10.320,10.367,10.441,10.519,10.601,
     * 9.997,10.041,10.076,10.093,10.111,10.149,10.193,10.243,
     * 9.956,9.997,10.029,10.041,10.049,10.076,10.107,10.137,
     * 9.880,9.919,9.952,9.965,9.975,9.997,10.025,10.049,
     * 9.556,9.617,9.673,9.708,9.740,9.772,9.810,9.840,
     * 9.013,9.072,9.127,9.193,9.258,9.336,9.423,9.446,
     * 8.966,9.021,9.076,9.140,9.201,9.281,9.365,9.387,
     * 8.729,8.777,8.821,8.880,8.937,9.009,9.090,9.114,
     * 8.683,8.731,8.775,8.834,8.891,8.963,9.044,9.068/
      DATA ((EDEN(I,5,1,L,3),I=33,40),L=1,NZEN)/
     * 10.533,10.633,10.722,10.786,10.841,10.911,10.984,11.037,
     * 10.521,10.617,10.701,10.762,10.814,10.876,10.943,10.992,
     * 10.391,10.480,10.558,10.616,10.667,10.736,10.810,10.869,
     * 10.201,10.279,10.344,10.389,10.425,10.494,10.567,10.639,
     * 10.053,10.086,10.114,10.127,10.130,10.164,10.204,10.250,
     * 10.013,10.041,10.061,10.064,10.057,10.076,10.100,10.124,
     * 9.939,9.967,9.987,9.988,9.981,9.993,10.013,10.025,
     * 9.637,9.684,9.725,9.745,9.757,9.777,9.804,9.823,
     * 9.061,9.114,9.164,9.223,9.274,9.346,9.425,9.439,
     * 9.013,9.064,9.114,9.170,9.217,9.290,9.367,9.382,
     * 8.780,8.821,8.859,8.910,8.951,9.021,9.093,9.107,
     * 8.734,8.775,8.813,8.864,8.906,8.975,9.048,9.061/
      DATA ((EDEN(I,1,2,L,3),I=33,40),L=1,NZEN)/
     * 10.322,10.428,10.525,10.626,10.741,10.840,10.944,11.021,
     * 10.210,10.307,10.394,10.490,10.598,10.700,10.807,10.898,
     * 10.086,10.179,10.260,10.350,10.452,10.558,10.667,10.771,
     * 9.924,10.000,10.072,10.149,10.241,10.334,10.433,10.544,
     * 9.815,9.861,9.900,9.952,10.017,10.079,10.149,10.225,
     * 9.772,9.816,9.855,9.903,9.964,10.017,10.076,10.140,
     * 9.688,9.735,9.776,9.827,9.890,9.940,9.997,10.057,
     * 9.326,9.400,9.470,9.547,9.636,9.701,9.772,9.839,
     * 8.855,8.918,8.979,9.072,9.179,9.279,9.386,9.431,
     * 8.806,8.867,8.926,9.017,9.124,9.223,9.328,9.375,
     * 8.566,8.620,8.671,8.758,8.857,8.952,9.053,9.100,
     * 8.520,8.574,8.625,8.712,8.812,8.907,9.007,9.055/
      DATA ((EDEN(I,2,2,L,3),I=33,40),L=1,NZEN)/
     * 10.405,10.513,10.611,10.704,10.806,10.890,10.979,11.041,
     * 10.260,10.358,10.446,10.535,10.631,10.728,10.827,10.911,
     * 10.134,10.225,10.312,10.394,10.487,10.585,10.688,10.782,
     * 9.975,10.053,10.121,10.193,10.274,10.362,10.455,10.556,
     * 9.863,9.909,9.948,9.996,10.049,10.107,10.170,10.238,
     * 9.821,9.863,9.899,9.943,9.994,10.041,10.097,10.152,
     * 9.738,9.782,9.820,9.866,9.919,9.966,10.017,10.068,
     * 9.377,9.449,9.517,9.589,9.667,9.728,9.794,9.853,
     * 8.903,8.966,9.025,9.117,9.212,9.305,9.407,9.444,
     * 8.853,8.915,8.976,9.064,9.158,9.250,9.348,9.387,
     * 8.614,8.667,8.719,8.801,8.890,8.980,9.076,9.111,
     * 8.568,8.622,8.674,8.756,8.845,8.934,9.030,9.065/
      DATA ((EDEN(I,3,2,L,3),I=33,40),L=1,NZEN)/
     * 10.441,10.548,10.644,10.731,10.820,10.900,10.984,11.041,
     * 10.334,10.431,10.519,10.602,10.688,10.773,10.863,10.933,
     * 10.207,10.299,10.382,10.459,10.539,10.630,10.726,10.808,
     * 10.037,10.114,10.182,10.248,10.314,10.400,10.489,10.581,
     * 9.915,9.958,9.993,10.029,10.068,10.124,10.182,10.241,
     * 9.873,9.913,9.947,9.977,10.009,10.053,10.104,10.146,
     * 9.794,9.834,9.867,9.899,9.933,9.976,10.025,10.061,
     * 9.449,9.513,9.575,9.632,9.690,9.744,9.803,9.848,
     * 8.947,9.009,9.068,9.149,9.233,9.320,9.415,9.444,
     * 8.898,8.958,9.017,9.097,9.179,9.265,9.358,9.386,
     * 8.661,8.711,8.760,8.834,8.912,8.995,9.083,9.111,
     * 8.615,8.665,8.714,8.789,8.866,8.949,9.037,9.065/
      DATA ((EDEN(I,4,2,L,3),I=33,40),L=1,NZEN)/
     * 10.462,10.566,10.662,10.740,10.815,10.893,10.975,11.033,
     * 10.425,10.522,10.612,10.687,10.758,10.833,10.910,10.968,
     * 10.292,10.386,10.470,10.543,10.611,10.692,10.777,10.844,
     * 10.107,10.188,10.262,10.322,10.378,10.455,10.534,10.616,
     * 9.963,10.009,10.053,10.079,10.104,10.146,10.193,10.246,
     * 9.921,9.965,10.004,10.021,10.037,10.064,10.100,10.134,
     * 9.844,9.888,9.927,9.946,9.962,9.987,10.017,10.037,
     * 9.526,9.590,9.650,9.691,9.728,9.764,9.803,9.833,
     * 8.986,9.049,9.107,9.179,9.248,9.330,9.415,9.435,
     * 8.939,8.998,9.057,9.127,9.193,9.274,9.358,9.378,
     * 8.701,8.753,8.803,8.867,8.927,9.004,9.083,9.104,
     * 8.655,8.707,8.758,8.821,8.882,8.959,9.037,9.058/
      DATA ((EDEN(I,5,2,L,3),I=33,40),L=1,NZEN)/
     * 10.512,10.615,10.709,10.781,10.844,10.916,10.991,11.041,
     * 10.505,10.605,10.697,10.765,10.825,10.888,10.954,10.999,
     * 10.369,10.464,10.551,10.618,10.677,10.748,10.822,10.878,
     * 10.161,10.248,10.326,10.384,10.431,10.504,10.579,10.650,
     * 9.990,10.033,10.072,10.097,10.117,10.158,10.201,10.250,
     * 9.946,9.985,10.017,10.029,10.033,10.061,10.090,10.117,
     * 9.872,9.910,9.943,9.954,9.958,9.978,10.000,10.017,
     * 9.572,9.627,9.681,9.713,9.736,9.764,9.794,9.816,
     * 9.017,9.079,9.140,9.207,9.265,9.340,9.420,9.431,
     * 8.969,9.029,9.086,9.152,9.210,9.283,9.362,9.375,
     * 8.733,8.782,8.831,8.891,8.943,9.013,9.086,9.100,
     * 8.687,8.737,8.785,8.845,8.897,8.967,9.041,9.055/
      DATA ((EDEN(I,1,3,L,3),I=33,40),L=1,NZEN)/
     * 10.338,10.444,10.542,10.642,10.755,10.849,10.948,11.025,
     * 10.215,10.310,10.396,10.491,10.600,10.702,10.808,10.899,
     * 10.086,10.179,10.262,10.352,10.455,10.560,10.670,10.772,
     * 9.924,10.000,10.072,10.149,10.241,10.334,10.435,10.545,
     * 9.811,9.859,9.900,9.951,10.017,10.079,10.146,10.225,
     * 9.769,9.814,9.851,9.900,9.961,10.013,10.072,10.137,
     * 9.687,9.732,9.772,9.823,9.887,9.938,9.994,10.053,
     * 9.326,9.400,9.470,9.547,9.636,9.701,9.772,9.838,
     * 8.852,8.914,8.975,9.068,9.173,9.274,9.380,9.427,
     * 8.802,8.863,8.922,9.013,9.121,9.217,9.324,9.369,
     * 8.563,8.617,8.667,8.754,8.852,8.947,9.049,9.093,
     * 8.518,8.571,8.622,8.708,8.807,8.902,9.003,9.048/
      DATA ((EDEN(I,2,3,L,3),I=33,40),L=1,NZEN)/
     * 10.387,10.494,10.594,10.691,10.795,10.882,10.972,11.037,
     * 10.260,10.358,10.449,10.543,10.642,10.738,10.836,10.915,
     * 10.130,10.225,10.314,10.401,10.497,10.595,10.699,10.790,
     * 9.965,10.045,10.117,10.196,10.279,10.369,10.462,10.563,
     * 9.848,9.895,9.938,9.989,10.045,10.107,10.170,10.241,
     * 9.804,9.851,9.892,9.939,9.993,10.041,10.093,10.149,
     * 9.721,9.769,9.812,9.862,9.918,9.966,10.017,10.068,
     * 9.365,9.439,9.511,9.587,9.669,9.730,9.796,9.853,
     * 8.889,8.954,9.017,9.107,9.207,9.301,9.401,9.438,
     * 8.839,8.903,8.965,9.057,9.152,9.246,9.344,9.380,
     * 8.600,8.654,8.708,8.793,8.885,8.975,9.072,9.107,
     * 8.554,8.608,8.662,8.747,8.840,8.930,9.026,9.061/
      DATA ((EDEN(I,3,3,L,3),I=33,40),L=1,NZEN)/
     * 10.413,10.521,10.622,10.716,10.812,10.894,10.979,11.037,
     * 10.322,10.423,10.517,10.607,10.699,10.787,10.875,10.943,
     * 10.190,10.288,10.378,10.464,10.551,10.644,10.739,10.818,
     * 10.009,10.093,10.170,10.246,10.322,10.410,10.500,10.591,
     * 9.884,9.930,9.971,10.017,10.061,10.121,10.185,10.243,
     * 9.839,9.884,9.925,9.961,9.999,10.049,10.100,10.143,
     * 9.759,9.805,9.848,9.885,9.924,9.971,10.021,10.053,
     * 9.415,9.484,9.555,9.618,9.683,9.743,9.805,9.845,
     * 8.921,8.987,9.053,9.140,9.228,9.318,9.412,9.438,
     * 8.873,8.937,9.000,9.086,9.173,9.262,9.354,9.380,
     * 8.634,8.688,8.744,8.824,8.906,8.992,9.079,9.107,
     * 8.589,8.643,8.698,8.778,8.860,8.946,9.033,9.061/
      DATA ((EDEN(I,4,3,L,3),I=33,40),L=1,NZEN)/
     * 10.441,10.551,10.655,10.744,10.830,10.910,10.988,11.041,
     * 10.391,10.496,10.594,10.682,10.766,10.843,10.920,10.973,
     * 10.255,10.356,10.450,10.535,10.617,10.702,10.786,10.852,
     * 10.057,10.149,10.236,10.307,10.378,10.461,10.543,10.623,
     * 9.903,9.958,10.009,10.049,10.086,10.140,10.190,10.243,
     * 9.859,9.908,9.954,9.988,10.017,10.057,10.093,10.124,
     * 9.780,9.831,9.879,9.912,9.942,9.975,10.009,10.029,
     * 9.455,9.528,9.601,9.658,9.709,9.753,9.797,9.823,
     * 8.943,9.013,9.083,9.167,9.248,9.332,9.417,9.433,
     * 8.893,8.960,9.029,9.114,9.193,9.274,9.358,9.375,
     * 8.658,8.715,8.775,8.852,8.926,9.004,9.083,9.100,
     * 8.612,8.669,8.729,8.807,8.880,8.959,9.037,9.055/
      DATA ((EDEN(I,5,3,L,3),I=33,40),L=1,NZEN)/
     * 10.487,10.599,10.706,10.793,10.872,10.946,11.017,11.061,
     * 10.462,10.571,10.673,10.756,10.829,10.898,10.963,11.004,
     * 10.322,10.425,10.524,10.607,10.681,10.757,10.831,10.883,
     * 10.100,10.199,10.290,10.365,10.430,10.509,10.587,10.656,
     * 9.910,9.966,10.017,10.064,10.097,10.149,10.199,10.248,
     * 9.864,9.911,9.957,9.989,10.013,10.049,10.079,10.111,
     * 9.787,9.834,9.881,9.914,9.937,9.964,9.989,10.004,
     * 9.481,9.549,9.618,9.670,9.712,9.747,9.782,9.803,
     * 8.957,9.029,9.104,9.188,9.260,9.342,9.420,9.428,
     * 8.908,8.977,9.049,9.134,9.204,9.286,9.362,9.371,
     * 8.673,8.734,8.797,8.872,8.937,9.013,9.086,9.097,
     * 8.627,8.688,8.752,8.826,8.891,8.967,9.041,9.051/
      DATA ((EDEN(I,1,4,L,3),I=33,40),L=1,NZEN)/
     * 10.354,10.462,10.561,10.660,10.772,10.864,10.960,11.029,
     * 10.215,10.310,10.396,10.490,10.598,10.699,10.806,10.895,
     * 10.086,10.179,10.260,10.350,10.452,10.558,10.667,10.769,
     * 9.921,9.999,10.068,10.146,10.238,10.332,10.431,10.542,
     * 9.811,9.857,9.897,9.948,10.013,10.076,10.143,10.220,
     * 9.766,9.810,9.848,9.895,9.956,10.009,10.068,10.134,
     * 9.684,9.729,9.769,9.820,9.884,9.933,9.989,10.049,
     * 9.328,9.400,9.468,9.544,9.632,9.695,9.766,9.833,
     * 8.846,8.908,8.968,9.061,9.167,9.267,9.373,9.420,
     * 8.799,8.859,8.916,9.009,9.114,9.212,9.318,9.364,
     * 8.559,8.611,8.660,8.747,8.846,8.941,9.041,9.090,
     * 8.513,8.565,8.614,8.701,8.800,8.895,8.996,9.044/
      DATA ((EDEN(I,2,4,L,3),I=33,40),L=1,NZEN)/
     * 10.362,10.473,10.576,10.679,10.789,10.880,10.971,11.033,
     * 10.253,10.352,10.447,10.544,10.649,10.744,10.841,10.918,
     * 10.121,10.220,10.310,10.403,10.502,10.603,10.705,10.794,
     * 9.948,10.033,10.111,10.193,10.283,10.375,10.468,10.568,
     * 9.825,9.877,9.925,9.981,10.045,10.107,10.173,10.241,
     * 9.782,9.833,9.879,9.930,9.988,10.041,10.097,10.149,
     * 9.700,9.751,9.798,9.853,9.915,9.965,10.017,10.064,
     * 9.348,9.427,9.502,9.582,9.667,9.730,9.796,9.852,
     * 8.867,8.935,9.004,9.100,9.201,9.297,9.394,9.430,
     * 8.820,8.885,8.951,9.045,9.146,9.241,9.338,9.375,
     * 8.581,8.638,8.698,8.786,8.881,8.971,9.064,9.100,
     * 8.535,8.593,8.652,8.740,8.835,8.926,9.019,9.055/
      DATA ((EDEN(I,3,4,L,3),I=33,40),L=1,NZEN)/
     * 10.387,10.500,10.610,10.713,10.818,10.903,10.985,11.041,
     * 10.297,10.403,10.505,10.606,10.708,10.799,10.886,10.949,
     * 10.161,10.265,10.364,10.461,10.559,10.655,10.750,10.826,
     * 9.976,10.068,10.155,10.241,10.326,10.418,10.509,10.599,
     * 9.839,9.893,9.945,10.000,10.053,10.121,10.185,10.243,
     * 9.796,9.846,9.894,9.941,9.989,10.045,10.100,10.140,
     * 9.714,9.766,9.819,9.866,9.914,9.966,10.017,10.049,
     * 9.369,9.447,9.528,9.601,9.675,9.739,9.801,9.841,
     * 8.888,8.958,9.029,9.130,9.228,9.318,9.410,9.433,
     * 8.839,8.907,8.979,9.076,9.170,9.262,9.354,9.377,
     * 8.599,8.660,8.723,8.814,8.904,8.992,9.079,9.104,
     * 8.553,8.614,8.678,8.768,8.858,8.946,9.033,9.058/
      DATA ((EDEN(I,4,4,L,3),I=33,40),L=1,NZEN)/
     * 10.442,10.561,10.678,10.782,10.881,10.957,11.025,11.064,
     * 10.342,10.455,10.567,10.672,10.772,10.856,10.932,10.981,
     * 10.201,10.312,10.422,10.524,10.621,10.713,10.799,10.860,
     * 9.993,10.093,10.196,10.290,10.378,10.468,10.553,10.630,
     * 9.829,9.891,9.954,10.017,10.076,10.134,10.188,10.243,
     * 9.782,9.841,9.900,9.953,10.000,10.045,10.086,10.117,
     * 9.702,9.761,9.823,9.876,9.922,9.964,10.000,10.021,
     * 9.365,9.450,9.540,9.618,9.690,9.742,9.788,9.816,
     * 8.889,8.968,9.053,9.155,9.250,9.338,9.422,9.433,
     * 8.841,8.918,9.000,9.100,9.193,9.281,9.364,9.377,
     * 8.603,8.671,8.745,8.838,8.927,9.009,9.090,9.100,
     * 8.557,8.625,8.699,8.792,8.881,8.963,9.044,9.055/
      DATA ((EDEN(I,5,4,L,3),I=33,40),L=1,NZEN)/
     * 10.501,10.626,10.754,10.860,10.955,11.021,11.079,11.104,
     * 10.391,10.509,10.629,10.737,10.834,10.911,10.977,11.013,
     * 10.243,10.360,10.477,10.585,10.683,10.769,10.844,10.892,
     * 10.004,10.117,10.236,10.336,10.427,10.516,10.597,10.663,
     * 9.788,9.859,9.933,10.009,10.072,10.137,10.193,10.246,
     * 9.736,9.801,9.870,9.930,9.980,10.029,10.068,10.097,
     * 9.652,9.718,9.789,9.848,9.897,9.940,9.975,9.989,
     * 9.330,9.420,9.517,9.600,9.672,9.724,9.768,9.787,
     * 8.877,8.965,9.061,9.170,9.265,9.352,9.431,9.433,
     * 8.828,8.913,9.009,9.114,9.210,9.297,9.373,9.375,
     * 8.590,8.666,8.751,8.851,8.941,9.021,9.097,9.100,
     * 8.544,8.620,8.705,8.806,8.895,8.975,9.051,9.055/
      DATA ((EDEN(I,1,5,L,3),I=33,40),L=1,NZEN)/
     * 10.371,10.480,10.580,10.679,10.792,10.880,10.973,11.041,
     * 10.217,10.312,10.398,10.493,10.600,10.700,10.806,10.894,
     * 10.090,10.182,10.265,10.352,10.455,10.559,10.667,10.769,
     * 9.924,10.000,10.072,10.149,10.238,10.332,10.433,10.543,
     * 9.807,9.854,9.893,9.945,10.009,10.072,10.140,10.217,
     * 9.765,9.810,9.848,9.894,9.953,10.004,10.064,10.127,
     * 9.683,9.728,9.769,9.818,9.881,9.931,9.987,10.045,
     * 9.334,9.405,9.473,9.545,9.632,9.695,9.766,9.831,
     * 8.843,8.904,8.962,9.057,9.161,9.262,9.367,9.415,
     * 8.792,8.852,8.911,9.004,9.107,9.204,9.310,9.356,
     * 8.555,8.606,8.656,8.742,8.841,8.935,9.037,9.083,
     * 8.509,8.561,8.610,8.696,8.795,8.889,8.992,9.037/
      DATA ((EDEN(I,2,5,L,3),I=33,40),L=1,NZEN)/
     * 10.344,10.459,10.567,10.677,10.792,10.884,10.974,11.037,
     * 10.248,10.350,10.447,10.549,10.658,10.754,10.850,10.924,
     * 10.114,10.212,10.307,10.407,10.511,10.612,10.713,10.800,
     * 9.936,10.025,10.107,10.193,10.288,10.384,10.477,10.575,
     * 9.807,9.863,9.916,9.977,10.045,10.111,10.173,10.241,
     * 9.764,9.816,9.866,9.922,9.985,10.041,10.097,10.149,
     * 9.683,9.736,9.788,9.847,9.912,9.964,10.017,10.064,
     * 9.336,9.415,9.494,9.579,9.668,9.732,9.796,9.851,
     * 8.851,8.921,8.992,9.093,9.199,9.294,9.393,9.427,
     * 8.803,8.870,8.939,9.037,9.143,9.238,9.334,9.369,
     * 8.565,8.623,8.684,8.777,8.876,8.968,9.061,9.097,
     * 8.519,8.577,8.638,8.732,8.830,8.923,9.015,9.051/
      DATA ((EDEN(I,3,5,L,3),I=33,40),L=1,NZEN)/
     * 10.378,10.498,10.617,10.730,10.844,10.927,11.004,11.053,
     * 10.281,10.391,10.500,10.611,10.722,10.813,10.898,10.957,
     * 10.143,10.250,10.356,10.464,10.572,10.670,10.763,10.834,
     * 9.950,10.045,10.143,10.238,10.332,10.431,10.522,10.609,
     * 9.804,9.863,9.924,9.987,10.053,10.124,10.188,10.246,
     * 9.760,9.815,9.872,9.928,9.985,10.045,10.097,10.137,
     * 9.616,9.648,9.686,9.731,9.775,9.815,9.851,9.872,
     * 9.336,9.420,9.507,9.589,9.671,9.739,9.801,9.838,
     * 8.857,8.931,9.013,9.117,9.223,9.318,9.410,9.430,
     * 8.806,8.879,8.959,9.064,9.167,9.262,9.352,9.373,
     * 8.569,8.633,8.704,8.803,8.902,8.992,9.079,9.097,
     * 8.524,8.588,8.658,8.758,8.856,8.946,9.033,9.051/
      DATA ((EDEN(I,4,5,L,3),I=33,40),L=1,NZEN)/
     * 10.477,10.605,10.736,10.848,10.953,11.017,11.072,11.097,
     * 10.310,10.428,10.551,10.671,10.785,10.872,10.948,10.993,
     * 10.164,10.281,10.403,10.521,10.634,10.730,10.814,10.872,
     * 9.946,10.057,10.170,10.281,10.386,10.483,10.569,10.643,
     * 9.769,9.838,9.914,9.993,10.068,10.137,10.193,10.246,
     * 9.720,9.787,9.859,9.928,9.991,10.045,10.086,10.114,
     * 9.638,9.705,9.780,9.849,9.912,9.960,9.998,10.017,
     * 9.301,9.394,9.497,9.592,9.680,9.739,9.787,9.810,
     * 8.844,8.928,9.021,9.140,9.248,9.342,9.425,9.431,
     * 8.793,8.877,8.972,9.086,9.196,9.286,9.367,9.375,
     * 8.555,8.630,8.715,8.824,8.927,9.013,9.090,9.100,
     * 8.509,8.585,8.669,8.778,8.881,8.967,9.044,9.055/
      DATA ((EDEN(I,5,5,L,3),I=33,40),L=1,NZEN)/
     * 10.567,10.705,10.849,10.960,11.057,11.107,11.143,11.143,
     * 10.334,10.464,10.601,10.731,10.849,10.931,10.997,11.025,
     * 10.179,10.307,10.447,10.577,10.695,10.789,10.866,10.907,
     * 9.926,10.057,10.196,10.320,10.435,10.533,10.615,10.679,
     * 9.691,9.776,9.870,9.969,10.057,10.137,10.199,10.250,
     * 9.633,9.712,9.800,9.885,9.959,10.021,10.064,10.093,
     * 9.544,9.625,9.717,9.802,9.876,9.929,9.967,9.979,
     * 9.212,9.318,9.441,9.551,9.648,9.712,9.760,9.777,
     * 8.812,8.913,9.029,9.155,9.272,9.364,9.441,9.436,
     * 8.764,8.861,8.973,9.100,9.217,9.307,9.382,9.378,
     * 8.522,8.612,8.714,8.836,8.945,9.033,9.104,9.100,
     * 8.477,8.566,8.669,8.791,8.900,8.988,9.058,9.055/
      DATA ((EDEN(I,1,6,L,3),I=33,40),L=1,NZEN)/
     * 10.386,10.497,10.599,10.697,10.810,10.897,10.989,11.053,
     * 10.230,10.324,10.410,10.504,10.611,10.710,10.814,10.901,
     * 10.100,10.193,10.274,10.364,10.465,10.569,10.678,10.777,
     * 9.928,10.009,10.079,10.155,10.246,10.340,10.442,10.553,
     * 9.807,9.854,9.893,9.945,10.009,10.072,10.140,10.220,
     * 9.765,9.810,9.848,9.894,9.953,10.004,10.064,10.127,
     * 9.685,9.729,9.769,9.818,9.881,9.930,9.985,10.041,
     * 9.342,9.412,9.477,9.550,9.636,9.697,9.766,9.831,
     * 8.836,8.898,8.958,9.049,9.155,9.255,9.360,9.407,
     * 8.789,8.848,8.905,8.996,9.100,9.199,9.303,9.348,
     * 8.551,8.603,8.652,8.736,8.834,8.928,9.029,9.076,
     * 8.506,8.557,8.606,8.691,8.788,8.882,8.984,9.030/
      DATA ((EDEN(I,2,6,L,3),I=33,40),L=1,NZEN)/
     * 10.342,10.458,10.571,10.683,10.802,10.894,10.985,11.045,
     * 10.255,10.360,10.458,10.562,10.674,10.768,10.861,10.931,
     * 10.124,10.223,10.318,10.420,10.528,10.628,10.728,10.810,
     * 9.936,10.025,10.114,10.204,10.301,10.396,10.491,10.588,
     * 9.799,9.854,9.908,9.975,10.045,10.114,10.179,10.248,
     * 9.755,9.809,9.862,9.921,9.986,10.041,10.100,10.149,
     * 9.676,9.730,9.785,9.846,9.913,9.966,10.017,10.061,
     * 9.338,9.417,9.497,9.582,9.672,9.737,9.801,9.853,
     * 8.838,8.907,8.980,9.083,9.190,9.288,9.386,9.418,
     * 8.789,8.857,8.928,9.029,9.137,9.233,9.328,9.362,
     * 8.553,8.611,8.672,8.769,8.871,8.962,9.053,9.086,
     * 8.507,8.565,8.626,8.723,8.825,8.917,9.007,9.041/
      DATA ((EDEN(I,3,6,L,3),I=33,40),L=1,NZEN)/
     * 10.387,10.511,10.635,10.754,10.873,10.953,11.025,11.068,
     * 10.281,10.394,10.507,10.624,10.741,10.831,10.913,10.969,
     * 10.143,10.253,10.364,10.477,10.592,10.690,10.781,10.848,
     * 9.943,10.041,10.143,10.248,10.350,10.450,10.540,10.624,
     * 9.785,9.847,9.912,9.984,10.057,10.130,10.196,10.253,
     * 9.741,9.798,9.860,9.923,9.986,10.045,10.100,10.137,
     * 9.661,9.719,9.783,9.846,9.909,9.967,10.017,10.045,
     * 9.328,9.412,9.502,9.589,9.675,9.744,9.805,9.839,
     * 8.836,8.912,8.996,9.107,9.220,9.314,9.405,9.422,
     * 8.785,8.860,8.944,9.053,9.161,9.258,9.348,9.365,
     * 8.549,8.614,8.688,8.792,8.896,8.988,9.072,9.090,
     * 8.503,8.568,8.643,8.747,8.850,8.942,9.026,9.044/
      DATA ((EDEN(I,4,6,L,3),I=33,40),L=1,NZEN)/
     * 10.522,10.655,10.795,10.906,11.009,11.064,11.104,11.117,
     * 10.297,10.420,10.551,10.681,10.806,10.893,10.966,11.004,
     * 10.149,10.272,10.401,10.531,10.654,10.751,10.834,10.886,
     * 9.926,10.041,10.167,10.288,10.403,10.504,10.590,10.661,
     * 9.736,9.810,9.893,9.986,10.072,10.146,10.204,10.258,
     * 9.688,9.757,9.837,9.917,9.990,10.045,10.090,10.117,
     * 9.604,9.676,9.757,9.837,9.910,9.961,9.999,10.013,
     * 9.270,9.367,9.479,9.583,9.682,9.743,9.790,9.810,
     * 8.812,8.901,9.004,9.130,9.248,9.342,9.423,9.428,
     * 8.764,8.851,8.952,9.076,9.193,9.286,9.365,9.369,
     * 8.525,8.603,8.695,8.814,8.925,9.013,9.090,9.093,
     * 8.479,8.557,8.649,8.768,8.879,8.967,9.044,9.048/
      DATA ((EDEN(I,5,6,L,3),I=33,40),L=1,NZEN)/
     * 10.643,10.782,10.933,11.037,11.130,11.164,11.176,11.161,
     * 10.305,10.442,10.592,10.737,10.869,10.954,11.017,11.041,
     * 10.149,10.286,10.436,10.583,10.717,10.814,10.889,10.925,
     * 9.886,10.025,10.179,10.322,10.453,10.558,10.640,10.699,
     * 9.640,9.733,9.839,9.959,10.064,10.149,10.210,10.262,
     * 9.580,9.666,9.766,9.866,9.954,10.021,10.072,10.097,
     * 9.489,9.577,9.681,9.782,9.871,9.930,9.968,9.978,
     * 9.152,9.270,9.405,9.531,9.644,9.713,9.760,9.775,
     * 8.777,8.883,9.009,9.149,9.276,9.371,9.446,9.436,
     * 8.726,8.829,8.951,9.093,9.223,9.314,9.387,9.378,
     * 8.483,8.579,8.693,8.829,8.950,9.041,9.111,9.100,
     * 8.437,8.533,8.647,8.783,8.905,8.996,9.065,9.055/
      DATA ((EDEN(I,1,7,L,3),I=33,40),L=1,NZEN)/
     * 10.396,10.507,10.609,10.707,10.818,10.903,10.992,11.057,
     * 10.236,10.332,10.418,10.512,10.620,10.719,10.822,10.907,
     * 10.107,10.199,10.281,10.371,10.474,10.577,10.686,10.784,
     * 9.931,10.013,10.083,10.161,10.255,10.350,10.450,10.560,
     * 9.807,9.854,9.893,9.946,10.013,10.076,10.146,10.225,
     * 9.765,9.810,9.848,9.894,9.953,10.004,10.064,10.130,
     * 9.686,9.730,9.769,9.818,9.881,9.930,9.985,10.041,
     * 9.344,9.413,9.479,9.551,9.636,9.697,9.766,9.831,
     * 8.839,8.900,8.958,9.049,9.152,9.253,9.358,9.403,
     * 8.789,8.848,8.905,8.996,9.100,9.196,9.301,9.346,
     * 8.553,8.603,8.652,8.736,8.834,8.927,9.025,9.072,
     * 8.507,8.557,8.606,8.691,8.788,8.882,8.980,9.026/
      DATA ((EDEN(I,2,7,L,3),I=33,40),L=1,NZEN)/
     * 10.350,10.467,10.579,10.692,10.812,10.904,10.993,11.053,
     * 10.265,10.367,10.467,10.573,10.685,10.778,10.869,10.938,
     * 10.130,10.230,10.328,10.430,10.539,10.639,10.737,10.819,
     * 9.940,10.033,10.121,10.212,10.312,10.407,10.501,10.597,
     * 9.798,9.855,9.911,9.977,10.049,10.117,10.182,10.250,
     * 9.757,9.810,9.860,9.921,9.987,10.041,10.097,10.149,
     * 9.679,9.731,9.783,9.846,9.914,9.966,10.017,10.064,
     * 9.346,9.423,9.502,9.587,9.678,9.740,9.801,9.854,
     * 8.836,8.905,8.977,9.079,9.190,9.286,9.382,9.415,
     * 8.787,8.854,8.924,9.025,9.134,9.230,9.324,9.358,
     * 8.551,8.610,8.671,8.767,8.869,8.960,9.053,9.083,
     * 8.506,8.564,8.625,8.721,8.823,8.915,9.007,9.037/
      DATA ((EDEN(I,3,7,L,3),I=33,40),L=1,NZEN)/
     * 10.396,10.520,10.645,10.765,10.884,10.964,11.033,11.072,
     * 10.292,10.405,10.519,10.636,10.754,10.842,10.921,10.975,
     * 10.155,10.262,10.375,10.490,10.605,10.702,10.790,10.856,
     * 9.949,10.049,10.152,10.258,10.364,10.462,10.553,10.633,
     * 9.787,9.850,9.917,9.989,10.061,10.137,10.204,10.260,
     * 9.744,9.802,9.865,9.928,9.990,10.053,10.104,10.140,
     * 9.666,9.723,9.785,9.849,9.914,9.971,10.021,10.045,
     * 9.340,9.422,9.511,9.598,9.683,9.751,9.810,9.843,
     * 8.832,8.906,8.989,9.104,9.215,9.312,9.400,9.418,
     * 8.781,8.854,8.937,9.049,9.158,9.255,9.342,9.360,
     * 8.545,8.611,8.686,8.790,8.894,8.985,9.068,9.086,
     * 8.500,8.565,8.640,8.745,8.849,8.939,9.022,9.041/
      DATA ((EDEN(I,4,7,L,3),I=33,40),L=1,NZEN)/
     * 10.534,10.667,10.806,10.916,11.021,11.072,11.111,11.117,
     * 10.305,10.430,10.561,10.693,10.818,10.904,10.975,11.009,
     * 10.161,10.283,10.412,10.544,10.669,10.765,10.844,10.894,
     * 9.934,10.053,10.176,10.301,10.418,10.519,10.603,10.671,
     * 9.741,9.817,9.902,9.997,10.083,10.155,10.212,10.265,
     * 9.693,9.763,9.842,9.924,9.999,10.057,10.097,10.124,
     * 9.612,9.682,9.764,9.845,9.920,9.969,10.004,10.017,
     * 9.286,9.380,9.490,9.594,9.692,9.751,9.795,9.814,
     * 8.810,8.897,8.999,9.127,9.246,9.340,9.420,9.423,
     * 8.758,8.844,8.946,9.072,9.190,9.283,9.362,9.365,
     * 8.521,8.599,8.691,8.811,8.924,9.013,9.086,9.090,
     * 8.475,8.553,8.645,8.765,8.878,8.967,9.041,9.044/
      DATA ((EDEN(I,5,7,L,3),I=33,40),L=1,NZEN)/
     * 10.655,10.791,10.940,11.045,11.137,11.167,11.179,11.161,
     * 10.312,10.449,10.600,10.748,10.883,10.967,11.029,11.049,
     * 10.155,10.294,10.444,10.594,10.731,10.826,10.899,10.932,
     * 9.898,10.037,10.193,10.336,10.467,10.572,10.653,10.708,
     * 9.651,9.744,9.851,9.971,10.079,10.161,10.225,10.276,
     * 9.593,9.678,9.777,9.880,9.968,10.037,10.083,10.107,
     * 9.504,9.590,9.694,9.796,9.884,9.940,9.974,9.983,
     * 9.170,9.286,9.420,9.547,9.660,9.726,9.771,9.782,
     * 8.772,8.877,9.004,9.146,9.276,9.371,9.444,9.433,
     * 8.723,8.824,8.946,9.090,9.220,9.314,9.386,9.375,
     * 8.481,8.575,8.689,8.827,8.950,9.041,9.107,9.097,
     * 8.436,8.529,8.644,8.781,8.904,8.996,9.061,9.051/
      DATA ((EDEN(I,1,8,L,3),I=33,40),L=1,NZEN)/
     * 10.391,10.501,10.603,10.701,10.812,10.898,10.989,11.053,
     * 10.230,10.324,10.412,10.507,10.614,10.714,10.820,10.906,
     * 10.100,10.193,10.274,10.365,10.468,10.573,10.682,10.782,
     * 9.928,10.009,10.079,10.158,10.250,10.346,10.447,10.558,
     * 9.807,9.854,9.893,9.946,10.013,10.076,10.146,10.225,
     * 9.766,9.810,9.848,9.895,9.956,10.009,10.068,10.130,
     * 9.685,9.729,9.769,9.818,9.881,9.931,9.987,10.045,
     * 9.336,9.408,9.476,9.549,9.636,9.697,9.766,9.833,
     * 8.843,8.904,8.962,9.053,9.158,9.258,9.365,9.410,
     * 8.792,8.852,8.909,9.000,9.107,9.204,9.307,9.352,
     * 8.555,8.606,8.656,8.741,8.838,8.932,9.033,9.079,
     * 8.509,8.561,8.610,8.695,8.792,8.887,8.988,9.033/
      DATA ((EDEN(I,2,8,L,3),I=33,40),L=1,NZEN)/
     * 10.364,10.477,10.589,10.699,10.817,10.905,10.993,11.053,
     * 10.258,10.360,10.459,10.563,10.675,10.770,10.864,10.935,
     * 10.124,10.223,10.320,10.422,10.529,10.630,10.729,10.813,
     * 9.943,10.029,10.114,10.207,10.303,10.400,10.494,10.590,
     * 9.810,9.864,9.916,9.980,10.053,10.117,10.182,10.248,
     * 9.767,9.817,9.866,9.924,9.989,10.045,10.097,10.149,
     * 9.687,9.738,9.789,9.849,9.916,9.968,10.017,10.064,
     * 9.346,9.423,9.501,9.585,9.677,9.739,9.801,9.854,
     * 8.849,8.916,8.987,9.090,9.196,9.292,9.389,9.422,
     * 8.801,8.865,8.933,9.033,9.140,9.236,9.332,9.364,
     * 8.561,8.619,8.681,8.775,8.876,8.967,9.057,9.090,
     * 8.515,8.573,8.635,8.729,8.830,8.921,9.011,9.044/
      DATA ((EDEN(I,3,8,L,3),I=33,40),L=1,NZEN)/
     * 10.396,10.517,10.638,10.755,10.872,10.952,11.025,11.068,
     * 10.292,10.403,10.513,10.628,10.743,10.832,10.913,10.969,
     * 10.155,10.262,10.371,10.483,10.595,10.692,10.781,10.848,
     * 9.959,10.057,10.155,10.255,10.356,10.453,10.543,10.625,
     * 9.808,9.867,9.928,9.997,10.064,10.137,10.201,10.258,
     * 9.765,9.819,9.876,9.937,9.998,10.057,10.107,10.143,
     * 9.687,9.741,9.801,9.862,9.922,9.976,10.025,10.049,
     * 9.356,9.435,9.520,9.604,9.688,9.753,9.811,9.844,
     * 8.848,8.922,9.004,9.114,9.220,9.316,9.405,9.423,
     * 8.798,8.870,8.950,9.057,9.167,9.260,9.348,9.367,
     * 8.562,8.625,8.697,8.799,8.900,8.990,9.072,9.093,
     * 8.517,8.580,8.651,8.754,8.855,8.944,9.026,9.048/
      DATA ((EDEN(I,4,8,L,3),I=33,40),L=1,NZEN)/
     * 10.497,10.625,10.760,10.873,10.979,11.037,11.083,11.104,
     * 10.318,10.438,10.563,10.688,10.808,10.893,10.964,11.004,
     * 10.176,10.292,10.417,10.542,10.660,10.754,10.834,10.886,
     * 9.961,10.072,10.188,10.303,10.413,10.511,10.592,10.662,
     * 9.780,9.849,9.927,10.013,10.090,10.158,10.212,10.262,
     * 9.733,9.799,9.873,9.945,10.009,10.061,10.100,10.127,
     * 9.654,9.719,9.794,9.868,9.936,9.979,10.009,10.025,
     * 9.328,9.418,9.520,9.616,9.706,9.760,9.803,9.821,
     * 8.833,8.918,9.017,9.137,9.248,9.342,9.422,9.427,
     * 8.787,8.868,8.961,9.083,9.193,9.286,9.364,9.369,
     * 8.548,8.620,8.705,8.820,8.926,9.013,9.090,9.093,
     * 8.502,8.574,8.659,8.774,8.881,8.967,9.044,9.048/
      DATA ((EDEN(I,5,8,L,3),I=33,40),L=1,NZEN)/
     * 10.587,10.722,10.868,10.980,11.079,11.124,11.149,11.146,
     * 10.332,10.464,10.606,10.745,10.872,10.953,11.013,11.037,
     * 10.182,10.312,10.453,10.593,10.720,10.813,10.885,10.921,
     * 9.942,10.068,10.210,10.340,10.461,10.561,10.639,10.696,
     * 9.717,9.799,9.895,10.000,10.093,10.170,10.223,10.270,
     * 9.665,9.740,9.827,9.916,9.993,10.049,10.090,10.114,
     * 9.581,9.656,9.746,9.836,9.913,9.961,9.990,9.996,
     * 9.253,9.354,9.474,9.587,9.687,9.744,9.782,9.794,
     * 8.809,8.905,9.017,9.155,9.279,9.369,9.442,9.435,
     * 8.758,8.854,8.967,9.100,9.223,9.312,9.384,9.377,
     * 8.520,8.605,8.709,8.837,8.952,9.041,9.111,9.100,
     * 8.474,8.560,8.664,8.791,8.906,8.996,9.065,9.055/
      DATA ((EDEN(I,1,9,L,3),I=33,40),L=1,NZEN)/
     * 10.373,10.481,10.582,10.680,10.792,10.880,10.973,11.041,
     * 10.210,10.305,10.393,10.486,10.593,10.695,10.801,10.892,
     * 10.083,10.176,10.258,10.346,10.449,10.554,10.664,10.766,
     * 9.921,9.998,10.064,10.143,10.236,10.328,10.428,10.539,
     * 9.807,9.856,9.897,9.947,10.009,10.072,10.140,10.217,
     * 9.766,9.810,9.848,9.895,9.956,10.009,10.068,10.130,
     * 9.683,9.728,9.769,9.820,9.884,9.933,9.989,10.049,
     * 9.326,9.398,9.467,9.544,9.632,9.695,9.766,9.833,
     * 8.846,8.908,8.969,9.064,9.167,9.270,9.377,9.422,
     * 8.799,8.859,8.918,9.009,9.114,9.212,9.318,9.365,
     * 8.559,8.611,8.660,8.747,8.848,8.943,9.045,9.090,
     * 8.513,8.565,8.614,8.702,8.802,8.897,9.000,9.044/
      DATA ((EDEN(I,2,9,L,3),I=33,40),L=1,NZEN)/
     * 10.375,10.487,10.594,10.700,10.814,10.900,10.986,11.045,
     * 10.241,10.340,10.436,10.537,10.644,10.741,10.838,10.916,
     * 10.111,10.207,10.299,10.396,10.498,10.600,10.701,10.790,
     * 9.944,10.025,10.104,10.188,10.281,10.373,10.465,10.565,
     * 9.826,9.877,9.924,9.983,10.049,10.111,10.173,10.238,
     * 9.783,9.832,9.878,9.932,9.993,10.045,10.097,10.149,
     * 9.701,9.750,9.798,9.854,9.917,9.968,10.017,10.064,
     * 9.348,9.423,9.500,9.582,9.670,9.732,9.795,9.851,
     * 8.866,8.932,9.000,9.100,9.207,9.303,9.401,9.435,
     * 8.815,8.881,8.948,9.045,9.152,9.248,9.344,9.377,
     * 8.577,8.634,8.694,8.786,8.885,8.977,9.068,9.104,
     * 8.532,8.589,8.648,8.740,8.840,8.931,9.022,9.058/
      DATA ((EDEN(I,3,9,L,3),I=33,40),L=1,NZEN)/
     * 10.394,10.511,10.624,10.733,10.844,10.928,11.009,11.053,
     * 10.283,10.389,10.493,10.600,10.708,10.799,10.884,10.946,
     * 10.149,10.253,10.352,10.455,10.559,10.656,10.748,10.822,
     * 9.971,10.061,10.149,10.238,10.328,10.422,10.509,10.595,
     * 9.841,9.894,9.947,10.009,10.068,10.130,10.193,10.246,
     * 9.799,9.848,9.897,9.951,10.004,10.057,10.107,10.143,
     * 9.719,9.768,9.819,9.874,9.931,9.981,10.025,10.057,
     * 9.377,9.450,9.529,9.609,9.688,9.751,9.809,9.846,
     * 8.880,8.949,9.025,9.127,9.233,9.326,9.417,9.438,
     * 8.831,8.899,8.973,9.076,9.179,9.270,9.360,9.380,
     * 8.593,8.653,8.719,8.815,8.911,9.000,9.086,9.107,
     * 8.548,8.607,8.673,8.769,8.865,8.954,9.041,9.061/
      DATA ((EDEN(I,4,9,L,3),I=33,40),L=1,NZEN)/
     * 10.447,10.567,10.688,10.799,10.903,10.977,11.041,11.076,
     * 10.330,10.441,10.556,10.668,10.775,10.860,10.933,10.980,
     * 10.190,10.301,10.413,10.522,10.627,10.719,10.801,10.858,
     * 9.994,10.093,10.196,10.294,10.386,10.476,10.556,10.629,
     * 9.841,9.902,9.967,10.033,10.097,10.152,10.201,10.248,
     * 9.797,9.856,9.919,9.975,10.025,10.072,10.104,10.130,
     * 9.719,9.776,9.839,9.898,9.950,9.989,10.017,10.033,
     * 9.391,9.471,9.561,9.642,9.718,9.766,9.806,9.827,
     * 8.884,8.960,9.045,9.155,9.255,9.346,9.428,9.436,
     * 8.833,8.909,8.994,9.100,9.201,9.290,9.369,9.378,
     * 8.597,8.664,8.740,8.841,8.936,9.021,9.097,9.104,
     * 8.551,8.618,8.694,8.795,8.890,8.975,9.051,9.058/
      DATA ((EDEN(I,5,9,L,3),I=33,40),L=1,NZEN)/
     * 10.497,10.623,10.754,10.868,10.971,11.037,11.086,11.104,
     * 10.369,10.489,10.614,10.733,10.841,10.918,10.980,11.009,
     * 10.225,10.342,10.465,10.583,10.689,10.776,10.848,10.890,
     * 10.009,10.117,10.233,10.342,10.438,10.528,10.602,10.663,
     * 9.825,9.890,9.962,10.041,10.107,10.167,10.215,10.255,
     * 9.779,9.839,9.907,9.972,10.025,10.068,10.097,10.117,
     * 9.701,9.760,9.829,9.893,9.945,9.982,10.004,10.009,
     * 9.384,9.465,9.559,9.646,9.721,9.767,9.800,9.809,
     * 8.880,8.962,9.057,9.173,9.279,9.367,9.441,9.439,
     * 8.830,8.911,9.004,9.121,9.223,9.310,9.384,9.382,
     * 8.593,8.665,8.750,8.859,8.957,9.037,9.107,9.104,
     * 8.548,8.619,8.704,8.813,8.911,8.992,9.061,9.058/
      DATA ((EDEN(I,1,10,L,3),I=33,40),L=1,NZEN)/
     * 10.346,10.455,10.553,10.653,10.766,10.859,10.957,11.029,
     * 10.190,10.286,10.371,10.464,10.569,10.673,10.781,10.876,
     * 10.068,10.155,10.238,10.324,10.425,10.530,10.639,10.745,
     * 9.914,9.988,10.053,10.130,10.220,10.310,10.407,10.517,
     * 9.811,9.857,9.897,9.947,10.009,10.072,10.140,10.212,
     * 9.767,9.812,9.851,9.899,9.959,10.013,10.072,10.134,
     * 9.682,9.728,9.769,9.820,9.884,9.936,9.994,10.053,
     * 9.316,9.391,9.461,9.538,9.627,9.694,9.766,9.834,
     * 8.852,8.914,8.975,9.072,9.179,9.281,9.387,9.435,
     * 8.802,8.864,8.924,9.017,9.124,9.225,9.330,9.378,
     * 8.562,8.616,8.667,8.756,8.857,8.954,9.057,9.104,
     * 8.517,8.570,8.622,8.710,8.812,8.908,9.011,9.058/
      DATA ((EDEN(I,2,10,L,3),I=33,40),L=1,NZEN)/
     * 10.386,10.497,10.601,10.701,10.809,10.895,10.983,11.045,
     * 10.225,10.322,10.413,10.508,10.610,10.708,10.808,10.892,
     * 10.100,10.193,10.281,10.369,10.465,10.566,10.668,10.762,
     * 9.947,10.025,10.093,10.173,10.258,10.346,10.436,10.535,
     * 9.844,9.891,9.933,9.986,10.045,10.104,10.164,10.230,
     * 9.801,9.846,9.887,9.938,9.996,10.045,10.097,10.149,
     * 9.716,9.763,9.808,9.860,9.921,9.969,10.021,10.068,
     * 9.350,9.425,9.497,9.577,9.665,9.727,9.792,9.849,
     * 8.885,8.950,9.017,9.114,9.217,9.314,9.413,9.450,
     * 8.835,8.900,8.965,9.061,9.161,9.258,9.356,9.393,
     * 8.595,8.650,8.707,8.797,8.894,8.987,9.083,9.117,
     * 8.550,8.605,8.661,8.752,8.849,8.941,9.037,9.072/
      DATA ((EDEN(I,3,10,L,3),I=33,40),L=1,NZEN)/
     * 10.405,10.515,10.621,10.722,10.824,10.909,10.993,11.045,
     * 10.279,10.378,10.474,10.571,10.667,10.759,10.849,10.920,
     * 10.152,10.248,10.338,10.428,10.520,10.616,10.709,10.791,
     * 9.991,10.068,10.146,10.223,10.301,10.387,10.473,10.561,
     * 9.881,9.926,9.969,10.017,10.068,10.124,10.182,10.233,
     * 9.838,9.881,9.923,9.967,10.013,10.061,10.107,10.146,
     * 9.756,9.799,9.843,9.890,9.939,9.985,10.029,10.061,
     * 9.401,9.471,9.543,9.615,9.688,9.747,9.806,9.847,
     * 8.915,8.982,9.053,9.146,9.246,9.336,9.428,9.455,
     * 8.868,8.932,9.000,9.093,9.188,9.281,9.371,9.396,
     * 8.627,8.683,8.742,8.831,8.920,9.009,9.097,9.121,
     * 8.582,8.637,8.696,8.785,8.874,8.963,9.051,9.075/
      DATA ((EDEN(I,4,10,L,3),I=33,40),L=1,NZEN)/
     * 10.425,10.537,10.645,10.744,10.839,10.921,10.997,11.045,
     * 10.346,10.450,10.551,10.647,10.740,10.822,10.899,10.954,
     * 10.215,10.314,10.413,10.504,10.591,10.679,10.761,10.828,
     * 10.037,10.124,10.210,10.288,10.362,10.442,10.519,10.595,
     * 9.906,9.961,10.017,10.061,10.100,10.149,10.193,10.236,
     * 9.866,9.916,9.966,10.004,10.041,10.076,10.107,10.134,
     * 9.787,9.838,9.889,9.929,9.965,9.998,10.025,10.041,
     * 9.456,9.529,9.604,9.669,9.728,9.771,9.810,9.833,
     * 8.939,9.009,9.083,9.176,9.265,9.352,9.435,9.449,
     * 8.888,8.956,9.029,9.121,9.210,9.294,9.378,9.393,
     * 8.652,8.712,8.777,8.862,8.943,9.025,9.104,9.117,
     * 8.606,8.666,8.732,8.816,8.897,8.980,9.058,9.072/
      DATA ((EDEN(I,5,10,L,3),I=33,40),L=1,NZEN)/
     * 10.458,10.571,10.683,10.785,10.875,10.953,11.021,11.061,
     * 10.412,10.519,10.625,10.722,10.808,10.881,10.947,10.987,
     * 10.276,10.378,10.481,10.575,10.658,10.739,10.811,10.862,
     * 10.076,10.167,10.262,10.344,10.415,10.494,10.565,10.630,
     * 9.918,9.970,10.025,10.076,10.114,10.161,10.201,10.241,
     * 9.876,9.922,9.971,10.013,10.041,10.076,10.100,10.117,
     * 9.799,9.846,9.895,9.937,9.967,9.994,10.013,10.021,
     * 9.489,9.554,9.626,9.688,9.739,9.776,9.806,9.818,
     * 8.949,9.021,9.100,9.199,9.283,9.365,9.442,9.447,
     * 8.903,8.971,9.045,9.143,9.228,9.310,9.386,9.389,
     * 8.667,8.728,8.796,8.883,8.959,9.037,9.111,9.114,
     * 8.621,8.682,8.750,8.837,8.913,8.992,9.065,9.068/
      DATA ((EDEN(I,1,11,L,3),I=33,40),L=1,NZEN)/
     * 10.320,10.427,10.525,10.627,10.742,10.841,10.944,11.021,
     * 10.179,10.274,10.358,10.450,10.556,10.660,10.769,10.866,
     * 10.057,10.146,10.228,10.314,10.413,10.519,10.627,10.734,
     * 9.911,9.984,10.049,10.124,10.210,10.301,10.398,10.508,
     * 9.811,9.857,9.897,9.948,10.013,10.072,10.140,10.212,
     * 9.767,9.812,9.851,9.900,9.961,10.013,10.072,10.137,
     * 9.682,9.728,9.769,9.821,9.887,9.939,9.997,10.057,
     * 9.312,9.386,9.456,9.535,9.626,9.693,9.766,9.834,
     * 8.855,8.919,8.981,9.076,9.185,9.288,9.396,9.444,
     * 8.806,8.868,8.928,9.021,9.130,9.230,9.338,9.386,
     * 8.565,8.619,8.671,8.760,8.862,8.960,9.064,9.111,
     * 8.519,8.573,8.625,8.715,8.816,8.914,9.019,9.065/
      DATA ((EDEN(I,2,11,L,3),I=33,40),L=1,NZEN)/
     * 10.400,10.508,10.609,10.705,10.808,10.894,10.984,11.049,
     * 10.223,10.316,10.403,10.493,10.589,10.688,10.790,10.877,
     * 10.100,10.190,10.272,10.356,10.447,10.545,10.647,10.744,
     * 9.960,10.029,10.093,10.164,10.246,10.330,10.417,10.519,
     * 9.862,9.905,9.942,9.991,10.045,10.104,10.161,10.225,
     * 9.816,9.859,9.897,9.943,9.996,10.045,10.097,10.149,
     * 9.732,9.775,9.813,9.863,9.921,9.968,10.021,10.068,
     * 9.362,9.433,9.501,9.577,9.660,9.722,9.789,9.848,
     * 8.900,8.965,9.029,9.121,9.223,9.320,9.422,9.459,
     * 8.852,8.915,8.977,9.068,9.167,9.262,9.364,9.401,
     * 8.612,8.667,8.720,8.806,8.899,8.991,9.090,9.127,
     * 8.566,8.621,8.674,8.760,8.853,8.945,9.044,9.081/
      DATA ((EDEN(I,3,11,L,3),I=33,40),L=1,NZEN)/
     * 10.428,10.535,10.636,10.728,10.822,10.903,10.987,11.045,
     * 10.288,10.384,10.471,10.556,10.643,10.735,10.827,10.903,
     * 10.164,10.255,10.338,10.417,10.498,10.591,10.686,10.772,
     * 10.017,10.086,10.152,10.217,10.288,10.369,10.452,10.542,
     * 9.913,9.955,9.991,10.029,10.072,10.124,10.179,10.230,
     * 9.872,9.911,9.945,9.979,10.017,10.061,10.107,10.146,
     * 9.789,9.829,9.866,9.903,9.943,9.985,10.033,10.064,
     * 9.430,9.496,9.559,9.623,9.688,9.744,9.803,9.847,
     * 8.946,9.009,9.072,9.158,9.248,9.338,9.433,9.462,
     * 8.898,8.958,9.017,9.104,9.193,9.283,9.377,9.405,
     * 8.657,8.709,8.761,8.842,8.924,9.013,9.100,9.130,
     * 8.611,8.664,8.715,8.796,8.878,8.967,9.055,9.085/
      DATA ((EDEN(I,4,11,L,3),I=33,40),L=1,NZEN)/
     * 10.439,10.544,10.642,10.729,10.812,10.893,10.975,11.029,
     * 10.377,10.473,10.563,10.643,10.720,10.799,10.879,10.940,
     * 10.250,10.342,10.428,10.502,10.573,10.656,10.739,10.811,
     * 10.083,10.161,10.233,10.292,10.350,10.423,10.498,10.579,
     * 9.963,10.009,10.053,10.083,10.107,10.146,10.188,10.233,
     * 9.923,9.967,10.009,10.029,10.049,10.079,10.107,10.134,
     * 9.846,9.889,9.929,9.955,9.976,10.004,10.029,10.049,
     * 9.516,9.581,9.644,9.692,9.736,9.772,9.811,9.838,
     * 8.986,9.049,9.111,9.193,9.267,9.350,9.435,9.455,
     * 8.937,8.998,9.061,9.137,9.212,9.294,9.378,9.398,
     * 8.699,8.752,8.806,8.879,8.946,9.025,9.104,9.124,
     * 8.653,8.706,8.760,8.833,8.901,8.980,9.058,9.078/
      DATA ((EDEN(I,5,11,L,3),I=33,40),L=1,NZEN)/
     * 10.477,10.581,10.679,10.760,10.833,10.910,10.986,11.037,
     * 10.459,10.558,10.649,10.726,10.793,10.862,10.929,10.974,
     * 10.330,10.422,10.509,10.581,10.644,10.719,10.792,10.849,
     * 10.149,10.225,10.299,10.356,10.407,10.477,10.547,10.616,
     * 10.009,10.049,10.086,10.111,10.124,10.161,10.196,10.236,
     * 9.970,10.004,10.033,10.053,10.061,10.083,10.104,10.121,
     * 9.895,9.928,9.959,9.976,9.984,10.000,10.017,10.029,
     * 9.588,9.638,9.690,9.727,9.754,9.781,9.808,9.822,
     * 9.021,9.083,9.143,9.217,9.283,9.362,9.441,9.452,
     * 8.974,9.029,9.090,9.164,9.228,9.305,9.384,9.393,
     * 8.739,8.788,8.839,8.905,8.961,9.033,9.107,9.117,
     * 8.693,8.742,8.793,8.859,8.915,8.988,9.061,9.072/
      DATA ((EDEN(I,1,12,L,3),I=33,40),L=1,NZEN)/
     * 10.307,10.413,10.511,10.614,10.731,10.831,10.936,11.017,
     * 10.182,10.279,10.364,10.456,10.563,10.667,10.777,10.872,
     * 10.061,10.152,10.233,10.320,10.420,10.525,10.634,10.741,
     * 9.914,9.987,10.053,10.127,10.215,10.307,10.405,10.515,
     * 9.811,9.859,9.900,9.951,10.017,10.076,10.143,10.217,
     * 9.769,9.815,9.855,9.903,9.964,10.017,10.076,10.137,
     * 9.683,9.731,9.772,9.823,9.887,9.940,9.999,10.057,
     * 9.312,9.387,9.458,9.537,9.627,9.695,9.769,9.836,
     * 8.858,8.922,8.985,9.079,9.188,9.290,9.398,9.446,
     * 8.809,8.872,8.932,9.025,9.134,9.233,9.342,9.389,
     * 8.567,8.622,8.675,8.764,8.866,8.963,9.068,9.114,
     * 8.521,8.576,8.629,8.718,8.821,8.917,9.022,9.068/
      DATA ((EDEN(I,2,12,L,3),I=33,40),L=1,NZEN)/
     * 10.410,10.520,10.618,10.711,10.812,10.897,10.986,11.049,
     * 10.236,10.328,10.413,10.498,10.592,10.690,10.792,10.881,
     * 10.111,10.201,10.281,10.362,10.450,10.548,10.650,10.748,
     * 9.970,10.041,10.100,10.170,10.248,10.332,10.420,10.522,
     * 9.873,9.914,9.949,9.995,10.049,10.104,10.161,10.228,
     * 9.827,9.869,9.904,9.946,9.997,10.045,10.097,10.152,
     * 9.743,9.785,9.821,9.867,9.922,9.968,10.021,10.072,
     * 9.371,9.441,9.508,9.581,9.661,9.723,9.790,9.850,
     * 8.910,8.974,9.033,9.124,9.223,9.318,9.420,9.459,
     * 8.862,8.923,8.983,9.072,9.167,9.262,9.364,9.403,
     * 8.621,8.675,8.728,8.810,8.899,8.991,9.090,9.127,
     * 8.575,8.629,8.682,8.764,8.854,8.945,9.044,9.081/
      DATA ((EDEN(I,3,12,L,3),I=33,40),L=1,NZEN)/
     * 10.452,10.559,10.655,10.740,10.828,10.907,10.991,11.049,
     * 10.307,10.401,10.486,10.565,10.647,10.736,10.828,10.905,
     * 10.188,10.274,10.352,10.427,10.502,10.592,10.688,10.775,
     * 10.037,10.104,10.167,10.228,10.290,10.369,10.453,10.545,
     * 9.933,9.971,10.000,10.037,10.072,10.124,10.179,10.233,
     * 9.893,9.928,9.956,9.985,10.017,10.061,10.111,10.149,
     * 9.810,9.846,9.877,9.910,9.945,9.985,10.029,10.068,
     * 9.452,9.515,9.573,9.630,9.690,9.745,9.806,9.849,
     * 8.962,9.021,9.079,9.161,9.246,9.336,9.431,9.462,
     * 8.914,8.973,9.029,9.111,9.193,9.281,9.375,9.405,
     * 8.675,8.725,8.772,8.848,8.925,9.009,9.100,9.130,
     * 8.629,8.679,8.727,8.802,8.879,8.963,9.055,9.085/
      DATA ((EDEN(I,4,12,L,3),I=33,40),L=1,NZEN)/
     * 10.465,10.569,10.663,10.740,10.813,10.891,10.972,11.029,
     * 10.410,10.502,10.587,10.658,10.725,10.801,10.880,10.943,
     * 10.286,10.373,10.450,10.517,10.579,10.659,10.743,10.816,
     * 10.117,10.193,10.258,10.310,10.356,10.427,10.502,10.584,
     * 9.998,10.041,10.076,10.097,10.111,10.149,10.190,10.238,
     * 9.959,9.997,10.029,10.041,10.053,10.079,10.107,10.137,
     * 9.880,9.919,9.952,9.968,9.980,10.004,10.029,10.053,
     * 9.555,9.615,9.671,9.708,9.741,9.774,9.812,9.841,
     * 9.013,9.072,9.127,9.196,9.262,9.342,9.428,9.452,
     * 8.963,9.021,9.076,9.143,9.207,9.288,9.373,9.394,
     * 8.726,8.775,8.822,8.883,8.941,9.017,9.097,9.121,
     * 8.680,8.729,8.776,8.837,8.895,8.971,9.051,9.075/
      DATA ((EDEN(I,5,12,L,3),I=33,40),L=1,NZEN)/
     * 10.517,10.618,10.708,10.775,10.835,10.906,10.981,11.033,
     * 10.505,10.600,10.684,10.747,10.801,10.865,10.931,10.980,
     * 10.378,10.465,10.543,10.602,10.653,10.723,10.797,10.856,
     * 10.199,10.270,10.332,10.378,10.415,10.483,10.553,10.624,
     * 10.057,10.090,10.117,10.127,10.130,10.164,10.199,10.241,
     * 10.021,10.045,10.064,10.068,10.061,10.079,10.100,10.124,
     * 9.945,9.970,9.990,9.992,9.986,10.000,10.017,10.029,
     * 9.644,9.688,9.727,9.748,9.760,9.782,9.807,9.823,
     * 9.064,9.114,9.164,9.225,9.279,9.352,9.431,9.446,
     * 9.013,9.064,9.111,9.170,9.223,9.297,9.375,9.389,
     * 8.779,8.820,8.859,8.912,8.956,9.025,9.100,9.114,
     * 8.733,8.774,8.813,8.866,8.910,8.980,9.055,9.068/
      DATA ((EDEN(I,1,1,L,1),I=41,48),L=1,NZEN)/
     * 10.949,10.997,11.049,11.093,11.137,11.161,11.173,11.176,
     * 10.897,10.942,10.993,11.033,11.072,11.100,11.111,11.117,
     * 10.801,10.859,10.923,10.968,11.013,11.041,11.053,11.061,
     * 10.604,10.687,10.776,10.838,10.899,10.933,10.950,10.959,
     * 10.196,10.294,10.401,10.502,10.604,10.670,10.718,10.742,
     * 10.083,10.137,10.196,10.292,10.386,10.468,10.531,10.574,
     * 10.037,10.061,10.090,10.130,10.167,10.212,10.238,10.279,
     * 9.958,9.981,10.013,10.045,10.079,10.097,10.090,10.079,
     * 9.723,9.713,9.710,9.714,9.723,9.717,9.687,9.646,
     * 9.670,9.660,9.657,9.662,9.669,9.663,9.633,9.593,
     * 9.398,9.387,9.386,9.393,9.401,9.396,9.367,9.328,
     * 9.352,9.342,9.340,9.347,9.356,9.350,9.322,9.283/
      DATA ((EDEN(I,2,1,L,1),I=41,48),L=1,NZEN)/
     * 10.968,11.009,11.053,11.086,11.117,11.140,11.155,11.164,
     * 10.903,10.946,10.993,11.025,11.057,11.083,11.097,11.114,
     * 10.807,10.863,10.922,10.959,10.996,11.021,11.037,11.053,
     * 10.611,10.691,10.775,10.828,10.883,10.914,10.936,10.952,
     * 10.204,10.299,10.400,10.493,10.587,10.650,10.702,10.733,
     * 10.090,10.143,10.196,10.281,10.367,10.447,10.516,10.565,
     * 10.045,10.064,10.090,10.117,10.149,10.193,10.223,10.270,
     * 9.966,9.986,10.009,10.033,10.061,10.076,10.079,10.068,
     * 9.731,9.718,9.710,9.707,9.707,9.699,9.673,9.639,
     * 9.678,9.666,9.658,9.654,9.655,9.646,9.620,9.587,
     * 9.405,9.394,9.387,9.384,9.386,9.378,9.352,9.320,
     * 9.359,9.349,9.342,9.338,9.340,9.333,9.306,9.274/
      DATA ((EDEN(I,3,1,L,1),I=41,48),L=1,NZEN)/
     * 10.967,11.009,11.045,11.072,11.100,11.124,11.143,11.158,
     * 10.921,10.960,10.997,11.017,11.041,11.061,11.079,11.100,
     * 10.829,10.879,10.927,10.954,10.980,11.004,11.021,11.045,
     * 10.638,10.712,10.785,10.825,10.867,10.896,10.920,10.943,
     * 10.220,10.316,10.413,10.496,10.579,10.636,10.688,10.725,
     * 10.086,10.143,10.199,10.279,10.362,10.438,10.508,10.561,
     * 10.037,10.057,10.076,10.104,10.130,10.179,10.217,10.274,
     * 9.956,9.975,9.994,10.017,10.041,10.061,10.072,10.079,
     * 9.727,9.713,9.700,9.688,9.681,9.672,9.651,9.628,
     * 9.673,9.659,9.646,9.635,9.628,9.619,9.599,9.575,
     * 9.400,9.387,9.375,9.365,9.360,9.352,9.332,9.310,
     * 9.354,9.342,9.329,9.320,9.314,9.306,9.287,9.264/
      DATA ((EDEN(I,4,1,L,1),I=41,48),L=1,NZEN)/
     * 10.951,10.997,11.041,11.061,11.086,11.111,11.137,11.164,
     * 10.943,10.975,11.004,11.017,11.029,11.049,11.072,11.100,
     * 10.858,10.899,10.937,10.952,10.968,10.989,11.013,11.041,
     * 10.679,10.742,10.801,10.829,10.857,10.882,10.907,10.938,
     * 10.253,10.348,10.439,10.507,10.574,10.625,10.676,10.718,
     * 10.086,10.146,10.204,10.276,10.350,10.420,10.490,10.548,
     * 10.013,10.029,10.041,10.057,10.072,10.117,10.161,10.225,
     * 9.937,9.948,9.958,9.959,9.963,9.970,9.972,9.975,
     * 9.713,9.697,9.679,9.660,9.644,9.632,9.615,9.599,
     * 9.660,9.644,9.626,9.606,9.591,9.580,9.562,9.544,
     * 9.386,9.373,9.356,9.336,9.322,9.312,9.297,9.279,
     * 9.340,9.327,9.310,9.291,9.276,9.266,9.251,9.233/
      DATA ((EDEN(I,5,1,L,1),I=41,48),L=1,NZEN)/
     * 10.951,10.999,11.037,11.053,11.068,11.093,11.124,11.161,
     * 10.965,10.991,11.009,11.013,11.013,11.029,11.053,11.090,
     * 10.884,10.919,10.946,10.949,10.953,10.969,10.994,11.033,
     * 10.715,10.769,10.816,10.829,10.843,10.863,10.890,10.927,
     * 10.290,10.380,10.464,10.516,10.567,10.611,10.660,10.706,
     * 10.093,10.158,10.217,10.281,10.346,10.410,10.477,10.539,
     * 9.998,10.009,10.013,10.025,10.041,10.090,10.143,10.215,
     * 9.923,9.925,9.922,9.917,9.915,9.924,9.933,9.949,
     * 9.703,9.686,9.663,9.634,9.610,9.598,9.585,9.580,
     * 9.649,9.632,9.610,9.581,9.556,9.545,9.533,9.525,
     * 9.378,9.362,9.338,9.312,9.288,9.276,9.265,9.260,
     * 9.333,9.316,9.293,9.266,9.242,9.231,9.219,9.214/
      DATA ((EDEN(I,1,2,L,1),I=41,48),L=1,NZEN)/
     * 10.947,10.994,11.045,11.090,11.134,11.161,11.173,11.176,
     * 10.903,10.945,10.994,11.033,11.072,11.097,11.107,11.114,
     * 10.810,10.864,10.924,10.967,11.009,11.037,11.049,11.057,
     * 10.619,10.697,10.782,10.841,10.899,10.931,10.946,10.956,
     * 10.207,10.307,10.413,10.512,10.612,10.675,10.719,10.741,
     * 10.083,10.140,10.207,10.301,10.396,10.477,10.538,10.577,
     * 10.033,10.057,10.086,10.127,10.167,10.215,10.243,10.286,
     * 9.955,9.978,10.009,10.041,10.076,10.093,10.090,10.079,
     * 9.714,9.702,9.699,9.704,9.712,9.706,9.676,9.636,
     * 9.661,9.650,9.646,9.651,9.660,9.653,9.623,9.584,
     * 9.387,9.378,9.377,9.382,9.391,9.386,9.358,9.320,
     * 9.342,9.333,9.331,9.336,9.345,9.340,9.312,9.274/
      DATA ((EDEN(I,2,2,L,1),I=41,48),L=1,NZEN)/
     * 10.961,11.004,11.045,11.079,11.114,11.140,11.155,11.164,
     * 10.910,10.950,10.994,11.025,11.057,11.079,11.093,11.107,
     * 10.820,10.871,10.925,10.960,10.995,11.021,11.033,11.049,
     * 10.628,10.704,10.782,10.833,10.883,10.913,10.932,10.947,
     * 10.217,10.314,10.417,10.507,10.597,10.656,10.703,10.732,
     * 10.090,10.146,10.207,10.294,10.382,10.458,10.521,10.567,
     * 10.041,10.061,10.086,10.117,10.152,10.199,10.228,10.274,
     * 9.963,9.984,10.009,10.033,10.061,10.076,10.072,10.068,
     * 9.720,9.708,9.699,9.695,9.695,9.688,9.661,9.628,
     * 9.667,9.654,9.646,9.643,9.642,9.634,9.607,9.574,
     * 9.394,9.382,9.375,9.373,9.373,9.367,9.342,9.310,
     * 9.349,9.336,9.329,9.327,9.327,9.322,9.297,9.264/
      DATA ((EDEN(I,3,2,L,1),I=41,48),L=1,NZEN)/
     * 10.959,11.000,11.041,11.068,11.097,11.117,11.137,11.155,
     * 10.927,10.962,10.997,11.017,11.037,11.061,11.076,11.097,
     * 10.839,10.885,10.930,10.955,10.979,11.000,11.017,11.037,
     * 10.655,10.724,10.792,10.831,10.869,10.895,10.915,10.936,
     * 10.233,10.332,10.431,10.511,10.589,10.642,10.689,10.722,
     * 10.086,10.149,10.212,10.292,10.377,10.449,10.513,10.562,
     * 10.029,10.049,10.072,10.104,10.137,10.182,10.223,10.276,
     * 9.950,9.968,9.989,10.013,10.037,10.057,10.068,10.076,
     * 9.714,9.700,9.688,9.677,9.669,9.660,9.638,9.614,
     * 9.662,9.646,9.633,9.623,9.616,9.607,9.585,9.561,
     * 9.389,9.375,9.362,9.354,9.348,9.340,9.322,9.299,
     * 9.343,9.329,9.316,9.308,9.303,9.295,9.276,9.253/
      DATA ((EDEN(I,4,2,L,1),I=41,48),L=1,NZEN)/
     * 10.948,10.994,11.037,11.061,11.086,11.107,11.130,11.155,
     * 10.948,10.977,11.004,11.013,11.025,11.045,11.064,11.090,
     * 10.865,10.903,10.937,10.952,10.967,10.985,11.004,11.033,
     * 10.691,10.750,10.806,10.831,10.857,10.879,10.901,10.928,
     * 10.267,10.362,10.453,10.519,10.583,10.630,10.676,10.713,
     * 10.083,10.152,10.215,10.290,10.365,10.431,10.496,10.548,
     * 10.004,10.021,10.037,10.057,10.079,10.130,10.176,10.236,
     * 9.928,9.941,9.952,9.957,9.964,9.975,9.979,9.983,
     * 9.702,9.685,9.667,9.648,9.632,9.621,9.603,9.585,
     * 9.649,9.632,9.615,9.595,9.579,9.568,9.549,9.531,
     * 9.375,9.360,9.344,9.324,9.310,9.301,9.286,9.267,
     * 9.329,9.314,9.299,9.279,9.264,9.255,9.240,9.221/
      DATA ((EDEN(I,5,2,L,1),I=41,48),L=1,NZEN)/
     * 10.953,11.000,11.041,11.057,11.072,11.093,11.121,11.152,
     * 10.966,10.991,11.009,11.009,11.013,11.025,11.045,11.079,
     * 10.887,10.920,10.945,10.948,10.951,10.966,10.987,11.021,
     * 10.723,10.774,10.818,10.831,10.843,10.860,10.884,10.917,
     * 10.301,10.391,10.474,10.524,10.575,10.614,10.657,10.699,
     * 10.093,10.164,10.228,10.292,10.358,10.417,10.480,10.537,
     * 9.987,10.000,10.009,10.029,10.049,10.100,10.155,10.223,
     * 9.912,9.915,9.914,9.914,9.916,9.929,9.942,9.958,
     * 9.693,9.675,9.652,9.624,9.600,9.588,9.574,9.566,
     * 9.640,9.621,9.599,9.572,9.548,9.534,9.520,9.511,
     * 9.365,9.348,9.328,9.301,9.279,9.267,9.255,9.248,
     * 9.320,9.303,9.283,9.255,9.233,9.221,9.210,9.202/
      DATA ((EDEN(I,1,3,L,1),I=41,48),L=1,NZEN)/
     * 10.947,10.992,11.041,11.086,11.130,11.155,11.167,11.170,
     * 10.901,10.943,10.992,11.029,11.068,11.093,11.104,11.114,
     * 10.812,10.864,10.923,10.965,11.009,11.033,11.045,11.053,
     * 10.621,10.698,10.782,10.839,10.897,10.929,10.944,10.953,
     * 10.207,10.310,10.417,10.515,10.614,10.674,10.717,10.738,
     * 10.083,10.140,10.207,10.301,10.396,10.476,10.537,10.575,
     * 10.029,10.053,10.083,10.124,10.164,10.212,10.241,10.283,
     * 9.952,9.974,10.004,10.037,10.068,10.090,10.086,10.076,
     * 9.708,9.696,9.694,9.698,9.706,9.700,9.669,9.630,
     * 9.655,9.644,9.640,9.645,9.652,9.646,9.617,9.579,
     * 9.382,9.373,9.369,9.375,9.382,9.378,9.352,9.314,
     * 9.336,9.327,9.323,9.329,9.336,9.333,9.306,9.268/
      DATA ((EDEN(I,2,3,L,1),I=41,48),L=1,NZEN)/
     * 10.952,10.994,11.041,11.076,11.111,11.134,11.149,11.158,
     * 10.911,10.950,10.993,11.021,11.053,11.076,11.090,11.100,
     * 10.822,10.872,10.924,10.959,10.993,11.017,11.029,11.041,
     * 10.635,10.708,10.785,10.834,10.883,10.911,10.928,10.941,
     * 10.220,10.320,10.423,10.512,10.601,10.658,10.702,10.727,
     * 10.090,10.149,10.210,10.297,10.386,10.461,10.522,10.565,
     * 10.033,10.057,10.083,10.117,10.149,10.196,10.228,10.274,
     * 9.959,9.979,10.004,10.029,10.057,10.072,10.072,10.064,
     * 9.712,9.698,9.690,9.688,9.688,9.679,9.651,9.617,
     * 9.659,9.645,9.637,9.634,9.634,9.626,9.599,9.563,
     * 9.387,9.375,9.367,9.365,9.365,9.360,9.334,9.301,
     * 9.342,9.329,9.322,9.320,9.320,9.314,9.289,9.255/
      DATA ((EDEN(I,3,3,L,1),I=41,48),L=1,NZEN)/
     * 10.951,10.993,11.037,11.064,11.090,11.114,11.134,11.146,
     * 10.929,10.962,10.996,11.017,11.037,11.057,11.068,11.086,
     * 10.844,10.886,10.929,10.954,10.977,10.997,11.009,11.025,
     * 10.664,10.728,10.795,10.832,10.869,10.893,10.910,10.926,
     * 10.241,10.338,10.439,10.517,10.595,10.645,10.688,10.714,
     * 10.086,10.149,10.215,10.299,10.384,10.453,10.513,10.558,
     * 10.021,10.041,10.068,10.100,10.137,10.185,10.225,10.276,
     * 9.943,9.962,9.985,10.009,10.033,10.053,10.064,10.072,
     * 9.707,9.691,9.678,9.668,9.661,9.651,9.628,9.601,
     * 9.654,9.638,9.625,9.615,9.607,9.598,9.574,9.547,
     * 9.382,9.365,9.354,9.346,9.340,9.332,9.310,9.286,
     * 9.336,9.320,9.308,9.301,9.295,9.287,9.264,9.240/
      DATA ((EDEN(I,4,3,L,1),I=41,48),L=1,NZEN)/
     * 10.952,10.994,11.033,11.057,11.079,11.100,11.124,11.140,
     * 10.949,10.976,11.000,11.013,11.025,11.037,11.053,11.076,
     * 10.869,10.904,10.937,10.951,10.966,10.981,10.996,11.017,
     * 10.696,10.753,10.808,10.833,10.857,10.876,10.894,10.914,
     * 10.270,10.365,10.461,10.525,10.590,10.631,10.671,10.701,
     * 10.083,10.152,10.223,10.299,10.375,10.438,10.497,10.542,
     * 9.994,10.013,10.033,10.061,10.090,10.137,10.182,10.241,
     * 9.919,9.932,9.946,9.956,9.968,9.983,9.989,9.993,
     * 9.696,9.678,9.659,9.641,9.626,9.613,9.591,9.569,
     * 9.643,9.624,9.606,9.589,9.573,9.561,9.539,9.516,
     * 9.369,9.352,9.336,9.320,9.305,9.294,9.274,9.253,
     * 9.323,9.306,9.291,9.274,9.260,9.249,9.228,9.207/
      DATA ((EDEN(I,5,3,L,1),I=41,48),L=1,NZEN)/
     * 10.966,11.004,11.041,11.057,11.072,11.090,11.111,11.130,
     * 10.966,10.988,11.004,11.009,11.009,11.021,11.033,11.061,
     * 10.889,10.918,10.943,10.947,10.949,10.961,10.977,11.000,
     * 10.725,10.773,10.818,10.831,10.843,10.857,10.874,10.898,
     * 10.303,10.393,10.480,10.530,10.580,10.615,10.652,10.685,
     * 10.090,10.161,10.230,10.299,10.367,10.422,10.479,10.526,
     * 9.975,9.988,9.998,10.025,10.053,10.107,10.161,10.223,
     * 9.899,9.902,9.904,9.911,9.919,9.936,9.951,9.968,
     * 9.686,9.665,9.642,9.617,9.594,9.581,9.563,9.549,
     * 9.633,9.612,9.590,9.565,9.542,9.528,9.509,9.494,
     * 9.360,9.340,9.320,9.297,9.274,9.262,9.246,9.230,
     * 9.314,9.295,9.274,9.251,9.228,9.217,9.200,9.185/
      DATA ((EDEN(I,1,4,L,1),I=41,48),L=1,NZEN)/
     * 10.951,10.993,11.041,11.083,11.124,11.152,11.164,11.164,
     * 10.897,10.939,10.987,11.025,11.064,11.090,11.100,11.111,
     * 10.808,10.860,10.919,10.962,11.004,11.029,11.041,11.049,
     * 10.618,10.695,10.779,10.836,10.894,10.926,10.943,10.950,
     * 10.204,10.305,10.413,10.512,10.611,10.672,10.716,10.737,
     * 10.076,10.137,10.201,10.299,10.394,10.476,10.537,10.575,
     * 10.025,10.049,10.079,10.121,10.161,10.210,10.238,10.281,
     * 9.947,9.969,9.999,10.029,10.064,10.086,10.083,10.072,
     * 9.702,9.692,9.689,9.694,9.702,9.695,9.666,9.626,
     * 9.650,9.639,9.636,9.641,9.649,9.643,9.614,9.574,
     * 9.378,9.367,9.365,9.373,9.380,9.377,9.346,9.310,
     * 9.333,9.322,9.320,9.327,9.334,9.331,9.301,9.264/
      DATA ((EDEN(I,2,4,L,1),I=41,48),L=1,NZEN)/
     * 10.949,10.991,11.037,11.076,11.111,11.130,11.143,11.149,
     * 10.910,10.947,10.989,11.021,11.049,11.072,11.083,11.090,
     * 10.822,10.870,10.922,10.957,10.991,11.013,11.025,11.033,
     * 10.636,10.708,10.784,10.833,10.881,10.909,10.924,10.933,
     * 10.223,10.322,10.425,10.515,10.603,10.658,10.701,10.722,
     * 10.086,10.146,10.212,10.301,10.391,10.464,10.524,10.562,
     * 10.029,10.053,10.079,10.114,10.149,10.196,10.225,10.270,
     * 9.954,9.973,9.999,10.025,10.053,10.068,10.064,10.057,
     * 9.705,9.691,9.683,9.681,9.682,9.673,9.644,9.607,
     * 9.652,9.637,9.630,9.628,9.628,9.619,9.591,9.554,
     * 9.380,9.367,9.362,9.360,9.360,9.352,9.326,9.290,
     * 9.334,9.322,9.316,9.314,9.314,9.306,9.281,9.244/
      DATA ((EDEN(I,3,4,L,1),I=41,48),L=1,NZEN)/
     * 10.951,10.990,11.033,11.061,11.090,11.111,11.121,11.130,
     * 10.929,10.960,10.994,11.017,11.033,11.053,11.061,11.072,
     * 10.844,10.885,10.928,10.953,10.976,10.993,11.004,11.013,
     * 10.667,10.729,10.795,10.833,10.869,10.890,10.904,10.914,
     * 10.243,10.342,10.444,10.521,10.600,10.647,10.685,10.706,
     * 10.083,10.146,10.217,10.303,10.389,10.456,10.515,10.551,
     * 10.009,10.033,10.061,10.097,10.130,10.185,10.223,10.272,
     * 9.936,9.955,9.979,10.004,10.033,10.053,10.061,10.064,
     * 9.701,9.682,9.670,9.662,9.655,9.644,9.618,9.585,
     * 9.648,9.630,9.617,9.609,9.602,9.591,9.565,9.533,
     * 9.375,9.358,9.348,9.340,9.334,9.326,9.301,9.272,
     * 9.329,9.312,9.303,9.295,9.289,9.281,9.255,9.226/
      DATA ((EDEN(I,4,4,L,1),I=41,48),L=1,NZEN)/
     * 10.965,10.998,11.033,11.057,11.076,11.090,11.104,11.114,
     * 10.951,10.975,11.000,11.013,11.021,11.033,11.041,11.053,
     * 10.870,10.903,10.937,10.951,10.964,10.976,10.985,10.996,
     * 10.700,10.753,10.808,10.834,10.858,10.873,10.884,10.895,
     * 10.274,10.369,10.465,10.531,10.594,10.633,10.667,10.687,
     * 10.079,10.149,10.225,10.305,10.386,10.444,10.497,10.533,
     * 9.987,10.004,10.025,10.057,10.093,10.143,10.188,10.238,
     * 9.910,9.922,9.939,9.955,9.973,9.991,9.998,10.004,
     * 9.693,9.670,9.652,9.636,9.622,9.607,9.581,9.553,
     * 9.640,9.618,9.600,9.584,9.569,9.554,9.528,9.498,
     * 9.367,9.346,9.330,9.316,9.301,9.290,9.265,9.238,
     * 9.322,9.301,9.285,9.270,9.255,9.244,9.219,9.192/
      DATA ((EDEN(I,5,4,L,1),I=41,48),L=1,NZEN)/
     * 10.989,11.017,11.045,11.057,11.064,11.076,11.086,11.097,
     * 10.969,10.987,11.004,11.009,11.009,11.013,11.021,11.033,
     * 10.893,10.917,10.943,10.947,10.950,10.956,10.964,10.976,
     * 10.730,10.775,10.820,10.833,10.845,10.854,10.863,10.874,
     * 10.307,10.396,10.486,10.538,10.588,10.617,10.645,10.667,
     * 10.083,10.158,10.236,10.307,10.378,10.428,10.477,10.513,
     * 9.964,9.975,9.989,10.025,10.064,10.114,10.164,10.217,
     * 9.886,9.888,9.892,9.906,9.920,9.942,9.958,9.974,
     * 9.686,9.660,9.636,9.614,9.591,9.574,9.550,9.526,
     * 9.633,9.606,9.583,9.560,9.538,9.521,9.497,9.473,
     * 9.358,9.334,9.314,9.292,9.272,9.258,9.236,9.212,
     * 9.312,9.289,9.268,9.246,9.226,9.212,9.190,9.166/
      DATA ((EDEN(I,1,5,L,1),I=41,48),L=1,NZEN)/
     * 10.962,11.000,11.045,11.086,11.130,11.152,11.161,11.161,
     * 10.895,10.938,10.986,11.025,11.064,11.090,11.100,11.107,
     * 10.805,10.859,10.918,10.960,11.004,11.029,11.041,11.049,
     * 10.618,10.695,10.778,10.836,10.894,10.926,10.942,10.949,
     * 10.204,10.307,10.415,10.513,10.612,10.674,10.718,10.738,
     * 10.072,10.134,10.201,10.299,10.396,10.477,10.540,10.577,
     * 10.021,10.045,10.076,10.117,10.158,10.210,10.241,10.286,
     * 9.943,9.966,9.995,10.025,10.061,10.083,10.079,10.068,
     * 9.697,9.687,9.684,9.689,9.697,9.691,9.662,9.622,
     * 9.643,9.633,9.631,9.636,9.645,9.639,9.610,9.569,
     * 9.371,9.362,9.362,9.367,9.377,9.373,9.342,9.305,
     * 9.325,9.316,9.316,9.322,9.331,9.327,9.297,9.260/
      DATA ((EDEN(I,2,5,L,1),I=41,48),L=1,NZEN)/
     * 10.953,10.993,11.037,11.076,11.111,11.130,11.140,11.143,
     * 10.910,10.946,10.988,11.021,11.049,11.068,11.079,11.086,
     * 10.824,10.869,10.920,10.956,10.990,11.013,11.021,11.029,
     * 10.640,10.710,10.785,10.834,10.883,10.908,10.921,10.928,
     * 10.225,10.324,10.430,10.519,10.607,10.662,10.702,10.719,
     * 10.083,10.146,10.212,10.305,10.396,10.468,10.526,10.562,
     * 10.025,10.049,10.076,10.111,10.146,10.196,10.228,10.272,
     * 9.951,9.971,9.998,10.025,10.053,10.068,10.064,10.049,
     * 9.698,9.684,9.677,9.676,9.677,9.667,9.636,9.597,
     * 9.646,9.630,9.623,9.623,9.624,9.614,9.583,9.544,
     * 9.375,9.360,9.354,9.354,9.354,9.348,9.320,9.281,
     * 9.329,9.314,9.308,9.308,9.308,9.303,9.274,9.235/
      DATA ((EDEN(I,3,5,L,1),I=41,48),L=1,NZEN)/
     * 10.959,10.993,11.033,11.061,11.086,11.104,11.114,11.117,
     * 10.931,10.960,10.993,11.013,11.033,11.049,11.057,11.061,
     * 10.848,10.885,10.928,10.953,10.976,10.992,10.998,11.004,
     * 10.672,10.732,10.797,10.834,10.870,10.889,10.899,10.903,
     * 10.248,10.346,10.450,10.529,10.605,10.650,10.684,10.699,
     * 10.076,10.146,10.220,10.310,10.398,10.464,10.517,10.548,
     * 10.004,10.025,10.057,10.097,10.137,10.188,10.225,10.267,
     * 9.930,9.948,9.974,10.000,10.029,10.049,10.057,10.053,
     * 9.694,9.675,9.663,9.655,9.649,9.636,9.607,9.572,
     * 9.641,9.621,9.610,9.602,9.597,9.584,9.555,9.519,
     * 9.367,9.350,9.338,9.334,9.330,9.320,9.292,9.258,
     * 9.322,9.304,9.293,9.289,9.285,9.274,9.246,9.212/
      DATA ((EDEN(I,4,5,L,1),I=41,48),L=1,NZEN)/
     * 10.980,11.004,11.029,11.049,11.068,11.079,11.090,11.090,
     * 10.953,10.974,10.999,11.013,11.021,11.029,11.033,11.033,
     * 10.874,10.903,10.937,10.952,10.964,10.973,10.977,10.979,
     * 10.707,10.756,10.811,10.837,10.860,10.872,10.876,10.880,
     * 10.279,10.375,10.474,10.539,10.603,10.637,10.665,10.677,
     * 10.079,10.149,10.230,10.314,10.396,10.452,10.500,10.526,
     * 9.979,9.997,10.021,10.061,10.100,10.152,10.196,10.241,
     * 9.902,9.914,9.932,9.954,9.976,9.997,10.004,10.009,
     * 9.688,9.663,9.645,9.630,9.616,9.600,9.571,9.537,
     * 9.634,9.610,9.592,9.577,9.563,9.548,9.519,9.483,
     * 9.362,9.338,9.322,9.307,9.294,9.281,9.255,9.223,
     * 9.316,9.293,9.276,9.262,9.249,9.235,9.210,9.177/
      DATA ((EDEN(I,5,5,L,1),I=41,48),L=1,NZEN)/
     * 11.004,11.017,11.037,11.045,11.053,11.057,11.061,11.061,
     * 10.975,10.989,11.009,11.009,11.009,11.009,11.009,11.009,
     * 10.900,10.920,10.944,10.950,10.951,10.952,10.952,10.953,
     * 10.740,10.780,10.825,10.838,10.848,10.852,10.852,10.854,
     * 10.318,10.405,10.497,10.548,10.597,10.621,10.641,10.651,
     * 10.086,10.161,10.243,10.318,10.391,10.438,10.480,10.504,
     * 9.956,9.966,9.983,10.029,10.072,10.127,10.173,10.220,
     * 9.877,9.876,9.882,9.902,9.922,9.947,9.964,9.979,
     * 9.683,9.652,9.629,9.607,9.585,9.565,9.535,9.504,
     * 9.629,9.599,9.576,9.554,9.531,9.513,9.484,9.452,
     * 9.358,9.328,9.305,9.286,9.265,9.248,9.223,9.193,
     * 9.312,9.283,9.260,9.240,9.219,9.202,9.177,9.147/
      DATA ((EDEN(I,1,6,L,1),I=41,48),L=1,NZEN)/
     * 10.970,11.009,11.049,11.090,11.130,11.152,11.161,11.155,
     * 10.898,10.939,10.986,11.025,11.064,11.090,11.100,11.107,
     * 10.810,10.862,10.919,10.961,11.004,11.029,11.041,11.049,
     * 10.626,10.702,10.782,10.838,10.895,10.926,10.942,10.949,
     * 10.212,10.316,10.423,10.521,10.620,10.679,10.721,10.740,
     * 10.072,10.137,10.210,10.307,10.407,10.486,10.545,10.582,
     * 10.017,10.041,10.072,10.114,10.158,10.210,10.243,10.290,
     * 9.941,9.964,9.995,10.025,10.061,10.079,10.076,10.061,
     * 9.689,9.679,9.677,9.683,9.691,9.685,9.655,9.616,
     * 9.636,9.626,9.623,9.629,9.637,9.632,9.602,9.562,
     * 9.364,9.354,9.354,9.362,9.371,9.365,9.336,9.299,
     * 9.318,9.308,9.308,9.316,9.325,9.320,9.291,9.253/
      DATA ((EDEN(I,2,6,L,1),I=41,48),L=1,NZEN)/
     * 10.960,10.998,11.041,11.076,11.111,11.130,11.140,11.137,
     * 10.912,10.947,10.988,11.017,11.049,11.068,11.076,11.079,
     * 10.827,10.872,10.922,10.957,10.990,11.013,11.017,11.025,
     * 10.650,10.716,10.790,10.837,10.884,10.909,10.920,10.924,
     * 10.236,10.334,10.441,10.529,10.617,10.667,10.704,10.719,
     * 10.086,10.149,10.220,10.314,10.407,10.477,10.533,10.565,
     * 10.025,10.045,10.072,10.111,10.149,10.201,10.236,10.274,
     * 9.948,9.967,9.993,10.021,10.049,10.064,10.057,10.045,
     * 9.690,9.675,9.668,9.667,9.668,9.659,9.627,9.587,
     * 9.636,9.621,9.615,9.615,9.617,9.606,9.574,9.534,
     * 9.365,9.352,9.346,9.346,9.348,9.340,9.312,9.272,
     * 9.320,9.306,9.301,9.301,9.303,9.295,9.266,9.226/
      DATA ((EDEN(I,3,6,L,1),I=41,48),L=1,NZEN)/
     * 10.967,10.996,11.033,11.061,11.086,11.104,11.111,11.107,
     * 10.933,10.960,10.993,11.013,11.033,11.049,11.049,11.049,
     * 10.852,10.887,10.928,10.954,10.978,10.991,10.994,10.995,
     * 10.682,10.738,10.801,10.838,10.872,10.889,10.895,10.896,
     * 10.262,10.358,10.461,10.539,10.614,10.656,10.685,10.695,
     * 10.079,10.152,10.230,10.320,10.410,10.473,10.522,10.548,
     * 10.000,10.021,10.049,10.093,10.137,10.190,10.230,10.270,
     * 9.926,9.943,9.969,9.999,10.029,10.049,10.053,10.049,
     * 9.685,9.665,9.652,9.646,9.641,9.628,9.597,9.559,
     * 9.632,9.612,9.600,9.594,9.589,9.576,9.544,9.505,
     * 9.360,9.340,9.332,9.326,9.322,9.312,9.281,9.246,
     * 9.314,9.295,9.287,9.281,9.276,9.266,9.235,9.200/
      DATA ((EDEN(I,4,6,L,1),I=41,48),L=1,NZEN)/
     * 10.985,11.000,11.025,11.045,11.061,11.068,11.072,11.068,
     * 10.957,10.974,10.999,11.013,11.021,11.029,11.025,11.021,
     * 10.881,10.905,10.937,10.953,10.965,10.971,10.970,10.968,
     * 10.718,10.763,10.815,10.840,10.862,10.871,10.872,10.869,
     * 10.294,10.387,10.487,10.550,10.612,10.642,10.664,10.669,
     * 10.079,10.158,10.243,10.328,10.410,10.462,10.504,10.524,
     * 9.975,9.993,10.021,10.064,10.107,10.161,10.204,10.243,
     * 9.898,9.908,9.928,9.952,9.976,9.997,10.004,10.004,
     * 9.680,9.653,9.635,9.622,9.610,9.591,9.559,9.520,
     * 9.627,9.600,9.583,9.569,9.556,9.539,9.507,9.468,
     * 9.354,9.328,9.314,9.301,9.290,9.274,9.243,9.207,
     * 9.308,9.283,9.268,9.255,9.244,9.228,9.197,9.161/
      DATA ((EDEN(I,5,6,L,1),I=41,48),L=1,NZEN)/
     * 11.004,11.009,11.021,11.033,11.037,11.041,11.037,11.033,
     * 10.981,10.990,11.009,11.013,11.009,11.009,11.000,10.993,
     * 10.907,10.923,10.948,10.953,10.953,10.951,10.944,10.938,
     * 10.752,10.787,10.830,10.843,10.851,10.851,10.846,10.839,
     * 10.336,10.420,10.513,10.562,10.609,10.627,10.640,10.641,
     * 10.097,10.173,10.262,10.336,10.408,10.449,10.483,10.498,
     * 9.952,9.962,9.981,10.029,10.079,10.134,10.182,10.220,
     * 9.872,9.868,9.875,9.899,9.922,9.947,9.962,9.975,
     * 9.678,9.643,9.620,9.599,9.576,9.555,9.521,9.484,
     * 9.625,9.591,9.567,9.547,9.525,9.502,9.468,9.433,
     * 9.350,9.318,9.297,9.276,9.258,9.238,9.207,9.173,
     * 9.304,9.272,9.251,9.231,9.212,9.192,9.161,9.127/
      DATA ((EDEN(I,1,7,L,1),I=41,48),L=1,NZEN)/
     * 10.973,11.009,11.049,11.090,11.130,11.152,11.161,11.155,
     * 10.902,10.942,10.988,11.025,11.064,11.086,11.097,11.107,
     * 10.816,10.866,10.922,10.963,11.004,11.029,11.041,11.049,
     * 10.634,10.708,10.787,10.841,10.897,10.927,10.942,10.949,
     * 10.220,10.324,10.435,10.529,10.625,10.683,10.723,10.741,
     * 10.076,10.143,10.215,10.314,10.415,10.493,10.550,10.587,
     * 10.017,10.041,10.072,10.117,10.161,10.215,10.250,10.294,
     * 9.940,9.964,9.995,10.025,10.061,10.079,10.076,10.061,
     * 9.686,9.676,9.673,9.679,9.686,9.680,9.650,9.612,
     * 9.632,9.622,9.620,9.625,9.633,9.627,9.598,9.559,
     * 9.360,9.350,9.350,9.356,9.365,9.362,9.334,9.297,
     * 9.314,9.304,9.304,9.310,9.320,9.316,9.289,9.251/
      DATA ((EDEN(I,2,7,L,1),I=41,48),L=1,NZEN)/
     * 10.963,11.000,11.045,11.079,11.114,11.134,11.140,11.137,
     * 10.916,10.949,10.989,11.017,11.049,11.068,11.076,11.079,
     * 10.834,10.875,10.923,10.957,10.990,11.013,11.017,11.021,
     * 10.657,10.722,10.793,10.839,10.885,10.909,10.920,10.923,
     * 10.246,10.344,10.450,10.537,10.621,10.671,10.706,10.719,
     * 10.086,10.152,10.225,10.320,10.415,10.484,10.538,10.567,
     * 10.021,10.041,10.072,10.111,10.152,10.204,10.238,10.279,
     * 9.947,9.966,9.993,10.021,10.049,10.061,10.057,10.041,
     * 9.685,9.669,9.663,9.663,9.664,9.653,9.622,9.582,
     * 9.632,9.617,9.610,9.610,9.612,9.601,9.569,9.529,
     * 9.360,9.346,9.340,9.342,9.344,9.336,9.305,9.265,
     * 9.314,9.301,9.295,9.297,9.299,9.291,9.260,9.219/
      DATA ((EDEN(I,3,7,L,1),I=41,48),L=1,NZEN)/
     * 10.968,10.997,11.033,11.061,11.086,11.104,11.107,11.104,
     * 10.936,10.960,10.992,11.013,11.033,11.045,11.049,11.049,
     * 10.857,10.889,10.928,10.953,10.976,10.990,10.993,10.993,
     * 10.688,10.743,10.803,10.839,10.873,10.889,10.894,10.894,
     * 10.272,10.367,10.470,10.545,10.619,10.659,10.687,10.695,
     * 10.083,10.155,10.236,10.328,10.417,10.477,10.525,10.548,
     * 10.000,10.021,10.049,10.093,10.140,10.193,10.233,10.272,
     * 9.925,9.942,9.968,9.997,10.025,10.045,10.053,10.045,
     * 9.679,9.658,9.646,9.640,9.635,9.622,9.591,9.551,
     * 9.625,9.604,9.593,9.588,9.584,9.571,9.539,9.498,
     * 9.354,9.334,9.326,9.322,9.318,9.307,9.276,9.238,
     * 9.308,9.289,9.281,9.276,9.272,9.262,9.231,9.192/
      DATA ((EDEN(I,4,7,L,1),I=41,48),L=1,NZEN)/
     * 10.984,10.999,11.021,11.041,11.061,11.068,11.072,11.068,
     * 10.959,10.974,10.998,11.013,11.021,11.029,11.025,11.021,
     * 10.884,10.907,10.938,10.953,10.965,10.971,10.969,10.965,
     * 10.724,10.766,10.817,10.841,10.862,10.870,10.870,10.866,
     * 10.307,10.396,10.494,10.556,10.616,10.644,10.664,10.667,
     * 10.086,10.164,10.253,10.334,10.415,10.465,10.505,10.522,
     * 9.974,9.992,10.021,10.064,10.111,10.164,10.204,10.241,
     * 9.897,9.907,9.926,9.950,9.974,9.992,9.995,9.993,
     * 9.674,9.646,9.629,9.616,9.603,9.585,9.553,9.512,
     * 9.621,9.593,9.576,9.563,9.551,9.533,9.500,9.459,
     * 9.348,9.322,9.305,9.294,9.286,9.270,9.238,9.199,
     * 9.303,9.276,9.260,9.249,9.240,9.224,9.192,9.153/
      DATA ((EDEN(I,5,7,L,1),I=41,48),L=1,NZEN)/
     * 10.999,11.004,11.017,11.029,11.033,11.037,11.037,11.029,
     * 10.981,10.990,11.009,11.013,11.009,11.009,11.000,10.990,
     * 10.909,10.924,10.947,10.953,10.953,10.950,10.942,10.934,
     * 10.757,10.790,10.831,10.843,10.851,10.850,10.844,10.836,
     * 10.348,10.430,10.520,10.568,10.612,10.629,10.639,10.637,
     * 10.100,10.182,10.270,10.344,10.415,10.453,10.484,10.496,
     * 9.952,9.963,9.984,10.037,10.086,10.140,10.185,10.217,
     * 9.871,9.867,9.875,9.898,9.920,9.944,9.956,9.964,
     * 9.672,9.637,9.614,9.593,9.572,9.549,9.513,9.476,
     * 9.619,9.584,9.561,9.540,9.519,9.497,9.462,9.423,
     * 9.344,9.312,9.292,9.272,9.253,9.233,9.201,9.164,
     * 9.299,9.266,9.246,9.226,9.207,9.187,9.156,9.119/
      DATA ((EDEN(I,1,8,L,1),I=41,48),L=1,NZEN)/
     * 10.969,11.009,11.049,11.090,11.130,11.152,11.161,11.158,
     * 10.902,10.943,10.989,11.025,11.064,11.090,11.100,11.107,
     * 10.816,10.866,10.922,10.964,11.004,11.033,11.041,11.049,
     * 10.632,10.707,10.787,10.842,10.897,10.928,10.943,10.949,
     * 10.217,10.320,10.431,10.526,10.622,10.681,10.722,10.741,
     * 10.076,10.143,10.215,10.312,10.412,10.489,10.548,10.583,
     * 10.017,10.045,10.076,10.117,10.161,10.215,10.248,10.292,
     * 9.943,9.965,9.995,10.025,10.061,10.083,10.079,10.068,
     * 9.691,9.680,9.677,9.682,9.690,9.684,9.654,9.616,
     * 9.638,9.627,9.624,9.629,9.637,9.631,9.601,9.562,
     * 9.365,9.356,9.354,9.360,9.369,9.365,9.336,9.299,
     * 9.320,9.310,9.308,9.314,9.323,9.320,9.291,9.253/
      DATA ((EDEN(I,2,8,L,1),I=41,48),L=1,NZEN)/
     * 10.963,11.000,11.041,11.079,11.114,11.134,11.140,11.137,
     * 10.915,10.949,10.990,11.021,11.049,11.068,11.076,11.083,
     * 10.831,10.875,10.925,10.959,10.992,11.013,11.021,11.025,
     * 10.654,10.719,10.792,10.839,10.885,10.909,10.920,10.925,
     * 10.241,10.338,10.444,10.531,10.618,10.668,10.703,10.719,
     * 10.086,10.152,10.223,10.314,10.407,10.479,10.534,10.566,
     * 10.025,10.045,10.076,10.111,10.149,10.201,10.236,10.276,
     * 9.950,9.968,9.993,10.021,10.049,10.064,10.057,10.045,
     * 9.692,9.676,9.668,9.667,9.668,9.659,9.627,9.588,
     * 9.638,9.623,9.616,9.615,9.616,9.605,9.575,9.534,
     * 9.367,9.352,9.346,9.346,9.348,9.340,9.312,9.272,
     * 9.322,9.306,9.301,9.301,9.303,9.295,9.266,9.226/
      DATA ((EDEN(I,3,8,L,1),I=41,48),L=1,NZEN)/
     * 10.966,10.996,11.033,11.061,11.086,11.104,11.111,11.111,
     * 10.935,10.960,10.993,11.013,11.033,11.049,11.053,11.053,
     * 10.853,10.888,10.929,10.954,10.976,10.991,10.995,10.997,
     * 10.684,10.739,10.801,10.838,10.872,10.889,10.896,10.898,
     * 10.265,10.360,10.464,10.539,10.614,10.655,10.685,10.695,
     * 10.083,10.155,10.233,10.322,10.410,10.471,10.521,10.547,
     * 10.004,10.025,10.053,10.097,10.140,10.193,10.230,10.270,
     * 9.930,9.946,9.971,10.000,10.029,10.049,10.049,10.045,
     * 9.685,9.664,9.652,9.646,9.640,9.628,9.598,9.560,
     * 9.631,9.611,9.599,9.593,9.588,9.575,9.544,9.507,
     * 9.360,9.340,9.330,9.326,9.322,9.312,9.283,9.246,
     * 9.314,9.295,9.285,9.281,9.276,9.266,9.238,9.200/
      DATA ((EDEN(I,4,8,L,1),I=41,48),L=1,NZEN)/
     * 10.980,11.000,11.025,11.049,11.064,11.079,11.086,11.083,
     * 10.956,10.974,10.998,11.013,11.021,11.029,11.029,11.029,
     * 10.880,10.905,10.936,10.952,10.964,10.972,10.974,10.972,
     * 10.717,10.761,10.814,10.839,10.862,10.871,10.873,10.873,
     * 10.299,10.389,10.486,10.549,10.611,10.641,10.664,10.670,
     * 10.086,10.161,10.243,10.326,10.407,10.459,10.501,10.522,
     * 9.978,9.995,10.021,10.061,10.104,10.155,10.196,10.233,
     * 9.902,9.911,9.930,9.950,9.970,9.986,9.989,9.982,
     * 9.677,9.650,9.634,9.621,9.609,9.590,9.558,9.519,
     * 9.624,9.598,9.581,9.568,9.554,9.537,9.505,9.465,
     * 9.352,9.326,9.312,9.299,9.290,9.274,9.243,9.204,
     * 9.306,9.281,9.266,9.253,9.244,9.228,9.197,9.158/
      DATA ((EDEN(I,5,8,L,1),I=41,48),L=1,NZEN)/
     * 10.999,11.009,11.025,11.041,11.049,11.053,11.057,11.053,
     * 10.977,10.987,11.004,11.009,11.009,11.009,11.004,11.000,
     * 10.903,10.920,10.944,10.951,10.953,10.953,10.949,10.944,
     * 10.748,10.783,10.826,10.839,10.849,10.851,10.849,10.844,
     * 10.336,10.418,10.508,10.558,10.604,10.625,10.639,10.642,
     * 10.100,10.176,10.260,10.332,10.403,10.444,10.480,10.496,
     * 9.959,9.968,9.986,10.033,10.076,10.127,10.173,10.210,
     * 9.880,9.876,9.882,9.900,9.918,9.936,9.944,9.949,
     * 9.674,9.640,9.618,9.598,9.576,9.555,9.521,9.484,
     * 9.621,9.588,9.566,9.545,9.524,9.502,9.468,9.431,
     * 9.346,9.316,9.294,9.276,9.258,9.238,9.207,9.173,
     * 9.301,9.270,9.249,9.231,9.212,9.192,9.161,9.127/
      DATA ((EDEN(I,1,9,L,1),I=41,48),L=1,NZEN)/
     * 10.960,10.999,11.045,11.086,11.124,11.149,11.161,11.161,
     * 10.897,10.940,10.988,11.025,11.064,11.090,11.100,11.111,
     * 10.806,10.860,10.919,10.963,11.004,11.033,11.045,11.053,
     * 10.615,10.694,10.778,10.836,10.895,10.927,10.943,10.950,
     * 10.201,10.303,10.412,10.511,10.611,10.673,10.717,10.738,
     * 10.076,10.134,10.199,10.297,10.394,10.476,10.538,10.576,
     * 10.025,10.049,10.079,10.121,10.161,10.212,10.241,10.283,
     * 9.947,9.969,9.999,10.029,10.064,10.086,10.083,10.072,
     * 9.704,9.694,9.691,9.695,9.704,9.697,9.667,9.628,
     * 9.651,9.640,9.638,9.643,9.651,9.645,9.615,9.575,
     * 9.378,9.369,9.367,9.375,9.382,9.378,9.348,9.312,
     * 9.333,9.323,9.322,9.329,9.336,9.333,9.303,9.266/
      DATA ((EDEN(I,2,9,L,1),I=41,48),L=1,NZEN)/
     * 10.960,10.998,11.041,11.076,11.111,11.130,11.140,11.143,
     * 10.908,10.945,10.987,11.021,11.049,11.072,11.083,11.090,
     * 10.820,10.867,10.920,10.956,10.990,11.013,11.021,11.033,
     * 10.631,10.704,10.782,10.832,10.882,10.908,10.923,10.931,
     * 10.217,10.316,10.420,10.511,10.601,10.657,10.699,10.719,
     * 10.083,10.140,10.204,10.297,10.389,10.462,10.522,10.559,
     * 10.029,10.053,10.079,10.114,10.149,10.196,10.225,10.267,
     * 9.955,9.973,9.998,10.025,10.053,10.068,10.064,10.049,
     * 9.708,9.692,9.685,9.684,9.685,9.675,9.644,9.605,
     * 9.654,9.639,9.632,9.630,9.631,9.621,9.591,9.553,
     * 9.380,9.367,9.362,9.362,9.364,9.356,9.326,9.288,
     * 9.334,9.322,9.316,9.316,9.318,9.310,9.281,9.242/
      DATA ((EDEN(I,3,9,L,1),I=41,48),L=1,NZEN)/
     * 10.960,10.995,11.033,11.064,11.090,11.107,11.117,11.121,
     * 10.925,10.956,10.991,11.013,11.033,11.049,11.057,11.064,
     * 10.841,10.881,10.925,10.951,10.976,10.993,11.000,11.009,
     * 10.661,10.723,10.791,10.831,10.869,10.890,10.901,10.908,
     * 10.238,10.334,10.438,10.517,10.597,10.644,10.682,10.700,
     * 10.079,10.143,10.212,10.299,10.386,10.455,10.511,10.544,
     * 10.013,10.033,10.061,10.097,10.134,10.182,10.217,10.262,
     * 9.938,9.955,9.978,10.004,10.033,10.053,10.057,10.053,
     * 9.701,9.682,9.670,9.663,9.657,9.645,9.616,9.580,
     * 9.648,9.629,9.618,9.611,9.604,9.592,9.565,9.528,
     * 9.375,9.358,9.346,9.340,9.336,9.326,9.299,9.265,
     * 9.329,9.312,9.301,9.295,9.291,9.281,9.253,9.219/
      DATA ((EDEN(I,4,9,L,1),I=41,48),L=1,NZEN)/
     * 10.968,10.997,11.029,11.053,11.072,11.086,11.100,11.104,
     * 10.945,10.968,10.996,11.013,11.021,11.033,11.041,11.045,
     * 10.866,10.897,10.932,10.949,10.963,10.975,10.982,10.989,
     * 10.694,10.746,10.803,10.831,10.858,10.873,10.881,10.887,
     * 10.267,10.360,10.458,10.526,10.592,10.631,10.663,10.679,
     * 10.079,10.143,10.217,10.299,10.380,10.439,10.490,10.521,
     * 9.989,10.004,10.021,10.057,10.090,10.137,10.176,10.220,
     * 9.913,9.922,9.938,9.952,9.967,9.977,9.976,9.968,
     * 9.691,9.667,9.651,9.636,9.623,9.606,9.576,9.542,
     * 9.637,9.615,9.598,9.584,9.572,9.554,9.524,9.489,
     * 9.365,9.342,9.326,9.314,9.303,9.288,9.260,9.225,
     * 9.320,9.297,9.281,9.268,9.257,9.242,9.214,9.180/
      DATA ((EDEN(I,5,9,L,1),I=41,48),L=1,NZEN)/
     * 10.985,11.009,11.037,11.053,11.061,11.072,11.083,11.086,
     * 10.963,10.980,11.000,11.004,11.009,11.013,11.017,11.021,
     * 10.886,10.910,10.938,10.945,10.949,10.955,10.959,10.965,
     * 10.723,10.766,10.813,10.830,10.844,10.852,10.859,10.864,
     * 10.299,10.386,10.476,10.531,10.585,10.614,10.640,10.654,
     * 10.083,10.152,10.228,10.301,10.373,10.423,10.470,10.497,
     * 9.970,9.977,9.989,10.021,10.057,10.104,10.149,10.193,
     * 9.894,9.891,9.894,9.906,9.917,9.929,9.934,9.936,
     * 9.685,9.656,9.634,9.614,9.593,9.574,9.544,9.512,
     * 9.631,9.603,9.581,9.561,9.540,9.520,9.491,9.458,
     * 9.358,9.330,9.310,9.292,9.272,9.255,9.228,9.196,
     * 9.312,9.285,9.264,9.246,9.226,9.210,9.182,9.150/
      DATA ((EDEN(I,1,10,L,1),I=41,48),L=1,NZEN)/
     * 10.953,10.995,11.045,11.086,11.130,11.155,11.164,11.167,
     * 10.886,10.933,10.986,11.025,11.068,11.093,11.104,11.114,
     * 10.792,10.850,10.914,10.961,11.009,11.037,11.049,11.057,
     * 10.592,10.677,10.766,10.829,10.893,10.928,10.946,10.955,
     * 10.185,10.283,10.387,10.491,10.597,10.663,10.712,10.736,
     * 10.076,10.130,10.188,10.281,10.375,10.459,10.525,10.568,
     * 10.033,10.057,10.083,10.121,10.161,10.207,10.233,10.272,
     * 9.953,9.976,10.009,10.041,10.076,10.093,10.086,10.076,
     * 9.719,9.709,9.708,9.713,9.722,9.715,9.685,9.644,
     * 9.667,9.657,9.655,9.660,9.668,9.662,9.632,9.591,
     * 9.394,9.386,9.384,9.391,9.400,9.394,9.365,9.326,
     * 9.349,9.340,9.338,9.345,9.354,9.349,9.320,9.281/
      DATA ((EDEN(I,2,10,L,1),I=41,48),L=1,NZEN)/
     * 10.960,10.999,11.041,11.076,11.111,11.134,11.146,11.152,
     * 10.897,10.939,10.985,11.021,11.053,11.076,11.090,11.100,
     * 10.801,10.857,10.916,10.954,10.993,11.017,11.029,11.041,
     * 10.604,10.684,10.768,10.823,10.879,10.909,10.927,10.939,
     * 10.196,10.290,10.391,10.487,10.583,10.645,10.695,10.721,
     * 10.083,10.134,10.188,10.276,10.364,10.442,10.509,10.553,
     * 10.041,10.061,10.083,10.114,10.146,10.188,10.215,10.258,
     * 9.961,9.979,10.004,10.029,10.057,10.072,10.072,10.061,
     * 9.725,9.712,9.705,9.703,9.704,9.695,9.666,9.628,
     * 9.673,9.659,9.652,9.650,9.652,9.642,9.613,9.575,
     * 9.400,9.387,9.382,9.382,9.384,9.375,9.346,9.310,
     * 9.354,9.342,9.336,9.336,9.338,9.329,9.301,9.264/
      DATA ((EDEN(I,3,10,L,1),I=41,48),L=1,NZEN)/
     * 10.958,10.996,11.037,11.068,11.097,11.114,11.127,11.134,
     * 10.914,10.950,10.989,11.013,11.037,11.053,11.068,11.079,
     * 10.823,10.870,10.921,10.949,10.977,10.997,11.009,11.021,
     * 10.632,10.703,10.778,10.822,10.865,10.890,10.908,10.920,
     * 10.212,10.307,10.407,10.493,10.577,10.632,10.679,10.705,
     * 10.079,10.134,10.190,10.274,10.360,10.433,10.498,10.542,
     * 10.025,10.045,10.064,10.097,10.127,10.176,10.210,10.258,
     * 9.948,9.965,9.986,10.013,10.037,10.057,10.061,10.061,
     * 9.721,9.704,9.693,9.685,9.679,9.667,9.640,9.607,
     * 9.668,9.651,9.639,9.631,9.625,9.614,9.588,9.554,
     * 9.394,9.378,9.369,9.360,9.356,9.346,9.322,9.290,
     * 9.349,9.333,9.323,9.314,9.310,9.301,9.276,9.244/
      DATA ((EDEN(I,4,10,L,1),I=41,48),L=1,NZEN)/
     * 10.954,10.993,11.033,11.057,11.079,11.097,11.114,11.127,
     * 10.936,10.965,10.995,11.009,11.025,11.041,11.053,11.068,
     * 10.851,10.888,10.928,10.947,10.966,10.981,10.994,11.009,
     * 10.668,10.729,10.792,10.824,10.856,10.876,10.892,10.908,
     * 10.238,10.330,10.428,10.500,10.572,10.620,10.663,10.689,
     * 10.072,10.130,10.190,10.270,10.348,10.417,10.480,10.522,
     * 10.000,10.013,10.029,10.053,10.072,10.117,10.152,10.204,
     * 9.926,9.936,9.947,9.955,9.965,9.971,9.969,9.961,
     * 9.709,9.689,9.673,9.657,9.643,9.627,9.602,9.572,
     * 9.657,9.636,9.620,9.604,9.590,9.574,9.548,9.517,
     * 9.382,9.364,9.348,9.332,9.320,9.305,9.281,9.253,
     * 9.336,9.318,9.303,9.287,9.274,9.260,9.235,9.207/
      DATA ((EDEN(I,5,10,L,1),I=41,48),L=1,NZEN)/
     * 10.963,11.000,11.037,11.057,11.072,11.086,11.104,11.121,
     * 10.953,10.976,10.999,11.004,11.009,11.021,11.033,11.049,
     * 10.872,10.903,10.934,10.943,10.949,10.960,10.974,10.991,
     * 10.700,10.751,10.802,10.823,10.841,10.856,10.871,10.888,
     * 10.267,10.358,10.447,10.507,10.563,10.603,10.642,10.670,
     * 10.076,10.137,10.196,10.270,10.342,10.401,10.461,10.504,
     * 9.984,9.990,9.997,10.017,10.037,10.086,10.130,10.185,
     * 9.909,9.907,9.908,9.912,9.916,9.925,9.928,9.929,
     * 9.700,9.677,9.655,9.632,9.612,9.594,9.572,9.547,
     * 9.647,9.623,9.602,9.580,9.559,9.542,9.519,9.491,
     * 9.373,9.350,9.330,9.310,9.290,9.274,9.253,9.228,
     * 9.327,9.304,9.285,9.264,9.244,9.228,9.207,9.182/
      DATA ((EDEN(I,1,11,L,1),I=41,48),L=1,NZEN)/
     * 10.948,10.994,11.045,11.086,11.130,11.155,11.167,11.170,
     * 10.881,10.931,10.986,11.029,11.068,11.097,11.107,11.117,
     * 10.783,10.845,10.913,10.961,11.009,11.037,11.049,11.061,
     * 10.580,10.667,10.760,10.826,10.893,10.929,10.948,10.957,
     * 10.176,10.272,10.375,10.481,10.589,10.658,10.710,10.736,
     * 10.076,10.127,10.182,10.274,10.367,10.453,10.521,10.565,
     * 10.037,10.061,10.086,10.124,10.164,10.207,10.228,10.270,
     * 9.958,9.979,10.009,10.041,10.076,10.093,10.090,10.076,
     * 9.730,9.720,9.719,9.724,9.733,9.727,9.696,9.655,
     * 9.677,9.667,9.666,9.671,9.680,9.674,9.643,9.602,
     * 9.403,9.394,9.394,9.401,9.410,9.405,9.377,9.336,
     * 9.357,9.349,9.349,9.356,9.364,9.359,9.331,9.291/
      DATA ((EDEN(I,2,11,L,1),I=41,48),L=1,NZEN)/
     * 10.965,11.004,11.045,11.079,11.114,11.137,11.149,11.158,
     * 10.889,10.936,10.985,11.021,11.053,11.079,11.093,11.107,
     * 10.791,10.849,10.912,10.952,10.993,11.021,11.033,11.049,
     * 10.587,10.671,10.759,10.818,10.876,10.910,10.932,10.946,
     * 10.185,10.279,10.375,10.473,10.571,10.638,10.694,10.724,
     * 10.083,10.130,10.185,10.267,10.350,10.433,10.502,10.553,
     * 10.045,10.064,10.090,10.117,10.146,10.185,10.212,10.255,
     * 9.966,9.985,10.009,10.033,10.061,10.076,10.076,10.064,
     * 9.736,9.724,9.718,9.716,9.718,9.708,9.680,9.644,
     * 9.684,9.672,9.665,9.663,9.665,9.655,9.627,9.591,
     * 9.410,9.400,9.393,9.393,9.396,9.387,9.360,9.324,
     * 9.364,9.354,9.347,9.347,9.350,9.342,9.314,9.279/
      DATA ((EDEN(I,3,11,L,1),I=41,48),L=1,NZEN)/
     * 10.962,11.000,11.041,11.068,11.097,11.117,11.134,11.146,
     * 10.908,10.948,10.989,11.013,11.037,11.061,11.076,11.093,
     * 10.814,10.865,10.918,10.948,10.977,11.000,11.017,11.033,
     * 10.615,10.692,10.769,10.816,10.863,10.892,10.915,10.933,
     * 10.199,10.292,10.389,10.477,10.565,10.625,10.679,10.713,
     * 10.083,10.130,10.182,10.262,10.344,10.423,10.493,10.545,
     * 10.037,10.053,10.072,10.100,10.127,10.170,10.204,10.255,
     * 9.956,9.973,9.993,10.017,10.041,10.061,10.064,10.068,
     * 9.733,9.719,9.707,9.698,9.691,9.681,9.657,9.628,
     * 9.681,9.666,9.653,9.644,9.638,9.628,9.604,9.575,
     * 9.407,9.393,9.382,9.375,9.369,9.360,9.338,9.312,
     * 9.361,9.347,9.336,9.329,9.323,9.314,9.293,9.266/
      DATA ((EDEN(I,4,11,L,1),I=41,48),L=1,NZEN)/
     * 10.948,10.993,11.037,11.061,11.086,11.107,11.127,11.146,
     * 10.931,10.964,10.995,11.013,11.025,11.045,11.064,11.086,
     * 10.843,10.886,10.927,10.947,10.967,10.985,11.004,11.029,
     * 10.654,10.721,10.786,10.820,10.853,10.877,10.900,10.925,
     * 10.225,10.320,10.413,10.487,10.560,10.614,10.665,10.702,
     * 10.072,10.127,10.182,10.258,10.332,10.405,10.474,10.529,
     * 10.013,10.025,10.037,10.053,10.068,10.111,10.146,10.204,
     * 9.936,9.945,9.956,9.959,9.964,9.969,9.967,9.961,
     * 9.720,9.703,9.687,9.669,9.654,9.641,9.620,9.595,
     * 9.667,9.650,9.633,9.616,9.601,9.588,9.567,9.542,
     * 9.394,9.378,9.362,9.346,9.330,9.318,9.299,9.274,
     * 9.349,9.333,9.316,9.301,9.285,9.272,9.253,9.228/
      DATA ((EDEN(I,5,11,L,1),I=41,48),L=1,NZEN)/
     * 10.950,10.997,11.037,11.057,11.076,11.093,11.117,11.143,
     * 10.951,10.978,11.000,11.004,11.009,11.025,11.045,11.076,
     * 10.867,10.904,10.936,10.943,10.951,10.966,10.985,11.013,
     * 10.691,10.747,10.800,10.820,10.840,10.859,10.882,10.910,
     * 10.260,10.350,10.436,10.496,10.555,10.600,10.647,10.688,
     * 10.079,10.134,10.190,10.258,10.328,10.393,10.458,10.515,
     * 9.997,10.004,10.004,10.021,10.033,10.079,10.121,10.185,
     * 9.921,9.921,9.919,9.916,9.915,9.922,9.926,9.931,
     * 9.710,9.690,9.669,9.644,9.622,9.607,9.590,9.574,
     * 9.657,9.637,9.616,9.591,9.568,9.554,9.537,9.519,
     * 9.384,9.365,9.344,9.320,9.299,9.286,9.270,9.253,
     * 9.338,9.320,9.299,9.274,9.253,9.240,9.224,9.207/
      DATA ((EDEN(I,1,12,L,1),I=41,48),L=1,NZEN)/
     * 10.949,10.997,11.049,11.093,11.137,11.161,11.173,11.176,
     * 10.886,10.934,10.989,11.029,11.072,11.097,11.111,11.117,
     * 10.789,10.849,10.916,10.964,11.013,11.041,11.053,11.061,
     * 10.587,10.672,10.764,10.830,10.896,10.931,10.950,10.959,
     * 10.182,10.279,10.382,10.487,10.593,10.663,10.713,10.739,
     * 10.079,10.130,10.185,10.279,10.373,10.458,10.524,10.568,
     * 10.041,10.064,10.090,10.127,10.164,10.210,10.233,10.272,
     * 9.960,9.982,10.013,10.045,10.079,10.097,10.090,10.079,
     * 9.731,9.721,9.719,9.724,9.733,9.727,9.696,9.655,
     * 9.678,9.667,9.666,9.671,9.680,9.673,9.642,9.602,
     * 9.405,9.396,9.394,9.401,9.410,9.405,9.377,9.336,
     * 9.359,9.350,9.349,9.356,9.364,9.359,9.331,9.291/
      DATA ((EDEN(I,2,12,L,1),I=41,48),L=1,NZEN)/
     * 10.969,11.009,11.053,11.086,11.117,11.140,11.155,11.161,
     * 10.893,10.940,10.989,11.021,11.057,11.079,11.097,11.111,
     * 10.795,10.854,10.915,10.956,10.996,11.021,11.037,11.053,
     * 10.591,10.676,10.763,10.822,10.879,10.913,10.936,10.952,
     * 10.190,10.283,10.380,10.477,10.575,10.642,10.697,10.730,
     * 10.086,10.137,10.188,10.270,10.354,10.438,10.508,10.559,
     * 10.045,10.068,10.093,10.121,10.149,10.190,10.215,10.262,
     * 9.968,9.988,10.013,10.037,10.064,10.079,10.079,10.068,
     * 9.739,9.727,9.719,9.717,9.718,9.708,9.682,9.647,
     * 9.686,9.674,9.667,9.664,9.665,9.656,9.629,9.594,
     * 9.412,9.401,9.396,9.393,9.394,9.387,9.364,9.330,
     * 9.366,9.356,9.350,9.347,9.349,9.342,9.318,9.285/
      DATA ((EDEN(I,3,12,L,1),I=41,48),L=1,NZEN)/
     * 10.967,11.009,11.045,11.072,11.100,11.121,11.137,11.152,
     * 10.912,10.953,10.993,11.017,11.041,11.061,11.079,11.100,
     * 10.816,10.870,10.922,10.951,10.980,11.004,11.021,11.041,
     * 10.620,10.697,10.775,10.819,10.865,10.894,10.919,10.941,
     * 10.204,10.299,10.394,10.480,10.567,10.628,10.684,10.721,
     * 10.086,10.137,10.188,10.267,10.348,10.427,10.497,10.553,
     * 10.037,10.057,10.076,10.104,10.130,10.176,10.212,10.265,
     * 9.959,9.978,9.998,10.017,10.041,10.061,10.072,10.076,
     * 9.735,9.721,9.708,9.698,9.691,9.682,9.660,9.634,
     * 9.682,9.668,9.655,9.645,9.638,9.629,9.607,9.581,
     * 9.408,9.394,9.384,9.375,9.369,9.360,9.340,9.316,
     * 9.362,9.349,9.338,9.329,9.323,9.314,9.295,9.270/
      DATA ((EDEN(I,4,12,L,1),I=41,48),L=1,NZEN)/
     * 10.950,10.997,11.041,11.061,11.086,11.111,11.137,11.161,
     * 10.935,10.969,11.000,11.013,11.025,11.049,11.068,11.100,
     * 10.848,10.892,10.931,10.949,10.967,10.988,11.013,11.041,
     * 10.662,10.728,10.791,10.823,10.854,10.880,10.907,10.936,
     * 10.233,10.328,10.420,10.491,10.563,10.617,10.670,10.713,
     * 10.079,10.137,10.190,10.262,10.336,10.408,10.480,10.540,
     * 10.017,10.029,10.041,10.057,10.068,10.111,10.149,10.215,
     * 9.939,9.950,9.960,9.960,9.963,9.968,9.968,9.967,
     * 9.720,9.704,9.688,9.669,9.653,9.641,9.623,9.604,
     * 9.667,9.651,9.634,9.616,9.600,9.589,9.569,9.550,
     * 9.393,9.378,9.362,9.344,9.328,9.320,9.303,9.283,
     * 9.347,9.333,9.316,9.299,9.283,9.274,9.257,9.238/
      DATA ((EDEN(I,5,12,L,1),I=41,48),L=1,NZEN)/
     * 10.949,10.998,11.037,11.053,11.072,11.093,11.127,11.158,
     * 10.957,10.985,11.009,11.009,11.013,11.029,11.053,11.090,
     * 10.874,10.911,10.941,10.946,10.952,10.969,10.993,11.029,
     * 10.700,10.757,10.807,10.824,10.841,10.862,10.889,10.925,
     * 10.270,10.362,10.447,10.502,10.559,10.604,10.656,10.702,
     * 10.086,10.146,10.201,10.265,10.332,10.398,10.467,10.530,
     * 10.000,10.009,10.009,10.021,10.033,10.083,10.130,10.199,
     * 9.925,9.926,9.923,9.917,9.913,9.920,9.927,9.937,
     * 9.708,9.692,9.670,9.642,9.619,9.606,9.593,9.585,
     * 9.656,9.638,9.617,9.590,9.566,9.554,9.539,9.529,
     * 9.382,9.365,9.344,9.318,9.297,9.286,9.272,9.265,
     * 9.336,9.320,9.299,9.272,9.251,9.240,9.226,9.219/
      DATA ((EDEN(I,1,1,L,2),I=41,48),L=1,NZEN)/
     * 11.025,11.072,11.124,11.167,11.210,11.236,11.248,11.248,
     * 10.946,10.995,11.049,11.097,11.140,11.170,11.182,11.190,
     * 10.840,10.903,10.971,11.021,11.072,11.104,11.117,11.124,
     * 10.632,10.716,10.807,10.876,10.945,10.981,11.000,11.009,
     * 10.246,10.332,10.427,10.528,10.630,10.696,10.744,10.769,
     * 10.130,10.182,10.243,10.332,10.422,10.497,10.554,10.594,
     * 10.057,10.086,10.124,10.173,10.223,10.262,10.281,10.314,
     * 9.908,9.940,9.984,10.033,10.083,10.104,10.100,10.090,
     * 9.584,9.579,9.587,9.611,9.636,9.630,9.597,9.559,
     * 9.530,9.525,9.533,9.556,9.581,9.574,9.540,9.504,
     * 9.255,9.250,9.260,9.286,9.312,9.305,9.274,9.238,
     * 9.210,9.205,9.214,9.240,9.266,9.260,9.228,9.192/
      DATA ((EDEN(I,2,1,L,2),I=41,48),L=1,NZEN)/
     * 11.045,11.086,11.127,11.158,11.190,11.215,11.230,11.238,
     * 10.952,11.000,11.053,11.090,11.127,11.152,11.170,11.188,
     * 10.847,10.906,10.969,11.013,11.057,11.086,11.104,11.117,
     * 10.637,10.719,10.805,10.866,10.927,10.962,10.987,11.004,
     * 10.253,10.336,10.423,10.517,10.612,10.676,10.728,10.760,
     * 10.140,10.188,10.243,10.322,10.403,10.477,10.538,10.585,
     * 10.064,10.090,10.124,10.164,10.207,10.243,10.265,10.305,
     * 9.917,9.945,9.982,10.025,10.068,10.086,10.083,10.079,
     * 9.595,9.587,9.589,9.604,9.622,9.613,9.583,9.553,
     * 9.539,9.531,9.534,9.550,9.567,9.559,9.529,9.497,
     * 9.265,9.258,9.262,9.279,9.299,9.290,9.260,9.230,
     * 9.219,9.212,9.217,9.233,9.253,9.244,9.214,9.185/
      DATA ((EDEN(I,3,1,L,2),I=41,48),L=1,NZEN)/
     * 11.041,11.083,11.121,11.146,11.173,11.196,11.215,11.230,
     * 10.969,11.013,11.053,11.079,11.107,11.134,11.152,11.173,
     * 10.866,10.920,10.974,11.009,11.041,11.064,11.086,11.111,
     * 10.660,10.736,10.812,10.862,10.912,10.944,10.971,10.994,
     * 10.262,10.346,10.433,10.516,10.601,10.661,10.714,10.753,
     * 10.134,10.185,10.241,10.316,10.391,10.464,10.528,10.581,
     * 10.053,10.079,10.111,10.146,10.188,10.228,10.260,10.307,
     * 9.905,9.932,9.966,10.004,10.045,10.068,10.079,10.090,
     * 9.590,9.581,9.580,9.588,9.597,9.589,9.565,9.544,
     * 9.535,9.526,9.524,9.531,9.542,9.533,9.509,9.489,
     * 9.260,9.253,9.253,9.260,9.272,9.265,9.243,9.223,
     * 9.214,9.207,9.207,9.214,9.226,9.219,9.197,9.177/
      DATA ((EDEN(I,4,1,L,2),I=41,48),L=1,NZEN)/
     * 11.029,11.072,11.111,11.134,11.158,11.182,11.207,11.236,
     * 10.993,11.029,11.064,11.079,11.097,11.121,11.143,11.173,
     * 10.894,10.941,10.984,11.004,11.029,11.053,11.076,11.107,
     * 10.696,10.763,10.828,10.865,10.901,10.929,10.957,10.988,
     * 10.281,10.365,10.447,10.519,10.591,10.644,10.696,10.740,
     * 10.127,10.179,10.233,10.301,10.371,10.438,10.502,10.560,
     * 10.025,10.049,10.072,10.097,10.121,10.158,10.190,10.248,
     * 9.882,9.902,9.925,9.944,9.966,9.974,9.974,9.979,
     * 9.576,9.566,9.560,9.558,9.559,9.545,9.524,9.509,
     * 9.522,9.512,9.504,9.502,9.502,9.490,9.468,9.452,
     * 9.248,9.238,9.230,9.230,9.230,9.220,9.201,9.185,
     * 9.202,9.192,9.185,9.185,9.185,9.174,9.156,9.139/
      DATA ((EDEN(I,5,1,L,2),I=41,48),L=1,NZEN)/
     * 11.029,11.076,11.114,11.127,11.143,11.167,11.199,11.236,
     * 11.013,11.049,11.072,11.079,11.083,11.100,11.127,11.161,
     * 10.920,10.961,10.994,11.004,11.013,11.033,11.057,11.097,
     * 10.730,10.790,10.843,10.866,10.888,10.911,10.941,10.978,
     * 10.305,10.389,10.467,10.524,10.582,10.629,10.681,10.730,
     * 10.127,10.182,10.233,10.294,10.360,10.422,10.486,10.549,
     * 10.004,10.021,10.033,10.057,10.079,10.121,10.164,10.230,
     * 9.864,9.874,9.883,9.897,9.913,9.922,9.930,9.949,
     * 9.568,9.555,9.542,9.531,9.524,9.512,9.496,9.491,
     * 9.513,9.500,9.486,9.476,9.468,9.456,9.439,9.435,
     * 9.238,9.225,9.212,9.204,9.199,9.188,9.170,9.167,
     * 9.192,9.180,9.166,9.158,9.153,9.142,9.125,9.122/
      DATA ((EDEN(I,1,2,L,2),I=41,48),L=1,NZEN)/
     * 11.025,11.068,11.121,11.164,11.204,11.233,11.243,11.246,
     * 10.953,11.000,11.053,11.097,11.140,11.167,11.179,11.188,
     * 10.849,10.908,10.973,11.021,11.072,11.100,11.114,11.121,
     * 10.645,10.726,10.813,10.879,10.945,10.980,10.999,11.009,
     * 10.255,10.342,10.438,10.538,10.638,10.702,10.746,10.769,
     * 10.134,10.188,10.248,10.338,10.430,10.504,10.560,10.598,
     * 10.053,10.083,10.121,10.173,10.223,10.265,10.283,10.318,
     * 9.908,9.938,9.980,10.029,10.083,10.100,10.097,10.086,
     * 9.576,9.571,9.579,9.602,9.626,9.620,9.585,9.549,
     * 9.522,9.516,9.522,9.547,9.572,9.565,9.531,9.494,
     * 9.248,9.243,9.250,9.276,9.301,9.297,9.265,9.228,
     * 9.202,9.197,9.205,9.231,9.255,9.251,9.219,9.182/
      DATA ((EDEN(I,2,2,L,2),I=41,48),L=1,NZEN)/
     * 11.037,11.076,11.121,11.155,11.190,11.212,11.225,11.236,
     * 10.960,11.004,11.057,11.090,11.127,11.149,11.164,11.182,
     * 10.858,10.914,10.974,11.013,11.057,11.083,11.100,11.114,
     * 10.654,10.732,10.814,10.872,10.929,10.962,10.984,11.000,
     * 10.265,10.350,10.439,10.530,10.622,10.683,10.730,10.760,
     * 10.140,10.193,10.250,10.332,10.413,10.486,10.544,10.588,
     * 10.064,10.090,10.124,10.164,10.207,10.246,10.270,10.307,
     * 9.913,9.943,9.982,10.025,10.068,10.086,10.083,10.079,
     * 9.583,9.575,9.579,9.593,9.611,9.601,9.572,9.542,
     * 9.529,9.521,9.524,9.539,9.556,9.547,9.517,9.486,
     * 9.255,9.248,9.250,9.267,9.286,9.279,9.250,9.220,
     * 9.210,9.202,9.205,9.221,9.240,9.233,9.205,9.174/
      DATA ((EDEN(I,3,2,L,2),I=41,48),L=1,NZEN)/
     * 11.033,11.072,11.114,11.140,11.167,11.193,11.212,11.228,
     * 10.977,11.017,11.057,11.083,11.107,11.130,11.152,11.170,
     * 10.877,10.927,10.978,11.009,11.041,11.064,11.083,11.104,
     * 10.676,10.747,10.820,10.867,10.914,10.944,10.968,10.989,
     * 10.274,10.358,10.446,10.529,10.612,10.668,10.716,10.751,
     * 10.134,10.190,10.248,10.326,10.403,10.473,10.534,10.583,
     * 10.045,10.072,10.104,10.146,10.188,10.233,10.265,10.312,
     * 9.898,9.927,9.962,10.004,10.045,10.068,10.079,10.086,
     * 9.577,9.568,9.567,9.575,9.585,9.576,9.551,9.531,
     * 9.524,9.513,9.512,9.520,9.530,9.522,9.497,9.476,
     * 9.250,9.241,9.241,9.250,9.260,9.255,9.230,9.212,
     * 9.205,9.195,9.195,9.205,9.214,9.210,9.185,9.166/
      DATA ((EDEN(I,4,2,L,2),I=41,48),L=1,NZEN)/
     * 11.025,11.068,11.111,11.134,11.155,11.176,11.201,11.228,
     * 10.997,11.033,11.064,11.079,11.097,11.117,11.137,11.164,
     * 10.901,10.945,10.985,11.004,11.025,11.049,11.068,11.097,
     * 10.708,10.772,10.834,10.867,10.901,10.927,10.953,10.980,
     * 10.290,10.377,10.461,10.530,10.601,10.650,10.698,10.737,
     * 10.127,10.182,10.241,10.312,10.384,10.449,10.509,10.561,
     * 10.017,10.041,10.068,10.097,10.127,10.167,10.204,10.258,
     * 9.874,9.895,9.920,9.944,9.969,9.981,9.985,9.991,
     * 9.566,9.555,9.548,9.547,9.548,9.535,9.512,9.497,
     * 9.511,9.498,9.491,9.491,9.493,9.480,9.456,9.441,
     * 9.236,9.225,9.220,9.220,9.223,9.212,9.190,9.176,
     * 9.190,9.180,9.174,9.174,9.177,9.166,9.145,9.130/
      DATA ((EDEN(I,5,2,L,2),I=41,48),L=1,NZEN)/
     * 11.033,11.079,11.117,11.130,11.143,11.164,11.196,11.225,
     * 11.017,11.049,11.072,11.079,11.083,11.097,11.121,11.152,
     * 10.924,10.962,10.993,11.004,11.013,11.029,11.053,11.086,
     * 10.738,10.794,10.846,10.867,10.889,10.910,10.936,10.968,
     * 10.314,10.396,10.474,10.533,10.591,10.634,10.680,10.724,
     * 10.127,10.185,10.241,10.305,10.371,10.430,10.491,10.549,
     * 9.999,10.013,10.029,10.057,10.090,10.134,10.176,10.241,
     * 9.856,9.866,9.877,9.896,9.916,9.932,9.944,9.965,
     * 9.559,9.544,9.531,9.522,9.516,9.502,9.484,9.479,
     * 9.502,9.489,9.476,9.467,9.461,9.447,9.430,9.422,
     * 9.228,9.215,9.201,9.196,9.190,9.179,9.161,9.155,
     * 9.182,9.169,9.156,9.150,9.145,9.133,9.116,9.110/
      DATA ((EDEN(I,1,3,L,2),I=41,48),L=1,NZEN)/
     * 11.025,11.068,11.117,11.158,11.201,11.228,11.241,11.243,
     * 10.953,10.998,11.049,11.093,11.137,11.164,11.179,11.188,
     * 10.851,10.908,10.972,11.021,11.068,11.097,11.111,11.117,
     * 10.648,10.727,10.813,10.878,10.943,10.978,10.996,11.004,
     * 10.255,10.344,10.441,10.539,10.638,10.701,10.744,10.766,
     * 10.130,10.185,10.248,10.338,10.430,10.504,10.558,10.597,
     * 10.049,10.079,10.121,10.167,10.217,10.262,10.283,10.318,
     * 9.905,9.934,9.978,10.029,10.079,10.100,10.093,10.086,
     * 9.571,9.565,9.573,9.595,9.620,9.614,9.580,9.544,
     * 9.516,9.511,9.519,9.540,9.566,9.559,9.526,9.489,
     * 9.243,9.238,9.246,9.270,9.294,9.290,9.258,9.223,
     * 9.197,9.192,9.200,9.224,9.249,9.244,9.212,9.177/
      DATA ((EDEN(I,2,3,L,2),I=41,48),L=1,NZEN)/
     * 11.029,11.068,11.114,11.149,11.185,11.210,11.225,11.233,
     * 10.961,11.004,11.049,11.086,11.124,11.146,11.158,11.173,
     * 10.862,10.916,10.974,11.013,11.053,11.079,11.093,11.107,
     * 10.661,10.736,10.815,10.872,10.928,10.960,10.980,10.993,
     * 10.270,10.354,10.444,10.535,10.625,10.684,10.728,10.755,
     * 10.140,10.193,10.253,10.334,10.418,10.489,10.545,10.587,
     * 10.057,10.083,10.121,10.161,10.207,10.246,10.270,10.307,
     * 9.910,9.939,9.978,10.021,10.068,10.083,10.083,10.076,
     * 9.575,9.567,9.571,9.585,9.602,9.594,9.563,9.531,
     * 9.521,9.512,9.515,9.530,9.548,9.539,9.508,9.476,
     * 9.246,9.238,9.243,9.260,9.279,9.272,9.241,9.212,
     * 9.200,9.192,9.197,9.214,9.233,9.226,9.195,9.166/
      DATA ((EDEN(I,3,3,L,2),I=41,48),L=1,NZEN)/
     * 11.029,11.068,11.107,11.137,11.164,11.188,11.207,11.217,
     * 10.979,11.017,11.057,11.083,11.107,11.127,11.143,11.158,
     * 10.881,10.928,10.978,11.009,11.037,11.061,11.076,11.093,
     * 10.684,10.754,10.825,10.870,10.914,10.942,10.962,10.979,
     * 10.276,10.364,10.453,10.537,10.618,10.671,10.715,10.744,
     * 10.130,10.188,10.248,10.330,10.412,10.479,10.537,10.580,
     * 10.037,10.064,10.100,10.143,10.188,10.233,10.265,10.310,
     * 9.894,9.922,9.957,10.000,10.041,10.068,10.076,10.086,
     * 9.571,9.560,9.558,9.567,9.577,9.568,9.542,9.519,
     * 9.516,9.504,9.502,9.512,9.522,9.513,9.487,9.464,
     * 9.241,9.230,9.230,9.241,9.253,9.246,9.220,9.199,
     * 9.195,9.185,9.185,9.195,9.207,9.200,9.174,9.153/
      DATA ((EDEN(I,4,3,L,2),I=41,48),L=1,NZEN)/
     * 11.029,11.068,11.107,11.134,11.155,11.173,11.193,11.212,
     * 11.000,11.033,11.064,11.079,11.093,11.111,11.130,11.149,
     * 10.906,10.946,10.986,11.009,11.025,11.045,11.061,11.083,
     * 10.713,10.775,10.836,10.869,10.902,10.925,10.945,10.966,
     * 10.292,10.380,10.467,10.538,10.609,10.653,10.695,10.727,
     * 10.121,10.179,10.243,10.318,10.393,10.455,10.512,10.558,
     * 10.009,10.033,10.061,10.097,10.134,10.176,10.212,10.262,
     * 9.866,9.887,9.914,9.944,9.976,9.991,9.996,10.004,
     * 9.560,9.547,9.539,9.540,9.542,9.529,9.504,9.484,
     * 9.505,9.490,9.483,9.484,9.487,9.474,9.449,9.428,
     * 9.230,9.217,9.212,9.215,9.217,9.204,9.182,9.164,
     * 9.185,9.172,9.166,9.169,9.172,9.158,9.136,9.119/
      DATA ((EDEN(I,5,3,L,2),I=41,48),L=1,NZEN)/
     * 11.045,11.083,11.117,11.130,11.143,11.161,11.182,11.204,
     * 11.017,11.045,11.068,11.076,11.079,11.093,11.111,11.134,
     * 10.925,10.960,10.993,11.004,11.013,11.025,11.041,11.068,
     * 10.740,10.794,10.845,10.867,10.889,10.905,10.925,10.950,
     * 10.316,10.398,10.479,10.538,10.595,10.635,10.677,10.711,
     * 10.121,10.179,10.241,10.310,10.378,10.435,10.491,10.540,
     * 9.987,10.004,10.021,10.057,10.093,10.140,10.185,10.243,
     * 9.843,9.855,9.871,9.897,9.924,9.942,9.955,9.977,
     * 9.551,9.534,9.522,9.516,9.511,9.496,9.474,9.464,
     * 9.496,9.479,9.467,9.461,9.455,9.441,9.418,9.407,
     * 9.220,9.204,9.193,9.188,9.185,9.173,9.152,9.143,
     * 9.174,9.158,9.147,9.142,9.139,9.127,9.107,9.097/
      DATA ((EDEN(I,1,4,L,2),I=41,48),L=1,NZEN)/
     * 11.029,11.068,11.114,11.155,11.199,11.225,11.236,11.238,
     * 10.947,10.993,11.045,11.090,11.134,11.161,11.176,11.182,
     * 10.846,10.904,10.968,11.017,11.064,11.097,11.107,11.117,
     * 10.644,10.723,10.810,10.875,10.940,10.975,10.993,11.000,
     * 10.253,10.340,10.436,10.537,10.635,10.698,10.742,10.764,
     * 10.127,10.182,10.246,10.336,10.428,10.502,10.558,10.595,
     * 10.045,10.076,10.114,10.164,10.215,10.258,10.279,10.314,
     * 9.898,9.930,9.975,10.025,10.076,10.097,10.090,10.083,
     * 9.566,9.560,9.568,9.592,9.616,9.611,9.576,9.540,
     * 9.511,9.505,9.513,9.537,9.562,9.555,9.521,9.484,
     * 9.236,9.233,9.241,9.265,9.292,9.288,9.255,9.220,
     * 9.190,9.187,9.195,9.219,9.246,9.242,9.210,9.174/
      DATA ((EDEN(I,2,4,L,2),I=41,48),L=1,NZEN)/
     * 11.025,11.064,11.111,11.146,11.185,11.207,11.220,11.223,
     * 10.960,11.004,11.049,11.083,11.117,11.143,11.158,11.164,
     * 10.862,10.914,10.970,11.013,11.053,11.076,11.090,11.100,
     * 10.662,10.736,10.815,10.872,10.928,10.958,10.975,10.985,
     * 10.267,10.354,10.446,10.537,10.627,10.685,10.728,10.750,
     * 10.137,10.190,10.250,10.336,10.420,10.490,10.544,10.582,
     * 10.053,10.079,10.114,10.158,10.204,10.246,10.267,10.303,
     * 9.906,9.934,9.974,10.017,10.064,10.079,10.076,10.068,
     * 9.568,9.560,9.563,9.580,9.598,9.588,9.555,9.521,
     * 9.513,9.504,9.507,9.524,9.542,9.533,9.500,9.467,
     * 9.238,9.230,9.236,9.253,9.272,9.265,9.236,9.201,
     * 9.192,9.185,9.190,9.207,9.226,9.219,9.190,9.156/
      DATA ((EDEN(I,3,4,L,2),I=41,48),L=1,NZEN)/
     * 11.029,11.064,11.107,11.134,11.161,11.182,11.196,11.204,
     * 10.981,11.017,11.053,11.079,11.104,11.124,11.137,11.146,
     * 10.884,10.928,10.977,11.009,11.037,11.057,11.068,11.079,
     * 10.688,10.754,10.826,10.872,10.916,10.940,10.955,10.966,
     * 10.279,10.365,10.458,10.540,10.623,10.673,10.713,10.735,
     * 10.127,10.185,10.248,10.332,10.417,10.481,10.535,10.573,
     * 10.029,10.057,10.093,10.140,10.188,10.230,10.262,10.305,
     * 9.886,9.914,9.952,9.997,10.045,10.068,10.072,10.079,
     * 9.563,9.551,9.549,9.560,9.572,9.561,9.531,9.504,
     * 9.508,9.496,9.494,9.505,9.517,9.507,9.477,9.449,
     * 9.233,9.223,9.223,9.236,9.248,9.241,9.212,9.185,
     * 9.187,9.177,9.177,9.190,9.202,9.195,9.166,9.139/
      DATA ((EDEN(I,4,4,L,2),I=41,48),L=1,NZEN)/
     * 11.041,11.072,11.107,11.127,11.146,11.164,11.176,11.188,
     * 11.000,11.029,11.061,11.079,11.093,11.107,11.117,11.127,
     * 10.907,10.945,10.985,11.004,11.025,11.037,11.049,11.061,
     * 10.719,10.777,10.838,10.872,10.904,10.922,10.935,10.947,
     * 10.294,10.380,10.471,10.543,10.615,10.656,10.691,10.714,
     * 10.114,10.176,10.246,10.324,10.401,10.461,10.513,10.550,
     * 9.998,10.025,10.057,10.097,10.140,10.185,10.223,10.267,
     * 9.853,9.876,9.908,9.943,9.980,10.000,10.009,10.017,
     * 9.556,9.539,9.533,9.535,9.538,9.522,9.494,9.468,
     * 9.501,9.483,9.476,9.479,9.483,9.468,9.439,9.413,
     * 9.225,9.210,9.204,9.210,9.212,9.201,9.173,9.149,
     * 9.180,9.164,9.158,9.164,9.166,9.156,9.127,9.103/
      DATA ((EDEN(I,5,4,L,2),I=41,48),L=1,NZEN)/
     * 11.064,11.090,11.117,11.130,11.140,11.149,11.158,11.170,
     * 11.017,11.041,11.064,11.072,11.079,11.086,11.097,11.107,
     * 10.930,10.960,10.992,11.004,11.013,11.021,11.029,11.041,
     * 10.745,10.795,10.848,10.870,10.891,10.903,10.914,10.926,
     * 10.314,10.396,10.483,10.544,10.603,10.637,10.670,10.694,
     * 10.111,10.173,10.241,10.314,10.387,10.441,10.491,10.529,
     * 9.971,9.989,10.013,10.057,10.100,10.149,10.193,10.243,
     * 9.828,9.838,9.855,9.892,9.927,9.951,9.966,9.986,
     * 9.549,9.528,9.516,9.511,9.507,9.490,9.464,9.444,
     * 9.493,9.471,9.459,9.456,9.452,9.435,9.408,9.387,
     * 9.217,9.199,9.188,9.185,9.182,9.167,9.143,9.124,
     * 9.172,9.153,9.142,9.139,9.136,9.122,9.097,9.078/
      DATA ((EDEN(I,1,5,L,2),I=41,48),L=1,NZEN)/
     * 11.037,11.076,11.121,11.161,11.201,11.225,11.233,11.233,
     * 10.946,10.993,11.045,11.090,11.134,11.158,11.170,11.179,
     * 10.846,10.903,10.966,11.013,11.064,11.093,11.107,11.114,
     * 10.644,10.723,10.808,10.874,10.940,10.975,10.993,11.000,
     * 10.253,10.340,10.436,10.537,10.637,10.699,10.743,10.765,
     * 10.124,10.179,10.246,10.336,10.430,10.504,10.560,10.597,
     * 10.041,10.072,10.111,10.161,10.212,10.258,10.279,10.314,
     * 9.894,9.927,9.971,10.021,10.072,10.093,10.086,10.076,
     * 9.560,9.555,9.563,9.588,9.613,9.605,9.572,9.535,
     * 9.505,9.500,9.508,9.531,9.558,9.551,9.517,9.480,
     * 9.230,9.228,9.236,9.260,9.288,9.283,9.250,9.215,
     * 9.185,9.182,9.190,9.214,9.242,9.238,9.205,9.169/
      DATA ((EDEN(I,2,5,L,2),I=41,48),L=1,NZEN)/
     * 11.029,11.068,11.114,11.149,11.185,11.204,11.212,11.215,
     * 10.961,11.000,11.049,11.083,11.117,11.140,11.152,11.155,
     * 10.863,10.913,10.970,11.013,11.053,11.076,11.086,11.093,
     * 10.667,10.738,10.816,10.873,10.928,10.957,10.972,10.979,
     * 10.270,10.356,10.449,10.542,10.632,10.688,10.728,10.747,
     * 10.137,10.190,10.253,10.340,10.427,10.496,10.548,10.581,
     * 10.049,10.076,10.114,10.158,10.204,10.246,10.267,10.303,
     * 9.906,9.932,9.970,10.017,10.061,10.079,10.072,10.064,
     * 9.562,9.553,9.556,9.574,9.592,9.581,9.548,9.512,
     * 9.507,9.497,9.500,9.519,9.537,9.526,9.493,9.456,
     * 9.233,9.225,9.228,9.248,9.267,9.260,9.228,9.193,
     * 9.187,9.180,9.182,9.202,9.221,9.214,9.182,9.147/
      DATA ((EDEN(I,3,5,L,2),I=41,48),L=1,NZEN)/
     * 11.029,11.064,11.104,11.134,11.161,11.179,11.188,11.188,
     * 10.982,11.013,11.053,11.079,11.104,11.121,11.127,11.134,
     * 10.886,10.929,10.978,11.009,11.037,11.053,11.064,11.068,
     * 10.694,10.757,10.827,10.873,10.918,10.939,10.950,10.956,
     * 10.281,10.369,10.464,10.547,10.629,10.677,10.712,10.728,
     * 10.124,10.182,10.250,10.338,10.425,10.487,10.538,10.569,
     * 10.021,10.049,10.090,10.140,10.188,10.233,10.262,10.301,
     * 9.879,9.906,9.947,9.994,10.041,10.064,10.068,10.072,
     * 9.555,9.542,9.542,9.553,9.565,9.554,9.521,9.490,
     * 9.501,9.487,9.486,9.498,9.511,9.498,9.467,9.435,
     * 9.228,9.215,9.215,9.230,9.243,9.233,9.201,9.170,
     * 9.182,9.169,9.169,9.185,9.197,9.187,9.156,9.125/
      DATA ((EDEN(I,4,5,L,2),I=41,48),L=1,NZEN)/
     * 11.049,11.076,11.104,11.124,11.140,11.152,11.161,11.164,
     * 11.004,11.029,11.061,11.079,11.093,11.104,11.107,11.111,
     * 10.913,10.946,10.985,11.009,11.025,11.037,11.041,11.045,
     * 10.725,10.780,10.840,10.874,10.906,10.920,10.928,10.931,
     * 10.301,10.386,10.479,10.551,10.622,10.660,10.689,10.704,
     * 10.111,10.176,10.248,10.332,10.413,10.470,10.516,10.544,
     * 9.990,10.017,10.049,10.100,10.146,10.193,10.228,10.267,
     * 9.848,9.869,9.901,9.944,9.986,10.009,10.013,10.021,
     * 9.550,9.530,9.524,9.529,9.533,9.516,9.484,9.453,
     * 9.494,9.474,9.468,9.473,9.477,9.462,9.430,9.398,
     * 9.217,9.201,9.196,9.201,9.207,9.196,9.164,9.137,
     * 9.172,9.156,9.150,9.156,9.161,9.150,9.119,9.091/
      DATA ((EDEN(I,5,5,L,2),I=41,48),L=1,NZEN)/
     * 11.079,11.090,11.111,11.121,11.124,11.130,11.137,11.137,
     * 11.025,11.045,11.068,11.076,11.079,11.083,11.086,11.083,
     * 10.936,10.962,10.994,11.004,11.013,11.017,11.017,11.021,
     * 10.755,10.800,10.852,10.875,10.894,10.901,10.904,10.906,
     * 10.324,10.405,10.493,10.555,10.613,10.642,10.667,10.679,
     * 10.111,10.176,10.248,10.326,10.401,10.450,10.494,10.521,
     * 9.959,9.977,10.004,10.057,10.107,10.158,10.201,10.243,
     * 9.816,9.825,9.847,9.888,9.927,9.956,9.973,9.991,
     * 9.545,9.520,9.507,9.504,9.501,9.481,9.450,9.423,
     * 9.490,9.464,9.452,9.449,9.446,9.427,9.394,9.369,
     * 9.215,9.190,9.179,9.179,9.176,9.161,9.130,9.107,
     * 9.169,9.145,9.133,9.133,9.130,9.116,9.085,9.061/
      DATA ((EDEN(I,1,6,L,2),I=41,48),L=1,NZEN)/
     * 11.045,11.083,11.124,11.164,11.201,11.225,11.233,11.230,
     * 10.949,10.994,11.045,11.090,11.134,11.158,11.170,11.179,
     * 10.851,10.906,10.968,11.017,11.064,11.093,11.107,11.114,
     * 10.652,10.730,10.814,10.878,10.941,10.975,10.993,11.000,
     * 10.258,10.348,10.446,10.544,10.643,10.705,10.747,10.768,
     * 10.124,10.182,10.248,10.342,10.436,10.511,10.566,10.601,
     * 10.041,10.068,10.107,10.161,10.212,10.260,10.283,10.320,
     * 9.894,9.925,9.968,10.017,10.068,10.090,10.086,10.076,
     * 9.553,9.547,9.556,9.580,9.605,9.599,9.566,9.529,
     * 9.497,9.493,9.501,9.525,9.550,9.544,9.511,9.474,
     * 9.223,9.220,9.230,9.255,9.283,9.276,9.246,9.210,
     * 9.177,9.174,9.185,9.210,9.238,9.231,9.200,9.164/
      DATA ((EDEN(I,2,6,L,2),I=41,48),L=1,NZEN)/
     * 11.037,11.072,11.117,11.149,11.185,11.204,11.212,11.212,
     * 10.965,11.004,11.049,11.083,11.117,11.140,11.149,11.152,
     * 10.869,10.917,10.972,11.013,11.053,11.076,11.086,11.090,
     * 10.676,10.745,10.822,10.877,10.930,10.958,10.971,10.975,
     * 10.276,10.364,10.458,10.550,10.640,10.694,10.731,10.747,
     * 10.137,10.193,10.258,10.346,10.435,10.502,10.554,10.584,
     * 10.045,10.072,10.111,10.155,10.204,10.246,10.272,10.305,
     * 9.902,9.929,9.969,10.013,10.057,10.076,10.072,10.061,
     * 9.553,9.543,9.548,9.566,9.584,9.574,9.539,9.501,
     * 9.498,9.489,9.493,9.511,9.529,9.519,9.484,9.446,
     * 9.225,9.215,9.220,9.241,9.260,9.253,9.217,9.182,
     * 9.180,9.169,9.174,9.195,9.214,9.207,9.172,9.136/
      DATA ((EDEN(I,3,6,L,2),I=41,48),L=1,NZEN)/
     * 11.037,11.068,11.107,11.137,11.161,11.176,11.179,11.179,
     * 10.985,11.017,11.053,11.079,11.104,11.121,11.124,11.124,
     * 10.893,10.932,10.978,11.009,11.037,11.053,11.061,11.061,
     * 10.704,10.764,10.833,10.877,10.920,10.939,10.947,10.948,
     * 10.290,10.378,10.474,10.556,10.637,10.682,10.713,10.725,
     * 10.124,10.188,10.258,10.346,10.435,10.496,10.542,10.568,
     * 10.017,10.045,10.086,10.137,10.190,10.236,10.267,10.301,
     * 9.878,9.903,9.943,9.990,10.037,10.061,10.064,10.064,
     * 9.547,9.531,9.531,9.545,9.558,9.545,9.512,9.477,
     * 9.491,9.477,9.477,9.490,9.502,9.490,9.456,9.422,
     * 9.217,9.204,9.207,9.220,9.236,9.225,9.193,9.158,
     * 9.172,9.158,9.161,9.174,9.190,9.180,9.147,9.113/
      DATA ((EDEN(I,4,6,L,2),I=41,48),L=1,NZEN)/
     * 11.057,11.076,11.097,11.117,11.134,11.143,11.149,11.143,
     * 11.009,11.029,11.061,11.079,11.093,11.100,11.100,11.097,
     * 10.919,10.949,10.987,11.009,11.025,11.033,11.037,11.033,
     * 10.736,10.786,10.844,10.878,10.908,10.921,10.924,10.922,
     * 10.314,10.396,10.490,10.562,10.631,10.667,10.690,10.698,
     * 10.117,10.182,10.258,10.342,10.425,10.479,10.520,10.542,
     * 9.986,10.013,10.049,10.100,10.152,10.199,10.236,10.270,
     * 9.843,9.864,9.900,9.943,9.986,10.009,10.013,10.017,
     * 9.542,9.520,9.515,9.520,9.525,9.508,9.473,9.439,
     * 9.486,9.464,9.459,9.465,9.470,9.453,9.418,9.384,
     * 9.212,9.193,9.188,9.196,9.204,9.188,9.155,9.121,
     * 9.166,9.147,9.142,9.150,9.158,9.142,9.110,9.075/
      DATA ((EDEN(I,5,6,L,2),I=41,48),L=1,NZEN)/
     * 11.079,11.083,11.097,11.104,11.111,11.114,11.111,11.107,
     * 11.033,11.049,11.072,11.079,11.083,11.079,11.076,11.068,
     * 10.945,10.967,10.997,11.009,11.013,11.013,11.009,11.004,
     * 10.768,10.808,10.859,10.880,10.898,10.901,10.898,10.893,
     * 10.340,10.418,10.508,10.568,10.624,10.648,10.666,10.670,
     * 10.114,10.182,10.260,10.340,10.417,10.461,10.497,10.515,
     * 9.958,9.973,10.000,10.061,10.114,10.164,10.207,10.243,
     * 9.810,9.818,9.840,9.885,9.928,9.957,9.973,9.989,
     * 9.540,9.511,9.497,9.496,9.493,9.473,9.438,9.407,
     * 9.484,9.455,9.442,9.441,9.438,9.418,9.382,9.350,
     * 9.207,9.179,9.170,9.170,9.167,9.152,9.117,9.090,
     * 9.161,9.133,9.125,9.125,9.122,9.107,9.072,9.044/
      DATA ((EDEN(I,1,7,L,2),I=41,48),L=1,NZEN)/
     * 11.049,11.083,11.124,11.164,11.204,11.225,11.233,11.230,
     * 10.955,10.999,11.049,11.090,11.134,11.158,11.170,11.179,
     * 10.857,10.911,10.971,11.017,11.064,11.093,11.107,11.114,
     * 10.661,10.736,10.818,10.881,10.943,10.976,10.993,11.000,
     * 10.265,10.354,10.453,10.551,10.650,10.709,10.750,10.769,
     * 10.127,10.185,10.253,10.348,10.442,10.517,10.571,10.605,
     * 10.041,10.068,10.107,10.161,10.215,10.262,10.286,10.322,
     * 9.894,9.925,9.968,10.017,10.068,10.090,10.086,10.076,
     * 9.549,9.544,9.551,9.575,9.601,9.594,9.561,9.525,
     * 9.494,9.489,9.497,9.521,9.547,9.540,9.507,9.470,
     * 9.220,9.217,9.225,9.250,9.276,9.272,9.241,9.204,
     * 9.174,9.172,9.180,9.205,9.231,9.226,9.195,9.158/
      DATA ((EDEN(I,2,7,L,2),I=41,48),L=1,NZEN)/
     * 11.037,11.076,11.114,11.152,11.188,11.204,11.212,11.210,
     * 10.969,11.009,11.053,11.086,11.117,11.140,11.149,11.152,
     * 10.874,10.921,10.974,11.013,11.053,11.076,11.083,11.086,
     * 10.684,10.751,10.825,10.880,10.932,10.959,10.971,10.976,
     * 10.286,10.371,10.467,10.556,10.645,10.697,10.732,10.747,
     * 10.140,10.196,10.265,10.352,10.441,10.508,10.558,10.587,
     * 10.045,10.072,10.111,10.158,10.207,10.250,10.274,10.307,
     * 9.902,9.929,9.969,10.013,10.057,10.076,10.068,10.057,
     * 9.549,9.539,9.543,9.561,9.580,9.569,9.534,9.497,
     * 9.493,9.483,9.487,9.505,9.525,9.515,9.480,9.441,
     * 9.220,9.212,9.217,9.236,9.255,9.248,9.215,9.176,
     * 9.174,9.166,9.172,9.190,9.210,9.202,9.169,9.130/
      DATA ((EDEN(I,3,7,L,2),I=41,48),L=1,NZEN)/
     * 11.041,11.072,11.107,11.137,11.161,11.176,11.179,11.176,
     * 10.988,11.017,11.053,11.079,11.104,11.117,11.124,11.124,
     * 10.897,10.934,10.980,11.009,11.037,11.053,11.061,11.061,
     * 10.712,10.769,10.835,10.879,10.920,10.940,10.948,10.947,
     * 10.301,10.387,10.481,10.563,10.642,10.685,10.713,10.724,
     * 10.130,10.193,10.265,10.352,10.441,10.500,10.545,10.569,
     * 10.017,10.045,10.086,10.140,10.190,10.238,10.270,10.303,
     * 9.877,9.904,9.945,9.993,10.041,10.061,10.061,10.061,
     * 9.542,9.526,9.526,9.539,9.553,9.540,9.507,9.471,
     * 9.486,9.471,9.471,9.484,9.498,9.486,9.452,9.415,
     * 9.212,9.199,9.201,9.217,9.230,9.220,9.188,9.152,
     * 9.166,9.153,9.156,9.172,9.185,9.174,9.142,9.107/
      DATA ((EDEN(I,4,7,L,2),I=41,48),L=1,NZEN)/
     * 11.057,11.072,11.097,11.117,11.134,11.143,11.146,11.143,
     * 11.013,11.033,11.061,11.079,11.093,11.100,11.097,11.093,
     * 10.923,10.951,10.988,11.009,11.025,11.033,11.033,11.029,
     * 10.743,10.790,10.848,10.880,10.910,10.921,10.923,10.919,
     * 10.322,10.405,10.498,10.568,10.635,10.668,10.690,10.695,
     * 10.121,10.188,10.267,10.350,10.431,10.483,10.522,10.540,
     * 9.986,10.009,10.049,10.100,10.155,10.201,10.233,10.267,
     * 9.843,9.863,9.899,9.942,9.983,10.004,10.009,10.009,
     * 9.535,9.513,9.508,9.515,9.520,9.502,9.467,9.430,
     * 9.480,9.458,9.453,9.459,9.464,9.447,9.412,9.375,
     * 9.207,9.185,9.179,9.190,9.199,9.182,9.149,9.114,
     * 9.161,9.139,9.133,9.145,9.153,9.136,9.103,9.068/
      DATA ((EDEN(I,5,7,L,2),I=41,48),L=1,NZEN)/
     * 11.072,11.079,11.090,11.104,11.111,11.114,11.111,11.104,
     * 11.033,11.045,11.072,11.079,11.083,11.079,11.072,11.064,
     * 10.948,10.968,10.997,11.009,11.017,11.017,11.009,11.000,
     * 10.774,10.812,10.860,10.881,10.898,10.900,10.897,10.889,
     * 10.348,10.427,10.515,10.573,10.628,10.650,10.665,10.667,
     * 10.124,10.190,10.270,10.348,10.423,10.464,10.497,10.512,
     * 9.958,9.974,10.004,10.061,10.117,10.167,10.207,10.241,
     * 9.812,9.818,9.839,9.885,9.928,9.954,9.965,9.978,
     * 9.534,9.504,9.491,9.490,9.487,9.467,9.430,9.396,
     * 9.479,9.449,9.436,9.435,9.431,9.412,9.377,9.342,
     * 9.201,9.173,9.161,9.164,9.164,9.146,9.111,9.079,
     * 9.156,9.127,9.116,9.119,9.119,9.100,9.065,9.033/
      DATA ((EDEN(I,1,8,L,2),I=41,48),L=1,NZEN)/
     * 11.045,11.079,11.121,11.161,11.201,11.225,11.233,11.230,
     * 10.955,10.999,11.049,11.090,11.134,11.161,11.176,11.182,
     * 10.856,10.910,10.971,11.021,11.068,11.093,11.107,11.114,
     * 10.659,10.735,10.818,10.881,10.943,10.976,10.993,11.000,
     * 10.262,10.352,10.452,10.550,10.647,10.708,10.748,10.769,
     * 10.127,10.188,10.253,10.346,10.441,10.515,10.568,10.603,
     * 10.041,10.072,10.111,10.164,10.215,10.262,10.286,10.322,
     * 9.894,9.927,9.971,10.021,10.072,10.093,10.090,10.079,
     * 9.554,9.549,9.556,9.580,9.604,9.599,9.565,9.529,
     * 9.500,9.494,9.501,9.525,9.550,9.544,9.511,9.474,
     * 9.228,9.223,9.230,9.255,9.281,9.276,9.246,9.210,
     * 9.182,9.177,9.185,9.210,9.235,9.231,9.200,9.164/
      DATA ((EDEN(I,2,8,L,2),I=41,48),L=1,NZEN)/
     * 11.041,11.076,11.117,11.149,11.185,11.204,11.212,11.212,
     * 10.968,11.009,11.053,11.086,11.117,11.140,11.149,11.155,
     * 10.872,10.920,10.974,11.013,11.053,11.076,11.086,11.090,
     * 10.679,10.747,10.824,10.879,10.931,10.959,10.972,10.977,
     * 10.281,10.367,10.462,10.553,10.641,10.694,10.731,10.747,
     * 10.140,10.196,10.260,10.348,10.438,10.504,10.554,10.584,
     * 10.045,10.076,10.111,10.158,10.207,10.248,10.274,10.307,
     * 9.905,9.931,9.970,10.017,10.061,10.076,10.072,10.061,
     * 9.554,9.544,9.548,9.566,9.584,9.574,9.540,9.502,
     * 9.500,9.489,9.493,9.511,9.529,9.520,9.486,9.447,
     * 9.228,9.217,9.223,9.241,9.260,9.253,9.220,9.185,
     * 9.182,9.172,9.177,9.195,9.214,9.207,9.174,9.139/
      DATA ((EDEN(I,3,8,L,2),I=41,48),L=1,NZEN)/
     * 11.037,11.068,11.107,11.137,11.161,11.179,11.185,11.185,
     * 10.986,11.017,11.053,11.079,11.104,11.121,11.124,11.127,
     * 10.894,10.932,10.979,11.009,11.037,11.057,11.061,11.064,
     * 10.706,10.766,10.833,10.877,10.919,10.940,10.949,10.951,
     * 10.297,10.382,10.476,10.558,10.637,10.681,10.713,10.725,
     * 10.127,10.190,10.260,10.348,10.435,10.494,10.542,10.568,
     * 10.025,10.049,10.090,10.140,10.190,10.238,10.267,10.301,
     * 9.881,9.907,9.946,9.993,10.041,10.061,10.064,10.064,
     * 9.548,9.533,9.531,9.545,9.558,9.545,9.512,9.479,
     * 9.491,9.477,9.477,9.490,9.502,9.491,9.458,9.423,
     * 9.217,9.204,9.207,9.223,9.236,9.225,9.193,9.161,
     * 9.172,9.158,9.161,9.177,9.190,9.180,9.147,9.116/
      DATA ((EDEN(I,4,8,L,2),I=41,48),L=1,NZEN)/
     * 11.053,11.072,11.097,11.121,11.140,11.152,11.155,11.152,
     * 11.009,11.029,11.057,11.076,11.093,11.104,11.104,11.104,
     * 10.919,10.948,10.986,11.009,11.025,11.037,11.037,11.037,
     * 10.736,10.785,10.843,10.877,10.907,10.920,10.926,10.925,
     * 10.316,10.398,10.490,10.561,10.629,10.665,10.689,10.698,
     * 10.121,10.185,10.258,10.342,10.423,10.476,10.517,10.539,
     * 9.993,10.017,10.049,10.097,10.146,10.190,10.223,10.258,
     * 9.848,9.867,9.900,9.941,9.980,9.997,9.998,9.995,
     * 9.540,9.519,9.513,9.519,9.524,9.507,9.470,9.435,
     * 9.484,9.464,9.458,9.464,9.468,9.452,9.417,9.380,
     * 9.210,9.190,9.188,9.193,9.199,9.185,9.149,9.117,
     * 9.164,9.145,9.142,9.147,9.153,9.139,9.103,9.072/
      DATA ((EDEN(I,5,8,L,2),I=41,48),L=1,NZEN)/
     * 11.068,11.083,11.100,11.114,11.121,11.127,11.127,11.124,
     * 11.029,11.045,11.068,11.079,11.083,11.083,11.079,11.076,
     * 10.941,10.963,10.994,11.004,11.013,11.017,11.013,11.009,
     * 10.763,10.805,10.855,10.877,10.895,10.901,10.901,10.897,
     * 10.340,10.417,10.504,10.563,10.620,10.646,10.665,10.670,
     * 10.121,10.185,10.260,10.336,10.412,10.456,10.493,10.512,
     * 9.966,9.981,10.009,10.061,10.111,10.158,10.196,10.230,
     * 9.822,9.828,9.849,9.888,9.924,9.945,9.952,9.961,
     * 9.537,9.509,9.497,9.496,9.493,9.471,9.436,9.403,
     * 9.481,9.453,9.441,9.441,9.438,9.417,9.380,9.346,
     * 9.204,9.179,9.170,9.170,9.167,9.149,9.117,9.086,
     * 9.158,9.133,9.125,9.125,9.122,9.103,9.072,9.041/
      DATA ((EDEN(I,1,9,L,2),I=41,48),L=1,NZEN)/
     * 11.037,11.076,11.117,11.158,11.199,11.223,11.233,11.236,
     * 10.946,10.993,11.045,11.090,11.134,11.161,11.176,11.182,
     * 10.844,10.903,10.967,11.017,11.064,11.097,11.107,11.117,
     * 10.641,10.722,10.808,10.874,10.940,10.975,10.995,11.004,
     * 10.250,10.338,10.433,10.534,10.634,10.698,10.743,10.766,
     * 10.127,10.179,10.243,10.334,10.427,10.502,10.558,10.595,
     * 10.045,10.076,10.114,10.164,10.215,10.258,10.279,10.314,
     * 9.898,9.930,9.975,10.025,10.076,10.097,10.090,10.083,
     * 9.567,9.562,9.571,9.594,9.619,9.612,9.579,9.542,
     * 9.512,9.507,9.515,9.538,9.563,9.558,9.524,9.486,
     * 9.238,9.233,9.243,9.267,9.294,9.290,9.258,9.220,
     * 9.192,9.187,9.197,9.221,9.249,9.244,9.212,9.174/
      DATA ((EDEN(I,2,9,L,2),I=41,48),L=1,NZEN)/
     * 11.037,11.072,11.114,11.149,11.185,11.204,11.215,11.215,
     * 10.958,11.000,11.049,11.083,11.117,11.143,11.155,11.161,
     * 10.858,10.911,10.969,11.013,11.053,11.076,11.090,11.097,
     * 10.657,10.731,10.812,10.870,10.927,10.957,10.974,10.982,
     * 10.265,10.348,10.439,10.533,10.625,10.683,10.726,10.747,
     * 10.134,10.185,10.246,10.332,10.418,10.489,10.543,10.579,
     * 10.053,10.079,10.114,10.158,10.204,10.243,10.265,10.299,
     * 9.906,9.933,9.974,10.017,10.064,10.079,10.076,10.064,
     * 9.572,9.561,9.565,9.582,9.600,9.590,9.556,9.520,
     * 9.516,9.507,9.509,9.528,9.545,9.535,9.501,9.465,
     * 9.241,9.233,9.238,9.258,9.276,9.267,9.236,9.201,
     * 9.195,9.187,9.192,9.212,9.231,9.221,9.190,9.156/
      DATA ((EDEN(I,3,9,L,2),I=41,48),L=1,NZEN)/
     * 11.037,11.068,11.107,11.137,11.164,11.182,11.190,11.193,
     * 10.976,11.013,11.053,11.079,11.104,11.124,11.134,11.140,
     * 10.878,10.923,10.973,11.009,11.037,11.057,11.068,11.072,
     * 10.682,10.748,10.820,10.869,10.915,10.939,10.953,10.960,
     * 10.274,10.358,10.450,10.537,10.621,10.670,10.709,10.728,
     * 10.127,10.182,10.246,10.330,10.413,10.479,10.531,10.565,
     * 10.033,10.057,10.093,10.140,10.188,10.230,10.258,10.294,
     * 9.889,9.914,9.951,9.996,10.041,10.064,10.068,10.068,
     * 9.563,9.550,9.550,9.562,9.574,9.562,9.530,9.498,
     * 9.509,9.496,9.494,9.507,9.520,9.508,9.476,9.442,
     * 9.236,9.223,9.223,9.236,9.250,9.241,9.210,9.179,
     * 9.190,9.177,9.177,9.190,9.205,9.195,9.164,9.133/
      DATA ((EDEN(I,4,9,L,2),I=41,48),L=1,NZEN)/
     * 11.041,11.072,11.104,11.127,11.146,11.161,11.173,11.176,
     * 10.996,11.025,11.057,11.076,11.093,11.107,11.114,11.121,
     * 10.902,10.938,10.980,11.004,11.025,11.037,11.049,11.053,
     * 10.712,10.769,10.831,10.869,10.904,10.921,10.932,10.939,
     * 10.290,10.373,10.464,10.538,10.611,10.652,10.687,10.703,
     * 10.114,10.173,10.238,10.318,10.396,10.453,10.504,10.535,
     * 10.000,10.021,10.053,10.093,10.134,10.173,10.204,10.241,
     * 9.860,9.877,9.906,9.940,9.973,9.985,9.981,9.978,
     * 9.555,9.537,9.530,9.535,9.539,9.522,9.489,9.456,
     * 9.500,9.481,9.474,9.480,9.484,9.467,9.433,9.400,
     * 9.225,9.207,9.204,9.210,9.212,9.199,9.167,9.134,
     * 9.180,9.161,9.158,9.164,9.166,9.153,9.122,9.088/
      DATA ((EDEN(I,5,9,L,2),I=41,48),L=1,NZEN)/
     * 11.061,11.083,11.111,11.124,11.137,11.146,11.152,11.155,
     * 11.013,11.037,11.064,11.072,11.079,11.086,11.090,11.097,
     * 10.922,10.952,10.986,11.000,11.013,11.017,11.025,11.029,
     * 10.738,10.786,10.840,10.866,10.890,10.901,10.909,10.914,
     * 10.312,10.389,10.474,10.539,10.600,10.634,10.664,10.679,
     * 10.114,10.170,10.236,10.310,10.382,10.435,10.480,10.511,
     * 9.981,9.992,10.013,10.053,10.097,10.137,10.173,10.215,
     * 9.836,9.842,9.860,9.891,9.920,9.936,9.940,9.944,
     * 9.548,9.525,9.515,9.512,9.508,9.489,9.456,9.427,
     * 9.493,9.470,9.458,9.456,9.453,9.433,9.401,9.371,
     * 9.217,9.196,9.185,9.185,9.182,9.164,9.134,9.104,
     * 9.172,9.150,9.139,9.139,9.136,9.119,9.088,9.058/
      DATA ((EDEN(I,1,10,L,2),I=41,48),L=1,NZEN)/
     * 11.029,11.068,11.117,11.158,11.201,11.225,11.236,11.241,
     * 10.936,10.988,11.045,11.090,11.137,11.164,11.179,11.188,
     * 10.830,10.893,10.961,11.013,11.064,11.097,11.111,11.117,
     * 10.620,10.704,10.796,10.866,10.937,10.975,10.996,11.004,
     * 10.236,10.320,10.413,10.517,10.620,10.688,10.737,10.763,
     * 10.127,10.176,10.233,10.322,10.413,10.490,10.547,10.588,
     * 10.049,10.079,10.121,10.167,10.215,10.255,10.272,10.305,
     * 9.901,9.933,9.978,10.029,10.079,10.100,10.093,10.083,
     * 9.583,9.576,9.584,9.610,9.635,9.628,9.594,9.556,
     * 9.528,9.522,9.530,9.555,9.581,9.574,9.540,9.501,
     * 9.253,9.248,9.258,9.283,9.312,9.305,9.272,9.236,
     * 9.207,9.202,9.212,9.238,9.266,9.260,9.226,9.190/
      DATA ((EDEN(I,2,10,L,2),I=41,48),L=1,NZEN)/
     * 11.037,11.076,11.117,11.152,11.185,11.207,11.220,11.223,
     * 10.945,10.993,11.045,11.083,11.117,11.146,11.161,11.173,
     * 10.839,10.898,10.962,11.009,11.053,11.079,11.093,11.104,
     * 10.629,10.711,10.797,10.860,10.922,10.957,10.979,10.990,
     * 10.246,10.326,10.415,10.511,10.607,10.670,10.719,10.747,
     * 10.134,10.179,10.233,10.316,10.398,10.471,10.529,10.573,
     * 10.061,10.086,10.121,10.161,10.201,10.236,10.255,10.290,
     * 9.910,9.938,9.978,10.021,10.064,10.083,10.076,10.068,
     * 9.589,9.581,9.584,9.601,9.618,9.609,9.576,9.540,
     * 9.534,9.525,9.529,9.547,9.565,9.554,9.521,9.486,
     * 9.258,9.250,9.255,9.274,9.292,9.286,9.253,9.220,
     * 9.212,9.205,9.210,9.228,9.246,9.240,9.207,9.174/
      DATA ((EDEN(I,3,10,L,2),I=41,48),L=1,NZEN)/
     * 11.033,11.068,11.111,11.140,11.164,11.185,11.199,11.207,
     * 10.962,11.004,11.045,11.076,11.104,11.127,11.140,11.152,
     * 10.860,10.912,10.967,11.004,11.037,11.057,11.076,11.086,
     * 10.653,10.728,10.805,10.857,10.908,10.938,10.958,10.972,
     * 10.253,10.334,10.423,10.511,10.599,10.657,10.704,10.732,
     * 10.124,10.173,10.230,10.310,10.391,10.459,10.517,10.561,
     * 10.041,10.064,10.097,10.140,10.182,10.223,10.250,10.290,
     * 9.897,9.922,9.956,9.999,10.041,10.064,10.068,10.072,
     * 9.584,9.573,9.572,9.582,9.594,9.582,9.553,9.524,
     * 9.529,9.517,9.516,9.528,9.539,9.528,9.498,9.468,
     * 9.255,9.243,9.243,9.255,9.270,9.260,9.230,9.204,
     * 9.210,9.197,9.197,9.210,9.224,9.214,9.185,9.158/
      DATA ((EDEN(I,4,10,L,2),I=41,48),L=1,NZEN)/
     * 11.029,11.068,11.107,11.130,11.149,11.170,11.190,11.201,
     * 10.983,11.017,11.053,11.076,11.093,11.111,11.124,11.140,
     * 10.884,10.928,10.974,11.000,11.025,11.041,11.057,11.072,
     * 10.685,10.750,10.817,10.859,10.899,10.922,10.942,10.957,
     * 10.267,10.348,10.435,10.512,10.589,10.638,10.683,10.712,
     * 10.114,10.164,10.217,10.292,10.367,10.431,10.490,10.534,
     * 10.017,10.033,10.057,10.086,10.117,10.155,10.185,10.228,
     * 9.873,9.890,9.916,9.941,9.966,9.975,9.972,9.968,
     * 9.574,9.559,9.553,9.555,9.558,9.542,9.512,9.483,
     * 9.517,9.502,9.497,9.500,9.502,9.487,9.456,9.427,
     * 9.243,9.228,9.225,9.228,9.233,9.217,9.188,9.161,
     * 9.197,9.182,9.180,9.182,9.187,9.172,9.142,9.116/
      DATA ((EDEN(I,5,10,L,2),I=41,48),L=1,NZEN)/
     * 11.041,11.076,11.114,11.130,11.143,11.158,11.176,11.190,
     * 11.000,11.029,11.057,11.068,11.079,11.093,11.107,11.124,
     * 10.906,10.943,10.980,10.996,11.009,11.021,11.037,11.057,
     * 10.713,10.769,10.828,10.857,10.884,10.901,10.920,10.938,
     * 10.286,10.364,10.447,10.513,10.579,10.621,10.664,10.693,
     * 10.107,10.158,10.212,10.283,10.354,10.413,10.471,10.515,
     * 9.993,10.004,10.017,10.049,10.079,10.121,10.155,10.201,
     * 9.849,9.855,9.867,9.892,9.916,9.926,9.928,9.932,
     * 9.565,9.545,9.535,9.530,9.526,9.509,9.481,9.459,
     * 9.509,9.490,9.479,9.474,9.471,9.453,9.425,9.401,
     * 9.217,9.207,9.207,9.217,9.230,9.220,9.190,9.164,
     * 9.172,9.161,9.161,9.172,9.185,9.174,9.145,9.119/
      DATA ((EDEN(I,1,11,L,2),I=41,48),L=1,NZEN)/
     * 11.025,11.068,11.117,11.161,11.204,11.230,11.241,11.243,
     * 10.929,10.982,11.041,11.090,11.137,11.167,11.182,11.190,
     * 10.822,10.887,10.959,11.013,11.068,11.100,11.114,11.121,
     * 10.607,10.695,10.790,10.862,10.935,10.975,10.998,11.009,
     * 10.230,10.314,10.403,10.508,10.614,10.684,10.735,10.762,
     * 10.127,10.173,10.230,10.318,10.403,10.483,10.542,10.584,
     * 10.057,10.083,10.121,10.170,10.217,10.255,10.270,10.301,
     * 9.905,9.936,9.980,10.029,10.083,10.100,10.097,10.086,
     * 9.591,9.587,9.595,9.620,9.646,9.639,9.605,9.566,
     * 9.535,9.531,9.540,9.565,9.591,9.584,9.549,9.511,
     * 9.262,9.258,9.270,9.294,9.322,9.316,9.281,9.246,
     * 9.217,9.212,9.224,9.249,9.276,9.270,9.235,9.200/
      DATA ((EDEN(I,2,11,L,2),I=41,48),L=1,NZEN)/
     * 11.041,11.079,11.117,11.152,11.188,11.210,11.223,11.230,
     * 10.937,10.988,11.041,11.083,11.124,11.149,11.164,11.179,
     * 10.827,10.891,10.958,11.004,11.053,11.079,11.097,11.111,
     * 10.614,10.698,10.787,10.854,10.920,10.957,10.983,10.997,
     * 10.238,10.318,10.401,10.498,10.595,10.663,10.717,10.750,
     * 10.134,10.179,10.230,10.310,10.387,10.462,10.522,10.571,
     * 10.064,10.090,10.124,10.161,10.201,10.236,10.255,10.290,
     * 9.913,9.943,9.981,10.025,10.068,10.083,10.079,10.072,
     * 9.601,9.593,9.595,9.613,9.631,9.621,9.590,9.556,
     * 9.544,9.537,9.542,9.558,9.576,9.566,9.534,9.500,
     * 9.270,9.262,9.267,9.286,9.305,9.297,9.267,9.233,
     * 9.224,9.217,9.221,9.240,9.260,9.251,9.221,9.187/
      DATA ((EDEN(I,3,11,L,2),I=41,48),L=1,NZEN)/
     * 11.037,11.076,11.117,11.143,11.167,11.190,11.207,11.220,
     * 10.954,11.000,11.049,11.076,11.104,11.130,11.149,11.167,
     * 10.848,10.905,10.963,11.000,11.037,11.061,11.083,11.097,
     * 10.635,10.715,10.797,10.851,10.906,10.939,10.965,10.984,
     * 10.243,10.324,10.410,10.498,10.587,10.649,10.703,10.739,
     * 10.127,10.173,10.225,10.301,10.377,10.449,10.512,10.563,
     * 10.049,10.076,10.104,10.143,10.182,10.220,10.248,10.290,
     * 9.901,9.929,9.964,10.004,10.045,10.068,10.072,10.079,
     * 9.597,9.588,9.587,9.595,9.606,9.597,9.569,9.543,
     * 9.542,9.531,9.531,9.540,9.551,9.540,9.513,9.487,
     * 9.267,9.258,9.258,9.270,9.281,9.272,9.248,9.223,
     * 9.221,9.212,9.212,9.224,9.235,9.226,9.202,9.177/
      DATA ((EDEN(I,4,11,L,2),I=41,48),L=1,NZEN)/
     * 11.025,11.068,11.111,11.134,11.155,11.176,11.199,11.220,
     * 10.977,11.017,11.053,11.072,11.093,11.114,11.134,11.158,
     * 10.876,10.925,10.973,10.999,11.025,11.045,11.068,11.090,
     * 10.671,10.741,10.810,10.852,10.894,10.923,10.950,10.974,
     * 10.258,10.338,10.422,10.498,10.576,10.632,10.685,10.723,
     * 10.114,10.161,10.212,10.283,10.352,10.420,10.484,10.539,
     * 10.025,10.045,10.064,10.090,10.114,10.146,10.176,10.225,
     * 9.881,9.899,9.922,9.943,9.966,9.970,9.965,9.964,
     * 9.584,9.572,9.566,9.567,9.568,9.554,9.528,9.507,
     * 9.529,9.517,9.511,9.511,9.513,9.500,9.473,9.450,
     * 9.253,9.243,9.238,9.238,9.241,9.228,9.204,9.182,
     * 9.207,9.197,9.192,9.192,9.195,9.182,9.158,9.136/
      DATA ((EDEN(I,5,11,L,2),I=41,48),L=1,NZEN)/
     * 11.029,11.072,11.111,11.127,11.143,11.164,11.190,11.215,
     * 11.000,11.033,11.061,11.072,11.079,11.097,11.121,11.146,
     * 10.903,10.944,10.982,10.997,11.009,11.029,11.049,11.079,
     * 10.704,10.766,10.824,10.854,10.883,10.905,10.931,10.960,
     * 10.279,10.358,10.438,10.502,10.567,10.616,10.667,10.708,
     * 10.111,10.158,10.207,10.272,10.338,10.401,10.467,10.524,
     * 10.000,10.013,10.021,10.045,10.072,10.107,10.146,10.201,
     * 9.859,9.866,9.876,9.892,9.909,9.918,9.922,9.930,
     * 9.575,9.560,9.549,9.542,9.535,9.520,9.498,9.484,
     * 9.519,9.504,9.491,9.486,9.481,9.464,9.442,9.427,
     * 9.243,9.228,9.217,9.212,9.210,9.193,9.173,9.158,
     * 9.197,9.182,9.172,9.166,9.164,9.147,9.127,9.113/
      DATA ((EDEN(I,1,12,L,2),I=41,48),L=1,NZEN)/
     * 11.025,11.068,11.121,11.164,11.210,11.236,11.248,11.248,
     * 10.936,10.988,11.045,11.093,11.140,11.170,11.182,11.190,
     * 10.827,10.892,10.963,11.017,11.072,11.100,11.117,11.124,
     * 10.614,10.701,10.794,10.867,10.940,10.979,11.000,11.009,
     * 10.236,10.318,10.410,10.515,10.618,10.688,10.739,10.766,
     * 10.127,10.179,10.236,10.322,10.410,10.487,10.545,10.588,
     * 10.057,10.086,10.124,10.173,10.220,10.258,10.274,10.307,
     * 9.908,9.940,9.984,10.033,10.083,10.104,10.100,10.086,
     * 9.593,9.588,9.597,9.620,9.645,9.638,9.605,9.567,
     * 9.538,9.533,9.542,9.566,9.591,9.584,9.549,9.511,
     * 9.262,9.258,9.270,9.294,9.320,9.314,9.281,9.246,
     * 9.217,9.212,9.224,9.249,9.274,9.268,9.235,9.200/
      DATA ((EDEN(I,2,12,L,2),I=41,48),L=1,NZEN)/
     * 11.045,11.083,11.124,11.155,11.190,11.212,11.228,11.236,
     * 10.941,10.993,11.045,11.086,11.127,11.152,11.170,11.185,
     * 10.832,10.895,10.962,11.009,11.053,11.083,11.104,11.117,
     * 10.619,10.704,10.792,10.857,10.922,10.960,10.986,11.004,
     * 10.243,10.322,10.407,10.502,10.599,10.667,10.721,10.756,
     * 10.137,10.185,10.236,10.312,10.389,10.465,10.528,10.577,
     * 10.064,10.090,10.124,10.164,10.204,10.238,10.258,10.297,
     * 9.916,9.944,9.982,10.025,10.068,10.086,10.083,10.079,
     * 9.602,9.595,9.599,9.614,9.631,9.622,9.591,9.560,
     * 9.545,9.539,9.543,9.559,9.576,9.567,9.537,9.504,
     * 9.272,9.265,9.270,9.288,9.305,9.297,9.270,9.238,
     * 9.226,9.219,9.224,9.242,9.260,9.251,9.224,9.192/
      DATA ((EDEN(I,3,12,L,2),I=41,48),L=1,NZEN)/
     * 11.041,11.083,11.121,11.146,11.173,11.193,11.210,11.225,
     * 10.958,11.004,11.049,11.079,11.107,11.130,11.152,11.176,
     * 10.852,10.910,10.967,11.004,11.037,11.064,11.086,11.107,
     * 10.642,10.721,10.801,10.854,10.907,10.941,10.969,10.992,
     * 10.248,10.330,10.415,10.502,10.589,10.652,10.708,10.747,
     * 10.130,10.179,10.230,10.305,10.380,10.453,10.519,10.573,
     * 10.053,10.079,10.111,10.146,10.185,10.223,10.255,10.301,
     * 9.905,9.932,9.965,10.004,10.045,10.068,10.079,10.086,
     * 9.598,9.589,9.588,9.597,9.606,9.598,9.573,9.550,
     * 9.544,9.535,9.533,9.542,9.551,9.543,9.517,9.494,
     * 9.267,9.260,9.260,9.270,9.281,9.274,9.250,9.228,
     * 9.221,9.214,9.214,9.224,9.235,9.228,9.205,9.182/
      DATA ((EDEN(I,4,12,L,2),I=41,48),L=1,NZEN)/
     * 11.025,11.072,11.111,11.134,11.158,11.182,11.207,11.233,
     * 10.983,11.021,11.061,11.079,11.097,11.117,11.143,11.170,
     * 10.883,10.931,10.978,11.000,11.025,11.049,11.076,11.104,
     * 10.679,10.749,10.816,10.857,10.897,10.926,10.956,10.985,
     * 10.267,10.348,10.430,10.504,10.580,10.636,10.691,10.735,
     * 10.121,10.170,10.220,10.288,10.354,10.425,10.491,10.550,
     * 10.029,10.049,10.068,10.090,10.114,10.149,10.179,10.236,
     * 9.885,9.903,9.924,9.943,9.962,9.970,9.969,9.970,
     * 9.584,9.574,9.567,9.567,9.568,9.555,9.531,9.515,
     * 9.529,9.519,9.512,9.511,9.513,9.498,9.476,9.458,
     * 9.255,9.243,9.238,9.238,9.241,9.228,9.207,9.190,
     * 9.210,9.197,9.192,9.192,9.195,9.182,9.161,9.145/
      DATA ((EDEN(I,5,12,L,2),I=41,48),L=1,NZEN)/
     * 11.029,11.076,11.114,11.127,11.143,11.167,11.196,11.230,
     * 11.009,11.037,11.064,11.072,11.083,11.100,11.124,11.161,
     * 10.911,10.953,10.989,11.000,11.013,11.033,11.057,11.093,
     * 10.715,10.777,10.833,10.859,10.885,10.910,10.940,10.975,
     * 10.292,10.371,10.447,10.509,10.572,10.621,10.675,10.723,
     * 10.117,10.167,10.215,10.279,10.342,10.407,10.474,10.538,
     * 10.004,10.017,10.029,10.049,10.072,10.111,10.149,10.215,
     * 9.864,9.871,9.878,9.890,9.905,9.914,9.921,9.936,
     * 9.574,9.561,9.549,9.540,9.533,9.520,9.502,9.496,
     * 9.519,9.505,9.493,9.484,9.477,9.464,9.446,9.436,
     * 9.243,9.230,9.220,9.212,9.207,9.193,9.176,9.170,
     * 9.197,9.185,9.174,9.166,9.161,9.147,9.130,9.125/
      DATA ((EDEN(I,1,1,L,3),I=41,48),L=1,NZEN)/
     * 11.093,11.143,11.199,11.246,11.294,11.322,11.336,11.340,
     * 10.978,11.037,11.104,11.158,11.210,11.243,11.262,11.274,
     * 10.860,10.930,11.004,11.068,11.130,11.167,11.185,11.196,
     * 10.640,10.728,10.823,10.903,10.982,11.025,11.053,11.064,
     * 10.297,10.367,10.449,10.550,10.650,10.719,10.769,10.798,
     * 10.201,10.248,10.303,10.384,10.462,10.531,10.580,10.620,
     * 10.117,10.149,10.196,10.250,10.305,10.338,10.348,10.371,
     * 9.912,9.951,10.009,10.072,10.137,10.161,10.155,10.146,
     * 9.491,9.491,9.509,9.549,9.589,9.583,9.548,9.513,
     * 9.433,9.433,9.450,9.491,9.533,9.528,9.491,9.456,
     * 9.158,9.161,9.179,9.223,9.265,9.260,9.225,9.193,
     * 9.113,9.116,9.133,9.177,9.219,9.214,9.180,9.147/
      DATA ((EDEN(I,2,1,L,3),I=41,48),L=1,NZEN)/
     * 11.114,11.158,11.201,11.241,11.276,11.301,11.318,11.330,
     * 10.982,11.041,11.100,11.146,11.193,11.225,11.250,11.270,
     * 10.865,10.933,11.004,11.057,11.114,11.146,11.173,11.190,
     * 10.642,10.728,10.819,10.891,10.963,11.009,11.037,11.061,
     * 10.303,10.371,10.446,10.538,10.629,10.697,10.751,10.788,
     * 10.212,10.253,10.303,10.373,10.444,10.511,10.562,10.610,
     * 10.127,10.155,10.196,10.243,10.290,10.318,10.330,10.360,
     * 9.920,9.957,10.009,10.064,10.121,10.143,10.140,10.140,
     * 9.501,9.498,9.512,9.543,9.575,9.567,9.537,9.508,
     * 9.444,9.441,9.453,9.486,9.519,9.511,9.480,9.452,
     * 9.167,9.167,9.182,9.215,9.250,9.243,9.212,9.188,
     * 9.122,9.122,9.136,9.169,9.205,9.197,9.166,9.142/
      DATA ((EDEN(I,3,1,L,3),I=41,48),L=1,NZEN)/
     * 11.114,11.155,11.196,11.225,11.253,11.279,11.301,11.320,
     * 10.999,11.053,11.104,11.140,11.176,11.207,11.233,11.260,
     * 10.881,10.945,11.009,11.053,11.097,11.130,11.158,11.185,
     * 10.661,10.744,10.827,10.888,10.949,10.989,11.025,11.049,
     * 10.301,10.373,10.446,10.531,10.618,10.683,10.740,10.783,
     * 10.199,10.243,10.292,10.360,10.430,10.496,10.553,10.607,
     * 10.111,10.143,10.179,10.223,10.270,10.303,10.328,10.367,
     * 9.907,9.943,9.989,10.045,10.100,10.127,10.134,10.149,
     * 9.496,9.493,9.501,9.525,9.551,9.544,9.520,9.504,
     * 9.439,9.436,9.444,9.470,9.496,9.489,9.464,9.446,
     * 9.164,9.164,9.173,9.199,9.228,9.220,9.199,9.182,
     * 9.119,9.119,9.127,9.153,9.182,9.174,9.153,9.136/
      DATA ((EDEN(I,4,1,L,3),I=41,48),L=1,NZEN)/
     * 11.097,11.143,11.185,11.210,11.236,11.262,11.292,11.322,
     * 11.025,11.072,11.114,11.140,11.164,11.193,11.223,11.258,
     * 10.911,10.968,11.021,11.053,11.086,11.117,11.146,11.179,
     * 10.692,10.767,10.840,10.888,10.937,10.972,11.009,11.041,
     * 10.301,10.375,10.447,10.522,10.599,10.657,10.714,10.763,
     * 10.179,10.223,10.265,10.326,10.387,10.452,10.512,10.573,
     * 10.083,10.107,10.134,10.161,10.188,10.215,10.238,10.283,
     * 9.885,9.913,9.947,9.979,10.013,10.021,10.021,10.029,
     * 9.483,9.477,9.480,9.494,9.509,9.496,9.471,9.458,
     * 9.427,9.420,9.423,9.436,9.452,9.439,9.415,9.400,
     * 9.152,9.146,9.152,9.164,9.182,9.170,9.146,9.134,
     * 9.107,9.100,9.107,9.119,9.136,9.125,9.100,9.088/
      DATA ((EDEN(I,5,1,L,3),I=41,48),L=1,NZEN)/
     * 11.097,11.146,11.185,11.201,11.220,11.246,11.281,11.318,
     * 11.049,11.090,11.127,11.140,11.155,11.179,11.210,11.248,
     * 10.938,10.989,11.033,11.053,11.072,11.097,11.130,11.170,
     * 10.724,10.792,10.854,10.890,10.925,10.956,10.992,11.033,
     * 10.312,10.384,10.452,10.519,10.585,10.639,10.697,10.752,
     * 10.164,10.207,10.246,10.303,10.364,10.427,10.493,10.559,
     * 10.057,10.072,10.083,10.111,10.137,10.167,10.201,10.260,
     * 9.860,9.877,9.897,9.926,9.956,9.968,9.975,9.996,
     * 9.473,9.464,9.459,9.465,9.474,9.461,9.441,9.439,
     * 9.417,9.408,9.403,9.410,9.417,9.403,9.384,9.380,
     * 9.140,9.134,9.130,9.137,9.146,9.134,9.114,9.111,
     * 9.094,9.088,9.085,9.091,9.100,9.088,9.068,9.065/
      DATA ((EDEN(I,1,2,L,3),I=41,48),L=1,NZEN)/
     * 11.093,11.143,11.196,11.243,11.290,11.320,11.332,11.338,
     * 10.985,11.041,11.104,11.158,11.210,11.243,11.262,11.274,
     * 10.870,10.938,11.013,11.072,11.130,11.167,11.185,11.196,
     * 10.652,10.738,10.831,10.908,10.984,11.025,11.053,11.064,
     * 10.301,10.375,10.458,10.558,10.658,10.724,10.772,10.799,
     * 10.201,10.248,10.307,10.387,10.468,10.537,10.584,10.623,
     * 10.117,10.149,10.193,10.250,10.305,10.338,10.348,10.373,
     * 9.912,9.951,10.009,10.072,10.137,10.158,10.152,10.146,
     * 9.483,9.481,9.500,9.539,9.580,9.574,9.539,9.505,
     * 9.425,9.425,9.442,9.483,9.524,9.519,9.483,9.449,
     * 9.152,9.152,9.170,9.212,9.255,9.250,9.217,9.185,
     * 9.107,9.107,9.125,9.166,9.210,9.205,9.172,9.139/
      DATA ((EDEN(I,2,2,L,3),I=41,48),L=1,NZEN)/
     * 11.107,11.149,11.196,11.236,11.274,11.299,11.316,11.328,
     * 10.994,11.049,11.104,11.149,11.196,11.228,11.248,11.265,
     * 10.877,10.943,11.013,11.061,11.114,11.146,11.170,11.190,
     * 10.660,10.743,10.831,10.900,10.968,11.009,11.037,11.057,
     * 10.310,10.380,10.458,10.549,10.640,10.705,10.755,10.789,
     * 10.212,10.255,10.307,10.380,10.452,10.519,10.568,10.613,
     * 10.124,10.155,10.196,10.243,10.290,10.320,10.334,10.362,
     * 9.920,9.957,10.009,10.064,10.121,10.140,10.137,10.137,
     * 9.491,9.489,9.500,9.533,9.566,9.558,9.525,9.498,
     * 9.433,9.431,9.442,9.476,9.508,9.501,9.468,9.441,
     * 9.158,9.158,9.173,9.207,9.238,9.233,9.204,9.176,
     * 9.113,9.113,9.127,9.161,9.192,9.187,9.158,9.130/
      DATA ((EDEN(I,3,2,L,3),I=41,48),L=1,NZEN)/
     * 11.104,11.149,11.190,11.220,11.250,11.276,11.297,11.318,
     * 11.009,11.061,11.111,11.146,11.182,11.210,11.233,11.255,
     * 10.894,10.955,11.017,11.057,11.097,11.127,11.155,11.179,
     * 10.678,10.757,10.838,10.896,10.954,10.991,11.021,11.045,
     * 10.305,10.380,10.458,10.544,10.629,10.691,10.744,10.783,
     * 10.196,10.243,10.294,10.367,10.439,10.504,10.560,10.611,
     * 10.104,10.137,10.173,10.223,10.270,10.305,10.330,10.369,
     * 9.904,9.942,9.989,10.045,10.100,10.127,10.134,10.149,
     * 9.486,9.481,9.490,9.515,9.540,9.534,9.509,9.491,
     * 9.428,9.423,9.431,9.458,9.484,9.477,9.452,9.435,
     * 9.152,9.149,9.161,9.188,9.215,9.210,9.188,9.173,
     * 9.107,9.103,9.116,9.142,9.169,9.164,9.142,9.127/
      DATA ((EDEN(I,4,2,L,3),I=41,48),L=1,NZEN)/
     * 11.097,11.143,11.185,11.207,11.233,11.258,11.286,11.314,
     * 11.033,11.076,11.114,11.140,11.167,11.193,11.223,11.250,
     * 10.920,10.974,11.025,11.057,11.086,11.114,11.140,11.170,
     * 10.705,10.777,10.848,10.895,10.941,10.974,11.004,11.037,
     * 10.307,10.382,10.458,10.534,10.611,10.667,10.719,10.763,
     * 10.176,10.223,10.270,10.334,10.400,10.464,10.522,10.577,
     * 10.072,10.100,10.130,10.161,10.193,10.225,10.250,10.294,
     * 9.876,9.906,9.943,9.981,10.021,10.033,10.033,10.041,
     * 9.471,9.464,9.467,9.483,9.500,9.487,9.462,9.447,
     * 9.415,9.408,9.410,9.427,9.442,9.430,9.405,9.389,
     * 9.140,9.134,9.137,9.155,9.173,9.161,9.137,9.124,
     * 9.094,9.088,9.091,9.110,9.127,9.116,9.091,9.078/
      DATA ((EDEN(I,5,2,L,3),I=41,48),L=1,NZEN)/
     * 11.100,11.146,11.185,11.204,11.220,11.246,11.279,11.310,
     * 11.053,11.093,11.124,11.140,11.155,11.176,11.204,11.241,
     * 10.944,10.990,11.033,11.053,11.072,11.097,11.127,11.161,
     * 10.732,10.798,10.860,10.894,10.928,10.956,10.989,11.025,
     * 10.314,10.389,10.461,10.529,10.597,10.647,10.701,10.750,
     * 10.158,10.204,10.250,10.314,10.377,10.438,10.501,10.562,
     * 10.045,10.064,10.083,10.114,10.146,10.182,10.215,10.272,
     * 9.854,9.871,9.893,9.928,9.964,9.980,9.991,10.013,
     * 9.462,9.453,9.449,9.456,9.467,9.453,9.433,9.430,
     * 9.407,9.396,9.393,9.400,9.410,9.396,9.377,9.371,
     * 9.130,9.121,9.117,9.127,9.140,9.127,9.111,9.104,
     * 9.085,9.075,9.072,9.081,9.094,9.081,9.065,9.058/
      DATA ((EDEN(I,1,3,L,3),I=41,48),L=1,NZEN)/
     * 11.093,11.143,11.196,11.241,11.286,11.316,11.330,11.336,
     * 10.985,11.041,11.104,11.155,11.207,11.241,11.260,11.270,
     * 10.872,10.938,11.009,11.068,11.127,11.161,11.182,11.193,
     * 10.654,10.739,10.831,10.907,10.982,11.025,11.049,11.061,
     * 10.301,10.375,10.459,10.559,10.658,10.723,10.770,10.797,
     * 10.201,10.248,10.305,10.387,10.468,10.537,10.584,10.622,
     * 10.111,10.146,10.193,10.248,10.303,10.336,10.348,10.371,
     * 9.909,9.949,10.004,10.068,10.134,10.155,10.152,10.143,
     * 9.477,9.477,9.494,9.534,9.574,9.568,9.534,9.501,
     * 9.420,9.420,9.438,9.479,9.517,9.513,9.479,9.444,
     * 9.146,9.146,9.167,9.207,9.250,9.246,9.212,9.182,
     * 9.100,9.100,9.122,9.161,9.205,9.200,9.166,9.136/
      DATA ((EDEN(I,2,3,L,3),I=41,48),L=1,NZEN)/
     * 11.100,11.143,11.193,11.230,11.267,11.294,11.312,11.322,
     * 10.994,11.049,11.104,11.149,11.193,11.223,11.243,11.260,
     * 10.881,10.944,11.013,11.061,11.114,11.146,11.167,11.182,
     * 10.666,10.747,10.833,10.901,10.968,11.009,11.033,11.049,
     * 10.310,10.382,10.462,10.554,10.645,10.708,10.755,10.785,
     * 10.210,10.253,10.305,10.382,10.456,10.521,10.571,10.612,
     * 10.121,10.152,10.193,10.241,10.288,10.318,10.332,10.360,
     * 9.918,9.955,10.004,10.064,10.121,10.140,10.137,10.134,
     * 9.483,9.480,9.493,9.525,9.558,9.550,9.517,9.489,
     * 9.425,9.422,9.436,9.468,9.501,9.493,9.461,9.431,
     * 9.152,9.149,9.164,9.199,9.233,9.228,9.196,9.167,
     * 9.107,9.103,9.119,9.153,9.187,9.182,9.150,9.122/
      DATA ((EDEN(I,3,3,L,3),I=41,48),L=1,NZEN)/
     * 11.097,11.140,11.185,11.215,11.248,11.272,11.292,11.307,
     * 11.017,11.061,11.111,11.143,11.176,11.204,11.225,11.246,
     * 10.900,10.959,11.017,11.057,11.097,11.127,11.152,11.170,
     * 10.686,10.763,10.843,10.899,10.955,10.990,11.017,11.037,
     * 10.307,10.384,10.464,10.550,10.636,10.695,10.743,10.777,
     * 10.190,10.241,10.294,10.369,10.444,10.509,10.562,10.607,
     * 10.097,10.127,10.170,10.217,10.267,10.305,10.330,10.365,
     * 9.896,9.933,9.983,10.041,10.100,10.124,10.130,10.143,
     * 9.476,9.470,9.479,9.505,9.533,9.525,9.498,9.479,
     * 9.418,9.413,9.423,9.449,9.476,9.468,9.442,9.422,
     * 9.146,9.140,9.152,9.179,9.210,9.204,9.179,9.161,
     * 9.100,9.094,9.107,9.133,9.164,9.158,9.133,9.116/
      DATA ((EDEN(I,4,3,L,3),I=41,48),L=1,NZEN)/
     * 11.100,11.140,11.182,11.207,11.233,11.255,11.279,11.299,
     * 11.033,11.076,11.117,11.143,11.167,11.188,11.210,11.233,
     * 10.924,10.974,11.025,11.057,11.086,11.111,11.137,11.158,
     * 10.711,10.781,10.851,10.898,10.943,10.973,11.000,11.025,
     * 10.305,10.384,10.464,10.542,10.619,10.671,10.719,10.755,
     * 10.164,10.215,10.270,10.340,10.412,10.473,10.528,10.576,
     * 10.061,10.090,10.124,10.164,10.201,10.236,10.260,10.303,
     * 9.865,9.895,9.937,9.981,10.025,10.045,10.049,10.057,
     * 9.465,9.456,9.459,9.477,9.494,9.481,9.455,9.438,
     * 9.407,9.400,9.403,9.420,9.438,9.425,9.400,9.382,
     * 9.134,9.127,9.130,9.149,9.167,9.158,9.134,9.117,
     * 9.088,9.081,9.085,9.103,9.122,9.113,9.088,9.072/
      DATA ((EDEN(I,5,3,L,3),I=41,48),L=1,NZEN)/
     * 11.111,11.152,11.188,11.207,11.223,11.243,11.267,11.292,
     * 11.053,11.090,11.121,11.137,11.152,11.170,11.196,11.223,
     * 10.944,10.990,11.033,11.053,11.072,11.093,11.117,11.143,
     * 10.735,10.798,10.860,10.895,10.928,10.953,10.981,11.009,
     * 10.312,10.387,10.464,10.534,10.603,10.650,10.699,10.739,
     * 10.152,10.199,10.248,10.318,10.387,10.447,10.505,10.558,
     * 10.033,10.053,10.076,10.114,10.155,10.193,10.230,10.281,
     * 9.841,9.860,9.887,9.929,9.971,9.993,10.004,10.029,
     * 9.455,9.444,9.441,9.452,9.462,9.449,9.427,9.418,
     * 9.398,9.386,9.384,9.394,9.405,9.393,9.369,9.360,
     * 9.124,9.111,9.111,9.124,9.137,9.124,9.104,9.097,
     * 9.078,9.065,9.065,9.078,9.091,9.078,9.058,9.051/
      DATA ((EDEN(I,1,4,L,3),I=41,48),L=1,NZEN)/
     * 11.097,11.143,11.196,11.241,11.286,11.314,11.328,11.330,
     * 10.981,11.037,11.100,11.152,11.204,11.238,11.255,11.267,
     * 10.867,10.933,11.004,11.064,11.124,11.158,11.179,11.190,
     * 10.650,10.735,10.827,10.904,10.980,11.021,11.045,11.057,
     * 10.297,10.371,10.455,10.555,10.655,10.721,10.768,10.794,
     * 10.196,10.243,10.301,10.384,10.465,10.534,10.581,10.619,
     * 10.107,10.140,10.188,10.243,10.299,10.332,10.344,10.367,
     * 9.904,9.944,10.000,10.064,10.130,10.152,10.146,10.140,
     * 9.471,9.471,9.490,9.530,9.572,9.566,9.530,9.497,
     * 9.415,9.415,9.433,9.474,9.515,9.509,9.474,9.439,
     * 9.140,9.143,9.161,9.204,9.246,9.243,9.210,9.176,
     * 9.094,9.097,9.116,9.158,9.200,9.197,9.164,9.130/
      DATA ((EDEN(I,2,4,L,3),I=41,48),L=1,NZEN)/
     * 11.097,11.140,11.188,11.228,11.267,11.292,11.307,11.314,
     * 10.994,11.045,11.104,11.146,11.188,11.220,11.238,11.250,
     * 10.883,10.944,11.009,11.061,11.114,11.143,11.161,11.173,
     * 10.668,10.748,10.834,10.901,10.968,11.004,11.029,11.041,
     * 10.310,10.382,10.462,10.555,10.647,10.708,10.753,10.780,
     * 10.204,10.250,10.305,10.382,10.458,10.521,10.569,10.607,
     * 10.117,10.146,10.188,10.238,10.286,10.318,10.330,10.356,
     * 9.915,9.950,10.000,10.061,10.117,10.137,10.134,10.127,
     * 9.474,9.470,9.484,9.517,9.550,9.543,9.509,9.479,
     * 9.417,9.415,9.428,9.462,9.496,9.487,9.453,9.422,
     * 9.143,9.143,9.158,9.193,9.228,9.220,9.188,9.158,
     * 9.097,9.097,9.113,9.147,9.182,9.174,9.142,9.113/
      DATA ((EDEN(I,3,4,L,3),I=41,48),L=1,NZEN)/
     * 11.097,11.137,11.182,11.215,11.243,11.267,11.283,11.294,
     * 11.013,11.061,11.107,11.143,11.176,11.201,11.217,11.230,
     * 10.904,10.959,11.017,11.057,11.097,11.124,11.143,11.155,
     * 10.690,10.765,10.844,10.901,10.957,10.989,11.013,11.025,
     * 10.305,10.382,10.465,10.554,10.641,10.697,10.741,10.768,
     * 10.185,10.233,10.292,10.371,10.449,10.512,10.562,10.601,
     * 10.090,10.121,10.164,10.215,10.265,10.303,10.326,10.360,
     * 9.890,9.926,9.977,10.037,10.097,10.121,10.127,10.137,
     * 9.468,9.461,9.470,9.498,9.525,9.517,9.489,9.465,
     * 9.410,9.403,9.413,9.442,9.470,9.462,9.433,9.408,
     * 9.137,9.130,9.143,9.173,9.204,9.196,9.167,9.146,
     * 9.091,9.085,9.097,9.127,9.158,9.150,9.122,9.100/
      DATA ((EDEN(I,4,4,L,3),I=41,48),L=1,NZEN)/
     * 11.107,11.143,11.182,11.207,11.230,11.248,11.265,11.276,
     * 11.033,11.072,11.114,11.140,11.167,11.185,11.201,11.212,
     * 10.927,10.975,11.025,11.057,11.086,11.107,11.121,11.137,
     * 10.715,10.782,10.854,10.900,10.945,10.970,10.991,11.004,
     * 10.303,10.382,10.467,10.548,10.626,10.676,10.718,10.744,
     * 10.158,10.210,10.270,10.346,10.422,10.481,10.531,10.572,
     * 10.049,10.079,10.117,10.164,10.210,10.248,10.274,10.312,
     * 9.854,9.886,9.931,9.983,10.033,10.057,10.061,10.072,
     * 9.459,9.449,9.452,9.471,9.490,9.477,9.447,9.427,
     * 9.401,9.391,9.394,9.415,9.435,9.422,9.391,9.369,
     * 9.127,9.117,9.121,9.143,9.164,9.155,9.127,9.107,
     * 9.081,9.072,9.075,9.097,9.119,9.110,9.081,9.061/
      DATA ((EDEN(I,5,4,L,3),I=41,48),L=1,NZEN)/
     * 11.134,11.161,11.190,11.207,11.220,11.233,11.250,11.262,
     * 11.053,11.086,11.121,11.137,11.152,11.164,11.179,11.193,
     * 10.947,10.989,11.033,11.053,11.072,11.086,11.104,11.121,
     * 10.737,10.798,10.862,10.898,10.932,10.951,10.970,10.986,
     * 10.305,10.384,10.467,10.539,10.611,10.654,10.695,10.723,
     * 10.140,10.188,10.246,10.322,10.396,10.453,10.507,10.549,
     * 10.017,10.033,10.061,10.111,10.158,10.201,10.238,10.283,
     * 9.820,9.840,9.872,9.925,9.977,10.004,10.017,10.041,
     * 9.452,9.436,9.433,9.446,9.458,9.442,9.417,9.401,
     * 9.394,9.378,9.377,9.389,9.401,9.387,9.360,9.344,
     * 9.117,9.104,9.100,9.117,9.130,9.121,9.093,9.083,
     * 9.072,9.058,9.055,9.072,9.085,9.075,9.048,9.037/
      DATA ((EDEN(I,1,5,L,3),I=41,48),L=1,NZEN)/
     * 11.107,11.149,11.199,11.243,11.288,11.314,11.324,11.326,
     * 10.980,11.033,11.097,11.149,11.204,11.236,11.253,11.265,
     * 10.867,10.932,11.004,11.064,11.124,11.158,11.179,11.190,
     * 10.652,10.736,10.827,10.904,10.980,11.021,11.045,11.057,
     * 10.294,10.371,10.455,10.555,10.656,10.723,10.769,10.795,
     * 10.193,10.241,10.299,10.382,10.465,10.535,10.583,10.621,
     * 10.104,10.137,10.185,10.241,10.294,10.330,10.340,10.365,
     * 9.902,9.941,9.997,10.064,10.127,10.149,10.143,10.137,
     * 9.465,9.465,9.484,9.525,9.566,9.560,9.525,9.491,
     * 9.407,9.408,9.428,9.470,9.509,9.505,9.470,9.435,
     * 9.137,9.137,9.158,9.199,9.241,9.238,9.204,9.170,
     * 9.091,9.091,9.113,9.153,9.195,9.192,9.158,9.125/
      DATA ((EDEN(I,2,5,L,3),I=41,48),L=1,NZEN)/
     * 11.100,11.140,11.188,11.228,11.267,11.292,11.305,11.307,
     * 10.997,11.045,11.104,11.146,11.188,11.217,11.233,11.243,
     * 10.886,10.945,11.009,11.061,11.114,11.143,11.158,11.167,
     * 10.673,10.751,10.836,10.903,10.969,11.004,11.025,11.037,
     * 10.312,10.384,10.467,10.560,10.652,10.712,10.755,10.777,
     * 10.204,10.248,10.303,10.384,10.461,10.525,10.573,10.607,
     * 10.114,10.143,10.185,10.236,10.283,10.316,10.328,10.354,
     * 9.912,9.948,10.000,10.057,10.117,10.134,10.127,10.121,
     * 9.468,9.464,9.477,9.511,9.545,9.537,9.501,9.468,
     * 9.410,9.407,9.422,9.455,9.490,9.480,9.446,9.412,
     * 9.137,9.134,9.149,9.185,9.223,9.215,9.182,9.149,
     * 9.091,9.088,9.103,9.139,9.177,9.169,9.136,9.103/
      DATA ((EDEN(I,3,5,L,3),I=41,48),L=1,NZEN)/
     * 11.104,11.140,11.179,11.212,11.243,11.265,11.276,11.283,
     * 11.017,11.061,11.107,11.143,11.176,11.199,11.215,11.220,
     * 10.908,10.960,11.017,11.061,11.097,11.124,11.140,11.146,
     * 10.697,10.769,10.848,10.905,10.960,10.989,11.009,11.017,
     * 10.305,10.384,10.471,10.561,10.648,10.701,10.741,10.762,
     * 10.182,10.230,10.292,10.373,10.453,10.516,10.565,10.598,
     * 9.903,9.930,9.975,10.037,10.100,10.130,10.140,10.158,
     * 9.884,9.920,9.973,10.033,10.097,10.121,10.124,10.130,
     * 9.459,9.452,9.462,9.491,9.520,9.511,9.479,9.452,
     * 9.403,9.394,9.405,9.435,9.464,9.455,9.423,9.394,
     * 9.130,9.124,9.134,9.164,9.196,9.190,9.158,9.134,
     * 9.085,9.078,9.088,9.119,9.150,9.145,9.113,9.088/
      DATA ((EDEN(I,4,5,L,3),I=41,48),L=1,NZEN)/
     * 11.124,11.149,11.179,11.201,11.223,11.241,11.253,11.255,
     * 11.041,11.076,11.114,11.143,11.167,11.182,11.190,11.196,
     * 10.933,10.976,11.025,11.057,11.086,11.104,11.114,11.121,
     * 10.723,10.787,10.858,10.905,10.948,10.970,10.984,10.992,
     * 10.305,10.386,10.474,10.556,10.636,10.681,10.718,10.736,
     * 10.152,10.207,10.272,10.352,10.433,10.490,10.538,10.569,
     * 10.041,10.072,10.111,10.164,10.215,10.255,10.283,10.316,
     * 9.845,9.877,9.926,9.984,10.041,10.064,10.068,10.079,
     * 9.450,9.438,9.442,9.465,9.486,9.473,9.441,9.415,
     * 9.394,9.380,9.386,9.408,9.430,9.417,9.384,9.358,
     * 9.121,9.107,9.114,9.137,9.161,9.149,9.121,9.097,
     * 9.075,9.061,9.068,9.091,9.116,9.103,9.075,9.051/
      DATA ((EDEN(I,5,5,L,3),I=41,48),L=1,NZEN)/
     * 11.149,11.164,11.188,11.201,11.212,11.220,11.225,11.228,
     * 11.061,11.090,11.124,11.143,11.155,11.161,11.167,11.170,
     * 10.956,10.992,11.033,11.057,11.076,11.086,11.093,11.097,
     * 10.748,10.805,10.868,10.904,10.936,10.950,10.961,10.967,
     * 10.310,10.389,10.476,10.551,10.622,10.661,10.693,10.712,
     * 10.130,10.185,10.246,10.330,10.410,10.464,10.511,10.543,
     * 10.000,10.021,10.053,10.107,10.164,10.210,10.246,10.286,
     * 9.810,9.828,9.862,9.922,9.979,10.009,10.025,10.045,
     * 9.446,9.427,9.423,9.439,9.452,9.436,9.405,9.386,
     * 9.389,9.369,9.367,9.382,9.396,9.380,9.350,9.328,
     * 9.114,9.093,9.093,9.111,9.127,9.114,9.086,9.068,
     * 9.068,9.048,9.048,9.065,9.081,9.068,9.041,9.022/
      DATA ((EDEN(I,1,6,L,3),I=41,48),L=1,NZEN)/
     * 11.114,11.155,11.204,11.248,11.290,11.316,11.324,11.326,
     * 10.985,11.041,11.100,11.152,11.204,11.238,11.255,11.267,
     * 10.873,10.938,11.009,11.064,11.124,11.158,11.179,11.190,
     * 10.660,10.743,10.833,10.907,10.982,11.025,11.049,11.061,
     * 10.299,10.377,10.462,10.562,10.663,10.728,10.774,10.798,
     * 10.193,10.243,10.301,10.386,10.471,10.540,10.590,10.626,
     * 10.100,10.137,10.182,10.238,10.294,10.330,10.342,10.369,
     * 9.902,9.941,9.997,10.061,10.127,10.149,10.143,10.134,
     * 9.458,9.459,9.479,9.519,9.560,9.554,9.520,9.486,
     * 9.401,9.401,9.422,9.462,9.504,9.498,9.464,9.428,
     * 9.127,9.130,9.152,9.193,9.236,9.230,9.199,9.167,
     * 9.081,9.085,9.107,9.147,9.190,9.185,9.153,9.122/
      DATA ((EDEN(I,2,6,L,3),I=41,48),L=1,NZEN)/
     * 11.107,11.146,11.193,11.233,11.270,11.292,11.303,11.303,
     * 11.000,11.049,11.107,11.149,11.193,11.220,11.236,11.241,
     * 10.893,10.950,11.017,11.064,11.114,11.140,11.155,11.164,
     * 10.684,10.760,10.843,10.909,10.973,11.009,11.025,11.033,
     * 10.316,10.389,10.474,10.568,10.661,10.719,10.759,10.778,
     * 10.204,10.250,10.307,10.387,10.468,10.531,10.577,10.610,
     * 10.111,10.140,10.182,10.233,10.283,10.316,10.330,10.354,
     * 9.912,9.946,9.997,10.057,10.117,10.134,10.124,10.117,
     * 9.459,9.455,9.468,9.504,9.538,9.529,9.493,9.458,
     * 9.401,9.398,9.412,9.447,9.481,9.473,9.438,9.401,
     * 9.130,9.127,9.143,9.179,9.215,9.207,9.173,9.140,
     * 9.085,9.081,9.097,9.133,9.169,9.161,9.127,9.094/
      DATA ((EDEN(I,3,6,L,3),I=41,48),L=1,NZEN)/
     * 11.111,11.143,11.182,11.215,11.243,11.262,11.272,11.274,
     * 11.025,11.064,11.111,11.146,11.176,11.199,11.212,11.215,
     * 10.916,10.965,11.021,11.064,11.100,11.124,11.137,11.140,
     * 10.709,10.777,10.854,10.910,10.964,10.990,11.004,11.009,
     * 10.314,10.393,10.481,10.571,10.657,10.708,10.744,10.760,
     * 10.179,10.233,10.297,10.380,10.462,10.524,10.568,10.597,
     * 10.079,10.111,10.155,10.212,10.267,10.305,10.326,10.354,
     * 9.881,9.918,9.971,10.033,10.097,10.121,10.124,10.124,
     * 9.450,9.442,9.453,9.483,9.513,9.502,9.470,9.439,
     * 9.394,9.386,9.394,9.427,9.456,9.447,9.413,9.382,
     * 9.121,9.114,9.127,9.158,9.190,9.182,9.152,9.124,
     * 9.075,9.068,9.081,9.113,9.145,9.136,9.107,9.078/
      DATA ((EDEN(I,4,6,L,3),I=41,48),L=1,NZEN)/
     * 11.130,11.149,11.176,11.201,11.220,11.233,11.238,11.238,
     * 11.049,11.079,11.117,11.143,11.167,11.179,11.188,11.185,
     * 10.942,10.981,11.029,11.061,11.090,11.104,11.111,11.114,
     * 10.736,10.795,10.864,10.910,10.953,10.971,10.982,10.983,
     * 10.316,10.396,10.486,10.568,10.647,10.688,10.719,10.732,
     * 10.155,10.210,10.276,10.362,10.444,10.500,10.543,10.568,
     * 10.037,10.064,10.107,10.164,10.220,10.260,10.288,10.318,
     * 9.841,9.872,9.921,9.983,10.041,10.068,10.072,10.079,
     * 9.444,9.428,9.433,9.456,9.479,9.465,9.430,9.401,
     * 9.386,9.371,9.377,9.400,9.422,9.408,9.375,9.344,
     * 9.111,9.097,9.104,9.130,9.155,9.143,9.111,9.086,
     * 9.065,9.051,9.058,9.085,9.110,9.097,9.065,9.041/
      DATA ((EDEN(I,5,6,L,3),I=41,48),L=1,NZEN)/
     * 11.152,11.161,11.179,11.190,11.199,11.207,11.207,11.204,
     * 11.072,11.093,11.124,11.143,11.155,11.161,11.161,11.158,
     * 10.967,10.998,11.037,11.061,11.076,11.083,11.086,11.083,
     * 10.763,10.814,10.876,10.911,10.942,10.952,10.956,10.955,
     * 10.324,10.401,10.491,10.565,10.635,10.668,10.694,10.704,
     * 10.134,10.190,10.258,10.342,10.425,10.474,10.515,10.540,
     * 9.998,10.017,10.049,10.111,10.170,10.215,10.250,10.286,
     * 9.802,9.820,9.857,9.921,9.982,10.013,10.029,10.045,
     * 9.439,9.417,9.415,9.431,9.444,9.428,9.393,9.369,
     * 9.382,9.360,9.356,9.375,9.389,9.373,9.338,9.314,
     * 9.107,9.083,9.083,9.104,9.121,9.107,9.076,9.053,
     * 9.061,9.037,9.037,9.058,9.075,9.061,9.030,9.007/
      DATA ((EDEN(I,1,7,L,3),I=41,48),L=1,NZEN)/
     * 11.117,11.158,11.204,11.248,11.290,11.316,11.324,11.324,
     * 10.990,11.045,11.104,11.152,11.204,11.238,11.255,11.267,
     * 10.880,10.943,11.013,11.068,11.127,11.161,11.182,11.190,
     * 10.668,10.750,10.840,10.913,10.985,11.025,11.049,11.061,
     * 10.303,10.382,10.468,10.569,10.669,10.732,10.777,10.801,
     * 10.196,10.246,10.305,10.391,10.476,10.545,10.594,10.629,
     * 10.100,10.137,10.182,10.241,10.297,10.332,10.346,10.373,
     * 9.902,9.941,9.997,10.061,10.127,10.149,10.143,10.134,
     * 9.456,9.456,9.476,9.516,9.555,9.550,9.516,9.483,
     * 9.398,9.400,9.418,9.459,9.500,9.494,9.461,9.427,
     * 9.127,9.127,9.146,9.190,9.233,9.228,9.193,9.161,
     * 9.081,9.081,9.100,9.145,9.187,9.182,9.147,9.116/
      DATA ((EDEN(I,2,7,L,3),I=41,48),L=1,NZEN)/
     * 11.111,11.149,11.193,11.233,11.270,11.292,11.303,11.303,
     * 11.004,11.053,11.104,11.149,11.193,11.220,11.236,11.241,
     * 10.900,10.956,11.021,11.068,11.117,11.146,11.158,11.164,
     * 10.692,10.766,10.848,10.912,10.975,11.009,11.029,11.037,
     * 10.322,10.396,10.481,10.574,10.667,10.722,10.761,10.780,
     * 10.204,10.250,10.310,10.393,10.473,10.537,10.582,10.612,
     * 10.111,10.140,10.182,10.236,10.286,10.320,10.332,10.356,
     * 9.914,9.948,9.999,10.057,10.117,10.134,10.127,10.117,
     * 9.455,9.450,9.465,9.500,9.534,9.525,9.489,9.455,
     * 9.398,9.393,9.408,9.444,9.479,9.468,9.433,9.398,
     * 9.124,9.121,9.137,9.176,9.212,9.204,9.170,9.137,
     * 9.078,9.075,9.091,9.130,9.166,9.158,9.125,9.091/
      DATA ((EDEN(I,3,7,L,3),I=41,48),L=1,NZEN)/
     * 11.114,11.143,11.182,11.215,11.243,11.262,11.274,11.272,
     * 11.029,11.064,11.111,11.146,11.182,11.201,11.210,11.212,
     * 10.922,10.969,11.025,11.064,11.100,11.124,11.134,11.137,
     * 10.717,10.783,10.858,10.913,10.966,10.992,11.004,11.009,
     * 10.322,10.400,10.489,10.576,10.663,10.711,10.745,10.759,
     * 10.182,10.236,10.299,10.386,10.470,10.528,10.572,10.598,
     * 10.079,10.111,10.155,10.212,10.267,10.307,10.328,10.354,
     * 9.884,9.918,9.971,10.033,10.097,10.121,10.121,10.121,
     * 9.446,9.436,9.447,9.479,9.508,9.498,9.465,9.435,
     * 9.387,9.378,9.391,9.422,9.453,9.442,9.410,9.378,
     * 9.114,9.107,9.121,9.152,9.188,9.179,9.146,9.117,
     * 9.068,9.061,9.075,9.107,9.142,9.133,9.100,9.072/
      DATA ((EDEN(I,4,7,L,3),I=41,48),L=1,NZEN)/
     * 11.130,11.149,11.173,11.196,11.217,11.230,11.238,11.236,
     * 11.049,11.079,11.121,11.146,11.170,11.182,11.188,11.185,
     * 10.946,10.985,11.033,11.064,11.090,11.107,11.114,11.111,
     * 10.744,10.801,10.868,10.913,10.955,10.973,10.981,10.981,
     * 10.324,10.401,10.493,10.573,10.651,10.691,10.719,10.730,
     * 10.158,10.215,10.283,10.367,10.450,10.502,10.544,10.567,
     * 10.037,10.064,10.111,10.167,10.220,10.260,10.286,10.314,
     * 9.844,9.874,9.923,9.983,10.041,10.064,10.064,10.068,
     * 9.436,9.422,9.427,9.452,9.473,9.459,9.423,9.393,
     * 9.380,9.365,9.371,9.394,9.417,9.403,9.367,9.336,
     * 9.104,9.093,9.100,9.127,9.152,9.140,9.104,9.076,
     * 9.058,9.048,9.055,9.081,9.107,9.094,9.058,9.030/
      DATA ((EDEN(I,5,7,L,3),I=41,48),L=1,NZEN)/
     * 11.149,11.155,11.170,11.188,11.196,11.201,11.204,11.199,
     * 11.076,11.097,11.127,11.146,11.158,11.164,11.164,11.155,
     * 10.971,11.000,11.041,11.064,11.079,11.086,11.086,11.083,
     * 10.769,10.819,10.879,10.913,10.943,10.952,10.956,10.952,
     * 10.332,10.410,10.498,10.571,10.639,10.671,10.694,10.701,
     * 10.140,10.196,10.262,10.348,10.430,10.479,10.516,10.538,
     * 10.000,10.017,10.049,10.114,10.170,10.215,10.248,10.281,
     * 9.805,9.822,9.859,9.921,9.980,10.009,10.021,10.033,
     * 9.435,9.412,9.408,9.427,9.441,9.423,9.386,9.360,
     * 9.378,9.354,9.352,9.369,9.384,9.367,9.330,9.303,
     * 9.100,9.079,9.079,9.097,9.114,9.100,9.068,9.041,
     * 9.055,9.033,9.033,9.051,9.068,9.055,9.022,8.996/
      DATA ((EDEN(I,1,8,L,3),I=41,48),L=1,NZEN)/
     * 11.114,11.155,11.201,11.246,11.288,11.314,11.324,11.326,
     * 10.990,11.045,11.104,11.155,11.207,11.238,11.255,11.267,
     * 10.879,10.942,11.013,11.072,11.127,11.161,11.182,11.190,
     * 10.666,10.748,10.838,10.911,10.984,11.025,11.049,11.061,
     * 10.303,10.380,10.467,10.567,10.667,10.731,10.775,10.799,
     * 10.196,10.246,10.305,10.389,10.473,10.543,10.592,10.628,
     * 10.104,10.137,10.185,10.241,10.297,10.332,10.346,10.373,
     * 9.904,9.943,9.999,10.064,10.127,10.149,10.143,10.137,
     * 9.462,9.461,9.479,9.519,9.560,9.554,9.520,9.487,
     * 9.405,9.405,9.423,9.462,9.504,9.498,9.464,9.430,
     * 9.130,9.134,9.152,9.193,9.236,9.233,9.199,9.167,
     * 9.085,9.088,9.107,9.147,9.190,9.187,9.153,9.122/
      DATA ((EDEN(I,2,8,L,3),I=41,48),L=1,NZEN)/
     * 11.111,11.149,11.193,11.233,11.270,11.292,11.303,11.303,
     * 11.004,11.053,11.107,11.149,11.193,11.220,11.236,11.243,
     * 10.895,10.952,11.017,11.064,11.114,11.143,11.161,11.167,
     * 10.687,10.762,10.845,10.910,10.974,11.009,11.029,11.037,
     * 10.318,10.393,10.477,10.569,10.661,10.719,10.759,10.779,
     * 10.204,10.250,10.307,10.389,10.468,10.531,10.577,10.610,
     * 10.111,10.143,10.185,10.236,10.286,10.318,10.332,10.356,
     * 9.914,9.948,10.000,10.061,10.117,10.134,10.127,10.117,
     * 9.462,9.458,9.471,9.507,9.539,9.530,9.494,9.461,
     * 9.405,9.400,9.415,9.449,9.483,9.474,9.439,9.403,
     * 9.130,9.127,9.143,9.179,9.215,9.210,9.176,9.143,
     * 9.085,9.081,9.097,9.133,9.169,9.164,9.130,9.097/
      DATA ((EDEN(I,3,8,L,3),I=41,48),L=1,NZEN)/
     * 11.111,11.143,11.182,11.215,11.248,11.265,11.274,11.276,
     * 11.025,11.064,11.111,11.146,11.176,11.199,11.212,11.217,
     * 10.917,10.966,11.021,11.064,11.100,11.124,11.137,11.140,
     * 10.710,10.777,10.854,10.910,10.964,10.991,11.009,11.013,
     * 10.316,10.394,10.481,10.571,10.656,10.707,10.744,10.760,
     * 10.185,10.236,10.299,10.382,10.462,10.522,10.568,10.597,
     * 10.083,10.114,10.158,10.215,10.267,10.305,10.326,10.354,
     * 9.886,9.921,9.974,10.037,10.097,10.121,10.121,10.121,
     * 9.452,9.442,9.452,9.484,9.513,9.504,9.471,9.441,
     * 9.394,9.386,9.396,9.428,9.458,9.449,9.415,9.384,
     * 9.121,9.114,9.127,9.158,9.190,9.182,9.152,9.124,
     * 9.075,9.068,9.081,9.113,9.145,9.136,9.107,9.078/
      DATA ((EDEN(I,4,8,L,3),I=41,48),L=1,NZEN)/
     * 11.124,11.146,11.176,11.201,11.223,11.238,11.246,11.246,
     * 11.045,11.076,11.117,11.143,11.167,11.182,11.190,11.193,
     * 10.941,10.981,11.029,11.061,11.090,11.107,11.114,11.117,
     * 10.735,10.794,10.863,10.909,10.952,10.972,10.983,10.986,
     * 10.320,10.396,10.486,10.566,10.643,10.686,10.717,10.730,
     * 10.161,10.212,10.279,10.360,10.439,10.493,10.535,10.561,
     * 10.045,10.072,10.111,10.161,10.212,10.248,10.272,10.299,
     * 9.850,9.878,9.924,9.980,10.033,10.053,10.053,10.053,
     * 9.442,9.428,9.433,9.456,9.477,9.461,9.425,9.394,
     * 9.386,9.371,9.377,9.400,9.420,9.405,9.369,9.338,
     * 9.111,9.097,9.104,9.130,9.152,9.140,9.107,9.076,
     * 9.065,9.051,9.058,9.085,9.107,9.094,9.061,9.030/
      DATA ((EDEN(I,5,8,L,3),I=41,48),L=1,NZEN)/
     * 11.146,11.158,11.179,11.196,11.207,11.215,11.220,11.217,
     * 11.064,11.090,11.121,11.140,11.155,11.161,11.167,11.164,
     * 10.962,10.995,11.033,11.057,11.076,11.086,11.090,11.090,
     * 10.760,10.812,10.873,10.908,10.940,10.952,10.959,10.959,
     * 10.326,10.400,10.487,10.561,10.630,10.665,10.691,10.702,
     * 10.143,10.193,10.258,10.338,10.418,10.468,10.511,10.534,
     * 10.013,10.025,10.057,10.114,10.164,10.207,10.236,10.270,
     * 9.818,9.833,9.866,9.923,9.977,10.000,10.004,10.017,
     * 9.439,9.417,9.415,9.431,9.444,9.427,9.391,9.362,
     * 9.382,9.360,9.358,9.375,9.389,9.371,9.334,9.305,
     * 9.107,9.086,9.086,9.104,9.121,9.104,9.072,9.045,
     * 9.061,9.041,9.041,9.058,9.075,9.058,9.026,9.000/
      DATA ((EDEN(I,1,9,L,3),I=41,48),L=1,NZEN)/
     * 11.104,11.149,11.199,11.241,11.286,11.312,11.324,11.326,
     * 10.980,11.037,11.100,11.152,11.204,11.238,11.255,11.267,
     * 10.865,10.931,11.004,11.064,11.124,11.158,11.179,11.190,
     * 10.647,10.733,10.826,10.903,10.979,11.021,11.045,11.057,
     * 10.294,10.369,10.452,10.554,10.654,10.721,10.769,10.794,
     * 10.196,10.243,10.301,10.382,10.464,10.533,10.581,10.619,
     * 10.107,10.143,10.188,10.243,10.299,10.332,10.342,10.367,
     * 9.904,9.943,9.999,10.064,10.130,10.152,10.146,10.140,
     * 9.474,9.473,9.491,9.533,9.573,9.567,9.531,9.498,
     * 9.417,9.417,9.435,9.476,9.516,9.511,9.476,9.441,
     * 9.143,9.143,9.164,9.204,9.248,9.243,9.210,9.176,
     * 9.097,9.097,9.119,9.158,9.202,9.197,9.164,9.130/
      DATA ((EDEN(I,2,9,L,3),I=41,48),L=1,NZEN)/
     * 11.107,11.146,11.190,11.230,11.267,11.290,11.303,11.307,
     * 10.993,11.045,11.100,11.143,11.188,11.217,11.238,11.248,
     * 10.878,10.940,11.009,11.057,11.107,11.140,11.158,11.170,
     * 10.663,10.743,10.830,10.899,10.967,11.004,11.029,11.037,
     * 10.307,10.378,10.458,10.551,10.644,10.706,10.751,10.777,
     * 10.204,10.248,10.301,10.378,10.453,10.519,10.567,10.603,
     * 10.117,10.146,10.188,10.238,10.286,10.316,10.326,10.350,
     * 9.914,9.949,10.000,10.061,10.117,10.137,10.127,10.121,
     * 9.477,9.473,9.486,9.520,9.554,9.545,9.509,9.477,
     * 9.418,9.415,9.430,9.464,9.497,9.489,9.453,9.420,
     * 9.146,9.143,9.158,9.196,9.230,9.223,9.188,9.155,
     * 9.100,9.097,9.113,9.150,9.185,9.177,9.142,9.110/
      DATA ((EDEN(I,3,9,L,3),I=41,48),L=1,NZEN)/
     * 11.104,11.140,11.185,11.217,11.248,11.267,11.281,11.286,
     * 11.009,11.053,11.100,11.140,11.176,11.201,11.215,11.225,
     * 10.898,10.951,11.013,11.057,11.097,11.124,11.140,11.149,
     * 10.684,10.757,10.838,10.897,10.955,10.987,11.009,11.017,
     * 10.303,10.377,10.459,10.549,10.637,10.693,10.736,10.760,
     * 10.185,10.230,10.288,10.367,10.444,10.507,10.556,10.592,
     * 10.093,10.121,10.164,10.215,10.267,10.301,10.320,10.350,
     * 9.892,9.927,9.977,10.037,10.097,10.121,10.124,10.124,
     * 9.470,9.462,9.471,9.501,9.529,9.519,9.487,9.458,
     * 9.412,9.405,9.415,9.444,9.473,9.462,9.431,9.401,
     * 9.140,9.134,9.143,9.176,9.207,9.196,9.167,9.140,
     * 9.094,9.088,9.097,9.130,9.161,9.150,9.122,9.094/
      DATA ((EDEN(I,4,9,L,3),I=41,48),L=1,NZEN)/
     * 11.111,11.143,11.176,11.201,11.225,11.246,11.260,11.265,
     * 11.029,11.068,11.107,11.137,11.161,11.182,11.199,11.207,
     * 10.920,10.967,11.017,11.053,11.086,11.107,11.121,11.127,
     * 10.708,10.775,10.846,10.896,10.943,10.968,10.986,10.996,
     * 10.303,10.377,10.459,10.542,10.621,10.669,10.709,10.732,
     * 10.161,10.207,10.262,10.338,10.413,10.471,10.520,10.554,
     * 10.057,10.079,10.114,10.158,10.201,10.233,10.250,10.281,
     * 9.860,9.886,9.929,9.978,10.025,10.037,10.037,10.033,
     * 9.459,9.446,9.449,9.471,9.491,9.474,9.439,9.410,
     * 9.401,9.389,9.393,9.415,9.435,9.418,9.384,9.352,
     * 9.127,9.114,9.121,9.143,9.164,9.152,9.117,9.086,
     * 9.081,9.068,9.075,9.097,9.119,9.107,9.072,9.041/
      DATA ((EDEN(I,5,9,L,3),I=41,48),L=1,NZEN)/
     * 11.130,11.152,11.182,11.201,11.215,11.228,11.241,11.246,
     * 11.045,11.079,11.114,11.134,11.152,11.164,11.176,11.185,
     * 10.941,10.981,11.025,11.049,11.072,11.086,11.097,11.107,
     * 10.731,10.788,10.852,10.893,10.930,10.949,10.965,10.974,
     * 10.307,10.378,10.459,10.534,10.606,10.648,10.686,10.707,
     * 10.146,10.188,10.241,10.316,10.391,10.446,10.494,10.528,
     * 10.025,10.041,10.064,10.111,10.152,10.188,10.215,10.250,
     * 9.832,9.846,9.874,9.923,9.969,9.987,9.991,9.998,
     * 9.453,9.435,9.433,9.447,9.459,9.441,9.408,9.380,
     * 9.394,9.377,9.375,9.391,9.403,9.384,9.350,9.324,
     * 9.117,9.100,9.100,9.117,9.134,9.117,9.086,9.061,
     * 9.072,9.055,9.055,9.072,9.088,9.072,9.041,9.015/
      DATA ((EDEN(I,1,10,L,3),I=41,48),L=1,NZEN)/
     * 11.097,11.143,11.193,11.238,11.286,11.312,11.324,11.330,
     * 10.966,11.025,11.093,11.149,11.204,11.238,11.255,11.267,
     * 10.848,10.919,10.997,11.061,11.124,11.158,11.179,11.190,
     * 10.626,10.714,10.811,10.892,10.973,11.017,11.045,11.061,
     * 10.288,10.358,10.438,10.539,10.639,10.710,10.760,10.790,
     * 10.196,10.241,10.294,10.375,10.453,10.522,10.572,10.612,
     * 10.111,10.146,10.190,10.246,10.301,10.332,10.340,10.364,
     * 9.907,9.945,10.000,10.068,10.134,10.155,10.149,10.140,
     * 9.489,9.489,9.508,9.548,9.588,9.582,9.547,9.512,
     * 9.430,9.430,9.449,9.490,9.531,9.525,9.490,9.455,
     * 9.155,9.158,9.176,9.220,9.262,9.258,9.223,9.190,
     * 9.110,9.113,9.130,9.174,9.217,9.212,9.177,9.145/
      DATA ((EDEN(I,2,10,L,3),I=41,48),L=1,NZEN)/
     * 11.107,11.146,11.193,11.230,11.267,11.292,11.307,11.314,
     * 10.975,11.033,11.093,11.140,11.188,11.220,11.238,11.253,
     * 10.857,10.924,10.997,11.053,11.107,11.140,11.161,11.176,
     * 10.635,10.720,10.812,10.885,10.958,11.000,11.029,11.041,
     * 10.294,10.364,10.438,10.531,10.625,10.692,10.744,10.775,
     * 10.204,10.246,10.294,10.367,10.438,10.504,10.554,10.597,
     * 10.121,10.149,10.190,10.238,10.286,10.314,10.324,10.348,
     * 9.912,9.949,10.000,10.061,10.117,10.137,10.130,10.124,
     * 9.494,9.491,9.505,9.538,9.572,9.562,9.528,9.496,
     * 9.438,9.435,9.449,9.481,9.516,9.507,9.471,9.439,
     * 9.164,9.161,9.176,9.212,9.246,9.238,9.207,9.173,
     * 9.119,9.116,9.130,9.166,9.200,9.192,9.161,9.127/
      DATA ((EDEN(I,3,10,L,3),I=41,48),L=1,NZEN)/
     * 11.100,11.143,11.185,11.217,11.248,11.270,11.286,11.297,
     * 10.992,11.041,11.097,11.137,11.173,11.199,11.220,11.236,
     * 10.874,10.937,11.000,11.049,11.093,11.124,11.143,11.158,
     * 10.654,10.734,10.820,10.883,10.946,10.983,11.013,11.029,
     * 10.292,10.362,10.438,10.528,10.616,10.679,10.730,10.763,
     * 10.190,10.230,10.281,10.354,10.427,10.491,10.544,10.589,
     * 10.100,10.130,10.170,10.217,10.265,10.297,10.316,10.348,
     * 9.898,9.932,9.979,10.037,10.097,10.121,10.124,10.130,
     * 9.490,9.484,9.493,9.521,9.549,9.538,9.508,9.483,
     * 9.433,9.427,9.436,9.464,9.491,9.481,9.452,9.427,
     * 9.158,9.152,9.164,9.193,9.223,9.215,9.185,9.161,
     * 9.113,9.107,9.119,9.147,9.177,9.169,9.139,9.116/
      DATA ((EDEN(I,4,10,L,3),I=41,48),L=1,NZEN)/
     * 11.097,11.137,11.179,11.204,11.230,11.250,11.272,11.286,
     * 11.013,11.057,11.104,11.134,11.161,11.185,11.204,11.223,
     * 10.900,10.953,11.009,11.045,11.079,11.104,11.127,11.143,
     * 10.679,10.752,10.828,10.881,10.933,10.964,10.992,11.009,
     * 10.286,10.358,10.435,10.516,10.595,10.652,10.702,10.735,
     * 10.164,10.207,10.253,10.320,10.386,10.447,10.501,10.548,
     * 10.068,10.090,10.121,10.155,10.190,10.215,10.230,10.265,
     * 9.869,9.895,9.934,9.976,10.017,10.025,10.021,10.017,
     * 9.479,9.468,9.471,9.490,9.508,9.491,9.459,9.433,
     * 9.422,9.412,9.415,9.433,9.452,9.436,9.403,9.375,
     * 9.146,9.137,9.143,9.161,9.182,9.167,9.134,9.107,
     * 9.100,9.091,9.097,9.116,9.136,9.122,9.088,9.061/
      DATA ((EDEN(I,5,10,L,3),I=41,48),L=1,NZEN)/
     * 11.107,11.143,11.182,11.201,11.220,11.238,11.260,11.276,
     * 11.033,11.072,11.107,11.130,11.149,11.164,11.185,11.204,
     * 10.921,10.968,11.017,11.045,11.068,11.086,11.107,11.127,
     * 10.704,10.770,10.838,10.880,10.920,10.945,10.971,10.992,
     * 10.288,10.358,10.431,10.508,10.581,10.631,10.680,10.715,
     * 10.146,10.185,10.225,10.294,10.362,10.422,10.477,10.526,
     * 10.041,10.053,10.072,10.107,10.140,10.170,10.196,10.236,
     * 9.846,9.858,9.881,9.922,9.961,9.974,9.976,9.984,
     * 9.468,9.455,9.452,9.465,9.477,9.459,9.430,9.408,
     * 9.412,9.396,9.394,9.408,9.420,9.401,9.371,9.350,
     * 9.134,9.121,9.121,9.134,9.149,9.134,9.104,9.083,
     * 9.088,9.075,9.075,9.088,9.103,9.088,9.058,9.037/
      DATA ((EDEN(I,1,11,L,3),I=41,48),L=1,NZEN)/
     * 11.093,11.143,11.196,11.241,11.286,11.314,11.328,11.332,
     * 10.959,11.021,11.090,11.146,11.204,11.238,11.260,11.270,
     * 10.838,10.911,10.992,11.057,11.124,11.161,11.182,11.193,
     * 10.615,10.705,10.803,10.888,10.971,11.017,11.049,11.061,
     * 10.286,10.352,10.430,10.531,10.632,10.705,10.758,10.789,
     * 10.199,10.241,10.294,10.373,10.449,10.519,10.567,10.609,
     * 10.117,10.149,10.193,10.248,10.303,10.332,10.340,10.362,
     * 9.907,9.946,10.004,10.068,10.134,10.155,10.152,10.143,
     * 9.497,9.497,9.516,9.556,9.598,9.591,9.555,9.520,
     * 9.439,9.439,9.459,9.500,9.542,9.535,9.500,9.462,
     * 9.164,9.164,9.185,9.228,9.272,9.267,9.233,9.196,
     * 9.119,9.119,9.139,9.182,9.226,9.221,9.187,9.150/
      DATA ((EDEN(I,2,11,L,3),I=41,48),L=1,NZEN)/
     * 11.111,11.152,11.196,11.233,11.270,11.297,11.314,11.322,
     * 10.964,11.025,11.090,11.137,11.185,11.217,11.241,11.258,
     * 10.843,10.914,10.990,11.049,11.104,11.140,11.167,11.182,
     * 10.620,10.708,10.800,10.876,10.952,10.998,11.029,11.049,
     * 10.292,10.356,10.427,10.521,10.614,10.684,10.740,10.777,
     * 10.207,10.246,10.294,10.362,10.430,10.497,10.548,10.594,
     * 10.124,10.155,10.193,10.241,10.288,10.314,10.322,10.348,
     * 9.915,9.951,10.004,10.061,10.117,10.137,10.134,10.130,
     * 9.507,9.504,9.516,9.549,9.583,9.574,9.540,9.511,
     * 9.449,9.446,9.459,9.493,9.526,9.517,9.484,9.453,
     * 9.173,9.173,9.188,9.223,9.258,9.250,9.217,9.188,
     * 9.127,9.127,9.142,9.177,9.212,9.205,9.172,9.142/
      DATA ((EDEN(I,3,11,L,3),I=41,48),L=1,NZEN)/
     * 11.104,11.146,11.188,11.220,11.250,11.274,11.292,11.307,
     * 10.982,11.037,11.093,11.134,11.170,11.201,11.225,11.248,
     * 10.861,10.927,10.996,11.041,11.090,11.121,11.149,11.170,
     * 10.637,10.721,10.808,10.874,10.940,10.980,11.013,11.037,
     * 10.290,10.356,10.427,10.515,10.602,10.669,10.727,10.768,
     * 10.193,10.233,10.281,10.348,10.417,10.483,10.539,10.590,
     * 10.107,10.137,10.176,10.220,10.265,10.299,10.318,10.352,
     * 9.901,9.938,9.986,10.041,10.100,10.124,10.130,10.137,
     * 9.502,9.498,9.507,9.533,9.560,9.551,9.524,9.501,
     * 9.446,9.441,9.449,9.476,9.504,9.494,9.467,9.444,
     * 9.170,9.167,9.176,9.204,9.233,9.225,9.199,9.179,
     * 9.125,9.122,9.130,9.158,9.187,9.180,9.153,9.133/
      DATA ((EDEN(I,4,11,L,3),I=41,48),L=1,NZEN)/
     * 11.093,11.137,11.179,11.207,11.233,11.255,11.281,11.303,
     * 11.004,11.053,11.100,11.130,11.158,11.188,11.215,11.241,
     * 10.889,10.947,11.004,11.041,11.079,11.107,11.134,11.161,
     * 10.667,10.743,10.820,10.874,10.928,10.964,10.997,11.025,
     * 10.286,10.352,10.423,10.502,10.582,10.642,10.700,10.744,
     * 10.170,10.210,10.253,10.312,10.371,10.435,10.493,10.549,
     * 10.079,10.100,10.130,10.155,10.185,10.207,10.223,10.260,
     * 9.879,9.904,9.940,9.975,10.009,10.017,10.013,10.013,
     * 9.490,9.481,9.486,9.502,9.520,9.504,9.474,9.453,
     * 9.431,9.425,9.428,9.444,9.461,9.446,9.417,9.394,
     * 9.158,9.149,9.155,9.173,9.190,9.176,9.149,9.127,
     * 9.113,9.103,9.110,9.127,9.145,9.130,9.103,9.081/
      DATA ((EDEN(I,5,11,L,3),I=41,48),L=1,NZEN)/
     * 11.097,11.143,11.182,11.201,11.220,11.243,11.272,11.299,
     * 11.029,11.072,11.111,11.130,11.149,11.170,11.196,11.228,
     * 10.916,10.967,11.017,11.041,11.064,11.090,11.117,11.149,
     * 10.696,10.766,10.833,10.875,10.916,10.946,10.979,11.013,
     * 10.288,10.356,10.423,10.496,10.568,10.623,10.680,10.728,
     * 10.152,10.188,10.225,10.286,10.346,10.408,10.470,10.531,
     * 10.049,10.061,10.079,10.104,10.130,10.158,10.182,10.233,
     * 9.853,9.867,9.888,9.921,9.953,9.963,9.965,9.976,
     * 9.480,9.468,9.465,9.474,9.484,9.468,9.444,9.431,
     * 9.422,9.410,9.408,9.418,9.427,9.412,9.386,9.371,
     * 9.146,9.137,9.134,9.143,9.155,9.140,9.117,9.104,
     * 9.100,9.091,9.088,9.097,9.110,9.094,9.072,9.058/
      DATA ((EDEN(I,1,12,L,3),I=41,48),L=1,NZEN)/
     * 11.093,11.143,11.199,11.246,11.290,11.320,11.332,11.338,
     * 10.965,11.029,11.097,11.152,11.207,11.243,11.262,11.274,
     * 10.845,10.918,10.998,11.064,11.127,11.164,11.185,11.196,
     * 10.622,10.712,10.810,10.893,10.975,11.021,11.049,11.064,
     * 10.288,10.358,10.438,10.538,10.638,10.711,10.763,10.793,
     * 10.201,10.243,10.299,10.377,10.453,10.522,10.572,10.614,
     * 10.117,10.149,10.196,10.250,10.305,10.336,10.344,10.365,
     * 9.909,9.949,10.004,10.072,10.137,10.158,10.152,10.146,
     * 9.498,9.498,9.516,9.556,9.598,9.591,9.555,9.520,
     * 9.441,9.441,9.459,9.500,9.542,9.535,9.500,9.462,
     * 9.164,9.167,9.188,9.230,9.272,9.267,9.233,9.199,
     * 9.119,9.122,9.142,9.185,9.226,9.221,9.187,9.153/
      DATA ((EDEN(I,2,12,L,3),I=41,48),L=1,NZEN)/
     * 11.114,11.155,11.199,11.236,11.274,11.299,11.318,11.326,
     * 10.969,11.029,11.093,11.143,11.188,11.223,11.246,11.267,
     * 10.848,10.920,10.995,11.053,11.107,11.146,11.170,11.190,
     * 10.624,10.713,10.807,10.881,10.956,11.000,11.033,11.057,
     * 10.297,10.362,10.433,10.525,10.618,10.688,10.744,10.782,
     * 10.210,10.250,10.299,10.365,10.435,10.501,10.553,10.601,
     * 10.127,10.155,10.196,10.243,10.290,10.318,10.328,10.354,
     * 9.918,9.955,10.004,10.061,10.121,10.140,10.137,10.137,
     * 9.508,9.505,9.519,9.551,9.584,9.576,9.543,9.515,
     * 9.450,9.449,9.462,9.494,9.528,9.519,9.487,9.458,
     * 9.176,9.176,9.190,9.223,9.258,9.250,9.220,9.193,
     * 9.130,9.130,9.145,9.177,9.212,9.205,9.174,9.147/
      DATA ((EDEN(I,3,12,L,3),I=41,48),L=1,NZEN)/
     * 11.114,11.155,11.196,11.223,11.253,11.276,11.299,11.318,
     * 10.986,11.041,11.100,11.137,11.173,11.204,11.233,11.258,
     * 10.866,10.933,11.000,11.049,11.093,11.127,11.155,11.179,
     * 10.642,10.727,10.813,10.877,10.943,10.985,11.021,11.045,
     * 10.292,10.362,10.433,10.519,10.606,10.673,10.732,10.775,
     * 10.196,10.241,10.286,10.352,10.420,10.486,10.544,10.599,
     * 10.111,10.143,10.179,10.223,10.267,10.299,10.320,10.358,
     * 9.905,9.942,9.989,10.045,10.100,10.124,10.130,10.143,
     * 9.505,9.501,9.509,9.534,9.560,9.553,9.526,9.508,
     * 9.447,9.442,9.452,9.477,9.504,9.496,9.470,9.450,
     * 9.173,9.170,9.179,9.207,9.236,9.228,9.204,9.185,
     * 9.127,9.125,9.133,9.161,9.190,9.182,9.158,9.139/
      DATA ((EDEN(I,4,12,L,3),I=41,48),L=1,NZEN)/
     * 11.097,11.143,11.185,11.210,11.233,11.258,11.288,11.316,
     * 11.013,11.061,11.104,11.134,11.161,11.190,11.223,11.253,
     * 10.897,10.956,11.013,11.049,11.083,11.111,11.140,11.173,
     * 10.675,10.752,10.827,10.879,10.930,10.968,11.004,11.037,
     * 10.292,10.362,10.431,10.508,10.585,10.647,10.706,10.755,
     * 10.176,10.217,10.258,10.316,10.375,10.439,10.500,10.561,
     * 10.086,10.107,10.134,10.158,10.185,10.210,10.225,10.270,
     * 9.885,9.911,9.944,9.977,10.009,10.017,10.017,10.021,
     * 9.491,9.484,9.487,9.501,9.517,9.504,9.479,9.462,
     * 9.433,9.427,9.430,9.444,9.461,9.447,9.420,9.403,
     * 9.158,9.152,9.158,9.173,9.190,9.176,9.152,9.134,
     * 9.113,9.107,9.113,9.127,9.145,9.130,9.107,9.088/
      DATA ((EDEN(I,5,12,L,3),I=41,48),L=1,NZEN)/
     * 11.093,11.143,11.182,11.201,11.220,11.246,11.279,11.316,
     * 11.037,11.079,11.117,11.134,11.152,11.176,11.207,11.243,
     * 10.926,10.978,11.025,11.045,11.068,11.093,11.127,11.164,
     * 10.708,10.777,10.843,10.881,10.920,10.951,10.988,11.025,
     * 10.297,10.367,10.435,10.504,10.573,10.629,10.688,10.743,
     * 10.158,10.196,10.233,10.290,10.350,10.413,10.479,10.545,
     * 10.053,10.068,10.079,10.104,10.127,10.158,10.188,10.241,
     * 9.857,9.872,9.890,9.919,9.948,9.958,9.963,9.980,
     * 9.479,9.470,9.467,9.473,9.481,9.467,9.446,9.441,
     * 9.422,9.413,9.410,9.417,9.425,9.410,9.389,9.382,
     * 9.146,9.137,9.134,9.143,9.152,9.140,9.121,9.114,
     * 9.100,9.091,9.088,9.097,9.107,9.094,9.075,9.068/
      DATA ((EDEN(I,1,1,L,1),I=49,56),L=1,NZEN)/
     * 11.173,11.164,11.152,11.127,11.111,11.104,11.121,11.114,
     * 11.124,11.121,11.117,11.093,11.079,11.072,11.090,11.083,
     * 11.064,11.064,11.064,11.045,11.033,11.029,11.045,11.033,
     * 10.965,10.963,10.970,10.957,10.947,10.943,10.957,10.945,
     * 10.762,10.766,10.784,10.775,10.767,10.764,10.772,10.754,
     * 10.613,10.631,10.669,10.666,10.663,10.657,10.659,10.636,
     * 10.318,10.360,10.441,10.462,10.481,10.483,10.486,10.459,
     * 10.061,10.049,10.061,10.068,10.068,10.072,10.068,10.061,
     * 9.604,9.569,9.536,9.528,9.510,9.527,9.528,9.546,
     * 9.551,9.513,9.474,9.441,9.398,9.352,9.288,9.217,
     * 9.288,9.253,9.217,9.185,9.146,9.104,9.045,8.978,
     * 9.242,9.207,9.172,9.139,9.100,9.058,9.000,8.932/
      DATA ((EDEN(I,2,1,L,1),I=49,56),L=1,NZEN)/
     * 11.173,11.167,11.155,11.134,11.114,11.100,11.114,11.111,
     * 11.127,11.130,11.124,11.100,11.083,11.072,11.086,11.079,
     * 11.068,11.076,11.072,11.053,11.037,11.025,11.037,11.029,
     * 10.965,10.973,10.978,10.964,10.952,10.942,10.951,10.941,
     * 10.762,10.775,10.792,10.782,10.772,10.761,10.764,10.749,
     * 10.613,10.638,10.678,10.672,10.666,10.654,10.651,10.631,
     * 10.316,10.364,10.452,10.468,10.483,10.479,10.477,10.452,
     * 10.061,10.057,10.076,10.083,10.079,10.076,10.061,10.049,
     * 9.605,9.579,9.557,9.565,9.562,9.585,9.596,9.608,
     * 9.551,9.521,9.483,9.449,9.403,9.350,9.286,9.212,
     * 9.288,9.260,9.225,9.196,9.155,9.104,9.041,8.972,
     * 9.242,9.214,9.180,9.150,9.110,9.058,8.996,8.926/
      DATA ((EDEN(I,3,1,L,1),I=49,56),L=1,NZEN)/
     * 11.173,11.176,11.170,11.146,11.127,11.107,11.117,11.121,
     * 11.124,11.134,11.137,11.114,11.097,11.076,11.086,11.086,
     * 11.064,11.079,11.083,11.064,11.049,11.029,11.037,11.037,
     * 10.964,10.979,10.988,10.973,10.960,10.944,10.951,10.947,
     * 10.760,10.780,10.800,10.790,10.779,10.763,10.763,10.754,
     * 10.614,10.645,10.683,10.679,10.674,10.657,10.651,10.638,
     * 10.328,10.384,10.450,10.470,10.486,10.480,10.479,10.461,
     * 10.083,10.093,10.124,10.127,10.124,10.114,10.100,10.086,
     * 9.605,9.591,9.588,9.594,9.589,9.608,9.617,9.631,
     * 9.551,9.531,9.517,9.483,9.439,9.382,9.316,9.250,
     * 9.288,9.270,9.260,9.228,9.188,9.134,9.072,9.009,
     * 9.242,9.224,9.214,9.182,9.142,9.088,9.026,8.963/
      DATA ((EDEN(I,4,1,L,1),I=49,56),L=1,NZEN)/
     * 11.193,11.204,11.204,11.182,11.161,11.134,11.137,11.143,
     * 11.130,11.149,11.167,11.146,11.127,11.100,11.104,11.107,
     * 11.072,11.093,11.111,11.093,11.079,11.053,11.053,11.057,
     * 10.969,10.992,11.009,10.997,10.984,10.960,10.959,10.958,
     * 10.760,10.787,10.812,10.802,10.790,10.764,10.757,10.748,
     * 10.606,10.645,10.679,10.678,10.674,10.647,10.633,10.619,
     * 10.292,10.356,10.398,10.427,10.447,10.435,10.430,10.412,
     * 9.978,9.984,9.986,9.973,9.949,9.922,9.890,9.875,
     * 9.583,9.575,9.584,9.544,9.493,9.440,9.379,9.348,
     * 9.529,9.515,9.525,9.480,9.422,9.339,9.250,9.162,
     * 9.262,9.253,9.262,9.217,9.161,9.083,8.998,8.915,
     * 9.217,9.207,9.217,9.172,9.116,9.037,8.953,8.869/
      DATA ((EDEN(I,5,1,L,1),I=49,56),L=1,NZEN)/
     * 11.199,11.217,11.217,11.196,11.176,11.137,11.134,11.143,
     * 11.127,11.155,11.185,11.161,11.140,11.104,11.100,11.104,
     * 11.068,11.097,11.127,11.107,11.090,11.053,11.045,11.053,
     * 10.965,10.997,11.025,11.009,10.993,10.959,10.951,10.955,
     * 10.754,10.789,10.823,10.812,10.799,10.763,10.747,10.743,
     * 10.602,10.647,10.686,10.685,10.679,10.642,10.622,10.612,
     * 10.290,10.358,10.394,10.425,10.449,10.427,10.413,10.400,
     * 9.967,9.985,9.983,9.968,9.942,9.908,9.874,9.857,
     * 9.577,9.582,9.585,9.532,9.466,9.392,9.312,9.282,
     * 9.520,9.517,9.543,9.494,9.433,9.340,9.243,9.158,
     * 9.258,9.255,9.279,9.233,9.173,9.086,8.992,8.912,
     * 9.212,9.210,9.233,9.187,9.127,9.041,8.946,8.866/
      DATA ((EDEN(I,1,2,L,1),I=49,56),L=1,NZEN)/
     * 11.173,11.164,11.152,11.127,11.111,11.104,11.124,11.114,
     * 11.121,11.117,11.117,11.097,11.079,11.072,11.090,11.083,
     * 11.064,11.061,11.064,11.045,11.033,11.029,11.045,11.037,
     * 10.961,10.961,10.969,10.957,10.948,10.945,10.960,10.946,
     * 10.760,10.763,10.781,10.773,10.766,10.764,10.773,10.754,
     * 10.613,10.629,10.666,10.663,10.660,10.656,10.660,10.636,
     * 10.322,10.362,10.436,10.456,10.476,10.480,10.484,10.456,
     * 10.061,10.049,10.061,10.064,10.064,10.068,10.061,10.053,
     * 9.595,9.562,9.535,9.525,9.507,9.523,9.523,9.539,
     * 9.542,9.505,9.473,9.439,9.398,9.350,9.288,9.215,
     * 9.279,9.246,9.217,9.185,9.146,9.100,9.041,8.975,
     * 9.233,9.200,9.172,9.139,9.100,9.055,8.996,8.929/
      DATA ((EDEN(I,2,2,L,1),I=49,56),L=1,NZEN)/
     * 11.170,11.167,11.158,11.137,11.117,11.107,11.121,11.114,
     * 11.121,11.124,11.124,11.104,11.086,11.076,11.090,11.083,
     * 11.061,11.068,11.072,11.057,11.041,11.033,11.045,11.033,
     * 10.960,10.967,10.977,10.965,10.955,10.948,10.959,10.945,
     * 10.758,10.769,10.790,10.781,10.773,10.766,10.771,10.752,
     * 10.612,10.634,10.673,10.670,10.666,10.657,10.657,10.633,
     * 10.320,10.365,10.442,10.462,10.480,10.479,10.480,10.452,
     * 10.061,10.057,10.072,10.076,10.072,10.068,10.057,10.045,
     * 9.594,9.568,9.552,9.552,9.542,9.560,9.562,9.572,
     * 9.540,9.511,9.483,9.449,9.405,9.354,9.288,9.212,
     * 9.276,9.250,9.225,9.196,9.155,9.107,9.041,8.970,
     * 9.231,9.205,9.180,9.150,9.110,9.061,8.996,8.925/
      DATA ((EDEN(I,3,2,L,1),I=49,56),L=1,NZEN)/
     * 11.170,11.173,11.170,11.149,11.130,11.117,11.127,11.124,
     * 11.114,11.124,11.137,11.117,11.100,11.086,11.093,11.090,
     * 11.057,11.068,11.083,11.064,11.053,11.037,11.049,11.041,
     * 10.955,10.968,10.985,10.973,10.963,10.951,10.959,10.951,
     * 10.754,10.772,10.796,10.789,10.781,10.770,10.772,10.759,
     * 10.610,10.639,10.678,10.678,10.675,10.663,10.660,10.643,
     * 10.330,10.384,10.439,10.465,10.486,10.484,10.486,10.465,
     * 10.083,10.093,10.114,10.124,10.124,10.117,10.100,10.090,
     * 9.591,9.577,9.580,9.578,9.564,9.578,9.576,9.593,
     * 9.538,9.519,9.515,9.483,9.441,9.387,9.320,9.253,
     * 9.276,9.260,9.258,9.228,9.188,9.140,9.076,9.013,
     * 9.231,9.214,9.212,9.182,9.142,9.094,9.030,8.967/
      DATA ((EDEN(I,4,2,L,1),I=49,56),L=1,NZEN)/
     * 11.182,11.193,11.193,11.176,11.158,11.137,11.143,11.140,
     * 11.117,11.137,11.158,11.140,11.124,11.104,11.111,11.107,
     * 11.061,11.079,11.104,11.090,11.076,11.057,11.061,11.053,
     * 10.956,10.978,11.004,10.993,10.983,10.965,10.967,10.959,
     * 10.750,10.775,10.806,10.799,10.790,10.771,10.765,10.751,
     * 10.601,10.636,10.675,10.676,10.674,10.654,10.644,10.624,
     * 10.297,10.358,10.398,10.430,10.456,10.449,10.446,10.422,
     * 9.989,9.999,10.004,9.997,9.980,9.961,9.934,9.919,
     * 9.569,9.562,9.573,9.535,9.485,9.441,9.386,9.366,
     * 9.515,9.501,9.521,9.480,9.425,9.352,9.267,9.179,
     * 9.250,9.238,9.258,9.220,9.170,9.100,9.017,8.932,
     * 9.205,9.192,9.212,9.174,9.125,9.055,8.971,8.887/
      DATA ((EDEN(I,5,2,L,1),I=49,56),L=1,NZEN)/
     * 11.185,11.201,11.207,11.185,11.167,11.137,11.134,11.134,
     * 11.111,11.137,11.170,11.152,11.134,11.104,11.100,11.100,
     * 11.053,11.079,11.114,11.097,11.083,11.057,11.049,11.049,
     * 10.951,10.979,11.013,11.000,10.989,10.963,10.957,10.952,
     * 10.742,10.775,10.814,10.807,10.797,10.769,10.755,10.744,
     * 10.594,10.635,10.680,10.682,10.680,10.652,10.633,10.617,
     * 10.292,10.358,10.401,10.433,10.459,10.444,10.435,10.413,
     * 9.978,10.004,10.004,10.004,9.989,9.968,9.939,9.922,
     * 9.561,9.567,9.558,9.509,9.448,9.393,9.327,9.315,
     * 9.504,9.501,9.531,9.490,9.435,9.356,9.262,9.179,
     * 9.243,9.241,9.267,9.228,9.176,9.100,9.013,8.932,
     * 9.197,9.195,9.221,9.182,9.130,9.055,8.967,8.887/
      DATA ((EDEN(I,1,3,L,1),I=49,56),L=1,NZEN)/
     * 11.170,11.161,11.146,11.124,11.107,11.100,11.121,11.114,
     * 11.117,11.114,11.114,11.093,11.079,11.072,11.090,11.083,
     * 11.057,11.057,11.064,11.045,11.033,11.029,11.045,11.033,
     * 10.958,10.957,10.968,10.955,10.946,10.944,10.960,10.946,
     * 10.757,10.760,10.778,10.771,10.765,10.763,10.773,10.754,
     * 10.611,10.626,10.662,10.660,10.657,10.654,10.659,10.635,
     * 10.322,10.362,10.431,10.453,10.473,10.477,10.483,10.453,
     * 10.057,10.049,10.057,10.061,10.061,10.064,10.057,10.049,
     * 9.590,9.556,9.532,9.526,9.511,9.528,9.529,9.545,
     * 9.537,9.501,9.470,9.436,9.393,9.346,9.283,9.210,
     * 9.274,9.238,9.212,9.182,9.140,9.097,9.037,8.971,
     * 9.228,9.192,9.166,9.136,9.094,9.051,8.992,8.925/
      DATA ((EDEN(I,2,3,L,1),I=49,56),L=1,NZEN)/
     * 11.164,11.161,11.155,11.134,11.121,11.114,11.127,11.121,
     * 11.111,11.114,11.121,11.104,11.090,11.083,11.097,11.086,
     * 11.053,11.057,11.068,11.053,11.045,11.037,11.053,11.037,
     * 10.951,10.957,10.974,10.964,10.957,10.953,10.965,10.948,
     * 10.751,10.760,10.783,10.778,10.774,10.771,10.777,10.756,
     * 10.605,10.627,10.666,10.666,10.665,10.661,10.662,10.636,
     * 10.318,10.362,10.431,10.456,10.479,10.481,10.484,10.453,
     * 10.053,10.049,10.061,10.064,10.064,10.064,10.053,10.041,
     * 9.582,9.556,9.544,9.539,9.524,9.538,9.534,9.542,
     * 9.528,9.500,9.479,9.447,9.405,9.356,9.290,9.212,
     * 9.265,9.238,9.220,9.193,9.152,9.107,9.045,8.970,
     * 9.219,9.192,9.174,9.147,9.107,9.061,9.000,8.925/
      DATA ((EDEN(I,3,3,L,1),I=49,56),L=1,NZEN)/
     * 11.155,11.161,11.161,11.146,11.134,11.124,11.137,11.130,
     * 11.100,11.107,11.124,11.111,11.100,11.093,11.104,11.097,
     * 11.041,11.053,11.072,11.061,11.053,11.049,11.057,11.049,
     * 10.942,10.953,10.975,10.969,10.965,10.962,10.971,10.958,
     * 10.741,10.758,10.786,10.785,10.783,10.780,10.783,10.766,
     * 10.600,10.627,10.667,10.673,10.677,10.673,10.671,10.650,
     * 10.326,10.375,10.430,10.461,10.489,10.494,10.497,10.473,
     * 10.076,10.090,10.104,10.121,10.127,10.127,10.111,10.097,
     * 9.575,9.561,9.563,9.555,9.534,9.549,9.544,9.563,
     * 9.521,9.502,9.507,9.481,9.444,9.398,9.332,9.262,
     * 9.260,9.243,9.248,9.225,9.190,9.149,9.086,9.021,
     * 9.214,9.197,9.202,9.180,9.145,9.103,9.041,8.975/
      DATA ((EDEN(I,4,3,L,1),I=49,56),L=1,NZEN)/
     * 11.158,11.170,11.176,11.164,11.152,11.143,11.149,11.140,
     * 11.097,11.111,11.140,11.130,11.121,11.111,11.117,11.104,
     * 11.037,11.053,11.086,11.076,11.072,11.061,11.068,11.053,
     * 10.936,10.954,10.986,10.983,10.980,10.972,10.975,10.960,
     * 10.732,10.754,10.790,10.791,10.790,10.782,10.778,10.758,
     * 10.588,10.619,10.663,10.672,10.677,10.669,10.661,10.634,
     * 10.297,10.352,10.398,10.436,10.468,10.473,10.471,10.441,
     * 9.999,10.013,10.021,10.029,10.029,10.025,10.004,9.986,
     * 9.550,9.543,9.544,9.510,9.463,9.444,9.403,9.401,
     * 9.496,9.483,9.505,9.474,9.431,9.375,9.294,9.210,
     * 9.233,9.223,9.243,9.217,9.179,9.124,9.045,8.964,
     * 9.187,9.177,9.197,9.172,9.133,9.078,9.000,8.918/
      DATA ((EDEN(I,5,3,L,1),I=49,56),L=1,NZEN)/
     * 11.152,11.170,11.179,11.167,11.158,11.143,11.143,11.134,
     * 11.086,11.104,11.143,11.134,11.124,11.111,11.107,11.100,
     * 11.025,11.049,11.086,11.083,11.076,11.061,11.061,11.049,
     * 10.925,10.949,10.988,10.986,10.983,10.972,10.968,10.954,
     * 10.719,10.747,10.794,10.797,10.796,10.784,10.773,10.754,
     * 10.576,10.614,10.667,10.679,10.685,10.672,10.656,10.631,
     * 10.286,10.346,10.408,10.447,10.480,10.480,10.471,10.442,
     * 9.990,10.017,10.029,10.045,10.053,10.057,10.037,10.021,
     * 9.538,9.544,9.506,9.471,9.423,9.414,9.379,9.394,
     * 9.481,9.479,9.507,9.480,9.441,9.384,9.301,9.220,
     * 9.220,9.220,9.246,9.223,9.188,9.134,9.053,8.977,
     * 9.174,9.174,9.200,9.177,9.142,9.088,9.007,8.932/
      DATA ((EDEN(I,1,4,L,1),I=49,56),L=1,NZEN)/
     * 11.161,11.152,11.137,11.114,11.100,11.093,11.111,11.104,
     * 11.114,11.107,11.107,11.086,11.068,11.064,11.083,11.072,
     * 11.057,11.053,11.057,11.037,11.025,11.021,11.037,11.025,
     * 10.955,10.954,10.963,10.950,10.941,10.938,10.952,10.937,
     * 10.754,10.758,10.775,10.767,10.760,10.759,10.768,10.748,
     * 10.610,10.625,10.662,10.658,10.654,10.651,10.655,10.630,
     * 10.320,10.360,10.438,10.456,10.474,10.477,10.481,10.452,
     * 10.053,10.041,10.053,10.061,10.061,10.064,10.061,10.053,
     * 9.585,9.553,9.527,9.530,9.524,9.547,9.556,9.570,
     * 9.533,9.496,9.456,9.425,9.384,9.336,9.274,9.201,
     * 9.270,9.236,9.201,9.173,9.134,9.090,9.033,8.964,
     * 9.224,9.190,9.156,9.127,9.088,9.044,8.988,8.919/
      DATA ((EDEN(I,2,4,L,1),I=49,56),L=1,NZEN)/
     * 11.152,11.149,11.143,11.127,11.117,11.114,11.130,11.114,
     * 11.100,11.100,11.111,11.097,11.086,11.083,11.100,11.083,
     * 11.041,11.045,11.057,11.049,11.041,11.041,11.053,11.037,
     * 10.941,10.945,10.964,10.958,10.955,10.955,10.967,10.947,
     * 10.741,10.750,10.776,10.775,10.773,10.775,10.780,10.756,
     * 10.600,10.619,10.659,10.663,10.667,10.667,10.667,10.637,
     * 10.312,10.356,10.430,10.458,10.481,10.489,10.491,10.456,
     * 10.045,10.037,10.049,10.061,10.064,10.072,10.061,10.049,
     * 9.569,9.544,9.529,9.523,9.507,9.522,9.514,9.523,
     * 9.516,9.486,9.468,9.439,9.401,9.358,9.290,9.212,
     * 9.255,9.225,9.210,9.185,9.149,9.111,9.045,8.971,
     * 9.210,9.180,9.164,9.139,9.103,9.065,9.000,8.925/
      DATA ((EDEN(I,3,4,L,1),I=49,56),L=1,NZEN)/
     * 11.137,11.140,11.143,11.134,11.130,11.130,11.143,11.130,
     * 11.079,11.086,11.107,11.100,11.100,11.100,11.114,11.097,
     * 11.025,11.029,11.057,11.053,11.053,11.057,11.068,11.049,
     * 10.924,10.932,10.960,10.961,10.964,10.970,10.980,10.960,
     * 10.727,10.740,10.773,10.780,10.785,10.791,10.794,10.770,
     * 10.589,10.613,10.656,10.669,10.680,10.686,10.684,10.657,
     * 10.318,10.364,10.423,10.462,10.496,10.512,10.515,10.486,
     * 10.064,10.076,10.090,10.114,10.130,10.143,10.130,10.114,
     * 9.554,9.540,9.535,9.527,9.507,9.530,9.524,9.548,
     * 9.502,9.481,9.487,9.470,9.441,9.407,9.342,9.272,
     * 9.243,9.223,9.230,9.215,9.190,9.158,9.097,9.029,
     * 9.197,9.177,9.185,9.169,9.145,9.113,9.051,8.984/
      DATA ((EDEN(I,4,4,L,1),I=49,56),L=1,NZEN)/
     * 11.124,11.134,11.143,11.143,11.143,11.149,11.158,11.140,
     * 11.064,11.076,11.111,11.111,11.111,11.117,11.124,11.107,
     * 11.009,11.021,11.057,11.057,11.064,11.068,11.076,11.057,
     * 10.907,10.922,10.958,10.966,10.975,10.983,10.987,10.964,
     * 10.708,10.726,10.769,10.781,10.792,10.799,10.797,10.769,
     * 10.571,10.598,10.649,10.668,10.684,10.691,10.684,10.651,
     * 10.290,10.342,10.403,10.449,10.490,10.509,10.508,10.471,
     * 10.009,10.029,10.041,10.068,10.086,10.100,10.086,10.064,
     * 9.526,9.519,9.493,9.477,9.447,9.469,9.455,9.472,
     * 9.473,9.459,9.480,9.465,9.439,9.403,9.330,9.246,
     * 9.212,9.201,9.220,9.210,9.185,9.152,9.083,9.004,
     * 9.166,9.156,9.174,9.164,9.139,9.107,9.037,8.959/
      DATA ((EDEN(I,5,4,L,1),I=49,56),L=1,NZEN)/
     * 11.111,11.121,11.137,11.140,11.146,11.152,11.152,11.134,
     * 11.045,11.061,11.100,11.107,11.114,11.121,11.121,11.100,
     * 10.991,11.009,11.049,11.057,11.068,11.076,11.072,11.053,
     * 10.890,10.909,10.951,10.965,10.978,10.988,10.986,10.962,
     * 10.690,10.713,10.765,10.783,10.797,10.808,10.800,10.772,
     * 10.551,10.585,10.649,10.674,10.693,10.703,10.692,10.659,
     * 10.276,10.332,10.415,10.465,10.509,10.529,10.522,10.489,
     * 9.994,10.029,10.049,10.097,10.130,10.158,10.146,10.127,
     * 9.507,9.511,9.425,9.427,9.416,9.472,9.486,9.523,
     * 9.453,9.450,9.470,9.467,9.450,9.425,9.354,9.281,
     * 9.193,9.193,9.212,9.212,9.199,9.176,9.111,9.037,
     * 9.147,9.147,9.166,9.166,9.153,9.130,9.065,8.992/
      DATA ((EDEN(I,1,5,L,1),I=49,56),L=1,NZEN)/
     * 11.158,11.146,11.130,11.104,11.083,11.076,11.093,11.083,
     * 11.111,11.107,11.097,11.072,11.057,11.049,11.064,11.053,
     * 11.053,11.053,11.049,11.029,11.013,11.004,11.021,11.004,
     * 10.952,10.952,10.957,10.943,10.932,10.926,10.939,10.921,
     * 10.754,10.758,10.775,10.764,10.755,10.753,10.761,10.737,
     * 10.612,10.627,10.665,10.659,10.652,10.646,10.648,10.622,
     * 10.324,10.364,10.455,10.470,10.481,10.480,10.481,10.449,
     * 10.049,10.037,10.053,10.064,10.068,10.079,10.076,10.064,
     * 9.580,9.548,9.519,9.534,9.539,9.572,9.588,9.605,
     * 9.526,9.490,9.441,9.408,9.367,9.324,9.265,9.196,
     * 9.265,9.230,9.188,9.158,9.121,9.083,9.025,8.958,
     * 9.219,9.185,9.142,9.113,9.075,9.037,8.980,8.912/
      DATA ((EDEN(I,2,5,L,1),I=49,56),L=1,NZEN)/
     * 11.143,11.137,11.130,11.117,11.107,11.111,11.124,11.104,
     * 11.090,11.090,11.097,11.086,11.079,11.079,11.093,11.072,
     * 11.033,11.033,11.049,11.037,11.033,11.037,11.049,11.025,
     * 10.933,10.936,10.955,10.951,10.950,10.953,10.963,10.939,
     * 10.736,10.742,10.769,10.771,10.772,10.777,10.781,10.751,
     * 10.595,10.613,10.658,10.664,10.668,10.670,10.669,10.635,
     * 10.314,10.354,10.436,10.467,10.493,10.501,10.500,10.459,
     * 10.037,10.029,10.037,10.057,10.068,10.083,10.079,10.064,
     * 9.558,9.530,9.507,9.506,9.493,9.514,9.508,9.520,
     * 9.504,9.473,9.449,9.425,9.391,9.354,9.288,9.210,
     * 9.241,9.212,9.193,9.173,9.143,9.107,9.045,8.970,
     * 9.195,9.166,9.147,9.127,9.097,9.061,9.000,8.924/
      DATA ((EDEN(I,3,5,L,1),I=49,56),L=1,NZEN)/
     * 11.121,11.121,11.124,11.121,11.124,11.134,11.146,11.127,
     * 11.064,11.064,11.090,11.090,11.093,11.104,11.114,11.093,
     * 11.009,11.013,11.037,11.041,11.049,11.061,11.068,11.045,
     * 10.908,10.914,10.943,10.952,10.961,10.975,10.984,10.959,
     * 10.714,10.724,10.760,10.774,10.785,10.799,10.802,10.773,
     * 10.580,10.600,10.648,10.667,10.684,10.697,10.695,10.662,
     * 10.312,10.356,10.427,10.471,10.511,10.531,10.531,10.496,
     * 10.053,10.064,10.079,10.114,10.140,10.164,10.152,10.137,
     * 9.537,9.521,9.504,9.507,9.496,9.537,9.539,9.567,
     * 9.483,9.461,9.465,9.456,9.436,9.412,9.350,9.276,
     * 9.225,9.204,9.210,9.204,9.188,9.164,9.104,9.037,
     * 9.180,9.158,9.164,9.158,9.142,9.119,9.058,8.992/
      DATA ((EDEN(I,4,5,L,1),I=49,56),L=1,NZEN)/
     * 11.093,11.100,11.114,11.124,11.137,11.152,11.161,11.137,
     * 11.041,11.045,11.079,11.090,11.104,11.124,11.130,11.104,
     * 10.984,10.992,11.025,11.041,11.061,11.079,11.086,11.057,
     * 10.885,10.895,10.932,10.952,10.972,10.993,10.997,10.967,
     * 10.690,10.705,10.750,10.773,10.793,10.815,10.814,10.778,
     * 10.555,10.580,10.636,10.666,10.691,10.712,10.706,10.667,
     * 10.286,10.334,10.410,10.465,10.513,10.543,10.540,10.497,
     * 10.013,10.033,10.049,10.093,10.130,10.161,10.146,10.127,
     * 9.505,9.496,9.462,9.478,9.481,9.537,9.545,9.567,
     * 9.452,9.438,9.452,9.453,9.444,9.425,9.358,9.276,
     * 9.193,9.182,9.198,9.203,9.193,9.179,9.114,9.037,
     * 9.147,9.136,9.150,9.156,9.147,9.133,9.068,8.992/
      DATA ((EDEN(I,5,5,L,1),I=49,56),L=1,NZEN)/
     * 11.064,11.072,11.098,11.115,11.137,11.164,11.167,11.140,
     * 11.013,11.025,11.062,11.084,11.107,11.134,11.137,11.107,
     * 10.958,10.971,11.009,11.037,11.061,11.090,11.090,11.061,
     * 10.859,10.874,10.918,10.947,10.975,11.004,11.004,10.974,
     * 10.665,10.684,10.738,10.772,10.803,10.833,10.828,10.793,
     * 10.531,10.561,10.630,10.670,10.704,10.733,10.724,10.687,
     * 10.267,10.322,10.417,10.480,10.537,10.575,10.571,10.531,
     * 9.998,10.037,10.064,10.134,10.188,10.236,10.228,10.207,
     * 9.479,9.480,9.413,9.464,9.500,9.592,9.623,9.659,
     * 9.425,9.422,9.438,9.458,9.464,9.464,9.403,9.332,
     * 9.167,9.167,9.182,9.207,9.217,9.220,9.161,9.093,
     * 9.122,9.122,9.136,9.161,9.172,9.174,9.116,9.048/
      DATA ((EDEN(I,1,6,L,1),I=49,56),L=1,NZEN)/
     * 11.149,11.137,11.117,11.093,11.072,11.061,11.079,11.064,
     * 11.111,11.104,11.086,11.061,11.041,11.033,11.049,11.033,
     * 11.053,11.049,11.041,11.017,11.000,10.993,11.004,10.989,
     * 10.952,10.951,10.953,10.936,10.921,10.915,10.926,10.905,
     * 10.757,10.758,10.773,10.761,10.751,10.745,10.751,10.726,
     * 10.616,10.629,10.667,10.658,10.648,10.641,10.642,10.613,
     * 10.332,10.369,10.471,10.480,10.486,10.481,10.480,10.442,
     * 10.045,10.033,10.057,10.072,10.079,10.093,10.093,10.079,
     * 9.574,9.542,9.515,9.539,9.555,9.592,9.614,9.629,
     * 9.520,9.483,9.425,9.396,9.356,9.314,9.255,9.188,
     * 9.258,9.223,9.173,9.146,9.111,9.072,9.017,8.954,
     * 9.212,9.177,9.127,9.100,9.065,9.026,8.971,8.908/
      DATA ((EDEN(I,2,6,L,1),I=49,56),L=1,NZEN)/
     * 11.134,11.127,11.121,11.107,11.097,11.100,11.114,11.090,
     * 11.083,11.079,11.086,11.076,11.072,11.072,11.086,11.061,
     * 11.025,11.025,11.037,11.029,11.025,11.029,11.041,11.013,
     * 10.927,10.928,10.947,10.944,10.943,10.949,10.959,10.929,
     * 10.732,10.737,10.766,10.768,10.769,10.776,10.780,10.745,
     * 10.594,10.610,10.658,10.664,10.667,10.672,10.670,10.632,
     * 10.316,10.354,10.449,10.476,10.500,10.509,10.505,10.461,
     * 10.029,10.021,10.037,10.061,10.076,10.100,10.097,10.079,
     * 9.545,9.519,9.493,9.497,9.492,9.523,9.522,9.536,
     * 9.491,9.459,9.433,9.412,9.382,9.348,9.283,9.204,
     * 9.230,9.201,9.176,9.161,9.134,9.104,9.045,8.969,
     * 9.185,9.156,9.130,9.116,9.088,9.058,9.000,8.924/
      DATA ((EDEN(I,3,6,L,1),I=49,56),L=1,NZEN)/
     * 11.100,11.100,11.107,11.111,11.117,11.134,11.143,11.121,
     * 11.049,11.053,11.076,11.079,11.086,11.104,11.114,11.090,
     * 10.996,10.998,11.025,11.033,11.045,11.061,11.068,11.041,
     * 10.897,10.902,10.931,10.944,10.959,10.977,10.985,10.955,
     * 10.707,10.714,10.753,10.770,10.786,10.806,10.807,10.773,
     * 10.574,10.591,10.643,10.667,10.687,10.705,10.701,10.665,
     * 10.312,10.352,10.430,10.477,10.520,10.545,10.543,10.501,
     * 10.045,10.053,10.072,10.114,10.146,10.176,10.167,10.152,
     * 9.521,9.505,9.484,9.501,9.505,9.559,9.568,9.593,
     * 9.467,9.444,9.446,9.444,9.431,9.416,9.354,9.279,
     * 9.210,9.188,9.190,9.193,9.182,9.170,9.111,9.041,
     * 9.164,9.142,9.145,9.147,9.136,9.125,9.065,8.996/
      DATA ((EDEN(I,4,6,L,1),I=49,56),L=1,NZEN)/
     * 11.068,11.072,11.090,11.111,11.134,11.158,11.167,11.137,
     * 11.021,11.025,11.057,11.079,11.100,11.127,11.137,11.104,
     * 10.967,10.973,11.004,11.029,11.053,11.083,11.090,11.057,
     * 10.868,10.876,10.914,10.942,10.969,11.000,11.004,10.969,
     * 10.678,10.688,10.735,10.768,10.797,10.827,10.824,10.785,
     * 10.547,10.567,10.626,10.664,10.697,10.725,10.718,10.674,
     * 10.286,10.330,10.408,10.470,10.525,10.562,10.558,10.511,
     * 10.004,10.029,10.045,10.100,10.146,10.188,10.173,10.149,
     * 9.486,9.476,9.460,9.499,9.524,9.594,9.605,9.623,
     * 9.433,9.417,9.430,9.444,9.442,9.436,9.369,9.288,
     * 9.173,9.161,9.176,9.190,9.193,9.190,9.127,9.049,
     * 9.127,9.116,9.130,9.145,9.147,9.145,9.081,9.003/
      DATA ((EDEN(I,5,6,L,1),I=49,56),L=1,NZEN)/
     * 11.029,11.037,11.068,11.100,11.134,11.173,11.179,11.146,
     * 10.991,10.998,11.033,11.068,11.104,11.143,11.149,11.114,
     * 10.935,10.945,10.986,11.022,11.057,11.100,11.104,11.068,
     * 10.838,10.849,10.893,10.936,10.975,11.017,11.021,10.982,
     * 10.646,10.663,10.719,10.764,10.806,10.849,10.845,10.804,
     * 10.517,10.543,10.614,10.666,10.710,10.752,10.744,10.701,
     * 10.265,10.314,10.412,10.486,10.551,10.601,10.593,10.549,
     * 9.993,10.033,10.061,10.143,10.210,10.267,10.258,10.233,
     * 9.453,9.455,9.453,9.516,9.563,9.662,9.688,9.717,
     * 9.401,9.398,9.412,9.447,9.467,9.481,9.423,9.350,
     * 9.146,9.146,9.161,9.199,9.223,9.238,9.182,9.111,
     * 9.100,9.100,9.116,9.153,9.177,9.192,9.136,9.065/
      DATA ((EDEN(I,1,7,L,1),I=49,56),L=1,NZEN)/
     * 11.149,11.134,11.117,11.090,11.068,11.057,11.076,11.061,
     * 11.111,11.104,11.086,11.061,11.041,11.029,11.045,11.029,
     * 11.053,11.049,11.037,11.017,10.998,10.989,11.004,10.985,
     * 10.952,10.951,10.952,10.933,10.919,10.912,10.924,10.902,
     * 10.757,10.758,10.772,10.760,10.749,10.744,10.750,10.723,
     * 10.618,10.630,10.668,10.658,10.647,10.640,10.641,10.612,
     * 10.336,10.373,10.474,10.481,10.487,10.481,10.479,10.441,
     * 10.045,10.033,10.061,10.076,10.083,10.097,10.097,10.083,
     * 9.571,9.539,9.516,9.544,9.562,9.599,9.621,9.638,
     * 9.517,9.480,9.423,9.393,9.354,9.312,9.255,9.188,
     * 9.255,9.220,9.170,9.143,9.111,9.072,9.017,8.954,
     * 9.210,9.174,9.125,9.097,9.065,9.026,8.971,8.908/
      DATA ((EDEN(I,2,7,L,1),I=49,56),L=1,NZEN)/
     * 11.134,11.124,11.117,11.104,11.097,11.100,11.114,11.090,
     * 11.083,11.079,11.083,11.072,11.068,11.072,11.083,11.057,
     * 11.025,11.025,11.037,11.029,11.025,11.029,11.037,11.013,
     * 10.926,10.927,10.944,10.943,10.943,10.949,10.957,10.927,
     * 10.732,10.736,10.764,10.766,10.769,10.776,10.779,10.744,
     * 10.595,10.610,10.656,10.663,10.667,10.672,10.669,10.631,
     * 10.318,10.356,10.450,10.477,10.501,10.511,10.505,10.461,
     * 10.025,10.021,10.037,10.064,10.083,10.107,10.104,10.086,
     * 9.540,9.515,9.489,9.499,9.499,9.533,9.536,9.551,
     * 9.487,9.455,9.428,9.408,9.378,9.348,9.286,9.207,
     * 9.225,9.196,9.173,9.158,9.134,9.104,9.041,8.969,
     * 9.180,9.150,9.127,9.113,9.088,9.058,8.996,8.924/
      DATA ((EDEN(I,3,7,L,1),I=49,56),L=1,NZEN)/
     * 11.100,11.097,11.104,11.107,11.117,11.134,11.146,11.121,
     * 11.049,11.049,11.072,11.076,11.086,11.104,11.114,11.090,
     * 10.993,10.995,11.021,11.029,11.045,11.061,11.072,11.041,
     * 10.894,10.899,10.928,10.943,10.960,10.980,10.986,10.955,
     * 10.703,10.711,10.749,10.768,10.786,10.808,10.809,10.774,
     * 10.572,10.589,10.640,10.666,10.687,10.707,10.702,10.665,
     * 10.312,10.352,10.430,10.477,10.521,10.548,10.544,10.502,
     * 10.041,10.049,10.068,10.114,10.146,10.179,10.170,10.152,
     * 9.513,9.500,9.480,9.504,9.516,9.573,9.584,9.609,
     * 9.459,9.438,9.441,9.441,9.430,9.415,9.352,9.279,
     * 9.201,9.182,9.188,9.190,9.182,9.170,9.111,9.037,
     * 9.156,9.136,9.142,9.145,9.136,9.125,9.065,8.992/
      DATA ((EDEN(I,4,7,L,1),I=49,56),L=1,NZEN)/
     * 11.064,11.068,11.086,11.111,11.134,11.164,11.170,11.140,
     * 11.017,11.021,11.053,11.079,11.104,11.134,11.140,11.107,
     * 10.964,10.968,11.004,11.029,11.057,11.086,11.093,11.061,
     * 10.865,10.872,10.911,10.942,10.972,11.004,11.009,10.971,
     * 10.673,10.683,10.731,10.765,10.797,10.830,10.828,10.785,
     * 10.542,10.561,10.621,10.661,10.696,10.728,10.719,10.673,
     * 10.281,10.324,10.405,10.468,10.524,10.562,10.556,10.507,
     * 9.994,10.013,10.029,10.090,10.134,10.176,10.161,10.134,
     * 9.476,9.465,9.454,9.495,9.524,9.594,9.603,9.617,
     * 9.422,9.407,9.423,9.436,9.436,9.430,9.362,9.276,
     * 9.164,9.149,9.167,9.185,9.188,9.185,9.117,9.037,
     * 9.119,9.103,9.122,9.139,9.142,9.139,9.072,8.992/
      DATA ((EDEN(I,5,7,L,1),I=49,56),L=1,NZEN)/
     * 11.025,11.033,11.064,11.100,11.137,11.179,11.185,11.149,
     * 10.985,10.992,11.029,11.068,11.107,11.149,11.152,11.117,
     * 10.930,10.939,10.980,11.021,11.064,11.104,11.107,11.072,
     * 10.832,10.843,10.888,10.935,10.979,11.021,11.025,10.984,
     * 10.641,10.656,10.712,10.763,10.808,10.852,10.847,10.803,
     * 10.512,10.537,10.606,10.662,10.710,10.754,10.744,10.697,
     * 10.258,10.307,10.400,10.479,10.547,10.598,10.589,10.540,
     * 9.976,10.017,10.041,10.124,10.190,10.250,10.238,10.210,
     * 9.442,9.442,9.437,9.503,9.552,9.650,9.673,9.697,
     * 9.391,9.386,9.403,9.441,9.462,9.476,9.412,9.332,
     * 9.134,9.130,9.149,9.190,9.215,9.230,9.170,9.093,
     * 9.088,9.085,9.103,9.145,9.169,9.185,9.125,9.048/
      DATA ((EDEN(I,1,8,L,1),I=49,56),L=1,NZEN)/
     * 11.152,11.140,11.124,11.097,11.076,11.068,11.090,11.076,
     * 11.111,11.107,11.090,11.068,11.049,11.041,11.061,11.045,
     * 11.053,11.053,11.045,11.021,11.004,11.000,11.017,10.999,
     * 10.953,10.952,10.954,10.938,10.926,10.920,10.933,10.913,
     * 10.757,10.759,10.773,10.762,10.752,10.748,10.755,10.731,
     * 10.616,10.629,10.665,10.657,10.649,10.643,10.645,10.617,
     * 10.334,10.371,10.464,10.474,10.483,10.480,10.480,10.446,
     * 10.049,10.041,10.061,10.072,10.079,10.090,10.090,10.076,
     * 9.575,9.543,9.524,9.547,9.560,9.595,9.615,9.630,
     * 9.520,9.484,9.433,9.403,9.364,9.322,9.265,9.196,
     * 9.258,9.225,9.182,9.155,9.121,9.079,9.025,8.960,
     * 9.212,9.180,9.136,9.110,9.075,9.033,8.980,8.914/
      DATA ((EDEN(I,2,8,L,1),I=49,56),L=1,NZEN)/
     * 11.134,11.127,11.121,11.107,11.100,11.104,11.117,11.097,
     * 11.086,11.083,11.090,11.079,11.072,11.076,11.086,11.068,
     * 11.029,11.029,11.041,11.033,11.029,11.033,11.041,11.021,
     * 10.929,10.931,10.948,10.945,10.945,10.951,10.961,10.935,
     * 10.732,10.738,10.765,10.767,10.769,10.776,10.780,10.749,
     * 10.595,10.611,10.655,10.662,10.667,10.671,10.669,10.634,
     * 10.316,10.356,10.444,10.471,10.494,10.505,10.504,10.462,
     * 10.033,10.025,10.041,10.064,10.079,10.100,10.093,10.079,
     * 9.547,9.520,9.503,9.514,9.515,9.546,9.549,9.563,
     * 9.494,9.462,9.438,9.418,9.387,9.352,9.288,9.212,
     * 9.233,9.204,9.182,9.164,9.140,9.107,9.049,8.974,
     * 9.187,9.158,9.136,9.119,9.094,9.061,9.003,8.928/
      DATA ((EDEN(I,3,8,L,1),I=49,56),L=1,NZEN)/
     * 11.107,11.107,11.114,11.114,11.121,11.137,11.146,11.127,
     * 11.057,11.057,11.079,11.083,11.090,11.107,11.117,11.093,
     * 11.000,11.004,11.025,11.033,11.045,11.061,11.072,11.045,
     * 10.900,10.905,10.934,10.947,10.961,10.979,10.987,10.960,
     * 10.708,10.716,10.752,10.769,10.785,10.804,10.806,10.775,
     * 10.574,10.592,10.641,10.665,10.686,10.702,10.699,10.665,
     * 10.310,10.352,10.423,10.471,10.515,10.538,10.537,10.500,
     * 10.041,10.053,10.072,10.114,10.143,10.170,10.158,10.140,
     * 9.522,9.508,9.499,9.518,9.525,9.576,9.583,9.605,
     * 9.468,9.447,9.450,9.449,9.433,9.415,9.352,9.276,
     * 9.210,9.190,9.196,9.196,9.185,9.167,9.107,9.037,
     * 9.164,9.145,9.150,9.150,9.139,9.122,9.061,8.992/
      DATA ((EDEN(I,4,8,L,1),I=49,56),L=1,NZEN)/
     * 11.079,11.086,11.104,11.121,11.143,11.164,11.173,11.146,
     * 11.029,11.033,11.068,11.090,11.111,11.134,11.140,11.114,
     * 10.974,10.980,11.017,11.041,11.064,11.090,11.093,11.064,
     * 10.874,10.883,10.921,10.948,10.976,11.004,11.004,10.974,
     * 10.679,10.690,10.738,10.769,10.796,10.823,10.819,10.782,
     * 10.545,10.566,10.624,10.660,10.692,10.718,10.708,10.666,
     * 10.274,10.318,10.391,10.453,10.509,10.543,10.537,10.490,
     * 9.979,9.994,10.009,10.057,10.093,10.127,10.111,10.086,
     * 9.481,9.471,9.453,9.478,9.489,9.543,9.541,9.548,
     * 9.430,9.412,9.433,9.439,9.431,9.417,9.342,9.253,
     * 9.170,9.155,9.176,9.185,9.182,9.167,9.097,9.013,
     * 9.125,9.110,9.130,9.139,9.136,9.122,9.051,8.967/
      DATA ((EDEN(I,5,8,L,1),I=49,56),L=1,NZEN)/
     * 11.049,11.057,11.083,11.114,11.146,11.180,11.183,11.153,
     * 10.999,11.009,11.049,11.083,11.117,11.149,11.152,11.121,
     * 10.945,10.955,10.997,11.033,11.068,11.104,11.104,11.072,
     * 10.844,10.857,10.903,10.944,10.983,11.017,11.017,10.982,
     * 10.649,10.666,10.723,10.766,10.806,10.841,10.833,10.794,
     * 10.517,10.542,10.611,10.661,10.704,10.739,10.725,10.681,
     * 10.253,10.301,10.387,10.462,10.526,10.571,10.560,10.513,
     * 9.960,9.990,10.013,10.079,10.137,10.188,10.170,10.143,
     * 9.452,9.449,9.393,9.444,9.479,9.564,9.578,9.599,
     * 9.400,9.391,9.415,9.441,9.450,9.449,9.377,9.290,
     * 9.143,9.137,9.158,9.188,9.201,9.204,9.134,9.053,
     * 9.097,9.091,9.116,9.144,9.156,9.158,9.088,9.007/
      DATA ((EDEN(I,1,9,L,1),I=49,56),L=1,NZEN)/
     * 11.158,11.149,11.134,11.111,11.093,11.086,11.107,11.097,
     * 11.114,11.107,11.100,11.079,11.064,11.057,11.076,11.064,
     * 11.057,11.053,11.049,11.033,11.017,11.013,11.033,11.017,
     * 10.955,10.954,10.958,10.944,10.934,10.931,10.946,10.931,
     * 10.757,10.760,10.773,10.764,10.756,10.754,10.763,10.743,
     * 10.612,10.627,10.662,10.657,10.652,10.647,10.650,10.626,
     * 10.322,10.362,10.446,10.462,10.476,10.477,10.481,10.452,
     * 10.053,10.041,10.061,10.068,10.072,10.076,10.072,10.061,
     * 9.587,9.554,9.535,9.550,9.556,9.588,9.604,9.618,
     * 9.533,9.496,9.453,9.423,9.382,9.338,9.279,9.207,
     * 9.272,9.236,9.199,9.170,9.134,9.093,9.033,8.968,
     * 9.226,9.190,9.153,9.125,9.088,9.048,8.988,8.922/
      DATA ((EDEN(I,2,9,L,1),I=49,56),L=1,NZEN)/
     * 11.143,11.140,11.134,11.117,11.111,11.111,11.127,11.114,
     * 11.093,11.093,11.097,11.086,11.079,11.079,11.093,11.079,
     * 11.037,11.037,11.049,11.037,11.033,11.037,11.049,11.033,
     * 10.938,10.941,10.954,10.950,10.948,10.953,10.964,10.944,
     * 10.738,10.745,10.769,10.769,10.769,10.774,10.779,10.754,
     * 10.594,10.614,10.655,10.661,10.665,10.667,10.667,10.637,
     * 10.307,10.350,10.433,10.461,10.486,10.496,10.496,10.461,
     * 10.037,10.033,10.053,10.068,10.079,10.090,10.079,10.064,
     * 9.566,9.538,9.526,9.540,9.543,9.574,9.577,9.587,
     * 9.513,9.481,9.455,9.433,9.400,9.362,9.297,9.220,
     * 9.248,9.220,9.199,9.179,9.149,9.114,9.053,8.980,
     * 9.202,9.174,9.153,9.133,9.103,9.068,9.007,8.934/
      DATA ((EDEN(I,3,9,L,1),I=49,56),L=1,NZEN)/
     * 11.124,11.124,11.127,11.124,11.127,11.134,11.146,11.134,
     * 11.072,11.076,11.093,11.093,11.097,11.104,11.114,11.100,
     * 11.013,11.021,11.041,11.045,11.049,11.057,11.068,11.053,
     * 10.914,10.922,10.945,10.953,10.963,10.974,10.981,10.963,
     * 10.718,10.729,10.760,10.773,10.784,10.795,10.797,10.774,
     * 10.579,10.601,10.645,10.665,10.681,10.691,10.688,10.662,
     * 10.305,10.352,10.418,10.462,10.501,10.521,10.521,10.491,
     * 10.049,10.057,10.083,10.114,10.137,10.155,10.143,10.127,
     * 9.545,9.529,9.533,9.549,9.554,9.594,9.597,9.615,
     * 9.491,9.470,9.471,9.462,9.442,9.413,9.350,9.279,
     * 9.233,9.212,9.217,9.210,9.190,9.167,9.104,9.037,
     * 9.187,9.166,9.172,9.164,9.145,9.122,9.058,8.992/
      DATA ((EDEN(I,4,9,L,1),I=49,56),L=1,NZEN)/
     * 11.111,11.117,11.130,11.140,11.149,11.164,11.170,11.152,
     * 11.053,11.061,11.093,11.104,11.117,11.130,11.137,11.121,
     * 10.998,11.009,11.041,11.053,11.072,11.083,11.090,11.068,
     * 10.895,10.907,10.942,10.960,10.979,10.994,10.997,10.975,
     * 10.695,10.711,10.753,10.775,10.794,10.809,10.803,10.775,
     * 10.554,10.579,10.631,10.660,10.685,10.699,10.688,10.654,
     * 10.265,10.314,10.378,10.435,10.484,10.511,10.507,10.467,
     * 9.963,9.973,9.988,10.021,10.045,10.064,10.041,10.021,
     * 9.508,9.496,9.494,9.494,9.482,9.502,9.475,9.474,
     * 9.455,9.436,9.456,9.450,9.430,9.398,9.318,9.225,
     * 9.193,9.176,9.196,9.193,9.176,9.146,9.072,8.983,
     * 9.147,9.130,9.150,9.147,9.130,9.100,9.026,8.937/
      DATA ((EDEN(I,5,9,L,1),I=49,56),L=1,NZEN)/
     * 11.093,11.100,11.117,11.137,11.158,11.173,11.173,11.155,
     * 11.033,11.045,11.084,11.105,11.124,11.144,11.142,11.121,
     * 10.975,10.989,11.029,11.053,11.076,11.093,11.090,11.072,
     * 10.873,10.889,10.932,10.960,10.986,11.004,11.000,10.977,
     * 10.672,10.691,10.744,10.775,10.803,10.822,10.809,10.780,
     * 10.530,10.560,10.624,10.663,10.695,10.713,10.695,10.661,
     * 10.243,10.297,10.378,10.442,10.500,10.529,10.517,10.477,
     * 9.941,9.964,9.983,10.033,10.072,10.104,10.083,10.061,
     * 9.484,9.481,9.435,9.446,9.442,9.486,9.474,9.485,
     * 9.430,9.420,9.444,9.450,9.441,9.417,9.336,9.248,
     * 9.167,9.161,9.185,9.196,9.190,9.167,9.090,9.004,
     * 9.122,9.116,9.139,9.150,9.145,9.122,9.044,8.959/
      DATA ((EDEN(I,1,10,L,1),I=49,56),L=1,NZEN)/
     * 11.167,11.155,11.143,11.121,11.104,11.100,11.121,11.114,
     * 11.117,11.114,11.107,11.090,11.076,11.068,11.090,11.083,
     * 11.061,11.057,11.057,11.041,11.029,11.025,11.041,11.033,
     * 10.960,10.958,10.961,10.949,10.940,10.939,10.955,10.943,
     * 10.757,10.761,10.775,10.767,10.760,10.759,10.768,10.751,
     * 10.609,10.626,10.661,10.657,10.654,10.651,10.656,10.634,
     * 10.312,10.352,10.433,10.453,10.471,10.476,10.483,10.455,
     * 10.057,10.045,10.061,10.068,10.068,10.072,10.064,10.057,
     * 9.602,9.567,9.541,9.549,9.546,9.577,9.591,9.605,
     * 9.549,9.511,9.470,9.438,9.396,9.350,9.288,9.217,
     * 9.286,9.250,9.212,9.185,9.146,9.104,9.045,8.977,
     * 9.240,9.205,9.166,9.139,9.100,9.058,9.000,8.931/
      DATA ((EDEN(I,2,10,L,1),I=49,56),L=1,NZEN)/
     * 11.155,11.152,11.146,11.127,11.114,11.111,11.130,11.121,
     * 11.107,11.107,11.107,11.093,11.083,11.083,11.097,11.090,
     * 11.049,11.053,11.057,11.045,11.037,11.037,11.049,11.037,
     * 10.949,10.953,10.962,10.955,10.951,10.951,10.962,10.949,
     * 10.746,10.756,10.775,10.772,10.770,10.770,10.775,10.756,
     * 10.597,10.619,10.659,10.663,10.665,10.663,10.662,10.638,
     * 10.301,10.346,10.430,10.456,10.480,10.486,10.489,10.459,
     * 10.049,10.041,10.061,10.076,10.079,10.083,10.072,10.057,
     * 9.589,9.561,9.547,9.561,9.565,9.597,9.606,9.615,
     * 9.535,9.504,9.471,9.446,9.408,9.365,9.299,9.225,
     * 9.272,9.243,9.215,9.190,9.158,9.117,9.053,8.983,
     * 9.226,9.197,9.169,9.145,9.113,9.072,9.007,8.937/
      DATA ((EDEN(I,3,10,L,1),I=49,56),L=1,NZEN)/
     * 11.140,11.143,11.143,11.134,11.130,11.127,11.140,11.137,
     * 11.090,11.097,11.111,11.100,11.097,11.097,11.111,11.104,
     * 11.033,11.041,11.057,11.053,11.053,11.053,11.061,11.053,
     * 10.931,10.943,10.959,10.959,10.961,10.964,10.973,10.962,
     * 10.732,10.747,10.772,10.777,10.781,10.784,10.785,10.770,
     * 10.585,10.614,10.655,10.667,10.677,10.679,10.676,10.656,
     * 10.303,10.354,10.422,10.458,10.491,10.504,10.505,10.483,
     * 10.061,10.068,10.100,10.121,10.134,10.143,10.127,10.111,
     * 9.575,9.558,9.562,9.579,9.584,9.617,9.621,9.637,
     * 9.521,9.500,9.493,9.476,9.447,9.410,9.344,9.276,
     * 9.260,9.241,9.236,9.223,9.196,9.161,9.100,9.033,
     * 9.214,9.195,9.190,9.177,9.150,9.116,9.055,8.988/
      DATA ((EDEN(I,4,10,L,1),I=49,56),L=1,NZEN)/
     * 11.143,11.149,11.163,11.157,11.158,11.158,11.161,11.158,
     * 11.083,11.097,11.124,11.124,11.127,11.124,11.130,11.124,
     * 11.025,11.041,11.064,11.068,11.076,11.076,11.079,11.072,
     * 10.925,10.941,10.968,10.974,10.981,10.983,10.985,10.973,
     * 10.717,10.738,10.772,10.783,10.792,10.793,10.786,10.768,
     * 10.567,10.600,10.643,10.663,10.679,10.679,10.667,10.642,
     * 10.260,10.316,10.373,10.422,10.464,10.476,10.473,10.444,
     * 9.956,9.961,9.978,9.991,9.993,9.997,9.972,9.952,
     * 9.544,9.530,9.540,9.525,9.498,9.485,9.438,9.419,
     * 9.490,9.471,9.486,9.464,9.430,9.378,9.294,9.204,
     * 9.225,9.210,9.223,9.204,9.173,9.127,9.045,8.958,
     * 9.180,9.164,9.177,9.158,9.127,9.081,9.000,8.912/
      DATA ((EDEN(I,5,10,L,1),I=49,56),L=1,NZEN)/
     * 11.137,11.146,11.155,11.161,11.167,11.161,11.158,11.155,
     * 11.068,11.086,11.121,11.127,11.134,11.130,11.124,11.121,
     * 11.013,11.033,11.064,11.076,11.083,11.079,11.076,11.068,
     * 10.908,10.930,10.966,10.978,10.989,10.989,10.981,10.971,
     * 10.701,10.727,10.772,10.788,10.801,10.798,10.782,10.766,
     * 10.550,10.588,10.643,10.667,10.687,10.684,10.665,10.641,
     * 10.241,10.303,10.375,10.428,10.473,10.483,10.471,10.444,
     * 9.934,9.950,9.972,9.996,10.009,10.017,9.993,9.971,
     * 9.526,9.524,9.514,9.497,9.466,9.453,9.399,9.391,
     * 9.470,9.459,9.481,9.467,9.438,9.387,9.297,9.212,
     * 9.207,9.199,9.220,9.207,9.182,9.137,9.049,8.966,
     * 9.161,9.153,9.174,9.161,9.136,9.091,9.003,8.920/
      DATA ((EDEN(I,1,11,L,1),I=49,56),L=1,NZEN)/
     * 11.170,11.161,11.146,11.124,11.107,11.104,11.124,11.117,
     * 11.121,11.117,11.111,11.093,11.076,11.072,11.090,11.086,
     * 11.064,11.061,11.061,11.041,11.029,11.025,11.045,11.037,
     * 10.963,10.961,10.964,10.952,10.943,10.941,10.956,10.946,
     * 10.760,10.764,10.779,10.770,10.763,10.760,10.769,10.754,
     * 10.605,10.625,10.664,10.661,10.659,10.654,10.657,10.637,
     * 10.305,10.348,10.433,10.455,10.474,10.479,10.484,10.459,
     * 10.057,10.045,10.061,10.068,10.068,10.072,10.064,10.057,
     * 9.612,9.576,9.543,9.545,9.537,9.561,9.571,9.589,
     * 9.558,9.520,9.477,9.444,9.401,9.354,9.292,9.223,
     * 9.294,9.258,9.217,9.190,9.152,9.107,9.049,8.982,
     * 9.249,9.212,9.172,9.145,9.107,9.061,9.003,8.936/
      DATA ((EDEN(I,2,11,L,1),I=49,56),L=1,NZEN)/
     * 11.161,11.158,11.152,11.130,11.114,11.107,11.124,11.117,
     * 11.117,11.121,11.114,11.097,11.083,11.076,11.090,11.086,
     * 11.061,11.064,11.064,11.049,11.037,11.029,11.045,11.037,
     * 10.958,10.964,10.968,10.958,10.950,10.945,10.956,10.946,
     * 10.754,10.766,10.782,10.775,10.769,10.763,10.768,10.754,
     * 10.601,10.626,10.667,10.666,10.664,10.657,10.656,10.636,
     * 10.299,10.348,10.439,10.461,10.479,10.480,10.481,10.456,
     * 10.053,10.045,10.072,10.079,10.079,10.079,10.064,10.053,
     * 9.607,9.579,9.557,9.572,9.577,9.608,9.623,9.634,
     * 9.554,9.522,9.480,9.450,9.410,9.362,9.297,9.223,
     * 9.288,9.258,9.223,9.196,9.158,9.114,9.049,8.982,
     * 9.242,9.212,9.177,9.150,9.113,9.068,9.003,8.936/
      DATA ((EDEN(I,3,11,L,1),I=49,56),L=1,NZEN)/
     * 11.158,11.161,11.158,11.143,11.127,11.117,11.127,11.130,
     * 11.111,11.121,11.124,11.107,11.097,11.086,11.097,11.097,
     * 11.053,11.061,11.068,11.057,11.049,11.041,11.049,11.049,
     * 10.951,10.963,10.972,10.965,10.959,10.952,10.959,10.956,
     * 10.746,10.765,10.785,10.782,10.779,10.771,10.771,10.763,
     * 10.597,10.628,10.667,10.672,10.674,10.666,10.661,10.647,
     * 10.305,10.360,10.433,10.462,10.487,10.489,10.489,10.473,
     * 10.072,10.079,10.111,10.127,10.130,10.127,10.111,10.097,
     * 9.600,9.583,9.580,9.596,9.601,9.631,9.643,9.657,
     * 9.547,9.525,9.507,9.481,9.446,9.398,9.330,9.267,
     * 9.283,9.262,9.250,9.228,9.196,9.149,9.086,9.025,
     * 9.238,9.217,9.205,9.182,9.150,9.103,9.041,8.980/
      DATA ((EDEN(I,4,11,L,1),I=49,56),L=1,NZEN)/
     * 11.167,11.179,11.185,11.170,11.161,11.146,11.149,11.152,
     * 11.111,11.127,11.146,11.137,11.127,11.114,11.117,11.117,
     * 11.053,11.072,11.090,11.083,11.076,11.064,11.064,11.064,
     * 10.950,10.969,10.989,10.986,10.982,10.970,10.970,10.968,
     * 10.740,10.765,10.792,10.792,10.790,10.776,10.767,10.758,
     * 10.584,10.622,10.660,10.668,10.674,10.659,10.646,10.630,
     * 10.265,10.326,10.378,10.417,10.449,10.449,10.444,10.423,
     * 9.958,9.960,9.973,9.971,9.959,9.944,9.914,9.894,
     * 9.574,9.561,9.567,9.542,9.506,9.469,9.412,9.384,
     * 9.519,9.501,9.508,9.473,9.427,9.358,9.270,9.182,
     * 9.253,9.238,9.246,9.215,9.170,9.104,9.017,8.934,
     * 9.207,9.192,9.200,9.169,9.125,9.058,8.971,8.888/
      DATA ((EDEN(I,5,11,L,1),I=49,56),L=1,NZEN)/
     * 11.170,11.185,11.193,11.182,11.170,11.149,11.146,11.152,
     * 11.104,11.127,11.155,11.146,11.140,11.117,11.114,11.117,
     * 11.045,11.072,11.097,11.093,11.086,11.068,11.061,11.064,
     * 10.940,10.968,10.997,10.995,10.992,10.973,10.964,10.965,
     * 10.730,10.761,10.797,10.800,10.799,10.778,10.761,10.754,
     * 10.573,10.617,10.663,10.675,10.682,10.660,10.638,10.625,
     * 10.253,10.320,10.377,10.418,10.453,10.446,10.433,10.417,
     * 9.940,9.954,9.964,9.966,9.957,9.942,9.911,9.890,
     * 9.562,9.561,9.563,9.530,9.485,9.431,9.355,9.328,
     * 9.505,9.497,9.516,9.483,9.438,9.362,9.265,9.182,
     * 9.241,9.233,9.253,9.223,9.179,9.107,9.017,8.934,
     * 9.195,9.187,9.207,9.177,9.133,9.061,8.971,8.888/
      DATA ((EDEN(I,1,12,L,1),I=49,56),L=1,NZEN)/
     * 11.173,11.164,11.146,11.124,11.111,11.104,11.121,11.117,
     * 11.124,11.121,11.114,11.093,11.079,11.072,11.090,11.083,
     * 11.068,11.064,11.061,11.045,11.033,11.025,11.041,11.037,
     * 10.965,10.964,10.968,10.954,10.944,10.941,10.956,10.945,
     * 10.761,10.766,10.782,10.773,10.766,10.762,10.770,10.754,
     * 10.610,10.629,10.668,10.665,10.662,10.656,10.659,10.637,
     * 10.312,10.352,10.439,10.461,10.480,10.483,10.487,10.461,
     * 10.061,10.049,10.061,10.072,10.072,10.076,10.068,10.061,
     * 9.612,9.576,9.540,9.533,9.518,9.540,9.546,9.564,
     * 9.559,9.520,9.476,9.442,9.400,9.354,9.292,9.223,
     * 9.294,9.258,9.217,9.188,9.149,9.104,9.045,8.980,
     * 9.249,9.212,9.172,9.142,9.103,9.058,9.000,8.935/
      DATA ((EDEN(I,2,12,L,1),I=49,56),L=1,NZEN)/
     * 11.167,11.164,11.155,11.130,11.114,11.100,11.117,11.113,
     * 11.124,11.130,11.121,11.100,11.083,11.072,11.083,11.079,
     * 11.068,11.072,11.068,11.049,11.037,11.025,11.037,11.033,
     * 10.966,10.973,10.974,10.961,10.950,10.941,10.950,10.942,
     * 10.761,10.774,10.789,10.780,10.770,10.760,10.763,10.749,
     * 10.607,10.634,10.675,10.670,10.665,10.653,10.651,10.631,
     * 10.307,10.358,10.450,10.468,10.481,10.479,10.479,10.453,
     * 10.057,10.053,10.079,10.083,10.083,10.079,10.064,10.053,
     * 9.613,9.583,9.558,9.571,9.573,9.603,9.618,9.631,
     * 9.559,9.526,9.483,9.450,9.407,9.354,9.288,9.217,
     * 9.294,9.265,9.225,9.196,9.155,9.107,9.045,8.976,
     * 9.249,9.219,9.180,9.150,9.110,9.061,9.000,8.930/
      DATA ((EDEN(I,3,12,L,1),I=49,56),L=1,NZEN)/
     * 11.167,11.173,11.167,11.146,11.127,11.107,11.121,11.124,
     * 11.121,11.134,11.130,11.111,11.093,11.079,11.086,11.090,
     * 11.064,11.076,11.079,11.061,11.049,11.029,11.037,11.041,
     * 10.961,10.975,10.982,10.969,10.958,10.943,10.950,10.949,
     * 10.757,10.777,10.796,10.787,10.778,10.763,10.763,10.756,
     * 10.607,10.640,10.679,10.678,10.673,10.658,10.652,10.640,
     * 10.316,10.373,10.447,10.470,10.487,10.481,10.480,10.464,
     * 10.083,10.090,10.121,10.130,10.127,10.121,10.104,10.090,
     * 9.611,9.594,9.588,9.600,9.602,9.628,9.641,9.656,
     * 9.555,9.535,9.515,9.483,9.441,9.386,9.318,9.255,
     * 9.290,9.272,9.258,9.228,9.190,9.141,9.078,9.014,
     * 9.244,9.226,9.212,9.182,9.145,9.094,9.030,8.967/
      DATA ((EDEN(I,4,12,L,1),I=49,56),L=1,NZEN)/
     * 11.188,11.199,11.196,11.179,11.161,11.137,11.140,11.149,
     * 11.130,11.149,11.161,11.143,11.130,11.104,11.107,11.114,
     * 11.072,11.090,11.107,11.090,11.079,11.053,11.057,11.061,
     * 10.966,10.989,11.004,10.995,10.984,10.961,10.960,10.962,
     * 10.756,10.783,10.807,10.799,10.790,10.766,10.757,10.751,
     * 10.600,10.639,10.674,10.675,10.673,10.648,10.635,10.622,
     * 10.279,10.342,10.391,10.422,10.446,10.435,10.430,10.413,
     * 9.967,9.969,9.976,9.964,9.941,9.915,9.882,9.867,
     * 9.588,9.577,9.581,9.548,9.504,9.453,9.393,9.362,
     * 9.531,9.516,9.522,9.477,9.422,9.342,9.253,9.167,
     * 9.267,9.253,9.258,9.217,9.164,9.086,9.000,8.919,
     * 9.221,9.207,9.212,9.172,9.119,9.041,8.954,8.873/
      DATA ((EDEN(I,5,12,L,1),I=49,56),L=1,NZEN)/
     * 11.193,11.210,11.215,11.193,11.173,11.140,11.137,11.146,
     * 11.124,11.152,11.176,11.158,11.143,11.107,11.104,11.111,
     * 11.068,11.097,11.121,11.107,11.090,11.057,11.053,11.057,
     * 10.963,10.992,11.017,11.009,10.996,10.964,10.955,10.960,
     * 10.750,10.785,10.817,10.810,10.799,10.766,10.749,10.747,
     * 10.594,10.640,10.679,10.681,10.679,10.645,10.624,10.616,
     * 10.272,10.342,10.384,10.418,10.447,10.427,10.413,10.401,
     * 9.950,9.965,9.967,9.954,9.930,9.900,9.866,9.849,
     * 9.580,9.582,9.587,9.540,9.483,9.410,9.330,9.295,
     * 9.521,9.517,9.538,9.493,9.435,9.344,9.246,9.161,
     * 9.258,9.255,9.274,9.230,9.176,9.090,8.994,8.913,
     * 9.212,9.210,9.228,9.185,9.130,9.044,8.948,8.867/
      DATA ((EDEN(I,1,1,L,2),I=49,56),L=1,NZEN)/
     * 11.246,11.236,11.220,11.199,11.185,11.176,11.196,11.201,
     * 11.193,11.190,11.182,11.161,11.149,11.143,11.158,11.161,
     * 11.130,11.127,11.124,11.104,11.090,11.086,11.104,11.104,
     * 11.017,11.013,11.013,10.998,10.987,10.984,11.000,10.998,
     * 10.790,10.793,10.800,10.786,10.775,10.772,10.784,10.777,
     * 10.631,10.646,10.673,10.662,10.652,10.647,10.656,10.645,
     * 10.344,10.375,10.438,10.444,10.452,10.452,10.461,10.447,
     * 10.076,10.061,10.064,10.053,10.045,10.041,10.037,10.033,
     * 9.519,9.481,9.434,9.430,9.422,9.465,9.507,9.543,
     * 9.464,9.422,9.364,9.314,9.262,9.220,9.176,9.127,
     * 9.199,9.158,9.104,9.061,9.013,8.971,8.929,8.882,
     * 9.153,9.113,9.058,9.015,8.967,8.926,8.883,8.836/
      DATA ((EDEN(I,2,1,L,2),I=49,56),L=1,NZEN)/
     * 11.246,11.241,11.230,11.210,11.190,11.179,11.196,11.199,
     * 11.201,11.201,11.193,11.170,11.152,11.140,11.155,11.158,
     * 11.134,11.137,11.134,11.111,11.097,11.086,11.100,11.100,
     * 11.017,11.021,11.021,11.004,10.991,10.982,10.996,10.993,
     * 10.790,10.801,10.809,10.793,10.780,10.770,10.778,10.772,
     * 10.630,10.653,10.682,10.668,10.657,10.645,10.649,10.638,
     * 10.342,10.380,10.450,10.452,10.455,10.449,10.453,10.438,
     * 10.076,10.068,10.083,10.068,10.053,10.041,10.033,10.025,
     * 9.521,9.491,9.465,9.480,9.489,9.538,9.587,9.614,
     * 9.465,9.431,9.371,9.324,9.272,9.223,9.173,9.121,
     * 9.201,9.167,9.114,9.068,9.017,8.972,8.924,8.874,
     * 9.156,9.122,9.068,9.022,8.971,8.926,8.878,8.829/
      DATA ((EDEN(I,3,1,L,2),I=49,56),L=1,NZEN)/
     * 11.246,11.250,11.246,11.223,11.201,11.185,11.199,11.207,
     * 11.196,11.204,11.207,11.185,11.167,11.149,11.161,11.167,
     * 11.130,11.143,11.143,11.124,11.107,11.093,11.104,11.107,
     * 11.017,11.029,11.033,11.017,11.000,10.987,10.997,11.000,
     * 10.790,10.808,10.820,10.804,10.790,10.775,10.781,10.780,
     * 10.633,10.664,10.690,10.679,10.667,10.652,10.654,10.650,
     * 10.356,10.400,10.455,10.461,10.467,10.458,10.462,10.456,
     * 10.100,10.107,10.127,10.117,10.100,10.086,10.079,10.072,
     * 9.524,9.508,9.498,9.513,9.524,9.568,9.613,9.641,
     * 9.468,9.446,9.408,9.362,9.310,9.260,9.210,9.164,
     * 9.204,9.182,9.149,9.107,9.057,9.009,8.963,8.919,
     * 9.158,9.136,9.103,9.061,9.011,8.963,8.918,8.873/
      DATA ((EDEN(I,4,1,L,2),I=49,56),L=1,NZEN)/
     * 11.262,11.276,11.276,11.253,11.233,11.207,11.215,11.220,
     * 11.204,11.223,11.230,11.212,11.193,11.167,11.173,11.179,
     * 11.137,11.158,11.170,11.149,11.130,11.107,11.111,11.114,
     * 11.017,11.041,11.053,11.033,11.017,10.995,10.999,11.000,
     * 10.784,10.810,10.823,10.807,10.790,10.765,10.762,10.759,
     * 10.619,10.655,10.677,10.667,10.655,10.628,10.621,10.612,
     * 10.305,10.360,10.391,10.401,10.408,10.393,10.393,10.380,
     * 9.986,9.988,9.981,9.951,9.916,9.885,9.861,9.845,
     * 9.497,9.484,9.466,9.423,9.373,9.340,9.313,9.307,
     * 9.438,9.420,9.398,9.334,9.262,9.182,9.107,9.037,
     * 9.170,9.155,9.134,9.072,9.000,8.926,8.854,8.785,
     * 9.125,9.110,9.088,9.026,8.954,8.880,8.808,8.740/
      DATA ((EDEN(I,5,1,L,2),I=49,56),L=1,NZEN)/
     * 11.272,11.290,11.294,11.270,11.246,11.212,11.210,11.217,
     * 11.199,11.225,11.248,11.225,11.204,11.170,11.167,11.176,
     * 11.134,11.161,11.182,11.161,11.143,11.111,11.107,11.114,
     * 11.017,11.045,11.064,11.045,11.029,10.996,10.992,10.997,
     * 10.780,10.813,10.834,10.817,10.798,10.763,10.753,10.753,
     * 10.613,10.657,10.684,10.673,10.661,10.625,10.611,10.605,
     * 10.299,10.360,10.387,10.400,10.408,10.384,10.378,10.369,
     * 9.972,9.988,9.973,9.944,9.908,9.874,9.849,9.833,
     * 9.490,9.491,9.466,9.406,9.337,9.279,9.228,9.224,
     * 9.431,9.423,9.412,9.346,9.274,9.185,9.100,9.033,
     * 9.164,9.158,9.146,9.083,9.013,8.926,8.846,8.780,
     * 9.119,9.113,9.100,9.037,8.967,8.880,8.801,8.734/
      DATA ((EDEN(I,1,2,L,2),I=49,56),L=1,NZEN)/
     * 11.246,11.236,11.220,11.199,11.185,11.179,11.201,11.201,
     * 11.193,11.188,11.182,11.161,11.149,11.143,11.158,11.161,
     * 11.127,11.124,11.124,11.104,11.090,11.086,11.107,11.104,
     * 11.013,11.013,11.013,10.998,10.988,10.985,11.004,10.999,
     * 10.789,10.791,10.799,10.785,10.775,10.772,10.785,10.777,
     * 10.631,10.645,10.669,10.659,10.651,10.646,10.656,10.644,
     * 10.348,10.378,10.433,10.439,10.447,10.449,10.459,10.444,
     * 10.076,10.061,10.061,10.053,10.037,10.033,10.033,10.029,
     * 9.511,9.474,9.435,9.431,9.422,9.465,9.503,9.538,
     * 9.456,9.415,9.362,9.312,9.260,9.217,9.173,9.124,
     * 9.190,9.152,9.104,9.057,9.009,8.968,8.925,8.876,
     * 9.145,9.107,9.058,9.011,8.963,8.923,8.879,8.830/
      DATA ((EDEN(I,2,2,L,2),I=49,56),L=1,NZEN)/
     * 11.243,11.241,11.233,11.210,11.190,11.182,11.199,11.199,
     * 11.193,11.196,11.190,11.170,11.155,11.146,11.161,11.161,
     * 11.127,11.130,11.130,11.111,11.097,11.090,11.104,11.104,
     * 11.013,11.017,11.021,11.004,10.993,10.987,11.000,10.996,
     * 10.787,10.796,10.807,10.792,10.780,10.774,10.783,10.774,
     * 10.629,10.650,10.677,10.666,10.656,10.647,10.653,10.640,
     * 10.346,10.382,10.442,10.447,10.452,10.447,10.453,10.438,
     * 10.072,10.064,10.072,10.061,10.045,10.037,10.029,10.021,
     * 9.511,9.481,9.456,9.463,9.466,9.509,9.549,9.576,
     * 9.455,9.422,9.371,9.322,9.270,9.223,9.173,9.117,
     * 9.190,9.158,9.114,9.068,9.017,8.972,8.924,8.872,
     * 9.145,9.113,9.068,9.022,8.971,8.926,8.879,8.826/
      DATA ((EDEN(I,3,2,L,2),I=49,56),L=1,NZEN)/
     * 11.241,11.248,11.243,11.223,11.207,11.193,11.207,11.210,
     * 11.188,11.196,11.201,11.182,11.167,11.155,11.170,11.173,
     * 11.121,11.130,11.143,11.124,11.111,11.100,11.111,11.111,
     * 11.009,11.021,11.029,11.017,11.004,10.995,11.006,11.005,
     * 10.785,10.801,10.816,10.803,10.792,10.782,10.788,10.785,
     * 10.631,10.659,10.685,10.677,10.669,10.658,10.662,10.655,
     * 10.358,10.401,10.449,10.458,10.465,10.462,10.468,10.459,
     * 10.097,10.107,10.121,10.111,10.100,10.090,10.083,10.072,
     * 9.512,9.497,9.488,9.494,9.496,9.534,9.570,9.600,
     * 9.455,9.433,9.405,9.360,9.310,9.262,9.215,9.167,
     * 9.193,9.173,9.146,9.104,9.057,9.013,8.968,8.922,
     * 9.147,9.127,9.100,9.058,9.011,8.967,8.922,8.876/
      DATA ((EDEN(I,4,2,L,2),I=49,56),L=1,NZEN)/
     * 11.253,11.265,11.267,11.248,11.230,11.210,11.215,11.217,
     * 11.190,11.207,11.225,11.204,11.188,11.170,11.176,11.176,
     * 11.124,11.143,11.158,11.143,11.130,11.111,11.117,11.117,
     * 11.009,11.025,11.045,11.029,11.017,11.000,11.004,11.004,
     * 10.776,10.799,10.818,10.805,10.792,10.774,10.772,10.764,
     * 10.615,10.648,10.673,10.666,10.658,10.638,10.633,10.620,
     * 10.314,10.364,10.393,10.407,10.418,10.410,10.410,10.394,
     * 9.998,10.004,9.999,9.978,9.953,9.929,9.909,9.893,
     * 9.483,9.474,9.461,9.419,9.371,9.350,9.326,9.330,
     * 9.425,9.408,9.394,9.336,9.272,9.201,9.130,9.061,
     * 9.161,9.143,9.130,9.076,9.013,8.945,8.878,8.810,
     * 9.116,9.097,9.085,9.030,8.967,8.900,8.832,8.764/
      DATA ((EDEN(I,5,2,L,2),I=49,56),L=1,NZEN)/
     * 11.255,11.272,11.279,11.258,11.238,11.212,11.212,11.215,
     * 11.185,11.210,11.236,11.217,11.201,11.173,11.170,11.173,
     * 11.121,11.143,11.170,11.152,11.137,11.111,11.111,11.111,
     * 11.004,11.029,11.053,11.037,11.025,11.000,10.998,10.997,
     * 10.769,10.800,10.827,10.814,10.799,10.773,10.765,10.758,
     * 10.609,10.648,10.681,10.674,10.665,10.637,10.625,10.615,
     * 10.305,10.364,10.396,10.412,10.425,10.408,10.403,10.389,
     * 9.989,10.009,9.999,9.983,9.961,9.938,9.918,9.902,
     * 9.477,9.480,9.446,9.389,9.326,9.289,9.256,9.270,
     * 9.418,9.410,9.405,9.346,9.281,9.204,9.130,9.064,
     * 9.152,9.146,9.140,9.083,9.021,8.948,8.877,8.812,
     * 9.107,9.100,9.094,9.037,8.975,8.903,8.831,8.766/
      DATA ((EDEN(I,1,3,L,2),I=49,56),L=1,NZEN)/
     * 11.243,11.233,11.220,11.199,11.185,11.176,11.196,11.199,
     * 11.190,11.185,11.179,11.158,11.143,11.140,11.158,11.158,
     * 11.121,11.117,11.121,11.100,11.090,11.086,11.104,11.104,
     * 11.009,11.009,11.009,10.996,10.986,10.983,11.000,10.997,
     * 10.785,10.787,10.795,10.782,10.772,10.771,10.785,10.776,
     * 10.630,10.643,10.667,10.656,10.648,10.644,10.654,10.642,
     * 10.348,10.377,10.430,10.436,10.444,10.446,10.456,10.441,
     * 10.076,10.057,10.057,10.045,10.033,10.029,10.029,10.025,
     * 9.507,9.470,9.433,9.434,9.433,9.475,9.513,9.545,
     * 9.450,9.410,9.358,9.310,9.260,9.215,9.170,9.117,
     * 9.188,9.149,9.097,9.053,9.004,8.965,8.921,8.873,
     * 9.142,9.103,9.051,9.007,8.959,8.919,8.875,8.827/
      DATA ((EDEN(I,2,3,L,2),I=49,56),L=1,NZEN)/
     * 11.238,11.233,11.225,11.207,11.193,11.188,11.207,11.204,
     * 11.182,11.185,11.185,11.170,11.158,11.149,11.164,11.161,
     * 11.117,11.121,11.127,11.111,11.100,11.097,11.111,11.107,
     * 11.004,11.009,11.017,11.004,10.995,10.993,11.009,11.000,
     * 10.780,10.787,10.800,10.790,10.781,10.779,10.789,10.777,
     * 10.625,10.644,10.670,10.663,10.657,10.652,10.659,10.643,
     * 10.344,10.378,10.431,10.441,10.452,10.452,10.459,10.441,
     * 10.072,10.064,10.068,10.061,10.045,10.041,10.033,10.025,
     * 9.500,9.471,9.448,9.451,9.450,9.489,9.523,9.550,
     * 9.444,9.410,9.367,9.322,9.272,9.228,9.179,9.124,
     * 9.179,9.149,9.111,9.064,9.017,8.977,8.930,8.877,
     * 9.133,9.103,9.065,9.019,8.971,8.932,8.885,8.832/
      DATA ((EDEN(I,3,3,L,2),I=49,56),L=1,NZEN)/
     * 11.228,11.236,11.236,11.217,11.207,11.201,11.215,11.215,
     * 11.173,11.179,11.193,11.179,11.170,11.164,11.179,11.175,
     * 11.107,11.117,11.134,11.121,11.111,11.107,11.121,11.117,
     * 10.994,11.004,11.021,11.013,11.004,11.004,11.017,11.013,
     * 10.773,10.787,10.806,10.799,10.794,10.792,10.800,10.793,
     * 10.622,10.647,10.675,10.674,10.673,10.670,10.675,10.665,
     * 10.354,10.396,10.439,10.455,10.471,10.476,10.484,10.473,
     * 10.097,10.104,10.114,10.114,10.111,10.107,10.097,10.090,
     * 9.497,9.481,9.472,9.473,9.470,9.506,9.535,9.569,
     * 9.441,9.418,9.400,9.360,9.316,9.276,9.230,9.182,
     * 9.179,9.158,9.140,9.104,9.064,9.025,8.983,8.936,
     * 9.133,9.113,9.094,9.058,9.019,8.980,8.937,8.890/
      DATA ((EDEN(I,4,3,L,2),I=49,56),L=1,NZEN)/
     * 11.230,11.243,11.250,11.238,11.228,11.217,11.225,11.220,
     * 11.167,11.179,11.204,11.193,11.185,11.176,11.185,11.179,
     * 11.100,11.117,11.140,11.130,11.124,11.117,11.127,11.121,
     * 10.986,11.004,11.025,11.021,11.017,11.009,11.017,11.009,
     * 10.759,10.780,10.805,10.799,10.794,10.787,10.788,10.775,
     * 10.603,10.633,10.665,10.665,10.665,10.657,10.654,10.636,
     * 10.314,10.362,10.398,10.418,10.438,10.439,10.442,10.422,
     * 10.013,10.025,10.025,10.017,10.004,9.997,9.982,9.967,
     * 9.467,9.458,9.436,9.402,9.361,9.364,9.354,9.374,
     * 9.410,9.393,9.386,9.338,9.286,9.233,9.170,9.104,
     * 9.146,9.130,9.124,9.079,9.029,8.978,8.918,8.854,
     * 9.100,9.085,9.078,9.033,8.984,8.932,8.872,8.808/
      DATA ((EDEN(I,5,3,L,2),I=49,56),L=1,NZEN)/
     * 11.228,11.243,11.250,11.238,11.230,11.217,11.217,11.215,
     * 11.158,11.176,11.207,11.199,11.190,11.179,11.179,11.176,
     * 11.093,11.114,11.143,11.137,11.130,11.121,11.121,11.114,
     * 10.976,11.000,11.029,11.025,11.021,11.013,11.013,11.004,
     * 10.748,10.775,10.810,10.806,10.802,10.792,10.786,10.775,
     * 10.592,10.628,10.670,10.673,10.674,10.663,10.654,10.638,
     * 10.303,10.358,10.407,10.430,10.450,10.450,10.447,10.430,
     * 10.004,10.029,10.033,10.033,10.033,10.029,10.017,10.004,
     * 9.456,9.461,9.402,9.363,9.319,9.331,9.331,9.366,
     * 9.398,9.391,9.386,9.344,9.297,9.246,9.182,9.121,
     * 9.134,9.130,9.124,9.086,9.041,8.993,8.933,8.876,
     * 9.088,9.085,9.078,9.041,8.996,8.947,8.888,8.830/
      DATA ((EDEN(I,1,4,L,2),I=49,56),L=1,NZEN)/
     * 11.236,11.225,11.210,11.190,11.176,11.170,11.188,11.188,
     * 11.188,11.182,11.173,11.152,11.137,11.130,11.152,11.149,
     * 11.121,11.117,11.114,11.097,11.083,11.076,11.097,11.093,
     * 11.004,11.004,11.004,10.990,10.980,10.977,10.994,10.989,
     * 10.783,10.785,10.792,10.779,10.769,10.766,10.780,10.770,
     * 10.628,10.641,10.666,10.654,10.644,10.640,10.650,10.637,
     * 10.346,10.375,10.435,10.439,10.444,10.444,10.453,10.438,
     * 10.068,10.053,10.057,10.045,10.033,10.033,10.029,10.025,
     * 9.502,9.465,9.433,9.446,9.453,9.500,9.544,9.574,
     * 9.446,9.405,9.346,9.301,9.250,9.207,9.161,9.111,
     * 9.182,9.143,9.090,9.045,8.997,8.958,8.915,8.867,
     * 9.136,9.097,9.044,9.000,8.952,8.912,8.870,8.822/
      DATA ((EDEN(I,2,4,L,2),I=49,56),L=1,NZEN)/
     * 11.228,11.223,11.217,11.201,11.190,11.190,11.210,11.201,
     * 11.170,11.170,11.176,11.161,11.152,11.152,11.167,11.161,
     * 11.107,11.107,11.114,11.104,11.097,11.097,11.114,11.104,
     * 10.993,10.995,11.004,10.998,10.993,10.994,11.009,10.999,
     * 10.771,10.777,10.793,10.785,10.780,10.782,10.792,10.779,
     * 10.619,10.635,10.664,10.660,10.657,10.657,10.663,10.645,
     * 10.338,10.373,10.427,10.439,10.453,10.459,10.465,10.446,
     * 10.061,10.053,10.057,10.053,10.045,10.045,10.037,10.029,
     * 9.487,9.458,9.433,9.433,9.431,9.470,9.501,9.528,
     * 9.431,9.398,9.356,9.314,9.267,9.228,9.179,9.124,
     * 9.170,9.137,9.097,9.057,9.013,8.980,8.935,8.879,
     * 9.125,9.091,9.051,9.011,8.967,8.933,8.887,8.832/
      DATA ((EDEN(I,3,4,L,2),I=49,56),L=1,NZEN)/
     * 11.212,11.212,11.215,11.210,11.207,11.210,11.223,11.217,
     * 11.155,11.158,11.176,11.170,11.167,11.173,11.188,11.179,
     * 11.090,11.093,11.117,11.111,11.111,11.117,11.130,11.121,
     * 10.976,10.984,11.004,11.004,11.004,11.013,11.025,11.017,
     * 10.757,10.769,10.792,10.794,10.797,10.803,10.812,10.800,
     * 10.611,10.632,10.665,10.670,10.676,10.683,10.688,10.673,
     * 10.346,10.386,10.430,10.453,10.477,10.491,10.500,10.484,
     * 10.086,10.093,10.100,10.111,10.114,10.121,10.114,10.107,
     * 9.477,9.462,9.445,9.445,9.440,9.483,9.510,9.548,
     * 9.422,9.400,9.382,9.350,9.316,9.288,9.243,9.193,
     * 9.161,9.140,9.124,9.097,9.064,9.037,8.996,8.948,
     * 9.116,9.094,9.078,9.051,9.019,8.992,8.950,8.902/
      DATA ((EDEN(I,4,4,L,2),I=49,56),L=1,NZEN)/
     * 11.199,11.207,11.215,11.215,11.217,11.225,11.236,11.225,
     * 11.140,11.149,11.176,11.176,11.179,11.185,11.193,11.182,
     * 11.072,11.086,11.111,11.114,11.121,11.127,11.137,11.124,
     * 10.960,10.973,11.000,11.009,11.013,11.025,11.029,11.017,
     * 10.738,10.754,10.786,10.792,10.800,10.808,10.810,10.792,
     * 10.588,10.614,10.653,10.665,10.676,10.683,10.682,10.660,
     * 10.314,10.358,10.405,10.435,10.465,10.481,10.484,10.461,
     * 10.025,10.041,10.045,10.057,10.064,10.076,10.064,10.049,
     * 9.446,9.436,9.398,9.384,9.363,9.403,9.419,9.454,
     * 9.389,9.373,9.364,9.336,9.301,9.272,9.217,9.155,
     * 9.127,9.114,9.104,9.079,9.049,9.021,8.968,8.908,
     * 9.081,9.068,9.058,9.033,9.003,8.975,8.922,8.862/
      DATA ((EDEN(I,5,4,L,2),I=49,56),L=1,NZEN)/
     * 11.182,11.193,11.210,11.215,11.223,11.230,11.233,11.223,
     * 11.124,11.137,11.167,11.176,11.182,11.193,11.196,11.185,
     * 11.057,11.072,11.107,11.114,11.124,11.137,11.140,11.127,
     * 10.942,10.961,10.998,11.009,11.021,11.029,11.033,11.021,
     * 10.720,10.743,10.783,10.797,10.809,10.821,10.820,10.803,
     * 10.569,10.602,10.654,10.672,10.688,10.700,10.695,10.676,
     * 10.294,10.348,10.417,10.453,10.487,10.508,10.508,10.489,
     * 10.013,10.045,10.057,10.086,10.111,10.134,10.127,10.114,
     * 9.428,9.431,9.337,9.344,9.343,9.415,9.456,9.505,
     * 9.373,9.367,9.358,9.342,9.320,9.303,9.253,9.201,
     * 9.111,9.107,9.100,9.086,9.068,9.053,9.009,8.956,
     * 9.065,9.061,9.055,9.041,9.022,9.007,8.963,8.910/
      DATA ((EDEN(I,1,5,L,2),I=49,56),L=1,NZEN)/
     * 11.228,11.217,11.201,11.179,11.158,11.152,11.173,11.170,
     * 11.182,11.176,11.161,11.140,11.124,11.117,11.137,11.134,
     * 11.117,11.114,11.107,11.086,11.072,11.064,11.083,11.079,
     * 11.004,11.000,11.000,10.984,10.971,10.967,10.984,10.976,
     * 10.783,10.785,10.791,10.776,10.764,10.760,10.772,10.761,
     * 10.630,10.642,10.668,10.654,10.643,10.637,10.645,10.631,
     * 10.346,10.377,10.449,10.449,10.450,10.447,10.453,10.436,
     * 10.064,10.049,10.057,10.049,10.041,10.041,10.041,10.033,
     * 9.497,9.461,9.435,9.460,9.481,9.532,9.581,9.610,
     * 9.439,9.400,9.332,9.288,9.241,9.199,9.155,9.107,
     * 9.176,9.137,9.076,9.033,8.989,8.952,8.911,8.866,
     * 9.130,9.091,9.030,8.988,8.944,8.906,8.865,8.820/
      DATA ((EDEN(I,2,5,L,2),I=49,56),L=1,NZEN)/
     * 11.217,11.210,11.204,11.193,11.185,11.188,11.201,11.193,
     * 11.161,11.158,11.164,11.152,11.146,11.149,11.164,11.155,
     * 11.097,11.097,11.104,11.097,11.090,11.093,11.107,11.097,
     * 10.984,10.985,10.997,10.992,10.990,10.993,11.004,10.993,
     * 10.765,10.769,10.787,10.782,10.780,10.785,10.794,10.776,
     * 10.614,10.628,10.662,10.659,10.659,10.662,10.666,10.644,
     * 10.336,10.369,10.433,10.447,10.461,10.468,10.474,10.449,
     * 10.057,10.045,10.049,10.049,10.049,10.057,10.049,10.041,
     * 9.476,9.446,9.414,9.419,9.419,9.462,9.491,9.520,
     * 9.418,9.384,9.338,9.301,9.260,9.228,9.179,9.124,
     * 9.158,9.124,9.083,9.049,9.009,8.978,8.933,8.879,
     * 9.113,9.078,9.037,9.003,8.963,8.932,8.887,8.833/
      DATA ((EDEN(I,3,5,L,2),I=49,56),L=1,NZEN)/
     * 11.190,11.190,11.196,11.196,11.199,11.212,11.228,11.217,
     * 11.137,11.140,11.155,11.158,11.164,11.176,11.188,11.176,
     * 11.072,11.076,11.093,11.100,11.107,11.121,11.134,11.121,
     * 10.960,10.966,10.988,10.995,11.004,11.017,11.029,11.017,
     * 10.745,10.754,10.780,10.788,10.798,10.813,10.820,10.804,
     * 10.601,10.619,10.656,10.668,10.681,10.695,10.699,10.679,
     * 10.340,10.377,10.430,10.459,10.489,10.509,10.516,10.496,
     * 10.076,10.083,10.090,10.107,10.124,10.140,10.134,10.124,
     * 9.459,9.442,9.420,9.429,9.435,9.491,9.522,9.562,
     * 9.403,9.380,9.360,9.338,9.314,9.297,9.253,9.201,
     * 9.143,9.121,9.104,9.086,9.061,9.045,9.004,8.958,
     * 9.097,9.075,9.058,9.041,9.015,9.000,8.959,8.912/
      DATA ((EDEN(I,4,5,L,2),I=49,56),L=1,NZEN)/
     * 11.167,11.173,11.185,11.199,11.212,11.233,11.243,11.228,
     * 11.114,11.121,11.146,11.158,11.173,11.193,11.204,11.188,
     * 11.049,11.057,11.086,11.100,11.117,11.140,11.146,11.130,
     * 10.937,10.946,10.976,10.994,11.013,11.037,11.045,11.025,
     * 10.721,10.733,10.767,10.786,10.804,10.827,10.830,10.808,
     * 10.575,10.598,10.642,10.665,10.686,10.708,10.707,10.681,
     * 10.310,10.350,10.410,10.450,10.489,10.517,10.520,10.493,
     * 10.029,10.049,10.057,10.086,10.111,10.134,10.127,10.111,
     * 9.427,9.417,9.390,9.410,9.424,9.488,9.517,9.554,
     * 9.371,9.356,9.342,9.330,9.314,9.303,9.255,9.196,
     * 9.111,9.097,9.086,9.076,9.064,9.053,9.009,8.950,
     * 9.065,9.051,9.041,9.030,9.019,9.007,8.963,8.905/
      DATA ((EDEN(I,5,5,L,2),I=49,56),L=1,NZEN)/
     * 11.140,11.146,11.170,11.190,11.215,11.246,11.253,11.236,
     * 11.086,11.100,11.130,11.155,11.179,11.207,11.215,11.196,
     * 11.025,11.037,11.072,11.097,11.124,11.152,11.158,11.140,
     * 10.912,10.927,10.964,10.992,11.021,11.053,11.057,11.037,
     * 10.695,10.715,10.758,10.788,10.817,10.849,10.851,10.830,
     * 10.551,10.581,10.637,10.671,10.703,10.735,10.734,10.711,
     * 10.290,10.340,10.420,10.471,10.519,10.558,10.560,10.538,
     * 10.013,10.053,10.072,10.124,10.167,10.210,10.210,10.193,
     * 9.401,9.403,9.366,9.411,9.451,9.547,9.594,9.642,
     * 9.346,9.342,9.332,9.340,9.344,9.352,9.312,9.262,
     * 9.086,9.086,9.079,9.090,9.093,9.104,9.064,9.017,
     * 9.041,9.041,9.033,9.044,9.048,9.058,9.019,8.971/
      DATA ((EDEN(I,1,6,L,2),I=49,56),L=1,NZEN)/
     * 11.225,11.210,11.190,11.167,11.146,11.140,11.158,11.155,
     * 11.182,11.176,11.155,11.130,11.114,11.104,11.121,11.117,
     * 11.117,11.114,11.100,11.079,11.061,11.053,11.068,11.064,
     * 11.004,11.000,10.997,10.977,10.963,10.957,10.972,10.963,
     * 10.785,10.785,10.791,10.774,10.760,10.755,10.766,10.754,
     * 10.634,10.644,10.670,10.654,10.640,10.634,10.640,10.625,
     * 10.354,10.382,10.462,10.458,10.455,10.450,10.453,10.433,
     * 10.061,10.045,10.061,10.057,10.049,10.053,10.057,10.049,
     * 9.490,9.456,9.436,9.472,9.504,9.559,9.611,9.637,
     * 9.433,9.393,9.320,9.276,9.230,9.193,9.152,9.107,
     * 9.170,9.130,9.068,9.025,8.983,8.946,8.907,8.865,
     * 9.125,9.085,9.022,8.980,8.937,8.901,8.862,8.819/
      DATA ((EDEN(I,2,6,L,2),I=49,56),L=1,NZEN)/
     * 11.207,11.199,11.193,11.182,11.176,11.179,11.193,11.182,
     * 11.155,11.152,11.152,11.143,11.137,11.143,11.158,11.143,
     * 11.090,11.090,11.097,11.090,11.086,11.090,11.104,11.090,
     * 10.978,10.979,10.991,10.986,10.985,10.992,11.004,10.986,
     * 10.761,10.764,10.783,10.780,10.779,10.785,10.793,10.772,
     * 10.614,10.625,10.661,10.659,10.659,10.664,10.667,10.643,
     * 10.338,10.369,10.441,10.455,10.468,10.477,10.480,10.450,
     * 10.049,10.037,10.045,10.053,10.057,10.068,10.064,10.053,
     * 9.464,9.436,9.405,9.417,9.427,9.475,9.508,9.537,
     * 9.407,9.371,9.324,9.292,9.255,9.225,9.179,9.124,
     * 9.146,9.111,9.068,9.037,9.004,8.977,8.933,8.881,
     * 9.100,9.065,9.022,8.992,8.959,8.932,8.887,8.835/
      DATA ((EDEN(I,3,6,L,2),I=49,56),L=1,NZEN)/
     * 11.176,11.176,11.182,11.188,11.199,11.215,11.228,11.212,
     * 11.124,11.124,11.143,11.149,11.158,11.176,11.190,11.176,
     * 11.061,11.064,11.083,11.093,11.104,11.124,11.134,11.117,
     * 10.950,10.954,10.976,10.988,11.000,11.021,11.033,11.017,
     * 10.737,10.744,10.772,10.785,10.799,10.819,10.826,10.806,
     * 10.595,10.611,10.650,10.667,10.683,10.702,10.705,10.682,
     * 10.336,10.371,10.433,10.465,10.497,10.522,10.526,10.501,
     * 10.064,10.072,10.083,10.107,10.127,10.152,10.146,10.134,
     * 9.444,9.428,9.407,9.431,9.450,9.516,9.552,9.588,
     * 9.387,9.364,9.342,9.328,9.310,9.299,9.258,9.207,
     * 9.127,9.107,9.086,9.076,9.061,9.053,9.009,8.962,
     * 9.081,9.061,9.041,9.030,9.015,9.007,8.963,8.917/
      DATA ((EDEN(I,4,6,L,2),I=49,56),L=1,NZEN)/
     * 11.143,11.146,11.164,11.188,11.210,11.238,11.248,11.228,
     * 11.097,11.100,11.124,11.149,11.173,11.201,11.210,11.190,
     * 11.033,11.037,11.068,11.090,11.117,11.146,11.155,11.134,
     * 10.922,10.928,10.958,10.985,11.013,11.045,11.053,11.029,
     * 10.708,10.719,10.754,10.781,10.808,10.839,10.842,10.816,
     * 10.566,10.585,10.632,10.662,10.691,10.722,10.721,10.691,
     * 10.307,10.346,10.410,10.458,10.502,10.538,10.539,10.508,
     * 10.025,10.045,10.057,10.093,10.127,10.161,10.152,10.134,
     * 9.408,9.398,9.410,9.446,9.476,9.549,9.576,9.608,
     * 9.352,9.336,9.324,9.324,9.318,9.316,9.270,9.210,
     * 9.093,9.079,9.068,9.072,9.068,9.068,9.025,8.965,
     * 9.048,9.033,9.022,9.026,9.022,9.022,8.980,8.919/
      DATA ((EDEN(I,5,6,L,2),I=49,56),L=1,NZEN)/
     * 11.104,11.114,11.143,11.179,11.217,11.258,11.265,11.243,
     * 11.064,11.072,11.107,11.143,11.179,11.220,11.228,11.204,
     * 11.004,11.013,11.045,11.083,11.124,11.164,11.173,11.149,
     * 10.892,10.904,10.941,10.981,11.021,11.064,11.072,11.049,
     * 10.679,10.695,10.740,10.782,10.822,10.867,10.870,10.844,
     * 10.539,10.565,10.622,10.668,10.711,10.756,10.754,10.728,
     * 10.286,10.332,10.415,10.476,10.534,10.584,10.585,10.558,
     * 10.009,10.049,10.072,10.134,10.190,10.243,10.241,10.220,
     * 9.380,9.380,9.424,9.477,9.523,9.621,9.660,9.698,
     * 9.326,9.322,9.314,9.336,9.352,9.375,9.336,9.286,
     * 9.068,9.068,9.061,9.086,9.107,9.130,9.090,9.041,
     * 9.022,9.022,9.015,9.041,9.061,9.085,9.044,8.996/
      DATA ((EDEN(I,1,7,L,2),I=49,56),L=1,NZEN)/
     * 11.225,11.210,11.190,11.164,11.143,11.137,11.157,11.153,
     * 11.182,11.176,11.155,11.130,11.111,11.104,11.121,11.114,
     * 11.117,11.114,11.097,11.076,11.057,11.053,11.068,11.061,
     * 11.004,11.004,10.996,10.976,10.961,10.955,10.969,10.960,
     * 10.786,10.786,10.790,10.773,10.760,10.754,10.765,10.751,
     * 10.637,10.646,10.670,10.654,10.640,10.633,10.639,10.623,
     * 10.356,10.386,10.465,10.459,10.456,10.450,10.453,10.433,
     * 10.064,10.049,10.064,10.061,10.053,10.057,10.057,10.053,
     * 9.487,9.453,9.444,9.481,9.517,9.569,9.621,9.646,
     * 9.431,9.391,9.318,9.276,9.230,9.193,9.152,9.107,
     * 9.167,9.130,9.064,9.025,8.981,8.946,8.910,8.867,
     * 9.122,9.085,9.019,8.980,8.935,8.901,8.864,8.821/
      DATA ((EDEN(I,2,7,L,2),I=49,56),L=1,NZEN)/
     * 11.204,11.193,11.188,11.176,11.170,11.177,11.194,11.179,
     * 11.155,11.149,11.152,11.140,11.134,11.140,11.155,11.143,
     * 11.090,11.086,11.093,11.086,11.083,11.090,11.100,11.086,
     * 10.979,10.978,10.988,10.984,10.984,10.991,11.004,10.985,
     * 10.761,10.763,10.781,10.778,10.778,10.786,10.794,10.772,
     * 10.614,10.625,10.660,10.658,10.659,10.665,10.667,10.643,
     * 10.340,10.371,10.444,10.456,10.470,10.479,10.481,10.452,
     * 10.045,10.037,10.049,10.057,10.061,10.076,10.072,10.061,
     * 9.459,9.431,9.404,9.423,9.439,9.491,9.525,9.554,
     * 9.403,9.367,9.320,9.288,9.253,9.225,9.179,9.127,
     * 9.140,9.107,9.064,9.037,9.004,8.980,8.937,8.884,
     * 9.094,9.061,9.019,8.992,8.959,8.934,8.891,8.838/
      DATA ((EDEN(I,3,7,L,2),I=49,56),L=1,NZEN)/
     * 11.173,11.173,11.179,11.185,11.196,11.212,11.225,11.212,
     * 11.121,11.121,11.140,11.146,11.158,11.179,11.190,11.176,
     * 11.057,11.061,11.079,11.090,11.104,11.124,11.134,11.117,
     * 10.947,10.951,10.974,10.987,11.000,11.025,11.033,11.017,
     * 10.735,10.741,10.769,10.783,10.799,10.821,10.827,10.806,
     * 10.594,10.609,10.648,10.666,10.684,10.704,10.707,10.683,
     * 10.336,10.369,10.433,10.465,10.498,10.524,10.528,10.502,
     * 10.061,10.068,10.079,10.104,10.127,10.155,10.149,10.137,
     * 9.438,9.423,9.409,9.439,9.466,9.535,9.572,9.606,
     * 9.382,9.358,9.338,9.326,9.310,9.301,9.258,9.207,
     * 9.121,9.100,9.083,9.076,9.061,9.053,9.013,8.962,
     * 9.075,9.055,9.037,9.030,9.015,9.007,8.967,8.917/
      DATA ((EDEN(I,4,7,L,2),I=49,56),L=1,NZEN)/
     * 11.140,11.143,11.161,11.185,11.212,11.243,11.253,11.230,
     * 11.093,11.093,11.124,11.149,11.176,11.204,11.215,11.193,
     * 11.029,11.033,11.061,11.090,11.117,11.149,11.158,11.134,
     * 10.918,10.925,10.954,10.984,11.013,11.049,11.053,11.029,
     * 10.704,10.713,10.749,10.778,10.808,10.841,10.843,10.814,
     * 10.562,10.581,10.626,10.659,10.690,10.722,10.719,10.688,
     * 10.303,10.340,10.403,10.452,10.498,10.535,10.535,10.500,
     * 10.013,10.029,10.037,10.079,10.114,10.149,10.137,10.117,
     * 9.398,9.387,9.404,9.442,9.476,9.552,9.580,9.603,
     * 9.342,9.326,9.316,9.316,9.310,9.310,9.260,9.196,
     * 9.083,9.068,9.057,9.061,9.061,9.061,9.013,8.951,
     * 9.037,9.022,9.011,9.015,9.015,9.015,8.967,8.906/
      DATA ((EDEN(I,5,7,L,2),I=49,56),L=1,NZEN)/
     * 11.100,11.111,11.137,11.179,11.220,11.262,11.270,11.246,
     * 11.057,11.068,11.100,11.140,11.182,11.225,11.230,11.207,
     * 10.998,11.009,11.042,11.083,11.124,11.170,11.176,11.149,
     * 10.886,10.897,10.936,10.980,11.025,11.068,11.076,11.049,
     * 10.673,10.688,10.732,10.778,10.822,10.869,10.869,10.839,
     * 10.533,10.558,10.614,10.663,10.708,10.754,10.751,10.719,
     * 10.279,10.324,10.404,10.468,10.526,10.577,10.576,10.544,
     * 9.997,10.033,10.053,10.114,10.170,10.223,10.215,10.193,
     * 9.369,9.367,9.408,9.459,9.503,9.603,9.638,9.675,
     * 9.314,9.310,9.301,9.324,9.342,9.364,9.320,9.262,
     * 9.057,9.053,9.049,9.076,9.093,9.114,9.072,9.017,
     * 9.011,9.007,9.003,9.030,9.048,9.068,9.026,8.971/
      DATA ((EDEN(I,1,8,L,2),I=49,56),L=1,NZEN)/
     * 11.225,11.212,11.196,11.173,11.155,11.149,11.167,11.164,
     * 11.182,11.176,11.158,11.137,11.117,11.111,11.130,11.127,
     * 11.121,11.114,11.100,11.079,11.064,11.061,11.076,11.072,
     * 11.004,11.004,10.999,10.980,10.967,10.963,10.979,10.970,
     * 10.785,10.785,10.791,10.775,10.761,10.759,10.771,10.758,
     * 10.635,10.645,10.668,10.654,10.641,10.635,10.643,10.628,
     * 10.354,10.384,10.456,10.455,10.453,10.450,10.455,10.435,
     * 10.064,10.049,10.064,10.057,10.049,10.049,10.053,10.045,
     * 9.490,9.455,9.448,9.481,9.511,9.563,9.612,9.637,
     * 9.435,9.394,9.328,9.286,9.238,9.199,9.158,9.111,
     * 9.173,9.134,9.072,9.033,8.989,8.953,8.913,8.869,
     * 9.127,9.088,9.026,8.988,8.944,8.907,8.868,8.823/
      DATA ((EDEN(I,2,8,L,2),I=49,56),L=1,NZEN)/
     * 11.210,11.201,11.193,11.185,11.179,11.182,11.199,11.188,
     * 11.161,11.155,11.152,11.146,11.143,11.146,11.161,11.149,
     * 11.093,11.093,11.097,11.090,11.086,11.093,11.107,11.093,
     * 10.981,10.981,10.992,10.987,10.985,10.992,11.004,10.990,
     * 10.763,10.766,10.782,10.779,10.779,10.785,10.794,10.775,
     * 10.614,10.627,10.660,10.658,10.659,10.664,10.667,10.645,
     * 10.338,10.371,10.438,10.452,10.467,10.474,10.479,10.452,
     * 10.053,10.041,10.053,10.057,10.061,10.072,10.068,10.057,
     * 9.465,9.436,9.416,9.437,9.455,9.508,9.542,9.569,
     * 9.408,9.375,9.330,9.297,9.260,9.230,9.185,9.130,
     * 9.149,9.114,9.076,9.045,9.009,8.983,8.939,8.886,
     * 9.103,9.068,9.030,9.000,8.963,8.937,8.893,8.841/
      DATA ((EDEN(I,3,8,L,2),I=49,56),L=1,NZEN)/
     * 11.182,11.182,11.188,11.190,11.199,11.215,11.230,11.217,
     * 11.130,11.130,11.143,11.152,11.161,11.179,11.190,11.179,
     * 11.064,11.068,11.086,11.093,11.107,11.124,11.134,11.121,
     * 10.954,10.958,10.979,10.990,11.004,11.021,11.033,11.017,
     * 10.739,10.746,10.772,10.785,10.799,10.818,10.824,10.806,
     * 10.595,10.612,10.649,10.665,10.681,10.699,10.702,10.682,
     * 10.336,10.371,10.428,10.461,10.493,10.516,10.521,10.499,
     * 10.064,10.068,10.083,10.107,10.124,10.146,10.140,10.127,
     * 9.446,9.430,9.423,9.451,9.475,9.540,9.575,9.609,
     * 9.389,9.365,9.348,9.332,9.312,9.299,9.255,9.204,
     * 9.130,9.107,9.090,9.079,9.061,9.049,9.009,8.960,
     * 9.085,9.061,9.044,9.033,9.015,9.003,8.963,8.915/
      DATA ((EDEN(I,4,8,L,2),I=49,56),L=1,NZEN)/
     * 11.152,11.158,11.176,11.196,11.217,11.243,11.250,11.233,
     * 11.104,11.107,11.134,11.155,11.176,11.204,11.212,11.193,
     * 11.037,11.045,11.076,11.097,11.121,11.149,11.155,11.134,
     * 10.926,10.934,10.964,10.989,11.017,11.045,11.049,11.029,
     * 10.709,10.719,10.754,10.780,10.804,10.832,10.833,10.806,
     * 10.563,10.583,10.626,10.655,10.683,10.709,10.706,10.675,
     * 10.294,10.332,10.389,10.435,10.480,10.513,10.512,10.479,
     * 9.995,10.009,10.013,10.045,10.072,10.100,10.086,10.064,
     * 9.403,9.391,9.382,9.408,9.429,9.492,9.513,9.534,
     * 9.348,9.328,9.318,9.310,9.297,9.286,9.230,9.161,
     * 9.086,9.068,9.061,9.057,9.045,9.037,8.983,8.916,
     * 9.041,9.022,9.015,9.011,9.000,8.992,8.937,8.870/
      DATA ((EDEN(I,5,8,L,2),I=49,56),L=1,NZEN)/
     * 11.124,11.130,11.158,11.190,11.225,11.260,11.262,11.243,
     * 11.076,11.083,11.117,11.152,11.188,11.223,11.225,11.204,
     * 11.013,11.021,11.057,11.093,11.130,11.164,11.170,11.146,
     * 10.898,10.910,10.948,10.987,11.025,11.064,11.064,11.041,
     * 10.679,10.695,10.740,10.780,10.817,10.854,10.851,10.823,
     * 10.535,10.560,10.615,10.658,10.698,10.734,10.728,10.696,
     * 10.270,10.314,10.387,10.447,10.502,10.547,10.542,10.509,
     * 9.975,10.004,10.017,10.068,10.114,10.161,10.149,10.127,
     * 9.375,9.371,9.340,9.387,9.427,9.517,9.548,9.581,
     * 9.320,9.310,9.305,9.316,9.320,9.326,9.274,9.210,
     * 9.057,9.053,9.049,9.061,9.068,9.076,9.025,8.965,
     * 9.011,9.007,9.003,9.015,9.022,9.030,8.980,8.919/
      DATA ((EDEN(I,1,9,L,2),I=49,56),L=1,NZEN)/
     * 11.230,11.223,11.207,11.185,11.167,11.164,11.185,11.185,
     * 11.188,11.182,11.167,11.146,11.130,11.127,11.146,11.146,
     * 11.121,11.117,11.111,11.090,11.076,11.072,11.090,11.090,
     * 11.009,11.004,11.000,10.985,10.975,10.972,10.989,10.985,
     * 10.785,10.785,10.790,10.776,10.764,10.763,10.776,10.766,
     * 10.629,10.642,10.666,10.653,10.643,10.639,10.648,10.635,
     * 10.346,10.375,10.441,10.442,10.447,10.446,10.455,10.439,
     * 10.068,10.053,10.061,10.053,10.045,10.041,10.041,10.033,
     * 9.502,9.465,9.449,9.475,9.496,9.548,9.597,9.624,
     * 9.446,9.405,9.344,9.299,9.250,9.210,9.167,9.117,
     * 9.185,9.143,9.086,9.045,8.999,8.961,8.921,8.874,
     * 9.139,9.097,9.041,9.000,8.953,8.916,8.875,8.829/
      DATA ((EDEN(I,2,9,L,2),I=49,56),L=1,NZEN)/
     * 11.215,11.210,11.207,11.193,11.185,11.188,11.204,11.199,
     * 11.164,11.164,11.167,11.155,11.149,11.149,11.167,11.161,
     * 11.100,11.100,11.107,11.097,11.090,11.093,11.111,11.104,
     * 10.988,10.990,10.997,10.991,10.989,10.993,11.009,10.997,
     * 10.767,10.772,10.786,10.781,10.778,10.782,10.792,10.778,
     * 10.614,10.629,10.660,10.657,10.656,10.659,10.665,10.647,
     * 10.332,10.365,10.430,10.444,10.459,10.465,10.471,10.450,
     * 10.057,10.045,10.061,10.061,10.061,10.061,10.057,10.045,
     * 9.483,9.453,9.439,9.461,9.481,9.533,9.573,9.596,
     * 9.428,9.393,9.346,9.310,9.270,9.236,9.190,9.134,
     * 9.164,9.130,9.090,9.057,9.017,8.986,8.941,8.889,
     * 9.119,9.085,9.044,9.011,8.971,8.941,8.895,8.843/
      DATA ((EDEN(I,3,9,L,2),I=49,56),L=1,NZEN)/
     * 11.196,11.199,11.201,11.199,11.201,11.212,11.228,11.223,
     * 11.143,11.146,11.161,11.161,11.167,11.176,11.188,11.182,
     * 11.079,11.083,11.097,11.100,11.111,11.121,11.134,11.124,
     * 10.967,10.973,10.991,10.997,11.004,11.017,11.029,11.021,
     * 10.748,10.758,10.781,10.787,10.796,10.809,10.816,10.804,
     * 10.599,10.620,10.654,10.666,10.678,10.689,10.692,10.678,
     * 10.332,10.371,10.425,10.453,10.481,10.500,10.507,10.490,
     * 10.068,10.076,10.093,10.107,10.121,10.134,10.127,10.117,
     * 9.468,9.450,9.451,9.478,9.501,9.558,9.592,9.623,
     * 9.412,9.387,9.367,9.344,9.318,9.297,9.250,9.201,
     * 9.149,9.127,9.111,9.090,9.068,9.049,9.004,8.957,
     * 9.103,9.081,9.065,9.044,9.022,9.003,8.959,8.911/
      DATA ((EDEN(I,4,9,L,2),I=49,56),L=1,NZEN)/
     * 11.182,11.188,11.201,11.212,11.223,11.241,11.248,11.236,
     * 11.127,11.134,11.158,11.170,11.185,11.199,11.207,11.196,
     * 11.061,11.072,11.097,11.111,11.127,11.140,11.146,11.134,
     * 10.946,10.957,10.984,11.000,11.017,11.033,11.037,11.021,
     * 10.723,10.736,10.767,10.782,10.799,10.815,10.815,10.794,
     * 10.569,10.592,10.632,10.652,10.672,10.687,10.682,10.657,
     * 10.281,10.322,10.375,10.415,10.453,10.476,10.476,10.447,
     * 9.977,9.985,9.991,10.009,10.021,10.033,10.017,9.998,
     * 9.427,9.410,9.398,9.400,9.398,9.436,9.440,9.455,
     * 9.369,9.346,9.336,9.314,9.286,9.258,9.196,9.124,
     * 9.104,9.086,9.076,9.057,9.029,9.004,8.944,8.875,
     * 9.058,9.041,9.030,9.011,8.984,8.959,8.899,8.829/
      DATA ((EDEN(I,5,9,L,2),I=49,56),L=1,NZEN)/
     * 11.161,11.173,11.193,11.212,11.233,11.250,11.253,11.241,
     * 11.107,11.117,11.149,11.170,11.193,11.212,11.212,11.199,
     * 11.037,11.053,11.086,11.111,11.134,11.152,11.152,11.140,
     * 10.924,10.939,10.975,11.000,11.025,11.045,11.045,11.029,
     * 10.700,10.719,10.760,10.785,10.810,10.829,10.823,10.802,
     * 10.545,10.574,10.627,10.656,10.684,10.703,10.693,10.668,
     * 10.260,10.307,10.375,10.425,10.471,10.498,10.493,10.465,
     * 9.954,9.975,9.989,10.021,10.049,10.076,10.061,10.041,
     * 9.401,9.398,9.341,9.355,9.362,9.420,9.438,9.466,
     * 9.344,9.332,9.327,9.317,9.299,9.281,9.220,9.152,
     * 9.083,9.072,9.064,9.061,9.045,9.029,8.971,8.906,
     * 9.037,9.026,9.019,9.015,9.000,8.984,8.925,8.861/
      DATA ((EDEN(I,1,10,L,2),I=49,56),L=1,NZEN)/
     * 11.238,11.228,11.215,11.193,11.179,11.176,11.196,11.199,
     * 11.190,11.185,11.173,11.155,11.140,11.137,11.158,11.161,
     * 11.121,11.117,11.114,11.097,11.083,11.083,11.100,11.100,
     * 11.009,11.004,11.004,10.989,10.979,10.979,10.998,10.995,
     * 10.785,10.787,10.792,10.778,10.768,10.766,10.780,10.773,
     * 10.625,10.640,10.665,10.654,10.645,10.642,10.651,10.641,
     * 10.336,10.367,10.430,10.436,10.444,10.446,10.456,10.442,
     * 10.068,10.053,10.064,10.053,10.041,10.037,10.033,10.029,
     * 9.517,9.479,9.449,9.463,9.475,9.527,9.577,9.609,
     * 9.461,9.418,9.358,9.312,9.262,9.220,9.176,9.127,
     * 9.196,9.155,9.100,9.057,9.009,8.970,8.929,8.881,
     * 9.150,9.110,9.055,9.011,8.963,8.925,8.883,8.835/
      DATA ((EDEN(I,2,10,L,2),I=49,56),L=1,NZEN)/
     * 11.225,11.223,11.215,11.201,11.190,11.190,11.207,11.207,
     * 11.179,11.179,11.173,11.158,11.152,11.149,11.167,11.167,
     * 11.111,11.114,11.114,11.100,11.093,11.093,11.111,11.107,
     * 10.999,11.000,11.004,10.995,10.989,10.991,11.004,11.000,
     * 10.773,10.781,10.792,10.783,10.778,10.778,10.787,10.778,
     * 10.614,10.633,10.664,10.659,10.655,10.653,10.658,10.645,
     * 10.324,10.362,10.428,10.439,10.452,10.456,10.462,10.446,
     * 10.057,10.049,10.064,10.061,10.053,10.049,10.041,10.033,
     * 9.504,9.473,9.454,9.476,9.495,9.549,9.595,9.620,
     * 9.449,9.413,9.362,9.320,9.274,9.236,9.188,9.130,
     * 9.185,9.149,9.104,9.064,9.021,8.985,8.939,8.886,
     * 9.139,9.103,9.058,9.019,8.975,8.939,8.893,8.840/
      DATA ((EDEN(I,3,10,L,2),I=49,56),L=1,NZEN)/
     * 11.215,11.217,11.217,11.210,11.207,11.207,11.220,11.223,
     * 11.161,11.170,11.173,11.167,11.167,11.170,11.185,11.185,
     * 11.097,11.104,11.114,11.111,11.111,11.111,11.124,11.124,
     * 10.985,10.993,11.004,11.004,11.004,11.009,11.021,11.017,
     * 10.761,10.775,10.792,10.792,10.792,10.796,10.802,10.797,
     * 10.605,10.630,10.663,10.667,10.671,10.675,10.679,10.669,
     * 10.330,10.371,10.428,10.449,10.470,10.481,10.489,10.477,
     * 10.076,10.079,10.104,10.111,10.111,10.114,10.107,10.097,
     * 9.496,9.476,9.473,9.498,9.519,9.573,9.613,9.642,
     * 9.439,9.413,9.386,9.354,9.318,9.286,9.238,9.190,
     * 9.176,9.152,9.127,9.100,9.064,9.033,8.991,8.944,
     * 9.130,9.107,9.081,9.055,9.019,8.988,8.945,8.899/
      DATA ((EDEN(I,4,10,L,2),I=49,56),L=1,NZEN)/
     * 11.215,11.223,11.230,11.230,11.233,11.233,11.238,11.238,
     * 11.155,11.167,11.188,11.188,11.190,11.190,11.199,11.196,
     * 11.090,11.104,11.121,11.124,11.130,11.130,11.137,11.134,
     * 10.974,10.987,11.009,11.013,11.017,11.021,11.025,11.017,
     * 10.742,10.761,10.785,10.789,10.794,10.794,10.793,10.780,
     * 10.580,10.610,10.642,10.653,10.663,10.663,10.657,10.639,
     * 10.272,10.322,10.367,10.398,10.428,10.438,10.438,10.417,
     * 9.968,9.969,9.976,9.975,9.969,9.966,9.946,9.926,
     * 9.458,9.441,9.427,9.412,9.390,9.399,9.388,9.392,
     * 9.401,9.377,9.361,9.323,9.276,9.228,9.161,9.086,
     * 9.134,9.114,9.097,9.061,9.017,8.973,8.907,8.837,
     * 9.088,9.068,9.051,9.015,8.971,8.927,8.862,8.791/
      DATA ((EDEN(I,5,10,L,2),I=49,56),L=1,NZEN)/
     * 11.207,11.217,11.230,11.233,11.241,11.240,11.240,11.238,
     * 11.140,11.158,11.188,11.190,11.196,11.196,11.196,11.193,
     * 11.076,11.093,11.124,11.130,11.137,11.137,11.134,11.130,
     * 10.958,10.978,11.004,11.017,11.025,11.029,11.025,11.017,
     * 10.725,10.750,10.785,10.794,10.803,10.802,10.792,10.781,
     * 10.562,10.598,10.642,10.658,10.672,10.670,10.657,10.640,
     * 10.255,10.310,10.369,10.405,10.439,10.449,10.441,10.423,
     * 9.941,9.956,9.969,9.979,9.984,9.989,9.970,9.952,
     * 9.441,9.436,9.405,9.382,9.352,9.361,9.343,9.357,
     * 9.382,9.367,9.356,9.324,9.288,9.241,9.170,9.100,
     * 9.137,9.114,9.092,9.050,9.004,8.959,8.897,8.841,
     * 9.091,9.068,9.044,9.003,8.959,8.913,8.851,8.796/
      DATA ((EDEN(I,1,11,L,2),I=49,56),L=1,NZEN)/
     * 11.243,11.233,11.217,11.196,11.185,11.179,11.201,11.204,
     * 11.193,11.188,11.179,11.158,11.143,11.140,11.158,11.164,
     * 11.127,11.124,11.121,11.100,11.090,11.086,11.104,11.107,
     * 11.013,11.009,11.009,10.993,10.983,10.980,10.999,10.998,
     * 10.786,10.789,10.795,10.782,10.771,10.768,10.781,10.776,
     * 10.624,10.640,10.667,10.657,10.649,10.644,10.653,10.643,
     * 10.330,10.364,10.430,10.438,10.446,10.447,10.458,10.446,
     * 10.072,10.053,10.068,10.057,10.045,10.037,10.033,10.033,
     * 9.525,9.486,9.443,9.449,9.451,9.503,9.553,9.589,
     * 9.470,9.427,9.365,9.318,9.267,9.225,9.179,9.130,
     * 9.204,9.161,9.107,9.061,9.013,8.973,8.930,8.884,
     * 9.158,9.116,9.061,9.015,8.967,8.927,8.885,8.838/
      DATA ((EDEN(I,2,11,L,2),I=49,56),L=1,NZEN)/
     * 11.236,11.230,11.223,11.204,11.188,11.182,11.201,11.204,
     * 11.190,11.190,11.182,11.164,11.152,11.146,11.161,11.164,
     * 11.121,11.127,11.124,11.107,11.093,11.090,11.104,11.107,
     * 11.009,11.013,11.013,10.998,10.989,10.985,11.000,10.999,
     * 10.780,10.791,10.799,10.787,10.777,10.772,10.781,10.775,
     * 10.617,10.640,10.671,10.662,10.654,10.647,10.652,10.642,
     * 10.326,10.364,10.436,10.444,10.452,10.450,10.456,10.442,
     * 10.064,10.053,10.076,10.068,10.057,10.049,10.041,10.029,
     * 9.521,9.489,9.465,9.488,9.505,9.560,9.611,9.640,
     * 9.465,9.428,9.369,9.324,9.274,9.230,9.182,9.130,
     * 9.201,9.164,9.111,9.068,9.021,8.981,8.935,8.885,
     * 9.156,9.119,9.065,9.022,8.975,8.935,8.889,8.839/
      DATA ((EDEN(I,3,11,L,2),I=49,56),L=1,NZEN)/
     * 11.230,11.233,11.230,11.215,11.207,11.196,11.210,11.217,
     * 11.182,11.193,11.190,11.176,11.167,11.158,11.173,11.179,
     * 11.114,11.124,11.123,11.115,11.107,11.100,11.114,11.117,
     * 11.000,11.013,11.017,11.009,11.000,10.995,11.004,11.009,
     * 10.775,10.792,10.805,10.797,10.790,10.783,10.789,10.789,
     * 10.615,10.644,10.675,10.671,10.668,10.662,10.664,10.660,
     * 10.334,10.378,10.441,10.453,10.467,10.467,10.473,10.467,
     * 10.086,10.090,10.117,10.114,10.107,10.104,10.093,10.086,
     * 9.519,9.498,9.489,9.513,9.533,9.587,9.636,9.667,
     * 9.462,9.436,9.400,9.360,9.316,9.275,9.229,9.182,
     * 9.199,9.173,9.140,9.104,9.064,9.025,8.979,8.934,
     * 9.153,9.127,9.094,9.058,9.019,8.980,8.933,8.889/
      DATA ((EDEN(I,4,11,L,2),I=49,56),L=1,NZEN)/
     * 11.241,11.250,11.253,11.243,11.233,11.220,11.225,11.230,
     * 11.182,11.199,11.210,11.199,11.190,11.179,11.185,11.188,
     * 11.114,11.134,11.143,11.137,11.134,11.117,11.124,11.127,
     * 10.998,11.017,11.029,11.021,11.017,11.004,11.009,11.009,
     * 10.763,10.787,10.803,10.797,10.790,10.776,10.773,10.768,
     * 10.594,10.630,10.657,10.656,10.656,10.640,10.633,10.622,
     * 10.276,10.330,10.373,10.393,10.412,10.407,10.407,10.393,
     * 9.966,9.963,9.968,9.950,9.927,9.908,9.885,9.866,
     * 9.486,9.470,9.450,9.421,9.385,9.371,9.350,9.347,
     * 9.428,9.405,9.380,9.328,9.270,9.201,9.130,9.057,
     * 9.161,9.140,9.114,9.064,9.009,8.945,8.874,8.805,
     * 9.116,9.094,9.068,9.019,8.963,8.899,8.828,8.759/
      DATA ((EDEN(I,5,11,L,2),I=49,56),L=1,NZEN)/
     * 11.241,11.258,11.265,11.255,11.246,11.228,11.225,11.230,
     * 11.176,11.199,11.220,11.212,11.204,11.185,11.179,11.188,
     * 11.107,11.130,11.152,11.146,11.143,11.124,11.121,11.124,
     * 10.990,11.017,11.037,11.029,11.025,11.009,11.004,11.009,
     * 10.754,10.783,10.809,10.804,10.799,10.779,10.768,10.765,
     * 10.584,10.625,10.660,10.663,10.663,10.642,10.627,10.619,
     * 10.262,10.322,10.369,10.393,10.415,10.405,10.398,10.387,
     * 9.942,9.954,9.954,9.942,9.924,9.906,9.884,9.865,
     * 9.474,9.471,9.447,9.404,9.356,9.321,9.276,9.276,
     * 9.415,9.401,9.386,9.334,9.279,9.207,9.127,9.057,
     * 9.146,9.137,9.121,9.072,9.017,8.948,8.872,8.806,
     * 9.100,9.091,9.075,9.026,8.971,8.903,8.826,8.760/
      DATA ((EDEN(I,1,12,L,2),I=49,56),L=1,NZEN)/
     * 11.246,11.236,11.220,11.199,11.185,11.179,11.201,11.204,
     * 11.193,11.190,11.182,11.161,11.149,11.143,11.158,11.164,
     * 11.130,11.127,11.121,11.100,11.090,11.086,11.104,11.104,
     * 11.017,11.013,11.013,10.996,10.985,10.981,10.999,10.998,
     * 10.789,10.792,10.799,10.785,10.774,10.770,10.782,10.777,
     * 10.627,10.643,10.671,10.660,10.651,10.646,10.655,10.645,
     * 10.336,10.369,10.438,10.442,10.450,10.452,10.461,10.447,
     * 10.076,10.057,10.068,10.057,10.045,10.041,10.037,10.037,
     * 9.526,9.487,9.438,9.436,9.430,9.478,9.524,9.564,
     * 9.470,9.428,9.364,9.316,9.267,9.225,9.179,9.130,
     * 9.204,9.164,9.107,9.061,9.013,8.973,8.930,8.884,
     * 9.158,9.119,9.061,9.015,8.967,8.927,8.885,8.838/
      DATA ((EDEN(I,2,12,L,2),I=49,56),L=1,NZEN)/
     * 11.243,11.238,11.225,11.204,11.188,11.179,11.196,11.201,
     * 11.196,11.199,11.188,11.167,11.152,11.140,11.155,11.158,
     * 11.130,11.134,11.127,11.107,11.093,11.083,11.100,11.100,
     * 11.017,11.021,11.016,10.999,10.990,10.981,10.994,10.993,
     * 10.788,10.799,10.806,10.791,10.778,10.769,10.777,10.772,
     * 10.625,10.649,10.679,10.667,10.656,10.645,10.648,10.639,
     * 10.332,10.373,10.447,10.450,10.455,10.449,10.455,10.441,
     * 10.072,10.061,10.083,10.072,10.057,10.045,10.037,10.029,
     * 9.526,9.494,9.468,9.488,9.503,9.556,9.607,9.637,
     * 9.471,9.435,9.371,9.324,9.272,9.225,9.176,9.124,
     * 9.207,9.170,9.114,9.072,9.021,8.976,8.929,8.880,
     * 9.161,9.125,9.068,9.026,8.975,8.930,8.883,8.834/
      DATA ((EDEN(I,3,12,L,2),I=49,56),L=1,NZEN)/
     * 11.241,11.246,11.238,11.220,11.204,11.188,11.201,11.210,
     * 11.196,11.207,11.199,11.179,11.164,11.149,11.158,11.170,
     * 11.127,11.140,11.140,11.121,11.107,11.093,11.104,11.111,
     * 11.013,11.025,11.025,11.013,11.000,10.988,10.998,11.004,
     * 10.787,10.805,10.816,10.801,10.789,10.776,10.781,10.782,
     * 10.626,10.657,10.687,10.677,10.667,10.654,10.656,10.653,
     * 10.346,10.391,10.453,10.461,10.467,10.461,10.465,10.459,
     * 10.097,10.100,10.127,10.117,10.107,10.093,10.083,10.076,
     * 9.528,9.511,9.498,9.520,9.537,9.587,9.639,9.668,
     * 9.473,9.447,9.407,9.362,9.314,9.265,9.215,9.170,
     * 9.207,9.185,9.149,9.107,9.061,9.013,8.969,8.925,
     * 9.161,9.139,9.103,9.061,9.015,8.967,8.923,8.880/
      DATA ((EDEN(I,4,12,L,2),I=49,56),L=1,NZEN)/
     * 11.258,11.270,11.270,11.250,11.233,11.210,11.217,11.225,
     * 11.201,11.217,11.228,11.207,11.190,11.170,11.176,11.185,
     * 11.134,11.152,11.161,11.146,11.134,11.111,11.114,11.121,
     * 11.017,11.037,11.045,11.029,11.017,10.997,11.000,11.004,
     * 10.780,10.805,10.818,10.803,10.790,10.766,10.762,10.760,
     * 10.611,10.648,10.670,10.663,10.653,10.630,10.622,10.614,
     * 10.290,10.344,10.382,10.395,10.407,10.393,10.391,10.380,
     * 9.972,9.971,9.968,9.941,9.908,9.878,9.853,9.837,
     * 9.500,9.486,9.463,9.424,9.380,9.352,9.328,9.322,
     * 9.441,9.420,9.393,9.330,9.265,9.185,9.107,9.037,
     * 9.173,9.155,9.127,9.068,9.004,8.928,8.856,8.786,
     * 9.127,9.110,9.081,9.022,8.959,8.883,8.810,8.740/
      DATA ((EDEN(I,5,12,L,2),I=49,56),L=1,NZEN)/
     * 11.267,11.283,11.286,11.265,11.248,11.215,11.212,11.225,
     * 11.199,11.223,11.243,11.225,11.207,11.176,11.173,11.182,
     * 11.130,11.155,11.179,11.161,11.146,11.114,11.111,11.117,
     * 11.013,11.041,11.057,11.041,11.029,10.999,10.995,11.000,
     * 10.774,10.806,10.828,10.814,10.798,10.766,10.755,10.755,
     * 10.604,10.648,10.676,10.669,10.660,10.627,10.613,10.607,
     * 10.281,10.342,10.377,10.393,10.405,10.384,10.377,10.369,
     * 9.953,9.965,9.955,9.927,9.894,9.862,9.836,9.818,
     * 9.491,9.490,9.466,9.412,9.350,9.294,9.244,9.235,
     * 9.431,9.420,9.405,9.342,9.274,9.185,9.100,9.033,
     * 9.164,9.155,9.137,9.079,9.013,8.927,8.846,8.779,
     * 9.119,9.110,9.091,9.033,8.967,8.882,8.801,8.733/
      DATA ((EDEN(I,1,1,L,3),I=49,56),L=1,NZEN)/
     * 11.342,11.334,11.320,11.301,11.290,11.286,11.307,11.322,
     * 11.283,11.281,11.272,11.255,11.243,11.241,11.260,11.274,
     * 11.204,11.204,11.201,11.185,11.176,11.173,11.196,11.207,
     * 11.076,11.072,11.076,11.061,11.053,11.053,11.072,11.083,
     * 10.823,10.826,10.836,10.820,10.809,10.809,10.827,10.834,
     * 10.656,10.670,10.696,10.682,10.672,10.669,10.685,10.688,
     * 10.391,10.412,10.465,10.461,10.461,10.461,10.476,10.476,
     * 10.137,10.117,10.124,10.100,10.079,10.072,10.072,10.072,
     * 9.477,9.438,9.396,9.407,9.421,9.496,9.580,9.635,
     * 9.420,9.377,9.310,9.258,9.207,9.170,9.143,9.114,
     * 9.158,9.117,9.057,9.004,8.954,8.920,8.894,8.866,
     * 9.113,9.072,9.011,8.959,8.908,8.874,8.848,8.821/
      DATA ((EDEN(I,2,1,L,3),I=49,56),L=1,NZEN)/
     * 11.338,11.336,11.328,11.307,11.294,11.283,11.303,11.316,
     * 11.286,11.290,11.281,11.260,11.248,11.241,11.260,11.272,
     * 11.207,11.215,11.212,11.193,11.179,11.173,11.193,11.201,
     * 11.076,11.083,11.086,11.072,11.061,11.053,11.072,11.079,
     * 10.823,10.834,10.845,10.828,10.816,10.809,10.824,10.829,
     * 10.655,10.677,10.707,10.690,10.677,10.668,10.680,10.682,
     * 10.387,10.417,10.480,10.471,10.465,10.459,10.471,10.470,
     * 10.137,10.124,10.143,10.117,10.090,10.076,10.072,10.068,
     * 9.480,9.447,9.443,9.474,9.505,9.580,9.666,9.712,
     * 9.423,9.386,9.324,9.270,9.215,9.173,9.143,9.111,
     * 9.158,9.124,9.068,9.017,8.963,8.924,8.894,8.863,
     * 9.113,9.078,9.022,8.971,8.918,8.878,8.849,8.818/
      DATA ((EDEN(I,3,1,L,3),I=49,56),L=1,NZEN)/
     * 11.338,11.344,11.342,11.322,11.307,11.294,11.312,11.326,
     * 11.283,11.297,11.294,11.276,11.260,11.248,11.265,11.279,
     * 11.210,11.223,11.223,11.207,11.193,11.182,11.199,11.210,
     * 11.076,11.090,11.097,11.079,11.068,11.057,11.076,11.086,
     * 10.825,10.845,10.857,10.841,10.827,10.818,10.832,10.842,
     * 10.662,10.691,10.719,10.705,10.693,10.682,10.693,10.701,
     * 10.405,10.442,10.496,10.490,10.486,10.479,10.491,10.497,
     * 10.161,10.167,10.190,10.167,10.143,10.124,10.121,10.117,
     * 9.487,9.471,9.476,9.513,9.547,9.620,9.704,9.746,
     * 9.430,9.407,9.362,9.310,9.258,9.215,9.188,9.158,
     * 9.167,9.146,9.107,9.057,9.009,8.967,8.939,8.912,
     * 9.122,9.100,9.061,9.011,8.963,8.921,8.893,8.866/
      DATA ((EDEN(I,4,1,L,3),I=49,56),L=1,NZEN)/
     * 11.350,11.365,11.367,11.346,11.330,11.310,11.320,11.330,
     * 11.288,11.310,11.316,11.297,11.281,11.260,11.267,11.279,
     * 11.212,11.233,11.243,11.225,11.210,11.188,11.196,11.204,
     * 11.076,11.100,11.107,11.090,11.076,11.057,11.064,11.072,
     * 10.812,10.838,10.848,10.829,10.811,10.788,10.791,10.794,
     * 10.633,10.670,10.688,10.672,10.657,10.632,10.632,10.630,
     * 10.330,10.377,10.405,10.401,10.398,10.380,10.384,10.378,
     * 10.037,10.037,10.029,9.986,9.943,9.906,9.887,9.870,
     * 9.447,9.433,9.399,9.367,9.333,9.332,9.350,9.365,
     * 9.387,9.364,9.316,9.243,9.167,9.093,9.033,8.978,
     * 9.121,9.097,9.053,8.982,8.907,8.835,8.778,8.725,
     * 9.075,9.051,9.007,8.936,8.862,8.789,8.732,8.679/
      DATA ((EDEN(I,5,1,L,3),I=49,56),L=1,NZEN)/
     * 11.356,11.378,11.384,11.362,11.342,11.312,11.314,11.326,
     * 11.288,11.316,11.334,11.312,11.292,11.262,11.267,11.276,
     * 11.210,11.238,11.258,11.238,11.220,11.190,11.193,11.201,
     * 11.076,11.104,11.121,11.104,11.086,11.057,11.061,11.068,
     * 10.807,10.841,10.857,10.838,10.818,10.788,10.785,10.789,
     * 10.627,10.671,10.694,10.678,10.661,10.629,10.623,10.623,
     * 10.320,10.375,10.401,10.400,10.396,10.371,10.371,10.369,
     * 10.021,10.033,10.017,9.978,9.933,9.895,9.875,9.858,
     * 9.441,9.441,9.399,9.344,9.288,9.257,9.245,9.266,
     * 9.378,9.367,9.326,9.250,9.170,9.090,9.025,8.971,
     * 9.111,9.100,9.061,8.989,8.912,8.832,8.770,8.719,
     * 9.065,9.055,9.015,8.943,8.866,8.786,8.724,8.673/
      DATA ((EDEN(I,1,2,L,3),I=49,56),L=1,NZEN)/
     * 11.338,11.330,11.318,11.299,11.286,11.283,11.307,11.320,
     * 11.279,11.276,11.272,11.255,11.243,11.241,11.260,11.274,
     * 11.204,11.204,11.201,11.185,11.176,11.173,11.196,11.207,
     * 11.072,11.072,11.076,11.061,11.053,11.053,11.076,11.083,
     * 10.823,10.825,10.834,10.819,10.809,10.810,10.829,10.834,
     * 10.658,10.670,10.695,10.680,10.670,10.668,10.684,10.687,
     * 10.393,10.413,10.461,10.456,10.456,10.458,10.473,10.473,
     * 10.137,10.117,10.121,10.097,10.076,10.064,10.064,10.064,
     * 9.470,9.433,9.396,9.410,9.426,9.499,9.580,9.631,
     * 9.413,9.371,9.307,9.255,9.201,9.164,9.137,9.107,
     * 9.152,9.111,9.053,9.000,8.951,8.916,8.890,8.861,
     * 9.107,9.065,9.007,8.954,8.905,8.871,8.844,8.815/
      DATA ((EDEN(I,2,2,L,3),I=49,56),L=1,NZEN)/
     * 11.336,11.334,11.326,11.307,11.294,11.288,11.307,11.318,
     * 11.279,11.283,11.281,11.262,11.248,11.243,11.262,11.272,
     * 11.204,11.210,11.210,11.196,11.185,11.179,11.199,11.204,
     * 11.072,11.079,11.083,11.072,11.061,11.057,11.076,11.079,
     * 10.820,10.831,10.841,10.826,10.814,10.810,10.826,10.829,
     * 10.655,10.675,10.702,10.688,10.676,10.669,10.681,10.681,
     * 10.389,10.417,10.470,10.464,10.461,10.458,10.468,10.465,
     * 10.134,10.124,10.134,10.107,10.083,10.068,10.064,10.057,
     * 9.470,9.439,9.428,9.454,9.480,9.551,9.630,9.672,
     * 9.412,9.377,9.318,9.265,9.210,9.170,9.137,9.104,
     * 9.149,9.117,9.064,9.013,8.960,8.920,8.888,8.855,
     * 9.103,9.072,9.019,8.967,8.914,8.874,8.842,8.809/
      DATA ((EDEN(I,3,2,L,3),I=49,56),L=1,NZEN)/
     * 11.336,11.342,11.340,11.322,11.310,11.301,11.318,11.330,
     * 11.279,11.288,11.292,11.276,11.262,11.253,11.270,11.281,
     * 11.201,11.215,11.223,11.207,11.196,11.188,11.207,11.215,
     * 11.072,11.086,11.097,11.083,11.072,11.068,11.083,11.093,
     * 10.821,10.839,10.854,10.840,10.829,10.823,10.837,10.845,
     * 10.660,10.688,10.714,10.703,10.695,10.688,10.699,10.704,
     * 10.407,10.442,10.487,10.486,10.486,10.483,10.497,10.501,
     * 10.161,10.164,10.182,10.161,10.140,10.127,10.124,10.121,
     * 9.476,9.461,9.460,9.492,9.521,9.588,9.663,9.700,
     * 9.418,9.396,9.358,9.310,9.258,9.220,9.190,9.161,
     * 9.155,9.137,9.104,9.057,9.009,8.970,8.942,8.914,
     * 9.110,9.091,9.058,9.011,8.963,8.924,8.896,8.869/
      DATA ((EDEN(I,4,2,L,3),I=49,56),L=1,NZEN)/
     * 11.340,11.356,11.360,11.342,11.328,11.312,11.322,11.330,
     * 11.279,11.297,11.310,11.294,11.281,11.265,11.274,11.279,
     * 11.201,11.220,11.236,11.220,11.207,11.193,11.204,11.207,
     * 11.068,11.086,11.104,11.090,11.076,11.064,11.072,11.076,
     * 10.806,10.831,10.845,10.829,10.815,10.799,10.804,10.804,
     * 10.632,10.667,10.688,10.675,10.663,10.646,10.648,10.643,
     * 10.340,10.384,10.413,10.413,10.413,10.401,10.408,10.400,
     * 10.053,10.053,10.049,10.017,9.981,9.952,9.936,9.921,
     * 9.436,9.425,9.399,9.373,9.344,9.353,9.370,9.392,
     * 9.377,9.356,9.318,9.250,9.179,9.117,9.064,9.009,
     * 9.114,9.093,9.053,8.990,8.923,8.862,8.810,8.759,
     * 9.068,9.048,9.007,8.944,8.877,8.816,8.764,8.713/
      DATA ((EDEN(I,5,2,L,3),I=49,56),L=1,NZEN)/
     * 11.344,11.364,11.373,11.354,11.336,11.314,11.316,11.324,
     * 11.276,11.301,11.322,11.303,11.288,11.267,11.272,11.276,
     * 11.196,11.225,11.248,11.230,11.215,11.196,11.201,11.207,
     * 11.061,11.090,11.111,11.097,11.086,11.064,11.068,11.072,
     * 10.800,10.831,10.852,10.838,10.823,10.801,10.799,10.801,
     * 10.625,10.667,10.695,10.683,10.670,10.647,10.644,10.640,
     * 10.330,10.384,10.417,10.418,10.420,10.403,10.405,10.400,
     * 10.041,10.057,10.049,10.021,9.989,9.962,9.948,9.934,
     * 9.430,9.433,9.391,9.344,9.295,9.285,9.290,9.320,
     * 9.367,9.358,9.324,9.258,9.190,9.124,9.068,9.017,
     * 9.104,9.093,9.061,8.998,8.931,8.867,8.815,8.766,
     * 9.058,9.048,9.015,8.953,8.885,8.822,8.769,8.721/
      DATA ((EDEN(I,1,3,L,3),I=49,56),L=1,NZEN)/
     * 11.336,11.328,11.314,11.297,11.283,11.281,11.303,11.316,
     * 11.276,11.274,11.270,11.250,11.241,11.238,11.260,11.272,
     * 11.201,11.199,11.199,11.182,11.173,11.170,11.193,11.204,
     * 11.072,11.068,11.072,11.061,11.053,11.054,11.073,11.080,
     * 10.820,10.823,10.832,10.817,10.807,10.807,10.826,10.831,
     * 10.656,10.668,10.689,10.676,10.667,10.666,10.682,10.684,
     * 10.393,10.413,10.458,10.453,10.453,10.454,10.471,10.469,
     * 10.134,10.117,10.117,10.093,10.068,10.061,10.061,10.061,
     * 9.465,9.428,9.400,9.420,9.442,9.513,9.594,9.642,
     * 9.408,9.367,9.305,9.250,9.199,9.161,9.134,9.104,
     * 9.146,9.107,9.049,8.997,8.946,8.912,8.885,8.856,
     * 9.100,9.061,9.003,8.951,8.901,8.866,8.840,8.810/
      DATA ((EDEN(I,2,3,L,3),I=49,56),L=1,NZEN)/
     * 11.328,11.328,11.322,11.305,11.297,11.292,11.312,11.320,
     * 11.272,11.274,11.276,11.260,11.250,11.248,11.267,11.274,
     * 11.196,11.201,11.207,11.190,11.182,11.179,11.199,11.204,
     * 11.064,11.068,11.079,11.068,11.061,11.061,11.079,11.083,
     * 10.814,10.823,10.835,10.823,10.814,10.814,10.831,10.831,
     * 10.650,10.668,10.695,10.683,10.675,10.672,10.685,10.683,
     * 10.386,10.413,10.459,10.458,10.459,10.459,10.471,10.467,
     * 10.130,10.117,10.124,10.100,10.076,10.064,10.061,10.053,
     * 9.459,9.430,9.414,9.436,9.457,9.524,9.596,9.634,
     * 9.401,9.367,9.314,9.261,9.208,9.171,9.137,9.100,
     * 9.140,9.107,9.057,9.009,8.956,8.920,8.889,8.854,
     * 9.094,9.061,9.011,8.963,8.910,8.874,8.844,8.808/
      DATA ((EDEN(I,3,3,L,3),I=49,56),L=1,NZEN)/
     * 11.324,11.330,11.330,11.320,11.312,11.310,11.326,11.334,
     * 11.262,11.272,11.283,11.272,11.265,11.262,11.279,11.288,
     * 11.188,11.199,11.212,11.204,11.199,11.199,11.215,11.220,
     * 11.057,11.072,11.086,11.079,11.076,11.076,11.093,11.100,
     * 10.810,10.826,10.843,10.836,10.832,10.833,10.848,10.852,
     * 10.652,10.677,10.704,10.699,10.696,10.696,10.708,10.712,
     * 10.403,10.436,10.476,10.480,10.487,10.493,10.508,10.508,
     * 10.155,10.158,10.170,10.155,10.143,10.137,10.134,10.130,
     * 9.461,9.446,9.442,9.466,9.492,9.555,9.620,9.659,
     * 9.403,9.382,9.348,9.303,9.260,9.228,9.199,9.170,
     * 9.143,9.124,9.093,9.049,9.009,8.979,8.952,8.923,
     * 9.097,9.078,9.048,9.003,8.963,8.933,8.906,8.877/
      DATA ((EDEN(I,4,3,L,3),I=49,56),L=1,NZEN)/
     * 11.320,11.334,11.340,11.330,11.324,11.318,11.328,11.332,
     * 11.258,11.272,11.290,11.283,11.279,11.272,11.283,11.286,
     * 11.182,11.199,11.220,11.212,11.210,11.201,11.212,11.215,
     * 11.049,11.064,11.086,11.079,11.076,11.076,11.083,11.086,
     * 10.792,10.814,10.834,10.827,10.822,10.818,10.826,10.822,
     * 10.625,10.656,10.683,10.679,10.677,10.671,10.675,10.667,
     * 10.348,10.389,10.423,10.431,10.439,10.441,10.449,10.439,
     * 10.072,10.079,10.079,10.061,10.041,10.025,10.013,10.004,
     * 9.423,9.413,9.389,9.372,9.356,9.382,9.411,9.444,
     * 9.365,9.346,9.316,9.262,9.207,9.161,9.117,9.068,
     * 9.104,9.086,9.053,9.004,8.951,8.908,8.865,8.820,
     * 9.058,9.041,9.007,8.959,8.905,8.862,8.819,8.774/
      DATA ((EDEN(I,5,3,L,3),I=49,56),L=1,NZEN)/
     * 11.318,11.334,11.346,11.336,11.330,11.322,11.328,11.332,
     * 11.250,11.270,11.297,11.290,11.286,11.276,11.281,11.283,
     * 11.170,11.193,11.223,11.217,11.215,11.207,11.210,11.212,
     * 11.037,11.064,11.093,11.090,11.086,11.079,11.086,11.086,
     * 10.781,10.810,10.839,10.835,10.832,10.826,10.829,10.827,
     * 10.613,10.651,10.689,10.688,10.688,10.681,10.681,10.676,
     * 10.334,10.384,10.433,10.444,10.456,10.456,10.461,10.455,
     * 10.057,10.079,10.083,10.076,10.064,10.057,10.049,10.041,
     * 9.413,9.420,9.365,9.341,9.315,9.347,9.377,9.424,
     * 9.354,9.346,9.316,9.270,9.220,9.179,9.138,9.097,
     * 9.090,9.086,9.057,9.013,8.967,8.928,8.887,8.848,
     * 9.044,9.041,9.011,8.967,8.921,8.883,8.841,8.802/
      DATA ((EDEN(I,1,4,L,3),I=49,56),L=1,NZEN)/
     * 11.330,11.322,11.307,11.290,11.276,11.274,11.297,11.307,
     * 11.274,11.270,11.262,11.246,11.233,11.230,11.250,11.262,
     * 11.199,11.196,11.196,11.176,11.164,11.164,11.185,11.196,
     * 11.068,11.064,11.068,11.057,11.049,11.050,11.070,11.076,
     * 10.816,10.820,10.829,10.814,10.804,10.804,10.824,10.828,
     * 10.654,10.666,10.690,10.675,10.665,10.663,10.679,10.680,
     * 10.389,10.410,10.461,10.455,10.453,10.453,10.468,10.467,
     * 10.130,10.111,10.117,10.093,10.068,10.061,10.061,10.061,
     * 9.461,9.425,9.409,9.440,9.472,9.546,9.627,9.673,
     * 9.403,9.362,9.297,9.246,9.193,9.158,9.130,9.100,
     * 9.143,9.100,9.041,8.992,8.942,8.908,8.883,8.854,
     * 9.097,9.055,8.996,8.946,8.896,8.863,8.837,8.808/
      DATA ((EDEN(I,2,4,L,3),I=49,56),L=1,NZEN)/
     * 11.318,11.316,11.312,11.299,11.292,11.292,11.312,11.318,
     * 11.258,11.260,11.265,11.253,11.248,11.250,11.270,11.274,
     * 11.182,11.185,11.196,11.185,11.182,11.182,11.201,11.204,
     * 11.053,11.057,11.068,11.061,11.061,11.064,11.079,11.083,
     * 10.805,10.812,10.827,10.818,10.814,10.818,10.834,10.833,
     * 10.644,10.660,10.688,10.679,10.675,10.677,10.689,10.685,
     * 10.380,10.405,10.453,10.455,10.459,10.464,10.477,10.470,
     * 10.121,10.107,10.111,10.093,10.076,10.068,10.064,10.060,
     * 9.447,9.417,9.396,9.417,9.440,9.507,9.574,9.613,
     * 9.389,9.354,9.301,9.253,9.204,9.173,9.140,9.104,
     * 9.127,9.093,9.045,9.000,8.954,8.922,8.892,8.856,
     * 9.081,9.048,9.000,8.954,8.908,8.876,8.846,8.810/
      DATA ((EDEN(I,3,4,L,3),I=49,56),L=1,NZEN)/
     * 11.303,11.307,11.312,11.310,11.310,11.316,11.332,11.338,
     * 11.243,11.248,11.265,11.262,11.262,11.270,11.288,11.292,
     * 11.170,11.176,11.196,11.193,11.196,11.204,11.223,11.225,
     * 11.041,11.049,11.068,11.072,11.076,11.083,11.100,11.104,
     * 10.795,10.808,10.831,10.831,10.834,10.844,10.860,10.861,
     * 10.640,10.663,10.692,10.695,10.701,10.710,10.723,10.723,
     * 10.394,10.427,10.467,10.481,10.497,10.511,10.525,10.522,
     * 10.146,10.149,10.158,10.152,10.149,10.152,10.152,10.146,
     * 9.442,9.428,9.420,9.445,9.468,9.533,9.590,9.634,
     * 9.386,9.364,9.332,9.299,9.262,9.243,9.217,9.185,
     * 9.127,9.107,9.076,9.045,9.013,8.993,8.968,8.938,
     * 9.081,9.061,9.030,9.000,8.967,8.948,8.922,8.892/
      DATA ((EDEN(I,4,4,L,3),I=49,56),L=1,NZEN)/
     * 11.292,11.301,11.314,11.316,11.322,11.330,11.340,11.340,
     * 11.228,11.238,11.265,11.270,11.274,11.283,11.294,11.294,
     * 11.155,11.167,11.193,11.196,11.204,11.215,11.228,11.225,
     * 11.021,11.037,11.061,11.068,11.079,11.090,11.104,11.100,
     * 10.773,10.791,10.818,10.823,10.832,10.844,10.853,10.847,
     * 10.614,10.640,10.675,10.683,10.693,10.703,10.710,10.701,
     * 10.350,10.389,10.433,10.452,10.474,10.490,10.500,10.489,
     * 10.086,10.097,10.100,10.100,10.100,10.104,10.097,10.086,
     * 9.407,9.398,9.378,9.383,9.389,9.438,9.475,9.520,
     * 9.350,9.334,9.303,9.270,9.236,9.212,9.176,9.137,
     * 9.090,9.076,9.045,9.017,8.983,8.962,8.928,8.887,
     * 9.044,9.030,9.000,8.971,8.937,8.916,8.882,8.841/
      DATA ((EDEN(I,5,4,L,3),I=49,56),L=1,NZEN)/
     * 11.276,11.290,11.307,11.316,11.328,11.340,11.348,11.346,
     * 11.212,11.228,11.260,11.270,11.281,11.294,11.303,11.301,
     * 11.140,11.155,11.188,11.199,11.212,11.225,11.236,11.233,
     * 11.004,11.025,11.057,11.072,11.086,11.104,11.114,11.111,
     * 10.756,10.781,10.820,10.831,10.845,10.862,10.869,10.865,
     * 10.595,10.629,10.679,10.694,10.710,10.727,10.732,10.728,
     * 10.334,10.382,10.446,10.474,10.501,10.525,10.534,10.528,
     * 10.068,10.097,10.114,10.130,10.143,10.161,10.161,10.155,
     * 9.391,9.394,9.339,9.359,9.379,9.455,9.511,9.568,
     * 9.332,9.328,9.301,9.283,9.262,9.255,9.225,9.193,
     * 9.072,9.072,9.045,9.029,9.013,9.004,8.978,8.945,
     * 9.026,9.026,9.000,8.984,8.967,8.959,8.932,8.900/
      DATA ((EDEN(I,1,5,L,3),I=49,56),L=1,NZEN)/
     * 11.324,11.314,11.299,11.279,11.265,11.262,11.283,11.294,
     * 11.274,11.270,11.253,11.236,11.223,11.220,11.241,11.250,
     * 11.199,11.193,11.188,11.170,11.158,11.155,11.179,11.185,
     * 11.064,11.064,11.068,11.053,11.041,11.037,11.061,11.068,
     * 10.817,10.819,10.828,10.812,10.800,10.800,10.818,10.822,
     * 10.655,10.666,10.693,10.676,10.664,10.662,10.677,10.678,
     * 10.389,10.410,10.473,10.464,10.458,10.456,10.470,10.467,
     * 10.127,10.107,10.117,10.093,10.076,10.068,10.068,10.068,
     * 9.455,9.420,9.423,9.465,9.508,9.586,9.670,9.710,
     * 9.398,9.356,9.286,9.236,9.188,9.155,9.130,9.100,
     * 9.137,9.097,9.033,8.985,8.938,8.906,8.881,8.854,
     * 9.091,9.051,8.988,8.940,8.892,8.860,8.836,8.809/
      DATA ((EDEN(I,2,5,L,3),I=49,56),L=1,NZEN)/
     * 11.310,11.305,11.301,11.292,11.288,11.292,11.312,11.314,
     * 11.250,11.250,11.255,11.246,11.243,11.248,11.265,11.267,
     * 11.176,11.176,11.185,11.179,11.179,11.182,11.199,11.201,
     * 11.045,11.045,11.064,11.057,11.057,11.064,11.083,11.079,
     * 10.799,10.804,10.822,10.816,10.814,10.821,10.836,10.832,
     * 10.640,10.654,10.684,10.679,10.677,10.682,10.694,10.688,
     * 10.378,10.401,10.456,10.459,10.467,10.474,10.484,10.474,
     * 10.114,10.100,10.104,10.090,10.079,10.079,10.076,10.068,
     * 9.435,9.405,9.383,9.407,9.434,9.502,9.567,9.606,
     * 9.377,9.340,9.288,9.243,9.201,9.173,9.143,9.107,
     * 9.117,9.083,9.033,8.992,8.951,8.925,8.896,8.861,
     * 9.072,9.037,8.988,8.946,8.905,8.880,8.850,8.815/
      DATA ((EDEN(I,3,5,L,3),I=49,56),L=1,NZEN)/
     * 11.288,11.288,11.294,11.299,11.307,11.322,11.338,11.338,
     * 11.225,11.230,11.248,11.253,11.260,11.276,11.292,11.292,
     * 11.152,11.158,11.179,11.188,11.196,11.212,11.228,11.228,
     * 11.025,11.029,11.053,11.064,11.076,11.093,11.107,11.107,
     * 10.784,10.794,10.818,10.826,10.837,10.855,10.869,10.868,
     * 10.631,10.650,10.684,10.693,10.706,10.723,10.735,10.732,
     * 10.179,10.196,10.223,10.220,10.220,10.217,10.199,10.158,
     * 10.134,10.140,10.146,10.152,10.155,10.170,10.167,10.161,
     * 9.427,9.410,9.410,9.442,9.473,9.542,9.597,9.641,
     * 9.369,9.346,9.314,9.288,9.262,9.253,9.228,9.196,
     * 9.111,9.090,9.061,9.037,9.013,9.004,8.981,8.951,
     * 9.065,9.044,9.015,8.992,8.967,8.959,8.935,8.905/
      DATA ((EDEN(I,4,5,L,3),I=49,56),L=1,NZEN)/
     * 11.260,11.270,11.286,11.301,11.318,11.342,11.356,11.350,
     * 11.204,11.212,11.238,11.253,11.272,11.294,11.310,11.303,
     * 11.130,11.140,11.167,11.185,11.207,11.230,11.241,11.236,
     * 11.000,11.013,11.045,11.061,11.083,11.107,11.121,11.114,
     * 10.757,10.772,10.803,10.822,10.842,10.868,10.879,10.870,
     * 10.603,10.627,10.667,10.688,10.709,10.733,10.741,10.731,
     * 10.352,10.389,10.442,10.473,10.504,10.534,10.543,10.531,
     * 10.093,10.107,10.117,10.130,10.146,10.167,10.164,10.152,
     * 9.393,9.384,9.411,9.441,9.470,9.540,9.582,9.621,
     * 9.334,9.320,9.292,9.276,9.260,9.258,9.228,9.188,
     * 9.076,9.064,9.037,9.025,9.009,9.009,8.980,8.942,
     * 9.030,9.019,8.992,8.980,8.963,8.963,8.934,8.896/
      DATA ((EDEN(I,5,5,L,3),I=49,56),L=1,NZEN)/
     * 11.236,11.248,11.272,11.297,11.324,11.356,11.369,11.365,
     * 11.179,11.193,11.225,11.250,11.279,11.312,11.322,11.318,
     * 11.104,11.121,11.155,11.185,11.215,11.248,11.258,11.253,
     * 10.977,10.993,11.029,11.061,11.097,11.130,11.143,11.137,
     * 10.734,10.756,10.798,10.829,10.860,10.898,10.907,10.903,
     * 10.581,10.612,10.667,10.699,10.732,10.770,10.778,10.773,
     * 10.330,10.380,10.452,10.497,10.542,10.583,10.593,10.587,
     * 10.072,10.107,10.134,10.167,10.204,10.241,10.243,10.236,
     * 9.369,9.373,9.418,9.458,9.497,9.588,9.641,9.692,
     * 9.314,9.312,9.290,9.297,9.301,9.316,9.294,9.265,
     * 9.053,9.057,9.037,9.045,9.053,9.068,9.049,9.017,
     * 9.007,9.011,8.992,9.000,9.007,9.022,9.003,8.971/
      DATA ((EDEN(I,1,6,L,3),I=49,56),L=1,NZEN)/
     * 11.322,11.310,11.294,11.272,11.255,11.250,11.274,11.283,
     * 11.274,11.267,11.250,11.228,11.212,11.210,11.230,11.241,
     * 11.199,11.193,11.185,11.164,11.149,11.146,11.167,11.176,
     * 11.068,11.064,11.064,11.049,11.037,11.033,11.053,11.057,
     * 10.820,10.820,10.830,10.812,10.799,10.797,10.815,10.818,
     * 10.659,10.668,10.695,10.678,10.664,10.660,10.674,10.675,
     * 10.393,10.413,10.483,10.471,10.462,10.459,10.471,10.466,
     * 10.121,10.104,10.121,10.100,10.083,10.076,10.079,10.079,
     * 9.450,9.417,9.436,9.489,9.543,9.618,9.701,9.739,
     * 9.393,9.350,9.281,9.230,9.185,9.152,9.130,9.104,
     * 9.130,9.090,9.029,8.981,8.934,8.905,8.883,8.858,
     * 9.085,9.044,8.984,8.935,8.889,8.859,8.837,8.812/
      DATA ((EDEN(I,2,6,L,3),I=49,56),L=1,NZEN)/
     * 11.301,11.297,11.292,11.286,11.283,11.288,11.305,11.307,
     * 11.246,11.246,11.248,11.241,11.238,11.246,11.262,11.262,
     * 11.170,11.170,11.179,11.173,11.173,11.179,11.196,11.196,
     * 11.041,11.041,11.057,11.053,11.057,11.064,11.083,11.079,
     * 10.797,10.800,10.819,10.815,10.815,10.824,10.838,10.833,
     * 10.639,10.651,10.683,10.679,10.679,10.686,10.696,10.688,
     * 10.378,10.400,10.461,10.465,10.473,10.481,10.491,10.479,
     * 10.107,10.093,10.100,10.093,10.086,10.090,10.090,10.083,
     * 9.425,9.396,9.378,9.415,9.451,9.521,9.583,9.621,
     * 9.367,9.330,9.276,9.238,9.199,9.176,9.146,9.111,
     * 9.107,9.072,9.021,8.985,8.949,8.928,8.900,8.866,
     * 9.061,9.026,8.975,8.940,8.904,8.883,8.855,8.821/
      DATA ((EDEN(I,3,6,L,3),I=49,56),L=1,NZEN)/
     * 11.274,11.274,11.283,11.292,11.305,11.324,11.340,11.340,
     * 11.217,11.217,11.236,11.248,11.260,11.281,11.297,11.294,
     * 11.143,11.146,11.167,11.179,11.193,11.215,11.233,11.228,
     * 11.017,11.021,11.045,11.057,11.076,11.097,11.111,11.111,
     * 10.777,10.785,10.810,10.823,10.839,10.863,10.876,10.872,
     * 10.626,10.642,10.679,10.692,10.708,10.732,10.743,10.736,
     * 10.382,10.412,10.462,10.486,10.512,10.538,10.549,10.540,
     * 10.124,10.130,10.140,10.149,10.158,10.179,10.179,10.170,
     * 9.412,9.396,9.409,9.452,9.495,9.570,9.622,9.663,
     * 9.354,9.330,9.301,9.281,9.262,9.260,9.236,9.201,
     * 9.097,9.076,9.045,9.029,9.013,9.013,8.987,8.956,
     * 9.051,9.030,9.000,8.984,8.967,8.967,8.941,8.910/
      DATA ((EDEN(I,4,6,L,3),I=49,56),L=1,NZEN)/
     * 11.241,11.246,11.267,11.290,11.318,11.350,11.365,11.358,
     * 11.188,11.193,11.220,11.246,11.272,11.305,11.318,11.312,
     * 11.117,11.124,11.152,11.176,11.204,11.238,11.250,11.243,
     * 10.986,10.995,11.025,11.053,11.086,11.121,11.134,11.124,
     * 10.747,10.760,10.792,10.819,10.848,10.883,10.894,10.883,
     * 10.597,10.618,10.659,10.688,10.718,10.751,10.759,10.747,
     * 10.352,10.387,10.444,10.480,10.520,10.558,10.566,10.550,
     * 10.090,10.107,10.117,10.143,10.167,10.196,10.193,10.179,
     * 9.377,9.369,9.460,9.495,9.531,9.603,9.640,9.675,
     * 9.320,9.307,9.281,9.276,9.272,9.279,9.250,9.212,
     * 9.061,9.049,9.029,9.025,9.021,9.033,9.004,8.966,
     * 9.015,9.003,8.984,8.980,8.975,8.988,8.959,8.920/
      DATA ((EDEN(I,5,6,L,3),I=49,56),L=1,NZEN)/
     * 11.204,11.215,11.243,11.286,11.326,11.371,11.382,11.375,
     * 11.158,11.170,11.201,11.241,11.281,11.328,11.340,11.332,
     * 11.086,11.097,11.130,11.173,11.217,11.262,11.274,11.265,
     * 10.958,10.972,11.013,11.053,11.097,11.146,11.158,11.149,
     * 10.719,10.739,10.781,10.824,10.868,10.918,10.929,10.921,
     * 10.571,10.599,10.653,10.698,10.743,10.792,10.801,10.792,
     * 10.328,10.375,10.449,10.504,10.558,10.612,10.621,10.611,
     * 10.072,10.107,10.134,10.179,10.228,10.276,10.276,10.265,
     * 9.350,9.354,9.490,9.526,9.561,9.656,9.700,9.744,
     * 9.294,9.294,9.277,9.297,9.316,9.346,9.324,9.292,
     * 9.037,9.041,9.025,9.049,9.068,9.100,9.079,9.045,
     * 8.992,8.996,8.980,9.003,9.022,9.055,9.033,9.000/
      DATA ((EDEN(I,1,7,L,3),I=49,56),L=1,NZEN)/
     * 11.320,11.307,11.290,11.270,11.255,11.250,11.270,11.279,
     * 11.274,11.270,11.248,11.225,11.210,11.204,11.225,11.236,
     * 11.199,11.193,11.182,11.161,11.149,11.146,11.167,11.173,
     * 11.068,11.064,11.064,11.049,11.037,11.033,11.053,11.057,
     * 10.822,10.822,10.829,10.812,10.799,10.797,10.815,10.818,
     * 10.662,10.671,10.696,10.678,10.664,10.661,10.675,10.675,
     * 10.398,10.417,10.486,10.473,10.464,10.461,10.473,10.468,
     * 10.124,10.104,10.121,10.100,10.083,10.079,10.083,10.083,
     * 9.447,9.413,9.448,9.504,9.559,9.632,9.714,9.748,
     * 9.389,9.348,9.279,9.230,9.185,9.152,9.130,9.104,
     * 9.127,9.090,9.029,8.982,8.936,8.907,8.885,8.861,
     * 9.081,9.044,8.984,8.936,8.890,8.861,8.840,8.815/
      DATA ((EDEN(I,2,7,L,3),I=49,56),L=1,NZEN)/
     * 11.301,11.292,11.288,11.281,11.281,11.288,11.305,11.305,
     * 11.246,11.243,11.243,11.236,11.236,11.243,11.260,11.260,
     * 11.170,11.170,11.176,11.173,11.173,11.182,11.196,11.196,
     * 11.041,11.041,11.057,11.053,11.057,11.064,11.079,11.076,
     * 10.797,10.800,10.818,10.814,10.815,10.825,10.839,10.833,
     * 10.641,10.652,10.683,10.679,10.679,10.688,10.698,10.689,
     * 10.380,10.401,10.464,10.467,10.474,10.484,10.493,10.481,
     * 10.107,10.093,10.100,10.093,10.086,10.093,10.093,10.086,
     * 9.420,9.393,9.385,9.426,9.469,9.541,9.605,9.638,
     * 9.362,9.326,9.274,9.236,9.199,9.179,9.149,9.114,
     * 9.104,9.068,9.021,8.984,8.950,8.930,8.903,8.869,
     * 9.058,9.022,8.975,8.938,8.904,8.885,8.857,8.823/
      DATA ((EDEN(I,3,7,L,3),I=49,56),L=1,NZEN)/
     * 11.270,11.270,11.279,11.290,11.303,11.326,11.342,11.340,
     * 11.215,11.215,11.230,11.243,11.258,11.281,11.297,11.294,
     * 11.140,11.143,11.164,11.176,11.193,11.215,11.230,11.228,
     * 11.013,11.017,11.041,11.057,11.076,11.100,11.114,11.111,
     * 10.775,10.782,10.808,10.822,10.839,10.865,10.879,10.874,
     * 10.625,10.640,10.677,10.691,10.709,10.734,10.744,10.738,
     * 10.382,10.410,10.462,10.487,10.513,10.542,10.551,10.543,
     * 10.121,10.124,10.137,10.149,10.161,10.182,10.182,10.173,
     * 9.407,9.394,9.415,9.466,9.517,9.594,9.648,9.685,
     * 9.348,9.324,9.297,9.279,9.265,9.262,9.236,9.204,
     * 9.090,9.068,9.045,9.029,9.013,9.013,8.990,8.958,
     * 9.044,9.022,9.000,8.984,8.967,8.967,8.944,8.912/
      DATA ((EDEN(I,4,7,L,3),I=49,56),L=1,NZEN)/
     * 11.236,11.241,11.262,11.290,11.318,11.352,11.365,11.358,
     * 11.185,11.190,11.217,11.243,11.274,11.307,11.318,11.312,
     * 11.111,11.121,11.146,11.176,11.207,11.241,11.253,11.243,
     * 10.983,10.991,11.021,11.053,11.083,11.121,11.134,11.121,
     * 10.743,10.754,10.787,10.815,10.846,10.882,10.890,10.877,
     * 10.592,10.612,10.652,10.682,10.713,10.748,10.754,10.738,
     * 10.344,10.377,10.431,10.471,10.512,10.549,10.555,10.537,
     * 10.076,10.090,10.100,10.124,10.149,10.179,10.170,10.155,
     * 9.365,9.358,9.454,9.492,9.529,9.601,9.635,9.663,
     * 9.310,9.292,9.267,9.262,9.260,9.265,9.233,9.190,
     * 9.053,9.037,9.017,9.013,9.009,9.017,8.986,8.943,
     * 9.007,8.992,8.971,8.967,8.963,8.971,8.941,8.898/
      DATA ((EDEN(I,5,7,L,3),I=49,56),L=1,NZEN)/
     * 11.199,11.210,11.241,11.286,11.328,11.375,11.384,11.375,
     * 11.155,11.164,11.196,11.241,11.283,11.330,11.342,11.330,
     * 11.079,11.093,11.127,11.173,11.217,11.267,11.276,11.265,
     * 10.954,10.966,11.004,11.053,11.100,11.149,11.161,11.146,
     * 10.713,10.731,10.772,10.819,10.865,10.916,10.925,10.912,
     * 10.563,10.591,10.643,10.690,10.737,10.788,10.794,10.780,
     * 10.320,10.364,10.435,10.490,10.547,10.601,10.607,10.591,
     * 10.057,10.086,10.111,10.158,10.204,10.255,10.253,10.236,
     * 9.338,9.340,9.471,9.510,9.549,9.642,9.682,9.721,
     * 9.283,9.281,9.263,9.281,9.301,9.328,9.301,9.262,
     * 9.025,9.025,9.013,9.033,9.053,9.083,9.057,9.017,
     * 8.980,8.980,8.967,8.988,9.007,9.037,9.011,8.971/
      DATA ((EDEN(I,1,8,L,3),I=49,56),L=1,NZEN)/
     * 11.322,11.312,11.297,11.276,11.260,11.258,11.281,11.290,
     * 11.274,11.270,11.250,11.230,11.217,11.215,11.238,11.246,
     * 11.199,11.196,11.185,11.167,11.155,11.152,11.173,11.182,
     * 11.068,11.064,11.064,11.049,11.037,11.033,11.057,11.061,
     * 10.821,10.822,10.828,10.812,10.800,10.799,10.817,10.820,
     * 10.661,10.670,10.695,10.677,10.664,10.661,10.677,10.678,
     * 10.396,10.415,10.479,10.468,10.461,10.459,10.473,10.470,
     * 10.127,10.111,10.121,10.100,10.083,10.076,10.079,10.079,
     * 9.452,9.417,9.448,9.498,9.550,9.623,9.704,9.738,
     * 9.394,9.352,9.286,9.236,9.188,9.158,9.134,9.107,
     * 9.134,9.093,9.033,8.986,8.941,8.910,8.887,8.861,
     * 9.088,9.048,8.988,8.941,8.895,8.864,8.841,8.815/
      DATA ((EDEN(I,2,8,L,3),I=49,56),L=1,NZEN)/
     * 11.303,11.297,11.292,11.286,11.283,11.290,11.307,11.310,
     * 11.248,11.248,11.248,11.241,11.238,11.246,11.265,11.267,
     * 11.176,11.176,11.179,11.173,11.173,11.182,11.199,11.199,
     * 11.045,11.045,11.061,11.053,11.053,11.064,11.083,11.079,
     * 10.798,10.801,10.819,10.814,10.814,10.823,10.838,10.834,
     * 10.640,10.652,10.683,10.679,10.679,10.686,10.696,10.690,
     * 10.380,10.403,10.461,10.465,10.473,10.481,10.491,10.481,
     * 10.111,10.097,10.107,10.097,10.090,10.093,10.090,10.083,
     * 9.427,9.398,9.397,9.441,9.485,9.556,9.621,9.655,
     * 9.369,9.332,9.281,9.241,9.204,9.182,9.152,9.117,
     * 9.107,9.076,9.025,8.989,8.954,8.932,8.904,8.870,
     * 9.061,9.030,8.980,8.944,8.908,8.886,8.858,8.825/
      DATA ((EDEN(I,3,8,L,3),I=49,56),L=1,NZEN)/
     * 11.276,11.276,11.283,11.292,11.305,11.324,11.342,11.342,
     * 11.220,11.223,11.238,11.248,11.260,11.279,11.297,11.297,
     * 11.146,11.149,11.170,11.182,11.196,11.215,11.230,11.230,
     * 11.021,11.025,11.045,11.057,11.076,11.097,11.114,11.114,
     * 10.778,10.787,10.811,10.823,10.838,10.860,10.874,10.872,
     * 10.626,10.643,10.678,10.691,10.708,10.729,10.740,10.736,
     * 10.382,10.410,10.461,10.483,10.509,10.534,10.545,10.539,
     * 10.124,10.127,10.140,10.149,10.158,10.176,10.176,10.167,
     * 9.413,9.400,9.425,9.475,9.526,9.601,9.655,9.692,
     * 9.356,9.332,9.303,9.283,9.265,9.258,9.233,9.201,
     * 9.097,9.076,9.049,9.033,9.013,9.009,8.986,8.955,
     * 9.051,9.030,9.003,8.988,8.967,8.963,8.940,8.909/
      DATA ((EDEN(I,4,8,L,3),I=49,56),L=1,NZEN)/
     * 11.248,11.255,11.274,11.297,11.322,11.350,11.362,11.356,
     * 11.196,11.201,11.225,11.250,11.276,11.305,11.316,11.307,
     * 11.117,11.127,11.155,11.182,11.207,11.236,11.248,11.241,
     * 10.991,11.000,11.029,11.057,11.086,11.114,11.124,11.114,
     * 10.746,10.757,10.788,10.812,10.838,10.868,10.875,10.862,
     * 10.590,10.610,10.649,10.674,10.701,10.729,10.733,10.717,
     * 10.330,10.364,10.415,10.450,10.487,10.520,10.525,10.505,
     * 10.057,10.064,10.072,10.090,10.107,10.127,10.121,10.100,
     * 9.365,9.356,9.404,9.438,9.474,9.541,9.574,9.600,
     * 9.310,9.290,9.262,9.248,9.236,9.230,9.193,9.143,
     * 9.049,9.033,9.009,8.995,8.983,8.981,8.944,8.897,
     * 9.003,8.988,8.963,8.949,8.937,8.935,8.899,8.851/
      DATA ((EDEN(I,5,8,L,3),I=49,56),L=1,NZEN)/
     * 11.220,11.228,11.255,11.294,11.332,11.371,11.380,11.369,
     * 11.167,11.179,11.210,11.248,11.288,11.326,11.334,11.322,
     * 11.093,11.104,11.140,11.179,11.220,11.258,11.265,11.253,
     * 10.963,10.977,11.013,11.057,11.097,11.137,11.143,11.130,
     * 10.719,10.736,10.776,10.815,10.854,10.896,10.900,10.886,
     * 10.562,10.590,10.641,10.681,10.720,10.761,10.763,10.747,
     * 10.305,10.348,10.415,10.465,10.516,10.560,10.565,10.547,
     * 10.033,10.061,10.076,10.114,10.149,10.190,10.185,10.167,
     * 9.340,9.338,9.389,9.431,9.471,9.560,9.601,9.637,
     * 9.283,9.276,9.253,9.260,9.265,9.279,9.246,9.201,
     * 9.021,9.017,9.000,9.009,9.017,9.033,9.000,8.957,
     * 8.975,8.971,8.954,8.963,8.971,8.988,8.954,8.911/
      DATA ((EDEN(I,1,9,L,3),I=49,56),L=1,NZEN)/
     * 11.324,11.316,11.303,11.283,11.272,11.270,11.294,11.305,
     * 11.274,11.270,11.258,11.238,11.225,11.225,11.248,11.260,
     * 11.199,11.196,11.190,11.173,11.161,11.161,11.182,11.193,
     * 11.068,11.064,11.068,11.053,11.041,11.041,11.064,11.072,
     * 10.818,10.820,10.826,10.811,10.800,10.801,10.820,10.824,
     * 10.655,10.667,10.690,10.675,10.664,10.662,10.678,10.680,
     * 10.389,10.410,10.467,10.461,10.456,10.456,10.471,10.470,
     * 10.130,10.111,10.121,10.100,10.079,10.072,10.072,10.072,
     * 9.462,9.425,9.437,9.480,9.524,9.600,9.684,9.724,
     * 9.405,9.362,9.297,9.246,9.196,9.164,9.137,9.111,
     * 9.143,9.100,9.045,8.995,8.946,8.914,8.890,8.862,
     * 9.097,9.055,9.000,8.949,8.901,8.869,8.844,8.815/
      DATA ((EDEN(I,2,9,L,3),I=49,56),L=1,NZEN)/
     * 11.310,11.305,11.301,11.292,11.288,11.292,11.314,11.318,
     * 11.255,11.258,11.255,11.248,11.243,11.248,11.267,11.272,
     * 11.182,11.182,11.188,11.179,11.179,11.185,11.201,11.207,
     * 11.049,11.053,11.061,11.057,11.057,11.064,11.079,11.083,
     * 10.800,10.806,10.821,10.815,10.814,10.820,10.834,10.834,
     * 10.638,10.653,10.682,10.677,10.675,10.680,10.692,10.689,
     * 10.375,10.400,10.456,10.459,10.465,10.473,10.484,10.479,
     * 10.114,10.100,10.114,10.100,10.090,10.086,10.083,10.076,
     * 9.442,9.412,9.415,9.457,9.499,9.577,9.653,9.685,
     * 9.386,9.348,9.297,9.253,9.210,9.182,9.152,9.117,
     * 9.124,9.090,9.041,9.000,8.960,8.933,8.905,8.870,
     * 9.078,9.044,8.996,8.954,8.914,8.888,8.859,8.825/
      DATA ((EDEN(I,3,9,L,3),I=49,56),L=1,NZEN)/
     * 11.290,11.292,11.297,11.301,11.307,11.320,11.338,11.344,
     * 11.236,11.238,11.250,11.255,11.262,11.278,11.296,11.301,
     * 11.158,11.164,11.179,11.185,11.196,11.210,11.225,11.228,
     * 11.029,11.037,11.057,11.064,11.076,11.090,11.104,11.107,
     * 10.785,10.796,10.818,10.825,10.835,10.851,10.865,10.866,
     * 10.628,10.649,10.682,10.690,10.702,10.717,10.728,10.728,
     * 10.380,10.410,10.461,10.479,10.500,10.519,10.531,10.529,
     * 10.130,10.130,10.149,10.152,10.152,10.161,10.161,10.155,
     * 9.433,9.415,9.435,9.486,9.538,9.612,9.673,9.709,
     * 9.375,9.350,9.318,9.292,9.265,9.250,9.223,9.190,
     * 9.114,9.093,9.064,9.037,9.013,9.000,8.976,8.945,
     * 9.068,9.048,9.019,8.992,8.967,8.954,8.931,8.900/
      DATA ((EDEN(I,4,9,L,3),I=49,56),L=1,NZEN)/
     * 11.272,11.281,11.297,11.310,11.326,11.344,11.354,11.352,
     * 11.215,11.225,11.248,11.260,11.276,11.294,11.305,11.303,
     * 11.140,11.149,11.176,11.190,11.210,11.228,11.238,11.233,
     * 11.009,11.017,11.041,11.061,11.079,11.100,11.107,11.104,
     * 10.755,10.770,10.797,10.811,10.828,10.846,10.852,10.842,
     * 10.591,10.615,10.650,10.666,10.684,10.701,10.703,10.689,
     * 10.316,10.350,10.398,10.425,10.455,10.477,10.481,10.465,
     * 10.033,10.037,10.045,10.049,10.053,10.061,10.049,10.033,
     * 9.382,9.367,9.371,9.395,9.418,9.472,9.502,9.525,
     * 9.324,9.301,9.270,9.241,9.210,9.188,9.143,9.093,
     * 9.061,9.037,9.013,8.984,8.957,8.938,8.895,8.844,
     * 9.015,8.992,8.967,8.938,8.911,8.893,8.852,8.800/
      DATA ((EDEN(I,5,9,L,3),I=49,56),L=1,NZEN)/
     * 11.255,11.267,11.288,11.310,11.334,11.358,11.364,11.358,
     * 11.193,11.210,11.241,11.262,11.288,11.310,11.314,11.310,
     * 11.117,11.134,11.167,11.193,11.220,11.241,11.246,11.238,
     * 10.986,11.004,11.037,11.064,11.090,11.114,11.121,11.114,
     * 10.732,10.753,10.790,10.814,10.840,10.864,10.865,10.854,
     * 10.566,10.597,10.645,10.671,10.698,10.721,10.719,10.706,
     * 10.290,10.332,10.398,10.436,10.474,10.502,10.504,10.489,
     * 10.009,10.025,10.041,10.061,10.079,10.104,10.093,10.076,
     * 9.360,9.354,9.332,9.359,9.385,9.456,9.492,9.526,
     * 9.301,9.288,9.260,9.243,9.228,9.217,9.176,9.127,
     * 9.041,9.025,9.004,8.989,8.975,8.968,8.926,8.880,
     * 8.996,8.980,8.959,8.943,8.929,8.922,8.881,8.834/
      DATA ((EDEN(I,1,10,L,3),I=49,56),L=1,NZEN)/
     * 11.330,11.322,11.307,11.292,11.281,11.279,11.303,11.318,
     * 11.274,11.272,11.262,11.248,11.236,11.236,11.258,11.270,
     * 11.199,11.196,11.196,11.179,11.167,11.170,11.193,11.204,
     * 11.068,11.064,11.068,11.053,11.045,11.045,11.068,11.079,
     * 10.816,10.819,10.828,10.812,10.801,10.803,10.823,10.829,
     * 10.649,10.663,10.688,10.674,10.665,10.664,10.679,10.683,
     * 10.382,10.403,10.459,10.455,10.453,10.455,10.471,10.471,
     * 10.130,10.111,10.124,10.100,10.079,10.068,10.068,10.068,
     * 9.474,9.435,9.425,9.456,9.490,9.569,9.656,9.705,
     * 9.417,9.373,9.307,9.255,9.204,9.170,9.143,9.114,
     * 9.152,9.111,9.053,9.004,8.953,8.920,8.894,8.865,
     * 9.107,9.065,9.007,8.959,8.907,8.874,8.848,8.819/
      DATA ((EDEN(I,2,10,L,3),I=49,56),L=1,NZEN)/
     * 11.318,11.316,11.310,11.299,11.292,11.292,11.314,11.324,
     * 11.262,11.265,11.262,11.253,11.248,11.248,11.267,11.276,
     * 11.188,11.190,11.193,11.185,11.179,11.182,11.199,11.210,
     * 11.057,11.061,11.068,11.057,11.057,11.061,11.079,11.083,
     * 10.804,10.813,10.827,10.818,10.812,10.814,10.829,10.833,
     * 10.637,10.657,10.688,10.679,10.674,10.675,10.687,10.687,
     * 10.371,10.398,10.458,10.458,10.461,10.465,10.479,10.474,
     * 10.117,10.104,10.127,10.107,10.090,10.083,10.079,10.072,
     * 9.462,9.430,9.433,9.471,9.511,9.591,9.675,9.715,
     * 9.405,9.367,9.310,9.262,9.215,9.182,9.152,9.117,
     * 9.143,9.104,9.053,9.009,8.965,8.934,8.904,8.870,
     * 9.097,9.058,9.007,8.963,8.919,8.888,8.858,8.825/
      DATA ((EDEN(I,3,10,L,3),I=49,56),L=1,NZEN)/
     * 11.307,11.310,11.312,11.310,11.310,11.314,11.330,11.340,
     * 11.248,11.258,11.267,11.262,11.262,11.267,11.283,11.294,
     * 11.173,11.182,11.196,11.193,11.196,11.201,11.217,11.225,
     * 11.045,11.053,11.068,11.068,11.072,11.076,11.093,11.104,
     * 10.794,10.810,10.830,10.829,10.831,10.838,10.852,10.859,
     * 10.632,10.658,10.692,10.693,10.697,10.704,10.716,10.720,
     * 10.380,10.413,10.467,10.479,10.491,10.504,10.517,10.520,
     * 10.137,10.140,10.164,10.158,10.152,10.152,10.149,10.146,
     * 9.458,9.438,9.451,9.497,9.544,9.626,9.704,9.742,
     * 9.401,9.375,9.338,9.303,9.267,9.243,9.215,9.185,
     * 9.140,9.114,9.083,9.049,9.017,8.994,8.968,8.939,
     * 9.094,9.068,9.037,9.003,8.971,8.949,8.922,8.893/
      DATA ((EDEN(I,4,10,L,3),I=49,56),L=1,NZEN)/
     * 11.301,11.312,11.324,11.324,11.328,11.332,11.342,11.346,
     * 11.241,11.255,11.272,11.274,11.281,11.283,11.294,11.299,
     * 11.161,11.176,11.199,11.201,11.207,11.212,11.223,11.225,
     * 11.029,11.045,11.068,11.068,11.076,11.079,11.090,11.090,
     * 10.770,10.790,10.810,10.812,10.816,10.820,10.825,10.821,
     * 10.595,10.625,10.656,10.661,10.667,10.670,10.671,10.662,
     * 10.301,10.342,10.387,10.403,10.422,10.431,10.436,10.423,
     * 10.017,10.017,10.025,10.009,9.997,9.989,9.973,9.954,
     * 9.408,9.389,9.368,9.371,9.372,9.409,9.437,9.457,
     * 9.350,9.324,9.286,9.238,9.188,9.146,9.097,9.041,
     * 9.086,9.061,9.025,8.978,8.931,8.892,8.843,8.790,
     * 9.041,9.015,8.980,8.932,8.885,8.846,8.797,8.744/
      DATA ((EDEN(I,5,10,L,3),I=49,56),L=1,NZEN)/
     * 11.294,11.307,11.322,11.328,11.338,11.342,11.346,11.350,
     * 11.228,11.248,11.272,11.279,11.288,11.292,11.297,11.299,
     * 11.152,11.173,11.196,11.207,11.217,11.223,11.225,11.225,
     * 11.017,11.037,11.064,11.076,11.086,11.090,11.093,11.093,
     * 10.753,10.780,10.811,10.819,10.827,10.831,10.830,10.826,
     * 10.577,10.614,10.657,10.667,10.679,10.682,10.678,10.670,
     * 10.281,10.328,10.389,10.413,10.436,10.447,10.447,10.438,
     * 9.997,10.009,10.021,10.021,10.017,10.017,10.009,9.990,
     * 9.393,9.387,9.353,9.346,9.339,9.373,9.391,9.418,
     * 9.332,9.316,9.281,9.243,9.201,9.164,9.114,9.064,
     * 9.068,9.053,9.021,8.984,8.945,8.912,8.863,8.815,
     * 9.022,9.007,8.975,8.938,8.900,8.866,8.817,8.769/
      DATA ((EDEN(I,1,11,L,3),I=49,56),L=1,NZEN)/
     * 11.334,11.326,11.314,11.297,11.286,11.283,11.307,11.322,
     * 11.276,11.274,11.267,11.250,11.241,11.238,11.260,11.276,
     * 11.201,11.199,11.196,11.182,11.173,11.170,11.193,11.207,
     * 11.072,11.068,11.068,11.057,11.049,11.049,11.072,11.083,
     * 10.816,10.820,10.829,10.814,10.804,10.805,10.825,10.832,
     * 10.647,10.663,10.690,10.677,10.667,10.666,10.682,10.685,
     * 10.378,10.400,10.459,10.455,10.455,10.456,10.473,10.474,
     * 10.130,10.111,10.124,10.100,10.079,10.072,10.072,10.072,
     * 9.481,9.441,9.410,9.432,9.454,9.536,9.626,9.681,
     * 9.425,9.380,9.312,9.260,9.207,9.173,9.146,9.117,
     * 9.161,9.117,9.057,9.004,8.956,8.922,8.897,8.869,
     * 9.116,9.072,9.011,8.959,8.910,8.876,8.851,8.823/
      DATA ((EDEN(I,2,11,L,3),I=49,56),L=1,NZEN)/
     * 11.326,11.326,11.320,11.303,11.292,11.290,11.312,11.322,
     * 11.272,11.276,11.272,11.255,11.246,11.243,11.265,11.276,
     * 11.199,11.201,11.199,11.188,11.179,11.176,11.196,11.207,
     * 11.064,11.072,11.076,11.064,11.053,11.053,11.072,11.083,
     * 10.811,10.822,10.834,10.821,10.811,10.809,10.825,10.831,
     * 10.640,10.663,10.696,10.683,10.674,10.671,10.683,10.686,
     * 10.373,10.400,10.467,10.464,10.462,10.462,10.474,10.474,
     * 10.124,10.114,10.137,10.117,10.093,10.083,10.079,10.076,
     * 9.479,9.444,9.447,9.485,9.520,9.602,9.690,9.737,
     * 9.420,9.382,9.320,9.270,9.220,9.182,9.153,9.118,
     * 9.155,9.121,9.064,9.017,8.968,8.932,8.904,8.872,
     * 9.110,9.075,9.019,8.971,8.922,8.887,8.858,8.826/
      DATA ((EDEN(I,3,11,L,3),I=49,56),L=1,NZEN)/
     * 11.322,11.328,11.328,11.316,11.307,11.303,11.322,11.336,
     * 11.270,11.279,11.281,11.270,11.262,11.258,11.276,11.290,
     * 11.190,11.204,11.210,11.199,11.190,11.188,11.204,11.220,
     * 11.061,11.072,11.083,11.072,11.068,11.064,11.083,11.093,
     * 10.808,10.826,10.841,10.833,10.827,10.825,10.839,10.849,
     * 10.640,10.670,10.703,10.697,10.693,10.691,10.702,10.710,
     * 10.386,10.420,10.480,10.483,10.487,10.490,10.504,10.509,
     * 10.149,10.149,10.179,10.164,10.149,10.140,10.137,10.134,
     * 9.480,9.459,9.467,9.511,9.553,9.635,9.724,9.767,
     * 9.422,9.396,9.352,9.310,9.265,9.230,9.204,9.176,
     * 9.158,9.134,9.097,9.057,9.013,8.983,8.956,8.928,
     * 9.113,9.088,9.051,9.011,8.967,8.937,8.910,8.883/
      DATA ((EDEN(I,4,11,L,3),I=49,56),L=1,NZEN)/
     * 11.328,11.340,11.344,11.336,11.328,11.318,11.328,11.340,
     * 11.267,11.286,11.294,11.286,11.281,11.270,11.279,11.290,
     * 11.190,11.210,11.220,11.212,11.207,11.199,11.210,11.215,
     * 11.053,11.072,11.086,11.079,11.072,11.064,11.072,11.079,
     * 10.788,10.813,10.826,10.817,10.810,10.800,10.803,10.803,
     * 10.606,10.642,10.668,10.662,10.656,10.644,10.644,10.640,
     * 10.301,10.346,10.387,10.393,10.400,10.394,10.400,10.393,
     * 10.017,10.009,10.017,9.986,9.955,9.931,9.912,9.894,
     * 9.435,9.415,9.381,9.364,9.346,9.364,9.389,9.406,
     * 9.375,9.348,9.299,9.238,9.173,9.114,9.057,9.000,
     * 9.107,9.083,9.037,8.977,8.915,8.857,8.801,8.747,
     * 9.061,9.037,8.994,8.933,8.870,8.811,8.755,8.702/
      DATA ((EDEN(I,5,11,L,3),I=49,56),L=1,NZEN)/
     * 11.330,11.346,11.356,11.348,11.340,11.326,11.328,11.338,
     * 11.262,11.286,11.305,11.299,11.292,11.279,11.281,11.288,
     * 11.182,11.210,11.228,11.223,11.215,11.201,11.204,11.212,
     * 11.045,11.072,11.093,11.090,11.086,11.072,11.072,11.079,
     * 10.777,10.809,10.831,10.825,10.819,10.803,10.799,10.801,
     * 10.595,10.637,10.670,10.667,10.664,10.647,10.640,10.638,
     * 10.283,10.336,10.382,10.393,10.403,10.394,10.394,10.389,
     * 9.991,10.000,10.000,9.978,9.952,9.931,9.914,9.897,
     * 9.423,9.418,9.379,9.346,9.311,9.305,9.302,9.321,
     * 9.362,9.344,9.301,9.241,9.179,9.117,9.053,9.000,
     * 9.097,9.079,9.037,8.980,8.920,8.860,8.801,8.750,
     * 9.051,9.033,8.992,8.934,8.874,8.814,8.756,8.704/
      DATA ((EDEN(I,1,12,L,3),I=49,56),L=1,NZEN)/
     * 11.338,11.330,11.318,11.299,11.286,11.286,11.310,11.322,
     * 11.279,11.276,11.270,11.253,11.243,11.241,11.265,11.276,
     * 11.204,11.204,11.201,11.185,11.176,11.173,11.196,11.207,
     * 11.072,11.072,11.072,11.061,11.053,11.053,11.072,11.083,
     * 10.820,10.823,10.833,10.818,10.808,10.807,10.825,10.833,
     * 10.652,10.667,10.695,10.680,10.670,10.668,10.684,10.688,
     * 10.384,10.405,10.464,10.459,10.459,10.461,10.476,10.477,
     * 10.134,10.114,10.124,10.104,10.083,10.072,10.072,10.076,
     * 9.481,9.442,9.399,9.413,9.429,9.510,9.597,9.656,
     * 9.425,9.380,9.314,9.260,9.207,9.173,9.146,9.117,
     * 9.161,9.117,9.057,9.004,8.956,8.922,8.897,8.869,
     * 9.116,9.072,9.011,8.959,8.910,8.876,8.851,8.823/
      DATA ((EDEN(I,2,12,L,3),I=49,56),L=1,NZEN)/
     * 11.334,11.332,11.324,11.305,11.290,11.286,11.307,11.318,
     * 11.283,11.288,11.279,11.260,11.248,11.241,11.258,11.272,
     * 11.207,11.212,11.207,11.190,11.179,11.173,11.193,11.204,
     * 11.076,11.079,11.083,11.068,11.057,11.049,11.068,11.079,
     * 10.819,10.831,10.842,10.825,10.812,10.807,10.822,10.828,
     * 10.648,10.671,10.704,10.688,10.676,10.668,10.679,10.683,
     * 10.380,10.408,10.476,10.469,10.465,10.461,10.473,10.471,
     * 10.130,10.121,10.143,10.117,10.093,10.079,10.076,10.072,
     * 9.484,9.450,9.449,9.484,9.518,9.597,9.686,9.735,
     * 9.427,9.389,9.324,9.270,9.217,9.176,9.146,9.114,
     * 9.164,9.127,9.068,9.017,8.967,8.928,8.898,8.868,
     * 9.119,9.081,9.022,8.971,8.921,8.882,8.852,8.822/
      DATA ((EDEN(I,3,12,L,3),I=49,56),L=1,NZEN)/
     * 11.334,11.340,11.336,11.320,11.307,11.297,11.314,11.328,
     * 11.281,11.292,11.288,11.274,11.262,11.250,11.265,11.281,
     * 11.204,11.217,11.220,11.204,11.190,11.182,11.201,11.215,
     * 11.072,11.086,11.093,11.079,11.068,11.057,11.072,11.086,
     * 10.819,10.839,10.852,10.838,10.826,10.818,10.830,10.841,
     * 10.652,10.683,10.715,10.702,10.692,10.682,10.693,10.701,
     * 10.396,10.431,10.490,10.487,10.486,10.480,10.493,10.498,
     * 10.155,10.158,10.188,10.164,10.143,10.127,10.124,10.121,
     * 9.490,9.471,9.479,9.518,9.556,9.632,9.720,9.765,
     * 9.431,9.407,9.360,9.310,9.260,9.220,9.190,9.164,
     * 9.167,9.146,9.104,9.057,9.009,8.970,8.943,8.915,
     * 9.122,9.100,9.058,9.011,8.963,8.924,8.897,8.870/
      DATA ((EDEN(I,4,12,L,3),I=49,56),L=1,NZEN)/
     * 11.344,11.360,11.364,11.346,11.330,11.312,11.320,11.332,
     * 11.286,11.305,11.310,11.292,11.279,11.260,11.272,11.283,
     * 11.207,11.228,11.236,11.220,11.207,11.188,11.196,11.210,
     * 11.072,11.093,11.104,11.086,11.072,11.057,11.064,11.076,
     * 10.804,10.831,10.841,10.823,10.808,10.787,10.790,10.794,
     * 10.623,10.661,10.680,10.667,10.654,10.632,10.631,10.630,
     * 10.314,10.360,10.398,10.396,10.393,10.378,10.382,10.377,
     * 10.025,10.021,10.017,9.979,9.938,9.902,9.881,9.864,
     * 9.449,9.431,9.394,9.368,9.338,9.346,9.365,9.381,
     * 9.387,9.364,9.310,9.238,9.164,9.093,9.033,8.978,
     * 9.121,9.097,9.045,8.978,8.907,8.837,8.780,8.725,
     * 9.075,9.051,9.000,8.932,8.861,8.791,8.734,8.679/
      DATA ((EDEN(I,5,12,L,3),I=49,56),L=1,NZEN)/
     * 11.350,11.371,11.378,11.360,11.344,11.318,11.320,11.332,
     * 11.283,11.310,11.326,11.310,11.292,11.267,11.272,11.281,
     * 11.204,11.233,11.248,11.233,11.220,11.196,11.196,11.207,
     * 11.068,11.097,11.114,11.100,11.083,11.061,11.061,11.072,
     * 10.799,10.833,10.849,10.832,10.816,10.788,10.784,10.790,
     * 10.614,10.659,10.685,10.671,10.658,10.629,10.623,10.623,
     * 10.299,10.354,10.387,10.389,10.389,10.369,10.367,10.365,
     * 10.000,10.009,10.000,9.961,9.920,9.884,9.862,9.844,
     * 9.439,9.438,9.396,9.348,9.297,9.271,9.259,9.272,
     * 9.377,9.362,9.318,9.246,9.170,9.090,9.021,8.968,
     * 9.107,9.097,9.053,8.982,8.908,8.831,8.767,8.714,
     * 9.061,9.051,9.007,8.936,8.862,8.785,8.721,8.669/
      DATA ((EDEN(I,1,1,L,1),I=57,64),L=1,NZEN)/
     * 11.114,11.124,11.130,11.121,11.107,11.140,11.173,11.173,
     * 11.079,11.090,11.093,11.083,11.068,11.100,11.130,11.127,
     * 11.029,11.037,11.037,11.025,11.009,11.037,11.064,11.064,
     * 10.937,10.940,10.940,10.926,10.904,10.931,10.956,10.951,
     * 10.740,10.739,10.735,10.719,10.693,10.714,10.733,10.726,
     * 10.619,10.613,10.606,10.590,10.563,10.580,10.595,10.588,
     * 10.436,10.423,10.412,10.393,10.364,10.369,10.375,10.367,
     * 10.057,10.045,10.037,10.017,9.989,9.968,9.952,9.938,
     * 9.566,9.595,9.626,9.641,9.651,9.644,9.645,9.647,
     * 9.149,9.111,9.072,9.033,8.994,8.960,8.934,8.914,
     * 8.914,8.875,8.837,8.801,8.760,8.724,8.697,8.676,
     * 8.869,8.829,8.791,8.756,8.715,8.679,8.651,8.630/
      DATA ((EDEN(I,2,1,L,1),I=57,64),L=1,NZEN)/
     * 11.111,11.114,11.114,11.107,11.093,11.124,11.149,11.149,
     * 11.079,11.079,11.076,11.068,11.053,11.083,11.107,11.104,
     * 11.029,11.025,11.021,11.013,10.994,11.021,11.045,11.037,
     * 10.934,10.930,10.923,10.910,10.890,10.913,10.933,10.927,
     * 10.737,10.728,10.717,10.700,10.677,10.694,10.709,10.700,
     * 10.613,10.602,10.589,10.569,10.545,10.558,10.571,10.560,
     * 10.428,10.408,10.389,10.369,10.342,10.342,10.346,10.336,
     * 10.041,10.025,10.013,9.986,9.957,9.933,9.915,9.899,
     * 9.619,9.631,9.644,9.644,9.642,9.627,9.620,9.618,
     * 9.140,9.093,9.045,9.004,8.963,8.925,8.897,8.874,
     * 8.905,8.858,8.813,8.773,8.732,8.690,8.659,8.635,
     * 8.859,8.812,8.767,8.727,8.686,8.644,8.613,8.590/
      DATA ((EDEN(I,3,1,L,1),I=57,64),L=1,NZEN)/
     * 11.124,11.124,11.117,11.114,11.107,11.134,11.152,11.158,
     * 11.090,11.086,11.079,11.076,11.068,11.090,11.111,11.114,
     * 11.037,11.033,11.025,11.021,11.009,11.029,11.049,11.049,
     * 10.945,10.939,10.928,10.919,10.907,10.925,10.941,10.941,
     * 10.748,10.737,10.725,10.713,10.698,10.712,10.723,10.722,
     * 10.627,10.614,10.599,10.587,10.572,10.581,10.590,10.588,
     * 10.446,10.428,10.410,10.394,10.377,10.378,10.380,10.378,
     * 10.072,10.061,10.049,10.029,10.009,9.989,9.975,9.966,
     * 9.645,9.660,9.676,9.685,9.695,9.687,9.687,9.695,
     * 9.185,9.137,9.093,9.057,9.021,8.989,8.964,8.948,
     * 8.944,8.900,8.856,8.820,8.785,8.750,8.723,8.707,
     * 8.899,8.854,8.810,8.774,8.739,8.704,8.678,8.661/
      DATA ((EDEN(I,4,1,L,1),I=57,64),L=1,NZEN)/
     * 11.152,11.137,11.117,11.104,11.090,11.104,11.111,11.104,
     * 11.114,11.097,11.076,11.061,11.045,11.057,11.068,11.057,
     * 11.057,11.041,11.017,11.000,10.984,10.994,11.000,10.988,
     * 10.959,10.937,10.913,10.893,10.872,10.879,10.884,10.868,
     * 10.741,10.715,10.687,10.662,10.636,10.638,10.639,10.620,
     * 10.605,10.575,10.543,10.515,10.486,10.484,10.481,10.461,
     * 10.393,10.354,10.314,10.279,10.243,10.233,10.223,10.201,
     * 9.859,9.840,9.822,9.787,9.756,9.727,9.704,9.677,
     * 9.318,9.309,9.301,9.296,9.297,9.278,9.270,9.265,
     * 9.076,8.991,8.908,8.839,8.777,8.722,8.676,8.634,
     * 8.829,8.751,8.674,8.609,8.549,8.496,8.449,8.410,
     * 8.784,8.706,8.628,8.563,8.503,8.450,8.403,8.364/
      DATA ((EDEN(I,5,1,L,1),I=57,64),L=1,NZEN)/
     * 11.152,11.127,11.097,11.083,11.068,11.086,11.097,11.090,
     * 11.114,11.086,11.057,11.037,11.021,11.041,11.053,11.041,
     * 11.057,11.029,11.000,10.979,10.961,10.976,10.987,10.974,
     * 10.958,10.927,10.894,10.870,10.848,10.860,10.868,10.854,
     * 10.739,10.703,10.666,10.637,10.612,10.619,10.623,10.605,
     * 10.601,10.561,10.521,10.489,10.461,10.462,10.465,10.446,
     * 10.386,10.338,10.292,10.253,10.217,10.212,10.207,10.185,
     * 9.839,9.818,9.797,9.762,9.733,9.705,9.684,9.661,
     * 9.250,9.242,9.236,9.239,9.252,9.242,9.238,9.243,
     * 9.072,8.983,8.892,8.822,8.759,8.708,8.667,8.628,
     * 8.828,8.743,8.658,8.591,8.533,8.481,8.438,8.401,
     * 8.782,8.697,8.612,8.545,8.487,8.436,8.392,8.356/
      DATA ((EDEN(I,1,2,L,1),I=57,64),L=1,NZEN)/
     * 11.114,11.124,11.130,11.121,11.104,11.140,11.167,11.167,
     * 11.079,11.090,11.093,11.083,11.064,11.097,11.127,11.124,
     * 11.029,11.037,11.037,11.025,11.009,11.037,11.064,11.061,
     * 10.937,10.940,10.941,10.926,10.903,10.930,10.953,10.948,
     * 10.740,10.739,10.735,10.718,10.692,10.712,10.730,10.723,
     * 10.618,10.613,10.607,10.589,10.561,10.576,10.592,10.584,
     * 10.431,10.420,10.408,10.389,10.360,10.365,10.371,10.362,
     * 10.049,10.041,10.033,10.009,9.980,9.959,9.944,9.929,
     * 9.558,9.585,9.617,9.631,9.640,9.632,9.631,9.635,
     * 9.146,9.104,9.064,9.029,8.986,8.951,8.924,8.904,
     * 8.909,8.870,8.833,8.796,8.754,8.716,8.688,8.667,
     * 8.863,8.824,8.787,8.750,8.709,8.670,8.642,8.621/
      DATA ((EDEN(I,2,2,L,1),I=57,64),L=1,NZEN)/
     * 11.114,11.114,11.114,11.104,11.090,11.117,11.143,11.143,
     * 11.079,11.079,11.079,11.068,11.049,11.076,11.100,11.097,
     * 11.029,11.025,11.021,11.009,10.991,11.017,11.038,11.034,
     * 10.935,10.931,10.925,10.908,10.886,10.908,10.928,10.921,
     * 10.736,10.728,10.718,10.698,10.673,10.689,10.704,10.695,
     * 10.614,10.601,10.588,10.568,10.542,10.553,10.565,10.554,
     * 10.427,10.408,10.389,10.365,10.334,10.336,10.340,10.328,
     * 10.033,10.017,10.004,9.978,9.949,9.924,9.906,9.889,
     * 9.581,9.596,9.613,9.616,9.616,9.604,9.599,9.599,
     * 9.137,9.086,9.041,8.999,8.954,8.915,8.886,8.863,
     * 8.898,8.852,8.808,8.766,8.722,8.681,8.650,8.625,
     * 8.852,8.807,8.762,8.720,8.676,8.635,8.605,8.580/
      DATA ((EDEN(I,3,2,L,1),I=57,64),L=1,NZEN)/
     * 11.124,11.121,11.117,11.111,11.100,11.127,11.146,11.152,
     * 11.090,11.086,11.079,11.072,11.061,11.086,11.104,11.107,
     * 11.037,11.033,11.025,11.017,11.004,11.025,11.041,11.045,
     * 10.945,10.938,10.927,10.916,10.901,10.919,10.934,10.934,
     * 10.748,10.737,10.724,10.711,10.695,10.708,10.719,10.717,
     * 10.627,10.614,10.599,10.585,10.568,10.576,10.585,10.584,
     * 10.446,10.428,10.410,10.393,10.375,10.375,10.377,10.375,
     * 10.076,10.061,10.049,10.029,10.009,9.987,9.973,9.964,
     * 9.609,9.630,9.653,9.666,9.680,9.674,9.678,9.686,
     * 9.185,9.137,9.093,9.057,9.021,8.987,8.962,8.947,
     * 8.945,8.899,8.855,8.818,8.783,8.747,8.720,8.705,
     * 8.899,8.854,8.809,8.772,8.737,8.702,8.674,8.659/
      DATA ((EDEN(I,4,2,L,1),I=57,64),L=1,NZEN)/
     * 11.140,11.127,11.107,11.090,11.076,11.090,11.100,11.090,
     * 11.104,11.086,11.068,11.049,11.029,11.045,11.057,11.045,
     * 11.049,11.033,11.009,10.990,10.969,10.982,10.991,10.978,
     * 10.952,10.930,10.906,10.883,10.859,10.868,10.874,10.860,
     * 10.737,10.712,10.685,10.657,10.628,10.632,10.634,10.617,
     * 10.604,10.574,10.543,10.513,10.481,10.481,10.481,10.462,
     * 10.396,10.360,10.322,10.286,10.248,10.238,10.233,10.212,
     * 9.900,9.879,9.859,9.825,9.794,9.763,9.740,9.716,
     * 9.342,9.345,9.350,9.351,9.358,9.347,9.343,9.343,
     * 9.086,9.009,8.934,8.869,8.808,8.758,8.718,8.682,
     * 8.845,8.772,8.699,8.637,8.580,8.529,8.487,8.452,
     * 8.799,8.726,8.653,8.592,8.534,8.483,8.441,8.406/
      DATA ((EDEN(I,5,2,L,1),I=57,64),L=1,NZEN)/
     * 11.137,11.114,11.083,11.064,11.049,11.068,11.086,11.076,
     * 11.100,11.076,11.045,11.025,11.004,11.025,11.041,11.029,
     * 11.045,11.017,10.986,10.963,10.943,10.961,10.975,10.964,
     * 10.947,10.915,10.881,10.856,10.832,10.847,10.860,10.846,
     * 10.732,10.697,10.660,10.630,10.603,10.613,10.621,10.605,
     * 10.599,10.560,10.519,10.487,10.458,10.464,10.468,10.453,
     * 10.391,10.346,10.301,10.262,10.228,10.225,10.225,10.207,
     * 9.901,9.878,9.855,9.820,9.792,9.764,9.744,9.726,
     * 9.296,9.306,9.318,9.330,9.350,9.347,9.353,9.363,
     * 9.090,9.009,8.929,8.866,8.812,8.766,8.729,8.700,
     * 8.848,8.771,8.695,8.633,8.581,8.534,8.496,8.465,
     * 8.802,8.725,8.649,8.588,8.535,8.488,8.450,8.420/
      DATA ((EDEN(I,1,3,L,1),I=57,64),L=1,NZEN)/
     * 11.111,11.121,11.127,11.117,11.100,11.134,11.164,11.164,
     * 11.079,11.086,11.090,11.083,11.064,11.093,11.121,11.117,
     * 11.029,11.033,11.037,11.025,11.004,11.033,11.057,11.057,
     * 10.937,10.939,10.939,10.925,10.902,10.927,10.948,10.943,
     * 10.740,10.737,10.733,10.716,10.691,10.709,10.727,10.719,
     * 10.617,10.611,10.604,10.586,10.560,10.574,10.587,10.579,
     * 10.430,10.418,10.405,10.386,10.356,10.360,10.365,10.356,
     * 10.041,10.033,10.025,10.004,9.974,9.952,9.936,9.921,
     * 9.562,9.589,9.618,9.630,9.637,9.628,9.626,9.630,
     * 9.140,9.097,9.057,9.021,8.979,8.943,8.916,8.895,
     * 8.904,8.864,8.825,8.789,8.747,8.709,8.679,8.658,
     * 8.858,8.818,8.779,8.743,8.702,8.664,8.634,8.612/
      DATA ((EDEN(I,2,3,L,1),I=57,64),L=1,NZEN)/
     * 11.114,11.114,11.114,11.100,11.083,11.111,11.134,11.130,
     * 11.079,11.079,11.079,11.064,11.045,11.068,11.090,11.086,
     * 11.029,11.029,11.025,11.009,10.986,11.009,11.029,11.021,
     * 10.936,10.931,10.925,10.906,10.881,10.901,10.918,10.909,
     * 10.737,10.729,10.718,10.697,10.668,10.681,10.694,10.682,
     * 10.613,10.601,10.588,10.565,10.535,10.544,10.553,10.542,
     * 10.425,10.407,10.387,10.362,10.328,10.328,10.328,10.316,
     * 10.029,10.013,9.999,9.971,9.940,9.914,9.895,9.877,
     * 9.550,9.567,9.586,9.593,9.599,9.585,9.579,9.579,
     * 9.134,9.086,9.037,8.991,8.944,8.905,8.876,8.851,
     * 8.897,8.849,8.802,8.759,8.713,8.671,8.639,8.614,
     * 8.851,8.803,8.756,8.713,8.668,8.625,8.594,8.568/
      DATA ((EDEN(I,3,3,L,1),I=57,64),L=1,NZEN)/
     * 11.124,11.121,11.114,11.107,11.093,11.117,11.134,11.140,
     * 11.090,11.086,11.079,11.068,11.053,11.076,11.093,11.093,
     * 11.037,11.033,11.025,11.013,10.997,11.017,11.029,11.033,
     * 10.945,10.938,10.927,10.913,10.894,10.911,10.924,10.923,
     * 10.749,10.738,10.725,10.709,10.689,10.700,10.709,10.707,
     * 10.629,10.616,10.602,10.585,10.565,10.571,10.577,10.575,
     * 10.449,10.431,10.413,10.394,10.373,10.371,10.373,10.369,
     * 10.083,10.068,10.057,10.033,10.013,9.989,9.973,9.963,
     * 9.580,9.608,9.637,9.653,9.670,9.667,9.674,9.682,
     * 9.190,9.143,9.100,9.061,9.025,8.988,8.961,8.946,
     * 8.951,8.906,8.862,8.823,8.786,8.748,8.721,8.704,
     * 8.905,8.860,8.816,8.778,8.740,8.702,8.675,8.658/
      DATA ((EDEN(I,4,3,L,1),I=57,64),L=1,NZEN)/
     * 11.130,11.117,11.100,11.079,11.057,11.072,11.086,11.076,
     * 11.093,11.079,11.061,11.041,11.017,11.029,11.041,11.029,
     * 11.041,11.025,11.004,10.982,10.957,10.968,10.976,10.964,
     * 10.945,10.925,10.903,10.877,10.848,10.857,10.863,10.849,
     * 10.737,10.713,10.687,10.657,10.625,10.628,10.630,10.615,
     * 10.607,10.580,10.550,10.519,10.486,10.486,10.486,10.468,
     * 10.410,10.375,10.340,10.305,10.270,10.260,10.250,10.236,
     * 9.965,9.944,9.923,9.888,9.856,9.824,9.801,9.780,
     * 9.395,9.412,9.429,9.442,9.458,9.447,9.446,9.450,
     * 9.117,9.049,8.981,8.923,8.869,8.821,8.784,8.755,
     * 8.878,8.812,8.747,8.689,8.636,8.587,8.548,8.520,
     * 8.832,8.766,8.701,8.644,8.591,8.541,8.502,8.474/
      DATA ((EDEN(I,5,3,L,1),I=57,64),L=1,NZEN)/
     * 11.124,11.100,11.072,11.053,11.029,11.053,11.072,11.068,
     * 11.090,11.064,11.033,11.013,10.990,11.013,11.029,11.021,
     * 11.033,11.009,10.979,10.955,10.930,10.949,10.966,10.958,
     * 10.938,10.910,10.878,10.851,10.823,10.840,10.854,10.845,
     * 10.732,10.700,10.666,10.636,10.607,10.618,10.627,10.617,
     * 10.605,10.571,10.534,10.504,10.474,10.480,10.487,10.477,
     * 10.412,10.373,10.334,10.301,10.270,10.267,10.265,10.255,
     * 9.998,9.973,9.948,9.916,9.889,9.860,9.841,9.829,
     * 9.404,9.430,9.455,9.474,9.499,9.495,9.504,9.516,
     * 9.137,9.068,9.000,8.948,8.904,8.861,8.830,8.810,
     * 8.895,8.831,8.766,8.714,8.670,8.624,8.590,8.569,
     * 8.850,8.785,8.721,8.669,8.624,8.579,8.544,8.524/
      DATA ((EDEN(I,1,4,L,1),I=57,64),L=1,NZEN)/
     * 11.097,11.107,11.114,11.104,11.090,11.124,11.155,11.155,
     * 11.064,11.076,11.079,11.068,11.049,11.083,11.111,11.111,
     * 11.017,11.021,11.025,11.013,10.993,11.021,11.049,11.045,
     * 10.926,10.928,10.928,10.913,10.889,10.916,10.941,10.935,
     * 10.732,10.729,10.725,10.707,10.679,10.700,10.719,10.711,
     * 10.610,10.604,10.598,10.579,10.551,10.567,10.582,10.573,
     * 10.425,10.412,10.400,10.378,10.348,10.354,10.360,10.350,
     * 10.045,10.033,10.025,10.000,9.970,9.948,9.933,9.918,
     * 9.586,9.608,9.632,9.643,9.646,9.634,9.629,9.630,
     * 9.134,9.093,9.053,9.017,8.974,8.939,8.913,8.891,
     * 8.899,8.860,8.823,8.785,8.743,8.705,8.676,8.654,
     * 8.854,8.814,8.777,8.740,8.697,8.659,8.630,8.608/
      DATA ((EDEN(I,2,4,L,1),I=57,64),L=1,NZEN)/
     * 11.104,11.107,11.104,11.090,11.068,11.093,11.117,11.114,
     * 11.072,11.072,11.068,11.053,11.029,11.053,11.076,11.068,
     * 11.021,11.021,11.013,10.996,10.970,10.993,11.013,11.004,
     * 10.930,10.924,10.916,10.895,10.866,10.886,10.903,10.893,
     * 10.734,10.723,10.711,10.687,10.655,10.667,10.679,10.667,
     * 10.611,10.597,10.582,10.556,10.524,10.531,10.540,10.528,
     * 10.423,10.403,10.382,10.354,10.320,10.318,10.318,10.305,
     * 10.033,10.017,10.004,9.971,9.936,9.909,9.890,9.870,
     * 9.533,9.554,9.576,9.582,9.587,9.574,9.572,9.572,
     * 9.130,9.083,9.033,8.988,8.940,8.899,8.869,8.843,
     * 8.895,8.848,8.801,8.756,8.708,8.666,8.632,8.606,
     * 8.850,8.802,8.756,8.710,8.663,8.620,8.587,8.561/
      DATA ((EDEN(I,3,4,L,1),I=57,64),L=1,NZEN)/
     * 11.117,11.114,11.107,11.093,11.076,11.097,11.114,11.117,
     * 11.083,11.079,11.072,11.057,11.037,11.057,11.072,11.072,
     * 11.033,11.025,11.017,11.000,10.979,10.997,11.013,11.009,
     * 10.941,10.932,10.921,10.903,10.879,10.893,10.905,10.902,
     * 10.747,10.736,10.722,10.702,10.676,10.684,10.693,10.688,
     * 10.630,10.615,10.600,10.579,10.554,10.559,10.563,10.560,
     * 10.455,10.435,10.415,10.393,10.367,10.364,10.362,10.356,
     * 10.097,10.083,10.064,10.041,10.013,9.986,9.968,9.956,
     * 9.570,9.600,9.632,9.648,9.665,9.660,9.667,9.675,
     * 9.196,9.149,9.104,9.064,9.025,8.985,8.956,8.940,
     * 8.957,8.912,8.867,8.827,8.787,8.747,8.717,8.698,
     * 8.911,8.866,8.821,8.781,8.742,8.701,8.671,8.652/
      DATA ((EDEN(I,4,4,L,1),I=57,64),L=1,NZEN)/
     * 11.117,11.107,11.090,11.064,11.037,11.053,11.064,11.057,
     * 11.086,11.072,11.053,11.029,10.999,11.013,11.021,11.013,
     * 11.033,11.017,10.997,10.970,10.940,10.951,10.960,10.949,
     * 10.939,10.919,10.897,10.869,10.836,10.843,10.850,10.837,
     * 10.738,10.715,10.690,10.658,10.623,10.625,10.628,10.614,
     * 10.615,10.589,10.561,10.529,10.494,10.491,10.491,10.477,
     * 10.431,10.400,10.367,10.332,10.297,10.284,10.278,10.263,
     * 10.041,10.017,9.990,9.955,9.922,9.886,9.861,9.843,
     * 9.483,9.509,9.537,9.544,9.554,9.541,9.540,9.545,
     * 9.155,9.097,9.037,8.982,8.931,8.883,8.847,8.823,
     * 8.919,8.860,8.801,8.747,8.697,8.647,8.609,8.583,
     * 8.873,8.814,8.755,8.702,8.651,8.602,8.563,8.537/
      DATA ((EDEN(I,5,4,L,1),I=57,64),L=1,NZEN)/
     * 11.114,11.093,11.068,11.045,11.021,11.045,11.064,11.064,
     * 11.079,11.057,11.033,11.009,10.984,11.004,11.021,11.021,
     * 11.029,11.004,10.978,10.954,10.927,10.945,10.962,10.960,
     * 10.934,10.910,10.881,10.855,10.827,10.842,10.856,10.852,
     * 10.740,10.713,10.683,10.656,10.628,10.636,10.645,10.641,
     * 10.622,10.593,10.562,10.537,10.509,10.512,10.519,10.515,
     * 10.450,10.418,10.384,10.356,10.330,10.322,10.322,10.318,
     * 10.100,10.076,10.049,10.017,9.989,9.957,9.937,9.926,
     * 9.553,9.583,9.614,9.628,9.650,9.640,9.643,9.653,
     * 9.199,9.143,9.086,9.041,9.004,8.958,8.927,8.912,
     * 8.960,8.906,8.851,8.804,8.764,8.719,8.686,8.668,
     * 8.914,8.860,8.806,8.758,8.718,8.673,8.640,8.623/
      DATA ((EDEN(I,1,5,L,1),I=57,64),L=1,NZEN)/
     * 11.079,11.086,11.090,11.079,11.064,11.107,11.146,11.147,
     * 11.045,11.053,11.053,11.045,11.025,11.068,11.104,11.104,
     * 10.997,11.000,11.000,10.989,10.968,11.009,11.041,11.041,
     * 10.907,10.908,10.906,10.890,10.866,10.902,10.934,10.930,
     * 10.718,10.713,10.708,10.689,10.661,10.689,10.717,10.709,
     * 10.600,10.592,10.583,10.563,10.535,10.559,10.581,10.573,
     * 10.418,10.405,10.391,10.369,10.340,10.352,10.364,10.354,
     * 10.057,10.045,10.033,10.004,9.974,9.954,9.940,9.925,
     * 9.622,9.641,9.661,9.668,9.668,9.656,9.652,9.650,
     * 9.127,9.090,9.053,9.017,8.976,8.943,8.919,8.898,
     * 8.894,8.857,8.823,8.787,8.747,8.711,8.684,8.662,
     * 8.848,8.812,8.777,8.742,8.702,8.665,8.638,8.616/
      DATA ((EDEN(I,2,5,L,1),I=57,64),L=1,NZEN)/
     * 11.086,11.090,11.083,11.064,11.041,11.072,11.104,11.097,
     * 11.057,11.057,11.049,11.029,11.004,11.033,11.061,11.053,
     * 11.009,11.004,10.997,10.974,10.944,10.972,10.998,10.989,
     * 10.916,10.910,10.900,10.874,10.841,10.867,10.890,10.879,
     * 10.723,10.711,10.697,10.669,10.632,10.651,10.668,10.655,
     * 10.603,10.588,10.572,10.542,10.505,10.519,10.531,10.517,
     * 10.422,10.400,10.377,10.346,10.307,10.307,10.312,10.299,
     * 10.049,10.029,10.009,9.974,9.936,9.908,9.888,9.868,
     * 9.531,9.554,9.577,9.584,9.591,9.576,9.572,9.571,
     * 9.127,9.079,9.033,8.987,8.939,8.897,8.866,8.841,
     * 8.894,8.846,8.800,8.756,8.709,8.665,8.630,8.605,
     * 8.849,8.801,8.754,8.710,8.664,8.619,8.585,8.560/
      DATA ((EDEN(I,3,5,L,1),I=57,64),L=1,NZEN)/
     * 11.104,11.100,11.093,11.076,11.049,11.072,11.093,11.093,
     * 11.072,11.068,11.057,11.037,11.013,11.033,11.053,11.049,
     * 11.025,11.017,11.004,10.984,10.956,10.975,10.992,10.988,
     * 10.933,10.922,10.909,10.886,10.856,10.872,10.886,10.881,
     * 10.744,10.730,10.713,10.689,10.658,10.667,10.677,10.669,
     * 10.628,10.612,10.594,10.569,10.539,10.543,10.549,10.542,
     * 10.459,10.438,10.415,10.389,10.358,10.352,10.350,10.342,
     * 10.117,10.097,10.076,10.045,10.013,9.981,9.960,9.945,
     * 9.590,9.620,9.650,9.662,9.675,9.664,9.665,9.670,
     * 9.199,9.152,9.107,9.064,9.021,8.980,8.949,8.930,
     * 8.963,8.917,8.872,8.830,8.787,8.742,8.708,8.688,
     * 8.917,8.871,8.826,8.784,8.742,8.696,8.663,8.642/
      DATA ((EDEN(I,4,5,L,1),I=57,64),L=1,NZEN)/
     * 11.111,11.097,11.079,11.049,11.017,11.033,11.049,11.037,
     * 11.076,11.061,11.041,11.013,10.980,10.993,11.004,10.995,
     * 11.025,11.009,10.989,10.959,10.922,10.934,10.944,10.932,
     * 10.933,10.914,10.892,10.859,10.821,10.829,10.838,10.824,
     * 10.740,10.718,10.692,10.658,10.618,10.620,10.623,10.609,
     * 10.623,10.598,10.571,10.537,10.497,10.493,10.493,10.479,
     * 10.450,10.420,10.386,10.350,10.312,10.297,10.288,10.274,
     * 10.100,10.068,10.041,10.000,9.961,9.921,9.893,9.872,
     * 9.584,9.605,9.625,9.627,9.632,9.609,9.602,9.600,
     * 9.188,9.130,9.076,9.021,8.971,8.918,8.880,8.854,
     * 8.952,8.895,8.839,8.785,8.735,8.680,8.640,8.614,
     * 8.907,8.850,8.793,8.740,8.689,8.635,8.595,8.568/
      DATA ((EDEN(I,5,5,L,1),I=57,64),L=1,NZEN)/
     * 11.107,11.090,11.068,11.045,11.017,11.037,11.057,11.057,
     * 11.076,11.057,11.033,11.009,10.980,10.999,11.017,11.017,
     * 11.025,11.004,10.981,10.956,10.926,10.942,10.957,10.956,
     * 10.937,10.914,10.888,10.862,10.831,10.843,10.856,10.852,
     * 10.752,10.728,10.701,10.674,10.644,10.648,10.656,10.651,
     * 10.642,10.618,10.590,10.563,10.535,10.533,10.537,10.533,
     * 10.486,10.456,10.425,10.398,10.369,10.356,10.352,10.346,
     * 10.176,10.146,10.117,10.079,10.049,10.009,9.982,9.969,
     * 9.687,9.711,9.734,9.739,9.751,9.728,9.721,9.724,
     * 9.253,9.201,9.149,9.104,9.061,9.009,8.975,8.958,
     * 9.013,8.963,8.912,8.865,8.823,8.769,8.732,8.714,
     * 8.967,8.918,8.866,8.819,8.777,8.724,8.687,8.669/
      DATA ((EDEN(I,1,6,L,1),I=57,64),L=1,NZEN)/
     * 11.053,11.061,11.064,11.057,11.037,11.094,11.144,11.143,
     * 11.023,11.028,11.029,11.017,11.000,11.053,11.100,11.100,
     * 10.977,10.980,10.978,10.965,10.943,10.993,11.037,11.037,
     * 10.889,10.888,10.884,10.867,10.843,10.889,10.932,10.928,
     * 10.704,10.698,10.689,10.670,10.641,10.679,10.717,10.709,
     * 10.588,10.579,10.568,10.549,10.521,10.553,10.583,10.575,
     * 10.410,10.394,10.380,10.360,10.330,10.348,10.367,10.358,
     * 10.068,10.053,10.037,10.009,9.976,9.959,9.947,9.932,
     * 9.647,9.666,9.686,9.692,9.693,9.676,9.667,9.665,
     * 9.124,9.086,9.053,9.021,8.982,8.950,8.927,8.907,
     * 8.894,8.857,8.823,8.790,8.752,8.717,8.692,8.670,
     * 8.848,8.812,8.777,8.744,8.706,8.671,8.646,8.624/
      DATA ((EDEN(I,2,6,L,1),I=57,64),L=1,NZEN)/
     * 11.072,11.072,11.064,11.041,11.013,11.053,11.093,11.086,
     * 11.041,11.037,11.029,11.009,10.976,11.017,11.049,11.041,
     * 10.992,10.986,10.976,10.952,10.919,10.955,10.989,10.980,
     * 10.902,10.893,10.881,10.853,10.817,10.850,10.881,10.869,
     * 10.713,10.699,10.683,10.651,10.612,10.637,10.663,10.649,
     * 10.595,10.577,10.559,10.528,10.489,10.508,10.526,10.512,
     * 10.417,10.393,10.369,10.336,10.297,10.302,10.311,10.295,
     * 10.061,10.037,10.017,9.978,9.936,9.909,9.890,9.869,
     * 9.549,9.570,9.592,9.602,9.609,9.590,9.580,9.578,
     * 9.127,9.079,9.033,8.988,8.941,8.899,8.869,8.843,
     * 8.894,8.848,8.801,8.757,8.710,8.666,8.632,8.605,
     * 8.849,8.802,8.756,8.712,8.664,8.620,8.587,8.560/
      DATA ((EDEN(I,3,6,L,1),I=57,64),L=1,NZEN)/
     * 11.093,11.086,11.079,11.057,11.025,11.053,11.079,11.076,
     * 11.064,11.053,11.045,11.021,10.989,11.013,11.037,11.033,
     * 11.013,11.004,10.990,10.966,10.933,10.956,10.977,10.970,
     * 10.925,10.912,10.897,10.870,10.835,10.854,10.872,10.864,
     * 10.738,10.722,10.703,10.675,10.639,10.651,10.664,10.654,
     * 10.625,10.606,10.587,10.558,10.522,10.528,10.537,10.528,
     * 10.458,10.435,10.408,10.380,10.346,10.338,10.338,10.328,
     * 10.130,10.104,10.079,10.045,10.009,9.974,9.949,9.932,
     * 9.615,9.642,9.670,9.677,9.684,9.666,9.661,9.663,
     * 9.201,9.155,9.107,9.064,9.017,8.972,8.939,8.918,
     * 8.966,8.919,8.872,8.828,8.783,8.734,8.698,8.676,
     * 8.920,8.873,8.826,8.782,8.737,8.688,8.652,8.630/
      DATA ((EDEN(I,4,6,L,1),I=57,64),L=1,NZEN)/
     * 11.104,11.086,11.068,11.033,10.994,11.013,11.029,11.013,
     * 11.072,11.053,11.033,10.998,10.957,10.971,10.985,10.970,
     * 11.021,11.000,10.979,10.943,10.900,10.912,10.924,10.908,
     * 10.930,10.908,10.883,10.845,10.800,10.809,10.818,10.801,
     * 10.740,10.714,10.686,10.646,10.600,10.601,10.605,10.587,
     * 10.625,10.597,10.566,10.526,10.481,10.476,10.476,10.456,
     * 10.458,10.423,10.387,10.346,10.299,10.283,10.272,10.253,
     * 10.117,10.083,10.049,10.004,9.957,9.911,9.877,9.852,
     * 9.633,9.646,9.658,9.652,9.647,9.617,9.600,9.592,
     * 9.196,9.137,9.079,9.021,8.967,8.908,8.865,8.836,
     * 8.961,8.901,8.842,8.785,8.731,8.670,8.625,8.594,
     * 8.915,8.856,8.796,8.740,8.685,8.624,8.580,8.549/
      DATA ((EDEN(I,5,6,L,1),I=57,64),L=1,NZEN)/
     * 11.107,11.086,11.061,11.029,10.995,11.013,11.029,11.021,
     * 11.076,11.053,11.029,10.997,10.959,10.974,10.989,10.981,
     * 11.025,11.004,10.976,10.943,10.905,10.917,10.930,10.921,
     * 10.938,10.913,10.884,10.850,10.810,10.818,10.828,10.818,
     * 10.756,10.729,10.700,10.666,10.627,10.625,10.629,10.618,
     * 10.649,10.621,10.590,10.556,10.519,10.511,10.509,10.498,
     * 10.496,10.462,10.428,10.393,10.356,10.336,10.324,10.312,
     * 10.196,10.161,10.124,10.079,10.037,9.990,9.956,9.937,
     * 9.736,9.752,9.767,9.761,9.760,9.725,9.708,9.703,
     * 9.265,9.210,9.155,9.100,9.053,8.992,8.948,8.925,
     * 9.029,8.973,8.916,8.862,8.814,8.751,8.706,8.681,
     * 8.984,8.927,8.871,8.816,8.768,8.706,8.660,8.635/
      DATA ((EDEN(I,1,7,L,1),I=57,64),L=1,NZEN)/
     * 11.053,11.061,11.064,11.053,11.033,11.090,11.143,11.146,
     * 11.021,11.025,11.029,11.017,10.997,11.053,11.100,11.100,
     * 10.972,10.976,10.975,10.962,10.941,10.992,11.041,11.037,
     * 10.885,10.885,10.881,10.865,10.840,10.889,10.934,10.930,
     * 10.700,10.695,10.688,10.668,10.640,10.680,10.719,10.712,
     * 10.585,10.576,10.567,10.548,10.520,10.553,10.585,10.577,
     * 10.408,10.394,10.380,10.360,10.330,10.350,10.371,10.362,
     * 10.072,10.053,10.037,10.013,9.980,9.962,9.950,9.936,
     * 9.656,9.673,9.692,9.698,9.700,9.681,9.671,9.671,
     * 9.124,9.090,9.057,9.021,8.985,8.953,8.930,8.911,
     * 8.894,8.860,8.827,8.794,8.757,8.720,8.694,8.673,
     * 8.848,8.814,8.782,8.749,8.711,8.674,8.648,8.627/
      DATA ((EDEN(I,2,7,L,1),I=57,64),L=1,NZEN)/
     * 11.068,11.064,11.061,11.037,11.009,11.053,11.093,11.086,
     * 11.037,11.033,11.025,11.004,10.972,11.013,11.049,11.041,
     * 10.988,10.982,10.974,10.948,10.914,10.953,10.989,10.980,
     * 10.899,10.890,10.878,10.849,10.814,10.848,10.881,10.870,
     * 10.710,10.696,10.680,10.648,10.609,10.636,10.664,10.649,
     * 10.593,10.575,10.556,10.525,10.486,10.507,10.528,10.513,
     * 10.417,10.393,10.367,10.334,10.294,10.301,10.312,10.297,
     * 10.064,10.041,10.017,9.980,9.938,9.910,9.890,9.870,
     * 9.564,9.585,9.607,9.612,9.613,9.594,9.586,9.585,
     * 9.127,9.079,9.033,8.989,8.943,8.901,8.871,8.845,
     * 8.895,8.849,8.804,8.760,8.714,8.669,8.634,8.609,
     * 8.850,8.804,8.758,8.715,8.669,8.624,8.589,8.563/
      DATA ((EDEN(I,3,7,L,1),I=57,64),L=1,NZEN)/
     * 11.093,11.086,11.076,11.049,11.021,11.049,11.076,11.068,
     * 11.061,11.053,11.041,11.013,10.982,11.009,11.033,11.025,
     * 11.013,11.000,10.987,10.961,10.926,10.950,10.973,10.964,
     * 10.923,10.910,10.893,10.865,10.829,10.848,10.867,10.858,
     * 10.737,10.720,10.700,10.670,10.633,10.644,10.658,10.647,
     * 10.624,10.604,10.582,10.553,10.517,10.522,10.530,10.520,
     * 10.458,10.433,10.405,10.375,10.338,10.332,10.330,10.318,
     * 10.127,10.100,10.076,10.041,10.000,9.966,9.941,9.922,
     * 9.628,9.651,9.673,9.678,9.681,9.661,9.655,9.654,
     * 9.199,9.152,9.104,9.057,9.013,8.963,8.928,8.906,
     * 8.963,8.916,8.869,8.823,8.777,8.726,8.689,8.665,
     * 8.918,8.870,8.823,8.777,8.731,8.680,8.644,8.619/
      DATA ((EDEN(I,4,7,L,1),I=57,64),L=1,NZEN)/
     * 11.107,11.086,11.064,11.025,10.981,10.998,11.013,10.996,
     * 11.072,11.053,11.029,10.989,10.944,10.958,10.972,10.952,
     * 11.021,11.000,10.974,10.933,10.886,10.898,10.910,10.889,
     * 10.931,10.905,10.877,10.834,10.785,10.793,10.802,10.780,
     * 10.739,10.709,10.678,10.633,10.582,10.582,10.585,10.562,
     * 10.622,10.590,10.555,10.511,10.461,10.455,10.453,10.430,
     * 10.450,10.412,10.371,10.324,10.274,10.255,10.246,10.220,
     * 10.100,10.061,10.025,9.974,9.923,9.874,9.838,9.810,
     * 9.624,9.630,9.634,9.623,9.613,9.577,9.556,9.543,
     * 9.182,9.117,9.053,8.992,8.932,8.870,8.823,8.790,
     * 8.947,8.883,8.819,8.758,8.700,8.634,8.584,8.551,
     * 8.901,8.837,8.773,8.712,8.654,8.589,8.539,8.506/
      DATA ((EDEN(I,5,7,L,1),I=57,64),L=1,NZEN)/
     * 11.111,11.083,11.053,11.017,10.972,10.987,11.000,10.988,
     * 11.076,11.049,11.021,10.981,10.936,10.948,10.961,10.945,
     * 11.025,10.999,10.967,10.927,10.880,10.890,10.902,10.885,
     * 10.938,10.907,10.873,10.831,10.784,10.789,10.797,10.780,
     * 10.753,10.719,10.683,10.640,10.593,10.590,10.591,10.574,
     * 10.642,10.607,10.569,10.528,10.483,10.471,10.468,10.450,
     * 10.481,10.442,10.401,10.358,10.314,10.290,10.274,10.258,
     * 10.170,10.127,10.086,10.037,9.989,9.934,9.895,9.870,
     * 9.710,9.718,9.724,9.713,9.707,9.665,9.642,9.632,
     * 9.241,9.179,9.117,9.057,9.000,8.935,8.887,8.858,
     * 9.004,8.943,8.880,8.818,8.763,8.695,8.643,8.614,
     * 8.959,8.897,8.834,8.772,8.718,8.649,8.598,8.568/
      DATA ((EDEN(I,1,8,L,1),I=57,64),L=1,NZEN)/
     * 11.068,11.079,11.083,11.076,11.057,11.107,11.149,11.152,
     * 11.033,11.045,11.049,11.037,11.021,11.064,11.107,11.107,
     * 10.987,10.993,10.995,10.983,10.963,11.004,11.045,11.045,
     * 10.898,10.900,10.900,10.885,10.862,10.901,10.939,10.934,
     * 10.711,10.708,10.702,10.685,10.658,10.690,10.722,10.714,
     * 10.592,10.587,10.580,10.561,10.533,10.560,10.585,10.579,
     * 10.415,10.403,10.391,10.369,10.340,10.354,10.369,10.362,
     * 10.064,10.053,10.037,10.013,9.980,9.961,9.949,9.933,
     * 9.647,9.663,9.682,9.689,9.691,9.677,9.672,9.669,
     * 9.130,9.093,9.057,9.025,8.986,8.953,8.929,8.908,
     * 8.897,8.861,8.827,8.793,8.754,8.719,8.692,8.671,
     * 8.851,8.815,8.782,8.747,8.709,8.673,8.646,8.625/
      DATA ((EDEN(I,2,8,L,1),I=57,64),L=1,NZEN)/
     * 11.083,11.079,11.076,11.057,11.029,11.068,11.104,11.097,
     * 11.049,11.045,11.041,11.021,10.994,11.029,11.061,11.053,
     * 11.000,10.996,10.987,10.966,10.936,10.968,10.998,10.989,
     * 10.911,10.902,10.891,10.866,10.834,10.863,10.889,10.878,
     * 10.719,10.706,10.691,10.664,10.627,10.649,10.669,10.656,
     * 10.600,10.584,10.566,10.538,10.501,10.517,10.533,10.519,
     * 10.420,10.398,10.375,10.344,10.307,10.310,10.316,10.301,
     * 10.061,10.037,10.017,9.982,9.942,9.913,9.893,9.872,
     * 9.573,9.591,9.610,9.615,9.618,9.600,9.593,9.590,
     * 9.134,9.086,9.037,8.994,8.946,8.904,8.873,8.848,
     * 8.899,8.853,8.809,8.764,8.718,8.671,8.636,8.611,
     * 8.854,8.807,8.763,8.718,8.672,8.625,8.591,8.565/
      DATA ((EDEN(I,3,8,L,1),I=57,64),L=1,NZEN)/
     * 11.104,11.097,11.086,11.064,11.037,11.061,11.083,11.079,
     * 11.072,11.064,11.053,11.029,11.000,11.021,11.041,11.037,
     * 11.025,11.013,10.998,10.974,10.943,10.963,10.982,10.975,
     * 10.932,10.919,10.903,10.877,10.844,10.860,10.876,10.867,
     * 10.744,10.727,10.708,10.680,10.645,10.655,10.666,10.654,
     * 10.628,10.609,10.588,10.561,10.526,10.530,10.537,10.526,
     * 10.459,10.435,10.408,10.380,10.346,10.338,10.334,10.324,
     * 10.117,10.097,10.072,10.037,10.004,9.968,9.942,9.924,
     * 9.623,9.645,9.668,9.675,9.681,9.662,9.656,9.655,
     * 9.199,9.149,9.104,9.057,9.013,8.964,8.930,8.908,
     * 8.963,8.915,8.867,8.822,8.776,8.727,8.690,8.667,
     * 8.918,8.870,8.822,8.776,8.730,8.681,8.644,8.621/
      DATA ((EDEN(I,4,8,L,1),I=57,64),L=1,NZEN)/
     * 11.117,11.097,11.072,11.033,10.992,11.004,11.017,10.997,
     * 11.083,11.061,11.033,10.998,10.953,10.964,10.975,10.953,
     * 11.033,11.009,10.980,10.941,10.894,10.903,10.912,10.889,
     * 10.939,10.911,10.881,10.838,10.790,10.795,10.801,10.777,
     * 10.740,10.708,10.674,10.629,10.579,10.576,10.577,10.551,
     * 10.618,10.583,10.547,10.501,10.449,10.441,10.438,10.412,
     * 10.438,10.396,10.354,10.303,10.250,10.230,10.217,10.190,
     * 10.053,10.017,9.980,9.929,9.879,9.829,9.792,9.763,
     * 9.548,9.556,9.565,9.552,9.542,9.507,9.486,9.475,
     * 9.152,9.083,9.017,8.949,8.886,8.823,8.775,8.739,
     * 8.918,8.850,8.782,8.716,8.653,8.589,8.539,8.502,
     * 8.872,8.804,8.736,8.670,8.607,8.542,8.492,8.455/
      DATA ((EDEN(I,5,8,L,1),I=57,64),L=1,NZEN)/
     * 11.121,11.090,11.053,11.013,10.968,10.981,10.995,10.975,
     * 11.086,11.053,11.017,10.977,10.930,10.941,10.952,10.931,
     * 11.033,11.000,10.963,10.921,10.872,10.881,10.891,10.869,
     * 10.942,10.906,10.866,10.822,10.772,10.776,10.783,10.760,
     * 10.747,10.708,10.666,10.620,10.571,10.566,10.567,10.544,
     * 10.629,10.589,10.544,10.498,10.449,10.438,10.435,10.412,
     * 10.456,10.412,10.362,10.316,10.267,10.243,10.228,10.204,
     * 10.104,10.064,10.021,9.969,9.920,9.867,9.828,9.801,
     * 9.609,9.619,9.627,9.615,9.609,9.571,9.549,9.540,
     * 9.193,9.124,9.053,8.990,8.931,8.865,8.816,8.785,
     * 8.958,8.889,8.820,8.755,8.697,8.627,8.576,8.543,
     * 8.912,8.844,8.774,8.709,8.651,8.582,8.531,8.497/
      DATA ((EDEN(I,1,9,L,1),I=57,64),L=1,NZEN)/
     * 11.093,11.104,11.114,11.107,11.090,11.127,11.158,11.158,
     * 11.061,11.072,11.076,11.068,11.053,11.086,11.114,11.114,
     * 11.013,11.017,11.021,11.013,10.995,11.025,11.053,11.049,
     * 10.921,10.924,10.925,10.913,10.892,10.920,10.944,10.940,
     * 10.727,10.726,10.723,10.708,10.683,10.704,10.723,10.716,
     * 10.606,10.602,10.595,10.580,10.555,10.571,10.587,10.579,
     * 10.425,10.413,10.401,10.382,10.356,10.362,10.367,10.360,
     * 10.053,10.045,10.033,10.013,9.982,9.960,9.946,9.931,
     * 9.632,9.651,9.672,9.677,9.676,9.663,9.658,9.658,
     * 9.140,9.100,9.064,9.029,8.987,8.952,8.927,8.905,
     * 8.904,8.867,8.833,8.796,8.754,8.718,8.689,8.668,
     * 8.858,8.822,8.787,8.750,8.709,8.672,8.644,8.623/
      DATA ((EDEN(I,2,9,L,1),I=57,64),L=1,NZEN)/
     * 11.100,11.104,11.104,11.090,11.068,11.093,11.117,11.111,
     * 11.068,11.068,11.064,11.053,11.029,11.053,11.072,11.068,
     * 11.017,11.017,11.013,10.995,10.970,10.992,11.013,11.004,
     * 10.926,10.922,10.914,10.894,10.866,10.885,10.901,10.892,
     * 10.731,10.721,10.710,10.687,10.656,10.668,10.679,10.667,
     * 10.609,10.597,10.582,10.558,10.526,10.533,10.540,10.528,
     * 10.427,10.407,10.387,10.360,10.326,10.322,10.322,10.307,
     * 10.049,10.029,10.013,9.984,9.949,9.920,9.898,9.878,
     * 9.594,9.609,9.624,9.628,9.630,9.610,9.600,9.597,
     * 9.140,9.093,9.045,9.000,8.953,8.911,8.880,8.854,
     * 8.906,8.859,8.812,8.769,8.723,8.678,8.642,8.616,
     * 8.861,8.813,8.766,8.724,8.678,8.632,8.597,8.570/
      DATA ((EDEN(I,3,9,L,1),I=57,64),L=1,NZEN)/
     * 11.121,11.117,11.107,11.093,11.072,11.090,11.104,11.104,
     * 11.086,11.083,11.072,11.057,11.033,11.049,11.064,11.061,
     * 11.037,11.029,11.017,11.000,10.976,10.990,11.000,10.997,
     * 10.945,10.934,10.921,10.900,10.874,10.885,10.895,10.888,
     * 10.751,10.736,10.721,10.699,10.671,10.676,10.681,10.674,
     * 10.633,10.617,10.600,10.577,10.549,10.549,10.551,10.544,
     * 10.459,10.438,10.415,10.391,10.362,10.354,10.350,10.342,
     * 10.107,10.090,10.068,10.041,10.009,9.978,9.955,9.940,
     * 9.629,9.651,9.674,9.680,9.686,9.671,9.667,9.669,
     * 9.201,9.155,9.107,9.064,9.021,8.977,8.944,8.924,
     * 8.963,8.916,8.869,8.827,8.785,8.738,8.703,8.682,
     * 8.918,8.870,8.823,8.781,8.739,8.692,8.658,8.636/
      DATA ((EDEN(I,4,9,L,1),I=57,64),L=1,NZEN)/
     * 11.137,11.114,11.090,11.061,11.025,11.033,11.037,11.017,
     * 11.100,11.079,11.053,11.021,10.984,10.989,10.993,10.972,
     * 11.045,11.025,10.996,10.963,10.924,10.927,10.929,10.906,
     * 10.950,10.923,10.894,10.857,10.816,10.816,10.815,10.791,
     * 10.744,10.713,10.679,10.639,10.594,10.588,10.583,10.556,
     * 10.617,10.581,10.544,10.502,10.456,10.446,10.438,10.412,
     * 10.423,10.382,10.340,10.292,10.241,10.220,10.204,10.176,
     * 9.990,9.960,9.928,9.881,9.835,9.789,9.754,9.724,
     * 9.463,9.469,9.472,9.464,9.458,9.428,9.412,9.405,
     * 9.127,9.053,8.978,8.910,8.844,8.782,8.734,8.696,
     * 8.889,8.817,8.744,8.678,8.614,8.550,8.500,8.462,
     * 8.844,8.771,8.699,8.632,8.568,8.504,8.454,8.417/
      DATA ((EDEN(I,5,9,L,1),I=57,64),L=1,NZEN)/
     * 11.134,11.100,11.064,11.033,10.993,11.004,11.013,10.991,
     * 11.097,11.064,11.029,10.992,10.952,10.960,10.967,10.946,
     * 11.045,11.009,10.972,10.934,10.892,10.898,10.904,10.881,
     * 10.949,10.912,10.871,10.831,10.787,10.789,10.792,10.768,
     * 10.744,10.703,10.659,10.616,10.571,10.566,10.563,10.539,
     * 10.620,10.576,10.529,10.484,10.438,10.428,10.423,10.400,
     * 10.431,10.382,10.332,10.286,10.238,10.215,10.201,10.179,
     * 10.025,9.990,9.951,9.903,9.859,9.810,9.775,9.748,
     * 9.488,9.498,9.509,9.503,9.502,9.471,9.456,9.452,
     * 9.152,9.076,8.998,8.931,8.870,8.808,8.761,8.728,
     * 8.913,8.838,8.763,8.698,8.639,8.574,8.524,8.490,
     * 8.867,8.792,8.718,8.652,8.594,8.528,8.478,8.444/
      DATA ((EDEN(I,1,10,L,1),I=57,64),L=1,NZEN)/
     * 11.114,11.127,11.134,11.130,11.117,11.143,11.167,11.167,
     * 11.079,11.090,11.097,11.090,11.076,11.100,11.124,11.121,
     * 11.029,11.037,11.041,11.033,11.017,11.041,11.061,11.057,
     * 10.936,10.941,10.943,10.932,10.913,10.933,10.949,10.944,
     * 10.739,10.739,10.737,10.723,10.701,10.713,10.725,10.719,
     * 10.617,10.613,10.609,10.593,10.569,10.579,10.588,10.580,
     * 10.431,10.423,10.413,10.394,10.367,10.367,10.369,10.360,
     * 10.049,10.040,10.031,10.012,9.985,9.962,9.946,9.931,
     * 9.623,9.641,9.662,9.669,9.671,9.657,9.652,9.652,
     * 9.149,9.107,9.072,9.033,8.991,8.955,8.928,8.907,
     * 8.912,8.873,8.837,8.800,8.759,8.720,8.692,8.669,
     * 8.866,8.828,8.791,8.754,8.713,8.674,8.646,8.624/
      DATA ((EDEN(I,2,10,L,1),I=57,64),L=1,NZEN)/
     * 11.117,11.121,11.121,11.111,11.097,11.117,11.134,11.127,
     * 11.083,11.086,11.083,11.076,11.057,11.076,11.090,11.083,
     * 11.033,11.033,11.029,11.017,10.997,11.013,11.025,11.017,
     * 10.939,10.936,10.930,10.915,10.893,10.905,10.915,10.906,
     * 10.740,10.732,10.723,10.704,10.679,10.685,10.691,10.679,
     * 10.616,10.605,10.593,10.574,10.548,10.550,10.551,10.540,
     * 10.433,10.415,10.396,10.373,10.342,10.336,10.332,10.320,
     * 10.041,10.029,10.013,9.988,9.957,9.928,9.907,9.889,
     * 9.623,9.635,9.647,9.645,9.640,9.623,9.615,9.612,
     * 9.149,9.100,9.057,9.013,8.965,8.922,8.889,8.865,
     * 8.912,8.866,8.820,8.777,8.732,8.688,8.653,8.626,
     * 8.866,8.820,8.774,8.732,8.687,8.642,8.607,8.581/
      DATA ((EDEN(I,3,10,L,1),I=57,64),L=1,NZEN)/
     * 11.134,11.130,11.124,11.114,11.100,11.117,11.127,11.127,
     * 11.100,11.093,11.086,11.076,11.061,11.076,11.083,11.083,
     * 11.045,11.041,11.033,11.021,11.004,11.013,11.021,11.021,
     * 10.953,10.945,10.934,10.920,10.901,10.909,10.915,10.911,
     * 10.757,10.744,10.731,10.715,10.695,10.697,10.700,10.695,
     * 10.636,10.622,10.606,10.591,10.571,10.569,10.569,10.565,
     * 10.459,10.441,10.422,10.401,10.378,10.371,10.365,10.360,
     * 10.097,10.079,10.064,10.041,10.017,9.989,9.970,9.958,
     * 9.649,9.666,9.683,9.690,9.699,9.686,9.684,9.689,
     * 9.204,9.158,9.111,9.072,9.029,8.989,8.960,8.941,
     * 8.964,8.918,8.872,8.832,8.792,8.750,8.719,8.699,
     * 8.918,8.872,8.826,8.786,8.747,8.704,8.673,8.653/
      DATA ((EDEN(I,4,10,L,1),I=57,64),L=1,NZEN)/
     * 11.152,11.134,11.111,11.090,11.061,11.064,11.064,11.045,
     * 11.114,11.097,11.072,11.049,11.021,11.021,11.017,11.000,
     * 11.061,11.041,11.017,10.989,10.958,10.957,10.953,10.933,
     * 10.960,10.937,10.910,10.880,10.847,10.843,10.838,10.815,
     * 10.747,10.719,10.687,10.654,10.617,10.607,10.597,10.572,
     * 10.616,10.582,10.548,10.511,10.471,10.458,10.446,10.420,
     * 10.413,10.373,10.330,10.288,10.243,10.220,10.199,10.173,
     * 9.928,9.902,9.876,9.836,9.797,9.754,9.721,9.691,
     * 9.395,9.394,9.392,9.385,9.382,9.357,9.343,9.335,
     * 9.107,9.029,8.948,8.878,8.812,8.750,8.699,8.658,
     * 8.866,8.790,8.715,8.646,8.582,8.519,8.468,8.428,
     * 8.821,8.745,8.669,8.601,8.536,8.473,8.423,8.382/
      DATA ((EDEN(I,5,10,L,1),I=57,64),L=1,NZEN)/
     * 11.149,11.121,11.086,11.061,11.029,11.037,11.037,11.021,
     * 11.111,11.079,11.045,11.017,10.987,10.992,10.994,10.975,
     * 11.057,11.025,10.987,10.957,10.926,10.928,10.929,10.908,
     * 10.958,10.922,10.883,10.850,10.816,10.815,10.813,10.791,
     * 10.745,10.704,10.662,10.624,10.587,10.581,10.576,10.551,
     * 10.614,10.571,10.524,10.484,10.444,10.435,10.427,10.401,
     * 10.412,10.364,10.312,10.267,10.223,10.204,10.188,10.165,
     * 9.943,9.912,9.879,9.837,9.799,9.757,9.723,9.699,
     * 9.374,9.378,9.382,9.383,9.390,9.369,9.361,9.361,
     * 9.117,9.033,8.950,8.881,8.820,8.758,8.710,8.675,
     * 8.875,8.795,8.715,8.648,8.589,8.526,8.477,8.441,
     * 8.829,8.749,8.669,8.603,8.543,8.481,8.431,8.395/
      DATA ((EDEN(I,1,11,L,1),I=57,64),L=1,NZEN)/
     * 11.121,11.130,11.140,11.134,11.121,11.146,11.170,11.170,
     * 11.086,11.097,11.100,11.097,11.083,11.107,11.127,11.127,
     * 11.033,11.041,11.045,11.037,11.021,11.045,11.064,11.061,
     * 10.941,10.945,10.947,10.937,10.918,10.937,10.953,10.948,
     * 10.742,10.742,10.740,10.727,10.704,10.717,10.728,10.722,
     * 10.621,10.617,10.611,10.597,10.574,10.582,10.591,10.583,
     * 10.438,10.427,10.415,10.398,10.371,10.371,10.371,10.364,
     * 10.053,10.045,10.037,10.017,9.991,9.968,9.950,9.936,
     * 9.607,9.628,9.651,9.660,9.664,9.653,9.651,9.652,
     * 9.152,9.114,9.076,9.037,8.997,8.960,8.932,8.912,
     * 8.917,8.877,8.839,8.803,8.763,8.725,8.695,8.674,
     * 8.871,8.832,8.794,8.758,8.717,8.679,8.650,8.628/
      DATA ((EDEN(I,2,11,L,1),I=57,64),L=1,NZEN)/
     * 11.117,11.124,11.124,11.117,11.104,11.127,11.143,11.140,
     * 11.086,11.090,11.086,11.079,11.064,11.083,11.100,11.097,
     * 11.033,11.033,11.033,11.021,11.004,11.021,11.037,11.029,
     * 10.941,10.938,10.931,10.920,10.901,10.915,10.926,10.918,
     * 10.741,10.734,10.724,10.708,10.687,10.695,10.702,10.692,
     * 10.619,10.607,10.595,10.577,10.554,10.558,10.562,10.551,
     * 10.435,10.415,10.396,10.377,10.350,10.344,10.342,10.330,
     * 10.041,10.029,10.017,9.990,9.960,9.934,9.916,9.899,
     * 9.645,9.653,9.662,9.659,9.654,9.637,9.629,9.622,
     * 9.152,9.104,9.057,9.013,8.971,8.929,8.897,8.874,
     * 8.913,8.866,8.822,8.780,8.736,8.694,8.661,8.636,
     * 8.868,8.821,8.776,8.734,8.691,8.648,8.615,8.591/
      DATA ((EDEN(I,3,11,L,1),I=57,64),L=1,NZEN)/
     * 11.137,11.134,11.127,11.124,11.114,11.130,11.143,11.146,
     * 11.100,11.097,11.090,11.083,11.076,11.090,11.100,11.104,
     * 11.049,11.041,11.033,11.029,11.017,11.029,11.037,11.037,
     * 10.954,10.946,10.936,10.927,10.913,10.923,10.930,10.929,
     * 10.755,10.744,10.732,10.720,10.705,10.710,10.714,10.712,
     * 10.634,10.621,10.606,10.594,10.579,10.581,10.583,10.580,
     * 10.456,10.438,10.420,10.403,10.384,10.378,10.377,10.373,
     * 10.083,10.072,10.057,10.037,10.017,9.995,9.978,9.968,
     * 9.671,9.684,9.698,9.704,9.709,9.699,9.698,9.701,
     * 9.199,9.152,9.107,9.068,9.033,8.996,8.968,8.951,
     * 8.958,8.913,8.869,8.831,8.794,8.755,8.727,8.709,
     * 8.912,8.868,8.823,8.785,8.749,8.709,8.681,8.664/
      DATA ((EDEN(I,4,11,L,1),I=57,64),L=1,NZEN)/
     * 11.158,11.143,11.124,11.107,11.090,11.093,11.090,11.076,
     * 11.121,11.104,11.083,11.064,11.045,11.049,11.045,11.029,
     * 11.068,11.045,11.025,11.004,10.984,10.983,10.978,10.961,
     * 10.966,10.943,10.918,10.896,10.872,10.867,10.861,10.841,
     * 10.748,10.720,10.690,10.665,10.636,10.626,10.616,10.594,
     * 10.614,10.581,10.548,10.519,10.487,10.473,10.459,10.436,
     * 10.403,10.364,10.322,10.286,10.248,10.225,10.207,10.179,
     * 9.872,9.852,9.831,9.797,9.766,9.728,9.699,9.669,
     * 9.352,9.343,9.335,9.325,9.318,9.296,9.283,9.275,
     * 9.090,9.009,8.924,8.854,8.787,8.725,8.673,8.629,
     * 8.846,8.768,8.690,8.622,8.560,8.497,8.444,8.403,
     * 8.800,8.722,8.644,8.576,8.514,8.451,8.398,8.357/
      DATA ((EDEN(I,5,11,L,1),I=57,64),L=1,NZEN)/
     * 11.158,11.130,11.100,11.083,11.064,11.068,11.072,11.057,
     * 11.121,11.093,11.061,11.041,11.017,11.025,11.025,11.009,
     * 11.068,11.037,11.004,10.980,10.957,10.959,10.958,10.941,
     * 10.964,10.931,10.896,10.870,10.844,10.843,10.841,10.821,
     * 10.745,10.708,10.668,10.637,10.607,10.601,10.594,10.573,
     * 10.610,10.568,10.525,10.490,10.458,10.449,10.439,10.417,
     * 10.398,10.348,10.299,10.260,10.220,10.201,10.188,10.161,
     * 9.866,9.841,9.816,9.779,9.747,9.711,9.684,9.658,
     * 9.294,9.288,9.282,9.281,9.287,9.271,9.268,9.268,
     * 9.093,9.004,8.912,8.840,8.776,8.717,8.668,8.627,
     * 8.848,8.763,8.679,8.609,8.547,8.487,8.438,8.398,
     * 8.802,8.717,8.633,8.563,8.501,8.441,8.392,8.352/
      DATA ((EDEN(I,1,12,L,1),I=57,64),L=1,NZEN)/
     * 11.117,11.127,11.134,11.127,11.114,11.146,11.173,11.173,
     * 11.083,11.093,11.097,11.090,11.076,11.104,11.130,11.127,
     * 11.033,11.037,11.041,11.033,11.017,11.041,11.064,11.064,
     * 10.940,10.942,10.942,10.930,10.911,10.934,10.955,10.951,
     * 10.741,10.740,10.736,10.722,10.699,10.716,10.732,10.725,
     * 10.620,10.615,10.609,10.593,10.568,10.582,10.594,10.588,
     * 10.439,10.427,10.415,10.396,10.369,10.371,10.377,10.367,
     * 10.057,10.049,10.041,10.021,9.993,9.971,9.955,9.941,
     * 9.586,9.610,9.637,9.648,9.654,9.646,9.647,9.650,
     * 9.152,9.114,9.076,9.037,8.998,8.962,8.936,8.916,
     * 8.917,8.877,8.839,8.805,8.765,8.728,8.700,8.679,
     * 8.871,8.832,8.794,8.759,8.719,8.682,8.654,8.633/
      DATA ((EDEN(I,2,12,L,1),I=57,64),L=1,NZEN)/
     * 11.112,11.116,11.117,11.111,11.100,11.127,11.149,11.146,
     * 11.083,11.083,11.083,11.076,11.061,11.086,11.104,11.104,
     * 11.029,11.029,11.025,11.017,11.000,11.025,11.041,11.037,
     * 10.937,10.933,10.927,10.915,10.897,10.916,10.932,10.926,
     * 10.739,10.731,10.721,10.705,10.683,10.696,10.708,10.699,
     * 10.615,10.604,10.592,10.575,10.551,10.560,10.568,10.559,
     * 10.431,10.413,10.394,10.373,10.346,10.346,10.348,10.336,
     * 10.041,10.029,10.017,9.991,9.962,9.937,9.919,9.903,
     * 9.643,9.651,9.661,9.661,9.658,9.640,9.630,9.625,
     * 9.146,9.100,9.053,9.013,8.969,8.931,8.901,8.878,
     * 8.908,8.862,8.818,8.777,8.736,8.695,8.664,8.640,
     * 8.862,8.816,8.772,8.732,8.690,8.649,8.618,8.595/
      DATA ((EDEN(I,3,12,L,1),I=57,64),L=1,NZEN)/
     * 11.130,11.127,11.124,11.121,11.114,11.134,11.153,11.156,
     * 11.097,11.093,11.086,11.079,11.072,11.093,11.107,11.111,
     * 11.045,11.037,11.029,11.025,11.017,11.033,11.045,11.049,
     * 10.950,10.942,10.931,10.924,10.912,10.927,10.939,10.939,
     * 10.751,10.740,10.728,10.718,10.704,10.713,10.721,10.720,
     * 10.630,10.617,10.603,10.591,10.577,10.583,10.589,10.587,
     * 10.450,10.433,10.415,10.400,10.382,10.380,10.380,10.378,
     * 10.076,10.064,10.053,10.033,10.017,9.994,9.979,9.969,
     * 9.670,9.683,9.699,9.704,9.711,9.700,9.700,9.703,
     * 9.190,9.143,9.100,9.064,9.029,8.993,8.967,8.952,
     * 8.950,8.905,8.861,8.825,8.792,8.754,8.726,8.710,
     * 8.905,8.859,8.815,8.780,8.746,8.708,8.680,8.664/
      DATA ((EDEN(I,4,12,L,1),I=57,64),L=1,NZEN)/
     * 11.158,11.143,11.124,11.111,11.097,11.104,11.107,11.097,
     * 11.121,11.104,11.083,11.068,11.053,11.061,11.064,11.049,
     * 11.064,11.045,11.025,11.009,10.991,10.996,10.996,10.982,
     * 10.964,10.943,10.919,10.899,10.879,10.880,10.879,10.862,
     * 10.746,10.719,10.692,10.667,10.641,10.637,10.632,10.612,
     * 10.610,10.579,10.547,10.517,10.489,10.481,10.474,10.453,
     * 10.396,10.358,10.318,10.281,10.246,10.230,10.215,10.190,
     * 9.851,9.833,9.814,9.782,9.752,9.718,9.690,9.663,
     * 9.328,9.316,9.302,9.292,9.288,9.265,9.254,9.245,
     * 9.079,8.993,8.909,8.839,8.773,8.714,8.665,8.620,
     * 8.834,8.754,8.675,8.607,8.545,8.486,8.436,8.394,
     * 8.789,8.709,8.629,8.562,8.500,8.440,8.390,8.349/
      DATA ((EDEN(I,5,12,L,1),I=57,64),L=1,NZEN)/
     * 11.158,11.134,11.107,11.090,11.076,11.086,11.093,11.083,
     * 11.121,11.093,11.064,11.049,11.033,11.041,11.049,11.033,
     * 11.068,11.037,11.009,10.987,10.969,10.977,10.981,10.967,
     * 10.964,10.933,10.900,10.877,10.857,10.862,10.863,10.846,
     * 10.744,10.708,10.670,10.642,10.617,10.617,10.615,10.595,
     * 10.606,10.566,10.524,10.493,10.464,10.461,10.456,10.435,
     * 10.387,10.342,10.294,10.255,10.220,10.207,10.193,10.170,
     * 9.829,9.808,9.786,9.751,9.721,9.690,9.667,9.640,
     * 9.257,9.243,9.229,9.226,9.232,9.220,9.217,9.217,
     * 9.076,8.982,8.890,8.818,8.754,8.696,8.647,8.606,
     * 8.829,8.742,8.656,8.587,8.525,8.468,8.422,8.380,
     * 8.783,8.696,8.610,8.541,8.479,8.423,8.376,8.334/
      DATA ((EDEN(I,1,1,L,2),I=57,64),L=1,NZEN)/
     * 11.210,11.223,11.236,11.236,11.230,11.258,11.279,11.283,
     * 11.167,11.182,11.193,11.193,11.185,11.210,11.230,11.236,
     * 11.107,11.121,11.130,11.127,11.121,11.143,11.164,11.167,
     * 11.000,11.009,11.017,11.013,11.004,11.025,11.041,11.045,
     * 10.773,10.779,10.784,10.780,10.769,10.787,10.803,10.806,
     * 10.636,10.640,10.643,10.639,10.628,10.643,10.658,10.661,
     * 10.433,10.431,10.430,10.423,10.412,10.420,10.430,10.433,
     * 10.029,10.025,10.025,10.013,10.000,9.989,9.985,9.983,
     * 9.580,9.608,9.638,9.659,9.676,9.678,9.687,9.700,
     * 9.076,9.049,9.025,9.004,8.978,8.958,8.945,8.940,
     * 8.833,8.806,8.782,8.761,8.738,8.716,8.702,8.695,
     * 8.787,8.760,8.736,8.715,8.692,8.670,8.656,8.650/
      DATA ((EDEN(I,2,1,L,2),I=57,64),L=1,NZEN)/
     * 11.204,11.215,11.217,11.220,11.215,11.236,11.255,11.258,
     * 11.164,11.170,11.176,11.173,11.167,11.190,11.207,11.210,
     * 11.104,11.107,11.111,11.111,11.104,11.124,11.140,11.140,
     * 10.994,10.997,10.998,10.995,10.985,11.004,11.017,11.017,
     * 10.766,10.766,10.765,10.760,10.750,10.763,10.777,10.777,
     * 10.629,10.626,10.623,10.617,10.606,10.618,10.629,10.630,
     * 10.423,10.417,10.408,10.400,10.386,10.391,10.398,10.400,
     * 10.017,10.004,9.999,9.984,9.968,9.955,9.947,9.944,
     * 9.638,9.649,9.662,9.668,9.673,9.666,9.666,9.673,
     * 9.064,9.029,9.000,8.973,8.947,8.924,8.908,8.901,
     * 8.822,8.788,8.757,8.731,8.706,8.680,8.664,8.656,
     * 8.776,8.742,8.712,8.685,8.660,8.635,8.618,8.610/
      DATA ((EDEN(I,3,1,L,2),I=57,64),L=1,NZEN)/
     * 11.217,11.223,11.225,11.228,11.230,11.250,11.267,11.279,
     * 11.176,11.179,11.179,11.185,11.185,11.204,11.220,11.230,
     * 11.117,11.117,11.117,11.121,11.117,11.137,11.152,11.161,
     * 11.004,11.009,11.004,11.004,11.004,11.021,11.033,11.045,
     * 10.781,10.780,10.777,10.777,10.775,10.790,10.803,10.812,
     * 10.646,10.644,10.640,10.640,10.638,10.650,10.663,10.672,
     * 10.449,10.442,10.435,10.433,10.430,10.438,10.446,10.455,
     * 10.057,10.049,10.045,10.033,10.029,10.021,10.017,10.021,
     * 9.663,9.682,9.702,9.713,9.726,9.728,9.738,9.752,
     * 9.114,9.083,9.053,9.029,9.013,8.995,8.985,8.984,
     * 8.869,8.838,8.809,8.787,8.769,8.750,8.738,8.737,
     * 8.823,8.792,8.763,8.741,8.724,8.704,8.692,8.691/
      DATA ((EDEN(I,4,1,L,2),I=57,64),L=1,NZEN)/
     * 11.230,11.220,11.207,11.199,11.190,11.196,11.199,11.193,
     * 11.188,11.176,11.161,11.152,11.140,11.146,11.149,11.140,
     * 11.121,11.107,11.093,11.083,11.072,11.076,11.076,11.068,
     * 11.004,10.989,10.972,10.959,10.945,10.948,10.947,10.937,
     * 10.756,10.736,10.715,10.699,10.683,10.682,10.680,10.668,
     * 10.602,10.580,10.555,10.537,10.519,10.515,10.511,10.500,
     * 10.365,10.334,10.305,10.281,10.258,10.248,10.241,10.228,
     * 9.825,9.807,9.790,9.766,9.748,9.726,9.709,9.695,
     * 9.296,9.300,9.304,9.309,9.317,9.315,9.316,9.323,
     * 8.960,8.890,8.823,8.772,8.727,8.687,8.655,8.631,
     * 8.710,8.644,8.582,8.531,8.489,8.449,8.417,8.393,
     * 8.664,8.599,8.536,8.486,8.4